/*
 *=============================================================================
 *                                  tSippRLE.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manipulate and render to Utah Raster Toolkit RLE files.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * Out own functions are used to manage comments to prevent memory lose that
 * occurs in the RLE library.
 *-----------------------------------------------------------------------------
 * $Id: tSippRLE.c,v 3.0 1993/01/20 06:43:44 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"
#include "sipp_pixmap.h"
#include "sipp_bitmap.h"

#ifndef TSIPP_NO_RLE

#include "rle.h"

/*
 * Type used to hold a file handle "fileNNN".
 */
typedef char fileHandle_t [10];

/*
 * RLE file table entry.
 */
typedef struct {
    fileHandle_t  handle;
    FILE         *filePtr;
    bool          readable;
    bool          writable;
    bool          eof;
    rle_hdr       rleHeader;
} rleFile_t, *rleFile_pt;

/*
 * Client-data for pixel setting call back used to hold a single row of data.
 * The data will be outputed when the last pixel is written.
 */
typedef struct {
    tSippGlob_pt  tSippGlobPtr;
    rleFile_pt    rleFilePtr;
    rle_hdr      *rleHdrPtr;
    int           xMax;
    rle_pixel   **rowPtr;
} renderData_t, *renderData_pt;

/*
 * Internal prototypes.
 */
static void
RLECleanHeader _ANSI_ARGS_((rle_hdr *rleHdrPtr,
                            bool     cleanComments));

static rleFile_pt
RLEHandleToPtr _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                            char           *handle));

static void
InitRLEHeader _ANSI_ARGS_((rleFile_pt    rleFilePtr));

static bool
ReadRLEHeader _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                           rleFile_pt    rleFilePtr,
                           char         *name));

static void
CopyRLEHeader _ANSI_ARGS_((rleFile_pt    srcFilePtr,
                           rleFile_pt    destFilePtr,
                           bool          mergeComments));

static bool
CloseRLEFile _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                          char           *command,
                          rleFile_pt      rleFilePtr));

static rleFile_pt
RLECommentCmdSetup _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                char           *handle,
                                bool            writeComment,
                                char           *name));

static void *
RLEOutputStart _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                            tSippOutputParms_pt   outputParmsPtr,
                            char                 *handle,
                            char                **comments));

static void
RLEOutputLine _ANSI_ARGS_((renderData_pt  renderDataPtr,
                           int            y,
                           u_char        *rowPtr));

static void
RLEOutputBitMap _ANSI_ARGS_((renderData_pt   renderDataPtr,
                             Sipp_bitmap    *bitMapPtr));

static bool
RLEOutputEnd _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                          tSippOutputParms_pt   outputParmsPtr,
                          renderData_pt         renderDataPtr));

static bool
ParseImageComment _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                               rleFile_pt            rleFilePtr,
                               tSippOutputParms_pt   outputParmsPtr));

static bool
RLEToRLECopy _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr,
                          char         *srcHandle,
                          rleFile_pt    srcFilePtr,
                          char         *destHandle));

static void
RLEToOtherImage _ANSI_ARGS_((tSippOutputParms_pt   outputParmsPtr,
                             rleFile_pt            srcFilePtr,
                             void                 *destClientData,
                             tSippStorageClass_pt  storageClassPtr));

static bool
RLECopyImage _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                          char                 *srcHandle,
                          char                 *destHandle,
                          tSippStorageClass_pt  destClassPtr,
                          bool                  clear));

static void
RLECleanUp _ANSI_ARGS_((tSippGlob_pt  tSippGlobPtr));

/*=============================================================================
 * RLECleanHeader --
 *   Free all dynamic memory associated with an RLE header.  This does not
 * free the header itself.
 * 
 * Parameters:
 *   o rleHdrPtr (I) - Pointer to the RLE file structure.
 *   o cleanComments (I) - If TRUE, comments are reset, if FALSE they are
 *     left alone.
 *-----------------------------------------------------------------------------
 */
static void
RLECleanHeader (rleHdrPtr, cleanComments)
    rle_hdr *rleHdrPtr;
    bool     cleanComments;
{
    if (rleHdrPtr->cmap != NULL) {
        sfree (rleHdrPtr->cmap);
        rleHdrPtr->cmap = NULL;
    }
    if (rleHdrPtr->bg_color != NULL) {
        sfree (rleHdrPtr->bg_color);
        rleHdrPtr->bg_color = NULL;
    }
    if (cleanComments)
        TSippFreeCom (&rleHdrPtr->comments);

}

/*=============================================================================
 * RLEHandleToPtr --
 *   Utility procedure to convert a RLE file handle to a RLE header pointer.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A RLE handle.
 * Returns:
 *   A pointer to the RLE file entry, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static rleFile_pt
RLEHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    rleFile_pt  rleFilePtr;

    rleFilePtr = (rleFile_pt)
        Tcl_HandleXlate (tSippGlobPtr->interp,
                         tSippGlobPtr->rleTblPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;
    return rleFilePtr;

}

/*=============================================================================
 * InitRLEHeader --
 *   Initialize  the header of a RLE file.  This is used to setup a header for
 * a file that is to be written.  If the header had data in it, RLECleanHeader
 * should be called before this function.  This function resets dynamic info
 * in the rleFilePtr
 * 
 * Parameters:
 *   o rleFilePtr (I) - Pointer to the RLE file structure containing the
 *     header.  The filePtr field must be initialized.
 *-----------------------------------------------------------------------------
 */
static void
InitRLEHeader (rleFilePtr)
    rleFile_pt    rleFilePtr;
{
    rleFilePtr->eof = TRUE;  /* Assume writing at the end. */

    rleFilePtr->rleHeader = rle_dflt_hdr;
    rleFilePtr->rleHeader.rle_file = rleFilePtr->filePtr;

    /*
     * Indicate that an image is zero sized until specified by reading,
     * rendering or copying.
     */
    rleFilePtr->rleHeader.xmin = 0;
    rleFilePtr->rleHeader.xmax = -1;
    rleFilePtr->rleHeader.ymin = 0;
    rleFilePtr->rleHeader.ymax = -1;

    /*
     * Allocate room for background RGB values.
     */
    rleFilePtr->rleHeader.bg_color = (int *) smalloc (3 * sizeof (int));

}

/*=============================================================================
 * ReadRLEHeader --
 *   Read the header of a RLE file.   If there is no data left in the file,
 * the EOF is set.  Care is taken not to lose any memory.  After a header
 * is read, the comments are copied and converted to the standard dynamic
 * layout.  All old dynamic data in the header is cleaned up before the read.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o rleFilePtr (I) - Pointer to the RLE file structure containing the
 *     header.  The filePtr field must be initialized.
 *   o name (I) - Name to indentify the file in an error message.  This is
 *     either the file name or handle.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured with error message in
 *   tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static bool
ReadRLEHeader (tSippGlobPtr, rleFilePtr, name)
    tSippGlob_pt  tSippGlobPtr;
    rleFile_pt    rleFilePtr;
    char         *name;
{
    int    status;
    char **oldArrayPtr;

    RLECleanHeader (&rleFilePtr->rleHeader, TRUE);  /* Clean comments */
    rleFilePtr->rleHeader.rle_file = rleFilePtr->filePtr;

    status = rle_get_setup (&rleFilePtr->rleHeader);

    /*
     * Convert comments to standard malloced string layout.
     */
    if (rleFilePtr->rleHeader.comments != NULL) {
        oldArrayPtr = rleFilePtr->rleHeader.comments;
        rleFilePtr->rleHeader.comments = TSippCopyCom (oldArrayPtr);
        sfree (oldArrayPtr);
    }

    switch (status) {
      case RLE_SUCCESS:
        rleFilePtr->eof = FALSE;
        return TRUE;
      case RLE_EMPTY:
      case RLE_EOF:
        rleFilePtr->eof = TRUE;
        return TRUE;
      case RLE_NOT_RLE:
        Tcl_AppendResult (tSippGlobPtr->interp, name, 
                          " is an invalid RLE file", (char *) NULL);
        return FALSE;
      case RLE_NO_SPACE:
        panic ("rle_get_setup: malloc failed");
    }

    return FALSE;  /* Should never make it here */

}

/*=============================================================================
 * CopyRLEHeader --
 *   Copy the header of one file to another.  All dynamic fields are duplicated
 * rather than sharing a pointer, making them independent.
 * 
 * Parameters:
 *   o srcFilePtr (I) - Pointer to the RLE file structure containing the
 *     source header.
 *   o destFilePtr (I) - Pointer to the RLE file structure containing the
 *     source header.
 *   o mergeComments (I) - If TRUE, then the comments will be merged, with
 *     the comments in destFile overriding srcFile.  If FALSE, the comments
 *     will be copied.
 *-----------------------------------------------------------------------------
 */
static void
CopyRLEHeader (srcFilePtr, destFilePtr, mergeComments)
    rleFile_pt    srcFilePtr;
    rleFile_pt    destFilePtr;
    bool          mergeComments;
{
    int    sizeBytes;
    char **saveComments;

    /*
     * Cleanup the header then copy all of the static fields.  The only thing
     * that changes is the stdio file pointer.  Then copy the dynamic fields.
     * Save the comments instead of releasing them if comments are being
     * merged.  The comment pointer saving code just copies a NULL if
     * mergeComments is FALSE;
     */
    RLECleanHeader (&destFilePtr->rleHeader, mergeComments == 0);

    saveComments = destFilePtr->rleHeader.comments;
    destFilePtr->rleHeader = srcFilePtr->rleHeader;
    destFilePtr->rleHeader.rle_file = destFilePtr->filePtr;
    destFilePtr->rleHeader.comments = saveComments;

    /*
     * Copy the color map
     */
    if (srcFilePtr->rleHeader.cmap != NULL) {
        sizeBytes = sizeof (rle_map) * srcFilePtr->rleHeader.cmaplen;
        destFilePtr->rleHeader.cmap = (rle_map *) smalloc (sizeBytes);
        memcpy (destFilePtr->rleHeader.cmap,
                srcFilePtr->rleHeader.cmap,
                sizeBytes);
    }

    /*
     * Copy the background color array.
     */
    if (srcFilePtr->rleHeader.bg_color != NULL) {
        sizeBytes = sizeof (int) * srcFilePtr->rleHeader.ncolors;
        destFilePtr->rleHeader.bg_color = (int *) smalloc (sizeBytes);
        memcpy (destFilePtr->rleHeader.bg_color,
                srcFilePtr->rleHeader.bg_color,
                sizeBytes);
    }

    /*
     * Copy or merge the comments.
     */
    if (mergeComments)
        TSippMergeCom (srcFilePtr->rleHeader.comments,
                       &destFilePtr->rleHeader.comments);
    else
        destFilePtr->rleHeader.comments =
            TSippCopyCom (srcFilePtr->rleHeader.comments);

}

/*=============================================================================
 * CloseRLEFile --
 *   Close an RLE file, deleting the handle entry.
 * 
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o command (I) - The command that called this function (argv [0]).
 *   o rleFilePtr (I) - Pointer to the file entry to close.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
CloseRLEFile (tSippGlobPtr, command, rleFilePtr)
    tSippGlob_pt    tSippGlobPtr;
    char           *command;
    rleFile_pt      rleFilePtr;
{
    int             result;
    char           *closeArgv [2];

    closeArgv [0] = command;
    closeArgv [1] = rleFilePtr->handle;

    result = Tcl_CloseCmd (NULL, tSippGlobPtr->interp, 2, closeArgv);

    RLECleanHeader (&rleFilePtr->rleHeader, TRUE);  /* Clean comments */

    Tcl_HandleFree (tSippGlobPtr->rleTblPtr, rleFilePtr);
    return (result == TCL_OK);

}

/*=============================================================================
 * SippRLEOpen --
 *   Implements the command:
 *     SippRLEOpen filename [access]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.  We cheat by calling
 * the Tcl "open" command executor to actually do the open, that way we get the
 * full semantics of the open command.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEOpen (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    fileHandle_t    fileHandle;
    rleFile_pt      rleFilePtr;
    OpenFile       *filePtr;
    char           *errorMsg;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " filename [access]", (char *) NULL);
        return TCL_ERROR;
    }
    
    /*
     * Use the open command to actually open the file.  If it succeeds, save
     * the handle, which will latter be stored in the RLE file table entry.
     */
    if (Tcl_OpenCmd (NULL, interp, argc, argv) != TCL_OK)
        return TCL_ERROR;

    strcpy (fileHandle, interp->result);
    Tcl_ResetResult (interp);

    if (TclGetOpenFile (interp, fileHandle, &filePtr) != TCL_OK)
         return TCL_ERROR;  /* Should never happen */

    /*
     * Make sure that the file will only be opened for reading or writting,
     * but not both.
     */
    if (filePtr->readable && filePtr->writable) {
        Tcl_AppendResult (interp, "RLE files may be opened for reading or ",
                          "writing, but not both", (char *) NULL);
        goto errorExit;
    }

    /*
     * Allocate and initialize the RLE header struct, defaulting all fields.
     * If its open for read access, then read the header.
     */
    rleFilePtr = (rleFile_pt) Tcl_HandleAlloc (tSippGlobPtr->rleTblPtr,
                                               interp->result);
    strcpy (rleFilePtr->handle, fileHandle);
    rleFilePtr->readable  = filePtr->readable;
    rleFilePtr->writable  = filePtr->writable;
    rleFilePtr->filePtr   = filePtr->f;

    InitRLEHeader (rleFilePtr);

    /*
     * If read access, get the header.  Must stash the handle in case an
     * error gets generated.
     */
    strcpy (fileHandle, interp->result);
    Tcl_ResetResult (interp);

    if (rleFilePtr->readable) {
        if (!ReadRLEHeader (tSippGlobPtr, rleFilePtr, fileHandle))
            goto errorExit;
    }

    strcpy (interp->result, fileHandle);
    return TCL_OK;

    /*
     * Close the file and clean up the without losing an error message.
     */
  errorExit:
    errorMsg = (char *) alloca (strlen (interp->result) + 1);
    strcpy (errorMsg, interp->result);
    Tcl_ResetResult (interp);

    CloseRLEFile (tSippGlobPtr, argv [0], rleFilePtr);

    Tcl_ResetResult (interp);
    Tcl_SetResult (interp, errorMsg, TCL_VOLATILE);

    return TCL_ERROR;

}

/*=============================================================================
 * SippRLEClose --
 *   Implements the command:
 *     SippRLEClose rlehandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEClose (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    rleFile_pt      rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, argv [1]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    if (CloseRLEFile (tSippGlobPtr, argv [0], rleFilePtr))
        return TCL_OK;
    else
        return TCL_ERROR;

}

/*=============================================================================
 * RLECommentCmdSetup --
 *   Utility procedure to set up for one of the comments commands.  If converts
 * the rlehandle and validates the comment name.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A RLE handle.
 *   o writeComment (I) - TRUE if the command is going to write the comment
 *     area.
 *   o name (I) - The name of the comment.  If NULL, then it is not validated
 * Returns:
 *   A pointer to the RLE file entry, or NULL if an error occured.  Errors
 * returned in tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static rleFile_pt
RLECommentCmdSetup (tSippGlobPtr, handle, writeComment, name)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
    bool            writeComment;
    char           *name;
{
    rleFile_pt  rleFilePtr;

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;

    /*
     * Check access.  You can read something that is write only, as the
     * header is stashed in memory.
     */
    if (writeComment & !rleFilePtr->writable) {
        TSippNotWritable (tSippGlobPtr->interp, name);
        return NULL;
    }
    if (!writeComment & !rleFilePtr->writable & rleFilePtr->eof) {
        TSippAtEOF (tSippGlobPtr->interp, name);
        return NULL;
    }

    if ((name != NULL)  && (strpbrk (name, " \f\n\r\t\v=") != NULL)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "name may not contain the ",
                          "`=' or whitespace characters", (char *) NULL);
        return NULL;
    }
    return rleFilePtr;

}

/*=============================================================================
 * SippRLEPutCom --
 *   Implements the command:
 *     SippRLEPutCom rlehandle name [value]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEPutCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    rleFile_pt      rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 3) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle name [value]", (char *) NULL);
        return TCL_ERROR;
    }
    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr, argv [1], TRUE, argv [2]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    TSippPutCom (&rleFilePtr->rleHeader.comments, argv [2], argv [3]);

    return TCL_OK;

}

/*=============================================================================
 * SippRLEGetCom --
 *   Implements the command:
 *     SippRLEGetCom rlehandle [name] [retvar | {}]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEGetCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    rleFile_pt      rleFilePtr;
    char           *comStr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 4)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle [name] [retvar | {}]", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr, argv [1], FALSE, argv [2]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    /*
     * If no name, return a keyed list of all comments.
     */
    if (argc == 2) {
        Tcl_SetResult (interp,
                       TSippGetAllCom (rleFilePtr->rleHeader.comments),
                       TCL_DYNAMIC);
        return TCL_OK;
    }

    /*
     * Search for a single comment.  If comment was not found, either return
     * false or leave result an empty string.  If its was found, either return
     * it in the variable or as the command result.
     */
    comStr = TSippGetCom (rleFilePtr->rleHeader.comments, argv [2]);

    if (comStr == NULL) {
        if (argc == 4)
            interp->result = "0";
    } else {
        if (argc == 4) {
            if (argv [3][0] != '\0') {
                if (Tcl_SetVar (interp, argv [3], comStr, 
                                TCL_LEAVE_ERR_MSG) == NULL)
                    return TCL_ERROR;
            }
            interp->result = "1";
        } else {
            Tcl_SetResult (interp, comStr, TCL_STATIC);
        }
    }
    return TCL_OK;

}

/*=============================================================================
 * SippRLEDelCom --
 *   Implements the command:
 *     SippRLEDelCom rlehandle [name]
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEDelCom (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    rleFile_pt      rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle [name]", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLECommentCmdSetup (tSippGlobPtr, argv [1], TRUE, argv [2]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    if (argc == 2)
        TSippFreeCom (&rleFilePtr->rleHeader.comments);
    else
        TSippDelCom (&rleFilePtr->rleHeader.comments, argv [2]);

    return TCL_OK;

}

/*=============================================================================
 * SippRLEInfo --
 *   Implements the command:
 *     SippRLEInfo rlehandle attribute
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEInfo (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    rleFile_pt      rleFilePtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " rlehandle attribute", (char *) NULL);
        return TCL_ERROR;
    }

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, argv [1]);
    if (rleFilePtr == NULL)
        return TCL_ERROR;

    if (STREQU (argv [2], "EOF")) {
        interp->result = rleFilePtr->eof ? "1" : "0";
        return TCL_OK;
    }

    if (rleFilePtr->eof)
        TSippAtEOF (tSippGlobPtr->interp, argv [1]);

    if (STREQU (argv [2], "XSIZE")) {
        sprintf (interp->result, "%d", (rleFilePtr->rleHeader.xmax -
                                        rleFilePtr->rleHeader.xmin) + 1);
        return TCL_OK;
    }
    if (STREQU (argv [2], "YSIZE")) {
        sprintf (interp->result, "%d", (rleFilePtr->rleHeader.ymax -
                                        rleFilePtr->rleHeader.ymin) + 1);
        return TCL_OK;
    }

    Tcl_AppendResult (interp, "unknown attribute \"", argv [2], 
                      "\", expected one of \"EOF\", \"XSIZE\". or \"YSIZE\"",
                      (char *) NULL);
    return TCL_ERROR;

}

/*=============================================================================
 * RLEOutputStart --
 *   Start output to an RLE file.  This routine is pointed to by the
 * RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - The RLE handle for the file to write.
 *   o comments (I) - If the source image has comments associated with it, they
 *     are passed here.  If the destination object can store them, it possible.
 *     NULL if no comments available.  Should contain standard comments when
 *     rendering.
 *
 * Returns:
 *   A pointer to be passed back into the other RLE output routines or NULL
 *   an a message in tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static void *
RLEOutputStart (tSippGlobPtr, outputParmsPtr, handle, comments)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    char                 *handle;
    char                **comments;
{
    rleFile_pt     rleFilePtr;
    renderData_pt  renderDataPtr;

    rleFilePtr = RLEHandleToPtr (tSippGlobPtr, handle);
    if (rleFilePtr == NULL)
        return NULL;

    if (!rleFilePtr->writable) {
        TSippNotWritable (tSippGlobPtr->interp, handle);
        return NULL;
    }

    if (comments != NULL) {
        TSippMergeCom (comments, &rleFilePtr->rleHeader.comments);
    }

    /*
     * Set up the RLE header for a 24 bit image and write it.
     */
    rleFilePtr->rleHeader.ncolors = 3;  /* Red, Green and Blue */
    rleFilePtr->rleHeader.xmin = 0;
    rleFilePtr->rleHeader.xmax = outputParmsPtr->imgData.xSize - 1;
    rleFilePtr->rleHeader.ymin = 0;
    rleFilePtr->rleHeader.ymax = outputParmsPtr->imgData.ySize - 1;

    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_RED);
    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_GREEN);
    RLE_SET_BIT (rleFilePtr->rleHeader, RLE_BLUE);

    rleFilePtr->rleHeader.bg_color [RLE_RED] =
        tSippGlobPtr->backgroundColor.red * 255;
    rleFilePtr->rleHeader.bg_color [RLE_GREEN] =
        tSippGlobPtr->backgroundColor.grn * 255;
    rleFilePtr->rleHeader.bg_color [RLE_BLUE] =
        tSippGlobPtr->backgroundColor.blu * 255;

    rleFilePtr->rleHeader.background = 2;        /* Use background color */

    /*
     * Write the header to file.
     */

    rle_put_setup (&rleFilePtr->rleHeader);

    /*
     * Setup the clientdata to pass back to the other rendering routines.
     */
    renderDataPtr = (renderData_pt) smalloc (sizeof (renderData_t));

    renderDataPtr->tSippGlobPtr = tSippGlobPtr;
    renderDataPtr->rleFilePtr   = rleFilePtr;
    renderDataPtr->rleHdrPtr    = &rleFilePtr->rleHeader;
    renderDataPtr->xMax         = renderDataPtr->rleHdrPtr->xmax;

    rle_row_alloc (&rleFilePtr->rleHeader, &renderDataPtr->rowPtr);

    return renderDataPtr;

}

/*=============================================================================
 * RLEOutputLine --
 *   Output a rendered line to an RLE file.  This routine is pointed to by the
 * RLE image storage class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the RLE rendering clientdata.
 *   o y (I) - The scan line that was just rendered.
 *   o rowPtr (I) - The pixels for the scanline that was just rendered.
 *-----------------------------------------------------------------------------
 */
static void
RLEOutputLine (renderDataPtr, y, rowPtr)
    renderData_pt  renderDataPtr;
    int            y;
    u_char        *rowPtr;
{
    int x;
    
    for (x = 0; x <= renderDataPtr->xMax; x++) {
        renderDataPtr->rowPtr [RLE_RED] [x]   = rowPtr [TSIPP_RED];
        renderDataPtr->rowPtr [RLE_GREEN] [x] = rowPtr [TSIPP_GREEN];
        renderDataPtr->rowPtr [RLE_BLUE] [x]  = rowPtr [TSIPP_BLUE];
        rowPtr += 3;
    }

    rle_putrow (renderDataPtr->rowPtr, renderDataPtr->xMax,
                renderDataPtr->rleHdrPtr);

}

/*=============================================================================
 * RLEOutputBitMap --
 *   Output a SIPP bit map to an RLE file.  This routine is pointed to by the
 * RLE image storage class table.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the RLE rendering clientdata.
 *   o bitMapPtr (I) - Pointer to the SIPP bit map structure.
 *-----------------------------------------------------------------------------
 */
static void
RLEOutputBitMap (renderDataPtr, bitMapPtr)
    renderData_pt   renderDataPtr;
    Sipp_bitmap    *bitMapPtr;
{
    tSippGlob_pt    tSippGlobPtr = renderDataPtr->tSippGlobPtr;
    rle_pixel     **rowPtr       = renderDataPtr->rowPtr;
    rle_hdr        *rleHdrPtr    = renderDataPtr->rleHdrPtr;
    int             xMax         = renderDataPtr->xMax;
    rle_pixel       lineColor [3], backgroundColor [3];
    int             x, y;
    u_char         *bitRowPtr;
    unsigned        bit;

    lineColor [RLE_RED]   = tSippGlobPtr->lineColor.red * 255;
    lineColor [RLE_GREEN] = tSippGlobPtr->lineColor.grn * 255;
    lineColor [RLE_BLUE]  = tSippGlobPtr->lineColor.blu * 255;

    backgroundColor [RLE_RED]   = tSippGlobPtr->backgroundColor.red * 255;
    backgroundColor [RLE_GREEN] = tSippGlobPtr->backgroundColor.grn * 255;
    backgroundColor [RLE_BLUE]  = tSippGlobPtr->backgroundColor.blu * 255;

    /*
     * Loop setting pixels based on the bit map and outputting rows to the
     * file.
     */

    for (y = rleHdrPtr->ymax - 1; y >= 0 ; y--) {
        bitRowPtr = &bitMapPtr->buffer [y * bitMapPtr->width_bytes];
        
        for (x = 0; x <= xMax; x++) {
            bit = (bitRowPtr [x >> 3] >> (7-(x & 7))) & 1;
            if (bit) {
                rowPtr [RLE_RED] [x]   = lineColor [RLE_RED];
                rowPtr [RLE_GREEN] [x] = lineColor [RLE_GREEN];
                rowPtr [RLE_BLUE] [x]  = lineColor [RLE_BLUE];
            } else {
                rowPtr [RLE_RED] [x]   = backgroundColor [RLE_RED];
                rowPtr [RLE_GREEN] [x] = backgroundColor [RLE_GREEN];
                rowPtr [RLE_BLUE] [x]  = backgroundColor [RLE_BLUE];
            }
        }
        rle_putrow (rowPtr, xMax, rleHdrPtr);
    }

}

/*=============================================================================
 * RLEOutputEnd --
 *   Finish up output to an RLE file.  This routine is pointed to by the
 * RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o renderDataPtr (I) - A pointer to the RLE rendering clientdata.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEOutputEnd (tSippGlobPtr, outputParmsPtr, renderDataPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    renderData_pt         renderDataPtr;
{
   rle_puteof (renderDataPtr->rleHdrPtr);

   rle_row_free (renderDataPtr->rleHdrPtr, renderDataPtr->rowPtr);
   sfree (renderDataPtr);

   fflush (renderDataPtr->rleHdrPtr->rle_file);

   /*
    * Reset the output file header, since we will start with a new image.
    */
   RLECleanHeader (renderDataPtr->rleHdrPtr, TRUE); /* Clean comments */
   InitRLEHeader (renderDataPtr->rleFilePtr);

   return TRUE;

}

/*=============================================================================
 * ParseImageComment --
 *   Parse the TSIPP_IMAGE comment stored in an RLE file header.  This
 * initializes fields in the outputParms structure.  If the comment is not,
 * found, then output parameters are unchanged.  If the comment is found, but
 * is not, valid, an error is returned.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o rleFilePtr (I) - Pointer to the RLE file structure containing the
 *     header.  The filePtr field must be initialized.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured with error message in
 *   tSippGlobPtr->interp->result.
 *-----------------------------------------------------------------------------
 */
static bool
ParseImageComment (tSippGlobPtr, rleFilePtr, outputParmsPtr)
    tSippGlob_pt          tSippGlobPtr;
    rleFile_pt            rleFilePtr;
    tSippOutputParms_pt   outputParmsPtr;
{
    char  *infoList, **infoArgv = NULL, *errorMsg;
    int    infoArgc;
    Color  backgroundColor, lineColor;

    infoList = TSippGetCom (rleFilePtr->rleHeader.comments, "TSIPP_IMAGE");
    if (infoList == NULL)
        return TRUE;

    if (Tcl_SplitList (tSippGlobPtr->interp, infoList,
                       &infoArgc, &infoArgv) != TCL_OK)
        goto errorExit;
    
    if (infoArgc != 5) {
        tSippGlobPtr->interp->result = "wrong number of elements in list";
        goto errorExit;
    }

    /*
     * Convert mode.
     */
    if (STREQU (infoArgv [0], "FLAT"))
        outputParmsPtr->imgData.mode = FLAT;
    else if (STREQU (infoArgv [0], "GOURAUD"))
        outputParmsPtr->imgData.mode = GOURAUD;
    else if (STREQU (infoArgv [0], "PHONG"))
        outputParmsPtr->imgData.mode = PHONG;
    else if (STREQU (infoArgv [0], "LINE"))
        outputParmsPtr->imgData.mode = LINE;
    else if (STREQU (infoArgv [0], "UNKNOWN24"))
        outputParmsPtr->imgData.mode = MODE_UNKNOWN24;
    else {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid mode: ",
                          infoArgv [0], (char *) NULL);
        goto errorExit;
    }

    /*
     * Convert over-sampling factor.
     */
    if (!TSippConvertPosUnsigned (tSippGlobPtr, infoArgv [1],
                                  &outputParmsPtr->imgData.overSampling))
        goto errorExit;

        
    /*
     * Convert field (interlace).
     */
    if (STREQU (infoArgv [2], "BOTH"))
        outputParmsPtr->imgData.field = BOTH;
    else if (STREQU (infoArgv [2], "ODD"))
        outputParmsPtr->imgData.field = ODD;
    else if (STREQU (infoArgv [2], "EVEN"))
        outputParmsPtr->imgData.field = EVEN;
    else  {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid field: ",
                          infoArgv [2], (char *) NULL);
        goto errorExit;
    }

    /*
     * Convert background and line colors.
     */
    if (!TSippConvertColor (tSippGlobPtr, infoArgv [3], &backgroundColor))
        goto errorExit;

    outputParmsPtr->imgData.backgroundColor [TSIPP_RED] =
        backgroundColor.red * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN] =
        backgroundColor.grn * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE] =
        backgroundColor.blu * 255;

    if (!TSippConvertColor (tSippGlobPtr, infoArgv [4], &lineColor))
        goto errorExit;

    outputParmsPtr->imgData.lineColor [TSIPP_RED]   = lineColor.red * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_GREEN] = lineColor.grn * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_BLUE]  = lineColor.blu * 255;

    sfree (infoArgv);
    return TRUE;

    /*
     * Handle error return.  Takes message in interp and adds more info.
     */
  errorExit:
    if (infoArgv == NULL)
        sfree (infoArgv);

    errorMsg = strdup (tSippGlobPtr->interp->result);
    Tcl_ResetResult (tSippGlobPtr->interp);
    Tcl_AppendResult (tSippGlobPtr->interp, "invalid format for TSIPP_IMAGE ",
                      "comment: ", errorMsg, (char *) NULL);
    sfree (errorMsg);
    return FALSE;

}

/*=============================================================================
 * RLEToRLECopy --
 *   Copy an image from an RLE file to another RLE file.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The handle for the source RLE file.  Used for error
 *     messages.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destHandle (I) - The handle for the destination RLE file.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLEToRLECopy (tSippGlobPtr, srcHandle, srcFilePtr, destHandle)
    tSippGlob_pt  tSippGlobPtr;
    char         *srcHandle;
    rleFile_pt    srcFilePtr;
    char         *destHandle;
{
    rleFile_pt  destFilePtr;

    destFilePtr = RLEHandleToPtr (tSippGlobPtr, destHandle);
    if (destFilePtr == NULL)
        return FALSE;

    if (!destFilePtr->writable) {
        TSippNotWritable (tSippGlobPtr->interp, destHandle);
        return FALSE;
    }

    /*
     * Copy the header and data.
     */

    CopyRLEHeader (srcFilePtr, destFilePtr, TRUE);  /* Merge the headers */
    
    rle_put_setup (&destFilePtr->rleHeader);

    rle_cp (&srcFilePtr->rleHeader, &destFilePtr->rleHeader);

    fflush (destFilePtr->filePtr);

    /*
     * Reset the headers.  Clear the write header and read in the next
     * input header, if available.
     */
    RLECleanHeader (&destFilePtr->rleHeader, TRUE); /* Clean comments */
    InitRLEHeader (destFilePtr);
    
    if (!ReadRLEHeader (tSippGlobPtr, srcFilePtr, srcHandle))
        return FALSE;
    
    return TRUE;

}

/*=============================================================================
 * RLEToOtherImage --
 *   Copy an image from an RLE file to some other image type.  This assumes
 * all setup has been done, it just copies the scanlines.
 *
 * Parameters:
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 *-----------------------------------------------------------------------------
 */
static void
RLEToOtherImage (outputParmsPtr, srcFilePtr, destClientData, storageClassPtr)
    tSippOutputParms_pt   outputParmsPtr;
    rleFile_pt            srcFilePtr;
    void                 *destClientData;
    tSippStorageClass_pt  storageClassPtr;
{
    int          x, xSize = outputParmsPtr->imgData.xSize;
    int          y, ySize = outputParmsPtr->imgData.ySize;
    int          redIdx, greenIdx, blueIdx;
    rle_pixel  **inRowPtr;
    u_char      *outRowPtr, *outFillPtr;
    void       (*outputLine) () = storageClassPtr->outputLine;

    /*
     * Set up the in and out row buffers.
     */
    rle_row_alloc (&srcFilePtr->rleHeader, &inRowPtr);
    outRowPtr = (u_char *) alloca (outputParmsPtr->imgData.xSize *
                                   sizeof (u_char) * 3);

    /*
     * Set up indices for copying the color channels.  Normally there will be
     * three color channel, but this will handle a monochrome or even something
     * strange with 2 colors.
     */
    redIdx = 0;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        greenIdx = 1;
    else
        greenIdx = srcFilePtr->rleHeader.ncolors - 1;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        blueIdx = 2;
    else
        blueIdx = srcFilePtr->rleHeader.ncolors - 1;

    /*
     * Copy the image.
     */
    for (y = ySize - 1; y >= 0; y--) {
        rle_getrow (&srcFilePtr->rleHeader, inRowPtr);
        outFillPtr = outRowPtr;

        for (x = 0; x < xSize; x++) {
            outFillPtr [TSIPP_RED]   = inRowPtr [redIdx] [x];
            outFillPtr [TSIPP_GREEN] = inRowPtr [greenIdx] [x];
            outFillPtr [TSIPP_BLUE]  = inRowPtr [blueIdx] [x];
            outFillPtr += 3;
        }
        (*outputLine) (destClientData, y, outRowPtr);
    }

    rle_row_free (&srcFilePtr->rleHeader, inRowPtr);
    
}

/*=============================================================================
 * RLEToOtherBitMap --
 *   Copy an image from an RLE file that we know from the comments was done
 * in line mode to some other image type.  This assumes all setup has been
 * done.
 *
 * Parameters:
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o srcFilePtr (I) - Pointer to the RLE file structure for the source file.
 *   o destClientData (I) - The clientdata returned by the output start
 *     function for the image destination.
 *   o storageClassPtr (I) - Pointer to the image storage class description
 *     for the target class.
 * Notes:
 *   Color indexing for less that 3 channels is still done to handle images
 * that have been post processed, but the results probably will not be very
 * nice.
 *-----------------------------------------------------------------------------
 */
static void
RLEToOtherBitMap (outputParmsPtr, srcFilePtr, destClientData, storageClassPtr)
    tSippOutputParms_pt   outputParmsPtr;
    rleFile_pt            srcFilePtr;
    void                 *destClientData;
    tSippStorageClass_pt  storageClassPtr;
{
    int            x, xSize = outputParmsPtr->imgData.xSize;
    int            y, ySize = outputParmsPtr->imgData.ySize;
    int            redIdx, greenIdx, blueIdx;
    rle_pixel    **inRowPtr;
    Sipp_bitmap   *bitMapPtr;
    u_char         backgroundColor [3], *bitMapRowPtr;

    /*
     * Set up the input row buffer and the bitmap.  Stash the background color.
     */
    rle_row_alloc (&srcFilePtr->rleHeader, &inRowPtr);
    bitMapPtr = sipp_bitmap_create (xSize, ySize);

    backgroundColor [TSIPP_RED] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_RED];
    backgroundColor [TSIPP_GREEN] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN];
    backgroundColor [TSIPP_BLUE] =
        outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE];

    /*
     * Set up indices for copying the color channels.  Normally there will be
     * three color channel, but this will handle a monochrome or even something
     * strange with 2 colors.
     */
    redIdx = 0;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        greenIdx = 1;
    else
        greenIdx = srcFilePtr->rleHeader.ncolors - 1;
    if (srcFilePtr->rleHeader.ncolors >= 2)
        blueIdx = 2;
    else
        blueIdx = srcFilePtr->rleHeader.ncolors - 1;

    /*
     * Read the image and fill in the bitmap.
     */
    for (y = ySize - 1; y >= 0; y--) {
        rle_getrow (&srcFilePtr->rleHeader, inRowPtr);
        bitMapRowPtr = bitMapPtr->buffer + (y * bitMapPtr->width_bytes);

        for (x = 0; x < xSize; x++) {
            if (inRowPtr [redIdx] [x]   != backgroundColor [TSIPP_RED] ||
                inRowPtr [greenIdx] [x] != backgroundColor [TSIPP_GREEN] ||
                inRowPtr [blueIdx] [x]  != backgroundColor [TSIPP_BLUE])

                bitMapRowPtr [x >> 3] |= 1 << (7 - (x & 7));
        }
    }

    (*storageClassPtr->outputBitMap) (destClientData, bitMapPtr);

    rle_row_free (&srcFilePtr->rleHeader, inRowPtr);
    sipp_bitmap_destruct (bitMapPtr);
    
}

/*=============================================================================
 * RLECopyImage --
 *   Copy an image from an RLE file to another image storage object.  This
 * routine is pointed to by the RLE image storage class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o srcHandle (I) - The RLE handle for the file to read.
 *   o destHandle (I) - The handle for the destination image storage object.
 *   o destClassPtr (I) - Pointer to the image storage class description
 *     for the destination class.
 *   o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *     don't clear.  Ignored if the ISO does not support this.
 * Returns:
 *   TRUE is all is OK, or FALSE if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
RLECopyImage (tSippGlobPtr, srcHandle, destHandle, destClassPtr, clear)
    tSippGlob_pt          tSippGlobPtr;
    char                 *srcHandle;
    char                 *destHandle;
    tSippStorageClass_pt  destClassPtr;
    bool                  clear;
{
    rleFile_pt           srcFilePtr;
    tSippOutputParms_t   outputParms;
    int                  status1, status2;
    void                *destClientData;

    srcFilePtr = RLEHandleToPtr (tSippGlobPtr, srcHandle);
    if (srcFilePtr == NULL)
        return FALSE;

    if (!srcFilePtr->readable) {
        TSippNotReadable (tSippGlobPtr->interp, srcHandle);
        return FALSE;
    }
    if (srcFilePtr->eof) {
        TSippAtEOF (tSippGlobPtr->interp, srcHandle);
        return FALSE;
    }

    /*
     * If the target is an RLE file, optimize the copy.
     */
    if (destClassPtr->copyImage == RLECopyImage)
        return RLEToRLECopy (tSippGlobPtr, srcHandle, srcFilePtr, destHandle);

    /*
     * Set up the general output parameters assoicated with this image.
     * Default them first, then try and get better values from the TSIPP_IMAGE
     * comment.
     */
    outputParms.imgData.xSize = (srcFilePtr->rleHeader.xmax -
                                 srcFilePtr->rleHeader.xmin) + 1;
    outputParms.imgData.ySize = (srcFilePtr->rleHeader.ymax -
                                 srcFilePtr->rleHeader.ymin) + 1;

    outputParms.imgData.mode         = MODE_UNKNOWN24;
    outputParms.imgData.overSampling = 1;
    outputParms.imgData.field        = BOTH;
    
    outputParms.imgData.backgroundColor [TSIPP_RED] =
        srcFilePtr->rleHeader.bg_color [RLE_RED];
    outputParms.imgData.backgroundColor [TSIPP_GREEN] =
        srcFilePtr->rleHeader.bg_color [RLE_GREEN];
    outputParms.imgData.backgroundColor [TSIPP_BLUE] =
        srcFilePtr->rleHeader.bg_color [RLE_BLUE];
        
    outputParms.imgData.lineColor [TSIPP_RED]   = 255;
    outputParms.imgData.lineColor [TSIPP_GREEN] = 255;
    outputParms.imgData.lineColor [TSIPP_BLUE]  = 255;

    if (!ParseImageComment (tSippGlobPtr, srcFilePtr, &outputParms))
        return FALSE;

    /*
     * Initialize other output parms fields.
     */
    outputParms.argv     = NULL;   
    outputParms.update   = 0;
    outputParms.maxDepth = 0;
    outputParms.clear    = clear;

    outputParms.bitMapOutput = (outputParms.imgData.mode == LINE) &&
                               (destClassPtr->bitMapOptimal);
    outputParms.scanDirection = TSIPP_SCAN_BOTTOM_UP;

    /*
     * Copy to other image source  If this is from a line image and the target
     * stores bitmaps in an optimal manner, output a bitmap.
     */
    destClientData =
        (*destClassPtr->outputStart) (tSippGlobPtr,
                                      &outputParms,
                                      destHandle,
                                      srcFilePtr->rleHeader.comments);
    if (destClientData == NULL)
        return FALSE;

    if ((outputParms.imgData.mode == LINE) && (destClassPtr->bitMapOptimal)) {
        RLEToOtherBitMap (&outputParms,
                          srcFilePtr,
                          destClientData,
                          destClassPtr);
    } else {
        RLEToOtherImage (&outputParms,
                         srcFilePtr,
                         destClientData,
                         destClassPtr);
    }

    status1 = (*destClassPtr->outputEnd) (tSippGlobPtr,
                                          &outputParms,
                                          destClientData);

    /*
     * Read in the next input header, if available.
     */
    status2 = ReadRLEHeader (tSippGlobPtr, srcFilePtr, srcHandle);

    return status1 && status2;

}

/*=============================================================================
 * TSippRLEInit --
 *   Initialized the Utah Raster Toolkit RLE commands and the RLE file
 *   table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRLEInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippRLEOpen",       (Tcl_CmdProc *) SippRLEOpen},
        {"SippRLEClose",      (Tcl_CmdProc *) SippRLEClose},
        {"SippRLEPutCom",     (Tcl_CmdProc *) SippRLEPutCom},
        {"SippRLEGetCom",     (Tcl_CmdProc *) SippRLEGetCom},
        {"SippRLEDelCom",     (Tcl_CmdProc *) SippRLEDelCom},
        {"SippRLEInfo",       (Tcl_CmdProc *) SippRLEInfo},
        {NULL,                NULL}
    };

    static tSippStorageClass_t storageClass = {
        "rlefile",                  /* handlePrefix  */
        7,                          /* prefixSize    */
        TSIPP_SCAN_BOTTOM_UP,       /* scanDirection */
        FALSE,                      /* bitMapOptimal */
        RLEOutputStart,
        RLEOutputLine,
        RLEOutputBitMap,
        RLEOutputEnd,
        RLECopyImage
    };

    tSippGlobPtr->rleTblPtr =
        Tcl_HandleTblInit ("rlefile", sizeof (rleFile_t), 4);
    tSippGlobPtr->rleCleanUpProc = RLECleanUp;

    TSippInitCmds (tSippGlobPtr, cmdTable);
    TSippAddStorageClass (tSippGlobPtr, &storageClass);

}

/*=============================================================================
 * RLECleanUp --
 *   Close all RLE files and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
RLECleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    int         walkKey = -1;
    rleFile_pt  rleFilePtr;

    while (TRUE) {
        rleFilePtr = Tcl_HandleWalk (tSippGlobPtr->rleTblPtr, &walkKey);
        if (rleFilePtr == NULL)
            break;
        if (!CloseRLEFile (tSippGlobPtr, "RLECleanUp", rleFilePtr))
            Tcl_ResetResult (tSippGlobPtr->interp);
    }
    Tcl_HandleTblRelease (tSippGlobPtr->rleTblPtr);
    tSippGlobPtr->rleTblPtr = NULL;

}

#else /* TSIPP_NO_RLE */

/*=============================================================================
 * SippRLEIsNotHere --
 *   Implements the a command that returns a error indicates that the Utah
 * Raster Toolkit is not available.  If this module is compiled with
 * TSIPP_NO_RLE defined, all RLE commands are bound to this command.
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippRLEIsNotHere (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, "The Utah Raster Toolkit RLE library has not ",
                      "been linked into\n", "this program. ", argv [0],
                      " is not available", (char *) NULL);
    return TCL_ERROR;

}

/*=============================================================================
 * TSippRLEInit --
 *   Initialization when the Utah Raster Toolkit is not available.  Bind all
 *   commands to a routine that returns an error.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRLEInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippRLEOpen",       (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEClose",      (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEPutCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEGetCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEDelCom",     (Tcl_CmdProc *) SippRLEIsNotHere},
        {"SippRLEInfo",       (Tcl_CmdProc *) SippRLEIsNotHere},
        {NULL,                NULL}
    };

    tSippGlobPtr->rleTblPtr = NULL;
    tSippGlobPtr->rleCleanUpProc = NULL;

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

#endif /* TSIPP_NO_RLE */
