/*
 *=============================================================================
 *                                  tSippPoly.c
 *-----------------------------------------------------------------------------
 * Tcl commands to manage SIPP polygons and surfaces.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippPoly.c,v 3.0 1993/01/20 06:43:40 markd Rel $
 *=============================================================================
 */

#include "tSippInt.h"
#include "primitives.h"

/*
 * Internal prototypes.
 */
static void
ClearPolygonStack ();

static void
BindSurfaceToHandle _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                                 Surface         *surfacePtr));

static void
SurfaceHandleCleanup _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr));

static bool
PushVertexList _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                            bool            textures,
                            bool            clockwise,
                            char           *vertexList));

static bool
PushVertexArgList _ANSI_ARGS_((tSippGlob_pt    tSippGlobPtr,
                               int             argc,    
                               char          **argv));
/*=============================================================================
 * ClearPolygonStack --
 *    Clear the polygon stack if it contains any polygons.
 *-----------------------------------------------------------------------------
 */
static void
ClearPolygonStack ()
{
    Surface       *surfacePtr;
    Surf_desc_hdr  surfDescHdr;

    surfDescHdr.ref_count = 1;
    surfDescHdr.free_func = NULL;

    /*
     * Pop any pending polygons into a surface, then delete it.
     */

    surfacePtr = surface_create (&surfDescHdr, NULL);
    if (surfacePtr != NULL)
        surface_delete (surfacePtr);

}

/*=============================================================================
 * BindSurfaceToHandle --
 *   Assigns a handle to the specified surface.
 * Parameters:
 *   o tSippGlobPtr (I) - Pointer to the Tcl SIPP globals. The handle is
 *     returned in interp->result.
 *   o surfacePtr (I) - A pointer to the surface.
 *-----------------------------------------------------------------------------
 */
static void
BindSurfaceToHandle (tSippGlobPtr, surfacePtr)
    tSippGlob_pt    tSippGlobPtr;
    Surface         *surfacePtr;
{
    Surface  **surfaceEntryPtr;

    surfaceEntryPtr = (Surface **)
        Tcl_HandleAlloc (tSippGlobPtr->surfaceTblPtr, 
                         tSippGlobPtr->interp->result);
    *surfaceEntryPtr = surfacePtr;

}

/*=============================================================================
 * SurfaceHandleCleanup --
 *    Delete all surface handles that are defined.  Note that if the surface
 * is not associated with an object, memory will be lost.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static void
SurfaceHandleCleanup (tSippGlobPtr)
    tSippGlob_pt   tSippGlobPtr;
{
    int       walkKey = -1;
    Surface **surfaceEntryPtr;

    while (TRUE) {
        surfaceEntryPtr = Tcl_HandleWalk (tSippGlobPtr->surfaceTblPtr,
                                          &walkKey);
        if (surfaceEntryPtr == NULL)
            break;

        surface_delete (*surfaceEntryPtr);  /* Decrements reference count */

        Tcl_HandleFree (tSippGlobPtr->surfaceTblPtr, surfaceEntryPtr);
    }

}

/*=============================================================================
 * TSippSurfaceHandleToPtr --
 *   Utility procedure to convert a surface handle to a surface pointer.
 *   For use of by functions outside of this module.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - A surface handle.
 * Returns:
 *   A pointer to the surface, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
Surface *
TSippSurfaceHandleToPtr (tSippGlobPtr, handle)
    tSippGlob_pt    tSippGlobPtr;
    char           *handle;
{
    Surface **surfaceEntryPtr;

    surfaceEntryPtr = (Surface **)
        Tcl_HandleXlate (tSippGlobPtr->interp, 
                         tSippGlobPtr->surfaceTblPtr, handle);
    if (surfaceEntryPtr == NULL)
        return NULL;
    return *surfaceEntryPtr;

}

/*=============================================================================
 * PushVertexList --
 *   Parse, convert a list of vertex coordinates, and optional texture
 * coordinates on the vertex stack.
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o textures (I) - TRUE if texture coordinates are includes, FALSE if not.
 *   o clockwise (I) - TRUE if vertices are clockwise, FALSE if counterclock-
 *     wise.
 *   o vertexList (I) - A list of lists of vertex coordinates or vertex and
 *     texture coordinates pairs.
 * Returns:
 *   TRUE if the list and numbers are valid, FALSE if there is an error.
 * Notes:
 *   There should be a way to clean up the vertex stack if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
PushVertexList (tSippGlobPtr, textures, clockwise, vertexList)
    tSippGlob_pt    tSippGlobPtr;
    bool            textures;
    bool            clockwise;
    char           *vertexList;
{
    int      vertexArgc, idx, idxLimit, idxIncr;
    char   **vertexArgv;
    Vector   vertex, textureCoord;

    if (Tcl_SplitList (tSippGlobPtr->interp, vertexList, &vertexArgc,
                       &vertexArgv) != TCL_OK)
        return FALSE;

    if (clockwise) {
        idx      = vertexArgc - 1;
        idxLimit = -1;
        idxIncr  = -1;
    } else {
        idx      = 0;
        idxLimit = vertexArgc;
        idxIncr  = 1;
    }


    if (textures) {
        for (; idx != idxLimit; idx += idxIncr) {
            if (!TSippConvertVertexTex (tSippGlobPtr, vertexArgv [idx],
                                        &vertex, &textureCoord))
                goto errorExit;
            vertex_tx_push (vertex.x, vertex.y, vertex.z, 
                            textureCoord.x, textureCoord.y, textureCoord.z);
        }
    } else {
        for (; idx != idxLimit; idx += idxIncr) {
            if (!TSippConvertVertex (tSippGlobPtr, vertexArgv [idx], &vertex))
                goto errorExit;
            vertex_push (vertex.x, vertex.y, vertex.z);
        }
    }

    sfree (vertexArgv);
    return TRUE;

    /*
     * Cleanup adter an error.  We pop any pending vertices into a polygon,
     * then clear the polygon stack.  This is a no-op if none have been pushed.
     */
errorExit:
    
    polygon_push ();
    ClearPolygonStack ();

    sfree (vertexArgv);
    return FALSE;

}

/*=============================================================================
 * SippPolygonPush --
 *   Implements the command:
 *       SippPolygonPush [-flags] vertexList
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPolygonPush (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;
    bool          textures, clockwise, gotTex, gotClock;
    char         *flag;
    int           argIdx;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    /*
     * Parse the flags and validate argument.
     */
    if (argc < 2)
        goto wrongArgs;
    
    textures = clockwise = gotTex = gotClock = FALSE;
    argIdx = 1;

    while ((argIdx < argc) && (argv [argIdx][0] == '-')) {
        flag = argv [argIdx];
        argIdx++;
        if (STREQU (flag, "-tex")) {
            if (gotTex)
                goto dupTex;
            gotTex = TRUE;
            textures = TRUE;
            continue;
        }
        if (STREQU (flag, "-notex")) {
            if (gotTex)
                goto dupTex;
            gotTex = TRUE;
            textures = FALSE;
            continue;
        }
        if (STREQU (flag, "-clock")) {
            if (gotClock)
                goto dupClock;
            gotClock = TRUE;
            clockwise = TRUE;
            continue;
        }
        if (STREQU (flag, "-counter")) {
            if (gotClock)
                goto dupClock;
            gotClock = TRUE;
            clockwise = FALSE;
            continue;
        }
        Tcl_AppendResult (interp, "invalid flag \"", flag,
                          " expected one of:  \"-tex\", \"-notex\" ",
                          "\"-clock\", or \"-counter\"",
                          (char *) NULL);
        return TCL_ERROR;
    }

    if (argIdx != argc - 1)
        goto wrongArgs;

    if (!PushVertexList (tSippGlobPtr, textures, clockwise, argv [argIdx]))
        return TCL_ERROR;

    polygon_push ();
    return TCL_OK;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                      " [-flags] vertexList", (char *) NULL);
    return TCL_ERROR;

  dupTex:
    interp->result = "can only specify one of \"-tex\" or \"-notex\"";
    return TCL_ERROR;

  dupClock:
    interp->result = "can only specify one of \"-clock\" or \"-counter\"";
    return TCL_ERROR;


}

/*=============================================================================
 * SippPolygonClear --
 *   Implements the command:
 *       SippPolygonClear
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippPolygonClear (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt  tSippGlobPtr = (tSippGlob_pt) clientData;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 1) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], (char *) NULL);
        return TCL_ERROR;
    }

    ClearPolygonStack ();

    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceCreate --
 *   Implements the command:
 *     SippSurfaceCreate shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceCreate (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    Surface        *surfacePtr;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], " shaderhandle",
                          (char *) NULL);
        return TCL_ERROR;
    }                     

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [1],
                                        &surfDescPtr);
    if (shaderPtr == NULL)
        return TCL_ERROR;

    surfacePtr = surface_create (surfDescPtr, shaderPtr);
    if (surfacePtr == NULL) {
        Tcl_AppendResult (interp, "the polygon stack is empty",
                         (char *) NULL);
        return TCL_ERROR;
    }

    BindSurfaceToHandle (tSippGlobPtr, surfacePtr);
    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceUnref --
 *   Implements the command:
 *     SippSurfaceUnref surfacelist|ALL
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceUnref (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    surfaceList;
    handleList_t    surfaceEntryList;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          " surfacelist|ALL", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (STREQU (argv [1], "ALL")) {
        SurfaceHandleCleanup (tSippGlobPtr);
        return TCL_OK;
    }

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [1], &surfaceList, &surfaceEntryList,
                                 NULL))
        return TCL_ERROR;

    for (idx = 0; idx < surfaceList.len; idx++) {
        surface_delete (surfaceList.ptr [idx]);  /* Decrements ref count */
        Tcl_HandleFree (tSippGlobPtr->surfaceTblPtr, 
                        surfaceEntryList.ptr [idx]);
    }

    TSippHandleListFree (&surfaceList);
    TSippHandleListFree (&surfaceEntryList);
    return TCL_OK;

}

/*=============================================================================
 * SippSurfaceSetShader --
 *   Implements the command:
 *     SippSurfaceSetShader surfacelist shaderhandle
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippSurfaceSetShader (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt    tSippGlobPtr = (tSippGlob_pt) clientData;
    int             idx;
    handleList_t    surfaceList;
    Shader         *shaderPtr;
    void           *surfDescPtr;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0],
                          "  surfacelist shaderhandle", (char *) NULL);
        return TCL_ERROR;
    }                     

    if (!TSippHandleListConvert (tSippGlobPtr, tSippGlobPtr->surfaceTblPtr,
                                 argv [1], &surfaceList, NULL, NULL))
        return TCL_ERROR;

    shaderPtr = TSippShaderHandleToPtr (tSippGlobPtr, argv [2], &surfDescPtr);
    if (shaderPtr == NULL)
        goto errorExit;

    for (idx = 0; idx < surfaceList.len; idx++)
        surface_set_shader ((Surface *) surfaceList.ptr [idx], surfDescPtr,
                            shaderPtr);

    TSippHandleListFree (&surfaceList);
    return TCL_OK;
errorExit:
    TSippHandleListFree (&surfaceList);
    return TCL_ERROR;

}

/*=============================================================================
 * TSippPolyInit --
 *   Initialized the polygon and surface commands, including creating the 
 *   polygon table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPolyInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        {"SippPolygonPush",      (Tcl_CmdProc *) SippPolygonPush},
        {"SippPolygonClear",     (Tcl_CmdProc *) SippPolygonClear},
        {"SippSurfaceCreate",    (Tcl_CmdProc *) SippSurfaceCreate},
        {"SippSurfaceUnref",     (Tcl_CmdProc *) SippSurfaceUnref},
        {"SippSurfaceSetShader", (Tcl_CmdProc *) SippSurfaceSetShader},
        {NULL,                   NULL}
    };

    tSippGlobPtr->surfaceTblPtr = 
        Tcl_HandleTblInit ("surface", sizeof (Surface *), 24);

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

/*=============================================================================
 * TSippPolyCleanUp --
 *   Cleanup the surface table and release all associated resources.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippPolyCleanUp (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    ClearPolygonStack ();

    SurfaceHandleCleanup (tSippGlobPtr);

    Tcl_HandleTblRelease (tSippGlobPtr->surfaceTblPtr);
    tSippGlobPtr->surfaceTblPtr = NULL;

}

