/*
 *=============================================================================
 *                              tSippRender.c
 *-----------------------------------------------------------------------------
 * Tcl rendering command to render to one or more image storage objects.  Also 
 * contains a command to copy from one image storage object to another.
 *-----------------------------------------------------------------------------
 * Copyright 1992-1993 Mark Diekhans
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Mark Diekhans makes
 * no representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *-----------------------------------------------------------------------------
 * $Id: tSippRender.c,v 3.0 1993/01/20 06:44:00 markd Rel $
 *-----------------------------------------------------------------------------
 * Scenes are rendered to one or more image storage object.  Each storage
 * class is defined by attributes stored in the tSippGlob.  An instance of
 * the storage class is specified by a handle.  Storage objects can be copied
 * as well as rendered to.
 *
 * The following attributes are defined for each image storage class:
 *
 *   o handlePrefix - The unique prefixed characters in the handle used to
 *     distinguish image storage classes.
 *
 *   o scanDirection - The scanning direction:
 *       o TSIPP_SCAN_BOTH - If the storage object can be written top to bottom
 *         or bottom to top.
 *       o TSIPP_SCAN_TOP_DOWN - If the storage object is written top to
 *         bottom.
 *       o TSIPP_SCAN_BOTTOM_UP - If the storage object is written bottom to
 *         top.
 *
 *   o bitMapOptimal - TRUE if storage of bitmaps in this format is optimized,
 *     FALSE if there is no special optimization for bitmaps.
 *
 *   o outputStart - Function called at the beginning of output.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o outputParmsPtr (I) - The parameters describing the image to
 *           output.
 *         o handle (I) - The handle for the storage object.
 *         o comments (I) - If the source image has comments associated
 *           with it, they are passed here.  If the destination object can
 *           store them, it possible.  NULL if no comments available.
 *       o Returns a generic client data pointer that is then passed in to the
 *         other routines, or NULL if an error occurs.
 *
 *   o outputLine - Function called to output each line.  If ODD or EVEN fields
 *     are being outputed, it is called for alternate lines.
 *       o Parameters:
 *         o clientdata returned from outputStart.
 *         o y (I) - The scan line that was just rendered.
 *         o rowPtr (I) - The pixels for the scanline that was just rendered.
 *
 *   o outputBitMap - Function called to output a bitmap when doing LINE
 *     rendering or output.
 *       o Parameters:
 *         o clientdata returned from outputStart.
 *         o bitMapPtr - Pointer to the SIPP bitmap structure to output. 
 *
 *   o outputEnd - Function called at the end of output.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o outputParmsPtr (I) - The parameters describing the image to
 *           output.
 *         o clientdata returned from outputStart.
 *       o Returns TRUE if all is ok, FALSE and an error message in
 *         tSippGlobPtr->interp->result if an error occurs.
 *
 *   o copyImage - Copy a single image from image storage object to another.
 *     This function is responsible for setting up the output parameters
 *     structure and calling the outputStart, outputEnd functions and using
 *     outputPixel or outputBitMap to output the image to the target image
 *     storage object.
 *       o Parameters:
 *         o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *         o srcHandle (I) - The handle for the source image storage object.
 *         o destHandle (I) - The handle for the destination image storage
 *           object.
 *         o destClassPtr (I) - Pointer to the image storage class description
 *           for the destination class.
 *         o clear (I) - If TRUE, clear target image before copying, if FALSE,
 *           don't clear.  Ignored if the ISO does not support this.
 *       o Returns TRUE is all is OK, or FALSE if an error occurs.
 *
 *=============================================================================
 */

#include "tSippInt.h"
#include "sipp_pixmap.h"
#include "sipp_bitmap.h"

/*
 * Output image storage object information for each of the objects being
 * written.  Some information is cached from the class entry for speed.
 */
typedef struct outputInfo_t {
    void                 *clientData;     /* Client data for output instance */
    void                (*outputLine) (); /* Call for each scan line.        */
    tSippStorageClass_pt  classPtr;       /* Class entry.                    */
    char                  handle [16];    /* Handle associated with object.  */
    bool                  initialized;    /* Startup code has been run.      */
    struct outputInfo_t  *nextPtr;        /* Next output object.             */
} outputInfo_t, *outputInfo_pt;

/*
 * Clientdata for pixel setting call back used to hold a scanline and
 * the output storage object callback information.
 */
typedef struct {
    int            xMax;          /* Maximum pixel number in a row.        */
    u_char        *rowPtr;        /* Buffer containing a scanline.         */
    u_char        *nextPixelPtr;  /* Next pixel in the scan line to write. */
    outputInfo_pt  outputListPtr; /* Output object to write scan lines to. */
} renderData_t, *renderData_pt;


/*
 * Internal prototypes.
 */
static tSippStorageClass_pt
FindOutputClass _ANSI_ARGS_((tSippGlob_pt   tSippGlobPtr,
                             char          *handle));

static void
PixelRender _ANSI_ARGS_((renderData_pt  renderDataPtr,
                         int            x,
                         int            y,
                         int            red,
                         int            green,
                         int            blue));

static void
RenderImage _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                         tSippOutputParms_pt  outputParmsPtr,
                         outputInfo_pt        outputListPtr));

static void
DrawImage _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                       tSippOutputParms_pt  outputParmsPtr,
                       outputInfo_pt        outputListPtr));

static void
AddStandardComment _ANSI_ARGS_((tSippOutputParms_pt     outputParmsPtr,
                                char                 ***commentsPtr));


static bool
ReturnScanDirectionError _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                                      tSippOutputParms_pt  outputParmsPtr,
                                      tSippStorageClass_pt storageClassPtr,
                                      outputInfo_pt        outputListPtr));

static bool
SetupOutputEntry _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                              tSippOutputParms_pt   outputParmsPtr,
                              char                 *handle,
                              outputInfo_pt        *outputListPtrPtr));

static outputInfo_pt
SetupOutputObjects _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                                tSippOutputParms_pt  outputParmsPtr,
                                char                *handleList));

static bool
FinishUpOutputObject _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                                  tSippOutputParms_pt  outputParmsPtr,
                                  outputInfo_pt        outputListPtr));

static bool
RequiresArgError _ANSI_ARGS_((tSippGlob_pt        tSippGlobPtr,
                              char                *flag));

static bool
ParseRenderFlags _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                              char                **argv,
                              tSippOutputParms_pt   outputParmsPtr,
                              int                  *nextArgPtr));

static bool
ParseRenderMode _ANSI_ARGS_((tSippGlob_pt         tSippGlobPtr,
                             char                *modeStr,
                             tSippOutputParms_pt  outputParmsPtr));

static bool
ParseRenderParms _ANSI_ARGS_((tSippGlob_pt          tSippGlobPtr,
                              int                   argc,
                              char                **argv,
                              tSippOutputParms_pt   outputParmsPtr,
                              char                **handleListPtr));

/*=============================================================================
 * FindOutputClass --
 *   Given a storage object handle, locate the image storage class in the
 * class table.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o handle (I) - The output handle.
 * Returns:
 *   A pointer to the storage class entry, or NULL if an error occurs.
 *-----------------------------------------------------------------------------
 */
static tSippStorageClass_pt
FindOutputClass (tSippGlobPtr, handle)
    tSippGlob_pt  tSippGlobPtr;
    char         *handle;
{
    int                  idx;
    tSippStorageClass_pt storageClassPtr;

    /*
     * Find an entry for this handle in the image storage class table.
     */
    for (idx = 0; tSippGlobPtr->storageClassPtrs [idx] != NULL; idx++) {
        storageClassPtr = tSippGlobPtr->storageClassPtrs [idx];
        if (STRNEQU (handle, storageClassPtr->handlePrefix,
                     storageClassPtr->prefixSize))
            break;
    }

    if (tSippGlobPtr->storageClassPtrs [idx] == NULL) {
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "invalid image storage object handle \"",
                          handle, "\"", (char *) NULL);
        return NULL;
    }

    return tSippGlobPtr->storageClassPtrs [idx];

}

/*=============================================================================
 * PixelRender --
 *   Pixel rendering call-back procedure.  SIPP doesn't have the concept of 
 * background and always invokes the call back for every pixel.
 *
 * Parameters:
 *   o renderDataPtr (I) - A pointer to the clientdata, It contains a buffer
 *     with the block of data currently being rendered.
 *   o x, y (I) - The pixel coordinate.
 *   o red, green, blue - (I) - The color values for the pixel.  These are
 *     in the range 0..255 (really should be u_char, but causes the compiler
 *     to complain about argument promotion).
 *-----------------------------------------------------------------------------
 */
static void
PixelRender (renderDataPtr, x, y, red, green, blue)
    renderData_pt  renderDataPtr;
    int            x;
    int            y;
    int            red;
    int            green;
    int            blue;
{
    outputInfo_pt  outputPtr;

    renderDataPtr->nextPixelPtr [TSIPP_RED]   = red;
    renderDataPtr->nextPixelPtr [TSIPP_GREEN] = green;
    renderDataPtr->nextPixelPtr [TSIPP_BLUE]  = blue;
    renderDataPtr->nextPixelPtr += 3;

    /*
     * Filled a scan line, output it to all outputs.
     */
    if (x == renderDataPtr->xMax) {
        outputPtr = renderDataPtr->outputListPtr;
        while (outputPtr != NULL) {
            (*outputPtr->outputLine) (outputPtr->clientData,
                                      y, renderDataPtr->rowPtr);
            outputPtr = outputPtr->nextPtr;
        }

        renderDataPtr->nextPixelPtr = renderDataPtr->rowPtr;
    }

}

/*=============================================================================
 * RenderImage --
 *   Render an image to a set of output objects.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - A pointer to a list of information on the outputs
 *     that the image is to be rendered to.
 *-----------------------------------------------------------------------------
 */
static void
RenderImage (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_pt         tSippGlobPtr;
    tSippOutputParms_pt  outputParmsPtr;
    outputInfo_pt        outputListPtr;
{
    renderData_t renderData;

    /*
     * Set up the clientdata SIPP will pass to our rendering function.
     */
    renderData.xMax          = outputParmsPtr->imgData.xSize - 1;
    renderData.rowPtr        = (u_char *) 
        alloca (outputParmsPtr->imgData.xSize * 3);
    renderData.nextPixelPtr  = renderData.rowPtr;
    renderData.outputListPtr = outputListPtr;

    if (outputParmsPtr->imgData.field == BOTH)
        render_image_func (outputParmsPtr->imgData.xSize,
                           outputParmsPtr->imgData.ySize,
                           PixelRender,
                           &renderData, 
                           outputParmsPtr->imgData.mode,
                           outputParmsPtr->imgData.overSampling);
    else
        render_field_func (outputParmsPtr->imgData.xSize,
                           outputParmsPtr->imgData.ySize,
                           PixelRender,
                           &renderData, 
                           outputParmsPtr->imgData.mode,
                           outputParmsPtr->imgData.overSampling,
                           outputParmsPtr->imgData.field);

}

/*=============================================================================
 * DrawImage --
 *   Render a line image to a list of output storage objects.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - A pointer to a list of information on the outputs
 *     that the image is to be rendered to.
 *-----------------------------------------------------------------------------
 */
static void
DrawImage (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_pt         tSippGlobPtr;
    tSippOutputParms_pt  outputParmsPtr;
    outputInfo_pt        outputListPtr;
{
    Sipp_bitmap   *bitMapPtr;
    outputInfo_pt  outputPtr;

    bitMapPtr = sipp_bitmap_create (outputParmsPtr->imgData.xSize,
                                    outputParmsPtr->imgData.ySize);

    render_image_func (outputParmsPtr->imgData.xSize,
                       outputParmsPtr->imgData.ySize,
                       sipp_bitmap_line,
                       bitMapPtr,
                       LINE, 0);

    outputPtr = outputListPtr;
    while (outputPtr != NULL) {
        (*outputPtr->classPtr->outputBitMap) (outputPtr->clientData,
                                              bitMapPtr);
        outputPtr = outputPtr->nextPtr;
    }

    sipp_bitmap_destruct (bitMapPtr);

}

/*=============================================================================
 * AddStandardComment --
 *    Add the standard Tcl-SIPP comments to an image.
 *
 * Parameters:
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o commentsPtr (I/O) - Pointer to comment array.
 *-----------------------------------------------------------------------------
 */
static void
AddStandardComment (outputParmsPtr, commentsPtr)
    tSippOutputParms_pt     outputParmsPtr;
    char                 ***commentsPtr;
{
    char  *infoArgv [5], *infoList, *bufPtr;
    char   overSamplingStr [16];
    char   backgroundColorStr [3 * TSIPP_DOUBLE_STR_SIZE];
    char   lineColorStr [3 * TSIPP_DOUBLE_STR_SIZE];

    TSippPutCom (commentsPtr, "TSIPP_VERSION", TSIPP_VERSION);

    if (outputParmsPtr->argv != NULL)
        TSippAddHist (commentsPtr, outputParmsPtr->argv);

    switch (outputParmsPtr->imgData.mode) {
      case FLAT:
        infoArgv [0] = "FLAT";
        break;
      case GOURAUD:
        infoArgv [0] = "GOURAUD";
        break;
      case PHONG:
        infoArgv [0] = "PHONG";
        break;
      case LINE:
        infoArgv [0] = "LINE";
        break;
      case MODE_UNKNOWN24:
        infoArgv [0] = "UNKNOWN24";
        break;
    }

    sprintf (overSamplingStr, "%d", outputParmsPtr->imgData.overSampling);
    infoArgv [1] = overSamplingStr;

    switch (outputParmsPtr->imgData.field) {
      case BOTH:
        infoArgv [2] = "BOTH";
        break;
      case ODD:
        infoArgv [2] = "ODD";
        break;
      case EVEN:
        infoArgv [2] = "EVEN";
        break;
    }

    bufPtr = TSippFormatDouble (
        ((double) outputParmsPtr->imgData.backgroundColor [TSIPP_RED]) /
                  255.0,
        backgroundColorStr);
    *bufPtr++ = ' ';

    bufPtr = TSippFormatDouble (
        ((double) outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN]) /
                  255.0,
        bufPtr);
    *bufPtr++ = ' ';

    TSippFormatDouble (
        ((double) outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE]) /
                  255.0,
        bufPtr);
    infoArgv [3] = backgroundColorStr;

    bufPtr = TSippFormatDouble (
        ((double) outputParmsPtr->imgData.lineColor [TSIPP_RED]) /
                  255.0,
       lineColorStr);
    *bufPtr++ = ' ';

    bufPtr = TSippFormatDouble (
        ((double) outputParmsPtr->imgData.lineColor [TSIPP_GREEN]) /
                  255.0,
        bufPtr);
    *bufPtr++ = ' ';

    TSippFormatDouble (
        ((double) outputParmsPtr->imgData.lineColor [TSIPP_BLUE]) /
                  255.0,
        bufPtr);
    infoArgv [4] = lineColorStr;

    infoList = Tcl_Merge (5, infoArgv);

    TSippPutCom (commentsPtr, "TSIPP_IMAGE", infoList);
    sfree (infoList);

}

/*=============================================================================
 * ReturnScanDirectionError --
 *   Determine which other image storage class or command line specification,
 * that the current image storage class conflicts with and return a meaningful
 * error message.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - Error returned in interp->result
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o storageClassPtr (I) - Storage class that conflicts with the current
 *     scan direction.
 *   o outputListPtr (I) - The head of the output list.
 * Returns:
 *   Always returnes FALSE.
 *-----------------------------------------------------------------------------
 */
static bool
ReturnScanDirectionError (tSippGlobPtr, outputParmsPtr, storageClassPtr,
                          outputListPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    tSippStorageClass_pt  storageClassPtr;
    outputInfo_pt         outputListPtr;
{
    char *flag;

    while (outputListPtr != NULL) {
        if (outputListPtr->classPtr->scanDirection != TSIPP_SCAN_BOTH)
            break;
        outputListPtr = outputListPtr->nextPtr;
    }

    if (outputListPtr != NULL) { 
        Tcl_AppendResult (tSippGlobPtr->interp, storageClassPtr->handlePrefix,
                          " scanning direction conflicts with ",
                          outputListPtr->classPtr->handlePrefix,
                          (char *) NULL);
    } else {
        if (outputParmsPtr->scanDirection == TSIPP_SCAN_TOP_DOWN) {
            flag = "-scandown";
        } else {
            flag = "-scanup";
        }

        Tcl_AppendResult (tSippGlobPtr->interp, storageClassPtr->handlePrefix,
                          " scanning direction conflicts with ", flag,
                          (char *) NULL);
    }

    return FALSE;

}

/*=============================================================================
 * SetupOutputEntry --
 *   Given a storage object handle, locate the image storage class in the
 * class table and set up a output list entry.  The initialization code is not
 * run at this time, since the scan direction might not be known.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handle (I) - The output handle.
 *   o outputListPtrPtr (I/O) - A pointer to the head of the output list.
 * Returns:
 *   TRUE if all is OK, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
SetupOutputEntry (tSippGlobPtr, outputParmsPtr, handle, outputListPtrPtr)
    tSippGlob_pt          tSippGlobPtr;
    tSippOutputParms_pt   outputParmsPtr;
    char                 *handle;
    outputInfo_pt        *outputListPtrPtr;
{
    tSippStorageClass_pt storageClassPtr;
    outputInfo_pt        outputInfoPtr;

    storageClassPtr = FindOutputClass (tSippGlobPtr, handle);
    if (storageClassPtr == NULL)
        return FALSE;

    /*
     * Make sure the storage object's rendering direction does not conflict
     * with the direction of any other object.  If there is a conflict,
     * find the conflicting class and output a helpful error message.
     */
    if ((outputParmsPtr->scanDirection != TSIPP_SCAN_BOTH) &&
        (outputParmsPtr->scanDirection != storageClassPtr->scanDirection)) {

        return ReturnScanDirectionError (tSippGlobPtr,     outputParmsPtr,
                                         storageClassPtr, *outputListPtrPtr);
    }

    /*
     * Stash the scanning direction, if its not SIPP_BOTH, this sets it.
     */
    outputParmsPtr->scanDirection = storageClassPtr->scanDirection;
    
    outputInfoPtr = (outputInfo_pt) smalloc (sizeof (outputInfo_t));
    
    outputInfoPtr->clientData  = NULL;
    outputInfoPtr->outputLine  = storageClassPtr->outputLine;
    outputInfoPtr->classPtr    = storageClassPtr;
    outputInfoPtr->initialized = FALSE;
    strcpy (outputInfoPtr->handle, handle);

    outputInfoPtr->nextPtr = *outputListPtrPtr;
    *outputListPtrPtr      = outputInfoPtr;

    return TRUE;

}

/*=============================================================================
 * SetupOutputObjects --
 *   Run through the output list and setup the output info entries for all 
 * output objects that were specified on the command line.  The startup
 * code is run for each object.  The standard comments will be passed to each
 * object.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o handleList (I) - String containing the list of output object handles.
 * Returns:
 *   A pointer to a list of output objects, or NULL if an error occured.
 *-----------------------------------------------------------------------------
 */
static outputInfo_pt
SetupOutputObjects (tSippGlobPtr, outputParmsPtr, handleList)
    tSippGlob_pt         tSippGlobPtr;
    tSippOutputParms_pt  outputParmsPtr;
    char                *handleList;
{
    int             handleArgc, idx;
    char          **handleArgv, *errorMsg;
    void           *outputClientData;
    outputInfo_pt   outputListPtr = NULL, outputInfoPtr;
    char          **comments = NULL;

    if (Tcl_SplitList (tSippGlobPtr->interp, handleList,
                       &handleArgc, &handleArgv) != TCL_OK) {
        return NULL;
    }

    /*
     * Run through and set up each of the outputs in the list.  This also
     * determines the scanning direction.
     */
    for (idx = 0; idx < handleArgc; idx++) {
        if (!SetupOutputEntry (tSippGlobPtr, outputParmsPtr, handleArgv [idx],
                               &outputListPtr))
            goto errorCleanUp;
    }

    AddStandardComment (outputParmsPtr, &comments);

    /*
     * Call the output output classes set up routines.
     */
    outputInfoPtr = outputListPtr;
    while (outputInfoPtr != NULL) {
        outputClientData =
            (*outputInfoPtr->classPtr->outputStart) (tSippGlobPtr,
                                                     outputParmsPtr,
                                                     outputInfoPtr->handle,
                                                     comments);
        if (outputClientData == NULL)
            goto errorCleanUp;
        outputInfoPtr->clientData  = outputClientData;
        outputInfoPtr->initialized = TRUE;

        outputInfoPtr = outputInfoPtr->nextPtr;
    }

    TSippFreeCom (&comments);
    sfree (handleArgv);
    return outputListPtr;

    /*
     * Got an error, run through and do the finish up operation.  At the least,
     * this will clean up each of the entries.  Save the error message, since
     * the finish up function can return an error.
     */
  errorCleanUp:
    TSippFreeCom (&comments);
    sfree (handleArgv);

    errorMsg = strdup (tSippGlobPtr->interp->result);
    Tcl_ResetResult (tSippGlobPtr->interp);

    FinishUpOutputObject (tSippGlobPtr, outputParmsPtr, outputListPtr);

    Tcl_ResetResult (tSippGlobPtr->interp);
    Tcl_SetResult (tSippGlobPtr->interp, errorMsg, TCL_DYNAMIC);
    return NULL;

}

/*=============================================================================
 * FinishUpOutputObject --
 *   Run through the output object list and run the renderEnd functions for
 * each object, if its been initialized, and release the output info entry.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *   o outputParmsPtr (I) - The parameters describing the image to output.
 *   o outputListPtr (I) - The list of output info entries.
 * Returns:
 *   TRUE if all is ok, FALSE and an error message in
 *  tSippGlobPtr->interp->result if an error occurs.
 *-----------------------------------------------------------------------------
 */
static bool
FinishUpOutputObject (tSippGlobPtr, outputParmsPtr, outputListPtr)
    tSippGlob_pt         tSippGlobPtr;
    tSippOutputParms_pt  outputParmsPtr;
    outputInfo_pt        outputListPtr;
{
    outputInfo_pt  freeEntryPtr;
    bool           status;
    char          *errorMsg = NULL;

    /*
     * Loop through doing output ends.  The first error message is saved and
     * returned.
     */
    while (outputListPtr != NULL) {
        if (outputListPtr->initialized) {
            status =
                (*outputListPtr->classPtr->outputEnd) (tSippGlobPtr,
                                                    outputParmsPtr,
                                                    outputListPtr->clientData);
            if (!status) {
                if (errorMsg != NULL)
                    errorMsg = strdup (tSippGlobPtr->interp->result);
                Tcl_ResetResult (tSippGlobPtr->interp);
            }
        }
        freeEntryPtr = outputListPtr;
        outputListPtr = outputListPtr->nextPtr;
        sfree (freeEntryPtr);
    }

    if (errorMsg != NULL) {
        Tcl_SetResult (tSippGlobPtr->interp, errorMsg, TCL_DYNAMIC);
        return FALSE;
    }
    return TRUE;

}

/*=============================================================================
 * RequiresArgError --
 *   Return a error indicating that a flag requires an argument.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o flag (I) - Flag that requires an argument.
 * Returns:
 *   FALSE, so function can be called in a return.
 *-----------------------------------------------------------------------------
 */
static bool
RequiresArgError (tSippGlobPtr, flag)
    tSippGlob_pt        tSippGlobPtr;
    char                *flag;
{
    Tcl_AppendResult (tSippGlobPtr->interp, "\"", flag,
                      "\" flag requires an argument", (char *) NULL);
    return FALSE;

}

/*=============================================================================
 * ParseRenderFlags --
 *   Parse the flags to the rendering commands.  Flags must be the first
 *  arguments.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o argv (I) - Command argument vector to parse. 
 *   o outputParmsPtr (O) - The parameters describing the image to output. 
 *     Fields are initialized from the flags.
 *   o nextArgPtr (O) - The index of the next argument after the flags is
 *     returned here.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderFlags (tSippGlobPtr, argv, outputParmsPtr, nextArgPtr)
    tSippGlob_pt          tSippGlobPtr;
    char                **argv;
    tSippOutputParms_pt   outputParmsPtr;
    int                  *nextArgPtr;
{
    char *argPtr;
    int   nextArg = 1;
    bool  gotField, gotClear, gotDirection;

    gotField = gotClear = gotDirection = FALSE;

    for (; (argv [nextArg] != NULL) && argv [nextArg][0] == '-'; nextArg++) {
        argPtr = argv [nextArg];

        if (STREQU (argPtr, "-both")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = BOTH;
            continue;
        }
        if (STREQU (argPtr, "-odd")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = ODD;
            continue;
        }
        if (STREQU (argPtr, "-even")) {
            if (gotField)
                goto dupField;
            gotField = TRUE;
            outputParmsPtr->imgData.field = EVEN;
            continue;
        }
        if (STREQU (argPtr, "-clear")) {
            if (gotClear)
                goto dupClear;
            gotClear = TRUE;
            outputParmsPtr->clear = TRUE;
            continue;
        }
        if (STREQU (argPtr, "-noclear")) {
            if (gotClear)
                goto dupClear;
            gotClear = TRUE;
            outputParmsPtr->clear = FALSE;
            continue;
        }
        if (STREQU (argPtr, "-either")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_BOTH;
            continue;
        }
        if (STREQU (argPtr, "-topdown")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_TOP_DOWN;
            continue;
        }
        if (STREQU (argPtr, "-bottomup")) {
            if (gotDirection)
                goto dupDirection;
            gotDirection = TRUE;
            outputParmsPtr->scanDirection = TSIPP_SCAN_BOTTOM_UP;
            continue;
        }
        if (STREQU (argPtr, "-update")) {
            unsigned update;

            if (argv [nextArg + 1] == NULL) {
                return RequiresArgError (tSippGlobPtr, argv [nextArg]);
            }
            if (Tcl_GetUnsigned (tSippGlobPtr->interp, argv [nextArg + 1],
                                 &update) != TCL_OK)
                return FALSE;
            outputParmsPtr->update = update;
            nextArg++;
            continue;
        }
        /*
         * Made it here, valid flag was not found.
         */
        Tcl_AppendResult (tSippGlobPtr->interp,
                          "invalid flag \"", argv [nextArg],
                          "\", expected one of \"-both\", \"-odd\", ",
                          "\"-even\", \"-update\", \"-clear\", \"-noclear\", ",
                          "\"-either\", \"-topdown\", or \"-bottomup\"",
                          (char *) NULL);
        return FALSE;
    }

    *nextArgPtr = nextArg;
    return TRUE;

    /*
     * Error return code for duplicated flags.
     */

  dupField:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-both\", \"-odd\", or ",
                      "\"-even\" may be specified", (char *) NULL);
    return FALSE;

  dupClear:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-clear\", or \"-noclear\"",
                      " may be specified", (char *) NULL);
    return FALSE;

  dupDirection:
    Tcl_AppendResult (tSippGlobPtr->interp,
                      "only one of \"-either\", \"-topdown\", or ",
                      "\"-bottomup\" may be specified", (char *) NULL);
    return FALSE;

}

/*=============================================================================
 * ParseRenderMode --
 *   Convert a string representing the SIPP render mode.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o modeStr (I) - The mode string to convert.
 *   o outputParmsPtr (O) - The output parameters to fill in from the command
 *     line.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderMode (tSippGlobPtr, modeStr, outputParmsPtr)
    tSippGlob_pt         tSippGlobPtr;
    char                *modeStr;
    tSippOutputParms_pt  outputParmsPtr;
{
    if (STREQU (modeStr, "PHONG"))
        outputParmsPtr->imgData.mode = PHONG;
    else
    if (STREQU (modeStr, "GOURAUD"))
        outputParmsPtr->imgData.mode = GOURAUD;
    else
    if (STREQU (modeStr, "FLAT"))
        outputParmsPtr->imgData.mode = FLAT;
    else
    if (STREQU (modeStr, "LINE")) {
        outputParmsPtr->imgData.mode = LINE;
    } else {
        Tcl_AppendResult (tSippGlobPtr->interp, "invalid rendering mode, ",
                          "expect one of `PHONG', `GOURAUD', `FLAT', ",
                          "or `LINE', got: ", modeStr,
                          (char *) NULL);
        return FALSE;
    }
    return TRUE;

}

/*=============================================================================
 * ParseRenderParms --
 *   Procedure to parse the render command parameters commands.  All
 * parameters are handled except for the handle list.
 * The command is in the form:
 *
 *  SippRender [-flags] outputlist xsize ysize [mode] [oversample]
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *     Errors are returned in interp->result.
 *   o argc, argv (I) - Arguments to the command.
 *   o outputParmsPtr (O) - The parameters describing the image to output.
 *     Info filled in from the command line args.
 *   o handleListPtr (O) - A pointer to the string from the argv containing the
 *     output object handle list.
 * Returns:
 *   TRUE if all is ok, FALSE if an error occured.
 *-----------------------------------------------------------------------------
 */
static bool
ParseRenderParms (tSippGlobPtr, argc, argv, outputParmsPtr, handleListPtr)
    tSippGlob_pt          tSippGlobPtr;
    int                   argc;
    char                **argv;
    tSippOutputParms_pt   outputParmsPtr;
    char                **handleListPtr;
{
    int nextArg;

    /*
     * Default all fields that might need it up front.
     */
    outputParmsPtr->imgData.mode         = PHONG;
    outputParmsPtr->imgData.overSampling = 1;
    outputParmsPtr->imgData.field        = BOTH;

    outputParmsPtr->imgData.backgroundColor [TSIPP_RED] =
        tSippGlobPtr->backgroundColor.red * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_GREEN] =
        tSippGlobPtr->backgroundColor.grn * 255;
    outputParmsPtr->imgData.backgroundColor [TSIPP_BLUE] =
        tSippGlobPtr->backgroundColor.blu * 255;

    outputParmsPtr->imgData.lineColor [TSIPP_RED] =
        tSippGlobPtr->lineColor.red * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_GREEN] =
        tSippGlobPtr->lineColor.grn * 255;
    outputParmsPtr->imgData.lineColor [TSIPP_BLUE] =
        tSippGlobPtr->lineColor.blu * 255;

    outputParmsPtr->argv           = argv;
    outputParmsPtr->update         = -1;
    outputParmsPtr->maxDepth       = 1024;
    outputParmsPtr->clear          = TRUE;
    outputParmsPtr->bitMapOutput   = FALSE;
    outputParmsPtr->scanDirection  = TSIPP_SCAN_BOTH;

    if (!ParseRenderFlags (tSippGlobPtr, argv, outputParmsPtr, &nextArg))
        return FALSE;

    if (((argc - nextArg) < 3) || ((argc - nextArg) > 5)) {
        Tcl_AppendResult (tSippGlobPtr->interp, "wrong # args: ", argv [0],
                          " [-flags] outputlist xsize ysize [mode] ",
                          "[oversample]", (char *) NULL);
        return FALSE;
    }

    *handleListPtr = argv [nextArg];
    nextArg++;

    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                  &outputParmsPtr->imgData.xSize))
        return FALSE;
    nextArg++;

    if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                  &outputParmsPtr->imgData.ySize))
        return FALSE;
    nextArg++;

    if ((nextArg < argc) && (argv [nextArg][0] != '\0')) {
        if (!ParseRenderMode (tSippGlobPtr, argv [nextArg], outputParmsPtr))
            return FALSE;
    }
    nextArg++;

    if ((nextArg < argc) && (argv [nextArg][0] != '\0')) {
        if (!TSippConvertPosUnsigned (tSippGlobPtr, argv [nextArg],
                                      &outputParmsPtr->imgData.overSampling))
            return FALSE;
    }

    if (outputParmsPtr->imgData.mode == LINE) {
        if (outputParmsPtr->imgData.field != BOTH) {
            Tcl_AppendResult (tSippGlobPtr->interp, "can't specify \"-odd\" ",
                              "or \"-even\" with a mode of \"LINE\"",
                              (char *) NULL);
            return FALSE;
        }
        outputParmsPtr->imgData.overSampling = 1;
        outputParmsPtr->bitMapOutput = TRUE;
    }
    if (outputParmsPtr->update < 0)
        outputParmsPtr->update = outputParmsPtr->imgData.xSize;

    return TRUE;

}

/*=============================================================================
 * SippRender --
 *   Implements the command:
 *     SippRender [-flags] outputList xsize ysize [mode] [oversample]
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *
 * Great care must be take with the updates and the abort flag.  If we did
 * an update before rendering begins and an abort takes place in that update,
 * then the abort might be lost, as the rendering clears it at the beginning.
 *-----------------------------------------------------------------------------
 */
static int
SippRender (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt        tSippGlobPtr = (tSippGlob_pt) clientData;
    tSippOutputParms_t  outputParms;
    outputInfo_pt       outputListPtr;
    char               *handleList;
    bool                ok;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);
                          
    if (!ParseRenderParms (tSippGlobPtr, argc, argv, &outputParms,
                           &handleList))
        return TCL_ERROR;

    /*
     * Setup the list of output objects.
     */
    outputListPtr = SetupOutputObjects (tSippGlobPtr, &outputParms,
                                        handleList);
    if (outputListPtr == NULL)
        return TCL_ERROR;

    /*
     * Flag rendering in progress to prevent modifying SIPP during an Tk event
     * update.
     */
    tSippGlobPtr->rendering = TRUE;

    /*
     * Set the scanning direction if not decided and tell SIPP about it.
     */
    if (outputParms.scanDirection == TSIPP_SCAN_BOTH) {
        outputParms.scanDirection = TSIPP_SCAN_TOP_DOWN;
    }
    sipp_render_direction (outputParms.scanDirection == TSIPP_SCAN_BOTTOM_UP);

    if (tSippGlobPtr->updateProc != NULL && outputParms.update > 0)
        sipp_set_update_callback (tSippGlobPtr->updateProc,
                                  tSippGlobPtr,
                                  outputParms.update);

    /*
     * Do the rendering.
     */
    if (outputParms.imgData.mode == LINE)
        DrawImage (tSippGlobPtr, &outputParms, outputListPtr);
    else
        RenderImage (tSippGlobPtr, &outputParms, outputListPtr);

    ok = FinishUpOutputObject (tSippGlobPtr, &outputParms, outputListPtr);

    sipp_set_update_callback (NULL, NULL, 0);

    /*
     * Finish up with an update.  Aborts in here would not matter, since
     * we are done.
     */
    if (tSippGlobPtr->updateProc != NULL && outputParms.update > 0)
        (*tSippGlobPtr->updateProc) (tSippGlobPtr);

    /*
     * An error due to a signal in a non-Tk environment will be stashed in the
     * globals.
     */
    if (tSippGlobPtr->delayedError != NULL) {
        Tcl_ResetResult (interp);
        Tcl_SetResult (interp, tSippGlobPtr->delayedError, TCL_DYNAMIC);
        tSippGlobPtr->delayedError = NULL;
        ok = FALSE;
    }

    tSippGlobPtr->rendering = FALSE;

    return ok ? TCL_OK : TCL_ERROR;

}

/*=============================================================================
 * SippCopy --
 *   Implements the command:
 *     SippCopy [-flags] inputHandle outputHandle
 *
 * Note:
 *   This procedure has standard Tcl command calling sematics.  ClientData
 * contains a pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
static int
SippCopy (clientData, interp, argc, argv)
    char       *clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    tSippGlob_pt          tSippGlobPtr = (tSippGlob_pt) clientData;
    tSippStorageClass_pt  inputClassPtr, outputClassPtr;
    int                   nextArg = 1;
    bool                  ok, clear = TRUE;

    if (tSippGlobPtr->rendering)
        return TSippNotWhileRendering (interp);

    if (argc < 2)
        goto wrongArgs;

    if (argv [1][0] == '-') {
        if (STREQU (argv [1], "-clear")) {
            clear = TRUE;
        } else if (STREQU (argv [1], "-noclear")) {
            clear = FALSE;
        } else {
            Tcl_AppendResult (tSippGlobPtr->interp,
                              "invalid flag \"", argv [1],
                              "\", expected one of  \"-clear\", \"-noclear\"",
                              (char *) NULL);
            return TCL_ERROR;
        }
        nextArg = 2;
    }

    if ((argc - nextArg) != 2)
        goto wrongArgs;

    /*
     * Find the output classes for each handle.
     */

    inputClassPtr = FindOutputClass (tSippGlobPtr, argv [nextArg]);
    if (inputClassPtr == NULL)
        return TCL_ERROR;
    
    outputClassPtr = FindOutputClass (tSippGlobPtr, argv [nextArg + 1]);
    if (outputClassPtr == NULL)
        return TCL_ERROR;

    /*
     * Make sure that the scanning direction of the two classes don't
     * conflict and the you can copy from the input image.
     */
    if ((inputClassPtr->scanDirection != TSIPP_SCAN_BOTH) &&
        (outputClassPtr->scanDirection != TSIPP_SCAN_BOTH) &&
        (inputClassPtr->scanDirection != outputClassPtr->scanDirection)) {
        Tcl_AppendResult (interp, argv [nextArg], 
                          " scanning direction conflicts with ",
                          argv [nextArg + 1], (char *) NULL);
        return TCL_ERROR;
    }

    if (inputClassPtr->copyImage == NULL) {
        Tcl_AppendResult (interp, argv [nextArg],
                          " can't be used as a copy source", (char *) NULL);
        return TCL_ERROR;
    }

    ok = (*inputClassPtr->copyImage) (tSippGlobPtr,
                                      argv [nextArg],
                                      argv [nextArg + 1],
                                      outputClassPtr,
                                      clear);

    return ok ? TCL_OK : TCL_ERROR;

  wrongArgs:
    Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                      " [-flags] inputHandle outputHandle",
                      (char *) NULL);
    return TCL_ERROR;
}

/*=============================================================================
 * TSippRenderInit --
 *   Initialized rendering commands.
 *
 * Parameters:
 *   o tSippGlobPtr (I) - A pointer to the Tcl SIPP global structure.
 *-----------------------------------------------------------------------------
 */
void
TSippRenderInit (tSippGlobPtr)
    tSippGlob_pt  tSippGlobPtr;
{
    static tSippTclCmdTbl_t cmdTable [] = {
        "SippRender", (Tcl_CmdProc *) SippRender,
        "SippCopy",   (Tcl_CmdProc *) SippCopy,
        {NULL,        NULL}
    };

    TSippInitCmds (tSippGlobPtr, cmdTable);

}

