# DOC
#
# awkproc.tcl - simple Tcl package for analysing/processing text files
#
# Copyright (C) 2001 Arjen Markus
#
#
# General information:
#
# AWKPROC is a simple Tcl package that has similar capabilities
# as the UNIX utility AWK.
# By using Tcl as the programming environment, one can create
# more flexible scripts than is possible with AWK itself.
#
# Though the package has similar capabilities the syntax is not
# compatible to AWK. That would have been a lot of (unnecessary) work.
#
# This package contains the following procedures:
# Pattern:       Define a simple pattern with some action
# RegPattern:    Define a regular expression with some action
# Content:       Manipulate the status information (the "content")
# DefaultAction: Define a default action per line
# BeginFile:     Define an action for the beginning of a file
# EndFile:       Define an action for the end of a file
# ProcessFiles:  Process the given files, using the given patterns
#
# To use it:
# package require AwkProc
#
# version 0.2: complete revision, june 2001
#
# ENDDOC
#
#  $Author$
#  $Date$
#  $Source$
#  $Log$
#

# --------------------------------------------------------------------
#   Package:   AwkProc
# --------------------------------------------------------------------
#
package provide AwkProc 0.2

namespace eval ::AwkProc {
   global LINE
   global NL
   global NLT
   global MATCH
   global FILENAME

   variable PatternActions   {}
   variable NoPatterns       0
   variable State
   variable Default_action   {}
   variable Default_set      0
   variable BeginFile_action {}
   variable BeginFile_set    0
   variable EndFile_action   {}
   variable EndFile_set      0
   variable infile           {}
   variable loop_level       {}
   variable skip_patterns    0

   namespace export Pattern Content DefaultAction BeginFile EndFile \
                    RegPattern ProcessFiles

# --------------------------------------------------------------------
#   Procedure: Pattern
#   Author:    Arjen Markus
#   Purpose:   Define a new pattern with an associated script
#   Context:   Used before processing the files
#   Summary:
#              Create a new command that implements the pattern and
#              executes the script. Register the command.
#   Arguments:
#   patt       The (regular) expression to be tested
#   action     The action to take if the pattern matches
#
#   To do:
#              Implement options! -nocase, -regexp, -glob
# --------------------------------------------------------------------
#
proc Pattern { patt action } {
   variable PatternActions
   variable NoPatterns

   incr NoPatterns

   # Note: simply find a substring - for the moment

   lappend PatternActions "[namespace current]::Pattern$NoPatterns"
   proc Pattern$NoPatterns { } \
      [format {
            global LINE
            global FILENAME
            global NL
            global NLT
            global MATCH
            if { [ string first %1$s $LINE ] > -1 } {
            %2$s }
            } [list $patt] $action ]

   return
}

# --------------------------------------------------------------------
#   Procedure: RegPattern
#   Author:    Arjen Markus
#   Purpose:   Define a regular expression with an associated script
#   Context:   Used before processing the files
#   Summary:
#              Create a new command that implements the pattern and
#              executes the script. Register the command.
#   Arguments:
#   patt       The (regular) expression to be tested
#   action     The action to take if the pattern matches
#
#   To do:
#              Implement options! -nocase, -regexp, -glob
# --------------------------------------------------------------------
#
proc RegPattern { patt action } {
   variable PatternActions
   variable NoPatterns

   incr NoPatterns

   # Note: simply find a substring - for the moment

   lappend PatternActions "[namespace current]::Pattern$NoPatterns"
   proc Pattern$NoPatterns { } \
      [format {
            global LINE
            global FILENAME
            global NL
            global NLT
            global MATCH
            if { [ regexp %1$s $LINE ] == 1 } {
            %2$s }
            } [list $patt] $action ]

   return
}

# --------------------------------------------------------------------
#   Procedure: Content
#   Author:    Arjen Markus
#   Purpose:   Manipulate the content/state information
#   Context:   Used in actions to keep track of the state
#   Summary:
#              Handle the following subcommands:
#              - init:       Initialise the content
#              - initProc:   Define the initialisation procedure
#              - export:     Execute the export procedure
#              - exportProc: Define the export procedure
#              - set:        Set a variable in the state array
#              - append:     Append text to a variable
#              - lappend:    Append text to a list variable
#              - get:        Return the current value
#              The state information is actually an array within
#              the namespace.
#   Arguments:
#   subcommand The subcommand to be executed
#   args       Any arguments to it
# --------------------------------------------------------------------
#
proc Content { subcommand args } {
   variable State

   switch $subcommand {
      init       { [namespace current]::initProc }
      export     { [namespace current]::exportProc }
      initProc   { proc [namespace current]::initProc {} [lindex $args 0] }
      exportProc { proc [namespace current]::exportProc {} [lindex $args 0] }
      set        { set var   [ lindex $args 0 ]
                   set value [ lindex $args 1 ]
                   set State($var) $value
                 }
      append     { set var   [ lindex $args 0 ]
                   set value [ lindex $args 1 ]
                   append State($var) $value
                 }
      lappend    { set var   [ lindex $args 0 ]
                   set value [ lindex $args 1 ]
                   lappend State($var) $value
                 }
      get        { set var   [ lindex $args 0 ]
                   return $State($var)
                 }
      default    { # Nothing
                 }
   }
   return
}

# --------------------------------------------------------------------
#   Procedure: GetNextLine
#   Author:    Arjen Markus
#   Purpose:   Get the next line from the file
#   Context:   Used in the actions
#   Summary:
#              Read the next line from the file
#   Arguments:
#              -- none --
#   Returns:
#              return value of gets
# --------------------------------------------------------------------
#
proc GetNextLine { } {
   variable infile
   global   LINE

   return [gets $infile LINE]
}

# --------------------------------------------------------------------
#   Procedure: SkipPatterns
#   Author:    Arjen Markus
#   Purpose:   Get the rest of the patterns
#   Context:   Used in the actions
#   Summary:
#              Set the variable skip_patterns so that the loop
#              over the patterns is broken upon return from the
#              current pattern.
#   Arguments:
#              -- none --
#   Note:
#              "uplevel x break" did not work. This may be a cleaner
#              method anyway.
# --------------------------------------------------------------------
#
proc SkipPatterns { } {
   variable skip_patterns

   set skip_patterns 1
   return
}

# --------------------------------------------------------------------
#   Procedure: DefaultAction, BeginFile, EndFile
#   Author:    Arjen Markus
#   Purpose:   Define a new special action
#   Context:   Used before processing the files
#   Summary:
#              Set the variables ..._action and ..._set
#   Arguments:
#   action     The action to take if the special event occurs
# --------------------------------------------------------------------
#
proc DefaultAction { action } {
   variable Default_action
   variable Default_set

   set Default_action $action
   set Default_set [ string length "$action" ]
   return
}

proc BeginFile { action } {
   variable BeginFile_action
   variable BeginFile_set

   set BeginFile_action $action
   set BeginFile_set [ string length "$action" ]
   return
}

proc EndFile { action } {
   variable EndFile_action
   variable EndFile_set

   set EndFile_action $action
   set EndFile_set [ string length "$action" ]
   return
}

# --------------------------------------------------------------------
#   Procedure: options
#   Author:    Arjen Markus
#   Purpose:   Define a new set of options
#   Context:   Used before processing the files
#   Summary:
#              Scan the arguments and set the options in the options
#              array, if recognised.
#   Arguments:
#   args       The action to take if the pattern matches
# --------------------------------------------------------------------
#
proc options { args } {
   variable Options

   foreach opt $args {
      switch $opt {
         "case"    { set Options(nocase)   0 }
         "nocase"  { set Options(nocase)   1 }
         "debug"   { set Options(debug)    1 }
         "nodebug" { set Options(debug)    0 }
         default   { puts "options: unknown option $opt" }
      }
   }
   return
}

# --------------------------------------------------------------------
#   Procedure: ProcessFiles
#   Author:    Arjen Markus
#   Purpose:   Process the input files with the given patterns and
#              conditions
#   Context:   Used after specifying the patterns and conditions
#   Summary:
#              Consider the arguments to be file names. Open the
#              file(s) and read the lines one by one. Then check
#              the patterns and conditions. If they match, execute
#              the action.
#   Note:
#              All of this is done in global context, to avoid
#              problems with scopes.
#   Arguments:
#   args       List of file names
# --------------------------------------------------------------------
#
proc ProcessFiles { args } {
   variable PatternActions
   variable Default_action
   variable Default_set
   variable BeginFile_action
   variable BeginFile_set
   variable EndFile_action
   variable EndFile_set
   variable infile
   variable loop_level
   variable skip_patterns
   global   LINE
   global   NL
   global   NLT
   global   MATCH
   global   FILENAME

   set NLT 0
   set loop_level [info level]
   foreach FILENAME $args {
      #
      # Open the input file and scan each line
      #
      set NL 0
      if [ catch { open $FILENAME "r" } infile ] {
         puts stderr "Cannot open $FILENAME: $infile"
      } else {
         set LINE ""
         if { $BeginFile_set } {
            eval $BeginFile_action
         }
         while { [ GetNextLine ] >= 0 } {
            incr NL
            incr NLT
            set MATCH 0
            set skip_patterns 0
            foreach pattern $PatternActions {
               $pattern
               if { $skip_patterns } {
                  break
               }
            }
            if { $Default_set } {
               eval $Default_action
            }
         }
         set LINE ""
         if { $EndFile_set } {
            eval $EndFile_action
         }
         close $infile
      }
   }
   return
}

#
# Initialise the procedures
#
Content initProc {
   variable State
   unset State
}
Content exportProc {
   variable State
   global LINE
   puts $LINE
}

}
# --------------------------------------------------------------------
#   End of package:   AwkProc
# --------------------------------------------------------------------


# Test procedure
namespace import ::AwkProc::*

if { 0 } {
global pr
global skip

set pr 0
set skip 0

#
# Note the ordering is important with these simple patterns!
#
proc SetKeyword { line } {
   set keyw [Content get keyw]
   set matched [regexp {: *([^ ].*)} $line match text]
   if { $matched } {
      Content set $keyw [string trim $text]
   } else {
      # Just a precaution!
      puts "No match: $line"
   }
}

proc AppendKeyword { line } {
   set keyw [Content get keyw]
   set text [string range $line 1 end]
   Content append $keyw "[string trim $text] "
}

RegPattern {^[ ]*# ENDDOC}  { Content set pr 0
                              Content set skip 1
                              SkipPatterns
                            }
RegPattern {^[ ]*# DOC}     { Content set pr 1
                              Content set skip 1
                              SkipPatterns
                            }
RegPattern {^[ ]*# ------}  { Content set pr [expr 1-[Content get pr]]
                              Content set skip 2
                              SkipPatterns
                            }

RegPattern {proc .*\{$}     { Content set proc 1
                              Content set procedure $LINE
                              SkipPatterns
                            }

Pattern "Purpose: "         { Content set keyw "purpose"
                              SetKeyword $LINE
                              SkipPatterns
                            }

Pattern "Context: "         { Content set keyw "context"
                              SetKeyword $LINE
                              SkipPatterns
                            }

Pattern "Summary:"          { # Note: no trailing space!
                              Content set keyw "summary"
                              # SetKeyword $LINE -- do not use this
                              Content set summary ""
                              SkipPatterns
                            }

Pattern "Arguments:"        { # Note: no trailing space!
                              Content set keyw "arguments"
                              # SetKeyword $LINE -- do not use this
                              Content set arguments ""
                              SkipPatterns
                            }

Pattern "Returns: "         { # Note: no trailing space!
                              Content set keyw "returns"
                              # SetKeyword $LINE -- do not use this
                              Content set returns ""
                              SkipPatterns
                            }

RegPattern {^# }            { # Inside comments before "proc"?
                              set pr [Content get pr]
                              if { $pr == 1 } {
                                 AppendKeyword $LINE
                              }
                              SkipPatterns
                            }

# Pattern "# DOC"     { global pr skip ; set pr 1 ; set skip 1 ; SkipPatterns }
# Pattern "# ---"     { global pr skip ; set pr 1 ; set skip 1 ; SkipPatterns }

BeginFile {
   puts "---"
}
EndFile {
   puts "---"
}

Content initProc {
   Content set keyw      "?"
   Content set summary   "?"
   Content set purpose   "?"
   Content set context   "?"
   Content set arguments "?"
   Content set returns   "?"
   Content set proc      0
   Content set procedure "?"
}

# We need to explicitly initialise the content now
Content init

Content exportProc {
   global LINE
   variable State
   if { $State(pr) && ! $State(skip) } {
      puts $LINE
   }
   if { $State(skip) == 1 } {
      Content set skip 0
   }
   if { $State(proc) } {
      puts "[Content get procedure]"
      puts "   Purpose:   [Content get purpose]"
      puts "   Context:   [Content get context]"
      puts "   Summary:   [Content get summary]"
      puts "   Arguments: [Content get arguments]"
      puts "   Returns:   [Content get returns]"
      Content init
   }
}

DefaultAction {
   Content export
   #Content set skip 0
}

# Now the actual processing: the command-line arguments contain
# file names
#
ProcessFiles awkproc.tcl
}
