# OpenVerse I/O Functions Module
# 
# This module handles all communctions with the server.
# It also handles connections.
# 
#
# Module Name		- I/O Functions Module
# Current Maintainter 	- Cruise <cruise@openverse.org>
# Sourced By		- Main Module
#
# Copyright (C) 1999 David Gale <cruise@openverse.org>
# For more information visit http://OpenVerse.org/
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
# USA.


#
# Sends pre-formatted commands to the server.
#
proc SendToServer {what} {
	global MV

	if {$MV(sock) == -1} {return}
	if $MV(debug) {puts "-> $what"}
	catch {
		puts $MV(sock) $what
		flush $MV(sock)
	} error
}

#
# Reads text from the OpenVerse server.
#
proc ReadFrom {} {
	global MV
	
	set input ""
	catch {gets $MV(sock) input} error
	if {$input == "NAMEINUSE"} {
		ProcChat $MV(nick) "The nick you have chosen is in use. Please choose another. You can use the setup button to do this or type '/nick YourNick' in the chat windows." 0 1 $MV(colors.system.baloon)
		if $MV(debug) {puts "<- Disconnected!"}
		Disconnect
		return
	}
	if {[eof $MV(sock)] == 1} {
		if $MV(debug) {puts "<- Disconnected!"}
		Disconnect
		return;
	}
	ProcessInput $input
}

#
# Process Command input
#
proc ProcessCommand {what} {
	global MV

	set cmd [string range $what 0 [expr [string first " " $what] -1]]
	set rest [string range $what [expr [string first " " $what] +1] end]
	set parms [split $rest " "]
	if {$cmd == ""} {set cmd $what}
	switch -exact -- [string tolower $cmd] {
		"/connect" {
			if {[lindex $parms 0] == "" || [lindex $parms 1] == ""} {
				ProcChat $MV(nick) "You must provide a servername and a port with the /connect command! (EX: /connect this.example.com 7000)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			Disconnect
			set MV(roomhost) [lindex $parms 0]
			set MV(roomport) [lindex $parms 1]
			Connect
		}
		"/aved" {
			AvEd
		}
		"/avatar" {
			if {[lindex $parms 0] == ""} {
				ProcChat $MV(nick) "You must provide an avatar name to user. (EX: /avatar default.av)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			AnimateMe [lindex $parms 0] 3
		}
		"/nick" {
			if {[lindex $parms 0] == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname! (EX: /nick I'm_A_Dork)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			set newnick [CheckThatNick [lindex $parms 0]]
			if {$newnick == ""} {return}
			if $MV(names) {KillName $MV(nick)}
			ChangeNick "$newnick"
			set MV($MV(nick).haschat) 0
			set MV($MV(nick).chatque) {}
			set MV($MV(nick).name_x_offset) $MV(anim.x_off)
			set MV($MV(nick).name_y_offset) $MV(anim.y_off)
			if $MV(names) {ShowName $MV(nick)}
			SendToServer "NICK $MV(nick)"
		}
		"/whois" {
			if {[lindex $parms 0] == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname! (EX: /whois Some_Dork)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			foreach person $MV(people) {
				if {[string tolower $person] == [string tolower [lindex $parms 0]]} {
					SendToServer "WHOIS $person"
				}
			}
		}
		"/msg" {
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			if {[lindex $parms 0] == "" || $rest == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname and message text! (EX: /msg Some_Dork HI!)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			foreach person $MV(people) {
				if {[string tolower $person] == [string tolower [lindex $parms 0]] || [lindex $parms 0] == "*"} {
					SendToServer "PRIVMSG $person $rest"
				}
			}
			ProcChat $MV(nick) "$rest" 0 2 $MV(colors.privmsg.baloon)
		}
		"/sing" {
			if {$rest == ""} {
				ProcChat $MV(nick) "You must provide some text with this command.! (EX: /sing Some_Dork Waaaay down on the swanee river!)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			SendToServer "SCHAT SING $rest"
		}
		"/sub" {
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			if {[lindex $parms 0] == "" || $rest == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname and message text! (EX: /sub Some_Dork COMMAND HI!)" 0 1 $MV(colors.system.baloon)
 				bell
				return
			}
			foreach person $MV(people) {
				if {[string tolower $person] == [string tolower [lindex $parms 0]] || [lindex $parms 0] == "*"} {
					SendToServer "SUB $person $rest"
				}
			}
		}
		"/push" {
			SendToServer "PUSH 100"
		}
		"/effect" {
			if {[lindex $parms 0] == ""} {
				ProcChat $MV(nick) "You must provide an effect type. (EX: /effect shiver)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			SendToServer "EFFECT [lindex $parms 0]"
		}
		"/dccsend" {
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			if {[lindex $parms 0] == "" || $rest == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname and file to send! (EX: /dccsend Some_Dork ~/MyFile.txt)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			foreach person $MV(people) {
				if {[string tolower $person] == [string tolower [lindex $parms 0]] || [lindex $parms 0] == "*"} {
					DCCSend $person [lindex $parms 1]
				}
			}
		}
		"/url" {
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			if {[lindex $parms 0] == "" || $rest == "" || [string first " " $what] == -1} {
				ProcChat $MV(nick) "You must provide a nickname and message text! (EX: /url Some_Dork http://example.com/!)" 0 1 $MV(colors.system.baloon)
				bell
				return
			}
			set sent 0
			foreach person $MV(people) {
				if {[string tolower $person] == [string tolower [lindex $parms 0]] || [lindex $parms 0] == "*"} {
					SendToServer "URL $person $rest"
					set sent 1
				}
			}
			if $sent {
				ProcURL $MV(nick) "$rest"
			}
		}
	}
}

#
# Connect to the current default server.
#
proc Connect {} {
	global MV

	set MV(sock) [socket $MV(roomhost) $MV(roomport)]
	fconfigure $MV(sock) -blocking 0 -buffering line
	set args [fconfigure $MV(sock) -sockname]
	InsertIntoChat 0 "***SYSTEM*** Connecting to $MV(roomhost) $MV(roomport)\n"
	set MV(MyIpAddress) [lindex $args 0]
	fileevent $MV(sock) readable ReadFrom
	SendToServer "AUTH $MV(nick) $MV(x) $MV(y) $MV(avatar) $MV(anim.x_off) $MV(anim.y_off) [file size "$MV(images)/$MV(avatar)"] $MV(anim.baloon_x) $MV(anim.baloon_y)"
	$MV(connect_button) configure -text "Disconnect" -command Disconnect
}

#
# Connect directly to a room.
#
proc ConnectToRoom {host port} {
	global MV

	Disconnect
	set MV(roomhost) $host
	set MV(roomport) $port
	set MV(sock) [socket $MV(roomhost) $MV(roomport)]
	fconfigure $MV(sock) -blocking 0
	set args [fconfigure $MV(sock) -sockname]
	InsertIntoChat 0 "***SYSTEM*** Connecting to $MV(roomhost) $MV(roomport)\n"
	set MV(MyIpAddress) [lindex $args 0]
	fileevent $MV(sock) readable ReadFrom
	SendToServer "AUTH $MV(nick) $MV(x) $MV(y) $MV(avatar) $MV(anim.x_off) $MV(anim.y_off) [file size "$MV(images)/$MV(avatar)"] $MV(anim.baloon_x) $MV(anim.baloon_y)"
	$MV(connect_button) configure -text "Disconnect" -command Disconnect
}

#
# Disconnect from the server.
#
proc Disconnect {} {
	global MV

	if {$MV(sock) == -1} {return}
	InsertIntoChat 0 "***SYSTEM*** Disconnected from $MV(roomhost) $MV(roomport)\n"
	catch {close $MV(sock)}
	set MV(sock) -1
	set MV(MyIpAddress) "0.0.0.0"
	wm title . "*** Not Connected! ***"
	$MV(connect_button) configure -text "Connect" -command Connect
	foreach who $MV(people) {
		unset MV($who.name)
		unset MV($who.avatar)
		unset MV($who.x)
		unset MV($who.y)
		unset MV($who.lastsaid)
		unset MV($who.name_x_offset)
		unset MV($who.name_y_offset)
		unset MV($who.baloon_x)
		unset MV($who.baloon_y)
		unset MV($who.haschat)
		unset MV($who.nomoremove)
		set MV($who.chatque) {}
		set idx [lsearch -exact $MV(people) $who]
		set MV(people) [lreplace $MV(people) $idx $idx]
		.top.c delete $MV($who.icon)
		if $MV(names) {KillName $who}
	}
	DoNames

	#
	# Proces Plugins!
	#
	foreach plugin $MV(plugin.traps.Disconnect) {
		if ![$MV(plugin.traps.Disconnect.$plugin)] {return}
	}
}

#
# Process Server input
#
proc ProcessInput {what} {
	global MV

	if $MV(debug) {puts "<- $what"}
	set cmd [string range $what 0 [expr [string first " " $what] -1]]
	set rest [string range $what [expr [string first " " $what] +1] end]
	set parms [split $rest " "]
	if {$cmd == ""} {set cmd $what}

	#
	# Proces Plugins!
	#
	foreach plugin $MV(plugin.traps.ProcessInput.pre) {
		if ![$MV(plugin.traps.ProcessInput.pre.$plugin) "$what" $cmd $parms $rest] {return}
	}
	switch -exact -- $cmd {
		"AVATAR" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.avatar) [lindex $parms 0]] != -1} {return}
			ChangeUserAvatar [lindex $parms 0] [lindex $parms 1] [lindex $parms 2] [lindex $parms 3] [lindex $parms 4] [lindex $parms 5] [lindex $parms 6]
		}
		"CHAT" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.speech) [lindex $parms 0]] != -1} {return}
			set parms [split $rest " "]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			ProcChat [lindex $parms 0] $rest 0 0 $MV(colors.chat.baloon)
		}
		"SCHAT" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.speech) [lindex $parms 0]] != -1} {return}
			set parms [split $rest " "]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			switch -exact -- [lindex $parms 0] {
				"SING" {
					ProcChat [lindex $parms 1] $rest 0 4 $MV(colors.chat.baloon)
				}
				default {
					ProcChat [lindex $parms 1] $rest 0 0 $MV(colors.chat.baloon)
				}
			}
		}
		"DCCGETAV" {
			if ![SanityCheck [lindex $parms 1]] {return}
			DCCGet $MV(roomhost) [lindex $parms 0] [lindex $parms 1] [lindex $parms 2] AVATAR *
		}
		"DCCGETROOM" {
			if ![SanityCheck [lindex $parms 1]] {return}
			DCCGet $MV(roomhost) [lindex $parms 0] [lindex $parms 1] [lindex $parms 2] ROOM *
		}
		"DCCSENDAV" {
			if ![SanityCheck [lindex $parms 1]] {return}
			DCCSendAv $MV(roomhost) [lindex $parms 0] [lindex $parms 1]
		}
		"EFFECT" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.effect) [lindex $parms 0]] != -1} {return}
			if $MV(honor_effects) {AvatarEffect [lindex $parms 0] [lindex $parms 1]}
		}
		"EXIT" {
			if !$MV(honor_exits) {return}
			if !$MV(downloadingroom) {
				set MV(roomhost) [lindex $parms 0]	
				set MV(roomport) [lindex $parms 1]
				Disconnect
				ProcChat $MV(nick) "Changing Rooms to [lindex $parms 0] [lindex $parms 1]" 0 1 $MV(colors.system.baloon)
				Connect	
			}
		}
		"MOVE" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.move) [lindex $parms 0]] != -1} {return}
			MoveUser [lindex $parms 0] [lindex $parms 1] [lindex $parms 2] [lindex $parms 3]
		}
		"NEW" {
			NewPerson [lindex $parms 0] [lindex $parms 1] [lindex $parms 2] [lindex $parms 3] [lindex $parms 4] [lindex $parms 5] [lindex $parms 6] [lindex $parms 7] [lindex $parms 8]
		}
		"NOMORE" {
			PersonLeft [lindex $parms 0]
		}
		"PING" {
			SendToServer "PONG"
		}
		"PUSH" {
			MoveTo [lindex $parms 0] [lindex $parms 1]
		}
		"PRIVMSG" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.speech) [lindex $parms 0]] != -1} {return}
			set parms [split $rest " "]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			ProcChat [lindex $parms 0] "$rest" 0 2 $MV(colors.privmsg.baloon)
		}
		"ROOM" {
			SetRoom [lindex $parms 0] [lindex $parms 1]
		}
		"ROOMNAME" {
			wm title . $rest
			set MV(roomname) $rest
		}
		"SUB" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.sub) [lindex $parms 0]] != -1} {return}
			set parms [split $rest " "]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			if {[lindex $parms 1] == ""} {return}
			switch -exact -- [lindex $parms 1] {
				"DCCGET" {
					if ![SanityCheck [lindex $parms 4]] {return}
					ProcURL [lindex $parms 0] "FILE://[lindex $parms 2]:[lindex $parms 3]/[lindex $parms 4]:[lindex $parms 5]"
				}
				"PERSONAL" {
					ProcPersonal [lindex $parms 0] "$rest"
				}
			}
		}
		"TOOBIG" {
			AnimateMe $MV(default_avatar_number) 1
			update idletasks
			ProcChat $MV(nick) "The server said 'TOOBIG'! This means that your avatar exceeds the server's limits.." 0 1 $MV(colors.system.baloon)
		}
		"URL" {
			if {[lsearch -exact $MV(ignore.all) [lindex $parms 0]] != -1 ||
				[lsearch -exact $MV(ignore.url) [lindex $parms 0]] != -1} {return}
			set parms [split $rest " "]
			set rest [string range $rest [expr [string first " " $rest] +1] end]
			ProcURL [lindex $parms 0] $rest
		}
		"USERS" {
			set MV(server_users) [lindex $parms 0]
		}
		"WHOIS" {
			WHOISUser [lindex $parms 0] [lindex $parms 1]
		}
	}
	#
	# Proces Plugins!
	#
	foreach plugin $MV(plugin.traps.ProcessInput.post) {
		if ![$MV(plugin.traps.ProcessInput.post.$plugin) "$what" $cmd $parms $rest] {return}
	}
}

proc SanityCheck {what} {
        if {[string first "../" $what] != -1} { return 0} else { return 1}
}

proc DownloadWindow {file} {
	global MV
	
	if {[string length $file] > 16} {
		set file [string range $file 0 15]
	}

	# find an empty one.
	set row -1
	set col -1
	for {set c 0} {$c < 26 && $row < 0} {incr c} {
		for {set s 1} {$s < 4 && $col < 0} {incr s} {
			if !$MV(dls.$c.$s) {
				set row $c
				set col $s
			}
		}
	}
	# Make it not empty :)
	set y [expr $row * 18 + 2]
	set xx [expr $col * 213]
	set MV(dls.$row.$col) 1
	set MV(dlbox.$row.$col) [.top.c create rectangle [expr $xx - 213] [expr $y +1] $xx [expr $y + 18 + 2] -fill black -outline white]
	set MV(dltxt.$row.$col) [.top.c create text [expr $xx - 140] [expr $y + 10] -text "$file" -fill white]
	set MV(dlstsbx.$row.$col) [.top.c create rectangle [expr $xx -68] [expr $y + 4] [expr $xx - 2] [expr $y + 16] -outline blue]
	set MV(dlsts.$row.$col) [.top.c create rectangle [expr $xx -66] [expr $y + 5] [expr $xx -66] [expr $y + 14] -fill red]
	set MV(dlperc.$row.$col) 0
	set MV(dlcps.$row.$col) 0.00

	#
	# Set the bounding box so a plugin can use this later.
	#
	set MV(dlbounds.$row.$col) [list [expr $xx - 213] [expr $y +1] $xx [expr $y + 18 + 2]]
	return "$row $col"
}

proc ShowDownloadStatus {row col} {
	global MV tl

	set y [expr $row * 18 + 2]
	set xx [expr $col * 213]
	set rightx [expr ($MV(dlperc.$row.$col) * 62) / 100]
	set leftx [expr $xx -66]

	.top.c delete $MV(dlsts.$row.$col)
	set MV(dlsts.$row.$col) [.top.c create rectangle $leftx [expr $y + 5] [expr $leftx + $rightx] [expr $y + 14] -fill red]
	if [winfo exists .top.c.dlcanc] {
		if $MV(dlactive.$row.$col) {
			set idx $MV(dlidx.$row.$col)
			.top.c.dlcanc.info2.inf configure -text "$tl($MV(DCC.$idx.sock))"
			.top.c.dlcanc.info3.inf configure -text "$MV(dlcps.$row.$col) Kcps"
		}
	}
}

proc KillDlwin {row col} {
	global MV

	.top.c delete $MV(dlbox.$row.$col)
	.top.c delete $MV(dltxt.$row.$col)
	.top.c delete $MV(dlstsbx.$row.$col)
	.top.c delete $MV(dlsts.$row.$col)
	set MV(dls.$row.$col) 0
}

### -- New download code.

# DCC GET CODE
#
# This has been styled using DCC code from Zircon an ircII client
# by Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
#
# This is our central location for DOWNLOADING files.
# 
proc DCCGet {host port file size type sender} {
	global MV

	if ![SanityCheck "$file"] {
		if $MV(debug) {puts "(!) $file fails SanityCheck"}
		return
	}


	#
	# make sure we are not already getting it.
	# Send a reject if we are!
	# TODO
	#

	lappend MV(downloads) "$file"

	switch -exact -- $type {
		"AVATAR" {set dldir "$MV(rem_images)"}
		"ROOM" {
			set dldir "$MV(roomdir)"
			set MV(downloadingroom) 1
		}
		"OBJECT" {set dldir "$MV(objects)"}
		default {set dldir "$MV(download_dir)"}
	}

	set sock [socket -async $host $port]
	fconfigure $sock -blocking 1

	set idx [incr MV(dcc_num)]	
	set MV(DCC.$idx.sender) $sender
	set MV(DCC.$idx.file) "$dldir/$file"
	set MV(DCC.$idx.size) $size
	set MV(DCC.$idx.type) $type
	set MV(DCC.$idx.posn) 0
	set MV(DCC.$idx.server) -1
	set MV(DCC.$idx.sock) $sock
	set MV(DCC.$idx.time) [clock seconds]
	set rc [DownloadWindow [file tail $MV(DCC.$idx.file)]]
	set row [string range $rc 0 [expr [string first " " $rc] -1]]
	set col [string range $rc [expr [string first " " $rc] +1] end]
	set MV(DCC.$idx.row) $row
	set MV(DCC.$idx.col) $col
	set MV(dlidx.$row.$col) $idx
	lappend MV(dcc_list) $idx
	fileevent $sock writable "startGet $idx"
}

proc startGet {index} {
    global MV

    set file $MV(DCC.$index.file)
    set posn $MV(DCC.$index.posn)
    fileevent $MV(DCC.$index.sock) writable {}
    fconfigure $MV(DCC.$index.sock) -buffering none -blocking 0 -translation binary -buffersize 4096
    set flags [list WRONLY CREAT]
    if {$posn == 0} { lappend flags TRUNC }
    if {![catch {open $file $flags 0600} outfile]} {
	if {$posn != 0} {
	    if {[catch {seek $outfile $posn start} msg]} {
	    	close $outfile
	    	endDCC Get $index 0 "Cannot seek on $file : $msg"
		return 0
	    }
		incr MV(DCC.$index.size) -$posn
	}
	uplevel #0 set tl($MV(DCC.$index.sock)) 0
	fconfigure $outfile -translation binary 
	fileevent $MV(DCC.$index.sock) readable "dccgevent $index [clock seconds] $outfile"
    } {
	endDCC Get $index 0 "Cannot write $file : $outfile"
        return 0
    }
    return 1
}

proc dccgevent {index st out} {
    global tl MV

    set xc 0

	set in $MV(DCC.$index.sock)
	set leng $MV(DCC.$index.size)
	uplevel #0 set MV(DCC.$index.time) [clock seconds]
	set fail_type 0

    if {[eof $in]} {
	if $MV(debug) {puts "(!) Dcc Ended, EOF detected."}
        if {$tl($in) < $leng} {
	    set msg "Transfer Interrupted"
		set fail_type 0
        } elseif {$tl($in) > $leng} {
   	    set msg "Too much data transferred!!"
		set fail_type 0
        } else {
    	    set sx s
	    if {[set st [expr {[clock seconds] - $st}]] == 0} {
	        set st 1
	        set sx {}
	    }
	    set xc 1
	    set msg "Transfer completed. [expr {$leng / ($st * 1024.0)}] Kbytes/sec"
		set fail_type 1
        }
    } {
        if {![catch {set buffer [read $in]} msg]} {
            incr tl($in) [set l [string length $buffer]]
		if $MV(debug) {puts "downloaded $l bytes ($tl($in) total) wrote to $out $MV(DCC.$index.file)"}
            if {[set dt [expr {[clock seconds] - $st}]] == 0 || $tl($in) == 0} {
                set elt 0
            } {
	        set elt [expr {($leng - $tl($in)) / ($tl($in) /([clock seconds] - $st))}]
	    }
	    if {$leng == 0} {
	    	set xt 0
	    } {
	        set xt [expr {($tl($in) * 100.0) / $leng}]
	    }
		
		set row $MV(DCC.$index.row)
		set col $MV(DCC.$index.col)
		if {$tl($in) != 0 && $MV(DCC.$index.size) != 0} {
			uplevel #0 set MV(dlperc.$row.$col) [expr ($tl($in) * 100) / $MV(DCC.$index.size)]
			if [expr [clock seconds] - $st] {
				uplevel #0 set MV(dlcps.$row.$col) [format "%0.2f" [expr ($tl($in) / ([clock seconds] - $st)) / 1024.00]]
			}
			ShowDownloadStatus $row $col
		}

            if {![catch {puts -nonewline $out $buffer} msg]} {
		catch {flush $out}
		if $l {
		        if {![catch {puts -nonewline $in [binary format I1 $tl($in)]} msg]} {
				flush $in
		            return
		        }
		}
		return
	    } else {
		set msg "Unable to report received size"
		set fail_type 0
		if $MV(debug) {puts "(!) Dcc Ended, Error Condition."}
	    }
	} else {
		# 0000 bytes triggered a read event?
		#
		return
	   #set msg "Unable to read from remote connection"
	   #set fail_type 0
	   #if $MV(debug) {puts "(!) Dcc Ended, Error Condition."}
	}
    }
    if $MV(debug) {puts "closing $out $MV(DCC.$index.file)"}
    catch {close $out} 
    endDCC Get $index $fail_type $msg
}

# DCC Send CODE
#
# This has been styled using DCC code from Zircon an ircII client
# by Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
#
# This is our central location for sending AVATAR files.
# It is a passive send.

proc DCCSendAv {host port what} {
	global MV

	set file "$MV(images)/$what"

	if {[file exists $file]} {
		if {![file readable $file]} {
            		if $MV(debug) {puts "($who) (DCCSend) Cannot read file $file."}
			return
		}

		set size [file size $file]
		set idx [incr MV(dcc_num)]
		set sock [socket -async $host $port]
		fconfigure $sock -blocking 1

		set MV(DCC.$idx.sender) "*"
		set MV(DCC.$idx.file) "$file"
		set MV(DCC.$idx.size) $size
		set MV(DCC.$idx.posn) 0
		set MV(DCC.$idx.time) [clock seconds]
		set MV(DCC.$idx.server) -1
		set MV(DCC.$idx.sock) $sock
		set MV(DCC.$idx.port) $port
		set MV(DCC.$idx.time) [clock seconds]
		set rc [DownloadWindow $MV(DCC.$idx.file)]
		set row [string range $rc 0 [expr [string first " " $rc] -1]]
		set col [string range $rc [expr [string first " " $rc] +1] end]
		set MV(dlidx.$row.$col) $idx
		set MV(DCC.$idx.row) $row
		set MV(DCC.$idx.col) $col
		lappend MV(dcc_list) $idx
		fileevent $sock writable "acceptAVSend $idx"
	} else {
		if $MV(debug) {puts "($who) (DCCSendAv) File $file does not exist."}
	}
	update idletasks
}

proc acceptAVSend {index} {
	global MV

	fileevent $MV(DCC.$index.sock) writable {}
	set chan $MV(DCC.$index.sock)
	
	if {[ catch {open $MV(DCC.$index.file) RDONLY} infile]} {
		endDCC Send $index 0 "Cannot read $MV(DCC.$index.file) : $infile"
		return 0
	}

	if {[set posn $MV(DCC.$index.posn)] != {} && $posn > 0} {
		if {[catch {seek $infile $posn start} msg]} {
			endDCC Send $index 0 "Cannot seek $MV(DCC.$index.file) : $msg"
			close $infile
			return 0
		}
		incr MV(DCC.$index.size) -$posn
	} 

	if {$MV(DCC.$index.size) == 0} {
		close $infile
		endDCC Send $index 1 "Transfer completed."
		return 1
	}

	set st [clock seconds]
	fconfigure $infile -translation binary

	if {[catch {set buffer [read $infile $MV(sendbuffer)]} msg]} {
		endDCC Send $index 0 "Error reading $file : $msg"
		close $infile
		return 0
	}

	global tl
	set tl($chan) [string length $buffer]
	fconfigure $chan -blocking 0 -buffering none -translation binary
	if {[catch {puts -nonewline $chan $buffer} msg]} {
		endDCC Send $index 0 "Write error : $msg"
		close $infile
		return 0
	}
	flush $chan
	if $MV(debug) {puts "($MV(DCC.$index.sender)) -- Accepted DCCSend"}
	fileevent $chan readable "dccSendEvent $index $st $infile"
}

# DCC Send CODE
#
# This has been styled using DCC code from Zircon an ircII client
# by Lindsay Marshall <lindsay.marshall@newcastle.ac.uk>
#
# This is our central location for SENDING ALL files.

proc DCCSend {who what} {
	global MV

	#if ![VerifyAvailable $who $what] {
		#puts "($who) -- Already Getting $what"
		#return
	#}

	set file "$what"
	set what "[file tail $file]"

	if {[file exists $file]} {
		if {![file readable $file]} {
            		if $MV(debug) {puts "($who) (DCCSend) Cannot read file $file."}
			return
		}

		set size [file size $file]
		set idx [incr MV(dcc_num)]
		set sock [socket -server "acceptSend $idx" 0]

		if {[catch {fconfigure $sock -sockname} port]} {
			if $MV(debug) {puts "($who) (DCCSend) Cannot get port for server - $port"}
		}

		set MV(DCC.$idx.sender) $who
		set MV(DCC.$idx.file) "$file"
		set MV(DCC.$idx.size) $size
		set MV(DCC.$idx.posn) 0
		set MV(DCC.$idx.time) [clock seconds]
		set MV(DCC.$idx.server) $sock
		set MV(DCC.$idx.sock) -1
		set MV(DCC.$idx.port) [lindex $port 2]
		set MV(DCC.$idx.time) [clock seconds]
		set rc [DownloadWindow $MV(DCC.$idx.file)]
		set row [string range $rc 0 [expr [string first " " $rc] -1]]
		set col [string range $rc [expr [string first " " $rc] +1] end]
		set MV(dlidx.$row.$col) $idx
		set MV(DCC.$idx.row) $row
		set MV(DCC.$idx.col) $col
		lappend MV(dcc_list) $idx
	
		SendToServer "SUB $who DCCGET $MV(MyIpAddress) [lindex $port 2] $what $size"
	} else {
		if $MV(debug) {puts "($who) (DCCSend) File $file does not exist."}
	}
}

proc acceptSend {index chan hst port} {
	global MV

	catch {close $MV(DCC.$index.server)}
	set MV(DCC.$index.server) -1
	uplevel #0 set MV(DCC.$index.sock) $chan
	
	if {[ catch {open $MV(DCC.$index.file) RDONLY} infile]} {
		endDCC Send $index 0 "Cannot read $MV(DCC.$index.file) : $infile"
		return 0
	}

	if {[set posn $MV(DCC.$index.posn)] != {} && $posn > 0} {
		if {[catch {seek $infile $posn start} msg]} {
			endDCC Send $index 0 "Cannot seek $MV(DCC.$index.file) : $msg"
			close $infile
			return 0
		}
		incr MV(DCC.$index.size) -$posn
	} 

	if {$MV(DCC.$index.size) == 0} {
		close $infile
		endDCC Send $index 1 "Transfer completed."
		return 1
	}

	set st [clock seconds]
	fconfigure $infile -translation binary

	if {[catch {set buffer [read $infile $MV(sendbuffer)]} msg]} {
		endDCC Send $index 0 "Error reading $file : $msg"
		close $infile
		return 0
	}

	global tl
	set tl($chan) [string length $buffer]
	fconfigure $chan -blocking 0 -buffering none -translation binary
	if {[catch {puts -nonewline $chan $buffer} msg]} {
		endDCC Send $index 0 "Write error : $msg"
		close $infile
		return 0
	}
	flush $chan
	fileevent $chan readable "dccSendEvent $index $st $infile"
}

proc dccSendEvent {index st fd} {
	global MV
	
	set sk $MV(DCC.$index.sock)
	uplevel #0 set MV(DCC.$index.time) [clock seconds]

	if {[eof $sk]} {
		endDCC Send $index 0 "Transfer interrupted"
		close $fd
		return
	}

	if {[catch {set l [read $sk 4]} msg]} {
		endDCC Send $index 0 "Read error : $msg"
		close $fd
		return
	}

	if {[string length $l] == 0} {
		endDCC Send $index 0 "Sync read error"
		close $fd
		return
	}

        global tl
	set cl 0
	binary scan $l I1 cl
	if {$cl != $tl($sk)} {return }

	if [eof $fd] {
		if {[set st [expr {[clock seconds] - $st}]] == 0} {
			    set st 1
		}
		close $fd
		endDCC Send $index 1 "Transfer completed"
		return
	}

	if {[catch {set buffer [read $fd $MV(sendbuffer)]} msg]} {
		endDCC Send $index 0 "Error reading $MV(DCC.$index.file) : $msg"
		close $fd
		return
	}

	if {[set lng [string length $buffer]] == 0} {
		if {[set st [expr {[clock seconds] - $st}]] == 0} {
			set st 1
		}
		close $fd
		endDCC Send $index 1 "Transfer completed."
		return
	}
	incr tl($sk) $lng
	if {[catch {puts -nonewline $sk $buffer} msg]} {
		endDCC Send $index 0 "Write error : $msg"
		close $fd
		return
	}
	flush $sk
		set row $MV(DCC.$index.row)
		set col $MV(DCC.$index.col)
	if {$tl($sk) != 0 && $MV(DCC.$index.size) != 0} {
		uplevel #0 set MV(dlperc.$row.$col) [expr ($tl($sk) * 100) / $MV(DCC.$index.size)]
		if [expr [clock seconds] - $st] {
			uplevel #0 set MV(dlcps.$row.$col) [format "%0.2f" [expr ($tl($sk) / ([clock seconds] - $st)) / 1024.00]]
		}
		ShowDownloadStatus $row $col
	}
	if {[set dt [expr {[clock seconds] - $st}]] == 0} {
		set elt 0
	} {
		set elt [expr {($MV(DCC.$index.size) - $tl($sk)) / ($tl($sk) /([clock seconds] - $st))}]
	}
	update idletasks
}

proc endDCC {type index fail_type debug} {
	global MV

	set idx [lsearch -exact $MV(downloads) [file tail $MV(DCC.$index.file)]]
	if {$index >= 0} {set MV(downloads) [lreplace $MV(downloads) $idx $idx]}
	if $MV(debug) {puts "(!) (DCC$type) - $debug $MV(DCC.$index.file)"}
	catch {close $MV(DCC.$index.sock)}
	if {$MV(DCC.$index.server) > 0} {
		catch {close $MV(DCC.$index.server)}
	}
	if {$type == "Get"} {
		switch -exact -- $MV(DCC.$index.type) {
			"AVATAR" {
				if $fail_type {
					foreach who $MV(people) {
						if {$MV($who.avatar) == [file tail $MV(DCC.$index.file)] && $MV($who.downloading) == 1} {
							catch {image create photo $who -file "$MV(rem_images)/$MV($who.avatar)"} err
							if {$err != $who} {
								catch {image create photo $who -file "$MV(images)/default.gif"} err
							}
							set MV($who.downloading) 0
						}
					}
				} else {
					SendToServer "DCCSENDAV [file tail $MV(DCC.$index.file)]"
					set MV($who.downloading) 1
				}
			}
			"ROOM" {
				if $fail_type {
					set MV(downloadingroom) 0
					catch {image create photo room -file "$MV(roomdir)/[file tail $MV(DCC.$index.file)]"}
				} else {
					SendToServer "DCCSENDROOM [file tail $MV(DCC.$index.file)]"
				}
			}
		}
	}
	KillDlwin $MV(DCC.$index.row) $MV(DCC.$index.col)
	if [winfo exists .top.c.dlcanc] {KillDlInfoWin $MV(DCC.$index.row) $MV(DCC.$index.col)}
	set idx [lsearch -exact $MV(dcc_list) $index]
	set MV(dcc_list) [lreplace $MV(dcc_list) $idx $idx]
}

proc CheckTimeouts {} {
	global MV

	set tme [clock seconds]
	foreach idx $MV(dcc_list) {
		if {[expr $tme - $MV(DCC.$idx.time)] > $MV(dcctimeout)} {
			if {$MV(DCC.$idx.server) > 0} {
				catch {close $MV(DCC.$idx.server)}
			}
			endDCC Timer $idx 0 "Connection Timed Out $MV(DCC.$idx.file)"
		}
	}
	after 5000 CheckTimeouts
}

proc KillDlInfoWin {row col} {
	global MV

	set MV(dlactive.$row.$col) 0
	destroy .top.c.dlcanc
}

proc DownloadClick {row col x y} {
	global MV tl


	if [winfo exists .top.c.dlcanc] {KillDlInfoWin $row $col}

        set where n     
        if {$row < 13} {set where n}
        if {$row > 13} {set where s}
        if {$row == 13} {set where s}
	if {$col == 1} {append where "w"}
	if {$col == 3} {append where "e"}

	frame .top.c.dlcanc -relief raised -borderwidth 2 \
		-bg $MV(colors.dl.frames.bg)

	.top.c create window $x $y -anchor $where -window .top.c.dlcanc
	set idx $MV(dlidx.$row.$col)
	set MV(dlactive.$row.$col) 1

	frame .top.c.dlcanc.info0 -relief sunken -borderwidth 2 \
		-bg $MV(colors.dl.frames.bg)
	frame .top.c.dlcanc.info1 -relief sunken -borderwidth 2 \
		-bg $MV(colors.dl.frames.bg)
	frame .top.c.dlcanc.info2 -relief sunken -borderwidth 2 \
		-bg $MV(colors.dl.frames.bg)
	frame .top.c.dlcanc.info3 -relief sunken -borderwidth 2 \
		-bg $MV(colors.dl.frames.bg)

	label .top.c.dlcanc.info0.txt -relief raised -borderwidth 2 -width 15 \
		-text "For / From" \
		-fg $MV(colors.dl.labels.fg) \
		-bg $MV(colors.dl.labels.bg)
	label .top.c.dlcanc.info1.txt -relief raised -borderwidth 2 -width 15 \
		-text "File" \
		-fg $MV(colors.dl.labels.fg) \
		-bg $MV(colors.dl.labels.bg)
	label .top.c.dlcanc.info2.txt -relief raised -borderwidth 2 -width 15 \
		-text "Bytes Trans" \
		-fg $MV(colors.dl.labels.fg) \
		-bg $MV(colors.dl.labels.bg)
	label .top.c.dlcanc.info3.txt -relief raised -borderwidth 2 -width 15 \
		-text "CPS" \
		-fg $MV(colors.dl.labels.fg) \
		-bg $MV(colors.dl.labels.bg)

	label .top.c.dlcanc.info0.inf -relief sunken -borderwidth 2 \
		-text "$MV(DCC.$idx.sender)" \
		-fg $MV(colors.dl.entries.fg) \
		-bg $MV(colors.dl.entries.bg)
	label .top.c.dlcanc.info1.inf -relief sunken -borderwidth 2 \
		-text "$MV(DCC.$idx.file)" \
		-fg $MV(colors.dl.entries.fg) \
		-bg $MV(colors.dl.entries.bg)
	label .top.c.dlcanc.info2.inf -relief sunken -borderwidth 2 \
		-text "$tl($MV(DCC.$idx.sock))" \
		-fg $MV(colors.dl.entries.fg) \
		-bg $MV(colors.dl.entries.bg)
	label .top.c.dlcanc.info3.inf -relief sunken -borderwidth 2 \
		-text "$MV(dlcps.$row.$col) Kbps" \
		-fg $MV(colors.dl.entries.fg) \
		-bg $MV(colors.dl.entries.bg)

	button .top.c.dlcanc.can -text "End Download" \
		-command "endDCC Cancel $idx 0 \"Canceled at user's Request\"" \
		-fg $MV(colors.dl.buttons.fg) \
		-bg $MV(colors.dl.buttons.bg) \
		-activeforeground $MV(colors.dl.buttons.afg) \
		-activebackground $MV(colors.dl.buttons.abg)
	button .top.c.dlcanc.nev -text "Nevermind" \
		-command {destroy .top.c.dlcanc} \
		-fg $MV(colors.dl.buttons.fg) \
		-bg $MV(colors.dl.buttons.bg) \
		-activeforeground $MV(colors.dl.buttons.afg) \
		-activebackground $MV(colors.dl.buttons.abg)

	pack .top.c.dlcanc.info0 -fill both -expand y
	pack .top.c.dlcanc.info1 -fill both -expand y
	pack .top.c.dlcanc.info2 -fill both -expand y
	pack .top.c.dlcanc.info3 -fill both -expand y

	pack .top.c.dlcanc.info0.txt -side left
	pack .top.c.dlcanc.info1.txt -side left
	pack .top.c.dlcanc.info2.txt -side left
	pack .top.c.dlcanc.info3.txt -side left

	pack .top.c.dlcanc.info0.inf -side left -fill both -expand y
	pack .top.c.dlcanc.info1.inf -side left -fill both -expand y
	pack .top.c.dlcanc.info2.inf -side left -fill both -expand y
	pack .top.c.dlcanc.info3.inf -side left -fill both -expand y

	pack .top.c.dlcanc.nev .top.c.dlcanc.can -side left -fill both -expand y

}

#
# We need to register as a plugin because
# there seems to be a limit on the number of bound items
# on the canvas or it's buggy or something. This works.
#
proc MV_DlWindows {x y} {
	global MV

	foreach dcc $MV(dcc_list) {
		set r $MV(DCC.$dcc.row)
		set c $MV(DCC.$dcc.col)
		if {$x >= [lindex $MV(dlbounds.$r.$c) 0] && \
			$x <= [lindex $MV(dlbounds.$r.$c) 2] && \
			$y >= [lindex $MV(dlbounds.$r.$c) 1] && \
			$y <= [lindex $MV(dlbounds.$r.$c) 3]} {
			 DownloadClick $r $c $x $y
			return 0
		}
	}
	return 1
}
