#  PreferencesUtils.tcl ---
#  
#  Part of the Whiteboard application. 
#  It defines some utilities for keeping the user preferences.
#   
#  The whiteboard application has been developed by:
#  
#	Mats Bengtsson, 
#	Hamngatan 21,
#	SE-58226, Linkoping Sweden,
#	matben@lin.foa.se,
#	phone: +46 13 136114
# 
#  It is distributed under the standard GPL.
#  See the README file for further details.
#  
#------------------------------------------------------------------------

proc PreferencesInit  {  }  {
    global  prefs
    
    if {[file exists $prefs(userPrefsFilePath)]}  {
	if {[catch {option readfile $prefs(userPrefsFilePath)} err]}  {
	    tk_messageBox -type ok -message  \
	      "Error reading preference file: $prefs(userPrefsFilePath)."
	}
    }
}

#   PreferencesAdd ---
#
#   'thePrefs': a list of lists where each sublist defines an item in the
#   following way:  {theVarName itsResourceName itsHardCodedDefaultValue}.

proc PreferencesAdd  { thePrefs }  {
    global  prefs dims shortcuts shortsMulticastQT
    
    append prefs(master) $thePrefs " "
    foreach item $thePrefs {
	set varName [lindex $item 0]
	set resourceName [lindex $item 1]
	set defaultValue [lindex $item 2]
	set value [PreferencesGetValue $varName $resourceName $defaultValue]
	set $varName $value
	#puts "PreferencesAdd:: varName=$varName, resourceName=$resourceName,\
	#  defaultValue=$defaultValue, value=$value"
    }   
}

proc PreferencesGetValue  { varName resourceName defValue }  {
    upvar #0 varName theVar
    
    set theVar [option get . $resourceName {}]
    #puts "PreferencesGetValue:: theVar=$theVar"
    # If not there {} then take the itsHardCodedDefaultValue.
    if {$theVar == {}}  {
	set theVar $defValue
    }
    return $theVar
}
    
proc PreferencesSaveToFile  {  }  {
    global  prefs dims tcl_platform shortcuts shortsMulticastQT

    # Work on a temporary file and switch later.
    set tmpFile $prefs(userPrefsFilePath).tmp
    if {[catch {open $tmpFile w} fid]}  {
	tk_messageBox -type ok -message "Error opening file $tmpFile."
	return
    }
    # Header information.
    puts $fid "!\n!   User preferences for the Whiteboard application."
    puts $fid "!   The date written at: [clock format [clock seconds]]\n!"
    # Only preferences indicated in the master copy are saved.
    foreach item $prefs(master) {
	set varName [lindex $item 0]
	set resourceName [lindex $item 1]
	# Need to make a double dereferencing.
	upvar 0 $varName var
	set value $var
	#puts "varName=$varName, resourceName=$resourceName, value=$value"
	puts $fid [format "%-24s\t%s" *${resourceName}: $value]
	
    }
    close $fid
    if {[catch {file rename -force $tmpFile $prefs(userPrefsFilePath)} msg]}  {
	tk_messageBox -type ok -message "Error renaming preferences file."
	return
    }
    if {$tcl_platform(platform) == "macintosh"}  {
	file attributes $prefs(userPrefsFilePath) -type pref
    }
}

#   PreferencesResetToAppDefaults ---
#
#   Resets the preferences in 'prefs(master)' to their hardcoded values.

proc PreferencesResetToAppDefaults  {  }  {
    global  prefs dims shortcuts shortsMulticastQT
    
    foreach item $prefs(master) {
	set varName [lindex $item 0]
	set defaultValue [lindex $item 2]
	set $varName $defaultValue
    }
}

proc PreferencesResetToUserDefaults  {  }  {
    global  prefs dims shortcuts shortsMulticastQT
    
    # Need to make a temporary storage in order not to duplicate items.
    set thePrefs $prefs(master)
    set prefs(master) {}
    # Read the user option database file once again.
    PreferencesInit
    PreferencesAdd $thePrefs
}

#---------------------------------------------------------------------
