#!/bin/sh
# the next line restarts using wish \
	exec wish "$0" -visual best "$@"
      
#  Whiteboard091.tcl ---
#  
#      This file is the main of the whiteboard application. It depends on
#      a number of other files. The 'lib' directory contains the other tcl
#      code that gets sourced here. The 'images' directory contains icons
#      and other images needed by this script. The 'items' directory
#      contains a library of canvas items that are accesable directly from
#      a menu.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

#--Descriptions of some central variables and their usage-----------------------
#            
#  The ip number is central to all administration of connections.
#  Each connection has a unique ip number from which all other necessary
#  variables are looked up using arrays:
#  
#  ipNum2Name:       maps ip number to the specific domain name.
#  
#  ipName2Num:       inverse of above.
#  
#  ipNum2Socket:     maps ip number to the specific socket that is used for 
#                    sending canvas commands and other commands. It is the 
#                    socket opened by the client, except in the case this is 
#                    the central server in a centralized network.
#                    
#  ipNum2ServSocket: maps ip number to the server side socket opened from
#                    a remote client.
#                    
#  ipNum2ServPort:   maps ip number to the specific remote server port number. 
#  
#  ipNum2nEnt:       maps ip number to the entry line in the connect panel from
#                    top (0) and down.
#                    
#  ipNum2User:       maps ip number to the user name.
#
#  To keep track of where images/movies exist at this clients disk
#  we have two synced lists 'knownFiles' and 'knownPaths' where an element
#  in 'knownFiles' which contains a files tail corresponds precisely
#  to this files full path in 'knownPaths'.
#  Also we have an array 'imageItno2FileTail' which gives the tail of 
#  the file from the items 'itpref/itno'.

#  Make sure that we are in the directory of the application itself.
set thisPath [info script]
if {$thisPath != ""}  {
    cd [file dirname $thisPath]
}
set thisPath [pwd]
set launchSecs [clock seconds]

# Level of detail for printouts. >= 2 for my outputs.
set debugLevel 0
# Level of detail for printouts for server. >= 2 for my outputs.
set debugServerLevel 0
# Macintosh only: if no debug printouts, no console. Also for windows?
if {([string compare $tcl_platform(platform) "macintosh"] == 0) &&   \
  $debugLevel == 0 && $debugServerLevel == 0}  {
    console hide
}

# Hide the main window during launch.
wm withdraw .

# Fonts needed in the splash screen and elsewhere.
if {[string compare $tcl_platform(platform) "unix"] == 0}  {
    set sysFont(s) {Helvetica 10 normal}
    set sysFont(sb) {Helvetica 10 bold}
    set sysFont(m) $sysFont(s)
} elseif {[string compare $tcl_platform(platform) "macintosh"] == 0}  {
    set sysFont(s) {Geneva 9 normal}
    set sysFont(sb) {Geneva 9 bold}
    set sysFont(m) application
} elseif {[string compare $tcl_platform(platform) "windows"] == 0}  {
    set sysFont(s) {Arial 7 normal}
    set sysFont(sb) {Arial 7 bold}
    set sysFont(m) $sysFont(s)
}

# Total screen dimension; needed in splash.
set dims(screenH) [winfo vrootheight .]
set dims(screenW) [winfo vrootwidth .]

# The splash screen is needed from the start.
set wDlgs(splash) .splash
if {[catch {source [file join lib SplashScreen.tcl]} msg]}  {
    tk_messageBox -message "Error:  $msg" -icon error -type ok
    exit
}
# Show it!
::SplashScreen::SplashScreen $wDlgs(splash)
update idletasks

# Need to set a trace on variable containing the splash start message.
proc TraceStartMessage { varName junk op }  {
    global  wDlgs
    
    # Need catch here if splash not shown. Update needed to force display (bad?).
    catch {${wDlgs(splash)}.can itemconfigure tsplash  \
      -text $::SplashScreen::startMsg}
    update idletasks
}
trace variable ::SplashScreen::startMsg w TraceStartMessage
set ::SplashScreen::startMsg "Sourcing tcl code..."

# These are auxilary procedures that we need.
foreach sourceName {  \
  SomeUtilities.tcl      \
  Base64Icons.tcl        \
  OpenConnection.tcl     \
  ProgressWindow.tcl     \
  Dialogs.tcl            \
  PutFile.tcl            \
  GetFile.tcl            \
  FilesAndCanvas.tcl     \
  EditShortcuts.tcl      \
  OpenMulticast.tcl      \
  PreferencesUtils.tcl   \
  ItemInspector.tcl      \
  EditFontFamilies.tcl   \
  TheServer.tcl          \
  SequenceGrabber.tcl}  {
    if {[catch {source [file join lib $sourceName]} msg]}  {
	after idle {tk_messageBox -message "Error sourcing $sourceName:  $msg"  \
	  -icon error -type ok; exit}
    }    
}

# Wants the progress bar of Alexander Schoepe. If not we make our own.
# Perhaps we should have this as a package instead.
if {[catch {source [file join lib progressbar.tcl]} msg]}  {
    set prefs(Progressbar) 0
} else {
    set prefs(Progressbar) 1
}

# The http package can be useful if using QuickTime.
if {![catch {package require http} msg]}  {
    set prefs(http) 1
} else {
    set prefs(http) 0
}

# Import routines that get exported from various namespaces in the lib files.
# This is a bit inconsistent since I sometimes use import namespace and 
# sometimes the fully qulified name; need to sort out this later.
# Note that the lib routines above need fully qualified names!

namespace import ::GetFile::*
namespace import ::PutFile::*
namespace import ::OpenConnection::*

# Define MIME types etc., and get packages.

if {[catch {source [file join lib MimeTypesAndPlugins.tcl]} msg]}  {
    tk_messageBox -message "Error sourcing MimeTypesAndPlugins.tcl  $msg"  \
      -icon error -type ok
    exit
}    


set ::SplashScreen::startMsg "Initialize..."

# ip numbers, port numbers, and names.
set thisServPort 8235
set remoteServPort 8235
set internalIPnum 127.0.0.1
set internalIPname "localhost"
# Set our IP number temporarily.
set thisIPnum $internalIPnum 

# Find user name.
if {[info exists env(USER)]}  {
    set thisUserName $env(USER)
} elseif {[info exists env(LOGIN)]}  {
    set thisUserName $env(LOGIN)
} elseif {[info exists env(USERNAME)]}  {
    set thisUserName $env(USERNAME)
} else {
    set thisUserName "Unknown"
}

# Keep lists of ip numbers for connected clients and servers.
# 'allIPnums' contains all ip nums that are either connected to, or from.
# It is the union of 'allIPnumsTo' and 'allIPnumsFrom'.
set allIPnums {}
# 'allIPnumsTo' contains all ip nums that we have made a client side connect to.
set allIPnumsTo {}
# 'allIPnumsFrom' contains all ip nums that are connected to our server.
set allIPnumsFrom {}
# 'allIPnumsToSend' is identical to 'allIPnumsTo' except when this is
# the server in a centralized network because then we does not make
# any connections, but all connections are connected 'from'.
set allIPnumsToSend {}

# Keep synced lists of all known file names (tails) and their corresponding
# paths. The position of a pair name-path in each list must be identical!
set knownFiles {}
set knownPaths {}
    
# Other inits.
# Status message shown in the status line.
set statMess {This line contains the status message!}
# Main canvas widget path.
set wCan .fmain.can
set servCan $wCan
set specServCan .serv.ca
set btFrame .fmain.frleft.frbt
set commFrame .fcomm.ent
set wStatMess .fcomm.stat.lbl
# Toplevel for progress window, only the prefix actually.
set wProgWin .pr
# Toplevel dialogs
set wDlgs(openConn) .opc
set wDlgs(infoServ) .infoserv
set wDlgs(infoClient) .infocli
set wDlgs(print) .prt
set wDlgs(editShorts) .tshcts
set wDlgs(editFonts) .edfnt
# Canvas size; these are also min sizes. Add new line of tools.
set dims(wCanOri) 350
set dims(hCanOri) [expr 328 + 28]
# Canvas size; with border.
set dims(wMinCanvas) [expr $dims(wCanOri) + 2]
set dims(hMinCanvas) [expr $dims(hCanOri) + 2]
set dims(x) 30
set dims(y) 30
# Total size of the application (not including menu); only temporary values.
set dims(wRoot) 1    
set dims(hRoot) 1
# As above but including the menu.
set dims(wTot) 1    
set dims(hTot) 1

# Running tag number must be unique for each item.
# It is always used in conjunction with the local prefix as $prefix/$itno.
# It is *never* reused during the lifetime of the application.
# It is updated *only* when writing to own canvas.
# When the server part writes to the canvas it is the prefix/no of the
# remote client that is used. Perhaps [clock seconds] may be used? 
# Should be a positive number though.
set itno 0
# Unique tag prefix for items created by this client.
set myItpref [info hostname]
if {$myItpref == ""}  {
    set myItpref $internalIPname
}
# On multiuser platforms (unix) prepend the user name; no spaces allowed.
if {([string compare $tcl_platform(platform) "unix"] == 0) &&  \
  [info exists env(USER)]}  {
    set user $thisUserName
    regsub -all " " $user "" user
    set myItpref ${user}@$myItpref
}
# Running number for entries in the comm frame.
set nEnt 0

# Useful time constants in seconds.
set tmsec(min) 60
set tmsec(hour) [expr 60*$tmsec(min)]
set tmsec(day) [expr 24*$tmsec(hour)]
set tmsec(week) [expr 7*$tmsec(day)]
set tmsec(30days) [expr 30*$tmsec(day)]

# Various constants.
set kPI 3.14159265359
set kRad2Grad [expr 180.0/$kPI]
set kGrad2Rad [expr $kPI/180.0]
set kTan225 [expr tan($kPI/8.0)]
set kTan675 [expr tan(3.0 * $kPI/8.0)]

#--- Preferences are set here --------------------------------------
# The application major and minor version numbers; should only be written to
# default file, never read.
set prefs(majorVers) 0
set prefs(minorVers) 91
# These are the hardcoded, application default, values, and can be
# overridden by the ones in user default file.
# The tool buttons.
set prefs(btState) 00
set prefs(btStateOld) 00
#
set prefs(bgColCanvas) #dedede
set prefs(bgColGeneral) #dedede
# fg and bg colors set in color selector; bgCol always white.
set prefs(fgCol) black
set prefs(bgCol) white
# Line thickness.
set prefs(penThick) 1	
# Brush thickness.
set prefs(brushThick) 8	
# Fill color for circles, polygons etc.
set prefs(fill) 0
# If and how polygons should be smoothed.
set prefs(smooth) 0
set prefs(splinesteps) 0
# Arc styles.
set prefs(arcstyle) "arc"
# Offset when duplicating canvas items and when opening images and movies.
set prefs(offsetCopy) 16
# Present anchor coordinates when importing images and movies.
# Gets translated with 'prefs(offsetCopy)' for each new imported item.
set importAnchor(x) $prefs(offsetCopy)
set importAnchor(y) $prefs(offsetCopy)
# Side of selecting box .
set prefs(aBBox) 2
# Want to fit all movies within canvas?
set prefs(autoFitMovies) 1
# Font prefs set in menus. Sizes according to html.
set prefs(fontSize) 2
set prefs(font) Helvetica
set prefs(fontWeight) normal
set prefs(canvasFonts) [list "Times" "Helvetica" "Courier"]
# Html sizes or point sizes when transferring text items?
set prefs(useHtmlSizes) 1
# Grid on or off.
set prefs(canGridOn) 0                  
# Grid spacing.
set prefs(gridDist) 40                 
# Only manipulate own items?
set prefs(privacy) 0    
# Should we check that server commands do not contain any potentially harmful
# instructions?
set prefs(checkSafety) 1
# Mark bounding box (1) or each coords (0).
set prefs(bboxOrCoords) 0
# Scale factor used when scaling canvas items.
set prefs(scaleFactor) 1.2
# Use common CG when scaling more than one item?
set prefs(scaleCommonCG) 0
# Constrain movements to 45 degrees, else 90 degree intervals.
set prefs(45) 1

# Network options: symmetric network, or a central server?
# This determines many other network options.
set prefs(symmetricNet) 1
# Connect automatically to connecting clients if 'symmetricNet'.
set prefs(autoConnect) 1                
# Disconnect automatically to disconnecting clients.
set prefs(autoDisconnect) $prefs(autoConnect)	
# When connecting to other client, connect automatically to all *its* clients.
set prefs(multiConnect) 1
# In a centralized network config there should be only one server;
# all other apps must be clients. Not relevant for 'symmetricNet' true.
set prefs(thisCentralServ) 0
# Start server when launching application, if not client only?
set prefs(autoStartServer) 1

# Open connection in async mode.
set prefs(asyncOpen) 1
# Safe server interpretator? (not working)
set prefs(makeSafeServ) 1
# Maximum time to wait for server to respond.
set prefs(timeout) 60
# When and how old is a cached file allowed to be before downloading a new?
set prefs(checkCache) "never"
# Animated 'Coccinella'?
if {$prefs(QuickTimeTcl)}  {
    set prefs(coccinellaMovie) 0
} else  {
    set prefs(coccinellaMovie) 0
}
# If we have got TclSpeech the default is to have it enabled.
if {$prefs(TclSpeech)}  {
    set prefs(TclSpeechOn) 1
} else {
    set prefs(TclSpeechOn) 0
}
set prefs(itemDir) [file join $thisPath items]
# Is the toolbar visible? (actually, rather a state than a prefs)
set prefs(visToolbar) 1
# If it is the first time the application is launched, then welcome.
set prefs(firstLaunch) 1
# The file name of the welcoming canvas.
set prefs(welcomeFile) [file join $thisPath myWelcome.can]
# Sometimes used for debugging.
set prefs(allowLocalServerCanvas) 0
# Shell print command in unix.
if {[info exists env(PRINTER)]}  {
    set prefs(unixPrintCmd) "lpr -P$env(PRINTER)"
} else  {
    set prefs(unixPrintCmd) "lpr"
}
# Postscript options.
set postscriptOpts {-pageheight 20c -pagewidth 16c -pageanchor nw}

#---- Shortcuts -----------------------------------------------------------
#----   domain names for open connection ----------------------------------
set shortcuts [list [list  \
  "user specified" mbmac laelia atlas vanessa other]   \
  [list  \
  remote.computer.name mbmac.lin.foa.se laelia.lin.foa.se   \
  atlas.sto.foa.se vanessa.lin.foa.se other.computer.name]]
#----   url's for streaming live movies -----------------------------------
set shortsMulticastQT {{   \
  "user specified"   \
  Bloomberg          \
  "Hard Radio"       \
  NPR  \
  "BBC World TV" }  {  \
  {}  \
  www.apple.com/quicktime/showcase/radio/bloomberg/bloombergradio.mov  \
  www.apple.com/quicktime/showcase/radio/hardradio/hardradio.mov  \
  www.apple.com/quicktime/showcase/radio/npr/npr.mov  \
  www.apple.com/quicktime/favorites/bbc_world1/bbc_world1.mov}}
#--------------------------------------------------------------------------
set compNameOrNum [lindex [lindex $shortcuts 1] 0]
# Number of bytes in copy buffert.
set chunkSize 4096
set chunkSize 8192
# Use a very unlikely combination for the separator of items in clipboard.
set clipItemSep " ANDqzU\06 "
# Default for 'clipToken' should always be "string" to be prepared
# for imports from other apps.
set clipToken "string"
# Status flags:
set isServerUp 0

# Mapping from error code to error message; 420+ own, rest HTTP codes.
set tclwbProtMsg(200) "OK"
set tclwbProtMsg(201) "Created"
set tclwbProtMsg(202) "Accepted"
set tclwbProtMsg(204) "No Content"
set tclwbProtMsg(301) "Moved Permanently"
set tclwbProtMsg(302) "Moved Temporarily"
set tclwbProtMsg(304) "Not Modified"
set tclwbProtMsg(320) "No other clients connected"
set tclwbProtMsg(400) "Bad Request"
set tclwbProtMsg(401) "Unauthorized"
set tclwbProtMsg(403) "Not Found"
set tclwbProtMsg(404) "Not Found"
set tclwbProtMsg(420) "File already cached"
set tclwbProtMsg(421) "MIME type unsupported"
set tclwbProtMsg(422) "MIME type not given"
set tclwbProtMsg(500) "Internal Server Error"
set tclwbProtMsg(501) "Not Implemented"
set tclwbProtMsg(502) "Bad Gateway"
set tclwbProtMsg(503) "Service Unavailable"

# Modifier keys and meny height (guess); add canvas border as well.
# System fonts used. Other system dependent stuff.
if {$tcl_platform(platform) == "unix"}	{
    #set osprefs(mod) Control
    set osprefs(mod) Command
    set clockClicksPerSec 1000000
    set prefs(userPrefsFilePath) "~/.whiteboard"
} elseif {$tcl_platform(platform) == "macintosh"}  {
    set osprefs(mod) Command
    set clockClicksPerSec 1000000
    if {[info exists env(PREF_FOLDER)]}  {
	set prefs(userPrefsFilePath)  \
	  [file join $env(PREF_FOLDER) "Whiteboard Prefs"]
    } else  {
	set prefs(userPrefsFilePath) "Whiteboard Prefs"
    }
} elseif {$tcl_platform(platform) == "windows"}	 {
    set osprefs(mod) Control
    #array set fontSize2Points {1 6 2 8 3 9 4 11 5 14 6 20}
    #array set fontPoints2Size {6 1 8 2 9 3 11 4 14 5 20 6}
    set clockClicksPerSec 1000000
    if {[info exists env(USERPROFILE)]}  {
	set prefs(userPrefsFilePath)  \
	  [file join $env(USERPROFILE) "Whiteboard Prefs"]
    } else  {
	set prefs(userPrefsFilePath) "Whiteboard Prefs"
    }
}
set ::SplashScreen::startMsg "Set and read preferences..."

# Set default light gray background.
if {$tcl_platform(platform) == "macintosh" || $tcl_platform(platform) == "unix"  \
  || $tcl_platform(platform) == "windows"}  {
    option add *Canvas.Background $prefs(bgColGeneral)
    option add *Frame.Background $prefs(bgColGeneral)
    option add *Label.Background $prefs(bgColGeneral)
    option add *Text.Background $prefs(bgColGeneral)
    option add *Message.Background $prefs(bgColGeneral)
    #option add *Button.Background $prefs(bgColGeneral)
    option add *Radiobutton.Background $prefs(bgColGeneral)
    option add *Progressbar.Background $prefs(bgColGeneral)
    option add *Entry.Background white
    option add *Entry.BorderWidth 1
}
if {$tcl_platform(platform) == "macintosh"}  {
    option add *Checkbutton.Background $prefs(bgColGeneral)
    option add *Button.HighlightBackground $prefs(bgColGeneral)
    #option add *Entry.Font application
    option add *Entry.Font $sysFont(m)
    option add *Listbox.Font $sysFont(m)
}
#tk_setPalette background $prefs(bgColGeneral) activeForeground white  \
#  selectBackground black selectForeground white highlightColor black
#option add *Entry.Background white
#option add *Listbox.Background white

# Manage the user preferences. Start by reading the preferences file.
PreferencesInit

# Set the user preferences from the preferences file if they are there,
# else take the hardcoded defaults.
# 'thePrefs': a list of lists where each sublist defines an item in the
# following way:  {theVarName itsResourceName itsHardCodedDefaultValue}.
PreferencesAdd [list  \
  [list prefs(majorVers)       prefs_majorVers       $prefs(majorVers)]  \
  [list prefs(minorVers)       prefs_minorVers       $prefs(minorVers)]  \
  [list prefs(btState)         prefs_btState         $prefs(btState)]  \
  [list prefs(bgColCanvas)     prefs_bgColCanvas     $prefs(bgColCanvas)]  \
  [list prefs(fgCol)           prefs_fgCol           $prefs(fgCol)]  \
  [list prefs(penThick)        prefs_penThick        $prefs(penThick)]  \
  [list prefs(brushThick)      prefs_brushThick      $prefs(brushThick)]  \
  [list prefs(fill)            prefs_fill            $prefs(fill)]  \
  [list prefs(arcstyle)        prefs_arcstyle        $prefs(arcstyle)]  \
  [list prefs(fontSize)        prefs_fontSize        $prefs(fontSize)] \
  [list prefs(font)            prefs_font            $prefs(font)]  \
  [list prefs(fontWeight)      prefs_fontWeight      $prefs(fontWeight)]  \
  [list prefs(canvasFonts)     prefs_canvasFonts     $prefs(canvasFonts)]  \
  [list prefs(smooth)          prefs_smooth          $prefs(smooth)]  \
  [list prefs(splinesteps)     prefs_splinesteps     $prefs(splinesteps)]  \
  [list prefs(canGridOn)       prefs_canGridOn       $prefs(canGridOn)]  \
  [list prefs(privacy)         prefs_privacy         $prefs(privacy)]   \
  [list prefs(45)              prefs_45              $prefs(45)]   \
  [list prefs(checkCache)      prefs_checkCache      $prefs(checkCache)]  \
  [list prefs(symmetricNet)    prefs_symmetricNet    $prefs(symmetricNet)]  \
  [list prefs(autoConnect)     prefs_autoConnect     $prefs(autoConnect)]  \
  [list prefs(multiConnect)    prefs_multiConnect    $prefs(multiConnect)]  \
  [list prefs(thisCentralServ) prefs_thisCentralServ $prefs(thisCentralServ)]  \
  [list postscriptOpts         postscriptOpts        $postscriptOpts]  \
  [list shortcuts              shortcuts             $shortcuts]  \
  [list shortsMulticastQT      shortsMulticastQT     $shortsMulticastQT]  \
  [list dims(x)                dims_x                $dims(x)]  \
  [list dims(y)                dims_y                $dims(y)]  \
  [list dims(wRoot)            dims_wRoot            $dims(wRoot)]  \
  [list dims(hRoot)            dims_hRoot            $dims(hRoot)]  \
  [list prefs(firstLaunch)     prefs_firstLaunch     $prefs(firstLaunch)]   \
  [list prefs(visToolbar)      prefs_visToolbar      $prefs(visToolbar)]   \
  [list prefs(TclSpeechOn)     prefs_TclSpeechOn     $prefs(TclSpeechOn)]   \
  [list prefs(unixPrintCmd)    prefs_unixPrintCmd    $prefs(unixPrintCmd)]  ]

# Function definitions
#--------------------------------------------------------------------------

# LabeledFrame --
#
#   A small utility that makes a nice frame with a label.
#   'wpath' is the widget path of the parent (it should be a frame); 
#   the return value is the widget path to the interior of the container.

set ::SplashScreen::startMsg "Read functions..."

proc LabeledFrame { wpath txt {opts {}} }  {
    global  sysFont
    
    pack [frame $wpath.st -borderwidth 0]  \
      -side top -fill both -pady 2 -padx 2 -expand true
    pack [frame $wpath.st.fr -relief groove -bd 2]  \
      -side top -fill both -expand true -padx 10 -pady 10 -ipadx 0 -ipady 0  \
      -in $wpath.st
    place [label $wpath.st.lbl -text $txt -font $sysFont(sb) -bd 0 -padx 6]  \
      -in $wpath.st -x 20 -y 14 -anchor sw
    return $wpath.st.fr
}

# CreateShortcutButtonPad --
#
#   Makes the top shortcut button pad. Switches between 'on' and 'off' state.
#   The 'subSpec' is only valid for 'init' where it can be 'off'.

proc CreateShortcutButtonPad  { what {subSpec {}} }  {
    global  wCan sysFont tcl_platform wDlgs
    
    set btNames {connect save_file open_file image printer stop}
    
    if {[string compare $what "init"] == 0}  {
	pack [frame .frtop -relief raised -borderwidth 0] -side top -fill x
	pack [frame .frtop.on -borderwidth 0] -fill x -side left -expand 1
	image create photo barvert -format gif  \
	  -file [file join images barvert.gif]
	image create photo barhoriz -format gif  \
	  -file [file join images barhoriz.gif]
	
	pack [label .frtop.on.barvert -image barvert -bd 1 -relief raised]  \
	  -padx 0 -pady 0 -side left
	pack [frame .frtop.on.fr -relief raised -borderwidth 1]   \
	  -side left -fill both -expand 1
	label .frtop.barhoriz -image barhoriz -relief raised -borderwidth 1
	bind .frtop.on.barvert <Button-1> [list CreateShortcutButtonPad "off"]
	
	# If one of the images exist, we asume we have got them all.
	if {[lsearch -exact [image names] "connect"] == -1}  {
	    set icol 1
	    foreach btnm $btNames {
		image create photo bt$btnm -format gif   \
		  -file [file join images $btnm.gif]
		pack [button .frtop.on.fr.$btnm -borderwidth 1 -image bt$btnm] \
		  -padx 4 -pady 2 -side left
		incr icol
	    }
	}
	
	# Make bindings.
	.frtop.on.fr.connect configure -command DoOpenConnection
	.frtop.on.fr.save_file configure -command "DoSaveCanvasFile $wCan"
	.frtop.on.fr.open_file configure -command "DoOpenCanvasFile $wCan"
	.frtop.on.fr.image configure -command "ImportImageOrMovieDlg $wCan"
	if {[string compare $tcl_platform(platform) "unix"] == 0}  {
	    .frtop.on.fr.printer configure  \
	      -command "::PrintPSonUnix::PrintPSonUnix $wDlgs(print) $wCan"
	} else {
	    .frtop.on.fr.printer configure -command "SavePostscript $wCan"
	}
	.frtop.on.fr.stop configure -command CancelAllPutGetAndPendingOpen
	
	# Do we want the toolbar to be collapsed at initialization?
	if {[string compare $subSpec "off"] == 0} {
	    pack forget .frtop.on
	    .frtop configure -bg gray75
	    pack .frtop.barhoriz -side left -padx 0 -pady 0
	    bind .frtop.barhoriz <Button-1> [list CreateShortcutButtonPad "on"]
	}
    } elseif {[string compare $what "off"] == 0}  {
	  
	# Fix the canvas size to what it is.
	$wCan configure -height [expr [winfo height $wCan] - 2]  \
	  -width [expr [winfo width $wCan] - 2]
	
	# Switch off the geometry constraint to let resize automatically.
	wm geometry . {}
	pack forget .frtop.on
	.frtop configure -bg gray75
	pack .frtop.barhoriz -side left -padx 0 -pady 0
	bind .frtop.barhoriz <Button-1> [list CreateShortcutButtonPad "on"]
	update idletasks
	
	# Organize the new geometry. First fix using wm geometry, then relax
	# canvas size.
	set newGeom [ParseWMGeometry .]
	wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
	$wCan configure -height 1 -width 1
	after idle {SetNewWMMinsize}

    } elseif {[string compare $what "on"] == 0}  {
	$wCan configure -height [expr [winfo height $wCan] - 2]  \
	  -width [expr [winfo width $wCan] - 2]
	
	# Switch off the geometry constraint to let resize automatically.
	wm geometry . {}
	pack forget .frtop.barhoriz
	pack .frtop.on -fill x -side left -expand 1
	bind .frtop.barhoriz <Button-1> [list CreateShortcutButtonPad "off"]
	update idletasks
	
	set newGeom [ParseWMGeometry .]
	wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
	$wCan configure -height 1 -width 1
	after idle {SetNewWMMinsize}
    }
}

proc IsShortcutButtonVisable {  } {
    
    return [expr [winfo height .frtop] > 16 ? 1 : 0]
}

# CreateAllButtons --
#
#   Makes the toolbar button pad for the drawing tools.

proc CreateAllButtons  {	 }  {
    global  btFrame btNo2Name btName2No idColSel prefs wColSel
    
    array set btNo2Name	 \
      {00 point 01 move 10 line 11 arrow 20 rect 21 oval 30 pen 31 brush  \
      40 text 41 del 50 paint 51 poly 60 arc 61 rot}
    array set btName2No	 \
      {point 00 move 01 line 10 arrow 11 rect 20 oval 21 pen 30 brush 31  \
      text 40 del 41 paint 50 poly 51 arc 60 rot 61}
    
    for {set icol 0} {$icol <= 1} {incr icol}  {
	for {set irow 0} {$irow <= 6} {incr irow}  {


	    set lwi [label $btFrame.bt$irow$icol -image im_off$irow$icol \
	      -borderwidth 0]
	    grid $lwi -row $irow -column $icol -padx 0 -pady 0
	    bind $lwi <Button-1> "ClickToolButton %W $btNo2Name($irow$icol)"
	}
    }
    
    # Color selector.
    image create photo imcolor -format gif -file [file join images col_switch.gif]
    set imheight [image height imcolor]
    set wColSel [canvas $btFrame.cacol -width 56 -height $imheight  \
      -highlightthickness 0]
    $btFrame.cacol create image 0 0 -anchor nw -image imcolor
    set idColSel [$btFrame.cacol create rect 7 7 33 30	\
      -fill $prefs(fgCol) -outline {}]
    $btFrame.cacol bind $idColSel <Button-1> {ColorSelector $prefs(fgCol)}
    
    # Black and white reset rectangle.
    image create photo bwrect -format gif -file [file join images transparent_rect.gif]
    #	set idColBW [$btFrame.cacol create rect 38 4 52 18]
    set idBWReset [$btFrame.cacol create image 4 34 -anchor nw -image bwrect]
    $btFrame.cacol bind $idBWReset <Button-1>  \
      "$wColSel itemconfigure $idColSel -fill black;  \
      set prefs(fgCol) black; set prefs(bgCol) white"
    
    # bg and fg switching.
    set idBWSwitch [$btFrame.cacol create image 38 4 -anchor nw -image bwrect]
    $btFrame.cacol bind $idBWSwitch <Button-1> \
      "SwitchBgAndFgCol $wColSel $idColSel"
    grid $btFrame.cacol -  -padx 0 -pady 0
}

proc SwitchBgAndFgCol { wColSel idColSel }  {
    global  prefs
    
    $wColSel itemconfigure $idColSel -fill $prefs(bgCol)
    set tmp $prefs(fgCol)
    set prefs(fgCol) $prefs(bgCol)
    set prefs(bgCol) $tmp
}

# StartStopAnimatedWave, AnimateWave --
#
#   Utility routines for animating the wave in the status message frame.

proc StartStopAnimatedWave  { w start }  {
    global  animateWave
    
    # Define speed and update frequency. Pix per sec and times per sec.
    set speed 150
    set freq 16

    if {$start}  {
	# Check if not already started.
	if {[info exists animateWave]}  {
	    return
	}
	set animateWave(pix) [expr int($speed/$freq)]
	set animateWave(wait) [expr int(1000.0/$freq)]
	set animateWave(id) [$w create image 0 0 -anchor nw -image im_wave]
	$w lower $animateWave(id)
	set animateWave(x) 0
	set animateWave(dir) 1
	set animateWave(killId) [after $animateWave(wait) [list AnimateWave $w]]
    } elseif {[info exists animateWave(killId)]}  {
	after cancel $animateWave(killId)
	$w delete $animateWave(id)
	catch {unset animateWave}
    }
}

proc AnimateWave  { w }  {
    global  dims animateWave
    
    set deltax [expr $animateWave(dir) * $animateWave(pix)]
    incr animateWave(x) $deltax
    if {$animateWave(x) > [expr $dims(wStatMess) - 80]}  {
	set animateWave(dir) -1
    } elseif {$animateWave(x) <= -60}  {
	set animateWave(dir) 1
    }
    $w move $animateWave(id) $deltax 0
    set animateWave(killId) [after $animateWave(wait) [list AnimateWave $w]]
}

# StartStopMovie --
#
#   With the new -loopstate and -palindromeloopstate of the Movie widget,
#   this procedure is not necessary anymore. Fixed.

proc StartStopMovie  { w start }  {

    if {$start}  {
	$w play
    } else  {
	$w stop
	$w settime 0
    }
}

# ClickToolButton --
#
#   Uhhh...  When a tool button is clicked. Mainly sets all button specific
#   bindings.

proc ClickToolButton { w btName }  {
    global  prefs btName2No wCan btFrame statMess debugLevel
    
    if {$debugLevel >= 2}  {
	#puts "ClickToolButton:: w=$w, btName=$btName"
    }
    set prefs(btState) $btName2No($btName)
    set irow [string index $prefs(btState) 0]
    set icol [string index $prefs(btState) 1]
    $btFrame.bt$irow$icol configure -image im_on$irow$icol
    if {$prefs(btState) != $prefs(btStateOld)}  {
	set irow [string index $prefs(btStateOld) 0]
	set icol [string index $prefs(btStateOld) 1]
	$btFrame.bt$irow$icol configure -image im_off$irow$icol
    }
    set prefs(btStateOld) $prefs(btState)
    RemoveAllBindings $wCan
    
    # Deselect text items.
    if {$btName != "text"}  {
	$wCan select clear
    }
    if {$btName == "del" || $btName == "text"}  {
	DeselectAll $wCan
    }
    
    # Cancel any outstanding polygon drawings.
    PolyFinish $wCan -10 -10
    
    $wCan config -cursor {}
    
    switch $btName {
	point {
	    bind $wCan <Button-1>  \
	      {MarkBbox %W 0; BoxBegin %W [%W canvasx %x] [%W canvasy %y] rect}
	    bind $wCan <Shift-Button-1>	{MarkBbox %W 1}
	    bind $wCan <B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 rect 1}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 rect 1}
	    bind $wCan <Double-Button-1>   \
	      {::ItemInspector::ItemInspector %W}
	    set statMess "Point or drag to mark"
	}
	move {
	    
	    # Bindings for moving items; movies need special class.
	    # The frame with the movie the mouse events, not the canvas.
	    # With shift constrained move.
	    bind $wCan <Button-1>  \
	      {InitXMove %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>  \
	      {DoXMove %W [%W canvasx %x] [%W canvasy %y] item}
	    bind $wCan <ButtonRelease-1>  \
	      {SendXMove %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <Shift-B1-Motion>  \
	      {DoXMove %W [%W canvasx %x] [%W canvasy %y] item 1}
	    
	    # Moving single coordinates.
	    $wCan bind tbbox <Button-1>  \
	      {InitXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    $wCan bind tbbox <B1-Motion>  \
	      {DoXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    $wCan bind tbbox <ButtonRelease-1>  \
	      {SendXMove %W [%W canvasx %x] [%W canvasy %y] point}
	    $wCan bind tbbox <Shift-B1-Motion>   \
	      {DoXMove %W [%W canvasx %x] [%W canvasy %y] point 1}
	    
	    # Moving movies.
	    bind QTFrame <Button-1> {InitXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind QTFrame <B1-Motion> {DoXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind QTFrame <ButtonRelease-1> {SendXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind QTFrame <Shift-B1-Motion> {DoXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie 1}
	    
	    # Moving sequence grabber.
	    bind SGFrame <Button-1> {InitXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind SGFrame <B1-Motion> {DoXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    bind SGFrame <ButtonRelease-1> {SendXMove $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie}
	    
	    $wCan config -cursor hand2
	    set statMess "Move tool"
	}
	line {
	    bind $wCan <Button-1>  \
	      {LineBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 0}
	    bind $wCan <Shift-B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 1}
	    bind $wCan <ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 0}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 1}
	    set statMess "Line tool"
	}
	arrow {
	    bind $wCan <Button-1>  \
	      {LineBegin %W [%W canvasx %x] [%W canvasy %y] arrow}
	    bind $wCan <B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 0 arrow}
	    bind $wCan <Shift-B1-Motion>	\
	      {LineDrag %W [%W canvasx %x] [%W canvasy %y] 1 arrow}
	    bind $wCan <ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 0 arrow}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {LineEnd %W [%W canvasx %x] [%W canvasy %y] 1 arrow}
	    set statMess "Arrow tool"
	}
	rect {
	    # Bindings for rectangle drawing.
	    bind $wCan <Button-1>  \
	      {BoxBegin %W [%W canvasx %x] [%W canvasy %y] rect}
	    bind $wCan <B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 rect}
	    bind $wCan <Shift-B1-Motion> { \
	      BoxDrag %W [%W canvasx %x] [%W canvasy %y] 1 rect}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 rect}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 1 rect}
	    set statMess "Rectangle tool"
	}
	oval {
	    bind $wCan <Button-1>  \
	      {BoxBegin %W [%W canvasx %x] [%W canvasy %y] oval}
	    bind $wCan <B1-Motion>	\
	      {BoxDrag %W [%W canvasx %x] [%W canvasy %y] 0 oval}
	    bind $wCan <Shift-B1-Motion>	\
	      {BoxDrag %W [%W canvasx %x] [%W canvasy %y] 1 oval}
	    bind $wCan <ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 0 oval}
	    bind $wCan <Shift-ButtonRelease-1>  \
	      {SendBox %W [%W canvasx %x] [%W canvasy %y] 1 oval}
	    set statMess "Oval tool"
	}
	text {
	    CanvasEditBind $wCan
	    $wCan config -cursor xterm
	    set statMess "Text tool"
	}
	del {
	    bind $wCan <Button-1>  \
	      {DeleteItem %W [%W canvasx %x] [%W canvasy %y]}
	    bind QTFrame <Button-1> {DeleteItem $wCan  \
	      [$wCan canvasx [expr [winfo x %W] + %x]]  \
	      [$wCan canvasy [expr [winfo y %W] + %y]] movie %W}
	    set statMess "Click to delete"
	}
	pen {
	    bind $wCan <Button-1>  \
	      {StrokeBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {StrokeDrag %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <ButtonRelease-1>  \
	      {StrokeEnd %W [%W canvasx %x] [%W canvasy %y]}
	    $wCan config -cursor pencil
	    set statMess "Pen tool"
	}
	brush {
	    bind $wCan <Button-1>  \
	      {StrokeBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>	\
	      {StrokeDrag %W [%W canvasx %x] [%W canvasy %y] $prefs(brushThick)}
	    bind $wCan <ButtonRelease-1>  \
	      {StrokeEnd %W [%W canvasx %x] [%W canvasy %y] $prefs(brushThick)}
	    set statMess "Brush tool"
	}
	paint {
	    bind $wCan  <Button-1>  \
	      {DoPaint %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan  <Shift-Button-1>  \
	      {DoPaint %W [%W canvasx %x] [%W canvasy %y] 1}
	    set statMess "Paint tool, shift-click to transparency"
	}
	poly {
            bind $wCan  <Button-1>  \
	      {PolySetPoint %W [%W canvasx %x] [%W canvasy %y]}
	    set statMess "Polygon tool, spacebar to cancel"
        }       
	arc {
	    bind $wCan <Button-1>  \
	      {ArcBegin %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <Shift-Button-1>  \
	      {ArcBegin %W [%W canvasx %x] [%W canvasy %y] 1}
	    set statMess "Arc tool, click to set center, spacebar to cancel"
	}
	rot {
	    bind $wCan <Button-1>  \
	      {InitRotateItem %W [%W canvasx %x] [%W canvasy %y]}
	    bind $wCan <B1-Motion>  \
	      {DoRotateItem %W [%W canvasx %x] [%W canvasy %y] 0}
	    bind $wCan <Shift-B1-Motion>  \
	      {DoRotateItem %W [%W canvasx %x] [%W canvasy %y] 1}
	    bind $wCan <ButtonRelease-1>  \
	      {FinRotateItem %W [%W canvasx %x] [%W canvasy %y]}
	    $wCan config -cursor exchange
	    set statMess "Rotate tool, select item first"
	}
    }
}

proc CanvasEditBind { c }  {
    
    bind $c <Button-1> \
      {CanvasFocus %W [%W canvasx %x] [%W canvasy %y]}
    bind $c <Button-2> \
      {CanvasTextPaste %W [%W canvasx %x] [%W canvasy %y]}
    #	bind $c <<Cut>> {CanvasTextCopy %W; CanvasTextDelete %W}
    #	bind $c <<Copy>> {CanvasTextCopy %W}
    #	bind $c <<Paste>> {CanvasTextPaste %W}
    $c bind text <Button-1>	 \
      {CanvasTextHit %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <B1-Motion>  \
      {CanvasTextDrag %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <Double-Button-1>  \
      {CanvasTextSelectWord %W [%W canvasx %x] [%W canvasy %y]}
    $c bind text <Delete> {CanvasTextDelete %W}
    $c bind text <Control-Key-d> {CanvasTextBackSpace %W}
    
    # Swallow any commands on mac's. 
    # Not working when doing chinese with command-T.
    $c bind text <Command-Any-Key> {puts "command any-key"}
    $c bind text <BackSpace> {CanvasTextBackSpace %W}
    $c bind text <Return> {CanvasNewLine %W}
    $c bind text <Any-Key> {CanvasTextInsert %W %A}
    $c bind text <Control-Any-Key> {CanvasTextInsert %W ""}
    $c bind text <Key-Right> {CanvasTextMoveRight %W}
    $c bind text <Key-Left> {CanvasTextMoveLeft %W}
    
    # Need some new string functions here.
    if {[info tclversion] >= 8.2}  {
	$c bind text <Key-Up> {CanvasTextMoveUpOrDown %W up}
	$c bind text <Key-Down> {CanvasTextMoveUpOrDown %W down}
    }
}

proc RemoveAllBindings { w }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "RemoveAllBindings::"
    }
    $w bind text <Button-1> {}
    $w bind text <B1-Motion> {}
    $w bind text <Any-Key> {}
    $w bind text <Double-Button-1> {}
    
    # Remove bindings on markers on selected items.
    $w bind tbbox <Button-1> {}
    $w bind tbbox <B1-Motion> {}
    $w bind tbbox <ButtonRelease-1> {}
    
    bind $w <Button> {}
    bind $w <Button-1> {}
    bind $w <Button-Motion> {}
    bind $w <ButtonRelease> {}
    bind $w <Shift-Button-1> {}
    bind $w <Double-Button-1> {}
    bind $w <Any-Key> {}
    bind $w <ButtonRelease-1> {}
    bind $w <B1-Motion> {}
    bind $w <Shift-B1-Motion> {}
    bind $w <Shift-ButtonRelease-1> {}
    bind QTFrame <Button-1> {}
    bind QTFrame <B1-Motion> {}
    bind QTFrame <ButtonRelease-1> {}
    focus .
    # Remove any text insertion...
    $w focus {}
}

# TraceStatusMessage --
#
#   Automatically update various status messages.

proc TraceStatusMessage { varName junk op }  {
    global  statMess wStatMess
    
    $wStatMess itemconfigure stattxt -text $statMess
}

# TracePreferences --
#
#   When certain preferences are set, they should trigger some change.

proc TracePreferences { varName index op }  {
    global  prefs dims wCan wColSel idColSel
    
    if {[string compare $varName "prefs"] == 0}  {
	switch $index {
	    bgColCanvas  {
		$wCan configure -bg $prefs(bgColCanvas)
	    }
	    fgCol  {
		$wColSel itemconfigure $idColSel -fill $prefs(fgCol)
		$wColSel raise $idColSel
	    }
	    canGridOn  {
		DoCanvasGrid
	    }
	}
    } elseif {[string compare $varName "dims"] == 0}  {
    
    }
}

# InitXMove --
#
#   'what' = "item": move an ordinary item.
#   'what' = "point": move one single point. Has always first priority.
#   'what' = "movie": QuickTime movie, make ghost rectangle instead.

proc InitXMove { w x y {what item} }  {
    global  xDrag kGrad2Rad debugLevel
    
    if {$debugLevel >= 2}  {
	puts "InitXMove:: w=$w, x=$x, y=$y, what=$what"
    }
    # If more than one item triggered, choose the "point".
    if {[info exists xDrag(what)] && $xDrag(what) == "point"}  {
	if {$debugLevel >= 2}  {
	    puts "InitXMove:: rejected"
	}
	return
    }
    set id_ {[0-9]+}
    set xDrag(what) $what
    set xDrag(baseX) $x
    set xDrag(baseY) $y
    set xDrag(anchorX) $x
    set xDrag(anchorY) $y
    # In some cases we need the anchor point to be an exact item specific coordinate.
    
    set xDrag(type) [$w type current]
    # Are we moving one point of a single segment line?
    set xDrag(singleSeg) 0
    if {$debugLevel >= 2}  {
	puts "InitXMove:: xDrag(type)=$xDrag(type), current=[CanvasGetItnoFrom $w current]"
    }    
    # If movie then make ghost rectangle. 
    # Movies do not obey the usual stacking order!
    if {$what == "movie"}  {
	set id [FindTypeFromOverlapping $w $x $y "movie"]
	if {$id == ""}  {
	    return
	}
	set it [CanvasGetItnoFrom $w $id]
	if {$it == ""}  {
	    return
	}
	$w addtag selectedmovie withtag $id
	set bbox [$w bbox $id]
	set x1 [expr [lindex $bbox 0] - 1]
	set y1 [expr [lindex $bbox 1] - 1]
	set x2 [expr [lindex $bbox 2] + 1]
	set y2 [expr [lindex $bbox 3] + 1]
	$w create rectangle $x1 $y1 $x2 $y2 -outline gray50 -width 3 \
	  -stipple gray50 -tags "ghostrect"
	
	# Moving a marker of a selected item, highlight marker.
      } elseif {$what == "point"}  {
	  
	  # 'current' must be a marker with tag 'tbbox'.
	  set id [$w find withtag current]
	  $w addtag hitBbox withtag $id
	  #puts "InitXMove:: w gettags current=[$w gettags current]"
	  
	  # Find associated id for the actual item. Saved in the tags of the marker.
	  if {![regexp " +id($id_)" [$w gettags current] match theItemId]}  {
	      #puts "no match: w gettags current=[$w gettags current]"
	      return
	  }
	  set xDrag(type) [$w type $theItemId]
	  if {($xDrag(type) == "text") || ($xDrag(type) == "image")}  {
	      #unset xDrag
	      return
	  }
	  # Make a highlightbox at the 'hitBbox' marker.
	  set bbox [$w bbox $id]
	  set x1 [expr [lindex $bbox 0] - 1]
	  set y1 [expr [lindex $bbox 1] - 1]
	  set x2 [expr [lindex $bbox 2] + 1]
	  set y2 [expr [lindex $bbox 3] + 1]
	  $w create rectangle $x1 $y1 $x2 $y2 -outline black -width 1 \
	    -tags "lightBbox id$theItemId" -fill white
	  
	  # Get the index of the coordinates that was 'hit'. Then update only
	  # this coordinate when moving.
	  # For rectangle and oval items a list with all four coordinates is used,
	  # but only the hit corner and the diagonally opposite one are kept.
	  set oldCoords [$w coords $theItemId]
	  #puts "x= $x, y=$y, oldCoords=$oldCoords"
	  if {([string compare $xDrag(type) "rectangle"] == 0) ||  \
	    ([string compare $xDrag(type) "oval"] == 0)}  {
	      
	      # Need to reconstruct all four coordinates as: 0---1
	      #                                              |   |
	      #                                              2---3
	      set fullListCoords [concat   \
		[lindex $oldCoords 0] [lindex $oldCoords 1]  \
		[lindex $oldCoords 2] [lindex $oldCoords 1]  \
		[lindex $oldCoords 0] [lindex $oldCoords 3]  \
		[lindex $oldCoords 2] [lindex $oldCoords 3] ]
	  } else  {
	      set fullListCoords $oldCoords
	  }
	  # Deal first with the arc points.
	  if {([string compare $xDrag(type) "arc"] == 0)}  {
	      set xDrag(coords) $fullListCoords
	      
	      # Some geometry. We have got the coordinates defining the box.
	      # Find out if we clicked the 'start' or 'extent' "point".
	      # Tricky part: be sure that the branch cut is at +-180 degrees!
	      # 'itemcget' gives angles 0-360, while atan2 gives -180-180.
	      set xDrag(arcX) $x
	      set xDrag(arcY) $y
	      set theCoords $xDrag(coords)
	      set x1 [lindex $fullListCoords 0]
	      set y1 [lindex $fullListCoords 1]
	      set x2 [lindex $fullListCoords 2]
	      set y2 [lindex $fullListCoords 3]
	      foreach {x1 y1 x2 y2} $fullListCoords {}
	      set r [expr abs(($x1 - $x2)/2.0)]
	      set cx [expr ($x1 + $x2)/2.0]
	      set cy [expr ($y1 + $y2)/2.0]
	      set xDrag(arcCX) $cx
	      set xDrag(arcCY) $cy
	      set startAng [$w itemcget $theItemId -start]
	      # Put branch cut at +-180!
	      if {$startAng > 180}  {
		  set startAng [expr $startAng - 360]
	      }
	      set extentAng [$w itemcget $theItemId -extent]
	      set xstart [expr $cx + $r * cos($kGrad2Rad * $startAng)]
	      set ystart [expr $cy - $r * sin($kGrad2Rad * $startAng)]
	      set xfin [expr $cx + $r * cos($kGrad2Rad * ($startAng + $extentAng))]
	      set yfin [expr $cy - $r * sin($kGrad2Rad * ($startAng + $extentAng))]
	      set dstart [expr hypot($xstart - $x,$ystart - $y)]
	      set dfin [expr hypot($xfin - $x,$yfin - $y)]
	      set xDrag(arcStart) $startAng
	      set xDrag(arcExtent) $extentAng
	      set xDrag(arcFin) [expr $startAng + $extentAng]
	      if {$dstart < $dfin}  {
		  set xDrag(arcHit) "start"
	      } else  {
		  set xDrag(arcHit) "extent"
	      }
	      #puts "InitXMove:: xDrag=[parray xDrag]"
	      
	  } else  {
	      # Deal with other item points.
	      # Find the one closest to the hit marker.
	      set n [llength $fullListCoords]
	      set minDist 1000
	      for {set i 0} {$i < $n} {incr i 2}  {
		  set len [expr hypot([lindex $fullListCoords $i] - $x,  \
		    [lindex $fullListCoords [expr $i + 1]] - $y)]
		  if {$len < $minDist}  {
		      set ind $i
		      set minDist $len
		  }
	      }
	      set ptInd [expr $ind/2]
	      #puts "ind=$ind, ptInd=$ptInd"
	      if {([string compare $xDrag(type) "rectangle"] == 0) ||  \
		([string compare $xDrag(type) "oval"] == 0)}  {
		  
		  # Keep only hit corner and the diagonally opposite one.
		  set coords [concat [lindex $fullListCoords $ind]  \
		    [lindex $fullListCoords [expr $ind + 1]] ]
		  if {$ptInd == 0}  {
		      set coords [lappend coords    \
			[lindex $fullListCoords 6] [lindex $fullListCoords 7] ]
		  } elseif {$ptInd == 1}  {
		      set coords [lappend coords    \
			[lindex $fullListCoords 4] [lindex $fullListCoords 5] ]
		  } elseif {$ptInd == 2}  {
		      set coords [lappend coords    \
			[lindex $fullListCoords 2] [lindex $fullListCoords 3] ]
		  } elseif {$ptInd == 3}  {
		      set coords [lappend coords    \
			[lindex $fullListCoords 0] [lindex $fullListCoords 1] ]
		  }	    
		  set ind 0
		  set fullListCoords $coords
	      }
	      # If moving a single line segment with shift, we need the
	      # anchor point to be the "other" point.
	      if {([string compare $xDrag(type) "line"] == 0) &&  \
		([llength $oldCoords] == 4)}  {
		  set xDrag(singleSeg) 1
		  # Other point denoted remote x and y.
		  if {$ind == 0} {
		      set xDrag(remX) [lindex $oldCoords 2]
		      set xDrag(remY) [lindex $oldCoords 3]
		  } else  {
		      set xDrag(remX) [lindex $oldCoords 0]
		      set xDrag(remY) [lindex $oldCoords 1]
		  }
		  #parray xDrag
	      }
	      if {$debugLevel >= 2}  {
		  puts "InitXMove:: ind=$ind, ptInd=$ptInd, fullListCoords=$fullListCoords"
	      }
	      set xDrag(hitInd) $ind
	      set xDrag(coords) $fullListCoords
	  }
	  
      } elseif {$what == "item"}  {
	  
	  # Add specific tag to the item being moved for later use.
	  set id [$w find withtag current]
	  $w addtag ismoved withtag $id
      }
  }

# DoXMove --
#
#   If selected items, move them, else move current item if exists.
#   It uses the xDrag array to keep track of start and current position.
#   If 'what' = "point", then drag the single point.

proc DoXMove { w x y what {shift 0} }	{
    global  xDrag kGrad2Rad kRad2Grad
    
    if {![info exists xDrag]}  {
	return
    }
    # If we drag a point, then reject events triggered by non-point events.
    if {([string compare $xDrag(what) "point"] == 0) &&   \
      ([string compare $what "point"] != 0)}  {
	return
    }
    # If dragging 'point' (marker) of a fixed size item, return.
    if {([string compare $xDrag(what) "point"] == 0) &&   \
      ( ([string compare $xDrag(type) "text"] == 0) ||   \
      ([string compare $xDrag(type) "image"] == 0) )}  {
	return
    }
    set id_ {[0-9]+}

    # If constrained to 90/45 degrees.
    # Should this be item dependent?
    if {$shift}  {
	if {([string compare $xDrag(what) "point"] == 0) &&  \
	  ([string compare $xDrag(type) "arc"] == 0)}  {
	    set newco [ConstrainedDrag $x $y $xDrag(arcCX) $xDrag(arcCY)]
	} else  {
	    # Are we moving one point of a single segment line?
	    if {$xDrag(singleSeg)}  {
		set newco [ConstrainedDrag $x $y $xDrag(remX) $xDrag(remY)]
	    } else  {
		set newco [ConstrainedDrag $x $y $xDrag(anchorX) $xDrag(anchorY)]
	    }
	}
	set x [lindex $newco 0]
	set y [lindex $newco 1]
	foreach {x y} $newco {}
    }
    # First, get canvas objects with tag 'selected'.
    set ids [$w find withtag selected]
    if {[string compare $what "item"] == 0}  {
	
	# If no selected items.
	if {[string length $ids] == 0}  {
	    # Be sure to exclude nonmovable items.
	    set tagsCurrent [$w gettags current]
	    set it [CanvasGetItnoFrom $w current]
	    #puts "DoXMove:: tagsCurrent=$tagsCurrent, it=$it"
	    if {[string length $it] == 0}  {
		return
	    }
	    #puts "DoXMove:: tagsCurrent=$tagsCurrent"
	    if { [lsearch $tagsCurrent grid] >= 0 }  {
		return
	    }
	    $w move current [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	} else  {
	    # If selected, move all items and markers.
	    foreach id $ids  {
		set it [CanvasGetItnoFrom $w $id]
		if {$it != ""}   {
		    $w move $id [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
		}
	    }
	    # Move markers with them.
	    $w move tbbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	} 
    } elseif {[string compare $what "movie"] == 0}  {
	
	# Moving a movie.
	$w move ghostrect [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	
    } elseif {[string compare $what "point"] == 0}  {
	
	# Find associated id for the actual item. Saved in the tags of the marker.
	if {![regexp " +id($id_)" [$w gettags hitBbox] match theItemId]}  {
	    #puts "no match: w gettags hitBbox=[$w gettags hitBbox]"
	    return
	}
	if {[lsearch [$w gettags current] hitBbox] == -1}  {
	    #puts "DoXMove:: Warning, no match"
	    return
	}
	# Find the item type of the item that is marked. Depending on type,
	# do different things.
	if {[string compare $xDrag(type) "arc"] == 0}  {
	    
	    # Some geometry. We have got the coordinates defining the box.
	    set theCoords $xDrag(coords)
	    set x1 [lindex $theCoords 0]
	    set y1 [lindex $theCoords 1]
	    set x2 [lindex $theCoords 2]
	    set y2 [lindex $theCoords 3]
	    foreach {x1 y1 x2 y2} $theCoords {}
	    set r [expr abs(($x1 - $x2)/2.0)]
	    set cx [expr ($x1 + $x2)/2.0]
	    set cy [expr ($y1 + $y2)/2.0]
	    set startAng [$w itemcget $theItemId -start]
	    set extentAng [$w itemcget $theItemId -extent]
	    set xstart [expr $cx + $r * cos($kGrad2Rad * $startAng)]
	    set ystart [expr $cy - $r * sin($kGrad2Rad * $startAng)]
	    set xfin [expr $cx + $r * cos($kGrad2Rad * ($startAng + $extentAng))]
	    set yfin [expr $cy - $r * sin($kGrad2Rad * ($startAng + $extentAng))]
	    set newAng [expr $kRad2Grad * atan2($cy - $y,-($cx - $x))]
	    #puts "cx=$cx, cy=$cy, startAng=$startAng, newAng=$newAng, extentAng=$extentAng,\
	    #  xfin=$xfin, yfin=$yfin"
	    
	    # Dragging the 'extent' point or the 'start' point?
	    if {[string compare $xDrag(arcHit) "extent"] == 0}  { 
		set extentAng [expr $newAng - $xDrag(arcStart)]
		
		# Same trick as when drawing it; take care of the branch cut.
		if {$xDrag(arcExtent) - $extentAng > 180}  {
		    set extentAng [expr $extentAng + 360]
		} elseif {$xDrag(arcExtent) - $extentAng < -180}  {
		    set extentAng [expr $extentAng - 360]
		}
		set xDrag(arcExtent) $extentAng
		
		# Update angle.
		$w itemconfigure $theItemId -extent $extentAng
		
		# Move highlight box.
		$w move hitBbox [expr $xfin - $xDrag(arcX)]   \
		  [expr $yfin - $xDrag(arcY)]
		$w move lightBbox [expr $xfin - $xDrag(arcX)]   \
		  [expr $yfin - $xDrag(arcY)]
		set xDrag(arcX) $xfin
		set xDrag(arcY) $yfin
	    } elseif {[string compare $xDrag(arcHit) "start"] == 0}  {

		# Need to update start angle as well as extent angle.
		set newExtentAng [expr $xDrag(arcFin) - $newAng]
		# Same trick as when drawing it; take care of the branch cut.
		if {$xDrag(arcExtent) - $newExtentAng > 180}  {
		    set newExtentAng [expr $newExtentAng + 360]
		} elseif {$xDrag(arcExtent) - $newExtentAng < -180}  {
		    set newExtentAng [expr $newExtentAng - 360]
		}
		set xDrag(arcExtent) $newExtentAng
		set xDrag(arcStart) $newAng
		$w itemconfigure $theItemId -start $newAng
		$w itemconfigure $theItemId -extent $newExtentAng
		
		# Move highlight box.
		$w move hitBbox [expr $xstart - $xDrag(arcX)]   \
		  [expr $ystart - $xDrag(arcY)]
		$w move lightBbox [expr $xstart - $xDrag(arcX)]   \
		  [expr $ystart - $xDrag(arcY)]
		set xDrag(arcX) $xstart
		set xDrag(arcY) $ystart
	    }
	} else  {

	    set ind $xDrag(hitInd)
	    set newCoords [lreplace $xDrag(coords) $ind [expr $ind + 1] $x $y]
	    eval $w coords $theItemId $newCoords
	    $w move hitBbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	    $w move lightBbox [expr $x - $xDrag(baseX)] [expr $y - $xDrag(baseY)]
	}
    }
    set xDrag(baseX) $x
    set xDrag(baseY) $y
}

# SendXMove --
#
#   Finished moving using DoXMove. Make sure that all connected clients
#   also moves either the selected items or the current item.

proc SendXMove { w x y {what item} }  {
    global  xDrag allIPnumsToSend ipNum2Socket debugLevel
    
    if {$debugLevel >= 2}  {
	puts "SendXMove:: what=$what, [info exists xDrag]"
    }
    if {![info exists xDrag]}  {
	return
    }
    # If we drag a point, then reject events triggered by non-point events.
    if {$xDrag(what) == "point" && $what != "point"}  {
	return
    }
    set id_ {[0-9]+}

    # Get item(s).
    # First, get canvas objects with tag 'selected', 'ismoved' or 'selectedmovie'.
    set ids [$w find withtag selected]
    if {[string compare $what "movie"] == 0}  {
	set id [$w find withtag selectedmovie]
    } else  {
	set id [$w find withtag ismoved]
    }
    set theItno [CanvasGetItnoFrom $w $id]
    if {$debugLevel >= 2}  {
	puts "SendXMove:: ids=$ids, id=$id, theItno=$theItno, x=$x, y=$y"
    }
    if {([string compare $what "item"] == 0) && $ids == "" && $theItno == ""}  {
	return
    }
    # Find item tags ('theItno') for the items being moved.
    if {([string compare $what "item"] == 0) ||   \
      ([string compare $what "movie"] == 0)}  {

	# If no selected items.
	if {$ids == ""}  {
	    
	    # We already have 'theItno' from above.
	} else  {
	    # If selected, move all items.
	    set theItno {}
	    foreach id $ids  {
		set it [CanvasGetItnoFrom $w $id]
		lappend theItno $it
	    }
	} 
	
	# Dragging points of items.
    } elseif {[string compare $what "point"] == 0}  {

	# Find associated id for the actual item. Saved in the tags of the marker.
	if {![regexp " +id($id_)" [$w gettags current] match theItemId]}  {
	    #puts "no match: w gettags current=[$w gettags current]"
	    return
	}
	set theItno [CanvasGetItnoFrom $w $theItemId]
	if {$theItno == ""}  {
	    return
	}
	# If endpoints overlap in line item, make closed polygon.
	# Find out if closed polygon or open line item. If closed, remove duplicate.
	set isClosed 0
	if {[string compare $xDrag(type) "line"] == 0} {
	    set n [llength $xDrag(coords)]
	    set len [expr hypot(  \
	      [lindex $xDrag(coords) [expr $n - 2]] -  \
	      [lindex $xDrag(coords) 0],  \
	      [lindex $xDrag(coords) [expr $n - 1]] -  \
	      [lindex $xDrag(coords) 1] )]
	    if {$len < 8}  {
		# Make the line segments to a closed polygon.
		#puts "isClosed"
		set isClosed 1
		# Get all actual options.
		set opcmd [CanvasItemConfigure $w $theItemId]
		set theCoords [$w coords $theItemId]
		set polyCoords [lreplace $theCoords end end]
		set polyCoords [lreplace $polyCoords end end]
		set cmd1 [list $w delete $theItemId]
		eval $cmd1
		# Make the closed polygon. Get rid of non-applicable options.
		foreach op {"-arrow" "-arrowshape" "-capstyle" "-joinstyle"}  {
		    set ind [lsearch -exact $opcmd $op]
		    if {$ind >= 0}  {
			set opcmd [lreplace $opcmd $ind [expr $ind + 1]]
		    }
		}
		# Replace -fill with -outline.
		set ind [lsearch -exact $opcmd "-fill"]
		if {$ind >= 0}  {
		    set opcmd [lreplace $opcmd $ind $ind "-outline"]
		}
		set opcmd [concat $opcmd "-fill {}"]

		# Update the new item id.
		set cmd2 "$w create polygon $polyCoords $opcmd"
		set theItemId [eval $w create polygon $polyCoords $opcmd]
	    }
	}
	if {!$isClosed}  {
	    if {[string compare $xDrag(type) "arc"] == 0} {
	    
		# The arc item: update both angles.
		set cmd "itemconfigure $theItno -start $xDrag(arcStart)   \
		  -extent $xDrag(arcExtent)"
		#puts "SendXMove:: xDrag(type)=arc, cmd=$cmd"
	    } else  {
		
		# Not arc, and not closed line item.
		set cmd "coords $theItno [$w coords $theItno]"
	    }
	}
	# Move all markers along.
	$w delete id$theItemId
	MarkBbox $w 0 $theItemId
    }
    # For QT movies: move the actual movie to the position of the ghost rectangle.
    if {[string compare $what "movie"] == 0}  {
	$w move selectedmovie [expr $x - $xDrag(anchorX)]  \
		[expr $y - $xDrag(anchorY)]
	$w dtag selectedmovie selectedmovie
	set cmd "coords $theItno [$w coords $theItno]"
    }
    # Delete the ghost rect or highlighted marker if any. Remove temporary tags.
    $w delete ghostrect
    $w delete lightBbox
    $w dtag all hitBbox 
    $w dtag all ismoved

    # Do send to all connected.
    foreach ip $allIPnumsToSend  {
	set s $ipNum2Socket($ip)
	if {[string compare $what "point"] == 0}  {
	    if {$isClosed}  {
		puts $s "CANVAS: $cmd1"
		puts $s "CANVAS: $cmd2"
	    } else  {
		puts $s "CANVAS: $cmd"
	    }
	} elseif {[string compare $what "movie"] == 0}  {
	    puts $s "CANVAS: $cmd"
	} else  {
	    foreach it $theItno  {
		puts $s "CANVAS: move $it [expr $x - $xDrag(anchorX)]  \
		  [expr $y - $xDrag(anchorY)]"
	    }
	}
    }
    catch {unset xDrag}
}

proc BoxBegin { w x y type }	{
    global  theBox

    set theBox($w,anchor) [list $x $y]
    catch {unset theBox($w,last)}
}

# BoxDrag --
#   
#   If 'shift' constrain to square or circle.
#   If 'mark', then draw a temporary marking rectangle.
#   If not 'mark', then draw ordinary rectangle if 'type' is rect,
#   or oval if 'type' is oval.

proc BoxDrag { w x y shift type {mark 0} }  {
    global  theBox itno myItpref prefs debugLevel
    
    if {$debugLevel >= 2}  {
	#puts "BoxDrag (entry):: type=$type, mark=$mark"
    }
    catch {$w delete $theBox($w,last)}
    # If not set anchor, just return.
    if {![info exists theBox($w,anchor)]}  {
	return
    }
    set boxOrig $theBox($w,anchor)
    # If 'shift' constrain to square or circle.
    if {$shift}  {
	set box [eval "ConstrainedBoxDrag $theBox($w,anchor) $x $y $type"]
	set boxOrig [lrange $box 0 1]
	set x [lindex $box 2]
	set y [lindex $box 3]
    }
    # Either mark rectangle or draw rectangle.
    if {$mark == 0}  {
	if {$prefs(fill) == 0}  {
	    set theBox($w,last) [eval {$w create $type} $boxOrig  \
	      {$x $y -outline $prefs(fgCol) -width $prefs(penThick)  \
	      -tags [list $type $myItpref/$itno] }]
	} else  {
	    set theBox($w,last) [eval {$w create $type} $boxOrig  \
	      {$x $y -outline $prefs(fgCol) -fill $prefs(fgCol)  \
	      -width $prefs(penThick) -tags [list $type $myItpref/$itno] }]
	}
    } else {
	set theBox($w,last) [eval {$w create $type} $boxOrig	\
	  {$x $y -outline gray50 -stipple gray50 -width 2 -tags "markbox" }]
    }
}

# SendBox --
#
#   If 'shift' constrain to square or circle.
#   If 'mark', then make the overlapping objects selected and delete the
#   temporary rectangle. If not 'mark', then let all other clients
#   write a rectangle or oval.
#   'type' should be 'rect' or 'oval'.

proc SendBox { w x y shift type {mark 0} }  {
    global  theBox itno myItpref prefs debugLevel  \
      allIPnumsToSend ipNum2Socket
    
    # If no theBox($w,anchor) defined just return.
    if {![info exists theBox($w,anchor)]}   {
	return
    }
    if {[lindex $theBox($w,anchor) 0] == $x && [lindex $theBox($w,anchor) 1] == $y}  {
	set nomove 1
	return
    } else {
	set nomove 0
    }
    if {$mark}  {
	set ids [eval {$w find overlapping} $theBox($w,anchor) {$x $y}]
	if {$debugLevel >= 2} {
	    puts "SendBox:: ids=$ids"
	}
	foreach id $ids {
	    MarkBbox $w 1 $id
	}
	$w delete withtag markbox
    }
    # Create real objects.
    if {!$mark && !$nomove && [llength $allIPnumsToSend] > 0}  {
	set boxOrig $theBox($w,anchor)
	# If 'shift' constrain to square or circle.
	if {$shift}  {
	    set box [eval "ConstrainedBoxDrag $theBox($w,anchor) $x $y $type"]
	    set boxOrig [lrange $box 0 1]
	    set x [lindex $box 2]
	    set y [lindex $box 3]
	}
	if {$prefs(fill) == 0}  {
	    set cmd "create $type $boxOrig $x $y	\
	      -tags {$type $myItpref/$itno} -outline $prefs(fgCol)  \
	      -width $prefs(penThick)"
	} else  {
	    set cmd "create $type $boxOrig $x $y	\
	      -tags {$type $myItpref/$itno} -outline $prefs(fgCol)  \
	      -width $prefs(penThick) -fill $prefs(fgCol)"
	}
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) "CANVAS: $cmd"
	}
    }
    # Only increment counter for real objects; not when marking.
    if {!$mark}  {
	if {$debugLevel >= 2} {
	    puts "SendBox:: incr itno from $itno"
	}
	incr itno
    }
    catch {unset theBox}
}

# ConstrainedBoxDrag --
#
#   With the 'shift' key pressed, the rect and oval items are contrained
#   to squares and circles respectively.

proc ConstrainedBoxDrag  { xanch yanch x y type }  {
    
    set deltax [expr $x - $xanch]
    set deltay [expr $y - $yanch]
    set prod [expr $deltax * $deltay]
    if {$type == "rect"}  {
	set boxOrig [list $xanch $yanch]
	if {$prod != 0}  {
	    set sign [expr $prod / abs($prod)]
	} else  {
	    set sign 1
	}
	if {[expr abs($deltax)] > [expr abs($deltay)]}  {
	    set x [expr $sign * ($y - $yanch) + $xanch]
	} else  {
	    set y [expr $sign * ($x - $xanch) + $yanch]
	}
	
	# A pure circle is not made with the bounding rect model.
	# The anchor and the present x, y define the diagonal instead.
    } elseif {$type == "oval"}  {
	set r [expr hypot($deltax, $deltay)/2.0]
	set midx [expr ($xanch + $x)/2.0]
	set midy [expr ($yanch + $y)/2.0]
	set boxOrig [list [expr int($midx - $r)] [expr int($midy - $r)]]
	set x [expr int($midx + $r)]
	set y [expr int($midy + $r)]
    }
    return [concat $boxOrig $x $y]
}

# ConstrainedDrag --
#
#   Compute new x and y coordinates constrained to 90 or 45 degree intervals.
#   Return list of new x and y coordinates.

proc ConstrainedDrag  { x y xanch yanch }  {
    global  prefs kTan225 kTan675
    
    # Constrain movements to 90 degree intervals.
    if {!$prefs(45)}  {
	if {[expr abs($x - $xanch)] > [expr abs($y - $yanch)]}  {
	    set y $yanch
	} else  {
	    set x $xanch
	}
	return [list $x $y]
    } else  {
	
	# 45 degree intervals.
	set deltax [expr $x - $xanch]
	set deltay [expr $y - $yanch]
	if {[expr abs($deltay/($deltax + 0.5))] <= $kTan225}  {
	    # constrain to x-axis.
	    set y $yanch
	    return [list $x $y]
	} elseif {[expr abs($deltay/($deltax + 0.5))] >= $kTan675}  {
	    # constrain to y-axis.
	    set x $xanch
	    return [list $x $y]
	} else  { 
	
	    # Do the same analysis in the coordinate system rotated 45 degree CCW.
	    set deltaxprim [expr 1./sqrt(2.0) * ($deltax + $deltay)]
	    set deltayprim [expr 1./sqrt(2.0) * (-$deltax + $deltay)]
	    if {[expr abs($deltayprim/($deltaxprim + 0.5))] <= $kTan225}  {
		# constrain to x'-axis.
		set x [expr $xanch + ($deltax + $deltay)/2.0]
		set y [expr $yanch + $x - $xanch]
	    } else  {
		# constrain to y'-axis.
		set y [expr $yanch + (-$deltax + $deltay)/2.0]
		set x [expr $xanch - $y + $yanch]
	    }
	    return [list $x $y]
	}
    }
}

# ArcBegin, ArcDrag, ArcFin, ArcCancel --
#
#   First click sets center, second button press sets start point.
#   The tricky part is to choose one of the two possible solutions, CW or CCW.

proc ArcBegin { w x y {shift 0} }  {
    global  arcBox kRad2Grad statMess tcl_platform debugLevel
    
    if {$debugLevel >= 2}  {
	puts "ArcBegin:: x=$x, y=$y, shift=$shift"
    }
    if {![info exists arcBox($w,setcent)] || $arcBox($w,setcent) == 0}  {
	
	# First button press.
	set arcBox($w,center) [list $x $y]
	set arcBox($w,setcent) 1
	# Hack.
	if {$tcl_platform(platform) == "macintosh"}  {
	    $w create oval [expr $x - 2] [expr $y - 2] [expr $x + 3] [expr $y + 3]  \
		    -outline gray50 -fill {} -tag tcent
	    $w create line [expr $x - 5] $y [expr $x + 5] $y -fill gray50 -tag tcent
	    $w create line $x [expr $y - 5] $x [expr $y + 5] -fill gray50 -tag tcent 
	} else  {
	    $w create oval [expr $x - 3] [expr $y - 3] [expr $x + 3] [expr $y + 3]  \
		    -outline gray50 -fill {} -tag tcent
	    $w create line [expr $x - 5] $y [expr $x + 6] $y -fill gray50 -tag tcent
	    $w create line $x [expr $y - 5] $x [expr $y + 6] -fill gray50 -tag tcent 
	}
	focus $w
	bind $w <KeyPress-space> {ArcCancel %W}
	set statMess "Arc tool, click and drag"
    } else  {
	
	# If second button press, bind mouse motion.
	set cx [lindex $arcBox($w,center) 0]
	set cy [lindex $arcBox($w,center) 1]
	if {$shift}  {
	    set newco [ConstrainedDrag $x $y $cx $cy]
	    set x [lindex $newco 0]
	    set y [lindex $newco 1]
	    foreach {x y} $newco {}
	}
	set arcBox($w,first) [list $x $y]
	set arcBox($w,startAng) [expr $kRad2Grad * atan2($cy - $y, -($cx - $x))]
	set arcBox($w,extent) 0.0
	set r [expr hypot($cx - $x, $cy - $y)]
	set x1 [expr $cx + $r]
	set y1 [expr $cy + $r]
	set arcBox($w,co1) [list $x1 $y1]
	set arcBox($w,co2) [list [expr $cx - $r] [expr $cy - $r]]
	bind $w <B1-Motion> {ArcDrag %W [%W canvasx %x] [%W canvasy %y]}
	bind $w <Shift-B1-Motion> {ArcDrag %W [%W canvasx %x] [%W canvasy %y] 1}
	bind $w <ButtonRelease-1> {ArcFin %W [%W canvasx %x] [%W canvasy %y]}
	if {$debugLevel >= 2}  {
	    puts "ArcBegin:: r=$r, cx =$cx, cy=$cy, x1=$x1, y1=$y1, arcBox(w,startAng)=$arcBox($w,startAng)"
	}
    }
    catch {unset arcBox($w,last)}
}

proc ArcDrag { w x y {shift 0} }  {
    global  arcBox wCan myItpref itno kRad2Grad prefs

    # If constrained to 90/45 degrees.
    if {$shift}  {
	set cx [lindex $arcBox($w,center) 0]
	set cy [lindex $arcBox($w,center) 1]
	foreach {cx cy} $arcBox($w,center) {}
	set newco [ConstrainedDrag $x $y $cx $cy]
	set x [lindex $newco 0]
	set y [lindex $newco 1]
	foreach {x y} $newco {}
    }
    # Choose one of two possible solutions, either CW or CCW.
    # Make sure that the 'extent' angle is more or less continuous.
    set stopAng [expr $kRad2Grad *   \
      atan2([lindex $arcBox($w,center) 1] - $y, -([lindex $arcBox($w,center) 0] - $x))]
    set extentAng [expr $stopAng - $arcBox($w,startAng)]
    if {$arcBox($w,extent) - $extentAng > 180}  {
	set extentAng [expr $extentAng + 360]
    } elseif {$arcBox($w,extent) - $extentAng < -180}  {
	set extentAng [expr $extentAng - 360]
    }
    set arcBox($w,extent) $extentAng
    #puts "ArcDrag:: stopAng=$stopAng, extentAng=$extentAng"
    catch {$w delete $arcBox($w,last)}
    if {$prefs(fill) == 0}  {
	set theFill "-fill {}"
    } else  {
	set theFill "-fill $prefs(fgCol)"
    }
    set arcBox($w,last) [eval {$wCan create arc} $arcBox($w,co1)   \
      $arcBox($w,co2) {-start $arcBox($w,startAng) -extent $extentAng  \
      -width $prefs(penThick) -style $prefs(arcstyle) -outline $prefs(fgCol)  \
      -tags [list arc $myItpref/$itno]} $theFill]
}

proc ArcFin { w x y }  {
    global  arcBox prefs allIPnumsToSend ipNum2Socket itno myItpref   \
      statMess debugLevel

    if {$debugLevel >= 2}  {
	puts "ArcFin:: enter"
    }
    set statMess "Arc tool, click to set center, spacebar to cancel"
    bind $w <B1-Motion> {}
    bind $w <ButtonRelease-1> {}
    bind $w <KeyPress-space> {}
    catch {$w delete tcent}
    
    # Let all other clients know.
    if {$prefs(fill) == 0}  {
	set theFill "-fill {}"
    } else  {
	set theFill "-fill $prefs(fgCol)"
    }
    set cmd "create arc $arcBox($w,co1)   \
      $arcBox($w,co2) -start $arcBox($w,startAng) -extent $arcBox($w,extent)  \
      -width $prefs(penThick) -style $prefs(arcstyle) -outline $prefs(fgCol)  \
      -tags {arc $myItpref/$itno} $theFill"
    if {$debugLevel >= 2}  {
	puts "ArcFin:: cmd=$cmd"
    }
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) "CANVAS: $cmd"
    }
    incr itno
    catch {unset arcBox}
}

proc ArcCancel { w }  {
    global  arcBox statMess
    
    set statMess "Arc tool, click to set center, spacebar to cancel"
    catch {$w delete tcent}
    catch {unset arcBox}
}

# DoPaint --
#
#   Fills item with the foreground color. If 'shift', then transparent.

proc DoPaint  { w x y {shift 0}}  {
    global  prefs allIPnumsToSend ipNum2Socket kRad2Grad debugLevel

    if {$debugLevel >= 2}  {
	puts "DoPaint:: w=$w, x=$x, y=$y, shift=$shift"
    }
    # Find items overlapping x and y. Doesn't work for transparent items.
    #set ids [$w find overlapping $x $y $x $y]
    # This is perhaps not an efficient solution.
    set ids [$w find all]

    foreach id $ids {
	set theType [$w type $id]

	# Sort out uninteresting items early.
	if {([string compare $theType "rectangle"] != 0) &&   \
	  ([string compare $theType "oval"] != 0) &&  \
	  ([string compare $theType "arc"] != 0)}  {
	    continue
	}
	# Must be in bounding box.
	set theBbox [$w bbox $id]

	if {$x >= [lindex $theBbox 0] && $x <= [lindex $theBbox 2] &&  \
	  $y >= [lindex $theBbox 1] && $y <= [lindex $theBbox 3]}  {
	    # OK, inside!
	    # Allow privacy.
	    set theItno [CanvasGetItnoFrom $w $id]
	    if {$theItno == ""}  {
		continue
	    }
	    set cmd {}
	    if {[string compare $theType "rectangle"] == 0}  {
		if {$shift == 0}  {
		    set cmd "itemconfigure $theItno -fill $prefs(fgCol)"
		} elseif {$shift == 1}  {
		    set cmd "itemconfigure $theItno -fill {}"
		}
		eval $w $cmd
	    } elseif {[string compare $theType "oval"] == 0}  {
		
		# Use ellipsis equation (1 = x^2/a^2 + y^2/b^2) to find if inside.
		set centx [expr ([lindex $theBbox 0] + [lindex $theBbox 2])/2.0]
		set centy [expr ([lindex $theBbox 1] + [lindex $theBbox 3])/2.0]
		set a [expr abs($centx - [lindex $theBbox 0])]
		set b [expr abs($centy - [lindex $theBbox 1])]
		if {[expr ($x-$centx)*($x-$centx)/($a*$a) +   \
		  ($y-$centy)*($y-$centy)/($b*$b)] <= 1}  {
		    # Inside!
		    if {$shift == 0}  {
			set cmd "itemconfigure $theItno -fill $prefs(fgCol)"
		    } elseif {$shift == 1}  {
			set cmd "itemconfigure $theItno -fill {}"
		    }
		    eval $w $cmd
		}
	    } elseif {[string compare $theType "arc"] == 0}  {
		set theCoords [$w coords $id]
		set cx [expr ([lindex $theCoords 0] + [lindex $theCoords 2])/2.0]
		set cy [expr ([lindex $theCoords 1] + [lindex $theCoords 3])/2.0]
		set r [expr abs([lindex $theCoords 2] - [lindex $theCoords 0])/2.0]
		set rp [expr hypot($x - $cx, $y - $cy)]
		
		# Sort out point outside the radius of the arc.
		if {$rp > $r}  {
		    continue
		}
		set phi [expr $kRad2Grad * atan2(-($y - $cy),$x - $cx)]
		if {$phi < 0}  {
		    set phi [expr $phi + 360]
		}
		set startPhi [$w itemcget $id -start]
		set extentPhi [$w itemcget $id -extent]
		if {$extentPhi >= 0}  {
		    set phi1 $startPhi
		    set phi2 [expr $startPhi + $extentPhi]
		} else  {
		    set phi1 [expr $startPhi + $extentPhi]
		    set phi2 $startPhi
		}
		# Put branch cut at 360 degrees. Count CCW.
		if {$phi1 > 360}  {
		    set phi1 [expr $phi1 - 360]
		} elseif {$phi1 < 0}  {
		    set phi1 [expr $phi1 + 360]
		}
		if {$phi2 > 360}  {
		    set phi2 [expr $phi2 - 360]
		} elseif {$phi2 < 0}  {
		    set phi2 [expr $phi2 + 360]
		}
		set inside 0
		
		# Keep track of if the arc covers the branch cut or not.
		if {$phi2 > $phi1}  {
		    if {$phi >= $phi1 && $phi <= $phi2}  {
			set inside 1
		    }
		} else  {
		    if {$phi >= $phi1 || $phi <= $phi2}  {
			set inside 1
		    }
		}
		if {$inside}  {
		    if {$shift == 0}  {
			set cmd "itemconfigure $theItno -fill $prefs(fgCol)"
		    } elseif {$shift == 1}  {
			set cmd "itemconfigure $theItno -fill {}"
		    }
		    eval $w $cmd
		}
	    }	    
	    if {[string length $cmd] > 1}  {
		foreach ip $allIPnumsToSend  {
		    puts $ipNum2Socket($ip) "CANVAS: $cmd"
		}
	    }
	}
    }

}

# PolySetPoint, PolyDrag, PolyFinish --
#
#   Polygon drawing routines.

proc PolySetPoint  { w x y }  {
    global  thePoly

    #puts "PolySetPoint::"
    if {![info exists thePoly(0)]}  {
	# First point.
	catch {unset thePoly}
	set thePoly(N) 0
	set thePoly(0) [list $x $y]
    } elseif {[expr   \
      hypot([lindex $thePoly(0) 0] - $x, [lindex $thePoly(0) 1] - $y)] < 6}  {
	# If this point close enough to 'thePoly(0)', close polygon.
	PolyDrag $w [lindex $thePoly(0) 0] [lindex $thePoly(0) 1]
	set thePoly(last) {}
	incr thePoly(N)
	set thePoly($thePoly(N)) $thePoly(0)
	PolyFinish $w [lindex $thePoly(0) 0] [lindex $thePoly(0) 1]
	return
    } else  {
	set thePoly(last) {}
	incr thePoly(N)
	#set thePoly($thePoly(N)) [list $x $y]
	set thePoly($thePoly(N)) $thePoly(xy)
    }
    #puts "[parray thePoly]"
    # Let the latest line segment follow the mouse movements.
    focus $w
    bind $w <Motion> {PolyDrag %W [%W canvasx %x] [%W canvasy %y]}
    bind $w <Shift-Motion> {PolyDrag %W [%W canvasx %x] [%W canvasy %y] 1}
    bind $w <KeyPress-space> {PolyFinish %W [%W canvasx %x] [%W canvasy %y]}
}               

proc PolyDrag  { w x y {shift 0} }  {
    global  thePoly prefs

    # Move one end point of the latest line segment of the polygon.
    # If anchor not set just return.
    if {![info exists thePoly(0)]}  {
	return
    }
    catch {$w delete $thePoly(last)}
    # Vertical or horizontal.
    if {$shift}  {
	set anch $thePoly($thePoly(N))
	set newco [ConstrainedDrag $x $y [lindex $anch 0] [lindex $anch 1]]
	set x [lindex $newco 0]
	set y [lindex $newco 1]
	foreach {x y} $newco {}
    }
    # Keep track of last coordinates. Important for 'shift'.
    set thePoly(xy) [list $x $y]
    set thePoly(last) [eval {$w create line} $thePoly($thePoly(N))  \
      {$x $y -tags "polylines" -fill $prefs(fgCol)  \
      -width $prefs(penThick)}]
}

proc PolyFinish { w x y }  {
    global  thePoly myItpref itno prefs allIPnumsToSend ipNum2Socket
    
    #puts "[parray thePoly]"
    bind $w <Motion> {}
    bind $w <KeyPress-space> {}
    # If anchor not set just return.
    if {![info exists thePoly(0)]}  {
	return
    }
    # If too few segment.
    if {$thePoly(N) <= 1}  {
	$w delete polylines
	catch {unset thePoly}
	return
    }
    # Delete last line segment.
    catch {$w delete $thePoly(last)}
    
    # Find out if closed polygon or open line item. If closed, remove duplicate.
    set isClosed 0
    if {[expr   \
      hypot([lindex $thePoly(0) 0] - $x, [lindex $thePoly(0) 1] - $y)] < 4}  {
	set isClosed 1
	unset thePoly($thePoly(N))
	incr thePoly(N) -1
    }
    # Transform the set of lines to a polygon (or line) item.
    set coords {}
    for {set i 0} {$i <= $thePoly(N)} {incr i}  {
	append coords $thePoly($i) " "
    }
    $w delete polylines
    if {$prefs(fill) == 0}  {
	set theFill "-fill {}"
    } else  {
	set theFill "-fill $prefs(fgCol)"
    }
    if {$isClosed}  {
	set cmd "create polygon $coords -tags {poly $myItpref/$itno}  \
	  -outline $prefs(fgCol) $theFill -width $prefs(penThick)  \
	  -smooth $prefs(smooth) -splinesteps $prefs(splinesteps)"
    } else  {
	set cmd "create line $coords -tags {poly $myItpref/$itno}  \
	  -fill $prefs(fgCol) -width $prefs(penThick)  \
	  -smooth $prefs(smooth) -splinesteps $prefs(splinesteps)"
    }
    eval $w $cmd
    catch {unset thePoly}
    
    # Let al other clients know.
    foreach ip $allIPnumsToSend  {
	set s $ipNum2Socket($ip)
	puts $s "CANVAS: $cmd"
    }
    incr itno
}

# InitRotateItem, DoRotateItem, FinRotateItem --
#
#

proc InitRotateItem  { w x y }  {
    global  rotDrag 
    
    # Only one single selected item is allowed to be rotated.
    set id [$w find withtag selected]
    if {[llength $id] != 1}  {
	return
    }
    set it [CanvasGetItnoFrom $w $id]
    if {[string length $it] == 0}  {
	return
    }
    # Certain item types cannot be rotated.
    set rotDrag(type) [$w type $id]
    if {([string compare $rotDrag(type) "text"] == 0)}  {
	return
    }
    # Get center of gravity.
    if {([string compare $rotDrag(type) "arc"] == 0)}  {
	set colist [$w coords $id]
    } else  {
	set colist [$w bbox $id]
    }
    set rotDrag(cgX) [expr ([lindex $colist 0] + [lindex $colist 2])/2.0]
    set rotDrag(cgY) [expr ([lindex $colist 1] + [lindex $colist 3])/2.0]
    set rotDrag(anchorX) $x
    set rotDrag(anchorY) $y
    set rotDrag(id) $id
    set rotDrag(itno) $it
    set rotDrag(lastAng) 0.0
    if {([string compare $rotDrag(type) "arc"] == 0)}  {
	set rotDrag(arcStart) [$w itemcget $id -start]
    }
    # Save coordinates relative cg.
    set theCoords [$w coords $id]
    set rotDrag(n) [expr [llength $theCoords]/2]    ;# Number of points.
    set i 0
    foreach {cx cy} $theCoords  {
	set rotDrag(x,$i) [expr $cx - $rotDrag(cgX)]
	set rotDrag(y,$i) [expr $cy - $rotDrag(cgY)]
	incr i
    }
    # Observe coordinate system.
    set rotDrag(startAng) [expr atan2($y - $rotDrag(cgY),$x - $rotDrag(cgX)) ]
    #puts "InitRotateItem:: rotDrag=[parray rotDrag]"
}

proc DoRotateItem  { w x y {shift 0} }  {
    global  rotDrag kPI kRad2Grad prefs
    
    if {![info exists rotDrag]}  {
	return
    }
    set newAng [expr atan2($y - $rotDrag(cgY),$x - $rotDrag(cgX))]
    set deltaAng [expr $rotDrag(startAng) - $newAng]
    set new {}
    set angRot 0.0
    
    # Certain items are only rotated in 90 degree intervals, other continuously.
    if {([string compare $rotDrag(type) "line"] == 0) ||  \
      ([string compare $rotDrag(type) "polygon"] == 0) ||  \
      ([string compare $rotDrag(type) "arc"] == 0)}  {
	if {$shift}  {
	    if {!$prefs(45)}  {
		set angRot [expr ($kPI/2.0) * round($deltaAng/($kPI/2.0))]
	    } elseif {$prefs(45)}  {
		set angRot [expr ($kPI/4.0) * round($deltaAng/($kPI/4.0))]
	    }
	} else  {
	    set angRot $deltaAng
	}
    } elseif {([string compare $rotDrag(type) "rectangle"] == 0) ||  \
	([string compare $rotDrag(type) "oval"] == 0)}  {
	
	# Find the rotated angle in steps of 90 degrees.
	set angRot [expr ($kPI/2.0) * round($deltaAng/($kPI/2.0))]
    }
    # Find the new coordinates; arc: only start angle.
    if {[expr abs($angRot)] > 1e-4 ||   \
      [expr abs($rotDrag(lastAng) - $angRot)] > 1e-4}  {
	set sinAng [expr sin($angRot)]
	set cosAng [expr cos($angRot)]
	if {[string compare $rotDrag(type) "arc"] == 0}  {
	    
	    # Different coordinate system for arcs...and units...
	    $w itemconfigure $rotDrag(id) -start   \
	      [expr $kRad2Grad * $angRot + $rotDrag(arcStart)]
	} else  {
	    # Compute new coordinates from the original ones.
	    for {set i 0} {$i < $rotDrag(n)} {incr i}  {
		lappend new [expr $rotDrag(cgX) + $cosAng * $rotDrag(x,$i) +  \
		  $sinAng * $rotDrag(y,$i)]
		lappend new [expr $rotDrag(cgY) - $sinAng * $rotDrag(x,$i) +  \
		  $cosAng * $rotDrag(y,$i)]
	    }
	    #puts "DoRotateItem:: new=$new"
	    eval $w coords $rotDrag(id) $new
	}
    }
    set rotDrag(lastAng) $angRot
}

proc FinRotateItem  { w x y }  {
    global  rotDrag allIPnumsToSend ipNum2Socket kRad2Grad
        
    if {![info exists rotDrag]}  {
	return
    }
    # Move all markers along.
    $w delete id$rotDrag(id)
    MarkBbox $w 0 $rotDrag(id)
    if {[string compare $rotDrag(type) "arc"] == 0}  {
	
	# Get new start angle.
	set cmd "itemconfigure $rotDrag(itno) -start   \
	      [$w itemcget $rotDrag(itno) -start]"
    } else  {
	# Or update all coordinates.
	set cmd "coords $rotDrag(itno) [$w coords $rotDrag(id)]"
    }
    #puts "FinRotateItem:: cmd=$cmd"
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) "CANVAS: $cmd"
    }
    catch {unset rotDrag}
}
	    
# DeleteItem --
#
#   Delete item in canvas 'w' with coordinates 'x' and 'y'.
#   'id' can be "current", "selected", "movie" or just an id number.

proc DeleteItem	 { w x y {id current} {moviePath {}} }  {
    global  debugLevel allIPnumsToSend ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "DeleteItem:: w=$w, x=$x, y=$y, id=$id, moviePath=$moviePath"
    }
    
    # Get item.
    if {$id == "current"}  {
	set theItno [CanvasGetItnoFrom $w current]
	if {$theItno == ""}  {
	    return
	}
	$w delete current
    } elseif {$id == "selected"}  {
	
	# First, get canvas objects with tag 'selected'.
	
	set ids [$w find withtag selected]
	if {[llength $ids] == 0}  {
	    return
	}
	set theItno ""
	foreach id $ids  {
	    set it [CanvasGetItnoFrom $w $id]
	    if {$it != ""}  {
		lappend theItno $it
		$w delete $it
	    }
	}
	# Remove select marks.
	DeselectAll $w
    } elseif {$id == "movie"}  {
	
	# Movie.
	
	set id [lindex [$w find closest $x $y 3] 0]
	set theItno [CanvasGetItnoFrom $w $id]
	#puts "   id=$id, theItno=$theItno"
	if {$theItno == ""}  {
	    return
	}
	# Delete both the window item and the movie.
	$w delete $id
	catch {destroy ${moviePath}.m}
    } else  {
	
	# 'id' is an actual item number.
	set theItno [CanvasGetItnoFrom $w $id]
	if {$theItno == ""}  {
	    return
	}
	$w delete $theItno
    }
    if {$theItno == ""}	  {
	return
    }
    foreach ip $allIPnumsToSend  {
        set s $ipNum2Socket($ip)
	foreach it $theItno  {
	    puts $s "CANVAS: delete $it"
	}
    }
}

proc FindTypeFromOverlapping { c x y type }  {    
    set ids [$c find overlapping [expr $x-2] [expr $y-2]  \
      [expr $x+2] [expr $y+2]]
    set id {}
    # Choose the first item with tags $type.
    foreach i $ids  {
	if {[lsearch [$c gettags $i] $type] >= 0}  {
	    # Found "$type".
	    set id $i
	    break
	}
    }
    #puts "FindTypeFromOverlapping:: ids=$ids, id=$id"
    return $id
}

# CanvasFocus --
#
#   Puts a text insert bar in the canvas. If already text item under the mouse
#   then give focus to that item. If 'forceNew', then always make a new item.

proc CanvasFocus { c x y {forceNew 0} }	{
    global  prefs itno myItpref debugLevel allIPnumsToSend ipNum2Socket   \
      fontSize2Points
    
    focus $c
    set id [FindTypeFromOverlapping $c $x $y "text"]
    #puts "CanvasFocus:: id=$id"
    if {($id == {}) || ([$c type $id] != "text") || $forceNew}  {
	
	# No text item under cursor, make a new empty text item.
	set cmd [list create text $x $y -text {} -tags [list text $myItpref/$itno] \
	  -anchor nw -fill $prefs(fgCol) \
	  -font [list $prefs(font) $fontSize2Points($prefs(fontSize))   \
	  $prefs(fontWeight)]]
	if {$debugLevel >= 2}  {
	    puts "$c $cmd"
	}
	set t [eval $c $cmd]
	$c focus $t
	$c select clear
	$c icursor $t 0
	
	# If 'useHtmlSizes', then transport the html sizes instead of point sizes.
	if {$prefs(useHtmlSizes)}  {
	    set cmd [list create text $x $y -text {}   \
	      -tags [list text $myItpref/$itno]  \
	      -anchor nw -fill $prefs(fgCol)  \
	      -font [list $prefs(font) $prefs(fontSize) $prefs(fontWeight)]]
	}
	foreach ip $allIPnumsToSend  {
	    set s $ipNum2Socket($ip)
	    puts $s "CANVAS: $cmd"
	}
	incr itno
    }
}

proc CanvasTextHit { c x y {select 1} }	 {
    
    $c focus current
    $c icursor current @$x,$y
    $c select clear
    $c select from current @$x,$y
}

proc CanvasTextDrag { c x y }  {
    global  tcl_platform
    
    DeselectAll $c
    $c select to current @$x,$y
    # Mac text bindings.????
    if {$tcl_platform(platform) == "macintosh"}  {
	#$c focus
    }
    # menus
    FixMenusWhenSelection $c
}

proc CanvasTextSelectWord { c x y }  {
    global  tcl_platform
    
    DeselectAll $c
    $c focus current
    # Mac text bindings.????
    if {$tcl_platform(platform) == "macintosh" ||   \
      $tcl_platform(platform) == "unix" ||  \
      $tcl_platform(platform) == "windows"}  {
	set id [$c find withtag current]
	if {$id == ""}  {
	    return
	}
	if {[$c type $id] != "text"}  {
	    return
	}
	set txt [$c itemcget $id -text]
	set ind [$c index $id @$x,$y]
	#puts "CanvasTextSelectWord:: txt=$txt, ind=$ind"
	# Find the boundaries of the word and select word.
	$c select from $id [string wordstart $txt $ind]
	$c select adjust $id [expr [string wordend $txt $ind] - 1]
	# menus
	FixMenusWhenSelection $c
    }    
}

proc CanvasTextDelete { c }	 {
    global	debugLevel allIPnumsToSend ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextDelete"
    }
    set idfocus [$c focus]
    if {[$c select item] != {}}	 {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set sfirst [$c index $idfocus sel.first]
	set slast [$c index $idfocus sel.last]
	$c dchars [$c select item] sel.first sel.last
	if {$debugLevel >= 2}  {
	    puts "sfirst: $sfirst, slast: $slast"
	}
        foreach ip $allIPnumsToSend  {
            set s $ipNum2Socket($ip)
	    puts $s [list "CANVAS:" dchars $theItno $sfirst $slast]
	}
    } elseif {$idfocus != {}}  {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set ind [expr [$c index $idfocus insert] - 1]
	$c dchars $idfocus $ind
	if {[llength $allIPnumsToSend] > 0}	{
	    # Find the 'itno'.
	    set theItno [CanvasGetItnoFrom $c focus]
	    if {$debugLevel >= 2}  {
		puts "theItno: $theItno, ind: $ind"
	    }
	    if {$theItno == ""}	 {
		return
	    }
	    foreach ip $allIPnumsToSend  {
		set s $ipNum2Socket($ip)
		puts $s [list "CANVAS:" dchars $theItno $ind]
	    }
	}
    }
}

proc CanvasDelChar { c }  {
    #puts "CanvasDelChar::"
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}  {
	return
    }
    set idfocus [$c focus]
    if {$idfocus != {}}	 {
	$c dchars $idfocus [expr [$c index $idfocus insert] -1]
    }
}

# CanvasTextBackSpace --
#
#   A backspace if selected text deletes that text.
#   A backspace if text item has focus deletes text left of insert cursor.

proc CanvasTextBackSpace { c }	{
    global  debugLevel allIPnumsToSend ipNum2Socket
    
    # Just return if any *items* selected.
    if {[llength [$c find withtag selected]] > 0}  {
	return
    }
    set selit [$c select item]
    if {$selit != {}} {
	set theItno [CanvasGetItnoFrom $c $selit]
	if {$theItno == ""}  {
	    return
	}
	
	# Be sure to get indices before deleting ;-)
	set indFirst [$c index $theItno sel.first]
	set indLast [$c index $theItno sel.last]
	$c dchars [$c select item] sel.first sel.last
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) [list "CANVAS:" dchars $theItno $indFirst $indLast]
	}
    } else  {
	set theItno [CanvasGetItnoFrom $c focus]
	if {$theItno == ""}  {
	    return
	}
	set t [$c focus]
	$c icursor $t [expr [$c index $t insert] -1]
	set indInsert [$c index $theItno insert]
	$c dchars $t insert
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) [list "CANVAS:" dchars $theItno $indInsert]
	}
    }
}

proc CanvasTextErase { c }	{
    #puts "CanvasTextErase"
    $c delete [$c focus]
}

proc CanvasNewLine { c }  {
    global  debugLevel allIPnumsToSend ipNum2Socket
    
    set nl_ "\\n"
    $c insert [$c focus] insert \n
    # Find the 'itno' .
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}	 {
	return
    }
    set ind [expr [$c index [$c focus] insert] - 1]
    if {$debugLevel >= 2}  {
	puts "theItno = $theItno, ind = $ind"
	puts "CANVAS: insert $theItno $ind $nl_"
    }
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) [list "CANVAS:" insert $theItno $ind $nl_]
    }
}

# CanvasGetItnoFrom --
#   
#   Finds the specific item identifier from 'fromWhat'.
#   If 'privacy' then returns empty string of not item made here or
#   if made by xxx...
#   fromWhat = "current": picks the current item
#   fromWhat = "focus": picks the item that has focus
#   fromWhat = canvas id number: takes the specified canvas id
#   If 'force' then do not respect privacy.

proc CanvasGetItnoFrom { c fromWhat {force 0}}  {
    global	myItpref prefs debugLevel
    
    # Find the 'itno'.
    set pre_ {[^/ ]+}
    set digit_ {[0-9]}
    set wild_ {[xX]+}
    if {[string compare $fromWhat "current"] == 0}  {
	set tcurr [$c gettags current]
    } elseif {[string compare $fromWhat "focus"] == 0}  {
	set tcurr [$c gettags [$c focus]]
    } else {
	set tcurr [$c gettags $fromWhat]
    }
    #puts "CanvasGetItnoFrom:: tcurr=$tcurr"
    if {$tcurr == ""}  {
	return {}
    }
    if {$prefs(privacy) && !$force}  {
	if {[regexp "(($wild_|$myItpref)/$digit_+)" "$tcurr" theItno] == 0}  {
	    return {}
	}
    } else  {
	if {[regexp "(^| )($pre_/$digit_+)" "$tcurr" junk junk2 theItno] == 0}  {
	    return {}
	}
    }
    return $theItno
}

# CanvasTextInsert --
#
#   Inserts text string 'char' at the insert point of the text item
#   with focus. Handles newlines as well.

proc CanvasTextInsert { c char }  {
    global  debugLevel allIPnumsToSend ipNum2Socket tcl_platform prefs
        
    set punct {[.,;?!]}
    set nl_ "\\n"
    
    # First, find out if there are any text item with focus.
    # If not, then make one.
    if {[llength [$c focus]] == 0}  {
	
    }
    # Find the 'itno'.
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == "" || $char == ""}	 {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextInsert:: theItno == {}"
	}
	return
    }
    # The index of the insertion point.
    set ind [$c index [$c focus] insert]

    # Mac text bindings: delete selection before inserting.
    if {$tcl_platform(platform) == "macintosh"}  {
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		CanvasTextDelete $c
		selection clear
	    }
	}
    }
    # The actual canvas text insertion; note that 'ind' is found before.
    $c insert [$c focus] insert $char
    if {$debugLevel >= 3}  {
	puts "CanvasTextInsert:: theItno = $theItno, ind = $ind, char: $char"
    }
    # If speech, speech last sentence if finished.
    if {$prefs(TclSpeech) && $prefs(TclSpeechOn)}  {
	if {[string match *${punct}* $char]}  {
	    speak [$c itemcget $theItno -text]
	}
    }
    # Need to treat the case with actual newlines in char string.
    # Write to all other clients; need to make a one liner first.
    if {1 || [llength $allIPnumsToSend] > 0}  {
	regsub -all "\n" $char $nl_ oneliner
	if {$debugLevel >= 3}  {
	    puts "CanvasTextInsert:: [list "CANVAS:" insert $theItno $ind $oneliner]"
	}
    }
    foreach ip $allIPnumsToSend  {
        set s $ipNum2Socket($ip)
	puts $s [list "CANVAS:" insert $theItno $ind $oneliner]
    }	
}

proc CanvasTextMoveRight { c }	{
    global  tcl_platform
    
    set foc [$c focus]
    
    # Mac text bindings: remove selection then move insert to end.
    if {$tcl_platform(platform) == "macintosh" ||  \
      $tcl_platform(platform) == "windows"}  {
	# If selection.
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		$c icursor $foc [expr [$c index $foc sel.last] + 1]
		$c select clear
	    }
	} else  {
	    $c icursor $foc [expr [$c index $foc insert] + 1]
	}
    } else  {
	$c icursor $foc [expr [$c index $foc insert] + 1]
    }
}

proc CanvasTextMoveLeft { c }  {
    global  tcl_platform
    
    set foc [$c focus]
    
    # Mac text bindings: remove selection then move insert to first.
    if {$tcl_platform(platform) == "macintosh" ||  \
      $tcl_platform(platform) == "windows"}  {
	# If selection.
	if {![catch {selection get} s]}  {
	    if {[llength $s] > 0}  {
		$c icursor $foc [expr [$c index $foc sel.first] + 0]
		$c select clear
	    }
	} else  {
	    $c icursor $foc [expr [$c index $foc insert] - 1]
	}
    } else  {
	$c icursor $foc [expr [$c index $foc insert] - 1]
    }
}

proc CanvasTextMoveUpOrDown { c upOrDown }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextMoveUpOrDown:: upOrDown=$upOrDown"
    }
    set foc [$c focus]
    # Find index of new character. Only for left justified text.
    set ind [$c index $foc insert]
    set theText [$c itemcget $foc -text]

    if {[string compare $upOrDown "up"] == 0}  {
	
	# Up one line. String operations.
	set indPrevNL [string last \n $theText [expr $ind - 1]]
	set indPrev2NL [string last \n $theText [expr $indPrevNL - 1]]
	#puts "indPrevNL=$indPrevNL, indPrev2NL=$indPrev2NL"
	# If first line.
	if {$indPrevNL == -1}  {
	    return
	}
	set ncharLeft [expr $ind - $indPrevNL - 1]
	set newInd [min [expr $indPrev2NL + $ncharLeft + 1] $indPrevNL]
	#puts "ncharLeft=$ncharLeft, newInd=$newInd"
	$c icursor $foc $newInd
    } else  {
	
	# Down one line.
	set indPrevNL [string last \n $theText [expr $ind - 1]]
	set indNextNL [string first \n $theText [expr $indPrevNL + 1]]
	set indNext2NL [string first \n $theText [expr $indNextNL + 1]]
	# If last line.
	if {$indNextNL == -1}  {
	    return
	}
	set ncharLeft [expr $ind - $indPrevNL - 1]
	if {$indNext2NL == -1}  {
	    # Move to last line.
	    set newInd [expr $indNextNL + $ncharLeft + 1]
	} else  {
	    set newInd [min [expr $indNextNL + $ncharLeft + 1] $indNext2NL]
	}
	#puts "indPrevNL=$indPrevNL, indNextNL=$indNextNL, indNext2NL=$indNext2NL,  \
	#  ncharLeft=$ncharLeft, newInd=$newInd"
	$c icursor $foc $newInd
    }
}

# StrokeBegin, StrokeDrag, StrokeEnd --
#
#   Handles drawing of an arbitrary line. Uses global 'stroke' variable
#   to store all intermediate points on the line, and stroke(N) to store
#   the number of such points. If 'thick'=-1, then use 'prefs(penThick)',
#   else use the 'thick' argument as line thickness.

proc StrokeBegin { w x y }	{
    global  stroke
    
    catch {unset stroke}
    set stroke(N) 0
    set stroke(0) [list $x $y]
}

proc StrokeDrag { w x y {thick -1} }  {
    global  stroke prefs itno myItpref
    
    # If stroke not set just return.
    if {![info exists stroke(N)]}  {
	return
    }
    set coords $stroke($stroke(N))
    lappend coords $x $y
    incr stroke(N)
    set stroke($stroke(N)) [list $x $y]
    if {$thick == -1}  {
	set thisThick $prefs(penThick)
    } else {
	set thisThick $thick
    }
    eval {$w create line} $coords  \
	    {-tags [list segments $myItpref/$itno] -fill $prefs(fgCol)  \
	    -width $thisThick}
}

proc StrokeEnd { w x y {thick -1} }	 {
    global	stroke prefs itno myItpref debugLevel allIPnumsToSend ipNum2Socket
    
    # If stroke not set just return.
    set coords {}
    if {![info exists stroke(N)]}  {
	return
    }
    for {set i 0} {$i <= $stroke(N)} {incr i}  {
	append coords $stroke($i) " "
    }
    $w delete segments
    if {[llength $coords] <= 2}  {
	return
    }
    if {$thick == -1}  {
	set thisThick $prefs(penThick)
    } else {
	set thisThick $thick
    }
    set cmd "create line $coords  \
      -tags {line $myItpref/$itno} -joinstyle round  \
      -smooth $prefs(smooth) -splinesteps $prefs(splinesteps) \
      -fill $prefs(fgCol) -width $thisThick"
    if {$debugLevel >= 2}  {
	puts "StrokeEnd:: cmd=$cmd"
    }
    eval $w $cmd
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) "CANVAS: $cmd"
    }
    incr itno
    catch {unset stroke}
}

# LineBegin, LineDrag, LineEnd --
# 
#   Handles drawing of a straight line. Uses global 'theLine' variable
#   to store anchor point and end point of the line.
#   If 'shift' constrain the line to be vertical or horizontal.
#   If 'opt'=arrow draw an arrow at the final line end.

proc LineBegin { w x y {opt 0} }  {
    global  theLine
    
    set theLine($w,anchor) [list $x $y]
    catch {unset theLine($w,last)}
}

proc LineDrag { w x y shift {opt 0} }	 {
    global  theLine itno myItpref prefs
    
    # If anchor not set just return.
    if {![info exists theLine($w,anchor)]}  {
	return
    }

    catch {$w delete $theLine($w,last)}
    if {$opt == "arrow"}  {
	set extras [list -arrow last]
    } else	{
	set extras ""
    }
    # Verical or horizontal.
    if {$shift}  {
	set newco [ConstrainedDrag $x $y [lindex $theLine($w,anchor) 0]  \
	  [lindex $theLine($w,anchor) 1]]
	set x [lindex $newco 0]
	set y [lindex $newco 1]
	foreach {x y} $newco {}
    }
    set theLine($w,last) [eval {$w create line} $theLine($w,anchor)  \
      {$x $y -tags "line $myItpref/$itno" -fill $prefs(fgCol)  \
      -width $prefs(penThick)} $extras]
}

proc LineEnd { w x y shift {opt 0}}  {
    global  itno myItpref theLine prefs ipNum2Socket allIPnumsToSend
    
    # If anchor not set just return.
    if {![info exists theLine($w,anchor)]}  {
	return
    }
    if {$opt == "arrow"}  {
	set extras [list -arrow last]
    } else	{
	set extras ""
    }
    # Verical or horizontal.
    if {$shift}  {
	set newco [ConstrainedDrag $x $y [lindex $theLine($w,anchor) 0]  \
	  [lindex $theLine($w,anchor) 1]]
	set x [lindex $newco 0]
	set y [lindex $newco 1]
	foreach {x y} $newco {}
    }
    if {[llength $allIPnumsToSend] > 0}  {
	set cmd "create line $theLine($w,anchor) $x $y	\
	  -tags {line $myItpref/$itno} -joinstyle round	\
	  -smooth true -fill $prefs(fgCol) -width $prefs(penThick) $extras"
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) "CANVAS: $cmd"
	}
    }
    incr itno
    catch {unset theLine}
}

proc DoUndo  {  }  {
    tk_messageBox -message "Chicken!!! \nUndo is not for real men." \
      -icon info -type ok
}

proc SelectAll { w }  {
    global  itno myItpref debugLevel
    
    set ids [$w find all]
    if {$debugLevel >= 2}  {
	puts "SelectAll:: ids=$ids"
    }
    foreach id $ids {
	MarkBbox $w 1 $id
    }
}

proc DeselectAll { w }  {
    $w delete withtag tbbox
    $w dtag all selected
    # menus
    FixMenusWhenSelection $w
    .menu.edit entryconfigure *Raise* -state disabled
    .menu.edit entryconfigure *Lower* -state disabled
}

# MarkBbox --
#
#   Makes four tiny squares at the corners of the specified items.
#   'which' can either be "current" or an id. 
#   If 'shift', then just select item, else deselect all other first.

proc MarkBbox { w shift {which current}}  {
    global	prefs kGrad2Rad debugLevel
    
    if {$debugLevel >= 2}  {
	puts "MarkBbox (entry):: w=$w, shift=$shift, which=$which"
    }
    set a $prefs(aBBox)
    # If no shift key, deselect all.
    if {$shift == 0}  {
	DeselectAll $w
    }
    if {$which == "current"}  {
	set thebbox [$w bbox current]
    } else {
	set thebbox [$w bbox $which]
    }
    if {[llength $thebbox] == 0}  {
	return
    }
    if {$which == "current"}  {
	set itno [CanvasGetItnoFrom $w current]
	set id [$w find withtag current]
    } else  {
	set itno [CanvasGetItnoFrom $w $which]
	set id $which
    }
    if {[llength $itno] == 0 || [llength $id] == 0}  {
	return
    }
    # Movies may not be selected this way; temporary solution?
    if {[lsearch [$w gettags $itno] "movie"] >= 0}  {
	return
    }
    if {$debugLevel >= 2}  {
	puts "MarkBbox:: itno=$itno, id=$id"
    }
    # Add tag 'selected' to the selected item. Indicate to which item id
    # a marker belongs with adding a tag 'id$id'.
    set theType [$w type $which]
    $w addtag "selected" withtag $itno
    set theMarkTags "tbbox id$id"
    
    # If mark the bounding box. Also for all "regular" shapes.
    if {$prefs(bboxOrCoords) || ($theType == "oval") || ($theType == "text")  \
      || ($theType == "rectangle") || ($theType == "image")  \
      || ($theType == "window")}  {
	set x1 [lindex $thebbox 0]
	set y1 [lindex $thebbox 1]
	set x2 [lindex $thebbox 2]
	set y2 [lindex $thebbox 3]
	foreach {x1 y1 x2 y2} $thebbox {}
	$w create rect [expr $x1-$a] [expr $y1-$a] [expr $x1+$a] [expr $y1+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x1-$a] [expr $y2-$a] [expr $x1+$a] [expr $y2+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x2-$a] [expr $y1-$a] [expr $x2+$a] [expr $y1+$a] \
	  -tags $theMarkTags -fill white
	$w create rect [expr $x2-$a] [expr $y2-$a] [expr $x2+$a] [expr $y2+$a] \
	  -tags $theMarkTags -fill white
    } else  {
	
	set theCoords [$w coords $which]
	if {[string length $theCoords] == 0}  {
	    return
	}
	set n [llength $theCoords]
	# For an arc item, mark start and stop endpoints.
	# Beware, mixes of two coordinate systems, y <-> -y.
	if {[string compare $theType "arc"] == 0}  {
	    if {$n != 4}  {
		return
	    }
	    set x1 [lindex $theCoords 0]
	    set y1 [lindex $theCoords 1]
	    set x2 [lindex $theCoords 2]
	    set y2 [lindex $theCoords 3]
	    foreach {x1 y1 x2 y2} $theCoords {}
	    set r [expr abs(($x1 - $x2)/2.0)]
	    set cx [expr ($x1 + $x2)/2.0]
	    set cy [expr ($y1 + $y2)/2.0]
	    set startAng [$w itemcget $id -start]
	    set extentAng [$w itemcget $id -extent]
	    set xstart [expr $cx + $r * cos($kGrad2Rad * $startAng)]
	    set ystart [expr $cy - $r * sin($kGrad2Rad * $startAng)]
	    set xfin [expr $cx + $r * cos($kGrad2Rad * ($startAng + $extentAng))]
	    set yfin [expr $cy - $r * sin($kGrad2Rad * ($startAng + $extentAng))]
	    $w create rect [expr $xstart-$a] [expr $ystart-$a]   \
	      [expr $xstart+$a] [expr $ystart+$a] -tags $theMarkTags -fill white
	    $w create rect [expr $xfin-$a] [expr $yfin-$a]   \
	      [expr $xfin+$a] [expr $yfin+$a] -tags $theMarkTags -fill white
	    
	} else  {
	    # Mark each coordinate. {x0 y0 x1 y1 ... }
	    for {set i 0} {$i < $n} {incr i 2}  {
		set x [lindex $theCoords $i]
		set y [lindex $theCoords [expr $i + 1]]
		$w create rect [expr $x-$a] [expr $y-$a] [expr $x+$a] [expr $y+$a] \
		  -tags $theMarkTags -fill white
	    }
	}
    }
    # Bind key to be able to delete selected item.
    #bind . <BackSpace> "DeleteItem $w %x %y selected"
    #bind . <Control-Key-d> "DeleteItem $w %x %y selected"
    focus $w
    
    # Enable cut and paste.
    FixMenusWhenSelection $w
    .menu.edit entryconfigure *Raise* -state normal
    .menu.edit entryconfigure *Lower* -state normal
}

proc RaiseOrLowerItems { w {what raise} }  {
    global  prefs allIPnumsToSend ipNum2Socket
    
    if {$what == "raise"}  {
	set doWhat $what
	set which all
    } elseif {$what == "lower"}  {
	if {$prefs(canGridOn)}  {
	    set doWhat "raise"   
	    set which grid
	} else  {
	    set doWhat $what
	    set which all
	}
    }
    $w $doWhat selected $which
    $w $doWhat tbbox $which
    
    # if connected
    if {[llength $allIPnumsToSend] > 0}  {
	# first, get canvas objects with tag 'selected'
	set ids [$w find withtag selected]
	set theItno {}
	foreach id $ids  {
	    lappend theItno [CanvasGetItnoFrom $w $id]
	}
	foreach ip $allIPnumsToSend  {
	    set s $ipNum2Socket($ip)
	    foreach it $theItno  {
		puts $s "CANVAS:  $what $it all"
	    }   
	}
    }
}

# CopySelectedToClipboard --
#
#   Copies the selection, either complete items or pure text, to the clipboard.
#   If there are no selected items, pure text is copied.
#   Set a flag 'clipToken' to tell which; "string" or "item".
#   The items are copied one by one using 'CopySingleItemToClipboard'.
#   doWhat: "cut" or "copy".
 
proc CopySelectedToClipboard  { w doWhat }  {
    global  debugLevel clipItemSep clipToken
    
    if {$debugLevel >= 2}  {
	puts "CopySelectedToClipboard:: w=$w, doWhat=$doWhat"
	puts "CopySelectedToClipboard:: focus=[focus], class=[winfo class [focus]]"
    }
    set wClass [winfo class $w]
    clipboard clear
    
    # Find out from where we copied: Entry
    if {[string compare $wClass "Entry"] == 0}  {
	set wsel [selection own]
	if {[catch {selection get} sel]}  {
	    return
	}
	clipboard append $sel
	set clipToken "string"
	if {[string compare $doWhat "cut"] == 0}  {
	    $w delete sel.first sel.last
	}
    } else {
	
	# Assume for the moment that we have copied from the Canvas.
	# First, get canvas objects with tag 'selected'.
	set ids [$w find withtag selected]	
	
	# If selected text within text item.
	if {[llength $ids] == 0}  {
	    CanvasTextCopy $w
	    if {[string compare $doWhat "cut"] == 0}  {
		CanvasTextDelete $w
	    }
	    set clipToken "string"
	} else {
	    # Loop over all selected items, use 'clipItemSep' as separator.
	    foreach id $ids  {
		CopySingleItemToClipboard $w $doWhat $id
		if {[lindex $ids end] != $id}  {
		    clipboard append $clipItemSep
		}
	    }
	    set clipToken "item"
	}
    }
    # Fix menus.
    .menu.edit entryconfigure *Paste* -state normal
}

# CopySingleItemToClipboard --
#
#   Copies the item given by 'id' to the clipboard.
#   doWhat: "cut" or "copy".

proc CopySingleItemToClipboard  { w doWhat id }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CopySingleItemToClipboard:: id=$id"
    }
    if {[llength $id] == 0}  {
	return
    }
    set theTags [$w gettags $id]

    # Get all actual options.
    set opcmd [CanvasItemConfigure $w $id]
    
    # Strip off options that are irrelevant; is helpful for other clients with
    # version numbers lower than this if they don't understand new options.
    set opcmd [CanvasStripItemOptions $opcmd]
    set itemType [$w type $id]
    set co [$w coords $id]
    set cmd [concat "create" $itemType $co $opcmd]
    if {$debugLevel >= 2}  {
	puts "CopySingleItemToClipboard:: cmd=$cmd"
    }
    
    # Copy the canvas object to the clipboard.
    clipboard append $cmd
    
    # If cut then delete items.
    if {$doWhat == "cut"}	 {
	DeleteItem $w 0 0 $id
	$w delete withtag tbbox
    } elseif {$doWhat == "copy"}	 {
	
    }
    .menu.edit entryconfigure *Paste* -state normal
}

# CanvasItemConfigure --
#
#   As canvas itemconfigure but only the actual options.

proc CanvasItemConfigure { w id }  {
    set opcmd {}
    set opts [$w itemconfigure $id]
    foreach opt $opts {
	set op [lindex $opt 0]
	set val [lindex $opt 4]
	lappend opcmd $op $val
    }
    return $opcmd
}

# CanvasStripItemOptions
#
#       Takes a list of '-option value' pairs and discards options that doesn't
#       make a difference, such as empty lists, zeros etc.
#       
# Arguments:
#       optList      the list of pairs '-option value'
# Results:
#       The modified '-option value' list.

proc CanvasStripItemOptions { optList } {
    
    array set opts $optList
    foreach name [array names opts] {
	
	# First, discard if empty list.
	
	if {[llength $opts($name)] == 0} {
	    unset opts($name)
	}
	
	# Pick options that can be discarded if zero.
	
	if {([string compare $name "-disabledwidth"] == 0) ||  \
	  ([string compare $name "-activewidth"] == 0) ||  \
	  ([string compare $name "-dashoffset"] == 0)} {
	    if {$opts($name) == 0} {
		unset opts($name)
	    }
	}
	if {([string compare $name "-offset"] == 0) ||  \
	  ([string compare $name "-outlineoffset"] == 0)} {
	    if {[string compare $opts($name) "0,0"] == 0} {
		unset opts($name)
	    }
	}
	
	# We take the opportunity to fix a bug(?) in 8.3.
	
	if {[string compare $name "-smooth"] == 0} {
	    if {[string compare $opts(-smooth) "bezier"] == 0} {
		set opts(-smooth) 1
	    }
	}
    }
    
    # And get back the modified list to return.
    return [array get opts]
}

# CanvasTextCopy --
#  
#   Just copies text from text items. If selected text, copy that,
#   else if text item has focus copy complete text item.

proc CanvasTextCopy { c }  {
    global  debugLevel
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextCopy::"
    }
    if {[$c select item] != {}}	 { 
	clipboard clear
	set t [$c select item]
	set text [$c itemcget $t -text]
	set start [$c index $t sel.first]
	set end [$c index $t sel.last]
	clipboard append [string range $text $start $end]
	if {$debugLevel >= 2}  {
	    puts "CanvasTextCopy:: select item] != {}, text=$text"
	}
    } elseif {[$c focus] != {}}	 {
	clipboard clear
	set t [$c focus]
	set text [$c itemcget $t -text]
	clipboard append $text
	if {$debugLevel >= 2}  {
	    puts "CanvasTextCopy:: focus] != {}, text=$text"
	}
    }
}

proc PasteFromClipboardTo { w }  {
    global  wCan
    
    set wClass [winfo class $w]
    #puts "PasteFromClipboard:: wClass=$wClass"
    if {[string compare $wClass "Canvas"] == 0}  {
	PasteFromClipboardToCanvas $w
    } elseif {[string compare $wClass "Wish"] == 0}  {
	# We assume that it is the canvas that should receive this.
	PasteFromClipboardToCanvas $wCan
    } elseif {[string compare $wClass "Entry"] == 0}  {
	event generate $w <<Paste>>
    } else {
	
	# Wild guess...
	event generate $w <<Paste>>
    }
}

# PasteFromClipboardToCanvas --
#
#   Depending on 'clipToken', either paste simple text string, or complete item(s).
#   Items are pasted one by one using 'PasteSingleFromClipboardToCanvas'.

proc PasteFromClipboardToCanvas { w }  {
    global  debugLevel clipItemSep clipToken
    
    $w delete withtag tbbox
    $w dtag all selected
    
    # Pick apart the clipboard content with the 'clipItemSep' separator.
    if {[catch {selection get -sel CLIPBOARD} cmds]}  {
	return
    }
    if {$debugLevel >= 2}  {
	puts "PasteFromClipboardToCanvas:: clipToken=$clipToken, cmds=$cmds"
    }
    $w delete withtag tbbox
    
    # Try to figure out if pute text string (clipToken="string") or complete
    # canvas create item command (clipToken="item").
    set tmpCmds $cmds
    # Check first if it has the potential of a canvas command.
    if {[regexp  "^create " $cmds]}  {
	set sep [string trim $clipItemSep]
	set firstCmd [CmdToken tmpCmds $sep]
	# Then check if it really is a canvas command.
	#puts "it has the potential of a canvas command:: firstCmd=$firstCmd"
	if {[info complete ".junk $firstCmd"]}  {
	    #puts "it is a canvas item command"
	    set clipToken "item"
	} else  {
	    set clipToken "string"	    
	} 
    } else  {
	set clipToken "string"	    
    }
        
    # Depending on clipToken, either paste simple text string, or complete item(s).
    if {$clipToken == "string"}  {
	
	# Find out if there is a current focus on a text item.
	if {[llength [$w focus]] == 0}  {
	    eval CanvasFocus $w [NewImportAnchor] 1
	}
	CanvasTextInsert $w $cmds
	
    } elseif {$clipToken == "item"}  {
	set sep [string trim $clipItemSep]
	set firstCmd [CmdToken cmds $sep]
	if {$debugLevel >= 2}  {
	    puts "PasteFromClipboardToCanvas:: firstCmd=$firstCmd, sep=$sep"
	}
	while {$firstCmd != -1}  {
	    PasteSingleFromClipboardToCanvas $w $firstCmd
	    set firstCmd [CmdToken cmds $sep]
	}
    }
    # Default for 'clipToken' should always be "string" to be prepared
    # for imports from other apps. Not 100% foolproof.
    set clipToken "string"
}

# PasteSingleFromClipboardToCanvas --
#
#   Evaluates the canvas create command given by 'cmd', but at a coordinate
#   offset, makes it the new selection and copies it again to clipboard.
#   Be sure to treat newlines correctly when sending command to clients.

proc PasteSingleFromClipboardToCanvas { w cmd } {
    global	itno myItpref prefs debugLevel allIPnumsToSend ipNum2Socket

    set nl_ "\\n"
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: cmd=$cmd"
    }
    # add new tags
    set itemType [lindex $cmd 1]
    set theTags [list $itemType $myItpref/$itno]
    lappend cmd -tags $theTags
    
    # make coordinate offset, first get coords
    set ind1 [lsearch $cmd \[0-9.\]*]
    set ind2 [expr [lsearch $cmd -*\[a-z\]*] - 1]
    set theCoords [lrange $cmd $ind1 $ind2]
    set cooOffset {}
    foreach coo $theCoords {
	lappend cooOffset [expr $coo + $prefs(offsetCopy)]
    }
    
    # paste back coordinates in cmd
    set newcmd [concat [lrange $cmd 0 [expr $ind1 - 1]] $cooOffset  \
	    [lrange $cmd [expr $ind2 + 1] end]]
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: newcmd=$newcmd"
    }
    # Actually draw canvas command.
    set id [eval $w $newcmd]
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: id=$id"
    }
    
    # Create new bbox and select item.
    MarkBbox $w 1 $id
    
    # Copy the newly pasted object to clipboard.
    CopySelectedToClipboard  $w copy
    
    # Write to all other clients; need to make a one liner first.
    regsub -all "\n" $newcmd $nl_ oneliner
    if {$debugLevel >= 2}  {
	puts "PasteSingleFromClipboardToCanvas:: oneliner=$oneliner"
    }
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) "CANVAS: $oneliner"
    }
    incr itno
}

# CanvasTextPaste --
#
#   Unix style paste using button 2.

proc CanvasTextPaste { c {x {}} {y {}} }  {
    global	debugLevel allIPnumsToSend ipNum2Socket
    
    if {$debugLevel >= 2}  {
	puts "CanvasTextPaste::"
    }
    # If no selection just return.
    if {[catch {selection get} _s] &&
    [catch {selection get -selection CLIPBOARD} _s]}  {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextPaste:: no selection"
	}
	return
    }
    
    # Once the text string is found use...
    CanvasTextInsert $c $_s
    return

    # old surplus code
    set id [$c focus]
    if {[string length $id] == 0}  {
	set id [$c find withtag current]
    }
    if {[string length $id] == 0}  {
	# no object under the mouse
	if {[string length $x] == 0}  {
	    # keyboard paste
	    set x [expr [winfo pointerx $c] - [winfo rootx $c]]
	    set y [expr [winfo pointery $c] - [winfo rooty $c]]
	}
	CanvasFocus $c $x $y
    } else {
	$c focus $id
    }
    set theItno [CanvasGetItnoFrom $c focus]
    if {$theItno == ""}	 {
	if {$debugLevel >= 2}  {
	    puts "CanvasTextPaste:: theItno={}"
	}
	return
    }
    # The actual insert command.
    $c insert [$c focus] insert $_s
    set ind [expr [$c index [$c focus] insert] - 0]
    foreach ip $allIPnumsToSend  {
	puts $ipNum2Socket($ip) [list "CANVAS:" insert $theItno $ind $_s]
    }	
}

proc ColorSelector { col }	{
    global  prefs wColSel idColSel
    
    set col [tk_chooseColor -initialcolor $col]
    if {[string length $col] > 0}	 {
	# The change should be triggered automatically through the trace.
	set prefs(fgCol) $col
	#$wColSel itemconfigure $idColSel -fill $prefs(fgCol)
	#$wColSel raise $idColSel
    }
}

proc SetCanvasBgColor { w }	 {
    global  prefs
    
    set col [tk_chooseColor -initialcolor $prefs(bgColCanvas)]
    if {[string length $col] > 0}  {
	# The change should be triggered automatically through the trace.
	set prefs(bgColCanvas) $col
	#$w configure -bg $prefs(bgColCanvas)
    }
}

proc DoCanvasGrid  {  }	 {
    # make a grid in the canvas; uses prefs(canGridOn) to toggle grid
    global  wCan prefs
    
    set length 1200
    set gridDist $prefs(gridDist)
    set canGridOn $prefs(canGridOn)
    if {$canGridOn == 0}  {
	$wCan delete grid
	return
    }
    for {set x $gridDist} {$x <= $length} {set x [expr $x + $gridDist]}	 {
	$wCan create line $x 0 $x $length	\
		-width 1 -fill gray50 -stipple gray50 -tags grid
    }
    for {set y $gridDist} {$y <= $length} {set y [expr $y + $gridDist]}  {
	$wCan create line 0 $y $length $y  \
		-width 1 -fill gray50 -stipple gray50 -tags grid
    }
    $wCan lower grid
}

# DoCloseClientConnection --
#
#   Handle closing down the client side connection (the 'to' part).

proc DoCloseClientConnection { ipNum }  {
    global  prefs allIPnumsToSend allIPnumsTo ipNum2Socket ipNum2Name debugLevel
        
    if {$debugLevel >= 2}  {
	puts "DoCloseClientConnection:: ipNum=$ipNum"    
    }
    # If it is not there, just return.
    set ind [lsearch $allIPnumsTo $ipNum]
    if {$ind == -1}  {
	return
    }
    # Do the actual closing.
    catch {close $ipNum2Socket($ipNum)}

    # Update the communication frame; remove connection 'to'.
    SetCommEntry $ipNum  0 -1

    # If no more connections left, make menus consistent.
    if {[llength $allIPnumsToSend] == 0}  {
	# In case we are the client in a centralized network, make sure we
	# can make a new connection when closed the old one.
	if {!$prefs(symmetricNet) && !$prefs(thisCentralServ)}  {
	    .menu.file entryconfigure "*Open Connection*" -state normal
	}
	.menu.file entryconfigure "*Put File*" -state disabled
	.menu.file entryconfigure "*Put Canvas*" -state disabled
	.menu.file entryconfigure "*Get Canvas*" -state disabled
	.menu.info entryconfigure "*On Clients*" -state disabled
	.menu.prefs entryconfigure "*Network Setup*" -state normal
    }
}

# DoCloseServerConnection --
#
#   Handles everything to close the server side connection (the 'from' part).

proc DoCloseServerConnection  { ipNum }  {
    global  ipNum2ServSocket allIPnumsToSend debugLevel
    
    if {$debugLevel >= 2}  {
	puts "DoCloseServerConnection:: ipNum=$ipNum"
    }
    # Switch off the comm 'from' button.
    SetCommEntry $ipNum -1 0
    catch {close $ipNum2ServSocket($ipNum)}

    # If no more connections left, make menus consistent.
    if {[llength $allIPnumsToSend] == 0}  {
	.menu.file entryconfigure "*Put File*" -state disabled
	.menu.file entryconfigure "*Put Canvas*" -state disabled
	.menu.file entryconfigure "*Get Canvas*" -state disabled
	.menu.info entryconfigure "*On Clients*" -state disabled
	.menu.prefs entryconfigure "*Network Setup*" -state normal
    }
}
    
proc SavePostscript { w }  {
    global  postscriptOpts
    
    set typelist {
	{"Postscript File" {".ps"}}
    }
    set ans [tk_getSaveFile -title "Save As Postscript"	 \
      -filetypes $typelist -defaultextension ".ps"]
    #puts "SavePostscript:: ans=$ans, postscriptOpts=$postscriptOpts"
    if {[string length $ans] > 0}	 {
	eval {$w postscript} $postscriptOpts {-file $ans}
    }
}

proc NewImportAnchor  {  }  {
    global  importAnchor prefs dims
    
    set x $importAnchor(x)
    set y $importAnchor(y)
    
    # Update 'importAnchor'.
    incr importAnchor(x) $prefs(offsetCopy)
    incr importAnchor(y) $prefs(offsetCopy)
    if {$importAnchor(x) > [expr $dims(wRoot) - 60] 
    || $importAnchor(y) > [expr $dims(hRoot) - 60]} {
	set importAnchor(x) $prefs(offsetCopy)
	set importAnchor(y) $prefs(offsetCopy)
    }
    return [list $x $y]
}

# ImportImageOrMovieDlg --
#
#   Handles the dialog of opening a file ans then lets 'DoImportImageOrMovie' 
#   do the rest. On Mac either file extension or 'type' must match.

proc ImportImageOrMovieDlg { w }  {
    global  typelistImageMovie
    
    set fileName [tk_getOpenFile -title "Open Image/Movie"   \
      -filetypes $typelistImageMovie]
    if {$fileName == ""} {
	return
    }
    # Once the file name is chosen continue...
    DoImportImageOrMovie $w $fileName [list coords: [NewImportAnchor]]
}

# DoImportImageOrMovie --
# 
#   Opens an image in the canvas and puts it to all other clients.
#   If QuickTime is supported, a movie can be opened as well.
#   
#   This is the preferred import procedure for QuickTimeTcl, xanim, and the snack
#   packages and helper application.
#   
#   The 'optList' is a list of 'key: value' pairs, resembling the html 
#   protocol for getting files, but where most keys correspond to a valid
#   "canvas create" option, and everything is on a single line.
#   
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc DoImportImageOrMovie { w fileName optList {where all} }  {
    global  myItpref itno allIPnumsToSend prefs macType2Suff  \
      tcl_platform xanimPipe2Frame xanimPipe2Item movieItemno2File   \
      suffix2MimeList prefMimeType2Package   \
      supportedMimeTypes debugLevel
    
    set dot_ {\.}
    set mime_ {[^/]+}

    set fileTail [file tail $fileName]
    set fext [string tolower [file extension $fileTail]]
    if {$debugLevel >= 2}  {
	puts "_  DoImportImageOrMovie:: fileName=$fileName, optList=$optList, \
	  where=$where, fext=$fext"
    }
    
    # The Mac allows the possibility of no extension; find one!
    if {$tcl_platform(platform) == "macintosh" && $fext == ""}  {
	set macType [file attributes $fileName -type]
	set fext $macType2Suff($macType)
    }
    set totBytes [file size $fileName]
    
    # Define a standard set of put/import options that may be overwritten by
    # the options in the procedure argument 'optList'.
    # We pick the first MIME type in the list.
    array set optArray [list   \
      Content-Type:     [lindex $suffix2MimeList($fext) 0]    \
      size:             $totBytes                             \
      coords:           {0 0}                                 \
      tags:             $myItpref/$itno                       ]
    
    # Now apply the 'optList' and possibly overwrite some of the default options.
    if {[llength $optList] > 0}  {
	array set optArray $optList
    }
    # Make it as a list for PutFile below.
    set putOpts [array get optArray]
    
    # Extract coordinates and tags which must be there. error checking?
    set x [lindex $optArray(coords:) 0]
    set y [lindex $optArray(coords:) 1]
    set useTag $optArray(tags:)
    
    # Depending on the MIME type do different things; the MIME type is the
    # primary key for classifying the file.    
    
    set theMIME $optArray(Content-Type:)
    regexp "(${mime_})/" $theMIME match mimeBase
   
    if {[string compare $mimeBase "image"] == 0}  {
	
	# Image: seem to work identically for all packages.
	# There probably is a problem here for the Img package since
	# a postscript file is considered an image but has mime type
	# 'application'.
	
	if {$where == "all" || $where == "own"}  {
	    
	    # Create internal image if not there already.
	    if {[lsearch -exact [image names] im_$fileTail] < 0}  {
		image create photo im_$fileTail -file $fileName
	    }
	    set cmd "create image $x $y   \
	      -image im_$fileTail -anchor nw -tags {image $useTag}"
	    eval $w $cmd
	    if {[info exists optArray(above:)]}  {
		
		# Need a catch here since we can't be sure that other item exists.
		catch {$w raise $useTag $optArray(above:)}
	    }
	}

	# Transfer image file to all other servers.
	# Be sure to keep the list structure of 'putOpts'.
	if {$where != "own" && [llength $allIPnumsToSend] > 0}  {
    
	    # Once the thing lives on the canvas, add a 'putOpts' "above".
	    set idBelow [$w find below $useTag]
	    if {[string length $idBelow] > 0}  {
		set itnoBelow [CanvasGetItnoFrom $w $idBelow 1]
		if {[string length $itnoBelow] > 0}  {
		    lappend putOpts "above:" $itnoBelow
		}
	    }
	    PutFile $fileName $where $putOpts
	}	
	
    } elseif {([lsearch $supportedMimeTypes(QuickTimeTcl) $theMIME] >= 0) ||  \
      ([lsearch $supportedMimeTypes(xanim) $theMIME] >= 0) ||  \
      ([lsearch $supportedMimeTypes(snack) $theMIME] >= 0)} {

	# QuickTime and other movies. Movies are put in frame with class 'QTFrame'
	# in order to catch mouse events.
	# Strip dots in filename, " " to _; small chars in pathname.
	# Xanim app: keep the class 'QTFrame'.
	
	set movieItemno2File($useTag) $fileName
	set fileRoot [string tolower [file rootname [file tail $fileName]]]
	regsub -all $dot_ $fileRoot "" tmp
	regsub -all " " $tmp "_" newName
	set fr ${w}.fr_${newName}${itno}
	if {$where == "all" || $where == "own"}  {
	    
	    # Make a frame for the movie; need special class to catch mouse events.
	    frame $fr -height 1 -width 1 -bg gray40 -class QTFrame
	   
	    # Need to search each package or helper application to find out
	    # which to choose for this type of file.
	    
	    if {[string compare $prefMimeType2Package($theMIME) "QuickTimeTcl"] == 0} {
	    
		# QuickTime:
		# Import the movie only if not exists already.
		
		if {[catch {Movie ${fr}.m -file $fileName} msg]}  {
		    tk_messageBox -message "Error:  $msg" -icon error -type ok
		    catch {destroy $fr}
		    return
		}
		set mpath $msg	
		if {$prefs(autoFitMovies)} {
		    set newSize [GetAutoFitSize $w $mpath]
		    #puts "GetAutoFitSize=$newSize"
		    eval $mpath configure  \
		      -width [lindex $newSize 0] -height [lindex $newSize 1]
		}
		$w create window $x $y -anchor nw -window $fr   \
		  -tags [list movie $useTag]
		pack $mpath -in $fr -padx 3 -pady 3
		update idletasks
		if {[info exists optArray(above:)]}  {
		    catch {$w raise $useTag $optArray(above:)}
		}
		set width [winfo width $mpath]
		set height [winfo height $mpath]
		if {$debugLevel >= 2}  {
		    puts "_  mpath=$mpath, newName=$newName, height=$height, \
		      width=$width"
		}

	    } elseif {[string compare $prefMimeType2Package($theMIME) "xanim"] == 0} {

		# Special handling using the 'xanim' application:
		# First, query the size of the movie without starting it.
		set size [XanimQuerySize $fileName]
		if {[llength $size] != 2}  {
		    return
		}
		set width [lindex $size 0]
		set height [lindex $size 1]
		$fr configure -width [expr $width + 6] -height [expr $height + 6]
		$w create window $x $y -anchor nw -window $fr -tags "movie $useTag"
		
		# Make special frame for xanim to draw in.
		set frxanim [frame $fr.xanim -container 1 -bg black  \
			-width $width -height $height]
		place $frxanim -in $fr -anchor nw -x 3 -y 3
		if {[info exists optArray(above:)]}  {
		    catch {$w raise $useTag $optArray(above:)}
		}
		
		# Important, make sure that the frame is mapped before continuing.
		update idletasks
		set xatomid [winfo id $frxanim]
		
		# Note trick to pipe stdout as well as stderr. Forks without &.
		if {[catch {open "|xanim +W$xatomid $fileName 2>@stdout"} xpipe]} {
		    puts "xanim err: xpipe=$xpipe"
		} else {
		    set xanimPipe2Frame($xpipe) $fr
		    set xanimPipe2Item($xpipe) $useTag
		    fileevent $xpipe readable "XanimReadOutput $xpipe"
		}
		
	    } elseif {[string compare $prefMimeType2Package($theMIME) "snack"] == 0} {
		
		# The snack plug-in for audio.
		
		
	    }
	}
	# Transfer movie file to all other servers.
	# Several options possible:
	#   1) flatten, put in httpd directory, and transfer via http.
	#   2) make hint track and serve using RTP.
	#   3) put as an ordinary binary file, perhaps flattened.
	
	if {($where != "own") && ([llength $allIPnumsToSend] > 0)}  {
	    
	    # Need to flatten QT movie first?
	    #set tmpflat "flatten_$fileRoot"
	    #$mpath flatten $tmpflat
	    
	    # Transfer the movie file to all other servers.
	    # Be sure to keep the list structure of 'putOpts'.
	    
	    # Once the thing lives on the canvas, add a 'putOpts' "above".
	    set idBelow [$w find below $useTag]
	    if {[string length $idBelow] > 0}  {
		set itnoBelow [CanvasGetItnoFrom $w $idBelow 1]
		if {[string length $itnoBelow] > 0}  {
		    lappend putOpts "above:" $itnoBelow
		}
	    }
	    PutFile $fileName $where $putOpts
	    
	    # Serve via http?
	    
	    # Serve via RTP?
	    
	}
    }
    # Add to the lists of known files.
    AddToKnownFiles $fileTail $fileName "$useTag"
    
    # Update 'itno' only when also writing to own canvas!
    if {$where == "all" || $where == "own"}  {
	incr itno
    }
}

# XanimQuerySize --
#
#   Gets size of the movie. If any error, return {}.
#   Check also version number ( >= 2.70 ).

proc XanimQuerySize { fileName }  {
    global  plugin
    
    set num_ {[0-9]+}
    set ver_ {[0-9]+\.[0-9]+}
    if {![catch {exec xanim +v +Zv $fileName} res]}  {
	
	# Check version number.
	if {[regexp "Rev +($ver_)" $res match ver]}  {
	    set plugin(xanim,ver) $ver
	    if {$ver < 2.7}  {
		puts stderr "Error: xanim must have at least version 2.7"
		return {}
	    }
	}
	
	# Ok, parse size.
	if {[regexp "Size=(${num_})x(${num_})" $res match w h]}  {
	    return [list $w $h]
	} else  {
	    return {}
	}
    } else  {
	# Error checking...
	puts "XanimQuerySize:: error, res=$res"
	return {}
    }
}

proc XanimReadOutput { xpipe }  {
    global  xanimPipe2Frame xanimPipe2Item wCan
    
    if [eof $xpipe]  {
	
	# Movie is stopped, cleanup.
	#puts "XanimReadOutput:: eof xpipe"
	set co [$wCan coords $xanimPipe2Item($xpipe)]
	DeleteItem $wCan [lindex $co 0] [lindex $co 1] movie   \
	  $xanimPipe2Frame($xpipe)
	catch {close $xpipe}
    } else  {
	
       # Read each line and try to figure out if anything went wrong.
       gets $xpipe line
       #puts "XanimReadOutput:: line=$line"
       if {[regexp -nocase "(unknown|error)" $line match junk]}  {
	   tk_messageBox -message "Something happened when trying to\
	     run 'xanim': $line" -icon info -type ok
       }
   }
}

# CanvasResizeItem --
#
#   Scales each selected item in canvas 'w' by a factor 'factor'. 
#   Not all item types are rescaled. 

proc CanvasResizeItem  { w factor }  {
    global  prefs ipNum2Socket allIPnumsToSend
    
    set ids [$w find withtag selected]
    if {[string length $ids] == 0}  {
	return
    }
    if {$prefs(scaleCommonCG)}  {
	set bbox [eval $w bbox $ids]
	set cgx [expr ([lindex $bbox 0] + [lindex $bbox 2])/2.0]
	set cgy [expr ([lindex $bbox 1] + [lindex $bbox 3])/2.0]
    }
    foreach id $ids  {
	set theItno [CanvasGetItnoFrom $w $id]
	#puts "theItno=$theItno"
	if {[string length $theItno] == 0}  {
	    continue
	}	
	# Sort out the nonrescalable ones.
	set theType [$w type $id]
	if {([string compare $theType "text"] == 0) ||   \
	  ([string compare $theType "image"] == 0) ||    \
	  ([string compare $theType "window"] == 0)}  {
	    continue
	}
	if {!$prefs(scaleCommonCG)}  {
	    set bbox [$w bbox $id]
	    set cgx [expr ([lindex $bbox 0] + [lindex $bbox 2])/2.0]
	    set cgy [expr ([lindex $bbox 1] + [lindex $bbox 3])/2.0]
	}
	set cmd "scale $theItno $cgx $cgy $factor $factor"
	eval $w $cmd
	
	# New markers.
	$w delete id$id
	MarkBbox $w 1 $id
	
	# Let other clients know.
	foreach ip $allIPnumsToSend  {
	    puts $ipNum2Socket($ip) "CANVAS: $cmd"
	}
    }    
}

# ResizeImage --
#
#   Uhh.. resizes the selected images. 'zoomFactor' is 0,1 for no resize,
#   2 for an enlargement with a factor of two, and
#   -2 for a size decrease to half size.   
#   'which' = "sel": selected images, or a specific image with tag 'which'.
#   'newTag' = "auto": use 'myItpref' and running 'itno', 
#   else 'newTag' is the tag to use.
#   where = "all": write to this canvas and all others.
#   where = "other": write only to remote client canvases.
#   where = ip number: write only to this remote client canvas and not to own.
#   where = "own": write only to this canvas and not to any other.

proc ResizeImage { w zoomFactor which newTag {where all} }  {
    global  itno myItpref ipNum2Socket allIPnumsToSend
    
    #puts "ResizeImage:: w=$w, zoomFactor=$zoomFactor"
    set scaleFactor 2
    set int_ {[-0-9]+}
    
    # Compute total resize factor.
    if {$zoomFactor >= 0 && $zoomFactor <= 1 }  {
	return
    } elseif {$zoomFactor == 2}  {
	set theScale 2
    } elseif {$zoomFactor == -2}  {
	set theScale 0.5
    } else  {
	return
    }
    if {$which == "sel"}  {
	# Choose selected images only.
	set ids [$w find withtag selected]
    } else  {
	# Choose image with tag 'which'.
	set ids [$w find withtag $which]
	if {[llength $ids] <= 0}  {
	    return
	}
    }
    set idsNew {}
    foreach id $ids  {
	
	if {$where == "all" || $where == "own"}  {
	    
	    set theType [$w type $id]
	    #puts "ResizeImage:: id=$id, theType=$theType, theScale=$theScale"
	    if {[string compare $theType "image"] != 0}  {
		continue
	    }
	    # Check if no privacy problems. Only if 'which' is the selected.
	    set itOrig [CanvasGetItnoFrom $w $id]
	    if {$which == "sel" && $itOrig == ""}  {
		continue
	    }
	    set theCoords [$w coords $id]
	    set theIm [$w itemcget $id -image]
	    # Resized photos add tag to name '_zoom2' for double size,
	    # '_zoom-2' for half size etc.
	    if {[regexp "_zoom(${int_})$" $theIm match sizeNo]}  {
		# This image already resized.
		#puts "# This image already resized."
		if {$zoomFactor == 2}  {
		    if {$sizeNo >= 2}  {
			set newSizeNo [expr $sizeNo * $zoomFactor]
		    } elseif {$sizeNo == -2} {
			set newSizeNo 0
		    } else {
			set newSizeNo [expr $sizeNo/$zoomFactor]
		    }
		} elseif {$zoomFactor == -2}  {
		    if {$sizeNo <= -2}  {
			set newSizeNo [expr -$sizeNo * $zoomFactor]
		    } elseif {$sizeNo == 2} {
			set newSizeNo 0
		    } else {
			set newSizeNo [expr -$sizeNo/$zoomFactor]
		    }
		}
		#puts "newSizeNo=$newSizeNo"
		if {$newSizeNo == 0}  {
		    # Get original image. Strip off the _zoom tag.
		    regsub "_zoom$sizeNo" $theIm  "" newImName
		} else  {
		    regsub "_zoom$sizeNo" $theIm "_zoom$newSizeNo" newImName
		}
	    } else  {
		# Add tag to name indicating that it has been resized.
		#puts "# Add tag to name indicating that it has been resized."
		set newSizeNo $zoomFactor
		set newImName ${theIm}_zoom${newSizeNo}
	    }
	    #puts "ResizeImage:: theCoords=$theCoords, theIm=$theIm, \
	    #  newImName=$newImName, newSizeNo=$newSizeNo"
	    
	    # Create new image for the scaled version if it does not exist before.
	    if {[lsearch -exact [image names] $newImName] < 0}  {
		#puts "new photo: zoomFactor=$zoomFactor"
		# Make new.
		image create photo $newImName
		if {$zoomFactor > 0}  {
		    $newImName copy $theIm -zoom $theScale
		} else  {
		    $newImName copy $theIm -subsample [expr round(1.0/$theScale)]
		}
	    }
	    # Choose this clients automatic tags or take 'newTag'.
	    if {$newTag == "auto"}  {
		set useTag $myItpref/$itno
	    } else  {
		set useTag $newTag
	    }
	    set cmd "create image $theCoords   \
	      -image $newImName -anchor nw -tags {image $useTag}"
	    set newId [eval $w $cmd]
	    lappend idsNew $newId
	}
	# This ends works on resizing "own" photos.
	
	# Make a list with all ip numbers to put file to.
	if {$where != "own"}  {
	    if {$where == "other" || $where == "all"}  {
		set allPutIP $allIPnumsToSend
	    } else  {
		set allPutIP $where
	    }    
	    # Loop over all connected servers or only the specified one.
	    foreach ip $allPutIP  {
		puts $ipNum2Socket($ip)  \
			"RESIZE IMAGE: $itOrig $useTag $zoomFactor"
	    }
	}
	# Remove old.
	$w delete $id
	incr itno
    }
    DeselectAll $w
    # Mark the new ones.
    foreach id $idsNew  {
	MarkBbox $w 1 $id
    }
}

# DoEraseAll --
# 
#   where = "all": erase this canvas and all others.
#   where = "other": erase only client canvases.
#   where = "own": erase only own canvas.

proc DoEraseAll { w { where all } }	 {
    global  allIPnumsToSend ipNum2Socket
    
    DeselectAll $w
    set ids [$w find all]
    foreach id $ids {
	# do not erase grid
	set theTags [$w gettags $id]
	if {[lsearch $theTags grid] >= 0}  {
	    continue
	}
	set it [CanvasGetItnoFrom $w $id]
	if {$it == ""}  {
	    continue
	}
	if {$where != "other"}  {
	    $w delete $id
	}
	if {$where != "own"}  {
	    foreach ip $allIPnumsToSend  {
		puts $ipNum2Socket($ip) "CANVAS: delete $it"
	    }
	}
    }
}

# DoOpenConnection --
#
#   Starts process to open a connection to another client.
#   Needs procedures from the OpenConnection.tcl file.
#   When opening in async mode, the calling sequence is:
#      DoOpenConnection -> OpenConnection -> PushBtConnect ->
#         -> DoConnect -> WhenSocketOpensInits,
#   where the last sequence is triggered by a fileevent.

proc DoOpenConnection {	 }	{
    global  wDlgs debugLevel
    
    if {$debugLevel >= 2}  {
	puts "DoOpenConnection:: enter"
    }
    
    # Complete dialog for choosing address etc.
    set ans [OpenConnection $wDlgs(openConn)]
    if {$debugLevel >= 2}  {
	puts "DoOpenConnection:: ans=$ans"
    }
}

# IsConnectedToQ --
#
#   Finds if connected to 'ipNameOrNum' by searching 'allIPnumsTo'.
#   It always answers that it is connected to itself ('thisIPnum').
#   This is also true if 'thisIPnum' equal to the standard 127.0.0.1.

proc IsConnectedToQ { ipNameOrNum }  {
    global  ipName2Num allIPnumsTo thisIPnum internalIPnum

    set ip_ {[a-zA-Z]+}
    
    # Find out if 'ipNameOrNum' is name or number.
    # If any character in 'ipNameOrNum' then assume it is a name.
    if {[regexp $ip_ $ipNameOrNum]}  {
	#puts "IsConnectedToQ:: ipNameOrNum=$ipNameOrNum, its a name!"
	if {[info exists ipName2Num($ipNameOrNum)]}  {
	    set ipnum $ipName2Num($ipNameOrNum)
	} else  {
	    return 0
	}
    } else  {
	set ipnum $ipNameOrNum
    }
    if {$ipnum == $internalIPnum}  {
	return 1
    } elseif {[lsearch -exact "$allIPnumsTo $thisIPnum" $ipnum] >= 0}  {
	return 1
    } else  {
	return 0
    }
}

#   Sets auto disconnect identical to autoConnect.

proc DoAutoConnect  {  }  {
    global  prefs
    set prefs(autoDisconnect) $prefs(autoConnect)
}

# CmdToken --
#   
#   Returns part of 'cmdName' up to 'separator' and deletes that part 
#   from 'cmdName'.

proc CmdToken { cmdName separator }  {
    upvar $cmdName theCmd
    
    # If nothing then return -1.
    if {[llength $theCmd] == 0} {
	return -1
    }
    set indSep [lsearch -exact $theCmd $separator]
    
    # If no separator then just return the remaining part.
    if {$indSep == -1}  {
	set firstPart $theCmd
	set theCmd {}
	return $firstPart
    }
    
    # If separator in -text then ???.
    if {[lindex $theCmd [expr $indSep - 1]] != "-text"}  {
	set firstPart [lrange $theCmd 0 [expr $indSep - 1]]
    } else {
	puts "Warning in CmdToken: -text part wrong"
    }
    set theCmd [lrange $theCmd [expr $indSep + 1] end]
    return $firstPart
}

# UpdateCommFrameHeader --
#
#   If network configuration is changed, update the communication
#   frame header.

proc UpdateCommFrameHeader  {  }  {
    global  prefs commFrame
    
    catch {destroy $commFrame.to $commFrame.from $commFrame.icon}
    
    # If symmetric network setup.
    if {$prefs(symmetricNet)}  {
	label $commFrame.to -text " To  "
	label $commFrame.from -text "From "
	grid $commFrame.to -column 2 -row 0 -sticky nws -pady 0
	grid $commFrame.from -column 3 -row 0 -sticky nws -pady 0
    } elseif {$prefs(thisCentralServ)}  {
	
	# Or if central server, only 'from' connections.
	label $commFrame.from -text "From "
	label $commFrame.icon -image bluebox
	grid $commFrame.from -sticky nws -column 2 -row 0 -pady 0
	grid $commFrame.icon -sticky nws -column 3 -row 0 -pady 0
    } else  {
	
	# If this is a client connected to a central server, no 'from' connections.
	label $commFrame.to -text " To  "
	label $commFrame.icon -image contact_off
	grid $commFrame.to -sticky nws -column 2 -row 0 -pady 0
	grid $commFrame.icon -sticky nws -column 3 -row 0 -pady 0
    }
}

# SetCommEntry --
#
#   Adds, removes or updates an entry in the communications frame.
#   If 'to' or 'from' is -1 then disregard this variable.
#   If neither 'to' or 'from', then remove the entry completely for this
#   specific ipNum.
#   It updates all lists of type 'allIPnums...', but doesn't do anything
#   with channels.
#   The actual job of handling the widgets are done in 'RemoveCommEntry' 
#   and 'CreateCommEntry'.

proc SetCommEntry  { ipNum to from }  { 
    global  commTo commFrom allIPnumsToSend debugLevel allIPnums allIPnumsTo  \
      allIPnumsFrom prefs
    
    if {$debugLevel >= 2}  {
	puts "SetCommEntry:: ipNum=$ipNum, to=$to, from=$from"
    }
    
    # Need to check if already exist before adding a completely new entry.
    set ind [lsearch $allIPnums $ipNum]
    if {$ind >= 0}  {
	set alreadyThere 1
    } else {
	set alreadyThere 0
    }
    if {!$alreadyThere}  {
	# Just init arrays.
	set commTo($ipNum) 0		
	set commFrom($ipNum) 0
    }
    
    # If wants to set 'to/from' that is not relevant. For centralized network.
    if {!$prefs(symmetricNet)}  {
	if {$prefs(thisCentralServ) && $from == -1}  {
	    return
	} elseif {!$prefs(thisCentralServ) && $to == -1}  {
	    return
	}
    }    
    if {$to >= 0}  {
	set commTo($ipNum) $to
    }
    if {$from >= 0}  {
	set commFrom($ipNum) $from
    }
    if {$debugLevel >= 2}  {
	puts "SetCommEntry:: alreadyThere=$alreadyThere, ipNum=$ipNum, ind=$ind"
    }
    
    # If it is not there and shouldn't be added, just return.
    if {!$alreadyThere && $commTo($ipNum) == 0 && $commFrom($ipNum) == 0}  {
	if {$debugLevel >= 2}  {
	    puts "SetCommEntry:: it is not there and shouldnt be added"
	}
	return
    }
    
    # Update 'allIPnumsTo' to contain each ip num connected to.
    set ind [lsearch $allIPnumsTo $ipNum]
    if {$ind == -1 && $commTo($ipNum) == 1}  {
	lappend allIPnumsTo $ipNum
    } elseif {$ind >= 0 && $commTo($ipNum) == 0}  {
	set allIPnumsTo [lreplace $allIPnumsTo $ind $ind]
    }
    
    # Update 'allIPnumsFrom' to contain each ip num connected to our server
    # from a remote client.
    set ind [lsearch $allIPnumsFrom $ipNum]
    if {$ind == -1 && $commFrom($ipNum) == 1}  {
	lappend allIPnumsFrom $ipNum
    } elseif {$ind >= 0 && $commFrom($ipNum) == 0}  {
	set allIPnumsFrom [lreplace $allIPnumsFrom $ind $ind]
    }
    
    # Update sending list. If this is the central server in a centralized net,
    # we should write on socket connected to the server, else on our client side
    # sockets.
    if {!$prefs(symmetricNet) && $prefs(thisCentralServ)}  {
	set allIPnumsToSend $allIPnumsFrom
    } else  {
	set allIPnumsToSend $allIPnumsTo
    }
    
    # Update 'allIPnums' to be the union of 'allIPnumsTo' and 'allIPnumsFrom'.
    # Tcl8.3: set allIPnums [lsort -unique [join $allIPnumsFrom $allIPnumsTo]]
    # If both to and from 0 then remove from list.
    set ind [lsearch $allIPnums $ipNum]
    if {$ind >= 0 && $commTo($ipNum) == 0 && $commFrom($ipNum) == 0}  {
	set allIPnums [lreplace $allIPnums $ind $ind]
    } elseif {$ind == -1}  {
	lappend allIPnums $ipNum
    }
    
    # If both 'to' and 'from' 0, then remove entry.
    if {$commTo($ipNum) == 0 && $commFrom($ipNum) == 0}  {
	RemoveCommEntry $ipNum
	# return here???
	return
    }
    
    # If already exist dont add widgets.
    if {$alreadyThere}  {
	return
    }
    CreateCommEntry $ipNum
}

proc CreateCommEntry { ipNum }  {
    global  wCan commFrame nEnt sysFont ipNum2User prefs commTo commFrom  \
      ipNum2User ipNum2Name allIPnumsTo ipNum2nEnt debugLevel
    
    # Size administration is very tricky; blood, sweat and tears...
    # Fix the canvas size to relax wm geometry. (Don't forget the border)
    $wCan configure -height [expr [winfo height $wCan] - 2]  \
      -width [expr [winfo width $wCan] - 2]
    # Switch off the geometry constraint to let resize automatically.
    wm geometry . {}
    
    # Add new status line.
    entry $commFrame.ad$nEnt -width 24 -font $sysFont(s)  \
      -textvariable ipNum2Name($ipNum) -relief sunken   \
      -bg $prefs(bgColGeneral)
    $commFrame.ad$nEnt configure -state disabled
    entry $commFrame.us$nEnt -width 16 -font $sysFont(s)   \
      -textvariable ipNum2User($ipNum) -relief sunken  \
      -bg $prefs(bgColGeneral)
    $commFrame.us$nEnt configure -state disabled
    
    # If symmetric network setup.
    if {$prefs(symmetricNet)}  {
	checkbutton $commFrame.to$nEnt -variable commTo($ipNum)   \
	  -highlightthickness 0 -state normal -command [list CheckCommTo $ipNum]
	checkbutton $commFrame.from$nEnt -variable commFrom($ipNum)  \
	  -highlightthickness 0 -state disabled
	grid $commFrame.ad$nEnt $commFrame.us$nEnt $commFrame.to$nEnt   \
	  $commFrame.from$nEnt -padx 4
    } elseif {$prefs(thisCentralServ)}  {
	
	# Or if central server, only 'from' connections.
	checkbutton $commFrame.from$nEnt -variable commFrom($ipNum)  \
	  -highlightthickness 0 -state disabled
	grid $commFrame.ad$nEnt $commFrame.us$nEnt $commFrame.from$nEnt -padx 4
    } else  {
    
        # If this is a client connected to a central server, no 'from' connections.
	checkbutton $commFrame.to$nEnt -variable commTo($ipNum)   \
	  -highlightthickness 0 -state normal -command [list CheckCommTo $ipNum]
	grid $commFrame.ad$nEnt $commFrame.us$nEnt $commFrame.to$nEnt -padx 4
	
	# Update "electric plug" icon if first connection.
	if {[llength $allIPnumsTo] == 1}  {
	    after 400 [list $commFrame.icon configure -image contact_on]
	}
    }
    update idletasks
    
    # Organize the new geometry. First fix using wm geometry, then relax
    # canvas size.
    set newGeom [ParseWMGeometry .]
    #puts "newGeom=$newGeom"
    wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
    $wCan configure -height 1 -width 1
    
    # Geometry considerations. Update geometry vars and set new minsize.
    after idle {SetNewWMMinsize}

    # Map ip name to nEnt.
    if {$debugLevel >= 2}  {
	puts "CreateCommEntry:: nEnt=$nEnt"
    }
    set ipNum2nEnt($ipNum) $nEnt
    
    # Step up running index. This must *never* be reused!
    incr nEnt
}

# debug stuff....
proc junkCommEntry  { ipNum }  {
    global ipNum2Name ipNum2User ipNum2Socket
    
    set ipNum2Name($ipNum) Mats$ipNum
    set ipNum2User($ipNum) "Mats Be"
    set ipNum2Socket($ipNum) sock99
    SetCommEntry $ipNum 1 1
}

# CheckCommTo --
#
#   This is the callback function when the checkbutton 'To' has been trigged.

proc CheckCommTo { ipNum }  {
    global  commTo debugLevel ipNum2Name ipNum2ServPort ipNum2nEnt
    
    #puts "CheckCommTo:: ipNum=$ipNum"
    # Find out which button triggered the event.
    set trigInd $ipNum2nEnt($ipNum)
       
    # Close connection.
    if {$commTo($ipNum) == 0}  {
	set res [tk_messageBox -message "Are you sure that you want\
	  to disconnect $ipNum2Name($ipNum)?" \
	  -icon warning -type yesno -default yes]
	if {$res == "no"}  {
	    # Reset.
	    set commTo($trigIP) 1
	    return
	} elseif {$res == "yes"}  {
	    DoCloseClientConnection $ipNum
	}
    } elseif {$commTo($ipNum) == 1}  {
	
	# Open connection. Let propagateSizeToClients = true.
	DoConnect $ipNum $ipNum2ServPort($ipNum) 1
	SetCommEntry $ipNum 1 -1
    }
}

# RemoveCommEntry --
#
#   Removes the complete entry in the communication frame for 'ipNum'.
#   It should not be called by itself; only from 'SetCommEntry'.

proc RemoveCommEntry  { ipNum }   {
    global  commFrame commTo commFrom ipNum2nEnt debugLevel wCan   \
      prefs allIPnumsTo
    
    # Find widget paths from ipNum and remove the entries.
    set no $ipNum2nEnt($ipNum)
    if {$debugLevel >= 2}  {
	puts "RemoveCommEntry:: no=$no"
    }
    # Size administration is very tricky; blood, sweat and tears...
    # Fix the canvas size to relax wm geometry.
    $wCan configure -height [expr [winfo height $wCan] - 2]  \
      -width [expr [winfo width $wCan] - 2]
    
    # Switch off the geometry constraint to let resize automatically.
    wm geometry . {}

    # Remove the widgets.
    catch {grid forget $commFrame.ad$no $commFrame.us$no $commFrame.to$no   \
      $commFrame.from$no}
    catch {destroy $commFrame.ad$no $commFrame.us$no $commFrame.to$no   \
      $commFrame.from$no}
    
    # Electric plug disconnect? Only for client only.
    if {!$prefs(symmetricNet) && !$prefs(thisCentralServ) &&  \
      [llength $allIPnumsTo] == 0}  {
	after 400 [list $commFrame.icon configure -image contact_off]
    }
    update idletasks
    
    # Organize the new geometry. First fix using wm geometry, then relax
    # canvas size.
    set newGeom [ParseWMGeometry .]
    wm geometry . [lindex $newGeom 0]x[lindex $newGeom 1]
    $wCan configure -height 1 -width 1

    # Geometry considerations. Update geometry vars and set new minsize.
    after idle {SetNewWMMinsize}
    if {$debugLevel >= 2}  {
	puts "exit RemoveCommEntry"
    }
}

# FontHtmlSizeToPointSize --
#
#   Change the -font option in a canvas command from html size type
#   to point size.

proc FontHtmlSizeToPointSize { canCmd }  {
    global  fontSize2Points
    
    set ind [lsearch -exact $canCmd "-font"]
    if {$ind >= 2}  {
	set fontSpec [lindex $canCmd [expr $ind +1]]
	set htmlFontSize [lindex $fontSpec 1]
	
	# Check that it is between 1 and 6.
	if {$htmlFontSize >= 1 && $htmlFontSize <= 6}  {
	    set newFontSpec   \
	      [lreplace $fontSpec 1 1 $fontSize2Points($htmlFontSize)]
	    
	    # Replace font specification in drawing command.
	    set canCmd [lreplace $canCmd [expr $ind + 1]   \
	      [expr $ind + 1] $newFontSpec]
	}
    }
    return $canCmd
}

# CreateFontSizeMapping --
#
#     Creates the mapping between Html sizes (1 2 3 4 5 6) and font point
#     sizes on this specific platform dynamically by measuring the length in
#     pixels of a fixed reference string.
#
# Results:
#     is put in the global variables 'fontSize2Points' and 'fontPoints2Size'.

proc CreateFontSizeMapping {  } {
    global  fontSize2Points fontPoints2Size
    
    # The reference is chosen to get point sizes on Mac as: 10, 12, 14, 18, 
    # 24, 36. Found via 'font measure {Times 10} {Mats Bengtsson}'.
    
    array set refHtmlSizeToLength {1 64 2 76 3 88 4 116 5 154 6 231}
    set refStr {Mats Bengtsson}
    
    # Pick to point sizes and reconstruct a linear relation from the font size
    # in points and the reference string length in pixels: y = kx + m.
    
    set p0 10
    set p1 36
    set y0 [font measure "Times $p0" $refStr]
    set y1 [font measure "Times $p1" $refStr]
    set k [expr ($y1 - $y0)/($p1 - $p0 + 0.0)]
    set m [expr $y1 - $k*$p1]
    
    # For what x (font size in points) do we get 'refHtmlSizeToLength(1)', etc.
    # x = (y - m)/k
    
    foreach htmlSize {1 2 3 4 5 6} {
	set fontSize2Points($htmlSize)  \
	  [expr int(floor( ($refHtmlSizeToLength($htmlSize) - $m)/$k + 0.5))]
    }
    
    # We also need the inverse mapping.
    
    foreach pt [array names fontSize2Points] {
	set fontPoints2Size($fontSize2Points($pt)) $pt
    }
}

# DoPutCanvasDlg --
#
#

proc DoPutCanvasDlg  { w }  {
    
    set ans [tk_messageBox -message   \
      "Warning! Syncing this canvas first erases all client canvases." \
      -icon warning -type okcancel -default ok]
    if {$ans != "ok"}  {
	return
    }
    # Erase all other client canvases.
    DoEraseAll $w "other"

    # Put this canvas to all others.
    DoPutCanvas $w all
}
    
# DoPutCanvas --
#   
#   Synchronizes, or puts, this canvas to all others. 
#   It uses a temporary file. Images don't work automatically.
#   If 'toIPnum' then put canvas 'w' only to that ip number.

proc DoPutCanvas { w {toIPnum all} }  {
    global  thisPath

    set tmpFile ".tmp[clock seconds].can"

    # Save canvas to temporary file.
    if {[catch [list open $tmpFile w] fileId]}  {
	tk_messageBox -message   \
	  "Cannot open $tmpFile for writing." \
	  -icon error -type ok
    }
    set absFilePath [file join $thisPath $tmpFile]
    CanvasToFile $w $fileId $absFilePath
    catch {close $fileId}

    if {[catch [list open $tmpFile r] fileId]}  {
	tk_messageBox -message   \
	  "Cannot open $tmpFile for reading." \
	  -icon error -type ok
    }
    # Distribute to all other client canvases.
    if {$toIPnum == "all"}  {
	FileToCanvas $w $fileId $absFilePath "other"
    } else  {
	FileToCanvas $w $fileId $absFilePath $toIPnum
    }
    catch {close $fileId}

    # Finally delete the temporary file.
    file delete $tmpFile
}

# DoGetCanvas --
#
#   Fetches the content of the canvas of a specified client.

proc DoGetCanvas { w }  {
    global  debugLevel ipNum2Name ipName2Num ipNum2Socket
    
    # The dialog to select remote client.
    set getCanIPNum [::GetCanvas::GetCanvas .getcan]
    if {$debugLevel >= 2}  {
	puts "DoGetCanvas:: getCanIPNum=$getCanIPNum"
    }
    if {$getCanIPNum == ""}  {
	return
    }    
    # Erase everything in own canvas.
    DoEraseAll $w "own"
    
    # GET CANVAS.
    catch {puts $ipNum2Socket($getCanIPNum) "GET CANVAS:"}
}

# FindWidgetGeometryAtLaunch --
#
#   Just after launch, find and set various geometries of the application.
#   'hRoot' excludes the menu height, 'hTot' includes it.
#   Note: [winfo height .#menu] gives the menu height when the menu is in the
#   root window; [wm geometry .] gives and sets dimensions without the menu;
#   [wm minsize .] gives and sets dimensions with the menu included.
#
#   wRoot, hRoot:      total size of the application not including any menu.
#   wTot, hTot:        total size of the application including any menu.
#   hTop:              height of the shortcut button frame at top.
#   hMenu:             height of any menu if present in the application window.
#   hStatus:           height of the status frame.
#   hComm:             height of the communication frame including all client
#                      frames.
#   hCommClean:        height of the communication frame excluding all client 
#                      frames.
#   wStatMess:         width of the status message frame.
#   wCanvas, hCanvas:  size of the actual canvas.
#   x, y:              position of the app window.

proc FindWidgetGeometryAtLaunch {  }  {
    global  dims wCan commFrame wStatMess tcl_platform
    
    # The actual dimensions.
    set dims(wRoot) [winfo width .]
    set dims(hRoot) [winfo height .]
    set dims(hTop) 0
    if {[winfo exists .frtop]}  {
	set dims(hTop) [winfo height .frtop]
    }
    set dims(hStatus) [winfo height .fcomm.st]
    set dims(hComm) [winfo height $commFrame]
    set dims(hCommClean) $dims(hComm)
    set dims(wStatMess) [winfo width $wStatMess]
    if {$tcl_platform(platform) != "macintosh"}  {
	set dims(hMenu) [winfo height .#menu]
    } else  {
	set dims(hMenu) 0
    }
    set dims(wCanvas) [winfo width $wCan]
    set dims(hCanvas) [winfo height $wCan]
    set dims(wTot) $dims(wRoot)
    set dims(hTot) [expr $dims(hRoot) + $dims(hMenu)]
    
    # Position of root window.
    set dimList [ParseWMGeometry .]
    set dims(x) [lindex $dimList 2]  
    set dims(y) [lindex $dimList 3]  

    # The minimum dimensions. Check if 'commFrame' is wider than wMinCanvas!
    # Take care of the case where there is no To or From checkbutton.
    
    set wMinCommFrame [expr [winfo width ${commFrame}.comm] +  \
      [winfo width ${commFrame}.user] +  \
      [image width im_handle] + 2]
    if {[winfo exists ${commFrame}.to]} {
	incr wMinCommFrame [winfo width ${commFrame}.to]
    }
    if {[winfo exists ${commFrame}.from]} {
	incr wMinCommFrame [winfo width ${commFrame}.from]
    }
    set dims(wMinRoot) [max [expr $dims(wMinCanvas) + 56] $wMinCommFrame]
    set dims(hMinRoot) [expr $dims(hMinCanvas) + $dims(hStatus) + $dims(hComm) +  \
      $dims(hTop)]
    set dims(wMinTot) $dims(wMinRoot)
    set dims(hMinTot) [expr $dims(hMinRoot) + $dims(hMenu)]
    
    # The minsize when no connected clients. Is updated when connect/disconnect.
    wm minsize . $dims(wMinTot) $dims(hMinTot)
}

proc ParseWMGeometry { w }  {
    set int_ {[0-9]+}
    set sign_ {\+|\-}
    regexp "(${int_})x(${int_})($sign_)(${int_})($sign_)(${int_})"   \
      [wm geometry $w] match wid hei junk1 x junk2 y
    return [list $wid $hei $x $y]
}

# CanvasSizeChange --
#   
#   If size change in canvas (application), then let other clients know.
#   
# Arguments:
#   where              "all" if tell all other connected clients,
#                       0 if none, and an ip number if only this one.
#   force               should we insist on telling other clients even if
#                       the canvas size not changed.

proc CanvasSizeChange { where {force 0} }  {
    global  allIPnumsToSend ipNum2Socket debugLevel wCan dims wStatMess
    
    # Get new sizes.
    update idletasks
    
    # Sizes without any menu.
    set w [winfo width .]
    set h [winfo height .]
    set wCanvas [winfo width $wCan]
    set hCanvas [winfo height $wCan]
    
    if {$debugLevel >= 2}  {
	puts "CanvasSizeChange:: w=$w, h=$h, wCanvas=$wCanvas, hCanvas=$hCanvas"
	puts "CanvasSizeChange:: force=$force, wm geometry .=[wm geometry .]"
    }
    
    # Only if size changed or if force.
    if {$where != "0"}  {
	if {($dims(wCanvas) != $wCanvas) || ($dims(hCanvas) != $hCanvas) || \
	  $force }  {
	    #puts "CanvasSizeChange:: propagate"
	    if {$where == "all"}  {
		foreach ip $allIPnumsToSend  {
		    puts $ipNum2Socket($ip) "RESIZE: $wCanvas $hCanvas"
		}
	    } else {
		
		# We must have a valid ip number.
		puts $ipNum2Socket($where) "RESIZE: $wCanvas $hCanvas"
	    }
	}
    }
    
    # Update actual size values. 'Root' no menu, 'Tot' with menu.
    set dims(wStatMess) [winfo width $wStatMess]
    set dims(wRoot) $w
    set dims(hRoot) $h
    set dims(wTot) $dims(wRoot)
    set dims(hTot) [expr $dims(hRoot) + $dims(hMenu)]
    set dims(wCanvas) $wCanvas
    set dims(hCanvas) $hCanvas
}

# SetCanvasSize --
#
#   From the canvas size, 'cw' and 'ch', set the total application size.

proc SetCanvasSize { cw ch }  {
    global  dims debugLevel
        
    # Compute new root size from the desired canvas size.
    set hRootFinal [expr $ch + $dims(hStatus) + $dims(hComm) + $dims(hTop)]
    set wRootFinal [expr $cw + 56]
    wm geometry . ${wRootFinal}x${hRootFinal}
    if {$debugLevel >= 2}  {
	puts "SetCanvasSize:: cw=$cw, ch=$ch, hRootFinal=$hRootFinal, \
	  wRootFinal=$wRootFinal"
    }
}

# SetNewWMMinsize --
#
#   If a new entry in the communication frame is added, or the shortcut button
#   frame is collapsed or expanded, we need to set a new minsize for the
#   total application size, and update the necassary 'dims' variables.
#   It must be called 'after idle' to be sure all windows have been updated
#   properly.

proc SetNewWMMinsize {  }  {
    global  dims commFrame debugLevel
    
    set dims(hTop) 0
    if {[winfo exists .frtop]}  {
	set dims(hTop) [winfo height .frtop]
    }
    set dims(hComm) [winfo height $commFrame]
    set dims(hMinRoot) [expr $dims(hMinCanvas) + $dims(hStatus) + $dims(hComm) +  \
      $dims(hTop)]
    set dims(hMinTot) [expr $dims(hMinRoot) + $dims(hMenu)]
    
    # Note: wm minsize is *with* the menu!!!
    wm minsize . $dims(wMinTot) $dims(hMinTot)
    
    if {$debugLevel >= 2}  {
	puts "SetNewWMMinsize:: dims(hComm)=$dims(hComm),  \
	  dims(hMinRoot)=$dims(hMinRoot), dims(hMinTot)=$dims(hMinTot), \
	  dims(hTop)=$dims(hTop)"
    }
}	    

# GetAutoFitSize ---
#
#    Gives a new smaller size of 'theMovie' if it is too large for canvas 'w'.
#    It is rescaled by factors of two.

proc GetAutoFitSize { w theMovie }  {

    set factor 2.0
    set canw [winfo width $w]
    set canh [winfo height $w]
    set msize [$theMovie size]
    set imw [lindex [lindex $msize 0] 1]
    set imh [lindex [lindex $msize 1] 1]
    set maxRatio [max [expr $imw/($canw + 0.0)] [expr $imh/($canh + 0.0)]]
    if {$maxRatio >= 1.0} {
	set k [expr ceil(log($maxRatio)/log(2.0))]
	return [list [expr int($imw/pow(2.0, $k))] [expr int($imh/pow(2.0, $k))]]
    } else {
	return [list $imw $imh]
    }
}


#   CancelAllPutGetAndPendingOpen ---
#
#   It is supposed to stop every put and get operation taking place.
#   This may happen when the user presses a stop button or something.

proc CancelAllPutGetAndPendingOpen  {  }  {

    PutFileCancelAll
    GetFileCancelAll
    OpenCancelAllPending
}

# GetTransferRateFromTiming --
#
#   From 'listClicksBytes' which is a list of {clockClicks bytes}, the transfer
#   rate is computed as bytes per seconds.
#   A moving average of the latest elements is used.

proc GetTransferRateFromTiming  { listClicksBytes }  {
    global  clockClicksPerSec debugLevel
    
  if {$debugLevel >= 2}  {
      #puts "GetTransferRateFromTiming:: entry"
  }
    set n [llength $listClicksBytes]
    set nAve 3
    set istart [expr $n - $nAve]
    if {$istart < 0}  {
	set istart 0
    }
    set iend [expr $n - 1]
    set sumBytes [expr [lindex [lindex $listClicksBytes $iend] 1] -  \
      [lindex [lindex $listClicksBytes $istart] 1]]
    set sumClicks [expr [lindex [lindex $listClicksBytes $iend] 0] -  \
      [lindex [lindex $listClicksBytes $istart] 0]]
    
    # Treat the case with wrap around. (Guess)
    if {$sumClicks <= 0}  {
	set sumClicks $clockClicksPerSec
    }
    set aveBytesPerClick [expr $sumBytes / ($sumClicks + 1.0)]
    set aveBytesPerSec [expr $aveBytesPerClick * $clockClicksPerSec]
    return $aveBytesPerSec
}

# BytesPerSecFormatted --
#
#   Returns the transfer rate as a nicely formatted text string.

proc BytesPerSecFormatted  { bytesPerSec }  {

    # Find format: bytes or k.
    if {$bytesPerSec < 1000}  {
	set txtRate "$bytesPerSec bytes/sec"
    } elseif {$bytesPerSec < 1000000}  {
	set txtRate [list [format "%.1f" [expr $bytesPerSec/1000.0] ]Kb/sec]
    } else {
	set txtRate [list [format "%.1f" [expr $bytesPerSec/1000000.0] ]Mb/sec]
    }
    return $txtRate
}

# AddToKnownFiles, GetKnownPathFromTail --
#
#   Keeps track of already opened or received images/movies files
#   through the synced lists 'knownFiles' and 'knownPaths'.
#   The 'fileTail' name is always the native file name which on the mac
#   my lack an extension.

proc AddToKnownFiles { fileTail filePath it }  {
    global  knownFiles knownPaths imageItno2FileTail
 
    # Check first if its not already there.
    if {[GetKnownPathFromTail $fileTail] == ""}  {
	lappend knownFiles "$fileTail"
	lappend knownPaths "$filePath"
    }
    set imageItno2FileTail($it) $fileTail
}
		
proc GetKnownPathFromTail { fileTail }  {
    global  knownFiles knownPaths tcl_platform
   
    set dot_ {\.}
    set ind [lsearch -exact $knownFiles $fileTail]
    
    # On mac it is only necessary that the rootnames agree.
    if {$tcl_platform(platform)  == "macintosh"}  {
	set fileRoot [file rootname $fileTail]
	set ind [lsearch -regexp $knownFiles "^${fileRoot}$dot_*|^${fileRoot}$"]	
    }
    
    # Return nothing if its not there.
    if {$ind < 0}  {
	return ""
    } else  {
	set path [lindex $knownPaths $ind]
	
	# Check if the file exists.
	if {[file exists $path]}  {
	    return $path
	} else  {
	    return ""
	}
    }
}

# IsFileInCacheQ --
#
#   Checks if file with tail of file name 'fileTail' exist in the incoming
#   directory taking into account the constraints of 'prefs(checkCache)'.

proc IsFileInCacheQ { fileTail }  {
    global  prefs thisPath
    
    if {$prefs(checkCache) == "never"}  {
	return 0
    }
    set fullName [file join $thisPath incoming $fileTail]

    set ans [file exists $fullName]
    if {$ans == 1}  {
	set ans [file isfile $fullName]
    } else  {
	return 0
    }
    # At this stage we are sure that the file is actually there.
    if {$prefs(checkCache) == "always"}  {
	return 1
    }
    # Check if the time limit has passed.
    if {![FileOlderThan $fullName $prefs(checkCache)]}  {
	return 1
    } else  {
	return 0
    }
}

# FileOlderThan --
#
#   Find out if file older than 'timespan'.
#   'timespan' can be: "launch", "min", "hour", "day", "30days".

proc FileOlderThan { filePath timespan }  {
    global  tmsec launchSecs
    
    if {[lsearch -exact {"launch" "min" "hour" "day" "30days"} $timespan] < 0}  {
	return 1
    }
    set fileTime [file mtime $filePath]
    set thisTime [clock seconds]
    set ans 1
    if {$timespan == "launch"}  {
	if {$fileTime > $launchSecs}  {
	    set ans 0
	}
    } else  {
	if {[expr $thisTime - $fileTime] < $tmsec($timespan)}  {
	    set ans 0
	}
    }
}

# AppGetFocus --
#
#   Check clipboard and activate corresponding menus.    

proc AppGetFocus {  }  {
    
    #puts "AppGetFocus::"
    # Check the clipboard or selection.
    if {[catch {selection get -selection CLIPBOARD} sel]}  {
	return
    } elseif {[string length $sel] > 0}  {
	.menu.edit entryconfigure *Paste* -state normal
   }
}

# DoQuit ---
#
#    Is called just before quitting to collect some state variables which
#    we want to save for next time.

proc DoQuit  {  }  {
    global  dims prefs wCan
    
    # Before quitting, save user preferences. 
    # Need to collect some of them first.
    # Position of root window.
    # We want to save wRoot and hRoot as they would be without any clients 
    # in the communication frame.
    
    foreach {dims(wRoot) hRoot dims(x) dims(y)} [ParseWMGeometry .] {}
    set dims(hRoot) [expr $dims(hCanvas) + $dims(hStatus) +  \
      $dims(hCommClean) + $dims(hTop)]
    set prefs(visToolbar) [IsShortcutButtonVisable]
    
    # If we used 'Edit/Revert To/Application Defaults' be sure to reset...
    set prefs(firstLaunch) 0
    
    # Save to the preference file and quit...
    PreferencesSaveToFile
    exit
}

#--- The Menus ------------------------------------------------------------------

proc FixMenusWhenSelection  { w }  {

    set wClass [winfo class $w]
    #puts "FixMenusWhenSelection:: w=$w, wClass=$wClass"
    
    # Do different things dependent on the type of widget.
    if {[string compare $wClass "Canvas"] == 0}  {
	if {([llength [$w find withtag selected]] == 0) &&  \
	  ([llength [$w select item]] == 0)}  {
	    
	    # There is no selection in the canvas.
	    .menu.edit entryconfigure *Cut* -state disabled
	    .menu.edit entryconfigure *Copy* -state disabled
	} else {
	    .menu.edit entryconfigure *Cut* -state normal
	    .menu.edit entryconfigure *Copy* -state normal
	}
    } elseif {[string compare $wClass "Entry"] == 0}  {
	if {[catch {selection get} sel]}  {
	    if {[catch {selection get -selection CLIPBOARD} sel]}  {
		# No selection or clipboard data.
		.menu.edit entryconfigure *Cut* -state disabled
		.menu.edit entryconfigure *Copy* -state disabled
	    }
	} else  {
	    .menu.edit entryconfigure *Cut* -state normal
	    .menu.edit entryconfigure *Copy* -state normal
	}
    }
}

# BuildItemMenu ---
#
#    Creates an item menu from all files in the specified directory.
#    
# Arguments:
# 
#    mt         The menu path.
#    itemDir    The directory to search the item files in.

proc BuildItemMenu { mt itemDir }  {
    global  wCan
    
    # Save old dir, and cd to the wanted one; glob works in present directory.
    set oldDir [pwd]
    cd $itemDir
    set allItemFiles [glob -nocomplain *]
    foreach itemFile $allItemFiles {
	
	# Keep only .can files and dirs.
	if {[string compare [file extension $itemFile] ".can"] == 0} {
	    $mt add command -label [file rootname $itemFile]  \
	      -command [list DrawCanvasItemFromFile $wCan  \
	      [file join $itemDir $itemFile]]
	} elseif {[file isdirectory $itemFile]} {
	    
	    # Build menus recursively. Consider: 1) large chars, 2) multi words.
	    regsub -all " " $itemFile "_" modFile
	    set subm [string tolower $modFile]
	    set msub [menu ${mt}.${subm} -tearoff 0]
	    $mt add cascade -label $itemFile -menu $msub
	    BuildItemMenu $msub [file join $itemDir $itemFile]
	}
    }
    cd $oldDir
}

# BuildFontMenu ---
# 
#    Creates the font selection menu, and removes any old.
#    
# Arguments:
# 
#    mt         The menu path.
#    allFonts   List of names of the fonts.

proc BuildFontMenu { mt allFonts }  {
    global  prefs

    $mt delete 0 end
    foreach afont $allFonts {
	$mt add radio -label $afont -variable prefs(font)   \
	  -command [list FontChanged "name"]
    }
    
    # Be sure that the presently selected font family is still there,
    # else choose helvetica.
    
    if {[lsearch -exact $allFonts $prefs(font)] == -1}  {
	set prefs(font) "Helvetica"
    }
}

# FontChanged ---
# 
#     Callback procedure for the font menu. When new font name, size or weight,
#     and we have focus on a text item, change the font spec of this item.
#
# Arguments:
#
#     what        name, size or weight.

proc FontChanged  { what }  {
    global  prefs wCan allIPnumsToSend ipNum2Socket fontSize2Points  \
      fontPoints2Size

    # If there is a focus on a text item, change the font for this item.
    set idfocus [$wCan focus]
    if {[string length $idfocus] > 0}  {
	set theItno [CanvasGetItnoFrom $wCan focus]
	if {[string length $theItno] == 0}  {
	    return
	}
	set fontSpec [$wCan itemcget $theItno -font]
	if {[llength $fontSpec] > 0}  {
	    array set whatToInd {name 0 size 1 weight 2}
	    array set whatToPref {name font size fontSize weight fontWeight}

	    # Need to translate html size to point size.
	    if {$what == "size"}  {
		set newFontSpec [lreplace $fontSpec   \
		  $whatToInd($what) $whatToInd($what)  \
		  $fontSize2Points($prefs($whatToPref($what)))]
	    } else {
		set newFontSpec [lreplace $fontSpec   \
		  $whatToInd($what) $whatToInd($what) $prefs($whatToPref($what))]
	    }

	    $wCan itemconfigure $theItno -font $newFontSpec
	    
	    # Need to translate back to actual point size to html size.
	    set theFontSizeInPoints [lindex $newFontSpec 1]
	    set newFontSpec [lreplace $newFontSpec 1 1   \
	      $fontPoints2Size($theFontSizeInPoints)]
	
	    # Let all remote client also know.
	    foreach ip $allIPnumsToSend  {
		puts $ipNum2Socket($ip)  \
		  [list "CANVAS:" itemconfigure $theItno -font $newFontSpec]
	    }	    
	}
    }
}

set ::SplashScreen::startMsg "Build menus..."
menu .menu -tearoff 0

#--- Apple ---------------------------------------------------------------------

if {[string compare $tcl_platform(platform) "macintosh"] == 0}  {
    set m [menu .menu.apple -tearoff 0]
    .menu add cascade -menu $m
    $m add command -label "About Whiteboard..." -command   \
      {::SplashScreen::SplashScreen $wDlgs(splash)}
}	    

#--- File ----------------------------------------------------------------------

set m [menu .menu.file -tearoff 0]
.menu add cascade -label "File " -menu $m -underline 0

# ...
$m add command -label "Start Server" -command "DoStartServer $thisServPort" 
if {!$prefs(symmetricNet) && !$prefs(thisCentralServ)}  {
    .menu.file entryconfigure "*Start Server*" -state disabled
}
# If we are the central server, never do 'Open Connection...'.
$m add command -label "Open Connection..."	\
  -command DoOpenConnection -accelerator $osprefs(mod)+O -underline 0
if {!$prefs(symmetricNet) && $prefs(thisCentralServ)}  {
    $m entryconfigure "*Open Connection*" -state disabled
}
$m add separator
$m add command -label "Open Image/Movie..." -command  \
  "ImportImageOrMovieDlg $wCan" \
  -accelerator $osprefs(mod)+I -underline 5
if {$prefs(QuickTimeTcl) && $prefs(http)}  {
    $m add command -label "Open URL Stream..."   \
      -command ::OpenMulticast::OpenMulticast
}
$m add separator
$m add command -label "Put Canvas" -command "DoPutCanvasDlg $wCan"  \
  -state disabled
$m add command -label "Get Canvas..." -command "DoGetCanvas $wCan"  \
  -state disabled
$m add command -label "Put File..." -command "PutFileDlg" -state disabled
$m add command -label "Stop Put/Get/Open" -command  \
  "CancelAllPutGetAndPendingOpen" \
  -state normal
$m add separator
$m add command -label "Open Canvas..." -command "DoOpenCanvasFile $wCan"
$m add command -label "Save Canvas..." -command "DoSaveCanvasFile $wCan"  \
  -accelerator $osprefs(mod)+S -underline 0
$m add separator
$m add command -label "Page Setup..." -command "::PSPageSetup::PSPageSetup .page"
$m add command -label "Save As Postscript..." -command "SavePostscript $wCan"
if {[string compare $tcl_platform(platform) "unix"] == 0}  {
    $m add command -label "Print Canvas..." -command   \
      "::PrintPSonUnix::PrintPSonUnix $wDlgs(print) $wCan" -underline 0
}
$m add command -label "Quit" -command DoQuit -accelerator $osprefs(mod)+Q

#--- Edit ----------------------------------------------------------------------

set m [menu .menu.edit -tearoff 0]
.menu add cascade -label "Edit " -menu $m -underline 0
$m add command -label "All" -command "SelectAll $wCan"  \
  -accelerator $osprefs(mod)+A -underline 0
$m add command -label "Erase All" -command "DoEraseAll $wCan"
$m add separator

# We have an alternative here: is virtual events as a level of indirection
# between events and the actions.

$m add command -label "Cut" -command "CopySelectedToClipboard \[focus] cut"  \
  -accelerator $osprefs(mod)+X -state disabled
$m add command -label "Copy" -command "CopySelectedToClipboard \[focus] copy"  \
  -accelerator $osprefs(mod)+C -state disabled -underline 0
$m add command -label "Paste" -command "PasteFromClipboardTo \[focus]"  \
  -accelerator $osprefs(mod)+V -state disabled -underline 0

$m add separator
$m add command -label "Raise" -command "RaiseOrLowerItems $wCan raise"  \
  -accelerator $osprefs(mod)+R -state disabled -underline 0
$m add command -label "Lower" -command "RaiseOrLowerItems $wCan lower"  \
  -accelerator $osprefs(mod)+L -state disabled -underline 0
$m add separator
$m add command -label "Larger" -command   \
  "CanvasResizeItem $wCan $prefs(scaleFactor)"  \
  -accelerator $osprefs(mod)+>
$m add command -label "Smaller" -command   \
  "CanvasResizeItem $wCan [expr 1.0/$prefs(scaleFactor)]"   \
  -accelerator $osprefs(mod)+<
$m add command -label "Image Larger" -command "ResizeImage $wCan 2 sel auto"
$m add command -label "Image Smaller" -command "ResizeImage $wCan -2 sel auto"

#--- Prefs ---------------------------------------------------------------------

set m [menu .menu.prefs -tearoff 0]
.menu add cascade -label "Preferences " -menu $m -underline 0
$m add command -label "Background Color..." -command "SetCanvasBgColor $wCan"
$m add checkbutton -label "Grid" -variable prefs(canGridOn)  \
  -onvalue 1 -offvalue 0
set mt [menu .menu.prefs.thick -tearoff 0]
$m add cascade -label "Thickness" -menu .menu.prefs.thick -underline 0
$mt add radio -label 1 -variable prefs(penThick)
$mt add radio -label 2 -variable prefs(penThick)
$mt add radio -label 4 -variable prefs(penThick)
$mt add radio -label 6 -variable prefs(penThick)
set mt [menu .menu.prefs.brthick -tearoff 0]
$m add cascade -label "Brush Thickness" -menu .menu.prefs.brthick
$mt add radio -label 8 -variable prefs(brushThick)
$mt add radio -label 10 -variable prefs(brushThick)
$mt add radio -label 12 -variable prefs(brushThick)
$mt add radio -label 16 -variable prefs(brushThick)
$m add checkbutton -label "Fill" -variable prefs(fill) -underline 0
set mt [menu .menu.prefs.smooth -tearoff 0]
$m add cascade -label "Line Smoothness" -menu .menu.prefs.smooth
$mt add radio -label None -value 0 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 0}
$mt add radio -label 2 -value 2 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 4 -value 4 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 6 -value 6 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 8 -value 8 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
$mt add radio -label 10 -value 10 -variable prefs(splinesteps)  \
  -command {set prefs(smooth) 1}
set mt [menu .menu.prefs.arcs -tearoff 0]
$m add cascade -label "Arcs" -menu .menu.prefs.arcs -underline 0
$mt add radio -label "Pieslice" -value pieslice -variable prefs(arcstyle)
$mt add radio -label "Chord" -value chord -variable prefs(arcstyle)
$mt add radio -label "Arc" -value arc -variable prefs(arcstyle)
set mt [menu .menu.prefs.cmove -tearoff 0]
$m add cascade -label "Shift Constrain" -menu $mt
$mt add radio -label "To 90 degrees" -variable prefs(45) -value 0
$mt add radio -label "To 45 degrees" -variable prefs(45) -value 1
$m add sep
set mt [menu .menu.prefs.font -tearoff 0]
$m add cascade -label "Font" -menu $mt -underline 0
BuildFontMenu $mt $prefs(canvasFonts)

set mt [menu .menu.prefs.size -tearoff 0]
$m add cascade -label "Size" -menu $mt -underline 0
$mt add radio -label "1" -variable prefs(fontSize) -command {FontChanged size}
$mt add radio -label "2" -variable prefs(fontSize) -command {FontChanged size}
$mt add radio -label "3" -variable prefs(fontSize) -command {FontChanged size}
$mt add radio -label "4" -variable prefs(fontSize) -command {FontChanged size}
$mt add radio -label "5" -variable prefs(fontSize) -command {FontChanged size}
$mt add radio -label "6" -variable prefs(fontSize) -command {FontChanged size}
set mt [menu .menu.prefs.weight -tearoff 0]
$m add cascade -label "Weight" -menu $mt -underline 0
$mt add radio -label "Normal" -variable prefs(fontWeight) -value normal  \
   -command {FontChanged weight}
$mt add radio -label "Bold" -variable prefs(fontWeight) -value bold  \
   -command {FontChanged weight}
$mt add radio -label "Italic" -variable prefs(fontWeight) -value italic  \
   -command {FontChanged weight}
$m add sep
if {$prefs(TclSpeech)}  {
    $m add checkbutton -label "Speech" -variable prefs(TclSpeechOn)
}
$m add checkbutton -label "Privacy" -variable prefs(privacy) -onvalue 1 -offvalue 0
set mt [menu .menu.prefs.cache -tearoff 0]
$m add cascade -label "Check Cache" -menu $mt
$mt add radio -label "Always" -value "always" -variable prefs(checkCache)
$mt add radio -label "At Launch" -value "launch" -variable prefs(checkCache)
$mt add radio -label "> 1 Hour" -value "hour" -variable prefs(checkCache)
$mt add radio -label "> 1 Day" -value "day" -variable prefs(checkCache)
$mt add radio -label "> 1 Week" -value "week" -variable prefs(checkCache)
$mt add radio -label "Never" -value "never" -variable prefs(checkCache)
$m add command -label "Edit Shortcuts..."   \
  -command "::EditShortcuts::EditShortcuts $wDlgs(editShorts) shortcuts"
$m add command -label "Edit Fonts..."    \
  -command "::EditFontFamilies::EditFontFamilies $wDlgs(editFonts)"
$m add command -label "Network Setup..." -command "::NetworkSetup::NetworkSetup"
$m add sep
set mt [menu .menu.prefs.revert -tearoff 0]
$m add cascade -label "Revert To" -menu $mt
$mt add command -label "User Defaults" -command PreferencesResetToUserDefaults
$mt add command -label "Application Defaults"   \
  -command PreferencesResetToAppDefaults

#--- Item menu (temporary placement) -------------------------------------------

set m [menu .menu.items -tearoff 0]
.menu add cascade -label "Items " -menu $m -underline 0
BuildItemMenu $m $prefs(itemDir)

#--- Camera/Mic ----------------------------------------------------------------
# Only supported by the sequence grabber in QuickTime.

if {$prefs(QuickTimeTcl)}  {
    set m [menu .menu.cam -tearoff 0]
    .menu add cascade -label "Camera/Mic " -menu $m -underline 0
    $m add command -label "Camera Action"  \
      -command [list DisplaySequenceGrabber $wCan] -state normal
    $m add checkbutton -label "Pause"  \
      -command [list SetVideoConfig $wCan pause]
    $m add checkbutton -label "Picture"  \
      -command [list SetVideoConfig $wCan picture]
    set mt [menu .menu.cam.size -tearoff 0]
    $m add cascade -label "Video Size" -menu $mt
    $mt add radio -label "Quarter" -value "quarter" -variable prefs(videoSize) \
      -command [list SetVideoConfig $wCan size]
    $mt add radio -label "Half" -value "half" -variable prefs(videoSize) \
      -command [list SetVideoConfig $wCan size]
    $mt add radio -label "Full" -value "full" -variable prefs(videoSize) \
      -command [list SetVideoConfig $wCan size]
    set mt [menu .menu.cam.zoom -tearoff 0]
    $m add cascade -label "Zoom" -menu $mt
    $mt add radio -label "x 1" -value 1.0 -variable prefs(videoZoom) \
      -command [list SetVideoConfig $wCan zoom]
    $mt add radio -label "x 2" -value 2.0 -variable prefs(videoZoom) \
      -command [list SetVideoConfig $wCan zoom]
    $mt add radio -label "x 3" -value 3.0 -variable prefs(videoZoom) \
      -command [list SetVideoConfig $wCan zoom]
    $mt add radio -label "x 4" -value 4.0 -variable prefs(videoZoom) \
      -command [list SetVideoConfig $wCan zoom]
    $m add command -label "Video Settings..."  \
      -command [list SetVideoConfig $wCan videosettings]
}

#--- Info ----------------------------------------------------------------------

# Here it's important to delay substutution until a menu is invoked.

set m [menu .menu.info -tearoff 0]
.menu add cascade -label "Info " -menu $m -underline 0
$m add command -label "On Server..."  \
  -command {ShowInfoServer $wDlgs(infoServ) $thisIPnum} -state normal  \
  -underline 3
$m add command -label "On Clients..."   \
  -command {ShowInfoClients $wDlgs(infoClient) $allIPnumsFrom}  \
  -state disabled -underline 3
$m add command -label "On Plugins..."   \
  -command [list InfoOnPlugins .plugs] -state normal -underline 3
if {[string compare $tcl_platform(platform) "macintosh"] != 0}  {
    $m add command -label "About Whiteboard..."   \
      -command [list ::SplashScreen::SplashScreen $wDlgs(splash)] -underline 6
}	    

#-------------------------------------------------------------------------------
# End menus; place the menubar.
. configure -menu .menu

set ::SplashScreen::startMsg "Build user interface..."

# Create the mapping between Html sizes and font point sizes dynamically.
CreateFontSizeMapping

# Shortcut buttons at top? Do we want the toolbar to be visible.
if {$prefs(visToolbar)} {
    CreateShortcutButtonPad init
} else {
    CreateShortcutButtonPad init off
}

# Make the tool button pad.
pack [frame .fmain] -side top -fill both -expand true
pack [frame .fmain.frleft] -side left -fill y
pack [frame $btFrame] -side top
pack [label .fmain.frleft.pad -relief raised -borderwidth 1]   \
  -fill both -expand true

# The 'Coccinella'.
if {$prefs(coccinellaMovie)}  {
    pack [Movie .fmain.frleft.padphoto -controller 0   \
      -file [file join images beetle<->igelpiga.mov]]   \
      -in .fmain.frleft.pad -side bottom
    .fmain.frleft.padphoto palindromeloopstate 1
} else  {
    image create photo igelpiga -format gif -file [file join images igelpiga.gif]
    pack [label .fmain.frleft.padphoto -borderwidth 0 -image igelpiga]   \
      -in .fmain.frleft.pad -side bottom
}

# ...and the drawing canvas.
canvas $wCan -height $dims(hCanOri) -width $dims(wCanOri) -relief raised -bd 1	\
  -highlightthickness 0 -background $prefs(bgColCanvas)
pack $wCan -fill both -expand true -side right

# Make the tool buttons and invoke the one from the prefs file.
CreateAllButtons
ClickToolButton $wCan $btNo2Name($prefs(btState))

# Make the connection frame. Get icons.
image create photo im_handle -format gif -file [file join images resizehandle.gif]
image create photo bluebox -format gif -file [file join images bluebox.gif]
image create photo contact_off -format gif -file [file join images contact_off.gif]
image create photo contact_on -format gif -file [file join images contact_on.gif]
pack [frame .fcomm] -side top -fill x

# Status message part.
pack [frame .fcomm.st -relief raised -borderwidth 1]  \
  -side top -fill x -pady 0
pack [frame .fcomm.stat -relief groove -bd 2]  \
  -side top -fill x -padx 10 -pady 2 -in .fcomm.st
image create photo im_wave -format gif -file [file join images wave.gif]
pack [canvas $wStatMess -bd 0 -highlightthickness 0 -height 14]   \
  -side left -pady 1 -padx 6 -fill x -expand true
$wStatMess create text 0 0 -anchor nw -text $statMess -font $sysFont(s)  \
  -tags stattxt

# The labels in comm frame.
pack [frame $commFrame  -relief raised -borderwidth 1] -side left
label $commFrame.comm -text "  Remote address:" -width 22 -anchor w
label $commFrame.user -text "  User:" -width 14 -anchor w
grid $commFrame.comm $commFrame.user -sticky nws -pady 0

# A min height was necessary here to make room for switching the icon of this row.
grid rowconfigure $commFrame 0 -minsize 21

# Rest of labels depend on the actual network setup.
UpdateCommFrameHeader
pack [frame .fcomm.pad -relief raised -borderwidth 1] -side top   \
  -fill both -expand true
pack [label .fcomm.pad.hand -relief flat -borderwidth 0 -image im_handle]  \
  -side right -anchor sw


# Trigger changes when certain variables are changed or set.
trace variable statMess w TraceStatusMessage
trace variable prefs w TracePreferences

# Add things that are defined in the prefs file and not updated else.
DoCanvasGrid

# Setting the window position never hurts.
wm geometry . +$dims(x)+$dims(y)

# Map the main window to screen if hidden before.
wm deiconify .

# Setting total (root) size however, should only be done if set in pref file!
# This need to be fixed!!!!!!!!!!!!!!!!!
if {$dims(wRoot) > 1 && $dims(hRoot) > 1}  {
    wm geometry . $dims(wRoot)x$dims(hRoot)
}

# Add virtual events.
#event add <<Cut>> <$osprefs(mod)-Key-x>
#event add <<Copy>> <$osprefs(mod)-Key-c>
#event add <<Paste>> <$osprefs(mod)-Key-v>

# Canvas copy/paste.
#bind $wCan <<Cut>> "CopySelectedToClipboard $wCan cut"
#bind $wCan <<Copy>> "CopySelectedToClipboard $wCan copy"
#bind $wCan <<Paste>> "PasteFromClipboardToCanvas $wCan"

# Entry copy/paste.
bind Entry <B1-Motion> {+ FixMenusWhenSelection %W}
bind Entry <Button-1> {+ FixMenusWhenSelection %W}
bind Entry <FocusIn> {+ FixMenusWhenSelection %W}

# Various bindings.
bind $wCan <BackSpace> "DeleteItem $wCan %x %y selected"
bind $wCan <Control-Key-d> "DeleteItem $wCan %x %y selected"

# This should be used for showing, for instance, text tracks of mp3.
#bind QTFrame <Enter> {puts "QTFrame <Enter>, text tracks of mp3"}

# Start the server. It was necessary to have an 'update idletasks' command here
# because when starting the script directly, and not from within wish, somehow
# there was a timing problem in 'DoStartServer'.
# Dont start the server if we are a client in a centralized network.

update idletasks
if {$prefs(autoStartServer) &&   \
  ($prefs(symmetricNet) || $prefs(thisCentralServ))}  {
    after 400 "DoStartServer $thisServPort"
}
wm title . "Whiteboard"

# A trick to let the window manager be finished before getting the geometry.
after 500 FindWidgetGeometryAtLaunch

bind . <FocusIn> AppGetFocus

# If user just click the close box, be sure to save prefs first.
wm protocol . WM_DELETE_WINDOW DoQuit

# Update size info when application is resized.
bind $wCan <Configure> {CanvasSizeChange "all"}

# Is it the first time it is launched, then show the welcome canvas.
if {$prefs(firstLaunch)}  {
    eval [list DoOpenCanvasFile $wCan $prefs(welcomeFile)]
}
set prefs(firstLaunch) 0

# At this point we should be finished with the launch and delete the splash screen.
set ::SplashScreen::startMsg ""
after 500 {catch {destroy $wDlgs(splash)}}

#*** The server part ***************************************************
#
#   The lines below is just a skeleton of how a safe Tk interpreter could
#   be setup. It needs a frame (with -container 1) to place its toplevel
#   into, and then to create a canvas for both the safe server interpreter
#   and the local drawing commands. Since the sharing of a common canvas
#   seems to be fairly complicated, this path has not been pursued?.
#   Perhaps it is possible to implement in a smart way via 'interp alias'.
#   The code is for possible implementations in the future.
#
# Create the interpreter. Only here for possible future implementation.

if {0 && $prefs(makeSafeServ)}  {
    ::safe::interpCreate serverInterp
    
    # We need to load Tk explicitly. 
    ::safe::loadTk serverInterp -use "a frame with -container 1"   
    if {$debugServerLevel >= 2}  {
	puts "issafe: [interp issafe serverInterp]"
    }
    interp eval serverInterp [list set servCan $servCan]
}

#-------------------------------------------------------------------------------
