
#  Dialogs.tcl ---
#  
#      This file is part of the whiteboard application. It implements some
#      of the dialogs. 
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

#  ::GetCanvas:: implements the dialog for choosing which client to get the 
#  canvas from.
    
namespace eval ::GetCanvas::  {
    
    namespace export GetCanvas
    
    # Wait for this variable to be set.
    variable finished
    
    # The ip name to get canvas from.
    variable getIPName
}

proc ::GetCanvas::GetCanvas { w }  {
    global  allIPnumsToSend ipNum2Name ipName2Num sysFont prefs tcl_platform
    
    variable finished
    variable getIPName
    
    # Build list of ip names.
    set ipNames {}
    foreach ip $allIPnumsToSend  {
	if {[info exists ipNum2Name($ip)]}  {
	    lappend ipNames $ipNum2Name($ip)
	}
    }
    if {[llength $ipNames] == 0}  {
	return 
    }
    set finished -1
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Get Canvas"
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    
    # Labelled frame.
    set wcfr [frame $w.frall.fr -borderwidth 0]
    pack $wcfr -side top -fill both -ipadx 10 -ipady 6 -in $w.frall
    set wcont [LabeledFrame $wcfr "Get Canvas"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont.frin]
    pack $frtot
    message $frtot.msg -borderwidth 0 -font $sysFont(s) -aspect 500 \
      -text "Choose client from which you want to get the canvas.\
      Your own canvas will be erased."
    eval {tk_optionMenu $frtot.optm [namespace current]::getIPName}  \
      $ipNames
    $frtot.optm configure -highlightthickness 0  \
      -background $prefs(bgColGeneral) -foreground black
    grid $frtot.msg -column 0 -row 0 -columnspan 2 -padx 4 -pady 2 -sticky news
    grid $frtot.optm -column 1 -row 1 -padx 4 -pady 0 -sticky e
    
    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]
    pack [button $frbot.btconn -text "    Get    " -default active \
      -command "set [namespace current]::finished 1"]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "   \
      -command "set [namespace current]::finished 2"]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -padx 8 -pady 6
    
    wm resizable $w 0 0
    bind $w <Return> "set [namespace current]::finished 1"
    
    # Grab and focus.
    focus $w
    catch {grab $w}
    
    # Wait here for a button press.
    tkwait variable [namespace current]::finished
    
    catch {grab release $w}
    catch {destroy $w}
    
    if {$finished == 1 &&  \
      [info exists ipName2Num($getIPName)]}  {
	return $ipName2Num($getIPName)
    } else {
	return ""
    }
}

#-- end ::GetCanvas:: ----------------------------------------------------------

#  InfoOnPlugins ---
#  
#      It implements the dialog for presenting the loaded packages or helper 
#      applications.

proc InfoOnPlugins  { w }  {
    global  sysFont prefs plugin supSuff tcl_platform
    
    # Check first of there are *any* plugins.
    set anyPlugin 0
    foreach plugtype $plugin(all)  {
	if {$prefs($plugtype)}  {
	    set anyPlugin 1
	    break
	}
    }
    if {!$anyPlugin}  {
	tk_messageBox -message   \
	  "There are no plugins or helper applications identified\
	  on this system." -icon warning -type ok
	return  
    }
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Plugin Info"
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall
    set txtLength 180
    
    # Try the known plugind and apps, and make a labelled frame for each.
    set n 0
    foreach plugtype $plugin(all)  {
	
	if {$prefs($plugtype)}  {
	    pack [frame $w.frtop$n -borderwidth 0] -in $w.frtop -fill x
	    set wcont [LabeledFrame $w.frtop$n $plugin($plugtype,full)]
	    
	    # Frame for everything inside the labeled container.
	    set fr [frame $wcont.fr]
	    label $fr.b1 -text "Name:" -font $sysFont(sb)
	    label $fr.b2 -text $plugtype -font $sysFont(s)
	    label $fr.a1 -text "Type:" -font $sysFont(sb)
	    label $fr.a2 -text $plugin($plugtype,type) -font $sysFont(s)
	    label $fr.d1 -text "Version:" -font $sysFont(sb)
	    if {[info exists plugin($plugtype,ver)]}  {
		set txt $plugin($plugtype,ver)
	    } else  {
		set txt "unknown"
	    }
	    label $fr.d2 -text $txt -font $sysFont(s)
	    label $fr.e1 -text "Suffixes:" -font $sysFont(sb)
	    if {[info exists supSuff($plugtype)] && \
	      [llength $supSuff($plugtype)] > 0}  {
		set txt [join [split $supSuff($plugtype)] ", "]
	    } else  {
		set txt "none"
	    }
	    label $fr.e2 -text $txt  \
	      -wraplength $txtLength -justify left -font $sysFont(s)
	    label $fr.c1 -text "Description:" -font $sysFont(sb)
	    label $fr.c2 -text $plugin($plugtype,desc)  \
	      -wraplength $txtLength -justify left -font $sysFont(s)
	    grid $fr.b1 -column 0 -row 0 -sticky ne
	    grid $fr.b2 -column 1 -row 0 -sticky nw
	    grid $fr.a1 -column 0 -row 1 -sticky ne
	    grid $fr.a2 -column 1 -row 1 -sticky nw
	    grid $fr.d1 -column 0 -row 2 -sticky ne
	    grid $fr.d2 -column 1 -row 2 -sticky nw
	    grid $fr.e1 -column 0 -row 3 -sticky ne
	    grid $fr.e2 -column 1 -row 3 -sticky nw
	    grid $fr.c1 -column 0 -row 4 -sticky ne
	    grid $fr.c2 -column 1 -row 4 -sticky nw
	    pack $fr -side left -padx 20 -fill x
	    pack $wcont -fill x    
	    incr n
	}
    }
    # Button part.
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both  \
      -padx 8 -pady 6
    pack [button $w.ok -text "  OK  " -default active -command "destroy $w"]  \
      -in $w.frbot -side right -padx 5 -pady 5
    wm resizable $w 0 0
    bind $w <Return> "$w.ok invoke"
    
    tkwait window $w
    grab release $w
}

#-- end InfoOnPlugins ----------------------------------------------------------

#  ::NetworkSetup:: ---
#  
#      It implements the dialog for choosing type of network topolgy and some 
#      options.

namespace eval ::NetworkSetup::  {
    
    variable tmpPrefs
    variable wopt
    variable finished
}
    
proc ::NetworkSetup::NetworkSetup  {  }  {
    global  sysFont prefs tcl_platform debugLevel
    
    variable tmpPrefs
    variable wopt
    variable finished

    set w .netsetup
    
    # Set the temporary variables.
    set tmpPrefs(symmetricNet) $prefs(symmetricNet)
    set tmpPrefs(autoConnect) $prefs(autoConnect)          
    set tmpPrefs(multiConnect) $prefs(multiConnect)
    set tmpPrefs(thisCentralServ) $prefs(thisCentralServ)
    set finished -1
    
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Network Setup"
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall    
    set wcont [LabeledFrame $w.frtop "Network Configuration"]
    
    # Frame for everything inside the labeled container.
    set fr [frame $wcont.fr]
    
    message $fr.msg -width 260 -font $sysFont(s) -text  \
      "The network can either be symmetric or centralized.\
      If centralized, only one server should be up running,\
      and all other whiteboards must be clients only."
    
    # The actual options.
    set fropt [frame $fr.fropt]
    set wopt $fropt
    
    # For the symmetric network config.
    radiobutton $fropt.symm -text "Symmetric" -font $sysFont(sb)   \
      -variable [namespace current]::tmpPrefs(symmetricNet)   \
      -value 1
    checkbutton $fropt.auto -text "Auto Connect" -font $sysFont(sb)  \
      -variable [namespace current]::tmpPrefs(autoConnect)
    message $fropt.automsg -width 160 -font $sysFont(s) -text  \
      "If a client connects to this server, connect to that clients\
      server automatically."
    checkbutton $fropt.multi -text "Multi Connect" -font $sysFont(sb)  \
      -variable [namespace current]::tmpPrefs(multiConnect)
    message $fropt.multimsg -width 160 -font $sysFont(s) -text  \
      "Connect automatically to all clients in this network when\
      connecting any of them."
    if {!$prefs(symmetricNet)}  { 
	$fropt.auto configure -state disabled
	$fropt.multi configure -state disabled
    }    
    
    # For the centralized network config.
    radiobutton $fropt.cent -text "Centralized\n(not completed)"  \
      -font $sysFont(sb)   \
      -variable [namespace current]::tmpPrefs(symmetricNet)   \
      -value 0
    radiobutton $fropt.servcent -text "Server" -font $sysFont(sb)    \
      -variable [namespace current]::tmpPrefs(thisCentralServ)  \
      -value 1
    message $fropt.centmsg -width 160 -font $sysFont(s) -text  \
      "Is this the central server? Don't forget to start it..."
    radiobutton $fropt.client -text "Client" -font $sysFont(sb)   \
      -variable [namespace current]::tmpPrefs(thisCentralServ)  \
      -value 0
    message $fropt.clientmsg -width 160 -font $sysFont(s) -text  \
      "Or is it just a plain client."
    if {$prefs(symmetricNet)}  { 
	$fropt.servcent configure -state disabled
	$fropt.client configure -state disabled
    }
    
    grid $fropt.symm -column 0 -row 0 -rowspan 4 -sticky nw -padx 10 -pady 1
    grid $fropt.auto -column 1 -row 0 -sticky w -padx 10 -pady 1
    grid $fropt.automsg -column 1 -row 1 -sticky w -padx 10 -pady 1
    grid $fropt.multi -column 1 -row 2 -sticky w -padx 10 -pady 1
    grid $fropt.multimsg -column 1 -row 3 -sticky w -padx 10 -pady 1
    grid $fropt.cent -column 0 -row 4 -rowspan 4 -sticky nw -padx 10 -pady 1
    grid $fropt.servcent -column 1 -row 4 -sticky w -padx 10 -pady 1
    grid $fropt.centmsg -column 1 -row 5 -sticky w -padx 10 -pady 1
    grid $fropt.client -column 1 -row 6 -sticky w -padx 10 -pady 1
    grid $fropt.clientmsg -column 1 -row 7 -sticky w -padx 10 -pady 1
    
    pack $fr.msg -side top -padx 2 -pady 6
    pack $fropt -side top -padx 5 -pady 6
    pack $fr -side left -padx 2    
    pack $wcont -fill x    
    
    # Button part.
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both  \
      -padx 8 -pady 6
    pack [button $w.ok -text "   OK   " -default active -command   \
      [list [namespace current]::NetworkSetupSave ]]  \
      -in $w.frbot -side right -padx 5 -pady 5
    pack [button $w.cancel -text "Cancel"   \
      -command "set [namespace current]::finished 0"]  \
      -in $w.frbot -side right -padx 5 -pady 5
    wm resizable $w 0 0
    bind $w <Return> "$w.ok invoke"
    
    # Grab and focus.
    focus $w
    catch {grab $w}
    trace variable [namespace current]::tmpPrefs(symmetricNet) w  \
      [namespace current]::TraceNetConfig
    
    tkwait variable [namespace current]::finished
    
    # Clean up.
    trace vdelete [namespace current]::tmpPrefs(symmetricNet) w  \
      [namespace current]::TraceNetConfig
    catch {unset finished}
    catch {grab release $w}
    destroy $w
}

#   NetworkSetupSave ---
#
#   Saves the parameters in the dialog to the actual preferences.

proc ::NetworkSetup::NetworkSetupSave  {  }  {
    global  prefs isServerUp debugLevel
    
    variable tmpPrefs
    variable wopt
    variable finished

    set prefs(symmetricNet) $tmpPrefs(symmetricNet)
    set prefs(autoConnect) $tmpPrefs(autoConnect)          
    set prefs(multiConnect) $tmpPrefs(multiConnect)
    set prefs(thisCentralServ) $tmpPrefs(thisCentralServ)
    
    # Update menus. First symmetric network config.
    if {$prefs(symmetricNet)}  { 
	if {$isServerUp}  {
	    .menu.file entryconfigure "*Start Server*" -state disabled
	} else  {
	    .menu.file entryconfigure "*Start Server*" -state normal
	}
	.menu.file entryconfigure "*Open Connection*" -state normal
    } elseif {!$prefs(symmetricNet)}  {
	
	# A network with a central server.
	if {$prefs(thisCentralServ)}  {
	    if {$isServerUp}  {
		.menu.file entryconfigure "*Start Server*" -state disabled
	    } else  {
		.menu.file entryconfigure "*Start Server*" -state normal
	    }
	    .menu.file entryconfigure "*Open Connection*" -state disabled
	} else  {
	    
	    # We are only a client.
	    .menu.file entryconfigure "*Start Server*" -state disabled
	    .menu.file entryconfigure "*Open Connection*" -state normal
	}
    }
    
    # Other consistency updates needed.
    UpdateCommFrameHeader
    set finished 1
}

proc ::NetworkSetup::TraceNetConfig { name index op }  {
    upvar #0 $name locName
    global  prefs
    
    variable wopt

    set fropt $wopt
    #puts "TraceNetConfig:: locName=$locName($index), index=$index, op=$op"
    if {$locName($index)}  { 
	$fropt.auto configure -state normal
	$fropt.multi configure -state normal
	$fropt.servcent configure -state disabled
	$fropt.client configure -state disabled
    } else  {
	$fropt.auto configure -state disabled
	$fropt.multi configure -state disabled
	$fropt.servcent configure -state normal
	$fropt.client configure -state normal
    }
}

#-- end ::NetworkSetup:: -------------------------------------------------------

#  ::PrintPSonUnix:: ---
#  
#      It implements the dialog for printing the canvas on Unix/Linux.

namespace eval ::PrintPSonUnix::  {
    
    variable psCmd
    variable finished
}

proc ::PrintPSonUnix::PrintPSonUnix  { w wCan }  {
    global  sysFont prefs postscriptOpts thisPath debugLevel
    
    variable psCmd
    variable finished
    
    if {$debugLevel >= 2}  {  
	puts "PrintPSonUnix (entry)::"
    }
    set finished -1
    set psCmd $prefs(unixPrintCmd)
    
    catch {toplevel $w}
    wm title $w "Print Canvas"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    set w1 [frame $w.frall.fr1]    
    set wcont1 [LabeledFrame $w1 "Print"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont1.frin]
    pack $frtot -padx 10 -pady 10
    
    message $frtot.msg -borderwidth 0 -font $sysFont(s) -aspect 1000 \
      -text "Shell print command, edit if desired."
    entry $frtot.entcmd -width 20 -font $sysFont(m)   \
      -textvariable [namespace current]::psCmd
    grid $frtot.msg -column 0 -row 0 -padx 4 -pady 2 -sticky news
    grid $frtot.entcmd -column 0 -row 1 -padx 4 -pady 2 -sticky news
    pack $w1 -fill x
    
    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]
    pack [button $frbot.btpr -text "   Print   " -default active  \
      -command "set [namespace current]::finished 1"]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "  \
      -command "set [namespace current]::finished 0"]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    wm resizable $w 0 0
    
    # Grab and focus.
    focus $w
    focus $frtot.entcmd
    bind $w <Return> "$frbot.btpr invoke"
    tkwait variable [namespace current]::finished
    
    # Print...
    if {$finished == 1}  {
	
	# Pipe instead of using a temporary file. Note eval!
	# Note extra braces to protect eval'ing postscript!
	
	if {[catch {eval exec $psCmd << {[eval $wCan postscript $postscriptOpts]}} msg]} {
	    tk_messageBox -message  \
	      "Error when printing: $msg" -icon error -type ok
	}
	set prefs(unixPrintCmd) $psCmd
    }
    catch {grab release $w}
    destroy $w
    return $finished
}

#-- end ::PrintPSonUnix:: ------------------------------------------------------

#  ::PSPageSetup:: ---
#  
#      It makes a dialog for choosing postscript options for the canvas.

namespace eval ::PSPageSetup::  {
    
    variable copyOfPostscriptOpts
    variable txtvarEnt
    variable menuBtVar
    variable allOptNames
    variable unitsFull2Short
    variable unitsShort2Full
    variable rotFull2Short
    variable rotShort2Full
    variable finished
}

proc ::PSPageSetup::PSPageSetup  { w }  {
    global  sysFont prefs postscriptOpts tcl_platform debugLevel
    
    variable copyOfPostscriptOpts
    variable txtvarEnt
    variable menuBtVar
    variable allOptNames
    variable unitsFull2Short
    variable unitsShort2Full
    variable rotFull2Short
    variable rotShort2Full
    variable finished
    
    if {$debugLevel >= 2}  {  
	puts "PSPageSetup (entry)::"
    }
    set finished -1
    set num_ {[0-9\.]+}
    
    # The options.
    set allOptNames {colormode height pageanchor pageheight  \
      pagewidth pagex pagey rotate width x y}
    set unitsShort {p c m i}
    set unitsFull {point cm mm inch}
    array set unitsFull2Short {point p cm c mm m inch i}
    array set unitsShort2Full {p point c cm m mm i inch}
    array set rotFull2Short {portrait 0 landscape 1}
    array set rotShort2Full {0 portrait 1 landscape}
    
    # List available options of special option menus.
    array set theMenuOpts {   \
      colormode {color grey mono}  \
      pageanchor {n ne e se s sw w nw center}  \
      rotate {portrait landscape}}
    
    # Take a copy of the actual options to work on.
    array set copyOfPostscriptOpts $postscriptOpts
    
    # Write this container as a simple proc with automatic sizing.
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Page Setup"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    set w1 [frame $w.frall.fr1]    
    set wcont [LabeledFrame $w1 "Postscript Page Setup"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont.frin]
    pack $frtot -padx 10 -pady 10
    
    message $frtot.msg -width 200 -font $sysFont(s) -text  \
      "Set any of the following options for the postscript\
      generated when printing or saving the canvas as\
      a postscript file."
    grid $frtot.msg -column 0 -columnspan 2 -row 0 -sticky news   \
      -padx 2 -pady 1
    
    # All the options.
    set iLine 0
    foreach optName $allOptNames  {
	incr iLine
	label $frtot.lbl$optName -text "${optName}:" -font $sysFont(sb)
	frame $frtot.fr$optName
	
	if {([string compare $optName "colormode"] == 0) ||  \
	  ([string compare $optName "pageanchor"] == 0) ||  \
	  ([string compare $optName "rotate"] == 0)}  {
	    
	    # Only menubutton.
	    # Get value if exists.
	    if {[info exists copyOfPostscriptOpts(-$optName)]}  {
		if {[string compare $optName "rotate"] == 0}  {
		    
		    # Get full menu name.
		    set menuBtVar($optName)   \
		      $rotShort2Full($copyOfPostscriptOpts(-$optName))
		} else  {
		    set menuBtVar($optName)   \
		      $copyOfPostscriptOpts(-$optName)
		}
	    } else  {
		set menuBtVar($optName)   \
		  [lindex $theMenuOpts($optName) 0]
	    }
	    set wMenu [eval {tk_optionMenu $frtot.menu$optName   \
	      [namespace current]::menuBtVar($optName)}    \
	      $theMenuOpts($optName)]
	    $wMenu configure -font $sysFont(sb)
	    $frtot.menu$optName configure -font $sysFont(sb)   \
	      -highlightthickness 0 -background $prefs(bgColGeneral)   \
	      -foreground black
	    pack $frtot.menu$optName -in $frtot.fr$optName
	    
	} else  {
	    
	    # Length option.
	    # Get value if exists. Need to separate value and unit.
	    if {[info exists copyOfPostscriptOpts(-$optName)]}  {
		set valUnit $copyOfPostscriptOpts(-$optName)
		regexp "(${num_})(p|c|m|i)" $valUnit match val unit
		set txtvarEnt($optName) $val
		set menuBtVar($optName) $unitsShort2Full($unit)
	    } else  {
		set txtvarEnt($optName) {}
		set menuBtVar($optName) [lindex $unitsFull 0]
	    }
	    entry $frtot.ent$optName -width 8 -font $sysFont(s)   \
	      -textvariable [namespace current]::txtvarEnt($optName)
	    set wMenu [eval {tk_optionMenu $frtot.menu$optName   \
	      [namespace current]::menuBtVar($optName)}   \
	      $unitsFull]
	    $wMenu configure -font $sysFont(sb)
	    $frtot.menu$optName configure -font $sysFont(sb)   \
	      -highlightthickness 0 -background $prefs(bgColGeneral)   \
	      -foreground black
	    pack $frtot.ent$optName $frtot.menu$optName   \
	      -in $frtot.fr$optName -side left
	}
	grid $frtot.lbl$optName -column 0 -row $iLine -sticky e -padx 2 -pady 1
	grid $frtot.fr$optName -column 1 -row $iLine -sticky w -padx 2 -pady 1
    }
    pack $w1 -fill x
    
    # Button part.
    set frbot [frame $w.frall.frbot -borderwidth 0]
    pack [button $frbot.btsave -text "   Save   " -default active  \
      -command [list [namespace current]::PushBtSave]]  \
      -side right -padx 5 -pady 5
    pack [button $frbot.btcancel -text " Cancel "  \
      -command "set [namespace current]::finished 0"]  \
      -side right -padx 5 -pady 5
    pack $frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    
    wm resizable $w 0 0
    bind $w <Return> "$frbot.btsave invoke"
    
    # Grab and focus.
    focus $w
    catch {grab $w}
    tkwait variable [namespace current]::finished
    catch {grab release $w}
    destroy $w
    return $finished
}
    
#   PushBtSave ---
#
#   Read out options from the panel and save in 'postscriptOpts'.

proc ::PSPageSetup::PushBtSave  {  }  {
    global  postscriptOpts debugLevel
    
    variable copyOfPostscriptOpts
    variable txtvarEnt
    variable menuBtVar
    variable allOptNames
    variable unitsFull2Short
    variable unitsShort2Full
    variable rotFull2Short
    variable rotShort2Full
    variable finished

    set num_ {([0-9]+|[0-9]+\.[0-9]*|\.[0-9]+)}
    set allNewOpts {}
    foreach optName $allOptNames  {
	
	if {([string compare $optName "colormode"] == 0) ||  \
	  ([string compare $optName "pageanchor"] == 0) ||  \
	  ([string compare $optName "rotate"] == 0)}  {
	    if {[string compare $optName "rotate"] == 0}  {
		
		# Get short name from full name in menu.
		set val $rotFull2Short($menuBtVar($optName))
	    } else  {
		set val $menuBtVar($optName)
	    }
	    lappend allNewOpts "-$optName" $val
	} else  {
	    # If length option in entry.
	    if {[string length $txtvarEnt($optName)] > 0}  {
		set unit $unitsFull2Short($menuBtVar($optName))
		
		# Check consistency of length value.
		if {![regexp "^${num_}$" $txtvarEnt($optName)]}  {
		    
		    # Not a valid number.
		    tk_messageBox -message   \
		      "Error: not a valid number for $optName" \
		      -icon error -type ok
		    return
		}
		set val $txtvarEnt($optName)$unit
		lappend allNewOpts "-$optName" $val
	    }
	    
	}
    }
    set postscriptOpts $allNewOpts
    set finished 1
}

#-- end ::PSPageSetup:: --------------------------------------------------------

#  ShowInfoClients ---
#  
#      It implements a dialog that shows client information.

proc ShowInfoClients  { w allIPnumsFrom }  {
    global  ipNum2ServSocket sysFont ipNum2User ipNum2Name ipNum2ServPort  \
      tcl_platform
    
    if {[llength $allIPnumsFrom] <= 0}  {
	return
    }
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Client Info"
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall
    
    
    # Treat each connected client in order and make a labelled frame for each.
    set n 0
    foreach ip $allIPnumsFrom  {
	set channel $ipNum2ServSocket($ip)
	set peername [fconfigure $channel -peername]
	set sockname [fconfigure $channel -sockname]
	set buff [fconfigure $channel -buffering]
	set block [fconfigure $channel -blocking]
	pack [frame $w.frtop$n -borderwidth 0] -in $w.frtop
	set wcont [LabeledFrame $w.frtop$n [lindex $peername 1]]
	
	# Frame for everything inside the labeled container.
	set fr [frame $wcont.fr]
	label $fr.a1 -text "IP number:" -font $sysFont(sb)
	label $fr.a2 -text "[lindex $peername 0]" -font $sysFont(s)
	label $fr.b1 -text "Host name:" -font $sysFont(sb)
	label $fr.b2 -text "[lindex $peername 1]" -font $sysFont(s)
	label $fr.c1 -text "User name:" -font $sysFont(sb)
	label $fr.c2 -text $ipNum2User($ip) -font $sysFont(s)
	label $fr.d1 -text "Port number:" -font $sysFont(sb)
	label $fr.d2 -text "$ipNum2ServPort($ip)" -font $sysFont(s)
	label $fr.e1 -text "Buffering:" -font $sysFont(sb)
	label $fr.e2 -text "$buff" -font $sysFont(s)
	label $fr.f1 -text "Blocking:" -font $sysFont(sb)
	label $fr.f2 -text "$block" -font $sysFont(s)
	grid $fr.a1 -column 0 -row 0 -sticky e
	grid $fr.a2 -column 1 -row 0 -sticky w
	grid $fr.b1 -column 0 -row 1 -sticky e
	grid $fr.b2 -column 1 -row 1 -sticky w
	grid $fr.c1 -column 0 -row 2 -sticky e
	grid $fr.c2 -column 1 -row 2 -sticky w
	grid $fr.d1 -column 0 -row 3 -sticky e
	grid $fr.d2 -column 1 -row 3 -sticky w
	grid $fr.e1 -column 0 -row 4 -sticky e
	grid $fr.e2 -column 1 -row 4 -sticky w
	grid $fr.f1 -column 0 -row 5 -sticky e
	grid $fr.f2 -column 1 -row 5 -sticky w
	pack $fr -side left -padx 20
	pack $wcont -fill x    
	incr n
    }
    # Button part.
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both  \
      -padx 8 -pady 6
    pack [button $w.ok -text "  OK  " -default active -command "destroy $w"]  \
      -in $w.frbot -side right -padx 5 -pady 5
    wm resizable $w 0 0
    bind $w <Return> "$w.ok invoke"

    tkwait window $w
    grab release $w
}

#-- end ShowInfoClients --------------------------------------------------------

#  ShowInfoServer ---
#  
#      It shows server information. Uses one of the 'allIPnumsTo' to get a 
#      channel which is used to obtain information. If not connected, then 
#      give only the hostname if available.

proc ShowInfoServer  { w thisIPnum }  {
    global  thisServPort sysFont thisUserName ipNum2ServSocket allIPnumsFrom  \
      isServerUp listenServSocket tcl_platform
    
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Server Info"
    pack [frame $w.frall -borderwidth 1 -relief raised]
    pack [frame $w.frtop -borderwidth 0] -in $w.frall    
    set wcont [LabeledFrame $w.frtop "Server Info"]
    
    # Frame for everything inside the labeled container.
    set fr [frame $wcont.fr]
    if {$isServerUp}  {
	set txt "yes"
    } else  {
	set txt "no"
    }
    label $fr.x1 -text "Is server up:" -font $sysFont(sb)
    label $fr.x2 -text $txt -font $sysFont(s)
    label $fr.a1 -text "This IP number:" -font $sysFont(sb)
    label $fr.b1 -text "Host name:" -font $sysFont(sb)
    label $fr.c1 -text "User name:" -font $sysFont(sb)
    label $fr.d1 -text "Port number:" -font $sysFont(sb)
    label $fr.e1 -text "Buffering:" -font $sysFont(sb)
    label $fr.f1 -text "Blocking:" -font $sysFont(sb)

    if {!$isServerUp}  {
	
	# Not yet started server.
	set theHostname [info hostname]
	if {[string length $theHostname] == 0}  {
	    set theHostname "not available"
	}
	
	# Try to get own ip number from a temporary server socket.
	if {[catch {socket -server puts -myaddr [info hostname] 0} s]} {
	    set txtIP $thisIPnum
	} else {
	    set txtIP [lindex [fconfigure $s -sockname] 0]
	    catch {close $s}
	}
	label $fr.a2 -text $txtIP -font $sysFont(s)
	label $fr.b2 -text $theHostname -font $sysFont(s)
	label $fr.c2 -text $thisUserName -font $sysFont(s)
	label $fr.d2 -text "not available" -font $sysFont(s)
	label $fr.e2 -text "not available" -font $sysFont(s)
	label $fr.f2 -text "not available" -font $sysFont(s)
	
    } elseif {$isServerUp && [llength $allIPnumsFrom] == 0}  {

	# Not yet connected but up.
	set theHostname [info hostname]
	if {[string length $theHostname] == 0}  {
	    set theHostname "not available"
	}
	set sockname [fconfigure $listenServSocket -sockname]
	label $fr.a2 -text $thisIPnum -font $sysFont(s)
	label $fr.b2 -text $theHostname -font $sysFont(s)
	label $fr.c2 -text $thisUserName -font $sysFont(s)
	label $fr.d2 -text $thisServPort -font $sysFont(s)
	label $fr.e2 -text "not available" -font $sysFont(s)
	label $fr.f2 -text "not available" -font $sysFont(s)

    } elseif {$isServerUp && [llength $allIPnumsFrom] > 0}  {
	
	# Take any ip and get server side channel.
	set channel $ipNum2ServSocket([lindex $allIPnumsFrom 0])
	set peername [fconfigure $channel -peername]
	set sockname [fconfigure $channel -sockname]
	set buff [fconfigure $channel -buffering]
	set block [fconfigure $channel -blocking]
	label $fr.a2 -text $thisIPnum -font $sysFont(s)
	label $fr.b2 -text "[info hostname]" -font $sysFont(s)
	label $fr.c2 -text $thisUserName -font $sysFont(s)
	label $fr.d2 -text $thisServPort -font $sysFont(s)
	label $fr.e2 -text $buff -font $sysFont(s)
	label $fr.f2 -text "$block" -font $sysFont(s)
    }
    grid $fr.x1 -column 0 -row 0 -sticky e
    grid $fr.x2 -column 1 -row 0 -sticky w
    grid $fr.a1 -column 0 -row 1 -sticky e
    grid $fr.a2 -column 1 -row 1 -sticky w
    grid $fr.b1 -column 0 -row 2 -sticky e
    grid $fr.b2 -column 1 -row 2 -sticky w
    grid $fr.c1 -column 0 -row 3 -sticky e
    grid $fr.c2 -column 1 -row 3 -sticky w
    grid $fr.d1 -column 0 -row 4 -sticky e
    grid $fr.d2 -column 1 -row 4 -sticky w
    grid $fr.e1 -column 0 -row 5 -sticky e
    grid $fr.e2 -column 1 -row 5 -sticky w
    grid $fr.f1 -column 0 -row 6 -sticky e
    grid $fr.f2 -column 1 -row 6 -sticky w
    pack $fr -side left -padx 20    
    pack $wcont -fill x    
        
    # button part
    pack [frame $w.frbot -borderwidth 0] -in $w.frall -fill both  \
      -padx 8 -pady 6
    pack [button $w.ok -text "  OK  " -default active -command "destroy $w"]  \
      -in $w.frbot -side right -padx 5 -pady 5
    wm resizable $w 0 0
    bind $w <Return> "$w.ok invoke"
    
    tkwait window $w
    grab release $w
}

#-- end ShowInfoServer ---------------------------------------------------------
