
#  GetFile.tcl ---
#  
#      This file is part of the whiteboard application. It contains a number
#      of procedures for performing a get operation over the network to
#      a disk file.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

namespace eval ::GetFile:: {
    
    # Exported routines.
    namespace export GetFile GetFileCancelAll GetFileFromClient
    
    # Internal vars only.
    # recordGetFile(sock): {fileId filePath}
    variable recordGetFile
    variable getFileKillerId
    variable timingGetFile
}

# GetFile --
#
#       Initializes a get operation to get a file from a remote server.
#       
# Arguments:
#       ip        the ip number of the remote server.
#       filePathRemote  the file name to get. It must be a pathname relative the
#                 servers base directory including any ../ if up dir. 
#                 Keep it a unix path style.
#       optList   (optional) is a list of key-value pairs that contains 
#                 additional information, typically if called as a response to
#                 a PUT NEW request.
# Results:
#       none.

proc ::GetFile::GetFile { ip filePathRemote {optList {}} } {
    global  statMess ipNum2ServPort debugLevel ipNum2Name
    
    if {$debugLevel >= 2}  {
	puts "+  GetFile:: ip=$ip, filePathRemote=$filePathRemote"
    }
                
    # Open new temporary socket only for this get operation.
    
    set statMess "Contacting client $ipNum2Name($ip). Waiting for response..."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "+  statMess=$statMess"
    }
    
    # The actual socket is opened.
    # In case something went wrong during socket open.
    if {[catch {socket -async $ip $ipNum2ServPort($ip)} s]}  {
	tk_messageBox -message   \
	  "Failed when trying to connect to $ipNum2Name($ip). $s" \
	  -icon error -type ok
	
	# Handle error.
 	GetFileShutDown $s ""
	return
    }    

    # Set in nonblocking mode and register the next event handler.
    fconfigure $s -blocking 0
    fconfigure $s -buffering line
            
    # Schedule timeout event.
    GetFileScheduleKiller $s {} $ip [file tail $filePathRemote]
    
    # If open socket in async mode, need to wait for fileevent.
    fileevent $s writable   \
      [list [namespace current]::GetFileSocketOpen $ip $s $filePathRemote  \
      $optList]
}

# GetFileSocketOpen --
#
#       Is called from previous procedure when the socket opens. Respond to
#       server and schedule reading server's response.
#       
# Arguments:
#       ip        the ip number of the remote server.
#       s         the channel socket.
#       filePathRemote  the file name to get. It must be a pathname relative the
#                 servers base directory including any ../ if up dir. 
#                 Keep it a unix path style.
#       optList   (optional) is a list of key-value pairs that contains 
#                 additional information, typically if called as a response to
#                 a PUT NEW request.
# Results:
#       none.

proc ::GetFile::GetFileSocketOpen { ip s filePathRemote {optList {}} } {
    global  ipNum2Name statMess debugLevel
    
    if {$debugLevel >= 2}  {
	puts "+    GetFileSocketOpen:: ip=$ip, filePathRemote=$filePathRemote, s=$s"
    }

    # Be sure to switch off any fileevents from previous procedures async open.
    fileevent $s writable {}
    
    # Schedule timeout event.
    GetFileScheduleKiller $s {} $ip [file tail $filePathRemote]
    
    # Set up event handler to wait for server response.
    
    fileevent $s readable [list [namespace current]::GetFileServerResponse  \
      $ip $s $filePathRemote $optList]
    set statMess "Client contacted: $ipNum2Name($ip); negotiating..."

    # This is the actual instruction to the remote server what to expect
    
    puts  $s "GET: $filePathRemote $optList"
    flush $s    
    if {$debugLevel >= 2}  {
	puts  "+    GET: $filePathRemote"
    }
}

# GetFileServerResponse --
#
#    Read the first line of the servers response and prepare to get the next
#    one if this was OK.
#    The protocol is typically:
#        TCLWB/1.0 200 OK
#        key1: value1 key2: value2 ...
#        and the data comes here...
#
# Arguments:
#       ip        the ip number of the remote server.
#       s         the channel socket.
#       filePathRemote  the file name to get. It must be a pathname relative the
#                 servers base directory including any ../ if up dir. 
#                 Keep it a unix path style.
#       optList   (optional) is a list of key-value pairs that contains 
#                 additional information, typically if called as a response to
#                 a PUT NEW request.
# Results:
#       none.

proc ::GetFile::GetFileServerResponse { ip s filePathRemote {optList {}} } {
    global  statMess tclwbProtMsg ipNum2Name debugLevel
    
    set int_ {[0-9]+}
    set any_ {.+}
    
    # Get server response.
    if {[gets $s line] == -1} {
	puts "Error reading server response. line=$line"
	set statMess "Error reading server response from $ipNum2Name($ip)"
 	GetFileShutDown $s ""
	return
    }
    if {$debugLevel >= 2}  {
	puts "+      GetFileServerResponse:: line=$line"
    }
	
    # Parse the servers repsonse. Catch problems.
    
    if {![regexp "^TCLWB/(${int_}\.${int_}) +($int_) +($any_)"  $line match  \
      version respCode msg]} {
	set statMess "The server at $ipNum2Name($ip) didn't respond with a\
	  well formed protocol"
 	GetFileShutDown $s "" 
	return
    } elseif {![info exists tclwbProtMsg($respCode)]}  {
	set statMess "The server at $ipNum2Name($ip) responded with an unkown code."
 	GetFileShutDown $s "" 
	return
    } elseif {$respCode != 200}  {
	set statMess "$tclwbProtMsg($respCode)"
 	GetFileShutDown $s "" 
	return
    } 
    
    set statMess "Client at $ipNum2Name($ip) responded."
    update idletasks
    if {$debugLevel >= 2}  {
	puts "+      GetFileServerResponse:: statMess=$statMess"
    }
    if {[eof $s]}  {

	# Shutdown!
 	GetFileShutDown $s "" 
	return
    }
    
    # Schedule timeout event.
    GetFileScheduleKiller $s {} $ip [file tail $filePathRemote]
    
    # Set up event handler to wait for servers next line.
    fileevent $s readable [list [namespace current]::GetFileReadOptLine  \
      $ip $s $filePathRemote $optList]
}

# GetFileReadOptLine --
#
#    Read the first line of the servers response and prepare to get the next
#    one if this was OK.
#    The protocol is typically:
#        TCLWB/1.0 200 OK
#        key1: value1 key2: value2 ...
#        and the data comes here...
#
# Arguments:
#       ip        the ip number of the remote server.
#       s         the channel socket.
#       filePathRemote  the file name to get. It must be a pathname relative the
#                 servers base directory including any ../ if up dir. 
#                 Keep it a unix path style.
#       optList   (optional) is a list of key-value pairs that contains 
#                 additional information, typically if called as a response to
#                 a PUT NEW request.
# Results:
#       none.

proc ::GetFile::GetFileReadOptLine { ip s filePathRemote {optList {}} } {
    global  statMess chunkSize tclwbProtMsg ipNum2Name debugLevel

    variable recordGetFile

    # Get next line that contains the 'optList'.
    # If given an 'optList' as an argument...
    
    if {[gets $s getOptList] == -1} {
	puts "Error reading server response. getOptList=$getOptList"
	set statMess "Error reading server response from $ipNum2Name($ip)"
 	GetFileShutDown $s "" 
	return
    }
    if {$debugLevel >= 2}  {
	puts "+      GetFileReadOptLine:: getOptList=$getOptList"
    }
    
    # Parse the 'optList', translate to an array first.
    array set arrayOfOpts $optList
    
    # Overwrite 'optList' with an updated 'getOptList'.
    array set arrayOfOpts $getOptList
    
    # Need better error handling here.....................
    
    if {[info exists arrayOfOpts(Content-Type:)]} {
	set theMime $arrayOfOpts(Content-Type:)
    } else {
	puts "Error:: MIME type not given."
	return
    }
    if {[info exists arrayOfOpts(size:)]} {
	set totBytes $arrayOfOpts(size:)
    } else {
	puts "Error:: size not given."
	return
    }

    # We need to get the local file path for the file from 'filePathRemote'.
    set fileTail [file tail $filePathRemote]
    set fileName [file join incoming $fileTail]
    
    # Open the file.
    if {[catch {open $fileName w} dest]} {
	tk_messageBox -message "Failed when trying to open $fileName." \
	  -icon error -type ok
	
	# Handle error.
 	GetFileShutDown $s $dest
	return 
    }
    
    # Store a record for this pair of socket fileId.
    set recordGetFile($s) [list $dest $fileName]
        
    # In order for the server to read a complete line, binary mode
    # must wait until the line oriented part is completed.
    # Use the MIME type to hint transfer mode for *socket write* and
    # for *file read*.

    if {[string match "text/*" $theMime]}  {
        fconfigure $s -translation auto
	fconfigure $dest -translation auto

	# For nonlatin characters to work be sure to use Unicode/UTF-8.
	if {[info tclversion] >= 8.1}  {
	    catch {fconfigure $s -encoding utf-8}
	}
    } else {
        fconfigure $s -translation {binary binary}
	fconfigure $dest -translation {binary binary}
    }
    
    # Be sure to switch off any fileevent before fcopy.
    fileevent $s readable {}
    fileevent $s writable {}
    
    # Schedule timeout event.
    GetFileScheduleKiller $s $dest $ip $fileTail
    if {$debugLevel >= 2}  {
	puts "+      GetFileReadOptLine:: start transfer"
    }

    # ...and finally, start transfer.
    fcopy $s $dest -size $chunkSize -command  \
      [list [namespace current]::GetFileCallback $s $dest $ip $fileTail  \
      $totBytes 0 $optList]
}

# GetFileFromClient --
#
#       When a client makes a put request to a server.
# Arguments:
#       ip        the ip number of the remote client.
#       s         the channel socket.
#       fileName  the file name to get. 
#       optList
# Results:
#       none.

proc ::GetFile::GetFileFromClient { ip s fileName optList }  {
    global  chunkSize debugServerLevel
    
    if {$debugServerLevel >= 2 }  {
	puts "GetFileFromClient:: "
    }
        
    # Parse the string 'optList' as key value pairs.
    # Here we could use, for instance, the MIME type, but we
    # only extract the file size for convenience in fcopy callbacks.
    
    if {[llength $optList] > 0}  {
	array set putArrOpts $optList
	if {[info exists putArrOpts(size:)]}  {
	    set totBytes $putArrOpts(size:)
	}
    }
    
    # Check receiving status, if OK returns file descriptor.
    set stat [GetFilePrepare $fileName $s $optList]
    if {([string compare $stat "noext"] == 0) ||   \
      ([string compare $stat "cached"] == 0) ||    \
      ([string compare $stat "nopen"] == 0)}  {
	return
    } 
    
    # We are sure it is a valid (open) file.
    set dest $stat
    
    # Schedule timeout killer event.
    GetFileScheduleKiller $s $dest $ip $fileName
    
    if {$debugServerLevel >= 2}  {
	puts "=HandleClientRequest::  \
	  start transfer [fconfigure $s]"
    }
    
    # Do the actual transfer through fcopy. 'optList's list
    # structure must be respected.
    
    if {$debugServerLevel >= 2}  {
	puts "=HandleClientRequest:: fcopy, registered handler"
    }
    fcopy $s $dest -size $chunkSize -command  \
      [list [namespace current]::GetFileCallback $s $dest $ip $fileName  \
      $totBytes 0 $optList]
}

# GetFilePrepare --
#
#   Checks if the file 'fileTail' should be received. 
#   Rejects if 1): extension not supported, 2): if cached, 3): otherwise known,
#   4): local disk file cannot be opened.
#   Configures channel according to file MIME type.

proc ::GetFile::GetFilePrepare  { fileTail channel optList }  {
    global  supportedMimeTypes prefs servCan tempChannel  \
      plugin prefMimeType2Package tclwbProtMsg debugServerLevel
    
    variable recordGetFile

    if {$debugServerLevel >= 2 }  {
	puts ">   GetFilePrepare:: fileTail=$fileTail, optList=$optList"
    }
    if {![file isdirectory incoming]}  {
	file mkdir incoming
    }
    set knownFilePath [GetKnownPathFromTail $fileTail]
    set inFile [file join incoming $fileTail]
    array set arrayOpts $optList
    
    # Get the MIME type.
    if {[info exists arrayOpts(Content-Type:)]} {
	set theMime $arrayOpts(Content-Type:)
    } else {
	puts "Error:: MIME type not given."
	puts $channel "TCLWB/1.0 422 $tclwbProtMsg(422)"
	flush $channel
	catch {close $channel}
	return "noext"
    }
	
    # Before doing anything more check if the MIME type is supported.
    if {[lsearch $supportedMimeTypes(all) $theMime] < 0}  {
	puts "Error:: MIME type is not supported."
	puts $channel "TCLWB/1.0 421 $tclwbProtMsg(421)"
	flush $channel
	catch {close $channel}
	return "noext"
    }
    
    if {[IsFileInCacheQ $fileTail]}  {

	# Before doing anything more check if the file is cached.
	puts $channel "TCLWB/1.0 420 $tclwbProtMsg(420)"
	flush $channel
	catch {close $channel}
	
	# Get the correct import procedure for this MIME type.
	set package $prefMimeType2Package($theMime)
	if {[string compare $package "0"] != 0} {
	    eval [list $plugin($package,importProc) $servCan $inFile  \
	      $optList "own"]
	}
	return "cached"
	
    } elseif {[llength $knownFilePath] > 0}  {

	# Before doing anything more check if the file is "known".
	# This part executes if file not cached but known.
	puts $channel "TCLWB/1.0 420 $tclwbProtMsg(420)"
	flush $channel
	catch {close $channel}
	
	# Get the correct import procedure for this MIME type.
	set package $prefMimeType2Package($theMime)
	if {[string compare $package "0"] != 0} {
	    eval [list $plugin($package,importProc) $servCan $knownFilePath  \
	      $optList "own"]
	}
	return "cached"
    }		
    
    # Check that the destination file opens correctly.
    if {[catch {open $inFile w} dest]}  {
	puts $channel "TCLWB/1.0 500 $tclwbProtMsg(500)"
	flush $channel
	catch {close $channel}
	tk_messageBox -message   \
	  "Server failed when trying to open $inFile: $dest" \
	  -icon error -type ok
	return "nopen"
    }    
    
    # Here we answer that it's ok to get on with the file transfer.
    puts $channel "TCLWB/1.0 200 $tclwbProtMsg(200)"
    flush $channel
    fconfigure $channel -blocking 0

    # Mark channel as temporary.
    set tempChannel($channel) 1
    
    # Disable callback for this channel. Important for fcopy!
    fileevent $channel readable {}
    
    # Use MIME type to hint transfer mode.
    if {[lsearch $supportedMimeTypes(bin) $theMime] >= 0}  {
	fconfigure $channel -translation {binary binary}
	fconfigure $channel -buffering full
	fconfigure $dest -translation {binary binary}
	
	# Be sure to switch off any encoding for binary.
	if {[info tclversion] >= 8.1}  {
	    catch {fconfigure $channel -encoding {}}
	} 
    } elseif {[lsearch $supportedMimeTypes(text) $theMime] >= 0}  {
	fconfigure $channel -translation auto
	fconfigure $dest -translation auto
	set optList junk
    }
    
    # Just to keep track of things.
    set recordGetFile($channel) [list $dest $inFile]
    
    # Everything is prepared to recieve the file; return the file descriptor.
    return $dest
}

# GetFileCallback --
#
#   Callback function to handel the server part of putting a file.
#   'in': socket, 'out': fileId, 'fileTail': tail of file name,
#   'totBytes': total size of file,...
#   'bytes' is the number of bytes copied and is appended by fcopy,
#   'error' is also appended by fcopy if there is one.

proc ::GetFile::GetFileCallback { in out ip fileTail totBytes sumBytes optList bytes {error {}} }  {
    global  chunkSize debugServerLevel wProgWin statMess ipNum2Name   \
      sysFont
    
    variable timingGetFile

    if {$debugServerLevel >= 2}  {
	puts ">     GetFileCallback:: (entry) error=$error, bytes=$bytes, \
	  optList=$optList"
    }
    if {$error != {}}  {
	tk_messageBox -message   \
	  "File transfer of file $fileTail failed with error: $error" \
	  -icon error -type ok
	catch {destroy $wProgWin$in}
	
	# Close socket and file.
	GetFileShutDown $in $out
    }
    
    # Check if socket already closed, perhaps the user pushed the cancel bt.
    if {[catch {eof $in}]}  {
	#puts "GetFileCallback:: catch eof in"
	return
    }
    
    # Handle the progress window; small files don't need one.
    set progWin 0
    if {$totBytes > [expr 4*$chunkSize]}  {
	set progWin 1
    }
    if {$progWin && $sumBytes == 0} {
	::ProgressWindow::ProgressWindow $wProgWin$in -name Transfer \
	  -filename $fileTail -font1 $sysFont(sb)  \
	  -cancelcmd [list [namespace current]::GetFileShutDown $in $out]
    }
    
    # Reset timing list. (microseconds on Mac)
    if {$sumBytes == 0}  {
	catch {unset timingGetFile($in)}
    }
    incr sumBytes $bytes
    
    # Store timing data in a list with each element a list {clicks sumBytes}.
    lappend timingGetFile($in) [list [clock clicks] $sumBytes]
    
    # Get transfer statistics.
    set bytesPerSec [GetTransferRateFromTiming  $timingGetFile($in)]
    set txtBytesPerSec [BytesPerSecFormatted $bytesPerSec]
    set percent [format "%3.0f" [expr 100*$sumBytes/($totBytes + 1.0)]]
    if {$sumBytes > [expr 3*$chunkSize]}  {
 	set secsLeft  \
	  [expr int(ceil(($totBytes - $sumBytes)/($bytesPerSec + 1.0)))]
	set txtTimeLeft ", $secsLeft secs remaining"
    } else  {
	set txtTimeLeft ""
    }
    if {$debugServerLevel >= 2}  {
	puts ">     GetFileCallback:: sumBytes=$sumBytes,  percent=$percent"
    }
    if {$progWin}  {

	# Update the progress bar in the progress window.
	::ProgressWindow::ProgressWindow $wProgWin$in configure -percent $percent
    }    
    
    # Status message.
    set statMess "Getting file: $fileTail from $ipNum2Name($ip) \
      (at ${txtBytesPerSec}${txtTimeLeft})"
    if {[eof $in] }  {

	# Consistency checking: totBytes must be equal to the actual bytes
	# received.
	if {$totBytes != $sumBytes}  {
	    puts ">     GetFileCallback:: eof in: totBytes=$totBytes, sumBytes=$sumBytes"
	}
	GetFileFinalize $in $out $fileTail $optList $bytes
	
	# Rebind this procedure.
    } else  {
	
	# Schedule timeout killer event.
	GetFileScheduleKiller $in $out $ip $fileTail
	
	# Not finished; rebind this callback.
	fcopy $in $out -size $chunkSize -command  \
	  [list [namespace current]::GetFileCallback $in $out $ip $fileTail  \
	  $totBytes $sumBytes $optList] \
    }
}

# GetFileFinalize --
#
#   Closes sockets and files. Calls to 'DoImportImageOrMovie' to actually show
#   image or movie in canvas.
#   Put file to receive file; handles via temporary socket.
#   The 'optList' is a list of 'key: value' pairs, resembling
#   the html protocol for getting files, but where most keys 
#   correspond to a valid "canvas create" option.

proc ::GetFile::GetFileFinalize { in out fileTail optList bytes {error {}} }  {
    global  servCan debugServerLevel wProgWin statMess plugin  \
      prefMimeType2Package
    
    if {$debugServerLevel >= 2}  {
	puts ">       GetFileFinalize:: (entry) bytes=$bytes, optList=$optList"
    }
    
    # Close socket and file.
    GetFileShutDown $in $out

    if {[string length $error] != 0}  {
	puts "error during file copy: $error"
	return
    }
    set inFile [file join [info script] incoming $fileTail]

    if {$debugServerLevel >= 2}  {
	puts ">       GetFileFinalize:: inFile=$inFile"
    }
    
    # Cleanup window and reset byte counter.
    catch {destroy $wProgWin$in}
    set statMess {}
    array set arrayOpts $optList
    
    # Do the actual work of showing the image/movie.
    # The 'optList' is just passed on.
    # Get the correct import procedure for this MIME type.    
    # Get the MIME type.
    
    if {[info exists arrayOpts(Content-Type:)]} {
	set theMime $arrayOpts(Content-Type:)
    } else {
	puts ">       Error:: MIME type not given."
	return {}
    }
    set impPackage $prefMimeType2Package($theMime)
    if {[string compare $impPackage "0"] != 0} {
	eval [list $plugin($impPackage,importProc) $servCan $inFile  \
	  $optList "own"]
    }
}

# GetFileScheduleKiller, GetFileKill, GetFileShutDown --
#
#   Utility routines to handle timeout events on get file operations.

proc ::GetFile::GetFileScheduleKiller { sock fid ip fileName }  {
    global  prefs
    
    variable getFileKillerId

    if {[info exists getFileKillerId($sock)]}  {
	after cancel $getFileKillerId($sock)
    }
    set getFileKillerId($sock) [after [expr 1000*$prefs(timeout)]   \
      [list [namespace current]::GetFileKill $sock $fid $ip $fileName]]
}

proc ::GetFile::GetFileKill { sock fid ip fileName }  {
    global  prefs ipNum2Name
    
    variable getFileKillerId

    if {![info exists getFileKillerId($sock)]}  {
	return
    }
    tk_messageBox -message   \
      "Timout when waiting for data for file $fileName from $ipNum2Name($ip)" \
      -icon error -type ok
    GetFileShutDown $sock $fid
}

proc ::GetFile::GetFileShutDown { sock fid }  {
    
    variable recordGetFile
    variable getFileKillerId
    variable timingGetFile

    #  Cleanup.
    catch {unset recordGetFile($sock)}
    catch {unset timingGetFile($sock)}
    if {[info exists getFileKillerId($sock)]}  {
	after cancel $getFileKillerId($sock)
    }
    catch {unset getFileKillerId($sock)}
    
    # Close.
    catch {close $sock}
    catch {close $fid}
}

#   GetFileCancelAll ---
#
#   It is supposed to stop ever get operation taking place.
#   This may happen when the user presses a stop button or something.
#   
#   recordGetFile(sock): {fileId filePath}; just to keep track of things.

proc ::GetFile::GetFileCancelAll  {  }  {
    global  debugLevel
    
    variable recordGetFile

    if {$debugLevel >= 2}  {
	puts "+GetFileCancelAll::"
	catch {parray recordGetFile}
    }
    
    # The Get part (reversed order!):
    if {[info exists recordGetFile]}  {
	array set tmpRecordGetFile [array get recordGetFile]
	foreach s [array names tmpRecordGetFile]  {
	    GetFileShutDown $s [lindex $tmpRecordGetFile($s) 0]
	}
	
	# Perhaps the files themselves should also be deleted?
	foreach s [array names tmpRecordGetFile]  {
	    file delete [lindex $tmpRecordGetFile($s) 1]
	}
    }    
}

#-------------------------------------------------------------------------------
