
#  OpenConnection.tcl ---
#  
#      This file is part of the whiteboard application. It creates a dialog 
#      for connecting to the server via TCP/IP, and provide some procedures 
#      to make the connection.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

namespace eval ::OpenConnection::  {
    
    namespace export OpenConnection OpenCancelAllPending DoConnect
    
    # The textvariable for the adress entry widget.
    variable txtvarEntIPnameOrNum
    
    # Variable in the shorts popup menu that is traced.
    variable menuShortVar
    
    # The actual adress connected to.
    variable connIPnum
    
    # The textvariable for the port number entry widget.
    variable compPort
    variable killerId
    
    # Is set when pressing cancel or open button.
    variable finished
}
    
proc ::OpenConnection::OpenConnection { w }  {
    global  internalIPnum remoteServPort sysFont prefs shortcuts tcl_platform
    
    variable txtvarEntIPnameOrNum
    variable menuShortVar
    variable connIPnum
    variable compPort
    variable finished

    set finished -1
    if {[winfo exists $w]} {
	return
    }
    if {[string compare $tcl_platform(platform) "macintosh"] == 0} {
	toplevel $w
	unsupported1 style $w documentProc
    } else {
	toplevel $w
    }
    wm title $w "Open Connection"
    
    # Global frame.
    pack [frame $w.frall -borderwidth 1 -relief raised] -fill both -expand 1
    
    # Ip part.
    frame $w.frip -borderwidth 0
    set wcont1 [LabeledFrame $w.frip "Connect to"]
    
    # Overall frame for whole container.
    set frtot [frame $wcont1.fr]
    pack $frtot
    message $frtot.msg -width 230 -font $sysFont(s) -text  \
      "Connect to a remote computer. Write remote computer name\
      or choose shortcut from the popup menu.\
      If necessary choose new remote port number."
    label $frtot.lblip -text "Shortcut:"
    entry $frtot.entip -width 30   \
      -textvariable [namespace current]::txtvarEntIPnameOrNum
    
    # The option menu. 
    set shorts [lindex $shortcuts 0]
    eval {tk_optionMenu $frtot.optm [namespace current]::menuShortVar} \
      $shorts 
    if {[string length $txtvarEntIPnameOrNum] == 0}  {
	set txtvarEntIPnameOrNum [lindex [lindex $shortcuts 1] 0]
    }
    $frtot.optm configure -highlightthickness 0  \
      -background $prefs(bgColGeneral) -foreground black
    grid $frtot.msg -column 0 -row 0 -columnspan 2 -sticky w -padx 6 -pady 2
    grid $frtot.lblip -column 0 -row 1 -sticky w -padx 6 -pady 2
    grid $frtot.optm -column 1 -row 1 -sticky e -padx 6 -pady 2
    grid $frtot.entip -column 0 -row 2 -sticky ew -padx 10 -columnspan 2
    pack $w.frip -side top -fill both -ipadx 10 -ipady 6 -in $w.frall
    
    # Port part.
    set ofr [frame $w.frport -borderwidth 0]
    set wcont2 [LabeledFrame $ofr "Port number"]
    label $wcont2.lport -text "Remote server port:"
    entry $wcont2.entport -width 6 -textvariable [namespace current]::compPort
    set compPort $remoteServPort
    grid $wcont2.lport -row 0 -column 0 -padx 6
    grid $wcont2.entport -row 0 -column 1 -padx 6
    pack $ofr -side top -fill both -ipadx 10 -ipady 6 -in $w.frall
    
    # Button part.
    frame $w.frbot -borderwidth 0
    pack [button $w.btconn -text "Connect" -default active  \
      -command [namespace current]::PushBtConnect]  \
      -in $w.frbot -side right -padx 5 -pady 5
    pack [button $w.btcancel -text " Cancel "   \
      -command "set [namespace current]::finished 0"]  \
      -in $w.frbot -side right -padx 5 -pady 5
    pack $w.frbot -side top -fill both -expand 1 -in $w.frall  \
      -padx 8 -pady 6
    
    bind $w <Return> [namespace current]::PushBtConnect
    trace variable [namespace current]::menuShortVar w  \
      [namespace current]::TraceOpenConnect
    
    wm resizable $w 0 0
    focus $w
    catch {grab $w}
    tkwait variable [namespace current]::finished
    
    # Clean up.
    trace vdelete [namespace current]::menuShortVar w  \
      [namespace current]::TraceOpenConnect
    catch {grab release $w}
    destroy $w
    
    if {$finished == 1}  {
	return $connIPnum
    } else {
	return ""
    }
}

proc ::OpenConnection::PushBtConnect {  }  {
    global  internalIPnum internalIPname
    
    variable txtvarEntIPnameOrNum
    variable connIPnum
    variable compPort
    variable finished
    
    # Always allow connections to 'internalIPname'.
    # This is because 'IsConnectedToQ' always answers this question with true.
    if {$txtvarEntIPnameOrNum == $internalIPnum ||  \
      $txtvarEntIPnameOrNum == $internalIPname} {
	set connIPnum [DoConnect $txtvarEntIPnameOrNum $compPort 1]
	set finished 1
	return
    }
    # Check if not already connected to the ip in question.
    if {[IsConnectedToQ $txtvarEntIPnameOrNum]}  {
	set finished 0
	return
    }
    set connIPnum [DoConnect $txtvarEntIPnameOrNum $compPort 1]
    set finished 1
}

proc ::OpenConnection::TraceOpenConnect { name junk1 junk2 }  {
    global  shortcuts
    
    # Call by name.
    upvar #0 $name locName

    variable txtvarEntIPnameOrNum

    # 'txtvarEntIPnameOrNum' is textvariable in entry widget
    set ind [lsearch [lindex $shortcuts 0] $locName]
    set txtvarEntIPnameOrNum [lindex [lindex $shortcuts 1] $ind]
}

#   'DoConnect' handles the complete connection process.
#   It makes the actual connection to a given ip address and
#   port number. It sets some arrays to keep track of each connection.
#   If open socket async, then need 'WhenSocketOpensInits' as callback.
#   If 'propagateSizeToClients', then let other clients know this canvas size,
#   which is the case if interactive open, else not (weird things happen).

proc ::OpenConnection::DoConnect { toNameOrNum toPort {propagateSizeToClients 1} }  {
    global  internalIPnum debugLevel prefs internalIPname errorCode statMess \
      wStatMess
    
    set nameOrIP $toNameOrNum
    set remoteServPort $toPort
    
    if {$debugLevel >= 2}  {
	puts "DoConnect:: nameOrIP: $nameOrIP, remoteServPort: $remoteServPort"
    }
    set statMess "Contacted $nameOrIP. Waiting for response..."
    StartStopAnimatedWave $wStatMess 1
    update idletasks
    
    # Handle the TCP/IP channel; if internal pick internalIPnum
    if {[string compare $nameOrIP $internalIPnum] == 0 ||
    [string compare $nameOrIP $internalIPname] == 0}  {
	if {$prefs(asyncOpen)}  {
	    set res [catch {socket -async -myaddr $internalIPnum  \
	      $internalIPnum $remoteServPort} server]
	} else  {
	    set res [catch {socket -myaddr $internalIPnum  \
	      $internalIPnum $remoteServPort} server]
	}
    } else {
	if {$prefs(asyncOpen)}  {
	    set res [catch {socket -async $nameOrIP $remoteServPort} server]
	} else  {
	    set res [catch {socket $nameOrIP $remoteServPort} server]
	}
    }
    if {$debugLevel >= 2}  {
	puts "DoConnect:: res=$res"
    }
    if {$res}  {
	tk_messageBox -message   \
	  "Couldn't create socket to server.\
	  Maybe networking is not functioning or server is down.\
	  The error code is: $errorCode" \
	  -icon error -type ok
	set statMess ""
	StartStopAnimatedWave $wStatMess 0
	update idletasks
	return {}
    }
    # Write line by line; encode newlines in text items as \n.
    fconfigure $server -buffering line
    
    # When socket writable the connection is opened.
    # Needs to be in nonblocking mode.
    fconfigure $server -blocking 0
    
    # For nonlatin characters to work be sure to use Unicode/UTF-8.
    if {[info tclversion] >= 8.1}  {
	catch {fconfigure $server -encoding utf-8}
    }
    # If open socket in async mode, need to wait for fileevent.
    if {$prefs(asyncOpen)}  {
	fileevent $server writable   \
	  [list [namespace current]::WhenSocketOpensInits $server   \
	  $remoteServPort $propagateSizeToClients]
	# Set up timer event for timeouts.
	OpenConnectionScheduleKiller $server
	set ans ""
	
	# Else, it is already open.
    } else  {
	set ans [[namespace current]::WhenSocketOpensInits   \
	  $server $remoteServPort $propagateSizeToClients]
    }
    return $ans
}

#   WhenSocketOpensInits ---
#
#   When socket is writable, it is open. Do all the necessary initializations.
#   If 'propagateSizeToClients', then let other clients know this canvas size.

proc ::OpenConnection::WhenSocketOpensInits { server remoteServPort {propagateSizeToClients 1} }  {
    global  ipNum2Name ipName2Num ipNum2Socket ipNum2ServPort thisServPort  \
      myItpref thisUserName debugLevel wStatMess thisIPnum  \
      internalIPnum wCan statMess allIPnumsTo prefs specServCan
    
    variable killerId    
    
    if {$debugLevel >= 2}  {
	puts "WhenSocketOpensInits:: (entry)"
    }
    StartStopAnimatedWave $wStatMess 0
    
    # No more event handlers here. See also below...
    fileevent $server writable {}
    
    #  Cancel timeout killer.
    if {[info exists killerId($server)]}  {
	after cancel $killerId($server)
    }
    # Check if something went wrong first.
    if {[catch {fconfigure $server -sockname} sockname]}  {
	tk_messageBox -message "Something went wrong. $sockname" \
	  -icon error -type ok
	set statMess ""
	return {}
    }
    if {[catch {fconfigure $server -peername} peername]}  {
	tk_messageBox -message "Something went wrong. $peername" \
	  -icon error -type ok
	set statMess ""
	return {}
    }
    # Save ip nums and names etc in arrays.
    if {$debugLevel >= 2}  {
	puts "WhenSocketOpensInits:: peername=$peername"
    }
    set ipNum [lindex $peername 0]
    set ipName [lindex $peername 1]
    set ipNum2Name($ipNum) $ipName
    set ipName2Num($ipName) $ipNum
    set ipNum2Socket($ipNum) $server
    set ipNum2ServPort($ipNum) $remoteServPort

    # Sometimes the DoStartServer just gives thisIPnum=0.0.0.0 ; fix this here.
    if {[string compare $thisIPnum "0.0.0.0"] == 0} {
	set thisIPnum [lindex $sockname 0]
    }
    set statMess "Client $ipName responded."
    
    # If a central server, then the single socket must be used full duplex.
    # This is only valid only for the clients.
    # It means that the socket we have just opened to the remote server
    # is used to read from as well as writing to.
    # We therefore set up an event handler similar to the server event
    # handler that is used to handle remote commands.
    
    if {!$prefs(symmetricNet) && !$prefs(thisCentralServ)}  {
	fileevent $server readable    \
	  "HandleClientRequest $server $ipNum $remoteServPort"
    }
    # Let the remote computer know port and itpref used by this client.
    puts $server [list "IDENTITY:" $thisServPort $myItpref $thisUserName]
    
    # Let connecting client now this 'allIPnumsTo' (for multi connect).????
    # 'allIPnumsTo' not yet updated with the new ip; doesn't matter!
    
    set listIPandPort {}
    foreach ip $allIPnumsTo {
	lappend listIPandPort $ip $ipNum2ServPort($ipNum)
    }
    puts $server "IPS CONNECTED: $listIPandPort"
    
    # If internal then open server toplevel canvas for writing
    # else write in the original canvas.
    
    if {$prefs(allowLocalServerCanvas) &&    \
      $ipNum == $internalIPnum && $remoteServPort == $thisServPort}  {
	set servCan $specServCan
	if {![winfo exists .serv]}  {
	    MakeSpecialServerCanvas .serv $servCan
	}
    } else {
	set servCan $wCan
    }
    # Add line in the communication entry. Also updates 'allIPnumsTo'.
    SetCommEntry $ipNum 1 -1    
    
    # Let all other now about the size change. propagateToClients and force.
    
    if {$propagateSizeToClients} {
	CanvasSizeChange $allIPnumsTo 1
    } else {
	CanvasSizeChange 0 1
    }
    
    # Update menus. If client only, allow only one connection.
    .menu.file entryconfigure "*Put File*" -state normal
    .menu.file entryconfigure "*Put Canvas*" -state normal
    .menu.file entryconfigure "*Get Canvas*" -state normal
    .menu.prefs entryconfigure "*Network Setup*" -state disabled
    if {!$prefs(symmetricNet) && !$prefs(thisCentralServ)}  { 
	.menu.file entryconfigure "*Open Connection*" -state disabled
    }
    return $ipNum
}

#   OpenConnectionScheduleKiller, OpenConnectionKill ---
#
#   Cancel 'OpenConnection' process if timeout.

proc ::OpenConnection::OpenConnectionScheduleKiller  { sock }  {
    global  prefs
    
    variable killerId    

    if {[info exists killerId($sock)]}  {
	after cancel $killerId($sock)
    }
    set killerId($sock) [after [expr 1000*$prefs(timeout)]   \
      [list [namespace current]::OpenConnectionKill $sock]]
}

proc ::OpenConnection::OpenConnectionKill  { sock }  {
    global  statMess
    
    variable killerId    

    catch {close $sock}
    set statMess "Timout when waiting for connection to open."
    if {[info exists killerId($sock)]}  {
	after cancel $killerId($sock)
    }
    catch {unset killerId($sock)}
    tk_messageBox -message   \
      "Timout when waiting for connection to open." \
      -icon error -type ok
}


#   OpenCancelAllPending ---
#
#   This may happen when the user presses a stop button or something.

proc ::OpenConnection::OpenCancelAllPending  {  }  {
    global  openConnectionKillerId debugLevel
    
    if {$debugLevel >= 2}  {
	puts "+OpenCancelAllPending::"
    }    
    
    # Pending Open connection:
    if {[info exists openConnectionKillerId]}  {
	foreach s [array names openConnectionKillerId]  {
	    
	    # Be sure to cancel any timeout events first.
	    after cancel $openConnectionKillerId($s)
	    
	    # Then close socket.
	    catch {close $s}
	}
    }
}

#---------------------------------------------------------------------