#!/bin/sh
# the next line restarts using tclsh \
	exec tclsh "$0" "$@"
      
#  ReflectorServer.tcl --
#  
#      This file is part of the whiteboard application. It implements a
#      reflector server that accepts connections from clients.
#      In short, incoming stuff from one client is written to all other
#      clients, see note below.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

#  We need to collect information about the various sockets: use the 
#  ip number as the first key into this database, and the unique group
#  identification number for the second key.
#
#  Variables:
#
#       semaphore(ip,group)      0 if no write operation is scheduled,
#                                1 if there is a write operation scheduled.
#       gidRun                   running transfer group identification number.
#       recordGroup(group,fd)    the file descriptor of the file to cache on disk,
#       recordGroup(group,fn)    and its original name (not the cached file name).
      


# Initializations.
# The tcl buffer size; must be the same for all put/get sockets.
set chunkSize 1024
# Should we save transfered files on disk?
set prefs(saveCopyOnDisk) 1
# Reject if no other clients to put to.
set prefs(rejectIfNoClients) 0

# Mapping from error code to error message; 420+ own, rest HTTP codes.

set tclwbProtMsg(200) "OK"
set tclwbProtMsg(201) "Created"
set tclwbProtMsg(202) "Accepted"
set tclwbProtMsg(204) "No Content"
set tclwbProtMsg(301) "Moved Permanently"
set tclwbProtMsg(302) "Moved Temporarily"
set tclwbProtMsg(304) "Not Modified"
set tclwbProtMsg(320) "No other clients connected"
set tclwbProtMsg(400) "Bad Request"
set tclwbProtMsg(401) "Unauthorized"
set tclwbProtMsg(403) "Not Found"
set tclwbProtMsg(404) "Not Found"
set tclwbProtMsg(420) "File already cached"
set tclwbProtMsg(421) "MIME type unsupported"
set tclwbProtMsg(422) "MIME type not given"
set tclwbProtMsg(423) "Group id not given"
set tclwbProtMsg(500) "Internal Server Error"
set tclwbProtMsg(501) "Not Implemented"
set tclwbProtMsg(502) "Bad Gateway"
set tclwbProtMsg(503) "Service Unavailable"

# StartReflectorServer --
#
#       Creates a listening socket that sets up a callback procedure on new
#       incoming connections.
#       
# Arguments:
#       port    the port to listen to.
# Results:
#       none.

proc StartReflectorServer { port } {
    global  gidRun
    
    if {[catch {socket -server SetupChannel $port} sock]}  {
	tk_messageBox -message   \
	  "Couldn't start server socket. Perhaps you are not connected."  \
	  -icon error -type ok
    } else {
	set gidRun 0
	
	# Make sure we have a cache directory in the present directory.
	if {![file exists cache]} {
	    file mkdir cache
	}
    }
}

# SetupChannel --
#
#       Sets up a file event to handle the first line received from a newly
#       opened socket.
#       
# Arguments:
#       s       the newly created socket.
#       ip      the remote ip number.
#       port    the remote port?.
# Results:
#       none.

proc SetupChannel { s ip port } {
    
    puts "SetupChannel: s=$s, ip=$ip, port=$port"
    fileevent $s readable [list HandleClientRequest $s $ip $port]
    fconfigure $s -blocking 0 -buffering line
}

# HandleClientRequest --
#
#       It reads the first line of each newly opened socket to this server,
#       and sets up new event handlers for primary or secondary sockets.
#       
# Arguments:
#       s       the newly created socket.
#       ip      the remote ip number.
#       port    the remote port?.
# Results:
#       none.

proc HandleClientRequest { s ip port } {
    global  gidRun allIps isPrimarySock ip2PrimarySock ip2UserName ipUserId  \
      chunkSize allIpsThisGroup tclwbProtMsg isOpen prefs ip2Sock
    
    puts "HandleClientRequest: s=$s, ip=$ip, port=$port"
    set portwrd_ {[0-9]+}
    set pre_ {[^/ ]+}
    # Matches list with braces.  
    # ($llist_|$wrd_)  :should match single item list or multi item list.
    set llist_ {\{[^\}]+\}}
    set wrd_ {[^ ]+}
    set anyornone_ {.*}
    
    
    
    if {[eof $s]} {
	puts "  HandleClientRequest: eof"

	# Tell all other clients that this ip is disconnected.
	ClientDisconnected $s $ip
	
    } elseif {[gets $s line] != -1} {

	# This is the first line of a newly opened socket.
	# Find out if we have a primary socket or a secondary one.
	
	if {[regexp "^IDENTITY: +($portwrd_) +($pre_) +($llist_|$wrd_)$" \
	  $line junk remPort id user]}  {

	    puts "  HandleClientRequest: IDENTITY id=$id, user=$user"
	    
	    # A client tells which server port number it has (irrelevant here), 
	    # its item prefix and its user name.
	    # This is a primary socket. Collect some useful information.
	    
	    set isPrimarySock($s) 1
	    set ip2PrimarySock($ip) $s
	    set ip2UserName($ip) $user
	    set ip2UserId($ip) $id
	    set allIps [array names ip2PrimarySock]
	    
	    # Set up new event handlers that reflects each incoming line to
	    # all other primary sockets.
	    
	    fileevent $s readable [list ReflectPrimaryCmdLine $s $ip]
	    
	    # Let all other clients know that a new client is connected.
	    
	    foreach ipClient $allIps {
		if {$ip == $ipClient} {
		    continue
		}
		puts $ip2PrimarySock($ipClient) "NEW CLIENT: $ip $user"
	    }	    
	    
	    
	} elseif {[regexp "^PUT: +($llist_|$wrd_) *($anyornone_)$" \
	  $line junk fileName optList]} {
	    
	    puts "  HandleClientRequest: PUT fileName=$fileName, optList=$optList"
	    
	    if {$prefs(rejectIfNoClients) && ([llength $allIps] <= 1)} {
		puts $s "TCLWB/1.0 320 $tclwbProtMsg(320)"
		flush $s
		catch {close $s}
		return
	    }
	    
	    # The clients need the group id so we may identify them when they 
	    # connect with a new socket and make a GET operation.
	    
	    incr gidRun
	    lappend optList {Group-Id:} $gidRun
	    array set putArrOpts $optList
	    if {[info exists putArrOpts(Content-Type:)]} {
		set theMime $putArrOpts(Content-Type:)
	    } else {
		puts "Error:: MIME type not given."
		puts $s "TCLWB/1.0 422 $tclwbProtMsg(422)"
		flush $s
		catch {close $s}
		return
	    }
	    
	    # Here we answer that it's ok to get on with the file transfer.
	    
	    puts $s "TCLWB/1.0 200 $tclwbProtMsg(200)"
	    flush $s
	        
	    # Make a new secondary group, and prepeare for a file transfer.
	    # Tell all other clients to prepare for a GET file operation
	    # on a new socket.
	    # Perhaps we only need the group id and Mime type in 'optList'.
	    
	    foreach ipClient $allIps {
		if {$ip == $ipClient} {
		    continue
		}
		puts $ip2PrimarySock($ipClient) "PUT NEW: $fileName $optList"
	    }
	    set isPrimarySock($s) 0
	    set allIpsThisGroup($gidRun) $allIps
	    
	    # From the Mime type, set translation mode on socket *read*.
	    
	    if {[string match "text/*" $theMime]}  {
		fconfigure $s -translation auto
	    } else {
		fconfigure $s -translation {binary binary}
	    }
	    fconfigure $s -blocking 0 -buffersize $chunkSize -buffering full
	    	    
	    # Prepare for a new group to be created.
	    
	    InitNewSecondaryGroup $s $ip $gidRun $fileName
	    
	} elseif {[regexp "^GET: +($llist_|$wrd_) *($anyornone_)$" \
	  $line junk fileName optList]} {

	    puts "  HandleClientRequest: GET fileName=$fileName, optList=$optList"

	    # Schedule write operations to this client on the secondary
	    # socket if OK.
	    # Find secondary group which is necessary for the identification.
	    
	    array set arrOpts $optList
	    if {[info exists arrOpts(Group-Id:)]} {
		set gid $arrOpts(Group-Id:)
	    } else {
		puts $s "TCLWB/1.0 423 $tclwbProtMsg(423)"
		flush $s
		catch {close $s}
		return
	    }
	    if {[info exists arrOpts(Content-Type:)]} {
		set theMime $arrOpts(Content-Type:)
	    } else {
		puts $s "TCLWB/1.0 320 $tclwbProtMsg(320)"
		flush $s
		catch {close $s}
		return
	    }
	    
	    # Respond to the client according to the GET protocol.
	    
	    puts $s "TCLWB/1.0 200 $tclwbProtMsg(200)"
	    puts $s $optList
	    flush $s

	    fconfigure $s -blocking 0 -buffersize $chunkSize -buffering full
	    
	    # From the Mime type, set translation mode on socket *write*.
	    
	    if {[string match "text/*" $theMime]}  {
		fconfigure $s -translation auto
	    } else {
		fconfigure $s -translation {binary binary}
	    }
	    
	    set isOpen($ip,$gid) 1
	    set isPrimarySock($s) 0
	    set ip2Sock($ip,$gid) $s
    
	    # If we have already got the first chunk of data, then schedule 
	    # write operations to this client on the secondary socket.
	    
	    if {[info exists chunk($gid,0)]} {
		fileevent $s writable [list WriteChunk $ip $s $gid 0]
	    } else {
		
		# When the first chunk is written, the trace will schedule
		# the write operation for this client.
		
		trace variable chunk($gid,0) w  \
		  [list TraceProcFirstChunk $ip $s $gid]
	    }
	    
	} else {
	    puts "Unknown instruction to the reflector server: $line"
	}
    }    
}

# ReflectPrimaryCmdLine --
#
#       It reads the incoming line and writes them to all other clients.
#       Blocking mode write since the lines written are usually much
#       smaller than the OS internal buffers (~64k).
#       
# Arguments:
#       s       the newly created socket.
#       ip      the remote ip number.
# Results:
#       none.

proc ReflectPrimaryCmdLine { s ip } {
    global  allIps isPrimarySock ip2PrimarySock
    
    puts "ReflectPrimaryCmdLine s=$s, ip=$ip"
    if {[eof $s]} {
	
	# Tell all other clients that this ip is disconnected.
	
	ClientDisconnected $s $ip
	
    } elseif {[gets $s line] != -1} {
	
	puts "  ReflectPrimaryCmdLine line=$line"

	# Loop over all clients except the incoming.

	foreach ipClient $allIps {
	    if {$ip == $ipClient} {
		continue
	    }
	    puts $ip2PrimarySock($ipClient) $line
	}
    }
}

# ClientDisconnected --
#
#       This is called when one client disconnects. Let all other connected
#       clients know, and cleanup.
#       
# Arguments:
#       s       socket that did the eof.
#       ip      and its ip number.
# Results:
#       none.

proc ClientDisconnected { s ip } {
    global  allIps isPrimarySock ip2PrimarySock
    
    puts "ClientDisconnected s=$s, ip=$ip"
    
    # Close the server side socket.
    
    catch {close $s}
    
    # Tell all other clients that this ip is disconnected.
    
    foreach ipClient $allIps {
	if {$ip == $ipClient} {
	    continue
	}
	puts $ip2PrimarySock($ipClient) "DISCONNECTED: $ip"
    }
    if {$isPrimarySock($s)} {
	catch {unset ip2PrimarySock($ip)}
	set ind [lsearch $allIps $ip]
	if {$ind >= 0} {
	    set allIps [lreplace $allIps $ind $ind]
	}
    }
}


#--- Stuff for secondary sockets -----------------------------------------------

# InitNewSecondaryGroup --
#
#       Takes care of various initializations for this transfer group,
#       and schedules reding the first data chunk.
#       
# Arguments:
#       s       socket that did the put operation.
#       ip      and its ip number.
#       gid     the group identification number for the group to be initiated.
#       fileName   the original file name (path???).
# Results:
#       none.

proc InitNewSecondaryGroup { s ip gid fileName } {
    global  semaphore allIps lastChunkWritten isOpen recordGroup
    
    puts "InitNewSecondaryGroup s=$s, ip=$ip, gid=$gid, fileName=$fileName"
    
    set recordGroup($gid,fn) $fileName
    foreach ipc $allIps {
	set isOpen($ipc,$gid) 0
	set semaphore($ipc,$gid) 0
	set lastChunkWritten($ipc,$gid) -1
    }
    
    # Schedule reading the first chunk of data.
    
    fileevent $s readable [list ReadChunk $ip $s $gid 0]
}

# TraceProcFirstChunk --
#
#       Gets called when the first chunk gets read in if the write socket
#       was open before chunk(...,0) existed.
#       
# Arguments:
#       ip      the remote ip number that made a GET request.
#       s       the socket.
#       groupId the unique group identifier that is identical for all sockets
#               that are involved on the transfer of one specific file
#       varName is "chunk"
#       ind     is "gid,0"
#       op      operation, r, w, u
# Results:
#       none.

proc TraceProcFirstChunk { ip s groupId varName ind op } {
    
    puts "TraceProcFirstChunk: ip=$ip, s=$s, groupId=$groupId, varName=$varName, ind=$ind, op=$op"

    if {$op == "w"} {
	fileevent $s writable [list WriteChunk $ip $s $groupId 0]
	
	# There may be a remaining trace that needs to be removed. 
	# Note: remove only this specific trace.
	
	catch {trace vdelete chunk($groupId,0) w   \
	  [list TraceProcFirstChunk $ip $s $groupId]}
    }
}

# ReadChunk --
#
#       Read one chunk of data, and reschedule reading the next chunk.
#       Reschedule writing to clients if stopped.
#       
# Arguments:
#       ip      the remote ip number.
#       s       the socket to read from.
#       groupId the unique group identifier that is identical for all sockets
#               that are involved on the transfer of one specific file
#       chunkNo all chunks within the same group have a unique number that
#               starts at 0 for the first chunk.
# Results:
#       none.

proc ReadChunk { ip s groupId chunkNo } {
    global  chunk chunkSize chunkRef allIpsThisGroup lastChunkWritten isOpen \
      semaphore endChunkNo recordGroup prefs ip2Sock

    puts "ReadChunk ip=$ip, s=$s, groupId=$groupId, chunkNo=$chunkNo"
    
    # Check first if input socket still open.
    
    if {[eof $s]} {
	catch {close $s}
	set endChunkNo($groupId) [expr $chunkNo - 1]
	puts "  ReadChunk eof; endChunkNo($groupId)=$endChunkNo($groupId)"
	if {$prefs(saveCopyOnDisk)} {
	    close $recordGroup($groupId,fd)
	}
	return
    }
    
    set chunk($groupId,$chunkNo) [read $s $chunkSize]
    
    if {$prefs(saveCopyOnDisk)} {
	if {$chunkNo == 0} {
	    set fileName cache${groupId}_$recordGroup($groupId,fn)
	    set fd [open [file join cache $fileName] w]
	    set recordGroup($groupId,fd) $fd
	    
	    # Set translation mode for the disk file *write*.
	    fconfigure $fd -translation [fconfigure $s -translation]
	}
	puts -nonewline $recordGroup($groupId,fd) $chunk($groupId,$chunkNo)
    }
    
    # Keep a reference count for each chunk in each group, and decrement it
    # with one each time it is written to a client.
    # Init it to the total number of clients minus the putting one.
    
    set chunkRef($groupId,$chunkNo)  \
      [expr [llength $allIpsThisGroup($groupId)] - 1]
    incr chunkNo
    
    # Schedule the next read operation.
    
    fileevent $s readable [list ReadChunk $ip $s $groupId $chunkNo]

    # Check if any of the write events have been stopped on an already open
    # get socket, and reschedule them at the point where they stopped.
    
    foreach ipClient $allIpsThisGroup($groupId) {
	if {$isOpen($ipClient,$groupId) &&  \
	  ($semaphore($ipClient,$groupId) == 0)} {
	    
	    # Reschedule write operation.
	    
	    set semaphore($ipClient,$groupId) 1
	    
	    fileevent $ip2Sock($ipClient,$groupId) writable  \
	      [list WriteChunk $ipClient $ip2Sock($ipClient,$groupId) $groupId \
	      [expr $lastChunkWritten($ipClient,$groupId) + 1]]
	}
    }
}

# WriteChunk --
#
#       Is called as a consequence to a fileevent writable on this socket.
#       The chunk is written, and a new read operation is rescheduled if
#       next chunk exists, else stop temporarily.
#       
# Arguments:
#       ip      the remote ip number.
#       s       the socket to write to.
#       groupId the unique group identifier that is identical for all sockets
#               that are involved on the transfer of one specific file.
#       chunkNo all chunks within the same group have a unique number that
#               starts at 0 for the first chunk.
# Results:
#       none.

proc WriteChunk { ip s groupId chunkNo } {
    global  chunk chunkSize chunkRef allIpsThisGroup lastChunkWritten  \
      endChunkNo semaphore

    puts "WriteChunk ip=$ip, s=$s, groupId=$groupId, chunkNo=$chunkNo"
    
    puts -nonewline $s $chunk($groupId,$chunkNo)
    incr chunkRef($groupId,$chunkNo) -1
    if {$chunkRef($groupId,$chunkNo) == 0} {
	unset chunk($groupId,$chunkNo)
    }
    set lastChunkWritten($ip,$groupId) $chunkNo
    
    # If this was the end chunk read, close down, and clean up.
    
    if {[info exists endChunkNo($groupId)]} {
	if {$chunkNo == $endChunkNo($groupId)} {
	    close $s
	    puts "  WriteChunk: endChunkNo($groupId)=$endChunkNo($groupId)"
	    
	    return
	}
    }
    
    incr chunkNo
    
    # If the next chunk has been read in, schedule the next write operation.
    
    if {[info exists chunk($groupId,$chunkNo)]} {
	fileevent $s writable [list WriteChunk $ip $s $groupId $chunkNo]
    } else {
	
	# No more chunks have been read in at this moment. We must stop here
	# and let 'ReadChunk' start us from where we are.
	# Save some info about where we have stopped.
	
	fileevent $s writable {}
	set semaphore($ip,$groupId) 0
	set lastChunkWritten($ip,$groupId) $chunkNo
    }
}
    
#-------------------------------------------------------------------------------

#  Make sure that we are in the directory of the application itself.

set thisPath [info script]
if {$thisPath != ""}  {
    cd [file dirname $thisPath]
}
StartReflectorServer 8144
vwait forever

#-------------------------------------------------------------------------------

