
#  TheServer.tcl ---
#  
#      This file is part of the whiteboard application. It implements the
#      server part and contains procedures for creating new server side sockets,
#      handling canvas operations and file transfer.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

# DoStartServer ---
#
#    This belongs to the server part, but is necessary for autoStartServer.
#    Some operations can be critical since the app is not yet completely launched.
#    Therfore 'after idle'.(?)

proc DoStartServer  { thisServPort }  {
    global  prefs isServerUp listenServSocket thisIPnum
    
    set res "ok"
    
    # If it's a centralized net config, ask first.
    if {!$prefs(symmetricNet)}  {
	after 1000 {set res [tk_messageBox -message   \
	  "Are you sure that this is the central server?" \
	  -icon warning -type okcancel -default ok]}
    }
    if {[string compare $res "ok"] == 0}  {
	if {[catch {socket -server SetupChannel $thisServPort} sock]}  {
	    after 500 {tk_messageBox -message   \
	      "Couldn't start server socket. Perhaps you are not connected."  \
	      -icon error -type ok}
	} else  {
	    set listenServSocket $sock
	    set isServerUp 1
	    # Sometimes this gives 0.0.0.0, why I don't know.
	    set thisIPnum [lindex [fconfigure $sock -sockname] 0]
	    .menu.file entryconfigure "*Start Server*" -state disabled
	}
    }
}

proc DoStopServer  {  }  {
    global  listenServSocket isServerUp
    
    catch {close $listenServSocket}
    set isServerUp 0
    .menu.file entryconfigure "*Start Server*" -state normal
}

# HandleClientRequest --
#
#   This is the actual server that reads client requests. 
#   The most important is the CANVAS command which is a complete
#   canvas command that is prefixed only by the widget path.
#

proc HandleClientRequest { channel ip port } {
    global  wCan servCan tempChannel ipNum2Name chunkSize debugServerLevel   \
      myItpref ipNum2User ipNum2ServSocket ipNum2Socket  \
      ipNum2ServPort prefs allIPnumsTo thisServPort thisIPnum thisUserName
    
    # regexp patterns. Defined globally to speedup???
    set wrd_ {[^ ]+}
    set optwrd_ {[^ ]*}
    set optlist_ {.*}
    set any_ {.+}
    set nothing_ {}
    # Matches list with braces.  
    # ($llist_|$wrd_)  :should match single item list or multi item list.
    set llist_ {\{[^\}]+\}}
    set pre_ {[^/ ]+}
    set portwrd_ {[0-9]+}
    set int_ {[0-9]+}
    set signint_ {[-0-9]+}
    # Special chars.
    set nl_ {\\n}
    set bs_ {\\}
    set lb_ {\{}
    set rb_ {\}}
    
    # If client closes socket.
    if {[eof $channel]} {
	if {$debugServerLevel >= 2}  {
	    puts "HandleClientRequest:: eof channel=$channel"
	}
	# Update entry only for nontemporary channels.
	
	if {![info exists tempChannel($channel)]}  {
	    DoCloseServerConnection $ip
	    
	    # If this is the central server there are no clients to disconnect.
	    if {$prefs(autoDisconnect) &&   \
	      !(!$prefs(symmetricNet) && $prefs(thisCentralServ))}  {
		DoCloseClientConnection $ip
	    }
	} else {
	    unset tempChannel($channel)
	}
	
	# Read one line at the time and find out what to do from the
	# leading word.
    } elseif {[gets $channel line] != -1} {
	if {$debugServerLevel >= 2}  {
	    puts "--->$ip:${port}:: $line"
	}
	
	# Check that line does not contain any embedded command.
	if {$prefs(checkSafety)}  {

	    # If any "[" that is not backslashed or embraced, then skip it.
	    
	    set ans [IsServerCommandSafe $line]
	    if {[string compare $ans "0"] == 0}  {
		puts "Warning: the following command to the server was considered\
		  potentially harmful:\n\t$line"
		return
	    } else {
		set line $ans
	    }
	}
	
	# If we are the central server in a centralized network, 
	# each incomming request should be reflected to every other client.
	if {$prefs(thisCentralServ)}  {
	    ReflectRequestToClients $line $ip
	}
	
	if {[regexp "^CANVAS: +(.*)$" $line junk instr]} {
	    
	    # Regular drawing commands in the canvas.
	    
	    # If html sizes in text items, be sure to translate them into
	    # platform specific point sizes.
	    
	    if {$prefs(useHtmlSizes) && ([lsearch -exact $instr "-font"] >= 0)}  {
		set instr [FontHtmlSizeToPointSize $instr]
	    }
	    
	    # Careful, make newline (backslash) substitutions only for the command
	    # being eval'd, else the tcl interpretation may be screwed up.
	    # Fix special chars such as braces since they are parsed 
	    # when doing 'subst' below. Pad extra backslash for each '\{' and
	    # '\}' to protect them.
	    # Seems like an identity operation but is not!
	    
	    regsub -all "$bs_$lb_" $instr "$bs_$lb_" padinstr
	    regsub -all "$bs_$rb_" $padinstr "$bs_$rb_" padinstr
	    set bsinstr [subst -nocommands -novariables $padinstr]
	    if {$debugServerLevel >= 4}  {
		puts "--->bsinstr: $bsinstr"
	    }

	    # Intercept the canvas command if delete to remove any markers
	    # *before* it is deleted! See below for other commands.

	    if {[string compare [lindex $instr 0] "delete"] == 0} {
		set theItno [lindex $instr 1]
		set id [$servCan find withtag $theItno]
		set idsMarker [$servCan find withtag id$id]
		
		# If we have selected the item in question.
		if {[string length $idsMarker] > 0}  {
		    $servCan delete id$id
		}		
	    }
	    
	    # Make the actual canvas command.
	    if {[catch {eval $servCan $bsinstr} id]}  {
		puts stderr "--->error: did not understand: $id"
	    }
	    
	    # Intercept the canvas command in certain cases:
	    # If moving a selected item, be sure to move the markers with it.
	    # The item can either be selected by remote client or here.
	    
	    if {([string compare [lindex $instr 0] "move"] == 0) ||  \
	      ([string compare [lindex $instr 0] "coords"] == 0) ||  \
	      ([string compare [lindex $instr 0] "scale"] == 0) ||  \
	      ([string compare [lindex $instr 0] "itemconfigure"] == 0)}  {
		set theItno [lindex $instr 1]
		set id [$servCan find withtag $theItno]
		set idsMarker [$servCan find withtag id$id]
		    
		# If we have selected the item in question.
		if {[string length $idsMarker] > 0}  {
		    $servCan delete id$id
		    MarkBbox $servCan 1 $id
		}
	    } 
	    
	    # If text then speak up to last punct.
	    
	    set type [$servCan type $id]
	    if {$prefs(TclSpeech) && $prefs(TclSpeechOn) && $type == "text"}  {
		speak [$w itemcget $id -text]
	    }
	    
	} elseif {[regexp "^IDENTITY: +($portwrd_) +($pre_) +($llist_|$wrd_)$" \
	  $line junk remPort id user]}  {

	    # A client tells which server port number it has, its item prefix
	    # and its user name.

	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: IDENTITY: remPort=$remPort, \
		  id=$id, user=$user"
	    }
	    
	    # Save port and socket for the server side in array.
	    # This is done here so we are sure that it is not a temporary socket
	    # for file transfer etc.
	    
	    set ipNum2ServSocket($ip) $channel
	    set ipNum2ServPort($ip) $remPort
	    
	    # If user is a list remove braces.
	    set ipNum2User($ip) [lindex $user 0]
	    
	    # Only of this is the central server in a centralized network 
	    # configuration.
	    
	    if {!$prefs(symmetricNet) && $prefs(thisCentralServ)}  {
		set ipNum2Socket($ip) $channel
		
		# Let the connecting client know our IDENTITY this way since
		# we are not doing any socket/connect from here.
		# Let the remote computer know port and itpref used by this client.
		
		puts $channel [list "IDENTITY:" $thisServPort $myItpref $thisUserName]

		# Update menus if this is the central server.
		.menu.file entryconfigure "*Put File*" -state normal
		.menu.file entryconfigure "*Put Canvas*" -state normal
		.menu.file entryconfigure "*Get Canvas*" -state normal
		.menu.prefs entryconfigure "*Network Setup*" -state disabled
	    }    
	    
	    # Add entry in the communication frame.
	    SetCommEntry $ip -1 1
	    .menu.info entryconfigure "*On Clients*" -state normal
	    
	    # Check that not own ip and user.
	    if {$ip == $thisIPnum &&   \
	      ([string compare [string tolower $user]  \
	      [string tolower $thisUserName]] == 0)}  {
		tk_messageBox -message   \
		  "A connecting client has chosen an ip number  \
		  and user name identical to your own." \
		  -icon warning -type ok
	    }
	    
	    # If auto connect, then make a connection to the client as well.
	    if {$prefs(symmetricNet) && $prefs(autoConnect)  &&   \
	      [lsearch $allIPnumsTo $ip] == -1}  {
		if {$debugServerLevel >= 2}  {
		    puts "HandleClientRequest:: autoConnect:  \
		      ip=$ip, name=$ipNum2Name($ip), remPort=$remPort"
		}
		
		# Handle the complete connection process.
		# Let propagateSizeToClients = false.
		::OpenConnection::DoConnect $ip $ipNum2ServPort($ip) 0
	    }
	    
	} elseif {[regexp "^IPS CONNECTED: +($any_|$nothing_)$" \
	  $line junk remListIPandPort]}  {

	    # A client tells which other ips it is connected to.
	    # 'remListIPandPort' contains: ip1 port1 ip2 port2 ...

	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: IPS CONNECTED:  \
		  remListIPandPort=$remListIPandPort"
	    }
	    
	    # If multi connect then connect to all other 'remAllIPnumsTo'.
	    if {$prefs(symmetricNet) && $prefs(multiConnect)}  {
		
		# Make temporary array that maps ip to port.
		array set arrayIP2Port $remListIPandPort
		foreach ipNum [array names arrayIP2Port] {
		    if {![IsConnectedToQ $ipNum]}  {		
			
			# Handle the complete connection process.
			# Let propagateSizeToClients = false.
			::OpenConnection::DoConnect $ipNum $arrayIP2Port($ipNum) 0
		    }
		}
	    }
	    
	} elseif {[regexp "^RESIZE: +($int_) +($int_)$" $line match w h]} {

	    # Received resize request.
	    
	    if {$debugServerLevel >= 2 }  {
		puts "HandleClientRequest:: RESIZE: w=$w, h=$h"
	    }
	    
	    # Change total size of application so that w and h is the canvas size.
	    # Be sure to not propagate this size change to other clients.
	    
	    bind $wCan <Configure> {CanvasSizeChange 0}
	    SetCanvasSize $w $h
	    update idletasks
	    bind $wCan <Configure> {CanvasSizeChange "all"}
	    
	} elseif {[regexp "^(PUT|GET): +($llist_|$wrd_) *($optlist_)$" \
	  $line what cmd fileName optList]} {
	    
	    # Put file to receive file; handles via temporary socket.
	    # The 'optList' is a list of 'key: value' pairs, resembling
	    # the html protocol for getting files, but where most keys 
	    # correspond to a valid "canvas create" option, and everything 
	    # is on a single line.
	    
	    if {$cmd == "PUT"}  {
	    
		# Some file is put on this server, take action.

		if {$debugServerLevel >= 2 }  {
		    puts "=HandleClientRequest:: PUT: cmd=$cmd, channel=$channel"
		    puts "    fileName=$fileName, optList=$optList"
		}
		set tempChannel($channel) 1
		
		# Be sure to strip off any path. (thisPath)??? Mac bug for /file?
		set fileName [file tail $fileName]
		::GetFile::GetFileFromClient $ip $channel $fileName $optList
		
	    } elseif {$cmd == "GET"}  {
		
		# A file is requested from this server. 'fileName' may be
		# a relative path so beware. This should be taken care for in
		# 'PutFileToClient'.
		
		if {$debugServerLevel >= 2 }  {
		    puts "=HandleClientRequest:: GET: cmd=$cmd, channel=$channel"
		    puts "    fileName=$fileName, optList=$optList"
		}
		set tempChannel($channel) 1
		::PutFile::PutFileToClient $channel $ip $fileName $optList
	    }

	} elseif {[regexp "^PUT NEW: +($llist_|$wrd_) *($optlist_)$" \
	  $line what fileName optList]} {

	    # We should open a new socket and request a GET operation on that
	    # socket with the options given.
	    
	    ::GetFile::GetFile $ip $fileName $optList
	    
	} elseif {[regexp "^GET CANVAS:" $line]} {

	    # The present client requests to put this canvas.

	    if {$debugServerLevel >= 2}  {
		puts "--->GET CANVAS:"
	    }
	    DoPutCanvas $wCan $ip
	    
	} elseif {[regexp "^RESIZE IMAGE: +($wrd_) +($wrd_) +($signint_)$"   \
	  $line match itOrig itNew zoomFactor]}  {
	    
	    # Image (photo) resizing.
	    
	    if {$debugServerLevel >= 2}  {
		puts "--->RESIZE IMAGE: itOrig=$itOrig, itNew=$itNew, \
		  zoomFactor=$zoomFactor"
	    }
	    ResizeImage $servCan $zoomFactor $itOrig $itNew "own"
	    
	} elseif {[regexp "^EOF" $line junk]} {

	    # This is probably unnecessary since caught by eof above.

	    if {$debugServerLevel >= 2}  {
		puts "--->Closed connection"
		puts "HandleClientRequest:: EOF read on channel=$channel"
	    }
	    
	    # Update entry only for nontemporary channels.
	    if {![info exists tempChannel($channel)]}  {
		SetCommEntry $ip  -1 0
	    } else {
		unset tempChannel($channel)
	    }
	    catch {close $channel}
	} else  {
	    if {$debugServerLevel >= 2}  {
		puts "---> unknown instruction"
	    }
	}
    }
}

# SetupChannel --
#   
#   Handles remote connections to the server port. Sets up the callback routine.

proc SetupChannel { channel ip port } {
    global  debugServerLevel ipNum2Name ipName2Num ipNum2Socket   \
      ipNum2ServPort thisIPnum prefs ipNum2ServSocket thisServPort   \
      myItpref thisUserName
    
    # This is the important code that sets up the server event handler.
    fileevent $channel readable [list HandleClientRequest $channel $ip $port]

    # Everything should be done with 'fileevent'.
    fconfigure $channel -blocking 0

    # Everything is lineoriented except binary transfer operations.
    fconfigure $channel -buffering line
    
    # For nonlatin characters to work be sure to use Unicode/UTF-8.
    if {[info tclversion] >= 8.1}  {
        catch {fconfigure $channel -encoding utf-8}
    }
    if {$debugServerLevel >= 2}  {
	puts "SetupChannel:: --->Connection made to $ip:${port} on \
	  channel $channel. Its characteristics are:"
	puts [fconfigure $channel]
    }
    
    # Save ip nums and names etc in arrays.
    set peername [fconfigure $channel -peername]
    set sockname [fconfigure $channel -sockname] 
    set ipNum [lindex $peername 0]
    set ipName [lindex $peername 1]
    set ipNum2Name($ipNum) $ipName
    set ipName2Num($ipName) $ipNum

    # Sometimes the DoStartServer just gives thisIPnum=0.0.0.0 ; fix this here.
    if {[string compare $thisIPnum "0.0.0.0"] == 0} {
	set thisIPnum [lindex $sockname 0]
    }
    if {![info exists ipNum2ServPort($thisIPnum)]}  {
	set ipNum2ServPort($thisIPnum) [lindex $sockname 2]
    }
}

# IsServerCommandSafe --
#
#     Look for any "[" that are not backslashed "\[" and not embraced {...[...}
#     
# Arguments:
#     cmd        Typically a canvas command, but can be any string.
#
# Returns:
#     0          If not safe
#     cmd        If safe; see comments above.

proc IsServerCommandSafe { cmd }  {
    
    # Patterns:
    # Verbatim [ that is not backslashed.
    set lbr_ {[^\\]\[}
    set lbr2_ {\[}
    set nolbr_ {[^\{]*}
    set norbr_ {[^\}]*}
    set noanybr_ {[^\}\{]*}
    set any_ {.*}

    if {[regexp "^(${any_})${lbr_}(${any_})$" $cmd match leftStr rightStr]} {
	
	# We have got one "[" that is not backslashed. Check if it's embraced.
	# Works only for one level of braces.
	
	if {[regexp "\{${noanybr_}${lbr2_}${noanybr_}\}" $cmd]} {
	    return $cmd
	} else {
	    return 0
	}
    } else {
	return $cmd
    }
}

# ReflectRequestToClients --
#
#   If this is the central server, each incoming request should be sent to
#   all other clients, except to the one that sent it originally.

proc ReflectRequestToClients { line ip }  {
    global  ipNum2Name ipName2Num ipNum2Socket allIPnumsToSend debugServerLevel
    
    # If only one client to send to, no one to reflect to.
    if {[llength $allIPnumsToSend] <= 1}  {
	return
    }
    if {$debugServerLevel >= 2}  {
	puts "ReflectRequestToClients:: line=$line, ip=$ip"
    }
    
    # Remove sending client from list to send to. Do it elsewhere? SetCommEntry???
    set ind [lsearch $allIPnumsToSend $ip]
    if {$ind >= 0}  {
	set reflectList [lreplace $allIPnumsToSend $ind $ind]
    } else  {
	set reflectList $allIPnumsToSend
    }
    foreach ipc $reflectList  {
	puts $ipNum2Socket($ipc) $line
    }
}

proc MakeSpecialServerCanvas { wtop wcan } {
    global  dims

    catch {toplevel $wtop}
    canvas $wcan -height $dims(hCanOri) -width $dims(wCanOri)   \
      -relief raised -bd 1 -scrollregion "0 0 $dims(wCanOri) $dims(hCanOri)" \
      -highlightthickness 0
    pack $wcan
    wm title $wtop "Whiteboard: server"
    wm resizable $wtop 0 0
}

#-------------------------------------------------------------------------------
