
#  moviecontroller.tcl ---
#  
#      This file is part of the whiteboard application. It implements a
#      QuickTime look alike movie controller widget.
#      
#  Copyright (c) 1999-2000  Mats Bengtsson
#  
#  See the README file for license, bugs etc.

#  Code idee from Alexander Schoepe's "Progressbar", thank you!
#
# ########################### USAGE #####################################
#
#   NAME
#      moviecontroller - a QuickTime-style controller for movies.
#   SYNOPSIS
#      moviecontroller pathName ?options?
#   OPTIONS
#      -audio, audio, Audio
#      -percent, percent, Percent
#      -command, command, Command
#      -variable, variable, Variable
#      -volume, volume, Volume
#      -width, width, Width
#   WIDGET COMMANDS
#      pathName cget option
#      pathName configure ?option? ?value option value ...?
#

namespace eval ::moviecontroller {

    # The public interface.
    namespace export moviecontroller

    # Globals same for all instances of this widget.
    variable widgetGlobals
    
    set widgetGlobals(debug) 2
}

# ::moviecontroller::Init --
#
#       Contains initializations need for the moviecontroller widget. It is
#       only necessary to invoke it for the first instance of a widget since
#       all stuff defined here are common for all widgets of this type.
#       
# Arguments:
#       none.
# Results:
#       Defines option arrays and icons for movie controllers.

proc ::moviecontroller::Init {  }  {
    
    variable widgetGlobals
    variable widgetOptions
    variable widgetCommands
    
    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::Init"
    }
    
    # List all allowed options with their database names and class names.
    
    array set widgetOptions {
	-audio         {audio         Audio      }      \
	-percent       {percent       Percent    }      \
	-command       {command       Command    }      \
	-variable      {variable      Variable   }      \
	-volume        {volume        Volume     }      \
	-width         {width         Width      }      \
    }
  
    # The legal widget commands.
    set widgetCommands {cget configure}
  
  
    # MIME translated gifs for the actual widget.
    # The volume buttons: off, normal, and full.
    
    set qtmc_vol_off {
R0lGODlhDwAOAPcAAP///+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v
7+/v7+9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAO9KAN7e3t7e3t7e
3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3s7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O
zs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozr29vb29vb29vb29vb29vb29vb29vb29vb29vb29vb29
vb29vb29vb29va2tra2tra2tra2tra2tra2tra2tra2tra2tra2tra2tra2tra2tra2trZyc
nJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnIyMjIyMjIyMjIyMjIyM
jIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjHNzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nz
c3Nzc3Nzc3Nzc3Nzc3Nzc2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2Nj
Y2NjY1JSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUlJSUkJCQkJCQkJC
QkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQghz5whz5whz5whz5whz5whz5whz
5whz5whz5whz5whz5whz5whz5whz5whCrQhCrQhCrQhCrQhCrQhCrQhCrQhCrQhCrQhCrQhC
rQhCrQhCrQhCrQhCrQDOCADOCADOCADOCADOCADOCADOCADOCADOCADOCADOCADOCADOCADO
CAB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQB7rQAAOQAAOQAAOQAA
OQAAOQAAOQAAOQAAOQAAOQAAOQAAOQAAOQAAOQAAOQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACwAAAAADwAOAAAIgwBXCOxAsIPAHDkEHszRIQDB
FQiPVKmCMMeRhgZXjDnCscqYjUdyBHCocZPFiWPiVLnosMOYTZtWSPQYJ87FDjBzykQZ58/N
nDB30mx0cQXQmBI/xiF6sKRJnn98RuQY56nSPzY5zqwSJ8fVRiAnih1D80+jrB1T1uwZFe1Y
tm3Jig0IADs=}

    set qtmc_vol {
R0lGODdhDwAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADwAOAAAEYXBIQYUkRF4iAh1YYRgYUXTWcBSscawFEXjqYo4HYpyecCyL
wQKHQJwEwOSgMMwljkng0pZTnITRaaFauNRswwIi8QyxELad08gSjRCE11gBG9kPrjm7lSuO
yXt3f4B4dhEAOw==}

    set qtmc_vol_full {
R0lGODlhDwAOAPcAAP////d71vd71vd71vd71vd71vd71vd71vd71vd71vd71vd71vd71vdS
vfdSvfdSvfdSvfdSvfdSvfdSvfdSvfdSvfdSvfdSvfdStfdStfdStfdStfdStfdStfdStfdS
tfdStfdStfdStfdStfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcQvfcACPcA
CPcACPcACPcACPcACPcACPcACPcACPcACPcACPcACO/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v
7+/v7+/v7+/v7++UhO+UhO+UhO+UhO+UhO+UhO+UhO+UhO+UhO+UhO+UhO9KAO9KAO9KAO9K
AO9KAO9KAO9KAO9KAO9KAO9KAO9KAN7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e3t7e
3s7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozr29vb29vb29vb29vb29vb29
vb29vb29vb29vb29vb29va291q291q291q291q291q291q291q291q291q291q291q2tra2t
ra2tra2tra2tra2tra2tra2tra2tra2tra2tra2tra2Uva2Uva2Uva2Uva2Uva2Uva2Uva2U
va2Uva2Uva2UvZycnJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnJycnIyMjIyMjIyM
jIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjHNzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nzc3Nz
c3Nzc3Nzc2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY2NjY1JSUlJSUlJSUlJSUlJS
UlJSUlJSUlJSUlJSUlJSUlJSUlJSUkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJCQkJC
QkJCQgC91gC91gC91gC91gC91gC91gC91gC91gC91gC91gC91gC91gAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACwAAAAADwAOAAAIigDRCOxCsItAOnQEHqTThQdB
NAgVhQqFkI6ihgbRtFLEMVSrjYro8HCocZvFbR5xhbrosEurbdvQSNzWCheui11g6kSDcpsi
XMRw6oTJk45PXM8u8hxa9GjSgyVNUvRJLGhEjrik0gxF7CZHiRNx0fmoiNgzkBPTtkpp1mvH
mjaBVnWrVu7ctWkDAgA7}

    # The play button.
    set qtmc_play {
R0lGODdhFgAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQgAAAAAA
AAAAAAAAACwAAAAAFgAOAAAEfXBIQYUkROp9iQjUgBWGgZ0oUXyWWLzGcbx0TQTgcozlgRi1
GkuwKM56CEQQoViEBsXoz5eoIRaKxDPK9SlqCgXiYMFwdS8mOHEwdKI7ggGRqNIObJOqoBvJ
6Eo1MjMwJiR/CQqEQYUlMYhijEExSUl1CYGSJEiXmG2OoI4RADs=}
    
    # The stop button.
    set qtmc_stop {
R0lGODdhDgAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADgAOAAAEWnBIQYUkRA4sAt1EYRhY0VlboRrHURCBtyzEPB6IcQrzMBMs
BOLjWxSBuASx9wsqLBgmrYVQXGrGXzJRKvxmhSRCJSIBVYiEwjVqt5KK8QonTCcScrf9jji0
IwA7}
    
    # The slider button.
    set qtmc_drag {
R0lGODdhDAAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADAAOAAAEV3AMYs5ByBBJiilEVyVgtYUiklSFwImHgg3ugAiekciSdOcI
BqJH+BkMweGkiKskUYXb8ZBgrApYqUWxoB6/lmomkThULomF4qNTJFQKhuKQo8YZi/IxAgA7}

    # The rewind button.
    set qtmc_rew {
R0lGODdhFQAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAAFQAOAAAEc3DISYUVkpCpuxeBNWiFYZBFqhYgNqrGcawrEYTDsRDLEiOG
xGzVEugWgx7hgGgmDETMEalkCmXRKXXHVJiGKcyyty10TbTMeNcrIBJPWgHlZi/cCYR8ZYIu
UwkKYDR9XzJvCnp7KT9Nb3CKhZI/cJAHfREAOw==}
    
    # The fast forward button.
    set qtmc_ff {
R0lGODdhDwAOALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADwAOAAAEYXDIIaqQhMxMRKhDVhiGWHhXWKzGcaxEECzLsCRjixjnRwu0
BOmAQJwuP9pNhwMllQmi4hSi2YIjxBRjDRIKiARORCPcvsSEcTXKlAppxStHarnCijWLWAyL
13VMYmoHdREAOw==}

    # The volume control drag button.
    set qtmcvol_drag {
R0lGODdhDgAMALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADgAMAAAEVnAQc5C16ZhiRjnFIEpGkhjGRAjCSHCmFo7SSykZQbgvVyoH
HeFyoSAYiIKOxRRQEshNYTpFVRQLCmprPUAThARC0/UugJ4E7qJmADsThIJBZywQG0MEADs=}

    # Top and bottom scale ends.
    set qtmcvol_top {
R0lGODdhDgAIALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQgAAAAAA
AAAAAAAAACwAAAAADgAIAAAEI3CQQqstRt6d57bdB2oUoijH6CGLkiQpFRYnctzVvMCGEWMR
ADs=}
    set qtmcvol_bot {
R0lGODdhDgAHALMAAP///+/v797e3s7Ozr29va2trZycnIyMjHNzc2NjY1JSUkJCQjExMQAA
AAAAAAAAACwAAAAADgAHAAAEHnCQUhI6JtNiJD0JZmycRxnHQZbTunatu8EySddnBAA7}

    # Make the actual images in tk.
    set widgetGlobals(qtmc_vol_off) [image create photo -data $qtmc_vol_off]
    set widgetGlobals(qtmc_vol) [image create photo -data $qtmc_vol]
    set widgetGlobals(qtmc_vol_full) [image create photo -data $qtmc_vol_full]
    set widgetGlobals(qtmc_play) [image create photo -data $qtmc_play]
    set widgetGlobals(qtmc_stop) [image create photo -data $qtmc_stop]
    set widgetGlobals(qtmc_drag) [image create photo -data $qtmc_drag]
    set widgetGlobals(qtmc_rew) [image create photo -data $qtmc_rew]
    set widgetGlobals(qtmc_ff) [image create photo -data $qtmc_ff]
    set widgetGlobals(qtmcvol_drag) [image create photo -data $qtmcvol_drag]
    set widgetGlobals(qtmcvol_top) [image create photo -data $qtmcvol_top]
    set widgetGlobals(qtmcvol_bot) [image create photo -data $qtmcvol_bot]
    
    # Unset the base64 coded gif data to save some space.
    unset qtmc_vol_off qtmc_vol qtmc_vol_full qtmc_play qtmc_stop   \
      qtmc_drag qtmc_rew qtmc_ff qtmcvol_drag qtmcvol_top qtmcvol_bot

    # Define coordinates for the images; only x coords needed; anchor nw.
    set widgetGlobals(w_vol) 15
    set widgetGlobals(w_play) 22
    set widgetGlobals(w_stop) 14
    set widgetGlobals(w_rew) 21
    set widgetGlobals(w_ff) 15
    
    # The volume control popup widget. Hardcoded values.
    set widgetGlobals(vol_w) 14
    set widgetGlobals(vol_h) 60
    set widgetGlobals(vol_bg) #adadad
    set widgetGlobals(hvol_top) 8
    set widgetGlobals(hvol_bot) 7
    set widgetGlobals(hvol_scaleh) [expr $widgetGlobals(vol_h) -  \
      $widgetGlobals(hvol_top) - $widgetGlobals(hvol_bot)]
    
    # Depending on the volume percentage, show different loud speakers on the
    # volume button. Here are the boundaries in pixels found from the 
    # percentages. Pixels relative top of volume canvas. Limits at 25% and 75%.
    
    set widgetGlobals(hvol_off) [expr int($widgetGlobals(hvol_top) +  \
      0.75 * $widgetGlobals(hvol_scaleh))]
    set widgetGlobals(hvol_full) [expr int($widgetGlobals(hvol_top) +  \
      0.25 * $widgetGlobals(hvol_scaleh))]
    
    # The slide ruler as canvas drawing commands.
    set widgetGlobals(todraw) {
	line tline1 #cecece {$lmark 1 $rmark 1}  \
	line tline2 #bdbdbd {$lmark 2 $rmark 2}  \
	line tline3 #adadad {$lmark 3 $rmark 3}  \
	line tline4 #adadad {$lmark 4 $rmark 4}  \
	line tline5 #424242 {$lmark 5 $rmark 5}  \
	line tline6 #737373 {$lmark 6 $rmark 6}  \
	line tline7 #8c8c8c {$lmark 7 $rmark 7}  \
	line tline8 #9c9c9c {$lmark 8 $rmark 8}  \
	line tline9 #9c9c9c {$lmark 9 $rmark 9}  \
	line tline10 #8c8c8c {$lmark 10 $rmark 10}  \
	line tline11 #adadad {$lmark 11 $rmark 11}  \
	line tline12 #adadad {$lmark 12 $rmark 12}  \
	line tline13 #adadad {$lmark 13 $rmark 13}  \
	line tline14 #9c9c9c {$lmark 14 $rmark 14}  \
	line tfr0 #000000 {0 0 [expr $width-1] 0 [expr $width-1] 15 0 15 0 0} \
    }
    
    # The icons to draw. Note that 'qtmc_vol' is drawn last on top of the other.
    set widgetGlobals(drawimages) {
	image tvoloff   qtmc_vol_off   {1 1}      nw \
	image tvolfull  qtmc_vol_full  {1 1}      nw \
	image tvol      qtmc_vol       {1 1}      nw \
	image tstop     qtmc_stop      {$xplay 1} nw \
	image tplay     qtmc_play      {$xplay 1} nw \
	image trew      qtmc_rew       {$rmark 1} nw \
	image tff       qtmc_ff        {[expr $rmark+21] 1} nw \
	image tdrag     qtmc_drag      {$xmark 1} n  \
    }
      
    # The black frame around everything together with the ruler.
    set widgetGlobals(drawvolume) {
	line tvol1 #cecece {0 $ytop 0 $ybot}  \
	line tvol2 #bdbdbd {1 $ytop 1 $ybot}  \
	line tvol3 #adadad {2 $ytop 2 $ybot}  \
	line tvol4 #adadad {3 $ytop 3 $ybot}  \
	line tvol5 #424242 {4 $ytop 4 $ybot}  \
	line tvol6 #737373 {5 $ytop 5 $ybot}  \
	line tvol7 #8c8c8c {6 $ytop 6 $ybot}  \
	line tvol8 #9c9c9c {7 $ytop 7 $ybot}  \
	line tvol9 #9c9c9c {8 $ytop 8 $ybot}  \
	line tvol10 #8c8c8c {9 $ytop 9 $ybot}  \
	line tvol11 #adadad {10 $ytop 10 $ybot}  \
	line tvol12 #adadad {11 $ytop 11 $ybot}   \
	line tvol13 #adadad {12 $ytop 12 $ybot}  \
	line tvol14 #9c9c9c {13 $ytop 13 $ybot}  \
    }
    set widgetGlobals(drawvolim) {
	image ttop      qtmcvol_top       {0 0}      nw \
	image tbot      qtmcvol_bot       {0 $ybot}  nw \
    }
  
    # Options for this widget
    option add *MovieController.audio         1
    option add *MovieController.percent       0
    option add *MovieController.command       {}
    option add *MovieController.variable      {}
    option add *MovieController.volume        50
    option add *MovieController.width         160
}

# ::moviecontroller::moviecontroller --
#
#       The constructor of this class; it creates an instance named 'w' of the
#       moviecontroller. 
#       
# Arguments:
#       w       the widget path.
#       args    (optional) list of key value pairs for the widget options.
# Results:
#       The widget path or an error. Calls the necessary procedures to make a 
#       complete movie controller widget.

proc ::moviecontroller::moviecontroller { w {args {}} }  {

    variable widgetGlobals
    variable widgetOptions

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::moviecontroller w=$w, args=$args"
    }
    
    # We need to make Init at least once.
    if {![info exists widgetOptions]}  {
	Init
    }
    
    # Error checking.
    foreach {name value} $args  {
	if {![info exists widgetOptions($name)]}  {
	    error "unknown option for the moviecontroller: $name"
	}
    }
    
    # Continues in the Build procedure.
    set wans [eval Build $w $args]
    return $wans
}

# ::moviecontroller::Build --
#
#       Parses options, creates widget command, and calls the Configure 
#       procedure to do the rest.
#       
# Arguments:
#       w       the widget path.
#       args    list of key value pairs for the widget options.
# Results:
#       The widget path or an error.

proc ::moviecontroller::Build { w args }  {

    variable widgetGlobals
    variable widgetOptions

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::Build w=$w, args=$args"
    }

    # Instance specific namespace
    namespace eval ::moviecontroller::${w} {
	variable options
	variable widgets
	variable wlocals
    }
    
    # Set simpler variable names.
    upvar ::moviecontroller::${w}::options options
    upvar ::moviecontroller::${w}::widgets widgets

    # We use a frame for this specific widget class.
    set widgets(this) [frame $w -class MovieController]
    
    # Set only the name here.
    set widgets(canvas) $w.mc
    set widgets(frame) ::moviecontroller::${w}::${w}
    
    # Need to get a unique toplevel name for our volume control.
    regsub -all {\.} $w {_} unpath
    set widgets(volctrltop) .__mc_vol$unpath
    set widgets(volctrlcan) $widgets(volctrltop).can
    
    # Necessary to remove the original frame procedure from the global
    # namespace into our own.
    rename ::$w $widgets(frame)
    
    # Parse options. First get widget defaults.
    foreach name [array names widgetOptions] {
	set optName [lindex $widgetOptions($name) 0]
	set optClass [lindex $widgetOptions($name) 1]
	set options($name) [option get $w $optName $optClass]
    }
    
    # Apply the options supplied in the widget command.
    # Overwrites defaults when option set in command.
    if {[llength $args] > 0}  {
	array set options $args
    }
    
    # Create the actual widget procedure.
    proc ::${w} { command args }   \
      "eval ::moviecontroller::WidgetProc {$w} \$command \$args"
    
    # The actual drawing takes place from 'Configure'.
    eval Configure $widgets(this) [array get options]

    return $w
}

# ::moviecontroller::WidgetProc --
#
#       This implements the methods; only two: cget and configure.
#       
# Arguments:
#       w       the widget path.
#       command the actual command; cget or configure.
#       args    list of key value pairs for the widget options.
# Results:
#

proc ::moviecontroller::WidgetProc { w command args }  {
    
    variable widgetGlobals
    variable widgetOptions
    variable widgetCommands
    upvar ::moviecontroller::${w}::options options
    
    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::WidgetProc w=$w, command=$command, args=$args"
    }
    
    # Error checking.
    if {[lsearch -exact $widgetCommands $command] == -1}  {
	error "unknown moviecontroller command: $command"
    }
    set result {}
    
    # Which command?
    switch $command {
	cget {
	    if {[llength $args] != 1}  {
		error "wrong # args: should be $w cget option"
	    }
	    set result $options($args)
	}
	configure {
	    set result [eval Configure $w $args]
	}
    }
    return $result
}

# ::moviecontroller::Configure --
#
#       Implements the "configure" widget command (method). 
#       
# Arguments:
#       w       the widget path.
#       args    list of key value pairs for the widget options.
# Results:
#

proc ::moviecontroller::Configure { w args }  {
    
    variable widgetGlobals
    variable widgetOptions
    upvar ${w}::options options
    upvar ${w}::widgets widgets
    upvar ${w}::wlocals wlocals
    
    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::Configure w=$w, args=$args"
    }
    
    # Error checking.
    foreach {name value} $args  {
	if {![info exists widgetOptions($name)]}  {
	    error "unknown option for the moviecontroller: $name"
	}
    }
    if {[llength $args] == 0}  {
	
	# Return all options.
	foreach opt [lsort [array names widgetOptions]] {
	    set optName [lindex $widgetOptions($opt) 0]
	    set optClass [lindex $widgetOptions($opt) 1]
	    set def [option get $w $optName $optClass]
	    lappend results [list $opt $optName $optClass $def $options($opt)]
	}
	return $results
    } elseif {[llength $args] == 1}  {
	
	# Return configuration value for this option.
	set opt $args
	set optName [lindex $widgetOptions($opt) 0]
	set optClass [lindex $widgetOptions($opt) 1]
	set def [option get $w $optName $optClass]
	return [list $opt $optName $optClass $def $options($opt)]
    }
    
    # Error checking.
    if {[expr {[llength $args]%2}] == 1}  {
	error "value for \"[lindex $args end]\" missing"
    }    
    
    # Process all configuration options.
    set needsRedraw 0
    array set opts $args
    foreach opt [array names opts] {
	
	# Constraint: 0 <= percent <= 100.
	if {[string compare $opt "-percent"] == 0} {
	    if {$opts(-percent) < 0} {
		set opts(-percent) 0
	    } elseif {$opts(-percent) > 100} {
		set opts(-percent) 100
	    }
	}
	set newValue $opts($opt)
	if {[info exists options($opt)]}  {
	    set oldValue $options($opt)
	} else  {
	    set oldValue {}
	}
	set options($opt) $newValue
	if {$widgetGlobals(debug) > 1}  {
	    puts "::moviecontroller::Configure opt=$opt, n=$newValue, o=$oldValue"
	}
	
	# Some options need action from the widgets side.
	switch -- $opt {
	    -width     {
		if {[winfo exists $widgets(canvas)]}  {
		    eval $widgets(frame) configure -width $newValue
		    eval $widgets(canvas) configure -width $newValue
		}
		set needsRedraw 1
	    }
	    -percent   {
		if {[winfo exists $widgets(canvas)]}  {
		    set xmark [expr $wlocals(min) +  \
		      ($wlocals(max) - $wlocals(min)) * $newValue/100.0]
		    eval $widgets(canvas) coords tdrag $xmark 1
		}
	    }
	    -variable  {
		
		# Remove any remaining old traces.
		if {[info procs Trace($w)] != ""} {
		    uplevel 3 trace vdelete $oldValue wu   \
		      ::moviecontroller::Trace($w)
		}
		
		# First, need to define a trace procedure,
		# second, set the trace on the traced variable.
		
		if {[string length $newValue] > 0}  {
		    
		    proc ::moviecontroller::Trace($w) {name elem op}  { 
			
			# Tricky part: we need the widget path 'w' here.
			# Either use quotes instead of braces for the 
			# procedure body, or as here, parse [info level 0] 
			# to get 'w' via the procedure name.
			
			set procName [lindex [info level 0] 0]
			regexp {::moviecontroller::Trace\(([^ ]+)\)}  \
			  $procName match wMatch
			switch -- $op {
			    w   {
				if {$elem != ""} {
				    upvar 1 ${name}(${elem}) val
				    catch {$wMatch configure -percent $val}
				} else  {
				    upvar 1 $name val
				    catch {$wMatch configure -percent $val}
				}
			    }
			    u   {
				after idle "catch {rename Trace($wMatch) {}}"
			    }
			}
		    }
		    
		    # Install the actual trace to the procedure above.
		    # Check level by [info level] to get it right.
		    uplevel 3 trace variable $newValue wu   \
		      ::moviecontroller::Trace($w)
		    		    
		    # Need to find out the correct namespace for the variable.
		    # If beginning with :: it is already fully qualified so do 
		    # nothing.
		    
		    set varName $options(-variable)
		    if {![string match "::*" $varName]} {
			
			# Get caller's namespace; make sure we have absolute 
			# paths.
			set ns [uplevel 3 namespace current]
			if {$ns != "::"} {
			    append ns "::"
			}
			set varName "$ns$varName"
			#puts "level=[info level], ns=$ns, new varName=$varName"
			set wlocals(varFull) $varName
			
			# Initialize it or overwrite it.
			set $varName 0
		    }
		}
	    }
	}
    }
    
    # And finally...
    if {$needsRedraw}  {
	Draw $w
    }
}

# ::moviecontroller::Draw --
#
#       This is the actual drawing routine. 'args' contains the complete set
#       of options if called from the widget creation procedure, but only
#       the relevant options if configured.
#       
# Arguments:
#       w       the widget path.
# Results:
#       none.

proc ::moviecontroller::Draw { w }  {

    variable widgetGlobals
    upvar ${w}::options options
    upvar ${w}::widgets widgets
    upvar ${w}::wlocals wlocals

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::Draw w=$w"
    }
    set width $options(-width)
    set percent $options(-percent)
    
    if {$options(-audio)}  {
	set xplay [expr $widgetGlobals(w_vol) + 1]
    } else  {
	set xplay 1
    }
    set lmark [expr $xplay + $widgetGlobals(w_play)]
    set rmark [expr $width - ($widgetGlobals(w_rew) + $widgetGlobals(w_ff) + 1)]
    set xmark [expr $lmark + ($rmark - $lmark)*$percent/100.0]
    set wlocals(min) $lmark
    set wlocals(max) $rmark
    set wlocals(xmark) $xmark
    set wlocals(maxmin) [expr $wlocals(max) - $wlocals(min)]

    # ...and finally, the actual drawing.
    if {![winfo exists $widgets(canvas)]}  {
	canvas $widgets(canvas) -width $width -height 16 -bd 0  \
	  -highlightthickness 0
	pack $widgets(canvas) -side left -fill both -anchor nw

	# Draw horizontal scale.
	foreach {type tag color coords} $widgetGlobals(todraw) {
	    eval $widgets(canvas) create $type $coords -tag $tag -fill $color
	}
	
	# Add all buttons as mime encoded gifs.
	foreach {type tag im coords anch} $widgetGlobals(drawimages) {
	    eval $widgets(canvas) create $type $coords -tag $tag -anchor $anch  \
	      -image $widgetGlobals($im)
	}
	
	# We need bindings to the drag scale, to the volume button, play and stop
	# button, rewind and fast forward button.
	
	$widgets(canvas) bind tdrag <Button-1>  \
	  [list ::moviecontroller::DragInit $w %x]
	$widgets(canvas) bind tdrag <B1-Motion>  \
	  [list ::moviecontroller::Drag $w %x]
	$widgets(canvas) bind tplay <Button-1>  \
	  [list ::moviecontroller::PlayStop $w play]
	$widgets(canvas) bind tstop <Button-1>  \
	  [list ::moviecontroller::PlayStop $w stop]
	$widgets(canvas) bind tvol <Button-1>  \
	  [list ::moviecontroller::Volume $w %y 1]
	$widgets(canvas) bind tvol <B1-Motion>  \
	  [list ::moviecontroller::VolDrag $w %y]
	$widgets(canvas) bind tvol <ButtonRelease>  \
	  [list ::moviecontroller::Volume $w %y 0]
	$widgets(canvas) bind tvoloff <Button-1>  \
	  [list ::moviecontroller::Volume $w %y 1]
	$widgets(canvas) bind tvoloff <B1-Motion>  \
	  [list ::moviecontroller::VolDrag $w %y]
	$widgets(canvas) bind tvoloff <ButtonRelease>  \
	  [list ::moviecontroller::Volume $w %y 0]
	$widgets(canvas) bind tvolfull <Button-1>  \
	  [list ::moviecontroller::Volume $w %y 1]
	$widgets(canvas) bind tvolfull <B1-Motion>  \
	  [list ::moviecontroller::VolDrag $w %y]
	$widgets(canvas) bind tvolfull <ButtonRelease>  \
	  [list ::moviecontroller::Volume $w %y 0]
	$widgets(canvas) bind trew <Button-1>  \
	  [list ::moviecontroller::RewFF $w rew 1]
	$widgets(canvas) bind trew <ButtonRelease>  \
	  [list ::moviecontroller::RewFF $w rew 0]
	$widgets(canvas) bind tff <Button-1>  \
	  [list ::moviecontroller::RewFF $w ff 1]
	$widgets(canvas) bind tff <ButtonRelease>  \
	  [list ::moviecontroller::RewFF $w ff 0]

	# It is also time to create our volume scale for the first time.
	DrawVolumeScale $w
    
    } else {

	# The widget is just configured. Only for the -width option.
	foreach {type tag color coords} $widgetGlobals(todraw) {
	    eval $widgets(canvas) coords $tag $coords
	}
	foreach {type tag im coords anch} $widgetGlobals(drawimages) {
	    eval $widgets(canvas) coords $tag $coords
	}
    }
}

# ::moviecontroller::DrawVolumeScale --
#
#       Draws the vertical volume scale in a separate toplevel window.
#       
# Arguments:
#       w       the widget path.
# Results:
#       none.

proc ::moviecontroller::DrawVolumeScale { w }  {

    variable widgetGlobals
    upvar ${w}::options options
    upvar ${w}::widgets widgets
    upvar ${w}::wlocals wlocals

    set volume $options(-volume)
    set ytop $widgetGlobals(hvol_top)
    set ybot [expr $widgetGlobals(vol_h) - $widgetGlobals(hvol_bot)]
     
    toplevel $widgets(volctrltop) -bg $widgetGlobals(vol_bg)
    wm overrideredirect $widgets(volctrltop) 1
    wm withdraw $widgets(volctrltop)
    pack [canvas $widgets(volctrlcan) -bd 0 -highlightthickness 0  \
      -bg $widgetGlobals(vol_bg) -width $widgetGlobals(vol_w)  \
      -height $widgetGlobals(vol_h)] -fill both
    foreach {type tag color coords} $widgetGlobals(drawvolume) {
	eval $widgets(volctrlcan) create $type $coords -tag $tag -fill $color
    }
    
    # Add all buttons as mime encoded gifs.
    foreach {type tag im coords anch} $widgetGlobals(drawvolim) {
	eval $widgets(volctrlcan) create $type $coords -tag $tag -anchor $anch \
	  -image $widgetGlobals($im)
    }
    
    # The loud speaker icon on the volume button is coded as:
    # 0: vol_off, 1: vol, 2: vol_full, in order to keep track of the present
    # icon to show.
    set wlocals(vol_icon) 1
    
    $widgets(volctrlcan) create image 1 [expr $widgetGlobals(vol_h)/2]  \
      -image $widgetGlobals(qtmcvol_drag) -tags tdrag -anchor w
    $widgets(volctrlcan) bind tdrag <B1-Motion>  \
      [list ::moviecontroller::VolDrag $w %x]
}

# ::moviecontroller::DragInit --
#
#       Invoked when the scale button is clicked. Sets the initial x 
#       coordinate.
# 
# Arguments:
#       w       the widget path.
#       x       the x coordinate of the mouse.
# Results:
#       none.

proc ::moviecontroller::DragInit { w x }  {

    variable widgetGlobals
    upvar ::moviecontroller::${w}::wlocals wlocals

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::DragInit w=$w, x=$x"
    }

    # Keep track of the latest position of the mouse.
    set wlocals(anchor) $x
    
    # Save the x offset between the mouse and the center of the mark 
    # (a few pixels).
    set wlocals(xoff) [expr $x - $wlocals(xmark)]
}

# ::moviecontroller::Drag --
#
#       Dragging the scale button. 
#    
# Arguments:
#       w       the widget path.
#       x       the x coordinate of the mouse.
# Results:
#       none.

proc ::moviecontroller::Drag { w x }  {

    upvar ::moviecontroller::${w}::widgets widgets
    upvar ::moviecontroller::${w}::options options
    upvar ::moviecontroller::${w}::wlocals wlocals

    # Mouse moved by 'dx' without any min/max constraints.
    set xanch $wlocals(anchor)
    set dx [expr $x - $xanch]
    set wlocals(anchor) $x
    
    # The new mark position.
    set wlocals(xmark) [lindex [$widgets(canvas) coords tdrag] 0]
    set newxmark [expr $wlocals(xmark) + $dx]
    
    # Take care of the max/min values.
    
    if {$newxmark > $wlocals(max)}  {
	set dx [expr $wlocals(max) - $wlocals(xmark)]
	set wlocals(anchor) [expr $wlocals(max) + $wlocals(xoff)]
    } elseif {$newxmark < $wlocals(min)}  {
	set dx [expr $wlocals(min) - $wlocals(xmark)]
	set wlocals(anchor) [expr $wlocals(min) + $wlocals(xoff)]
    }
    $widgets(canvas) move tdrag $dx 0
    set options(-percent) [expr   \
      100.0 * ($wlocals(xmark) - $wlocals(min))/$wlocals(maxmin)]
    
    # If we have a variable to set, set it.
    if {[llength $options(-variable)] > 0}  {
	
	# Need to get the correct namespace for the variable. This is done
	# previously in 'Configure' and then kept in 'wlocals(varFull)'.

	set varName $wlocals(varFull)
	
	# It is important to temporarily switch off the trace before setting
	# the variable, and then add it again.
	
	set cmd ::moviecontroller::Trace($w)
	trace vdelete $varName wu $cmd
	set $varName $options(-percent)
	trace variable $varName wu $cmd
    }
}

# ::moviecontroller::PlayStop --
#
#       Toggle play/stop button and evals any command we have registered.
#       
# Arguments:
#       w       the widget path.
#       what    is "play" or "stop".
# Results:
#       none.

proc ::moviecontroller::PlayStop  { w what }  {

    variable widgetGlobals
    upvar ::moviecontroller::${w}::options options
    upvar ::moviecontroller::${w}::widgets widgets

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::PlayStop w=$w, what=$what"
    }
    set cmd $options(-command)
    if {[string compare $what "play"] == 0}  {
	
	# Show stop button
	$widgets(canvas) raise tstop
    } elseif {[string compare $what "stop"] == 0}  {
	
	# Show play button.
	$widgets(canvas) lower tstop
    }
    
    # It should be evaluated in the global namespace.
    if {[llength $cmd] > 0}  {
	uplevel #0 $cmd $what
    }
}

# ::moviecontroller::RewFF --
#
#       Rewind of fast forward; evaluates any command registered.
#       
# Arguments:
#       w       the widget path.
#       what    is "rew" or "ff".
#       updown  1 if button pressed, 0 when released.
# Results:
#       none.

proc ::moviecontroller::RewFF  { w what updown }  {

    variable widgetGlobals
    upvar ::moviecontroller::${w}::options options
    set cmd $options(-command)

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::RewFF w=$w, what=$what, updown=$updown"
    }
    
    # It should be evaluated in the global namespace.
    if {[llength $cmd] > 0}  {
	uplevel #0 $cmd $what $updown
    }
}

# ::moviecontroller::Volume --
#
#       Show or hide the volume control.
#       
# Arguments:
#       w       the widget path.
#       y       is the y coordinate local to the original mc widget.
#       updown  1 if button pressed, 0 when released.
# Results:
#       None.

proc ::moviecontroller::Volume  { w y updown }  {
    
    variable widgetGlobals
    upvar ::moviecontroller::${w}::options options
    upvar ::moviecontroller::${w}::widgets widgets
    upvar ::moviecontroller::${w}::wlocals wlocals

    set volume $options(-volume)

    if {$widgetGlobals(debug) > 1}  {
	puts "::moviecontroller::Volume w=$w, y=$y, updown=$updown"
    }
    
    # Show volume scale.
    if {$updown}  {
		
	# Position the scale so that the volume control is centralized.
	set yposlow [expr int($widgetGlobals(hvol_bot) +  \
	  $volume/100.0 * $widgetGlobals(hvol_scaleh))]
	
	# Keep track of the relative y position of the mc canvas and the
	# pop up volume canvas; necessary when dragging volume control.
	
	set wlocals(yvolrel) [expr $widgetGlobals(vol_h) - $yposlow - 7]
	set xnw [expr [winfo rootx $w] - 16]
	set ynw [expr [winfo rooty $w] - $wlocals(yvolrel)]
	#puts "::moviecontroller::Volume xnw=$xnw, ynw=$ynw"
	wm geometry $widgets(volctrltop) +${xnw}+${ynw}
	wm deiconify $widgets(volctrltop)
	raise $widgets(volctrltop)

	# Set anchor point for the volume drag.
	set wlocals(volanch) $y	
    } else  {
	wm withdraw $widgets(volctrltop)
    }
}

# ::moviecontroller::VolDrag --
#
#       Dragging the volume control button. Raises the loud speaker icon of
#       the volume button depending on present volume.
#    
# Arguments:
#       w       the widget path.
#       y       the y coordinate of the mouse relative the original mc canvas.
# Results:
#       none.

proc ::moviecontroller::VolDrag  { w y }  {

    variable widgetGlobals
    upvar ::moviecontroller::${w}::options options
    upvar ::moviecontroller::${w}::widgets widgets
    upvar ::moviecontroller::${w}::wlocals wlocals

    # Mouse moved by 'dy'.
    set yanch $wlocals(volanch)
    set dy [expr $y - $yanch]

    # Set new anchor point for the volume drag.
    set wlocals(volanch) $y
    
    # Get actual position of the volume drag button.
    set ypos [lindex [$widgets(volctrlcan) coords tdrag] 1]
    set ybot [expr $widgetGlobals(hvol_top) + $widgetGlobals(hvol_scaleh)]
    set newymark [expr $ypos + $dy]

    # Impose min and max constraints. Note: y-axis upside down.
    # It is necessary to translate anchor point to mc canvas coords.
    
    if {$newymark < $widgetGlobals(hvol_top)}  {
	set dy [expr $widgetGlobals(hvol_top) - $ypos]
	set wlocals(volanch) \
	  [expr $widgetGlobals(hvol_top) - $wlocals(yvolrel)]
    } elseif {$newymark > $ybot}  {
	set dy [expr $ybot - $ypos]
	set wlocals(volanch) [expr $ybot - $wlocals(yvolrel)]
    }
    $widgets(volctrlcan) move tdrag 0 $dy    
    
    # Check to see if volume icon needs to be raised to indicate the present
    # volume setting.
    
    set yvol_off [expr $widgetGlobals(hvol_off) - $wlocals(yvolrel)]
    set yvol_full [expr $widgetGlobals(hvol_full) - $wlocals(yvolrel)]
    if {($y > $yvol_off) && ($wlocals(vol_icon) != 0)} {
	$widgets(canvas) raise tvoloff
	set wlocals(vol_icon) 0
    } elseif {($y < $yvol_full) && ($wlocals(vol_icon) != 2)} {
	$widgets(canvas) raise tvolfull
	set wlocals(vol_icon) 2
    } elseif {($y < $yvol_off) && ($y > $yvol_full) &&  \
      ($wlocals(vol_icon) != 1)} {
	$widgets(canvas) raise tvol
	set wlocals(vol_icon) 1
    }
    
    # Set the actual volume option variable.
    set options(-volume)   \
      [expr 100.0 * ($ybot - $ypos)/$widgetGlobals(hvol_scaleh)]
    
    # Any command should be evaluated in the global namespace.
    if {[llength $options(-command)] > 0} {
	uplevel #0 $options(-command) volume $options(-volume)
    }
}

#-------------------------------------------------------------------------------