//  Copyright 1999 McMillan Enterprises, Inc. -- www.mcmillan-inc.com
//  Copyright (C) 1999 Jean-Claude Wippler.  All rights reserved.
//
//  Storage class implementation and main entry point

#include <Python.h>
#include "PyStorage.h"
#include <PWOSequence.h>
#include <PWONumber.h>
#include "PyView.h"
#include "PyProperty.h"
#include "PyRowRef.h"
#include "mk4str.h"
#include "mk4io.h"

#if !defined _WIN32
#define __declspec(x)
#endif

static char mk4py_module_documentation[] =
"This is the Python interface of the embeddable MetaKit database library.\n"
"Example of use:\n"
"\n"
"    import Mk4py\n"
"    mk = Mk4py\n"
"\n"
"    s = mk.Storage('demo.dat', 1)\n"
"    v = s.getas('people[first:S,last:S,shoesize:I]')\n"
"\n"
"    v.append(first='John',last='Lennon',shoesize=44)\n"
"    v.append(first='Flash',last='Gordon',shoesize=42)\n"
"    s.commit()\n"
"\n"
"    def dump(v):\n"
"        print len(v)\n"
"        for r in v: print r.first, r.last, r.shoesize\n"
"\n"
"    v2 = v.sort(v.last)\n"
"    dump(v2)\n"
"    v[0].last = 'Doe'\n"
"    dump(v2)\n"
"    v2 = v.select(last='Doe')\n"
"    dump(v2)\n"
"    del s\n"
"\n"
"See the website area at http://www.equi4.com/metakit/ for full details.\n";

///////////////////////////////////////////////////////////////////////////////
// This was copied nearly verbatim from Mk4tcl 1.1, it implements file locking
// and scripted document access.  It also reflects load and save to Python.
//
// In Tcl, streaming I/O uses the Tcl channel interface for loading/saving.
// Not used for the core datafile I/O, so memory-mapping can still be used.

#if defined (unix) || defined (__unix__) || defined (__GNUC__)

#include <unistd.h>

#if HAVE_FLOCK
#include <sys/file.h>
#else
#include <fcntl.h>
#include <errno.h>

#define LOCK_SH 1    /* Shared lock.  */
#define LOCK_EX 2    /* Exclusive lock.  */
#define LOCK_UN 8    /* Unlock.  */
#define LOCK_NB 4    /* Don't block when locking.  */

    static int my_flock(int fd, int type)
    {
        struct flock fs;
        memset(&fs, 0, sizeof fs);
        fs.l_pid = getpid();

        switch (type & ~LOCK_NB)
        {
            case LOCK_EX:   fs.l_type = F_WRLCK; break;
            case LOCK_SH:   fs.l_type = F_RDLCK; break;
            case LOCK_UN:   fs.l_type = F_UNLCK; break;
            default:        errno = EINVAL; return -1;
        }

        return fcntl (fd, type & LOCK_NB ? F_SETLK : F_SETLKW, &fs);
    }

#define flock   my_flock
#endif

    bool LockingOpen(const char* name_, bool rw_, FILE*& file_)
    {
        file_ = fopen(name_, rw_ ? "r+b" : "rb");
        bool exists = file_ != 0;

        if (!exists && rw_)
            file_ = fopen(name_, "w+b");

        if (file_ != 0 && flock(fileno(file_),
                                (rw_ ? LOCK_EX : LOCK_SH) | LOCK_NB) == -1) {
            fclose(file_);
            file_ = 0;
        }

        return exists;
    }
    
#elif _WIN32
    
#include <fcntl.h>
#include <share.h>
#include <io.h>

#ifndef _SH_DENYRW
#define _SH_DENYRW SH_DENYRW
#define _SH_DENYWR SH_DENYWR
#endif

    bool LockingOpen(const char* name_, bool rw_, FILE*& file_)
    {
        int mode = (rw_ ? _O_RDWR : _O_RDONLY) | _O_BINARY;
        const int shared = rw_ ? _SH_DENYRW : _SH_DENYWR;

        int fd = _sopen(name_, mode, shared);
        bool exists = fd != -1;

        if (!exists && rw_)
            fd = _sopen(name_, mode | _O_CREAT | _O_EXCL, shared, 0666);

        if (fd != -1)
            file_ = _fdopen(fd, !exists ? "w+b" : rw_ ? "r+b" : "rb");

        return exists;
    }

#else
        // this is a version which does NOT lock - because we don't know how!

    bool LockingOpen(const char* name_, bool rw_, FILE*& file_)
    {
        file_ = fopen(name_, rw_ ? "r+b" : "rb");
        bool exists = file_ != 0;

        if (!exists && rw_)
            file_ = fopen(name_, "w+b");

        return exists;
    }

#endif

class PyStrategy : public c4_FileStrategy
{
    int _offset; // starting point of datafile

public:
    PyStrategy (FILE* file_ =0);
    virtual ~PyStrategy ();

        // the other members have overrides to allow datafiles at >0 offsets

    virtual bool DataOpen(const char* fileName_, bool allowWriting_)
    {
            // new in 1.1: logic to enforce proper file locking
            // either a single writer or multiple readers, no mix allowed

        bool exists = LockingOpen(fileName_, allowWriting_, _file);
        //d4_assert(_file != 0);

        if (exists && IsValid()) {
            ResetFileMapping();

                // look for the magic header bytes in the first 4 Kb
                // of the file, but only check on 16-byte boundaries
            
                // seek to start, since ResetFileMapping may have moved it!
            DataSeek(0);
            
            char buffer [4096];
            const int n = DataRead(buffer, sizeof buffer);

            for (const char* p = buffer; p < buffer + n; p += 16)
            {
                if (p[2] == '\x1A' && p[3] == '\x80' &&
                        (p[0] == '\x4A' && p[1] == '\x4C' ||
                         p[0] == '\x4C' && p[1] == '\x4A'))
                {
                    _offset = (int) (p - buffer);

                        // adjust file mapping if it is on
                    if (_mapStart != 0)
                        _mapStart += _offset;

                    break;
                }
            }
        }
        
        _cleanup = _file;

        return exists;
    }

    virtual void DataSeek(t4_i32 position_)
    {
        if (IsValid())
            c4_FileStrategy::DataSeek(position_ + _offset);
    }

    virtual int DataRead(void* buffer_, int length_)
    {
            // accept invalid file handle for in-memory use
        return IsValid() ? c4_FileStrategy::DataRead(buffer_, length_) : 0;
    }

    virtual void DataCommit(t4_i32 newSize_)
    {
        //d4_assert(IsValid());

        c4_FileStrategy::DataCommit(newSize_ > 0 ? newSize_ + _offset : 0);
    }

    virtual void ResetFileMapping()
    {
        if (_mapStart != 0)
            _mapStart -= _offset;

        c4_FileStrategy::ResetFileMapping();

        if (_mapStart != 0)
            _mapStart += _offset;
    }
};

PyStrategy::PyStrategy (FILE* file_)
    : c4_FileStrategy (file_), _offset (0)
{
}

PyStrategy::~PyStrategy ()
{
    if (_mapStart != 0)
        _mapStart -= _offset;
}

///////////////////////////////////////////////////////////////////////////////

class c4_PyStream : public c4_Stream
{
	PyObject* _stream;

public:
	c4_PyStream (PyObject* stream_);

    virtual int Read(void* buffer_, int length_);
    virtual void Write(const void* buffer_, int length_);
};

c4_PyStream::c4_PyStream (PyObject* stream_)
	: _stream (stream_)
{
}

int c4_PyStream::Read(void* buffer_, int length_)
{
	PyObject* o = PyObject_CallMethod(_stream, "read", "i", length_);
    int n = o != 0 ? PyString_Size(o) : 0;
    if (n > 0)
        memcpy(buffer_, PyString_AsString(o), n);
    return n;
}

void c4_PyStream::Write(const void* buffer_, int length_)
{
    PyObject_CallMethod(_stream, "write", "s#", buffer_, length_);
}

///////////////////////////////////////////////////////////////////////////////
// A "storage in a storage" strategy class for MetaKit

class SiasStrategy : public c4_Strategy
{
	c4_Storage& _storage;
	c4_View _view;
	c4_MemoProp _memo;
	int _row;
	t4_i32 _position;

public:
	SiasStrategy (c4_Storage& storage_, 
					const c4_View& view_, const c4_MemoProp& memo_, int row_)
		: _storage (storage_), _view (view_), _memo (memo_), _row (row_), _position (0)
	{
			// set up mapping if the memo itself is mapped in its entirety
		c4_Strategy& strategy = storage_.Strategy();
		if (strategy._mapStart != 0) {
			c4_RowRef r = _view[_row];
			c4_Bytes data = _memo(r).Access(0);
			const t4_byte* ptr = data.Contents();
			if (data.Size() == _memo(r).GetSize() &&
				strategy._mapStart <= ptr && ptr < strategy._mapLimit)
			{
				_mapStart = ptr;
				_mapLimit = ptr + data.Size();
			}
		}
	}

	virtual ~SiasStrategy ()
	{
		_view = c4_View();
		_mapStart = _mapLimit = 0;
	}

    virtual void DataSeek(t4_i32 position_)
    {
		_position = position_;
    }

    virtual int DataRead(void* buffer_, int length_)
    {
		int i = 0;

		while (i < length_)
		{
			c4_Bytes data = _memo(_view[_row]).Access(_position + i, length_ - i);
			int n = data.Size();
			if (n <= 0)
				break;
			memcpy((char*) buffer_ + i, data.Contents(), n);
			i += n;
		}

		_position += i;
		return i;
    }

    virtual bool DataWrite(const void* buffer_, int length_)
    {
		c4_Bytes data (buffer_, length_);
		bool ok = _memo(_view[_row]).Modify(data, _position);
		if (ok)
			_position += length_;
		return ok;
    }

    virtual void DataCommit(t4_i32 newSize_)
    {
		if (newSize_ > 0)
			_memo(_view[_row]).Modify(c4_Bytes (), newSize_);
    }

    virtual void ResetFileMapping()
    {
		_mapStart = _mapLimit = 0; // never called, but just in case
    }
};

///////////////////////////////////////////////////////////////////////////////

static char* autocommit__doc = 
"autocommit() -- turn on autocommit (i.e. commit when storage object is deleted)";

static PyObject* PyStorage_Autocommit(PyStorage *o, PyObject* _args) {
    try {
        o->AutoCommit();
        Py_INCREF(Py_None);
        return Py_None;
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* contents__doc = 
"contents() -- return view with one row, representing entire storage (internal use)";

static PyObject* PyStorage_Contents(PyStorage *o, PyObject* _args) {
    try {
        return new PyView(o->Contents().Container());
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* description__doc = 
"description(name='') -- return a description of named view, or of entire storage";

static PyObject* PyStorage_Description(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        PWOString nm ("");
		if (args.len() > 0)
			nm = args[0];
        PWOString rslt(o->Description(nm));
        return rslt.disOwn();
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* commit__doc = 
"commit() -- permanently commit data and structure changes to disk";

static PyObject* PyStorage_Commit(PyStorage *o, PyObject* _args) {
    try {
        if (!o->Commit())
             throw PWException(PyExc_IOError, "commit failed");
        Py_INCREF(Py_None);
        return Py_None;
   }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* rollback__doc = 
"rollback() -- revert data and structure as was last committed to disk";

static PyObject* PyStorage_Rollback(PyStorage *o, PyObject* _args) {
    try {
        if (!o->Rollback())
             throw PWException(PyExc_IOError, "rollback failed");
        Py_INCREF(Py_None);
        return Py_None;
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* view__doc = 
"view(viewname) -- return top-level view in storage, given its name";

static PyObject* PyStorage_View(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        PWOString nm(args[0]);
        return new PyView(o->View(nm));
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* getas__doc = 
"getas(description) -- return view, create / restructure as needed to match";

static PyObject* PyStorage_GetAs(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        PWOString descr(args[0]);
        return new PyView(o->GetAs(descr));
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* store__doc = 
"store(viewname, view) -- store a view, guessing its structure (deprecated)";

static PyObject* PyStorage_store(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        if (args.len() != 2)
            throw PWException(PyExc_ValueError, "store requires name and view object");
        PWOString nm(args[0]);
        PWOBase vw(args[1]);
        PyView *view = (PyView *)(PyObject* )args[1];
        return new PyView(o->Store(nm, *view));
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* load__doc = 
"load(file) -- replace storage object contents from file (or any obj supporting read)";

static PyObject* PyStorage_load(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        if (args.len() != 1)
            throw PWException(PyExc_ValueError, "load requires a file-like object");
        
		c4_PyStream stream (args[0]);
        o->LoadFrom(stream);

        Py_INCREF(Py_None);
        return Py_None;
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* save__doc = 
"save(file) -- store storage object contents to file (or any obj supporting write)";

static PyObject* PyStorage_save(PyStorage *o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        if (args.len() != 1)
            throw PWException(PyExc_ValueError, "save requires a file-like object");
        
		c4_PyStream stream (args[0]);
        o->SaveTo(stream);

        Py_INCREF(Py_None);
        return Py_None;
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static PyMethodDef StorageMethods[] = {
    {"getas", (PyCFunction)PyStorage_GetAs, METH_VARARGS, getas__doc},
    {"view",  (PyCFunction)PyStorage_View,  METH_VARARGS, view__doc},
    {"rollback", (PyCFunction)PyStorage_Rollback, METH_VARARGS, rollback__doc},
    {"commit", (PyCFunction)PyStorage_Commit, METH_VARARGS, commit__doc},
    {"description", (PyCFunction)PyStorage_Description, METH_VARARGS, description__doc},
    {"contents", (PyCFunction)PyStorage_Contents, METH_VARARGS, contents__doc},
    {"autocommit", (PyCFunction)PyStorage_Autocommit, METH_VARARGS, autocommit__doc},
    {"store", (PyCFunction)PyStorage_store, METH_VARARGS, store__doc},
    {"load", (PyCFunction)PyStorage_load, METH_VARARGS, load__doc},
    {"save", (PyCFunction)PyStorage_save, METH_VARARGS, save__doc},
    {0, 0, 0, 0}
};

static void PyStorage_dealloc(PyStorage *o) {
    //o->~PyStorage();
    delete o;
}

static int PyStorage_print(PyStorage *o, FILE *f, int) {
    fprintf(f, "<PyStorage object at %x>", o);
    return 0;
}

static PyObject* PyStorage_getattr(PyStorage *o, char *nm) {
    return Py_FindMethod(StorageMethods, o, nm);
}

PyTypeObject PyStoragetype = {
    PyObject_HEAD_INIT(&PyType_Type)
    0,
    "PyStorage",
    sizeof(PyStorage),
    0,
    (destructor)PyStorage_dealloc, /*tp_dealloc*/
    (printfunc)PyStorage_print, /*tp_print*/
    (getattrfunc)PyStorage_getattr, /*tp_getattr*/
    0,      /*tp_setattr*/
    (cmpfunc)0, /*tp_compare*/
    (reprfunc)0, /*tp_repr*/
    0,      /*tp_as_number*/
    0,  /*tp_as_sequence*/
    0,      /*tp_as_mapping*/
};

static char* storage__doc = 
"storage() -- create a new in-memory storage (can load/save, but not commit/rollback)\n"
"storage(file) -- attach a storage object to manage an already opened stdio file\n"
"storage(filename, rw) -- open file, rw=0: readonly, rw=1:modify (creates if absent)";

static PyObject* PyStorage_new(PyObject* o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        switch (args.len()) {
            case 0:
                return new PyStorage (*new PyStrategy, true);
            case 1:
                if (!PyFile_Check((PyObject* )args[0])) {
                    break;
                }
                return new PyStorage(*new PyStrategy (PyFile_AsFile(args[0])), true);
            case 4:	// Rrrrrr...
	{
		if (!PyStorage_Check((PyObject *)args[0]))
			throw PWException(PyExc_TypeError, "First arg must be a storage object");
		c4_Storage& storage = *(PyStorage*)(PyObject *)args[0];
		if (!PyView_Check((PyObject *)args[1]))
			throw PWException(PyExc_TypeError, "Second arg must be a view object");
		c4_View& view = *(PyView*)(PyObject *)args[1];
		if (!PyProperty_Check((PyObject *)args[2]))
			throw PWException(PyExc_TypeError, "Third arg must be a property object");
		c4_MemoProp& prop = *(c4_MemoProp*)(c4_Property*)(PyProperty*)(PyObject*)args[2];
		int row = PWONumber(args[3]);

        return new PyStorage(*new SiasStrategy (storage, view, prop, row), true);
	}
            default: {
                    PWOString fnm(args[0]);
                    if (PyInt_Check((PyObject* )args[1])) {
                        PWONumber canModify(args[1]);
                            // nastiness to use our own strategy class
                        PyStrategy* sy = new PyStrategy;
                        sy->DataOpen(fnm, (int)canModify ? true : false);
                        if (!sy->IsValid())
                        {
                            delete sy;
                            throw PWException(PyExc_IOError,
                                "can't open storage file (it might be in use)");
                        }
                        return new PyStorage (*sy, true);
                    }
/* there's no way we can do this with our own strategy class, so forget it
                    else {
                        PWOString descr(args[1]);
                        return new PyStorage(fnm, descr);
                    }
*/
            }
        }
        throw PWException(PyExc_ValueError, "not a file, or r/w parameter missing");
    }
    catch(PWException e) {
        return e.toPython();
    }
}

class PyViewer : public c4_CustomViewer
{
	PWOSequence _data;
	c4_View _template;
	c4_Row _tempRow;
	bool _byPos;

public:
	PyViewer (const PWOSequence& data_, const c4_View& template_, bool byPos_);
    virtual ~PyViewer ();
    
    virtual c4_View GetTemplate();
    virtual int GetSize();
    virtual bool GetItem(int row_, int col_, c4_Bytes& buf_);
};

PyViewer::PyViewer (const PWOSequence& data_, const c4_View& template_, bool byPos_)
	: _data (data_), _template (template_), _byPos (byPos_)
{
}

PyViewer::~PyViewer ()
{
}

c4_View PyViewer::GetTemplate()
{
	return _template;
}

int PyViewer::GetSize()
{
	return _data.len();
}

bool PyViewer::GetItem(int row_, int col_, c4_Bytes& buf_)
{
	const c4_Property& prop = _template.NthProperty(col_);
	if (_byPos)
	{
		PWOSequence item (_data[row_]);
		PyRowRef::setFromPython(_tempRow, prop, item[col_]);
		return prop (_tempRow).GetData(buf_);
	}

		// create a row with just this single property value
		// this detour handles dicts and objects, because makeRow does
	c4_Row one;
	PyView v (prop); // nasty, stack-based temp to get at makeRow
	v.makeRow(one, _data[row_]);
	return prop (one).GetData(buf_);
}

static PyObject* PyView_wrap(PyObject* o, PyObject* _args) {
    try {
        PWOSequence args(_args);
        PWOSequence seq(args[0]);
        PWOSequence types(args[1]);
		PWONumber usetuples(0);
        if (args.len() > 2)
			usetuples = args[2];

        c4_View templ;
		for (int i = 0; i < types.len(); ++i)
		{
			const c4_Property& prop = *(PyProperty*)(PyObject*) types[i];
			templ.AddProperty(prop);
		}

        c4_View cv = new PyViewer (seq, templ, (int) usetuples != 0);
		return new PyView (cv);
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static PyMethodDef Mk4Methods[] = {
    {"View", PyView_new, METH_VARARGS, "view() - create a new unattached view"},
    {"Storage", PyStorage_new, METH_VARARGS, storage__doc},
    {"Property", PyProperty_new, METH_VARARGS, 
        "property(type, name) -- create a property of given type and name"},
    {"Wrap", PyView_wrap, METH_VARARGS, 
		"wrap(seq,props,usetuples=0) - wrap a sequence as a read-only view"},
    {0, 0, 0, 0}
};

extern "C"
__declspec(dllexport)
#ifdef _DEBUG
void initMk4py_d() {
    Py_InitModule4("Mk4py_d", Mk4Methods, mk4py_module_documentation,
                    0,PYTHON_API_VERSION);
}
#else
void initMk4py() {
    Py_InitModule4("Mk4py", Mk4Methods, mk4py_module_documentation,
                    0,PYTHON_API_VERSION);
}
#endif
