//  Copyright 1999 McMillan Enterprises, Inc. -- www.mcmillan-inc.com
//  Copyright (C) 1999 Jean-Claude Wippler.  All rights reserved.
//
//  View class implementation

#include "PyView.h"
#include "PyProperty.h"
#include "PyRowRef.h"
#include <PWOMSequence.h>
#include <PWONumber.h>
#include <PWOMapping.h>

static char* structure__doc = 
"structure() -- return list of properties";

static PyObject* PyView_structure(PyView *o, PyObject* _args) {
	try {
		return o->structure();
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* insert__doc = 
"insert(position, obj) -- coerce obj to row and insert";

static PyObject* PyView_insert(PyView *o, PyObject* _args, PyObject* kwargs) {
	try {
		PWOSequence args(_args);
		if (args.len() == 1)
			o->insertAt(PWONumber(args[0]), kwargs);
		else
			o->insertAt(PWONumber (args[0]), args[1]);
		Py_INCREF(Py_None);
		return Py_None;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* append__doc = 
"append(obj) -- coerce obj (or keyword args) to row and append, returns pos";

static PyObject* PyView_append(PyView *o, PyObject* _args, PyObject* kwargs) {
    try {
        PWOSequence args(_args);
        PWONumber ndx(o->GetSize());
        if (args.len() == 0)
            o->insertAt(ndx, kwargs);
        else
            o->insertAt(ndx, args[0]);
		return ndx.disOwn();
    }
    catch(PWException e) {
        return e.toPython();
    }
}

static char* delete__doc = 
"delete(position) -- delete row at specified position";

static PyObject* PyView_delete(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		o->RemoveAt(PWONumber (args[0]));
		Py_INCREF(Py_None);
		return Py_None;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* addproperty__doc = 
"addproperty(property) -- add temp column to view, use getas() for persistent columns";

static PyObject* PyView_addproperty(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		PWOBase prop(args[0]);
		if (PyProperty_Check((PyObject* )prop)) {
			PWONumber rslt(o->AddProperty(*(PyProperty *)(PyObject* )prop));
			return rslt.disOwn();
		}
		throw PWException(PyExc_TypeError, "Not a Property object");
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* select__doc = 
"select(criteria) -- return virtual view with selected rows\n"
"select(crit_lo, crit_hi) -- select rows in specified range (inclusive)";

static PyObject* PyView_select(PyView *o, PyObject* _args, PyObject* kwargs) {
	try {
		c4_Row temp;
		PWOSequence args(_args);
		if (args.len() == 0) 
			return new PyView(o->Select(o->makeRow(temp, kwargs, false)), o);
		if (args.len() == 1) 
			return new PyView(o->Select(o->makeRow(temp, args[0], false)), o);
		else {
			c4_Row temp2;
			return new PyView(o->SelectRange(o->makeRow(temp, args[0], false), 
											 o->makeRow(temp2, args[1], false)), o);
		}
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* sort__doc = 
"sort() -- return virtual sorted view (native key order)\n"
"sort(property...) -- sort on the specified properties";

static PyObject* PyView_sort(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		if (args.len()) {
			PyView crit;
			crit.addProperties(args);
			return new PyView (o->SortOn(crit), o);
		}
		return new PyView (o->Sort(), o);
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* project__doc = 
"project(property...) -- returns virtual view with only the named columns";

static PyObject* PyView_project(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		PyView crit;
		crit.addProperties(args);
		return new PyView (o->Project(crit));
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* flatten__doc = 
"flatten(subview_property, outer=0) -- produces 'flat' view from nested view";

static PyObject* PyView_flatten(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		const c4_Property& subview = *(PyProperty *)(PyObject* )args[0];
		bool outer = false;
		if (args.len() > 1) {
			PWONumber flag(args[1]);
			if ((int)flag > 0)
				outer = true;
		}
		return new PyView (o->JoinProp((const c4_ViewProp&) subview, outer));
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* join__doc = 
"join(otherview, property..., outer=0) -- join view on prop of same name and type";

static PyObject* PyView_join(PyView *o, PyObject* _args) {
	try {
		PWOSequence args(_args);
		PyView *other = (PyView *)(PyObject* )args[0];
		bool outer = false;
		int last = args.len();
		if (PyInt_Check((PyObject*)args[last-1])) {
			PWONumber flag(args[--last]);
			if ((int)flag > 0)
				outer = true;
		}
		PyView crit;
		crit.addProperties(args.getSlice(1,last));
		return new PyView (o->Join(crit, *other, outer));
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* unique__doc = 
"unique() -- returns view without duplicate rows, i.e. a set";

static PyObject *PyView_unique(PyView *o, PyObject *_args) {
	try {
		return new PyView(o->Unique());
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* union__doc = 
"union(view2) -- produce the set union of both views";

static PyObject *PyView_union(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (PyView_Check((PyObject *)args[0]))
			return new PyView(o->Union(*(PyView *)(PyObject *)args[0]));
		else
			throw PWException(PyExc_TypeError, "Arg must be a view object");
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* intersect__doc = 
"intersect(view2) -- produce the set intersection of both views";

static PyObject *PyView_intersect(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (PyView_Check((PyObject *)args[0]))
			return new PyView(o->Intersect(*(PyView* )(PyObject *)args[0]));
		else
			throw PWException(PyExc_TypeError, "Arg must be a view object");
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* different__doc = 
"different(view2) -- produce the set difference of both views (XOR)";

static PyObject *PyView_different(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (PyView_Check((PyObject *)args[0]))
			return new PyView(o->Different(*(PyView* )(PyObject *)args[0]));
		else
			throw PWException(PyExc_TypeError, "Arg must be a view object");
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* minus__doc = 
"minus(view2) -- all rows in view, but not in view2";

static PyObject *PyView_minus(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (PyView_Check((PyObject *)args[0]))
			return new PyView(o->Minus(*(PyView* )(PyObject *)args[0]));
		else
			throw PWException(PyExc_TypeError, "Arg must be a view object");
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* find__doc = 
"find(criteria, start=0) -- return index of row found, matching criteria";

static PyObject *PyView_find(PyView *o, PyObject *_args, PyObject* kwargs) {
	PWONumber start(0);
	try {
		PWOSequence args(_args);
		if (args.len() == 0) {
			PWOMapping dict(kwargs);
			if (dict.hasKey("start")) {
				start = dict["start"];
				dict.delItem("start");
			}
		} else
			kwargs = args[0];
		c4_Row temp;
		return PWONumber(o->Find(o->makeRow(temp, kwargs, false), start)).disOwn();
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* search__doc = 
"search(criteria) -- binary search (native view order), returns match or insert pos";

static PyObject *PyView_search(PyView *o, PyObject *_args, PyObject* kwargs) {
	try {
		PWOSequence args(_args);
		if (args.len() != 0)
			kwargs = args[0];
		c4_Row temp;
		return PWONumber(o->Search(o->makeRow(temp, kwargs, false))).disOwn();
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* access__doc = 
"access(memoprop, rownum, offset, length=0) -- get (partial) memo property contents";

static PyObject *PyView_access(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (!PyProperty_Check((PyObject *)args[0]))
			throw PWException(PyExc_TypeError, "First arg must be a property");

		c4_MemoProp& prop = *(c4_MemoProp*)(c4_Property*)(PyProperty*)(PyObject*)args[0];

		int index = PyInt_AsLong(args[1]);
		if (index < 0 || index >= o->GetSize())
			throw PWException(PyExc_IndexError, "Index out of range");

		c4_RowRef row = o->GetAt(index);

		long offset = PyInt_AsLong(args[2]);
		int length = args.len() == 3 ? 0 : PyInt_AsLong(args[3]);
		if (length <= 0)
		{
			length = prop(row).GetSize() - offset;
			if (length < 0)
				length = 0;
		}

		PyObject* buffer = PyString_FromStringAndSize(0, length);
		int o = 0;

		while (o < length)
		{
			c4_Bytes buf = prop(row).Access(offset + o, length - o);
			int n = buf.Size();
			if (n == 0)
				break;
			memcpy(PyString_AS_STRING(buffer) + o, buf.Contents(), n);
			o += n;
		}

		if (o < length)
			_PyString_Resize(&buffer, o);

		return buffer;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* modify__doc = 
"modify(memoprop, rownum, string, offset, diff=0) -- store (partial) memo contents\n"
"diff removes (<0) or inserts (>0) bytes, and is adjusted to within sensible range";

static PyObject *PyView_modify(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (!PyProperty_Check((PyObject *)args[0]))
			throw PWException(PyExc_TypeError, "First arg must be a property");

		c4_MemoProp& prop = *(c4_MemoProp*)(c4_Property*)(PyProperty*)(PyObject*)args[0];

		int index = PWONumber(args[1]);
		if (index < 0 || index >= o->GetSize())
			throw PWException(PyExc_IndexError, "Index out of range");

		c4_RowRef row = o->GetAt(index);

		PWOString buffer (args[2]);
		c4_Bytes data ((void*)(const char*)buffer, buffer.len());

		long offset = PWONumber(args[3]);
		int diff = args.len() == 4 ? 0 : (int) PWONumber(args[4]);

		if (!prop(row).Modify(data, offset, diff))
			throw PWException(PyExc_TypeError, "Failed to modify memo field");

		Py_INCREF(Py_None);
		return Py_None;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* itemsize__doc = 
"itemsize(prop, rownum=0) -- return size of item (rownum only needed for S/B/M types)\n"
"with integer fields, a result of -1/-2/-4 means 1/2/4 bits per value, respectively";

static PyObject *PyView_itemsize(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (!PyProperty_Check((PyObject *)args[0]))
			throw PWException(PyExc_TypeError, "First arg must be a property");

		c4_MemoProp& prop = *(c4_MemoProp*)(c4_Property*)(PyProperty*)(PyObject*)args[0];
		int index = args.len() == 1 ? 0 : (int) PWONumber(args[1]);
		if (index < 0 || index >= o->GetSize())
			throw PWException(PyExc_IndexError, "Index out of range");

		return PWONumber(prop(o->GetAt(index)).GetSize()).disOwn();
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static char* relocaterows__doc = 
"relocaterows(from, count, dest, pos) -- relocate rows within views of same storage\n"
"from is source offset, count is number of rows, pos is destination offset\n"
"both views must have a compatible structure (field names may differ)";

static PyObject *PyView_relocaterows(PyView *o, PyObject *_args) {
	try {
		PWOSequence args(_args);
		if (!PyView_Check((PyObject *)args[2]))
			throw PWException(PyExc_TypeError, "Third arg must be a view object");
		
		PyView& dest = *(PyView *)(PyObject *)args[2];

		int from = PWONumber(args[0]);
		if (from < 0)
			from += o->GetSize();
		int count = PWONumber(args[1]);
		if (from < 0 || count < 0 || from + count > o->GetSize())
			throw PWException(PyExc_IndexError, "Source index out of range");

		int pos = PWONumber(args[3]);
		if (pos < 0)
			pos += dest.GetSize();
		if (pos < 0 || pos > dest.GetSize())
			throw PWException(PyExc_IndexError, "Destination index out of range");
		
		if (!o->RelocateRows(from, count, dest, pos))
			throw PWException(PyExc_TypeError, "Failed to relocate rows");

		Py_INCREF(Py_None);
		return Py_None;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static PyMethodDef ViewMethods[] = {
	{"insert", (PyCFunction)PyView_insert, METH_VARARGS | METH_KEYWORDS, insert__doc},
	{"append", (PyCFunction)PyView_append, METH_VARARGS | METH_KEYWORDS, append__doc},
	{"delete", (PyCFunction)PyView_delete, METH_VARARGS, delete__doc},
	{"structure", (PyCFunction)PyView_structure, METH_VARARGS, structure__doc},
	{"select", (PyCFunction)PyView_select, METH_VARARGS | METH_KEYWORDS, select__doc},
	{"addproperty", (PyCFunction)PyView_addproperty, METH_VARARGS, addproperty__doc},
	{"sort", (PyCFunction)PyView_sort, METH_VARARGS, sort__doc},
	{"project", (PyCFunction)PyView_project, METH_VARARGS, project__doc},
	{"flatten", (PyCFunction)PyView_flatten, METH_VARARGS, flatten__doc},
	{"join", (PyCFunction)PyView_join, METH_VARARGS, join__doc},
	{"union", (PyCFunction)PyView_union, METH_VARARGS, union__doc},
	{"intersect", (PyCFunction)PyView_intersect, METH_VARARGS, intersect__doc},
	{"different", (PyCFunction)PyView_different, METH_VARARGS, different__doc},
	{"minus", (PyCFunction)PyView_minus, METH_VARARGS, minus__doc},
	{"unique", (PyCFunction)PyView_unique, METH_VARARGS, unique__doc},
	{"find", (PyCFunction)PyView_find, METH_VARARGS | METH_KEYWORDS, find__doc},
	{"search", (PyCFunction)PyView_search, METH_VARARGS | METH_KEYWORDS, search__doc},
	{"access", (PyCFunction)PyView_access, METH_VARARGS, access__doc},
	{"modify", (PyCFunction)PyView_modify, METH_VARARGS, modify__doc},
	{"itemsize", (PyCFunction)PyView_itemsize, METH_VARARGS, itemsize__doc},
	{"relocaterows", (PyCFunction)PyView_relocaterows, METH_VARARGS, relocaterows__doc},
	{0, 0, 0, 0}
};
/*
	Duplicate(deep=0)  (__copy__ and __deepcopy__ as methods, too)
	Clone()
	Product(other)
	GroupBy(as, keys...) -> view with embedded view (as)
	Counts(as, keys...) -> view with new column (as)
*/
static int PyView_length(PyView *o) {
	return o->GetSize();
}

static PyObject* PyView_concat(PyView *o, PyView *other) {
	try {
		if (other->ob_type != &PyViewtype)
			throw PWException(PyExc_TypeError, "Not a PyView");
		return new PyView(o->Concat(*other));
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static PyObject* PyView_repeat(PyView *o, int n) {
	try {
		PyView* tmp = new PyView;
		while (--n >= 0) 
			tmp = new PyView(tmp->Concat(*o)); //!! a huge stack of views?
		return tmp;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static PyObject* PyView_getitem(PyView *o, int n) {
	try {
		PyObject *rslt = o->getItem(n);
		if (rslt == 0)
			PyErr_SetString(PyExc_IndexError, "Index out of range");
		return rslt;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static PyObject* PyView_getslice(PyView *o, int s, int e) {
	try {
		return o->getSlice(s,e);
	}
	catch(PWException e) {
		return e.toPython();
	}
}

static int PyView_setitem(PyView *o, int n, PyObject* v) {
	try {
		if (v == 0) {
			if (n < 0)
				n += o->GetSize();
			if (n >= o->GetSize() || n < 0)
				throw PWException(PyExc_IndexError, "Index out of range");
			o->RemoveAt(n);
			return 0;
		}

		return o->setItem(n,v);
	}
	catch(PWException e) {
		e.toPython();
		return -1;
	}
}

static int PyView_setslice(PyView *o, int s, int e, PyObject* v) {
	if (v == 0) {
		PWOSequence seq;
		return o->setSlice(s,e,seq);
	}

	return o->setSlice(s,e,PWOSequence(v));
}

static PySequenceMethods ViewAsSeq = {
	(inquiry)PyView_length, //sq_length
	(binaryfunc)PyView_concat, //sq_concat
	(intargfunc)PyView_repeat, //sq_repeat
	(intargfunc)PyView_getitem, //sq_item
	(intintargfunc)PyView_getslice, //sq_slice
	(intobjargproc)PyView_setitem, //sq_ass_item
	(intintobjargproc)PyView_setslice, //sq_ass_slice
};

static void PyView_dealloc(PyView *o) {
	//o->~PyView();
	delete o;
}

static int PyView_print(PyView *o, FILE *f, int) {
	fprintf(f, "<PyView object at %x>", (int)o);
	return 0;
}

static PyObject* PyView_getattr(PyView *o, char *nm) {
	PyObject *rslt;
	try {
		rslt = Py_FindMethod(ViewMethods, o, nm);
		if (rslt)
			return rslt;
		PyErr_Clear();
		int ndx = o->FindPropIndexByName(nm);
		if (ndx > -1) 
			return new PyProperty(o->NthProperty(ndx));
		throw PWException(PyExc_AttributeError, nm);
	}
	catch(PWException e) {
		return e.toPython();
	}
}

PyTypeObject PyViewtype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"PyView",
	sizeof(PyView),
	0,
	(destructor)PyView_dealloc, /*tp_dealloc*/
	(printfunc)PyView_print, /*tp_print*/
	(getattrfunc)PyView_getattr, /*tp_getattr*/
	0,		/*tp_setattr*/
	(cmpfunc)0, /*tp_compare*/
	(reprfunc)0, /*tp_repr*/
	0,		/*tp_as_number*/
	&ViewAsSeq,	/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
};

PyObject* PyView_new(PyObject* o, PyObject* _args) {
	try {
		return new PyView;
	}
	catch(PWException e) {
		return e.toPython();
	}
}

PyView::PyView() : PyHead(PyViewtype), _base (0) {
}

PyView::PyView(const c4_View& o, PyView *owner)
  : PyHead(PyViewtype), c4_View(o), _base (owner) {
	if (owner && owner->_base)
		_base = owner->_base;
}

c4_Row& PyView::makeRow(c4_Row& tmp, PyObject* o, bool useDefaults) {
    for (int i=0; i < NumProperties(); i++) {
        const c4_Property& prop = NthProperty(i);
        PyObject* attr = 0;
        if (o) {
            if (PyDict_Check(o))
            {
                attr = PyDict_GetItemString(o, (char *)prop.Name());
                Py_XINCREF(attr);
            }
            else if (PyList_Check(o))
            {
                attr = PyList_GetItem(o, i);
                Py_XINCREF(attr);
            }
            else 
                attr = PyObject_GetAttrString(o, (char *)prop.Name());
        }
        if (attr)
        {
            PyRowRef::setFromPython(tmp, prop, attr);
            Py_DECREF(attr);
        }
        else {
            PyErr_Clear();
            if (useDefaults)
                PyRowRef::setDefault(tmp, prop);
        }
    }
    if (!useDefaults)
        if (tmp.Container().NumProperties() == 0)
            throw PWException(PyExc_ValueError, "Object has no usable attributes");
    return tmp;
}

void PyView::insertAt(int i, PyObject* o) {
	c4_Row temp;
	InsertAt(i, makeRow(temp, o));
}

PyObject* PyView::structure() {
	int n = NumProperties();
	PWOList rslt(n);
	for (int i = 0; i < n; i++) {
		rslt[i] = new PyProperty(NthProperty(i));
	}
	return rslt.disOwn();
}

PyView *PyView::getSlice(int s, int e) {
	int sz = GetSize();
	if (s < 0)
		s += sz;
	if (e < 0)
		e += sz;
	if (e > sz)
		e = sz;
	if (s >= 0 && s < sz)
		if (e > s && e <= sz)
			return new PyView(Slice(s,e));
	return new PyView(Clone());
}

int PyView::setSlice(int s, int e, const PWOSequence& lst) {
	int sz = GetSize();
	if (s < 0)
		s += sz;
	if (e < 0)
		e += sz;
	if (e > sz)
		e = sz;
	int i = 0;
	for (; i < lst.len() && s < e; i++, s++)
		setItem(s, lst[i]);
	for (; i < lst.len(); i++, s++)
		insertAt(s, lst[i]);
	if (s < e)
		RemoveAt(s, e - s);
	return 0;
}

PyRowRef *PyView::getItem(int i) {
	if (i < 0)
		i += GetSize();
	if (i >= GetSize() || i < 0)
		return 0;
	if (_base) {
		c4_RowRef derived = GetAt(i);
		int ndx = _base->GetIndexOf(derived);
		if (ndx > -1)
			return new PyRowRef(_base->GetAt(ndx), 0);
		throw PWException(PyExc_RuntimeError, "Row not found in base view");
	}
	return  new PyRowRef(GetAt(i), _base);
}

int PyView::setItem(int i, PyObject* v) {
	if (PyRowRef_Check(v))
		return setItem(i, *(PyRowRef *)v);
	c4_Row temp;
	return setItem(i, makeRow(temp, v));
}

int PyView::addProperties(const PWOSequence& lst) {
	int i=0;
	for (; i<lst.len(); i++) {
		if (PyProperty_Check((PyObject* )lst[i])) {
			AddProperty(*(PyProperty*)(PyObject* )lst[i]);
		}
	}
	return i;
}
