# Copyright (C) 1999 Jean-Claude Wippler.  All rights reserved.
#
# object commands to access local files, FTP, or MetaKit (views / Tequila)

#   set h [iohan:local directory]
#       # creates a handler for local files
#
#   set h [iohan:ftp url ?userid? ?password?]
#       # creates a handler for files on FTP server (needs FTP pkg)
#
#   set h [iohan:mk view]			# assumes datafile is open
#       # creates a handler for entries stored in a MetaKit datafile
#   
#   set h [iohan:tequila array]		# assumes Tequila has been set up
#       # creates a handler for entries stored on a Tequila server
#   
#   array set listing [$h list ?wildcard?]
#       # sets of an array with all files, value is modification time
#        
#   set data [$h fetch filename]
#       # fetches the contents of a file
#       
#   set timestamp [$h store filename contents]
#       # stores contents into a file, returns new modification time
#       
#   set timestamp [$h append filename contents]
#       # appends contents to a file, creating it if necessary
#       
#   set ok [$h delete filename]
#       # delete file, return 1 if it existed and was removed
#       
#   $h destroy
#       # deletes the handler command object

if {![info exists iohan:seq]} {set iohan:seq 0}

proc iohan:local {dir} {
    set cmd iohan:#[incr ::iohan:seq]
    proc $cmd {args} [format {
        eval [list iohan:local: %s] $args
    } [list $cmd $dir]]
    return $cmd
}

proc iohan:local: {cmd dir action args} {
    eval [list iohan:local:$action $dir] $args
}

proc iohan:local:destroy {dir} {
    uplevel {rename $cmd ""}
}

proc iohan:local:list {dir {match *}} {
    set result {}
    
    set wd [pwd]
    catch {
        cd $dir
        foreach f [lsort [glob -nocomplain $match]] {
            if [file isfile $f] {
                lappend result $f [file mtime $f]
            }
        }
    }
    cd $wd
    
    return $result
}

proc iohan:local:fetch {dir file} {
    set path [file join $dir $file]
    
    set fd [open $path]
    fconfigure $fd -translation binary
    set result [read $fd [file size $path]]
    close $fd
    
    return $result
}

proc iohan:local:store {dir file contents} {
    set path [file join $dir $file]
    
    set fd [open $path w]
    fconfigure $fd -translation binary
    puts -nonewline $fd $contents
    close $fd
    
    return [file mtime $path]
}

proc iohan:local:append {dir file contents} {
    set path [file join $dir $file]
    
    set fd [open $path a+]
    fconfigure $fd -translation binary
    puts -nonewline $fd $contents
    close $fd
    
    return [file mtime $path]
}

proc iohan:local:delete {dir file} {
    set path [file join $dir $file]
    
    set ok [file isfile $path]
    file delete $path
    return $ok
}

    # utility call to set up the proper session context
proc iohan:ftp_session {{sess ""}} {
    upvar #0 iohan:ftp_state cb

	update idletasks
	
    if {[info exists cb(curr)]} {
        if {$cb(curr) == $sess} return
        
        tclLog [concat ftp:close $cb(curr)]
        FTP::Close
        unset cb(curr)
    }
    
    if {$sess == ""} return
    
    catch {parray FTP::ftp}
    
    if {![eval FTP::Open $sess]} {
    	after 1000 ;# one retry
    	if {![eval FTP::Open $sess]} {
        	error "cannot connect to [lindex $sess 0]"
        }
    }
    
	tclLog [concat ftp:open $sess]
    set cb(curr) $sess
    
    FTP::Type binary
    
    	# Close sessions after 2 minutes to avoid a bug in the FTP
    	# package (reporting an error after 10 minutes if last failed).
    	# This is harmless because open sessions act merely as cache.
    	# The session will be re-opened as soon as it is (re-)used.
    after cancel iohan:ftp_session
    after 120000 iohan:ftp_session
}

proc iohan:ftp {url {user anonymous} {pw some@where.com}} {
    package require FTP
    set FTP::VERBOSE 0

	proc FTP::DisplayMsg {msg {state ""}} {
		switch $state {
		  error		{tclLog [list ftp-error $msg]}
		}
	}
        # the following two lines are to undo all tkcon special-casing
    catch {alias ::FTP::List ""}
    catch {rename ::FTP::List_org ::FTP::List}
        
    set site $url
    set dir .
    regexp {^([^/]+)/?(.*)$} $url - site dir
    set sess [list $site $user $pw]
    
    iohan:ftp_session $sess
    
    set cmd iohan:#[incr ::iohan:seq]
    proc $cmd {args} [format {
        eval [list iohan:ftp: %s] $args
    } [list $cmd $sess $dir]]
    return $cmd
}

proc iohan:ftp: {cmd sess dir action args} {
    iohan:ftp_session $sess
    if {$dir == ""} {set dir .}
    eval [list iohan:ftp:$action $dir] $args
}

proc iohan:ftp:destroy {dir} {
    iohan:ftp_session
    uplevel {rename $cmd ""}
}

proc iohan:ftp:list {dir {match *}} {
    set result {}
    
    foreach line [::FTP::List $dir] {
        if {![string match -* $line]} continue
        set rc [scan $line "%s %s %s %s %s %s %s %s %s %s %s" \
                            perm l u g size d1 d2 d3 name link linksource]
        if {![string match $match $name]} continue
        lappend result $name [FTP::ModTime $dir/$name]
    }

    return $result
}

proc iohan:ftp:fetch {dir file} {
    set tmp xyzF
    
    set ok [FTP::Get $dir/$file $tmp]
    
    set fd [open $tmp]
    fconfigure $fd -translation binary
    set result [read $fd [file size $tmp]]
    close $fd
    
    file delete $tmp
    if {!$ok} {error "can't fetch $file"}
        
    return $result
}

proc iohan:ftp:store {dir file contents} {
    set tmp "tcl[clock seconds].tmp"
    
    set fd [open $tmp w]
    fconfigure $fd -translation binary
    puts -nonewline $fd $contents
    close $fd
    
    set ok [FTP::Put $tmp $dir/$file]
    
    file delete $tmp
    if {!$ok} {error "can't store $file"}
    
    return [FTP::ModTime $dir/$file]
}

proc iohan:ftp:append {dir file contents} {
    set tmp xyzF
    
    set fd [open $tmp w]
    fconfigure $fd -translation binary
    puts -nonewline $fd $contents
    close $fd
    
    set ok [FTP::Append $tmp $dir/$file]
    
    file delete $tmp
    if {!$ok} {error "can't fetch $file"}
    
    return [FTP::ModTime $dir/$file]
}

proc iohan:ftp:delete {dir file} {
    FTP::Delete $dir/$file
}

proc iohan:mk {dir} {
	mk::view layout $dir "name date:I contents:B"
    set cmd iohan:#[incr ::iohan:seq]
    proc $cmd {args} [format {
        eval [list iohan:mk: %s] $args
    } [list $cmd $dir]]
    return $cmd
}

proc iohan:mk_dirty {dir} {
	regsub {\..*} $dir {} dir
	after cancel mk::file commit $dir
	after 10000 mk::file commit $dir
}

proc iohan:mk: {cmd dir action args} {
    eval [list iohan:mk:$action $dir] $args
}

proc iohan:mk:destroy {dir} {
    uplevel {rename $cmd ""}
}

proc iohan:mk:list {dir {match *}} {
    set result {}
	foreach n [mk::select $dir -glob name $match] {
		eval lappend result [mk::get $dir!$n name date]
	}
    return $result
}

proc iohan:mk:fetch {dir file} {
	foreach n [mk::select $dir -count 1 name $file] {
    	return [mk::get $dir!$n contents]
    }
	error "cannot find MK entry '$file'"
}

proc iohan:mk:store {dir file contents} {
	iohan:mk_dirty $dir
	set d [clock seconds]
	foreach n [mk::select $dir -count 1 name $file] {
		mk::set $dir!$n name $file date $d contents $contents
    	return $d
    }
	mk::row append $dir name $file date $d contents $contents
    return $d
}

proc iohan:mk:append {dir file contents} {
		# could use the new mk::channel of Mk4tcl 1.2
    if {[catch {iohan:mk:fetch $dir $file} old]} {
    	set old ""
    }
    return [iohan:mk:store $dir $file $old$contents]
}

proc iohan:mk:delete {dir file} {
	foreach n [mk::select $dir -count 1 name $file] {
		iohan:mk_dirty $dir
		mk::row delete $dir!$n
		return 1
    }
	return 0
}

proc iohan:tequila {aname {type X}} {
	tequila::do Define $aname 0 $type
    set cmd iohan:#[incr ::iohan:seq]
    proc $cmd {args} [format {
        eval [list iohan:tequila: %s] $args
    } [list $cmd $aname]]
    return $cmd
}

proc iohan:tequila: {cmd aname action args} {
    eval [list iohan:tequila:$action $aname] $args
}

proc iohan:tequila:destroy {aname} {
    uplevel {rename $cmd ""}
}

proc iohan:tequila:list {aname {match *}} {
	set result [tequila::do Listing $aname]
	if {$match != "*"} {
    	array set a $result
    	set result [array get a $match]
	}
	return $result
}

proc iohan:tequila:fetch {aname file} {
	tequila::do Get $aname $file
}

proc iohan:tequila:store {aname file contents} {
	set d [clock seconds]
	tequila::do Set $aname $file $contents $d
    return $d
}

proc iohan:tequila:append {aname file contents} {
	tequila::do Append $aname $file $contents
	return [clock seconds]
}

proc iohan:tequila:delete {aname file} {
	if {[catch {tequila::do Get $aname $file}]} {
		return 0
	}
	tequila::do Unset $aname $file
	return 1
}
