#ifdef MALLOC_FUNC_CHECK
#include <malloc_dbg.h>
#endif

#include <stdio.h>
#include <tk.h>
#include <ndbm.h>
#include <fcntl.h>

#include "ndbmcmd.h"

/*
 * prototypes for library functions
 */

EXTERN VOID free _ANSI_ARGS_((VOID *));
EXTERN VOID perror _ANSI_ARGS_((const char *));

#ifndef MALLOC_FUNC_CHECK
EXTERN VOID *malloc _ANSI_ARGS_((long unsigned int ));
EXTERN VOID *realloc _ANSI_ARGS_((VOID *, long unsigned int ));
#endif

/*
 * DEFINES for error reporting
 */

#define ARGS1		"%s: \"%s\": error: %s"
#define ARGS2   	"%s: \"%s\" \"%s\": error: %s"
#define ARGS3		"%s: \"%s\" \"%s\" \"%s\": error: %s"

#define BAD_MODE 	"%s: bad mode \"%s\""

#define OPEN_USAGE 	"ndbm open <name> ?mode?"
#define CLOSE_USAGE 	"ndbm close <name>"
#define INSERT_USAGE 	"ndbm insert <name> <key> <content>"
#define STORE_USAGE 	"ndbm store <name> <key> <content>"
#define FETCH_USAGE 	"ndbm fetch <name> <key>"
#define DELETE_USAGE 	"ndbm delete <name> <key>"
#define LIST_USAGE 	"ndbm list <name>"
#define NDBM_USAGE	"ndbm <command> <name> ?parameter? ..."
#define FIRST_USAGE	"ndbm firstkey <name>"
#define NEXT_USAGE	"ndbm nextkey <name> <lastkey>"

/*
 * a database ..
 */

typedef struct db {
	char *name ;
	int mode ;
	DBM *db ;
} DB ;

/*
 * all open databases
 */

Tcl_HashTable databases ;

/*
 * counter for handles
 */

static handle ;

/*
 * prototypes for internal functions
 */

static int tcl_ndbm_close _ANSI_ARGS_((ClientData client, Tcl_Interp *interp, 
		   int argc, char **argv));
static int tcl_ndbm_insert _ANSI_ARGS_((ClientData client, Tcl_Interp *interp, 
		    int argc, char **argv));
static int tcl_ndbm_store _ANSI_ARGS_((ClientData client, Tcl_Interp *interp,
		   int argc, char **argv));
static int tcl_ndbm_fetch _ANSI_ARGS_((ClientData client, Tcl_Interp *interp,
		   int argc, char **argv));
static int tcl_ndbm_delete _ANSI_ARGS_((ClientData client, Tcl_Interp *interp,
		    int argc, char **argv));
static int tcl_ndbm_list _ANSI_ARGS_((ClientData client, Tcl_Interp *interp,
		  int argc, char **argv));
static int tcl_ndbm_open _ANSI_ARGS_((ClientData client, Tcl_Interp *interp,
		  int argc, char **argv));
static int tcl_ndbm_firstkey _ANSI_ARGS_((ClientData client, 
					  Tcl_Interp *interp, 
					  int argc, char **argv));
static int tcl_ndbm_nextkey _ANSI_ARGS_((ClientData client, 
					 Tcl_Interp *interp, 
					 int argc, char **argv));

/* ****************************************************************** */

/*
 * tcl_ndbm_close
 */

static int 
tcl_ndbm_close (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_HashEntry *entry ;
	char buf[512] ;

	/*
	 * check argument number
	 */
	if (argc != 2) {
		Tcl_AppendResult (interp,CLOSE_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	/*
	 * retrieve db from hash table
	 */

	db = (DB*) Tcl_GetHashValue (entry);

	/*
	 * close db and remove it from tables
	 */

	dbm_close (db->db) ;

	free (db->name);
	free ((char*)db);
	Tcl_DeleteHashEntry(entry);

	/*
	 * ok
	 */

	return (TCL_OK);

}

/* ****************************************************************** */

/*
 * tcl_ndbm_insert <name> <key> <content>
 */

static int 
tcl_ndbm_insert (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	char buf[512] ;
	datum key, content ;
	int ret ;
	DB *db ;
	Tcl_HashEntry *entry ;

	/*
	 * check argument number
	 */
	if (argc != 4) {
		Tcl_AppendResult (interp,INSERT_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	/*
	 * retrieve db
	 */

	db = (DB*) Tcl_GetHashValue (entry);

	/* prepare data and store it 
	 * if the key is allready stored in db 
	 * an error will be reported
	 */

	key.dptr = argv[2];
	key.dsize = strlen(argv[2]) + 1 ;

	content.dptr = argv[3];
	content.dsize = strlen(argv[3]) + 1 ;

	ret = dbm_store(db->db, key, content, DBM_INSERT);
	if (ret != 0) {
		sprintf(buf, ARGS3,argv[0],argv[1],argv[2],argv[3],"");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	return TCL_OK;
}

/* ****************************************************************** */

/*
 * tcl_ndbm_store <name> <key> <content>
 */

static int 
tcl_ndbm_store (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	char buf[512] ;
	datum key, content ;
	int ret ;
	DB *db ;
	Tcl_HashEntry *entry ;

	/*
	 * check argument number
	 */
	if (argc != 4) {
		Tcl_AppendResult (interp,STORE_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	/*
	 * retrieve db
	 */

	db = (DB*) Tcl_GetHashValue (entry);

	/* prepare data and store it 
	 * if the key is allready stored in db the new contents will
	 * will replace the old
	 */

	key.dptr = argv[2];
	key.dsize = strlen(argv[2]) + 1 ;

	content.dptr = argv[3];
	content.dsize = strlen(argv[3]) + 1 ;

	ret = dbm_store(db->db, key, content, DBM_REPLACE);
	if (ret != 0) {
		sprintf(buf, ARGS3,argv[0],argv[1],argv[2],argv[3],"");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	return TCL_OK;
}

/* ****************************************************************** */

/*
 * tcl_ndbm_fetch <name> <key>
 */

static int 
tcl_ndbm_fetch (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_HashEntry *entry ;
	datum key, content ;
	char buf[512] ;

	/*
	 * check argument number
	 */
	if (argc != 3) {
		Tcl_AppendResult (interp,FETCH_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}
	
	db = (DB*) Tcl_GetHashValue(entry);

	key.dptr = argv[2];
	key.dsize = strlen(argv[2]) + 1 ;

	content = dbm_fetch (db->db,key) ;
	
	if (content.dptr) {
		/*
	 	* found
	 	*/
		Tcl_AppendResult (interp, content.dptr, (char*)0);
	}

	return TCL_OK ;
}

/* ****************************************************************** */

/*
 * tcl_ndbm_delete <name> <key>
 */

static int 
tcl_ndbm_delete (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_HashEntry *entry;
	datum key;
	char buf[512] ;
	int ret ;


	/*
	 * check argument number
	 */
	if (argc != 3) {
		Tcl_AppendResult (interp,DELETE_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	db = (DB*) Tcl_GetHashValue(entry);

	/* check for read mode */

	if (db->mode == O_RDONLY) {
		sprintf(buf,ARGS1,argv[0],argv[1],"is opened for reading");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	key.dptr = argv[2];
	key.dsize = strlen(argv[2]) + 1 ;

	ret = dbm_delete(db->db, key) ;

	if (ret == 0) {
		return TCL_OK ;
	} else {
		/*
		 * not found
		 */
		sprintf(buf,ARGS2,argv[0],argv[1],argv[2], "can't delete");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}
}

/* ****************************************************************** */

/*
 * tcl_ndbm_list <name>
 * the key is not allowed to be longer than 1023 bytes!
 */

static int 
tcl_ndbm_list (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_DString result ;
	Tcl_HashEntry *entry ;
	datum key ;
	char buf[1024];

	/*
	 * check argument number
	 */
	if (argc != 2) {
		Tcl_AppendResult (interp,LIST_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	db = (DB*) Tcl_GetHashValue(entry) ;
	Tcl_DStringInit(&result);

	key = dbm_firstkey(db->db);
	if (!key.dptr) {
		/*
		 * empty db
		 */
		return TCL_OK ;
	}

	/* 
	 * copy key to result and free original key
	 */

	do {
		Tcl_DStringAppendElement(&result,key.dptr);
		/*
		 * remember key for ndbm_nextkey()
		 */
		strcpy(buf,key.dptr);
		free(key.dptr);
		key.dptr = buf ;
		key = dbm_nextkey(db->db,key) ;
	} while (key.dptr) ;

	Tcl_DStringResult(interp, &result);
	return TCL_OK ;
}

/*
 * tcl_ndbm_firstkey
 */

static int 
tcl_ndbm_firstkey (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_HashEntry *entry ;
	datum key ;
	char buf[512];

	/*
	 * check argument number
	 */
	if (argc != 2) {
		Tcl_AppendResult (interp,FIRST_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	db = (DB*) Tcl_GetHashValue(entry) ;

	key = dbm_firstkey(db->db);
	if (!key.dptr) {
		/*
		 * empty db
		 */
		return TCL_OK ;
	}

	Tcl_AppendResult (interp,key.dptr,(char*)0);
	free (key.dptr);
	return TCL_OK ;
}


/*
 * tcl_ndbm_nextkey
 */

static int 
tcl_ndbm_nextkey (client, interp, argc, argv)
	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	DB *db ;
	Tcl_HashEntry *entry ;
	datum oldkey, newkey ;
	char buf[512];

	/*
	 * check argument number
	 */
	if (argc != 3) {
		Tcl_AppendResult (interp,NEXT_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	entry = Tcl_FindHashEntry(&databases,argv[1]);

	if (!entry) {
		/*
		 * db not in hashtab
		 */
		sprintf(buf,ARGS1,argv[0],argv[1],"no such database");
		Tcl_AppendResult (interp,buf,(char*)0);
		return TCL_ERROR ;
	}

	db = (DB*) Tcl_GetHashValue(entry) ;

	
	oldkey.dsize = strlen(argv[2]) + 1;
	oldkey.dptr = argv[2] ;
	
	newkey = dbm_nextkey(db->db,oldkey);

	if (!newkey.dptr) {
		/*
		 * empty db
		 */
		return TCL_OK ;
	}

	Tcl_AppendResult (interp,newkey.dptr,(char*)0);
	free (newkey.dptr);
	return TCL_OK ;
}


/* ****************************************************************** */

/*
 * tcl_ndbm_open
 * 
 * mode in {r,rw,rwc,rwn}
 */

static int 
tcl_ndbm_open (client, interp, argc, argv)

	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	int mode, new ;
	char buf[512] ;
	DB *db ;
	Tcl_HashEntry *entry ;

	/*
	 * check argument number
	 */
	if (argc < 2 || argc > 3) {
		Tcl_AppendResult (interp,OPEN_USAGE,(char*)0);
		return(TCL_ERROR);
	}

	mode = O_RDONLY ;

	/*
	 * check mode string if given
	 */
	if (argc == 3) {
		
		if (strcmp(argv[2],"r")==0)
			mode = O_RDONLY ;
		else if (strcmp(argv[2],"rw")==0)
			mode = O_RDWR | O_SYNC ;
		else if (strcmp(argv[2],"rwc")==0)
			mode = O_CREAT | O_RDWR | O_SYNC ;
		else if (strcmp(argv[2],"rwn")==0)
			mode = O_CREAT | O_EXCL | O_RDWR | O_SYNC ;
		else {
			sprintf(buf, BAD_MODE, argv[0], argv[2]);
			Tcl_AppendResult (interp,buf,(char *)0);
			return (TCL_ERROR);
		}
	}

	/*
	 * open ndbm and register it
	 */

	/* name not in hashtab - create new db */

	db = (DB*) malloc (sizeof(DB)) ;

	if (!db) {
		/* malloc problem */
		perror ("malloc for db struct in db_open");
		exit (-1);
	}

	/*
	 * create new name and malloc space for it
	 * malloc extra space for name
	 */

	sprintf(buf,"gdbm%i",handle);
	handle++ ;

	db->name = (char *) malloc (strlen(buf)+1) ;
	if (!db->name) {
		/* malloc problem */
		perror ("malloc for name in db_open");
		exit (-1);
        }
	strcpy(db->name,buf);

	db->mode = mode ;
	db->db = dbm_open(argv[1],mode,0644);

	if (!db->db) {
		/*
		 * error occurred
		 * free previously allocated memory
		 */
		free (db->name);
		free ((char*) db);
		db = (DB*) NULL ;

		if (argc == 3) {
			sprintf(buf,ARGS2,argv[0],argv[1],argv[2],"");
		} else {
			sprintf(buf,ARGS1,argv[0],argv[1],"");
		}
		Tcl_AppendResult (interp,buf,(char*)0);
		return (TCL_ERROR);

	} else {
		/* 
		 * success
		 * enter db to hashtab
		 */
		entry = Tcl_CreateHashEntry(&databases,db->name,&new);
		Tcl_SetHashValue(entry,(char*)db);
		Tcl_AppendResult (interp,db->name,(char*)0);
		return TCL_OK;
	}

}

/*
 * tcl_ndbm
 * central command dispatcher
 */

static int 
tcl_ndbm (client, interp, argc, argv)

	ClientData client;
	Tcl_Interp *interp;
	int argc;
	char **argv;

{
	int i;
	char msg[80] ;

	static char *cmds[] = {
		"close",
		"delete",
		"fetch",
		"firstkey",
		"insert",
		"list",
		"nextkey",
		"open",
		"store",
		0
	};

	static (*f[])() = {
		tcl_ndbm_close,
		tcl_ndbm_delete,
		tcl_ndbm_fetch,
		tcl_ndbm_firstkey,
		tcl_ndbm_insert,
		tcl_ndbm_list,
		tcl_ndbm_nextkey,
		tcl_ndbm_open,
		tcl_ndbm_store
	};

	/*
	 * check argument number
	 */
	if (argc < 3) {
		Tcl_AppendResult (interp,NDBM_USAGE,(char*)0);
		return(TCL_ERROR);
	}
	
	for (i = 0 ; cmds[i] != NULL; i++) {
		if (strcmp(cmds[i],argv[1]) == 0) {
			/* 
			 * call appropriate function
			 */
			return (*f[i])(client,interp,argc-1,&argv[1]);
		}
	}
	
	/* command not found */

	sprintf (msg,"%s %s: no such command",argv[0],argv[1]);
	Tcl_AppendResult (interp,msg,(char*)0);
	return(TCL_ERROR);
}

/*
 * tcl_ndbm_init
 * register commands, init data structures
 */

int 
tcl_ndbm_init (interp)
	Tcl_Interp *interp;

{	
	Tcl_CreateCommand(interp,"ndbm", tcl_ndbm,
			  (ClientData)0, (Tcl_CmdDeleteProc *)0);

	/*
	 * init hash table
	 */

	Tcl_InitHashTable (&databases,TCL_STRING_KEYS);

	/*
	 * set handle to zero
	 */

	handle = 0 ;

	return TCL_OK ;
}
