/**********************************************************************

                  - TCLODBC COPYRIGHT NOTICE - 

This software is copyrighted by Roy Nurmi, contact address by email at
Roy.Nurmi@iki.fi. The following terms apply to all files associated with 
the software unless explicitly disclaimed in individual files.

The author hereby grant permission to use, copy, modify, distribute,
and license this software and its documentation for any purpose, provided
that existing copyright notices are retained in all copies and that this
notice is included verbatim in any distributions. No written agreement,
license, or royalty fee is required for any of the authorized uses.
Modifications to this software may be copyrighted by their authors
and need not follow the licensing terms described here, provided that
the new terms are clearly indicated on the first page of each file where
they apply.

IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY
DERIVATIVES THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
MODIFICATIONS.

GOVERNMENT USE: If you are acquiring this software on behalf of the
U.S. government, the Government shall have only "Restricted Rights"
in the software and related documentation as defined in the Federal 
Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
are acquiring the software on behalf of the Department of Defense, the
software shall be classified as "Commercial Computer Software" and the
Government shall have only "Restricted Rights" as defined in Clause
252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing, the
authors grant the U.S. Government and others acting in its behalf
permission to use and distribute the software in accordance with the
terms specified in this license. 

***********************************************************************/

//////////////////////////////////////////////////////////////////////////
//                                                                      //
//        ODBC-interface extension to tcl language by Roy Nurmi         //
//                             Version 1.4                              //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// VERSION HISTORY                                                      //
//                                                                      //
// 1.0 Initial version. Requires tcl 8.0 alpha 2 release                //
//                                                                      //
// 1.1 New version, compatible with tcl 8.0 beta 1 release              //
//                                                                      //
// 1.2 New version, enhanced functionality of statement columns command //
//                                                                      //                                                                     //
// 1.3 New version, added database configure option, index queries,     //
//                  new error object definition                         //                                                                     //
//                                                                      //                                                                     //
// 1.4 New version, added database datasources and drivers options      //
//                  and connecting with odbc connection string          //
//                                                                      //                                                                     //
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// INCLUDES
//

#ifdef WIN32

#include <windows.h>

extern "C" {
#include <string.h>
#include <tcl.h>
#include <sql.h>
#include <odbcinst.h>
#include <sqlext.h>
#include <stdlib.h>
}

#else

#define BOOL int
#define _declspec(x) int

extern "C" {
#include <string.h>
#include <tcl.h>
#include "sql.h"
#include "odbcinst.h"
#include "sqlext.h"
#include <stdlib.h>
}

#endif

#include "tclobj.hxx"


//////////////////////////////////////////////////////////////////////////
// DEFINES
//

// version number

#define TCLODBC_VERSION "1.4"

// This defines maximum lenghth of returned blob data. Some DBMS's return
// thoretical infinity (2^32 or something like that) as a maximum lenght 
// of a blob column instead of the current maximum. In this case this 
// defines the maximum buffer reserved.

#define MAX_BLOB_COLUMN_LENGTH  4096

// Errors are thrown with these two macros. In debug version also
// line information is included.

#ifdef _DEBUG
#define THROWSTR(str) \
{\
    TclObj errObj (str);\
    TclObj line (__LINE__);\
	Tcl_AppendToObj(errObj, " Line: ", -1);\
	Tcl_AppendToObj(errObj, line, -1);\
    throw errObj;\
}
#else
#define THROWSTR(str) \
{\
    throw TclObj(str);\
}
#endif
#ifdef _DEBUG
#define THROWOBJ(obj) \
{\
    TclObj line (__LINE__);\
	Tcl_AppendToObj(obj, " Line: ", -1);\
	Tcl_AppendToObj(obj, line, -1);\
    throw obj;\
}
#else
#define THROWOBJ(obj) \
{\
    throw obj;\
}
#endif

// A struct for storing strings with integer constants
struct NumStr 
{
	short numeric;
	char* string;
};

//////////////////////////////////////////////////////////////////////////
// DATABASE OBJECT CLASS HEADERS
//

// Common TclCmdObject ancestor to all database objects. This defines
// common destructing and command dispatching methods.

class TclCmdObject {
public:
    virtual ~TclCmdObject();
    // Virtual destructor for all database objects

    static void Destroy(ClientData);
    // Static function for destroying database objects

    static int Dispatch(ClientData clientData, Tcl_Interp *interp,
		int objc, Tcl_Obj *const objv[]);
    // Static function for dispatching commands to objects

private:
    virtual Dispatch(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) = 0;
    // Virtual function for class dependent implementation of
    // static Dispatch().
};


// Class TclDatabase provides a general database object, with
// connecting and transaction handling methods. Dispatch interface
// is defined for handling database object commands.

class TclDatabase : public TclCmdObject {
public:
    TclDatabase(char* db, char* uid, char* passwd);
    // Constructor creates a database object with connection
    // to a named datasource.

    TclDatabase(char* connectionstring);
    // Constructor creates a database object with a given connection
    // string.

    virtual ~TclDatabase();
    // Virtual destructor, disconnects from the database and destroys
    // the object.

    void Autocommit(BOOL on);
    // Autocommit sets the autocommit property on or off.

    void Transact(UWORD operation);
    // Transact is used for transaction handling. It is called with 
    // constant SQL_COMMIT or SQL_ROLLBACK.

   static TclObj Datasources();
    // List of all registered data source names

   static TclObj Drivers();
    // List of all registered drivers


private:
    virtual Dispatch(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]);
    HDBC dbc;
};

// Class TclStatement provides general statement handling methods, 
// including statement handle creation and destroying, executing 
// statement, and handling result set and result buffer.

class TclStatement : public TclCmdObject {
public:
    virtual ~TclStatement();
    // Destructor drops the statement handle and frees dynamic structures

    virtual void Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) = 0;
    // executes the statement with given arguments

    int ColumnCount();
    // returns the count of columns in current result set.

    TclObj Result();
    // The result set of an executed statement in a single Tcl_Obj.

    TclObj Value(Tcl_Interp *interp, int objc = 0, Tcl_Obj *const objv[] = NULL);
    // This first executes the statement, and then returns the result set

    TclObj Columns(int objc, Tcl_Obj *const objv[]);
    // returns the names of statement columns in a tcl list.

    BOOL Fetch(TclObj&);
    // Retches and returns the next row in the result set. Returns
    // false if no more data

protected:
    TclStatement(HDBC dbc);
    // Protected constructor creates a statement connected to a given data 
    // source. This is called from descendents constructors.

    HSTMT stmt;

private:
    virtual Dispatch(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]);
    int colCount;
    char **resultBuffer;

    void ReserveResultBuffer();
    void FreeResultBuffer();
};

class TclSqlStatement : public TclStatement {
public:
    TclSqlStatement(HDBC dbc, char* sql);
    // creates a sql statement object connected to a given data 
    // source. Sql clause is given as argument.

private:
    virtual void Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]);
};

class TclTableQuery : public TclStatement {
public:
    TclTableQuery(HDBC dbc);
    // creates a table query object connected to a given 
    // data source. 

private:
    virtual void Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]);
};

class TclColumnQuery : public TclStatement {
public:
    TclColumnQuery(HDBC dbc);
    // creates a column query object connected to a given 
    // data source. 

private:
    virtual void Execute(Tcl_Interp *interp, int objc, Tcl_Obj* const objv[]);
};

class TclIndexQuery : public TclStatement {
public:
    TclIndexQuery(HDBC dbc);
    // creates a index query object connected to a given 
    // data source. 

private:
    virtual void Execute(Tcl_Interp *interp, int objc, Tcl_Obj* const objv[]);
};


//////////////////////////////////////////////////////////////////////////
// GLOBAL VARIABLES
//

// ODBC environment handle
HENV env = 0;

// standard strings
char* strVersion = "tclodbc " TCLODBC_VERSION " (c) Roy Nurmi 1997";
char* strMemoryAllocationFailed = "Memory allocation failed";
char* strInvalidHandle = "Invalid handle";
char* strOK = "OK";
char* strTables = "tables";
char* strColumns = "columns";
char* strIndexes = "indexes";
char* strWrongArgs = "wrong # args";
char* strUsage = 
"Usage:\n"
"  database [connect] id datasourcename userid password\n"
"  database [connect] id connectionstring\n"
"  database configure operation driver attributelist\n"
"  database datasources\n"
"  database drivers\n"
"  database version\n";

NumStr sqlType [] = {
    // standard types
	{SQL_CHAR,         "CHAR"         },
	{SQL_NUMERIC,      "NUMERIC"      },
	{SQL_DECIMAL,      "DECIMAL"      },
	{SQL_INTEGER,      "INTEGER"      },
	{SQL_SMALLINT,     "SMALLINT"     },
	{SQL_FLOAT,        "FLOAT"        },
	{SQL_REAL,         "REAL"         },
	{SQL_DOUBLE,       "DOUBLE"       },
	{SQL_VARCHAR,      "VARCHAR"      },
    // extended types
	{SQL_DATE,         "DATE"         },
	{SQL_TIME,         "TIME"         },
	{SQL_TIMESTAMP,    "TIMESTAMP"    },
	{SQL_LONGVARCHAR,  "LONGVARCHAR"  },
	{SQL_BINARY,       "BINARY"       },
	{SQL_VARBINARY,    "VARBINARY"    },
	{SQL_LONGVARBINARY,"LONGVARBINARY"},
	{SQL_BIGINT,       "BIGINT"       },
	{SQL_TINYINT,      "TINYINT"      },
	{SQL_BIT,          "BIT"          },
	{0,                NULL           }
};

NumStr attrDef [] = {
	{SQL_COLUMN_LABEL,         "label"         },
	{SQL_COLUMN_TYPE,          "type"          },
	{SQL_COLUMN_TYPE_NAME,     "typename"      },
	{SQL_COLUMN_PRECISION,     "precision"     },
	{SQL_COLUMN_SCALE,         "scale"         },
	{SQL_COLUMN_DISPLAY_SIZE,  "displaysize"   },
	{SQL_COLUMN_NULLABLE,      "nullable"      },
	{SQL_COLUMN_UPDATABLE,     "updatable"     },
	{SQL_COLUMN_QUALIFIER_NAME,"qualifiername" },
	{SQL_COLUMN_OWNER_NAME,    "owner"         },
	{SQL_COLUMN_TABLE_NAME,    "tablename"     },
	{SQL_COLUMN_NAME,          "name"          },
	{0,                        NULL            }
};

NumStr configOp [] = {
	{ODBC_ADD_DSN,             "add_dsn"       },
	{ODBC_CONFIG_DSN,          "config_dsn"    },
	{ODBC_REMOVE_DSN,          "remove_dsn"    },
	{ODBC_ADD_SYS_DSN,         "add_sys_dsn"   },
	{ODBC_CONFIG_SYS_DSN,      "config_sys_dsn"},
	{ODBC_REMOVE_SYS_DSN,      "remove_sys_dsn"},
	{0,                        NULL            }
};

//////////////////////////////////////////////////////////////////////////
// MISANCELLOUS HELPER FUNCTIONS
//

TclObj SqlErr (HENV env, HDBC dbc, HSTMT stmt) {
    static char SqlMessage[SQL_MAX_MESSAGE_LENGTH];
    static char SqlState[6];
    SDWORD NativeError;
    SWORD Available;

    SQLError(env, dbc, stmt, 
        (UCHAR*) SqlState, &NativeError, (UCHAR*) SqlMessage, 
        SQL_MAX_MESSAGE_LENGTH-1, &Available);

    // sql error object is a triple:
    // {standard error code} {native error code} {error message}

    TclObj errObj;
    Tcl_ListObjAppendElement(NULL, errObj, TclObj(SqlState));
    Tcl_ListObjAppendElement(NULL, errObj, TclObj(NativeError));
    Tcl_ListObjAppendElement(NULL, errObj, TclObj(SqlMessage));

    return errObj;
}

short StrToNum (char *str, NumStr array[]) {

	short num;
    if (num = atoi(str)) 
        return num;

	for (int i = 0;;++i) {
        // return when a match is found or the string constant is NULL
		if (!array[i].string || !strcmp(str, array[i].string)) 
			return array[i].numeric;
	}
}

short SqlType (char *strType) {
	return StrToNum(strType, sqlType);
}

short AttrDef (char *strDef) {
	return StrToNum(strDef, attrDef);
}

short ConfigOp (char *strDef) {
	return StrToNum(strDef, configOp);
}

int StrSelect (char* str, char* strArray[])
{
    for (int i=0; strArray[i]; ++i) {
        if (!strcmp(str,strArray[i])) {
            // match found, return position
            return i;
        }
    }

    // not found
    return -1;
}

//////////////////////////////////////////////////////////////////////////
// TCL COMMAND INTERFACE
//

int tcl_database (ClientData clientData, Tcl_Interp *interp, int objc,
	Tcl_Obj *const objv[]) {
    char *name, *db, *uid, *password, *p;
    WORD operation;
    char *driver;
    TclObj attributes;
    Tcl_Obj *tmpObj;
    int i, attrcount;
    TclDatabase* pDataBase = NULL;
    int length;
    enum  OPTIONS     { CONFIGURE,  DATASOURCES,  DRIVERS,  VERSION,  CONNECT};
    char* Options[] = {"configure","datasources","drivers","version","connect",NULL};

    try {
        if (objc == 1) {
            // return usage
            Tcl_SetResult(interp, strUsage, TCL_STATIC);
            return TCL_OK;
        }

        switch (StrSelect(Tcl_GetStringFromObj(objv[1], NULL), Options)) {
        case CONFIGURE:
            if (objc != 5) 
                THROWSTR("wrong # args, should be configure operation driver attributes")
            operation = ConfigOp(Tcl_GetStringFromObj(objv[2],NULL));
            if (!operation)
                THROWSTR("invalid operation code")

            driver = Tcl_GetStringFromObj(objv[3], NULL);

            // construct attribute string, values separated by nulls, ending with
            // a double null
            if (Tcl_ListObjLength(interp, objv[4], &attrcount) != TCL_OK)
                THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
            for (i = 0; i < attrcount; ++i) {
                if (Tcl_ListObjIndex(interp, objv[4], i, &tmpObj) != TCL_OK)
                    THROWOBJ(TclObj(Tcl_GetObjResult(interp)))

	            Tcl_AppendToObj(attributes, Tcl_GetStringFromObj(tmpObj, NULL), -1);
                Tcl_AppendToObj(attributes, "\0", 1);
            }
            Tcl_AppendToObj(attributes, "\0", 1);

            if (SQLConfigDataSource(NULL, operation, driver, (char*) attributes)) {
                Tcl_SetResult(interp, strOK, TCL_STATIC);
                return TCL_OK;
            }
            else {
                THROWSTR("datasource configuration failed")
            }

        case DATASOURCES:
            // generate list of all databases
            Tcl_SetObjResult (interp, TclDatabase::Datasources());
            return TCL_OK;

        case DRIVERS:
            // generate list of all drivers available
            Tcl_SetObjResult (interp, TclDatabase::Drivers());
            return TCL_OK;

        case VERSION:
            // return version information
            Tcl_SetResult(interp, strVersion, TCL_STATIC);
            return TCL_OK;

        case CONNECT:
            --objc;
            ++objv;
            // fall through

        default:
            if (objc < 3 || objc > 5) 
                THROWSTR("wrong # args, should be database name connectionstring | (db [uid] [password])")

            name =  Tcl_GetStringFromObj(objv[1], &length);

            db = Tcl_GetStringFromObj(objv[2], &length);

			// search for '=' in dbname, indicating a odbc connection string
			for (p = db; *p != '\0' && *p != '='; ++p); 

			// connect using a connection string or a datasource name,
			if (objc == 3 && *p == '=') {
				pDataBase = new TclDatabase((char*) db);
			} else {
				uid = objc > 3 ? Tcl_GetStringFromObj(objv[3], &length) : "";
				password = objc > 4 ? Tcl_GetStringFromObj(objv[4], &length) : "";
				pDataBase = new TclDatabase(db, uid, password);
			}

            if (!pDataBase)
                THROWSTR(strMemoryAllocationFailed)

            Tcl_CreateObjCommand(interp, name, TclCmdObject::Dispatch, 
                pDataBase, TclCmdObject::Destroy);

            Tcl_SetObjResult (interp, objv[1]);
            return TCL_OK;
        }
    }
    catch( TclObj obj ) {
        if (pDataBase)
            delete pDataBase;
        Tcl_SetObjResult (interp, obj);
        return TCL_ERROR;
    }
}

//////////////////////////////////////////////////////////////////////////
// DATABASE OBJECT IMPLEMENTATION
//

void TclCmdObject::Destroy(ClientData o) {
    // virtual destructor call
    delete (TclCmdObject*) o;
}

TclCmdObject::~TclCmdObject() {
}

int TclCmdObject::Dispatch (ClientData clientData, Tcl_Interp *interp, 
                  int objc, Tcl_Obj* const objv[]) {
    TclCmdObject& o = *(TclCmdObject*) clientData;
    return o.Dispatch(interp, objc, objv);
}

TclDatabase::TclDatabase(char* db, char* uid, char* passwd) 
: dbc (SQL_NULL_HDBC) {

    RETCODE rc;

    // allocate connection handle
    rc = SQLAllocConnect(env, &dbc);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, SQL_NULL_HSTMT))

    // make connection
	rc = SQLConnect(dbc, 
        (UCHAR*) db, (SWORD) strlen(db), 
        (UCHAR*) uid, (SWORD) strlen(uid), 
        (UCHAR*) passwd, (SWORD) strlen(passwd));
    if (rc == SQL_ERROR) {
        TclObj error = SqlErr(env, dbc, SQL_NULL_HSTMT);
        SQLFreeConnect(dbc);
        THROWOBJ(error)
    }
}

TclDatabase::TclDatabase(char* connectionstring) 
: dbc (SQL_NULL_HDBC) {

    RETCODE rc;
    unsigned char szConnStrOut[256];
    SWORD pcbConnStrOut;

    // allocate connection handle
    rc = SQLAllocConnect(env, &dbc);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, SQL_NULL_HSTMT))

    // make connection
	rc = SQLDriverConnect(dbc, NULL, 
        (UCHAR*) connectionstring, (SWORD) strlen(connectionstring), 
        szConnStrOut, 255, &pcbConnStrOut, SQL_DRIVER_NOPROMPT);
    if (rc == SQL_ERROR) {
        TclObj error = SqlErr(env, dbc, SQL_NULL_HSTMT);
        SQLFreeConnect(dbc);
        THROWOBJ(error)
    }
}

TclDatabase::~TclDatabase() {
    if (dbc != SQL_NULL_HDBC) {
        SQLDisconnect(dbc);
        SQLFreeConnect(dbc);
    }
}

int TclDatabase::Dispatch (Tcl_Interp *interp, int objc, Tcl_Obj* const objv[]) {
    enum  COMMANDS     { STATEMENT,  S,  DISCONNECT,    TABLES,   COLUMNS,   INDEXES, AUTOCOMMIT,  COMMIT,  ROLLBACK};
    char* Commands[] = {"statement","s","disconnect",strTables,strColumns,strIndexes,"autocommit","commit","rollback",NULL};
    char* stmtName = NULL;
    char* stmtIniter = NULL;
    int   cmd;
    TclStatement *pStmt = NULL;

    try {
        // select command
        if (objc > 1)
            cmd = StrSelect(Tcl_GetStringFromObj(objv[1], NULL), Commands);
        else
            THROWSTR(strWrongArgs)

        // switch on command
        switch (StrSelect(Tcl_GetStringFromObj(objv[1], NULL), Commands)) {
        case STATEMENT:
        case S:
            if (objc != 4)
                THROWSTR("wrong # args, should be statement name initer")
            stmtName = Tcl_GetStringFromObj(objv[2], NULL);
            stmtIniter = Tcl_GetStringFromObj(objv[3], NULL);

            if (!strcmp(stmtIniter, strTables))
                pStmt = new TclTableQuery(dbc);
            else if (!strcmp(stmtIniter, strColumns))
                pStmt = new TclColumnQuery(dbc);
            else if (!strcmp(stmtIniter, strIndexes))
                pStmt = new TclIndexQuery(dbc);
            else
                pStmt = new TclSqlStatement(dbc, stmtIniter);

            if (!pStmt)
                THROWSTR(strMemoryAllocationFailed)

            Tcl_CreateObjCommand(interp, stmtName, TclCmdObject::Dispatch, 
                (ClientData) pStmt, TclCmdObject::Destroy);
            Tcl_SetObjResult (interp, objv[2]);
            break;

        case DISCONNECT:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            Tcl_DeleteCommand(interp, Tcl_GetStringFromObj(objv[0], NULL));
            Tcl_SetResult(interp, strOK, TCL_STATIC);
            break;

        case TABLES:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            Tcl_SetObjResult (interp, TclTableQuery(dbc).Value(interp));
            break;

        case COLUMNS:
            if (objc < 2 || objc > 3)
                THROWSTR("wrong # args, should be columns [tablename]")
            Tcl_SetObjResult (interp, TclColumnQuery(dbc).Value(interp, objc-2, objv+2));
            break;

        case INDEXES:
            if (objc != 3)
                THROWSTR("wrong # args, should be indexes tablename")
            Tcl_SetObjResult (interp, TclIndexQuery(dbc).Value(interp, objc-2, objv+2));
            break;

        case AUTOCOMMIT:
            if (objc != 3)
                THROWSTR("wrong # args, should be autocommit on|off")
            else if (!strcmp("on", Tcl_GetStringFromObj(objv[2], NULL)))
                Autocommit(TRUE);
            else if (!strcmp("off", Tcl_GetStringFromObj(objv[2], NULL)))
                Autocommit(FALSE);
            else
                THROWSTR("invalid state, should be on or off")
            Tcl_SetResult(interp, strOK, TCL_STATIC);
            break;

        case COMMIT:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            Transact(SQL_COMMIT);
            Tcl_SetResult(interp, strOK, TCL_STATIC);
           break;

        case ROLLBACK:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            Transact(SQL_ROLLBACK);
            Tcl_SetResult(interp, strOK, TCL_STATIC);
            break;

        default:
            if (objc < 2 || objc > 4)
                THROWSTR("wrong # args, should be sql [typedefs] [args]")
            Tcl_SetObjResult (interp, TclSqlStatement(dbc, Tcl_GetStringFromObj(objv[1], NULL))
                .Value(interp, objc-2, objv+2));
            break;
        }

        // command successful
        return TCL_OK;
    }
    catch (TclObj obj) {
        if (pStmt)
            delete pStmt;
        Tcl_SetObjResult (interp, obj);
        return TCL_ERROR;
    }
}

void TclDatabase::Autocommit(BOOL on) {
    RETCODE rc;
    UDWORD commit;

    commit = on ? SQL_AUTOCOMMIT_ON : SQL_AUTOCOMMIT_OFF;

    rc = SQLSetConnectOption(dbc, SQL_AUTOCOMMIT, commit);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, dbc, SQL_NULL_HSTMT))
}


void TclDatabase::Transact(UWORD op) {
    RETCODE rc;

    rc = SQLTransact(env, dbc, op);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, dbc, SQL_NULL_HSTMT))
    else if (rc == SQL_INVALID_HANDLE)
        THROWSTR(strInvalidHandle)
}

TclObj TclDatabase::Datasources() {

    RETCODE rc = 0;
    TclObj list;
    char dsn[SQL_MAX_DSN_LENGTH+1], descr[256];
    SWORD dsnLen, descrLen; 
    BOOL first = TRUE;

    while ((rc = SQLDataSources(env, 
        first ? SQL_FETCH_FIRST : SQL_FETCH_NEXT, 
        (UCHAR*)dsn, SQL_MAX_DSN_LENGTH+1, &dsnLen, 
        (UCHAR*)descr, 255, &descrLen)) == SQL_SUCCESS) 
    {
        TclObj item;
        TclObj dnsObj (dsn, dsnLen);
        TclObj descrObj (descr, descrLen);
        Tcl_ListObjAppendElement(NULL, item, dnsObj);
        Tcl_ListObjAppendElement(NULL, item, descrObj);
        Tcl_ListObjAppendElement(NULL, list, item);
        first = FALSE;
    }

    // success
    return list;
}

TclObj TclDatabase::Drivers() {

    RETCODE rc = 0;
    TclObj list;
    char driver[256], attrs[1024];
	char *attr;
    SWORD driverLen, attrsLen; 
    BOOL first = TRUE;

    while ((rc = SQLDrivers(env, 
        first ? SQL_FETCH_FIRST : SQL_FETCH_NEXT, 
        (UCHAR*)driver, 255, &driverLen, 
        (UCHAR*)attrs, 1023, &attrsLen)) == SQL_SUCCESS) 
    {
        TclObj item;
        TclObj driverObj (driver, driverLen);
        TclObj attrsObj;

		// loop over all attribute strings, list terminates with
		// double-null.
		for (attr = attrs; *attr; attr += strlen(attr)+1) {
			TclObj attrObj (attr);
			Tcl_ListObjAppendElement(NULL, attrsObj, attrObj);
		}

        Tcl_ListObjAppendElement(NULL, item, driverObj);
        Tcl_ListObjAppendElement(NULL, item, attrsObj);
        Tcl_ListObjAppendElement(NULL, list, item);
        first = FALSE;
    }

    // success
    return list;
}

TclStatement::TclStatement(HDBC dbc) 
: stmt(SQL_NULL_HSTMT), resultBuffer(NULL), colCount(-1) {
    RETCODE rc;

    rc = SQLAllocStmt(dbc, &stmt);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, dbc, stmt))
    else if (rc == SQL_INVALID_HANDLE) 
        THROWSTR(strInvalidHandle)
}

TclStatement::~TclStatement() {
    FreeResultBuffer();
    if (stmt != SQL_NULL_HSTMT)
        SQLFreeStmt(stmt, SQL_DROP);
}

int TclStatement::Dispatch(Tcl_Interp *interp, int objc, Tcl_Obj* const objv[]) {
    enum  COMMANDS     { DROP,  COLUMNS,  EXECUTE,  FETCH};
    char* Commands[] = {"drop","columns","execute","fetch",NULL};
    int   cmd;

    try {
        // select command
        if (objc > 1)
            cmd = StrSelect(Tcl_GetStringFromObj(objv[1], NULL), Commands);
        else
            cmd = -1;

        // switch on command
        switch (cmd) {
        case DROP:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            Tcl_DeleteCommand(interp, Tcl_GetStringFromObj(objv[0], NULL));
            Tcl_SetResult(interp, strOK, TCL_STATIC);
            break;

        case COLUMNS:
            if (objc < 2)
                THROWSTR(strWrongArgs)

			if (objc == 2) {
				// no arguments, create default argument
				Tcl_Obj* objv [1]; 
				TclObj label ("label");
				objv [0] = label;
				Tcl_SetObjResult (interp, Columns (1, objv));
			}
			else
				Tcl_SetObjResult (interp, Columns(objc-2, objv+2));
            break;

        case EXECUTE:
            if (objc < 2 || objc > 4)
                THROWSTR("wrong # args, should be execute [typedefs] [args]")
            Execute(interp, objc-2, objv+2);
            Tcl_SetResult(interp, strOK, TCL_STATIC);
            break;

        case FETCH:
            if (objc != 2)
                THROWSTR(strWrongArgs)
            {
                TclObj row;
                Fetch(row);
                Tcl_SetObjResult (interp, row);
            }
            break;

        default:
            Tcl_SetObjResult (interp, Value(interp, objc-1, objv+1));
            break;
        }

        // command successful
        return TCL_OK;
    }
    catch (TclObj obj) {
        Tcl_SetObjResult (interp, obj);
        return TCL_ERROR;
    }
}

void TclStatement::ReserveResultBuffer() {
    RETCODE rc;
    SDWORD colLen;
    static SDWORD cbValue;
    UWORD i;

    // allocate space for column array
    resultBuffer = (char**) Tcl_Alloc(ColumnCount()*sizeof(char*));
    if (!resultBuffer)
        THROWSTR(strMemoryAllocationFailed)
    memset (resultBuffer, 0, ColumnCount()*sizeof(char*));

    // allocate space for columns and bind them
    for (i=0; i<ColumnCount(); ++i) {
        // find out the max column length
        rc = SQLColAttributes(stmt, (UWORD)(i+1), SQL_COLUMN_DISPLAY_SIZE, 
            NULL, 0, NULL, &colLen);
        if (rc == SQL_ERROR) {
            THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
        }
        // allocate space
        if (colLen == SQL_NO_TOTAL || colLen > MAX_BLOB_COLUMN_LENGTH)
            colLen = MAX_BLOB_COLUMN_LENGTH;
        else
            colLen = colLen + 1; // +1 to include '\0'

        resultBuffer[i] = Tcl_Alloc(colLen*sizeof(char));
        if (!resultBuffer[i])
            THROWSTR(strMemoryAllocationFailed)
        memset (resultBuffer[i], 0, colLen*sizeof(char));
        // bind
        rc = SQLBindCol(stmt, (UWORD)(i+1), SQL_C_CHAR, resultBuffer[i], colLen, &cbValue);
        if (rc == SQL_ERROR) {
            THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
        }
    }
}

void TclStatement::FreeResultBuffer() {
    if (resultBuffer) {
        for (int i=0; i<ColumnCount(); ++i)
            if (resultBuffer[i])
                Tcl_Free(resultBuffer[i]);
        Tcl_Free((char*)resultBuffer);
        resultBuffer = NULL;
        colCount = -1;
    }
}


int TclStatement::ColumnCount() {
    RETCODE rc;

    // the column count of a single statement is always the same,
    // and it is enough to check it once
    if (colCount == -1) {
        SWORD tmp;
        rc = SQLNumResultCols(stmt, &tmp);
        if (rc == SQL_ERROR) {
            THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
        }
        colCount = tmp;
    }
    return colCount;
}

TclObj TclStatement::Result() {
    // if sql has result set, return it, otherways simply OK
    if (ColumnCount() > 0) {
        //fetch rows
        TclObj result;
        TclObj row;
        while (Fetch(row)) {
            Tcl_ListObjAppendElement(NULL, result, row);
        }
        return result;
    } else {
        return TclObj(strOK);
    }
}

TclObj TclStatement::Value(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
    // first execute
    Execute(interp, objc, objv);

    // return result in result
    return Result();
}

TclObj TclStatement::Columns(int objc, Tcl_Obj *const objv[]) {
    int i, arg;
    char   strData [256];
	SDWORD wordData;
    TclObj result;

	for (i = 1; i <= ColumnCount(); ++i) {
		// read column data for all specified argument
		TclObj element;
	    for (arg = 0; arg < objc; ++ arg) {
			UWORD attr = AttrDef(Tcl_GetStringFromObj(objv[arg],NULL));

			switch (attr) {
			case SQL_COLUMN_LABEL:
			case SQL_COLUMN_TYPE_NAME:
			case SQL_COLUMN_TABLE_NAME:
			case SQL_COLUMN_NAME:
			case SQL_COLUMN_OWNER_NAME:
			case SQL_COLUMN_QUALIFIER_NAME:
				if (SQLColAttributes(stmt, i, attr, 
					(UCHAR*)strData, 256, NULL, NULL) == SQL_ERROR)
					THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
				Tcl_ListObjAppendElement(NULL, element, TclObj(strData));
				break;

			case SQL_COLUMN_DISPLAY_SIZE:
			case SQL_COLUMN_TYPE:
			case SQL_COLUMN_PRECISION:
			case SQL_COLUMN_SCALE:
            case SQL_COLUMN_NULLABLE:
            case SQL_COLUMN_UPDATABLE:
				if (SQLColAttributes(stmt, i, attr, 
					NULL, 0, NULL, &wordData) == SQL_ERROR)
					THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
				Tcl_ListObjAppendElement(NULL, element, TclObj(wordData));
				break;

			default: 
				THROWSTR("Invalid column data definition")
				break;
			};
		}
		Tcl_ListObjAppendElement(NULL, result, element);
	}
    return result;
}

BOOL TclStatement::Fetch(TclObj& row) {
    RETCODE rc;
    UWORD i;

    // always clear old data first
    row = TclObj(); 

    //result buffer is reserved here, always only once per statement
    if (!resultBuffer && ColumnCount() > 0)
        ReserveResultBuffer();

    //fetch row
    rc = SQLFetch(stmt);
    switch (rc) {
    case SQL_NO_DATA_FOUND:
        return FALSE; 
    case SQL_ERROR:
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
    case SQL_INVALID_HANDLE:
        THROWSTR(strInvalidHandle)
    }

	for (i=0; i<ColumnCount(); ++i) {
        TclObj element (resultBuffer[i]);
        Tcl_ListObjAppendElement(NULL, row, element);
    }

    // success
    return TRUE;
}

TclSqlStatement::TclSqlStatement(HDBC dbc, char* sql) : TclStatement (dbc) {
    RETCODE rc;

    // prepare statement
    rc = SQLPrepare(stmt, (UCHAR*) sql, strlen(sql));
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, dbc, stmt))
}

void TclSqlStatement::Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
    RETCODE rc;
    int argc;
    BOOL hasArguments = FALSE;
    BOOL hasArgTypeDefs = FALSE;

    // syntax check
    switch (objc) {
    case 2:
        hasArgTypeDefs = TRUE;
    case 1:
        hasArguments = TRUE;
    case 0:
        break;
    default:
        THROWSTR("Invalid arguments, should be stmt [argtypelist] [arglist]")
    }

    if (hasArguments) { 
        // bind optional sql variables
        // sql argument count = count of items in the last argument
        if (Tcl_ListObjLength(interp, objv[objc-1], &argc) != TCL_OK)
            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))

        for (int i = 0; i < argc; ++i) {
            SDWORD cbName = SQL_NTS;
            int subc, sqlarglen;
            char* sqlArg;
            SWORD sqlType = SQL_VARCHAR;
            UDWORD cbColDef;
            SWORD ibScale = 0;
            Tcl_Obj *tmpObj, *typeDefObj;
            int tmpInt;
            int l;

            // actual argument
            if (Tcl_ListObjIndex(interp, objv[objc-1], i, &tmpObj) != TCL_OK)
                THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
            sqlArg = Tcl_GetStringFromObj(tmpObj, &sqlarglen); // last listitem
            cbColDef = sqlarglen;
            if (sqlarglen == 0)
                cbName = SQL_NULL_DATA;

            if (hasArgTypeDefs) {
                // corresponding dype definition
                if (Tcl_ListObjIndex(interp, objv[0], i, &typeDefObj) != TCL_OK)
                    THROWOBJ(TclObj(Tcl_GetObjResult(interp)))

                // typedef count may be less than actual arguments
                if (typeDefObj) {
                    // argument definition list length
                    if (Tcl_ListObjLength(interp, typeDefObj, &subc) != TCL_OK)
                        THROWOBJ(TclObj(Tcl_GetObjResult(interp)))

                    // type definition parts: {type lenght scale}
                    switch (subc) {
                    case 3:
                        if (Tcl_ListObjIndex(interp, typeDefObj, 2, &tmpObj) != TCL_OK)
                            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
                        if (Tcl_GetIntFromObj(interp, tmpObj, &tmpInt) != TCL_OK)
                            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
                        ibScale = (SWORD) tmpInt;
                    case 2:
                        if (Tcl_ListObjIndex(interp, typeDefObj, 1, &tmpObj) != TCL_OK)
                            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
                        if (Tcl_GetIntFromObj(interp, tmpObj, &tmpInt) != TCL_OK)
                            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
                        cbColDef = tmpInt;
                    case 1:
                        if (Tcl_ListObjIndex(interp, typeDefObj, 0, &tmpObj) != TCL_OK)
                            THROWOBJ(TclObj(Tcl_GetObjResult(interp)))
                        sqlType = SqlType(Tcl_GetStringFromObj(tmpObj, &l));
                    }
                }
            }

            rc = SQLBindParameter(stmt, (UWORD)(i+1), SQL_PARAM_INPUT, 
                    SQL_C_CHAR, sqlType, cbColDef, ibScale, sqlArg, 
                    sqlarglen+1, &cbName);
            if (rc == SQL_ERROR) 
                THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
            else if (rc == SQL_INVALID_HANDLE) 
                THROWSTR(strInvalidHandle)
        }
    }

    // execute sql, always close previous cursor if necessary
    rc = SQLFreeStmt(stmt, SQL_CLOSE);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))

    rc = SQLExecute(stmt);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))

}

TclTableQuery::TclTableQuery(HDBC dbc) : TclStatement(dbc) 
{}

void TclTableQuery::Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
    RETCODE rc;

    // close previous cursor
    rc = SQLFreeStmt(stmt, SQL_CLOSE);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))

    // read table information
    rc = SQLTables(stmt, NULL, 0, NULL, 0, NULL, 0, NULL, 0);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
}

TclColumnQuery::TclColumnQuery(HDBC dbc) : TclStatement(dbc) 
{}

void TclColumnQuery::Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
    RETCODE rc;
    int l;
    char* tableName = NULL;

    if (objc)
        tableName = Tcl_GetStringFromObj(objv[0], &l);

    // close previous cursor
    rc = SQLFreeStmt(stmt, SQL_CLOSE);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))

    // read column information
    rc = SQLColumns(stmt, NULL, 0, NULL, 0, (UCHAR*) tableName, 
        (SWORD) (tableName ? l : 0), NULL, 0);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
}


TclIndexQuery::TclIndexQuery(HDBC dbc) : TclStatement(dbc) 
{}

void TclIndexQuery::Execute(Tcl_Interp *interp, int objc, Tcl_Obj *const objv[]) {
    RETCODE rc;
    int l;
    char* tableName;

    if (objc != 1)
        THROWSTR("wrong # args, tablename is required")

    tableName = Tcl_GetStringFromObj(objv[0], &l);

    // close previous cursor
    rc = SQLFreeStmt(stmt, SQL_CLOSE);
    if (rc == SQL_ERROR)
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))

    // read index information
    rc = SQLStatistics(stmt, NULL, 0, NULL, 0, (UCHAR*) tableName,
        l, SQL_INDEX_ALL, SQL_ENSURE);
    if (rc == SQL_ERROR) 
        THROWOBJ(SqlErr(env, SQL_NULL_HDBC, stmt))
}

//////////////////////////////////////////////////////////////////////////
// TCL EXTENSION CLEANUP ROUTINE
//

void Tclodbc_Kill(ClientData clientData)
{
	SQLFreeEnv(env);
}


//////////////////////////////////////////////////////////////////////////
// TCL EXTENSION INITIALIZATION ROUTINE
//

extern "C" {
_declspec(dllexport)
Tclodbc_Init(Tcl_Interp *interp) 
{
	// allocate environment and create exit handler for deleting it on exit
    if (!env && SQLAllocEnv(&env) == SQL_ERROR) {
		if (env == SQL_NULL_HENV) 
            Tcl_SetResult(interp, strMemoryAllocationFailed, TCL_STATIC);
		else 
			Tcl_SetObjResult (interp, SqlErr(env, SQL_NULL_HDBC, SQL_NULL_HSTMT));
        return TCL_ERROR;

		Tcl_CreateExitHandler(Tclodbc_Kill, (ClientData) 0);
	}

    // create commands
    Tcl_CreateObjCommand(interp, "database", tcl_database, NULL, 
        (Tcl_CmdDeleteProc *) NULL);

    // provide package information
    Tcl_PkgProvide(interp, "tclodbc", TCLODBC_VERSION);
    return TCL_OK ;
}

} // extern "C"
