###############################################################################
###############################################################################
#####                          Ccombinado.tcl
###############################################################################
# This file contains a combobox type widget
###############################################################################
# I don't know where this widget comes from originally, someone in college
# 'Jess' used to give it, I think he had adapted it and I have changed it
# further. Anyway until someone tells me differently:
# Copyright 1999-2000 Jess, Andrs Garca. Distributed under the terms of the
# GPL. andresgarci@retemail.es
###############################################################################

namespace eval Ccombinado {

###############################################################################
# SetEntryText
#    Sets the selected value in the entry widget, after deleting the former
#    one.
###############################################################################
proc SetEntryText {v entry} {

    if { [set idx [$v curselection ]]!="" } {
        $entry delete 0 end
        $entry insert 0 [$v get $idx]
    }
    return
}

###############################################################################
# UnMapList
#    Erases the window used by the list.
###############################################################################
proc UnMapList { w } {
    grab release $w
    wm withdraw [winfo toplevel $w]

    return
}

###############################################################################
# MapList
#    Maps the window with the list
#
# Parameter
#    Window name
###############################################################################
proc MapList {w} {

    set d ${w}_desplegable
    set l $d.listado

    if {[winfo ismapped $d]} {
        grab release $d
        wm withdraw $d
    } else {
        set x1 [expr [winfo rootx ${w}.fondo.boton]-\
                [winfo reqwidth $l.cuadro]]
        set y1 [expr [winfo rooty ${w}.fondo.boton]+\
                [winfo height ${w}.fondo.boton]]
        wm geometry $d +$x1+$y1
        wm deiconify $d
        raise $d
        focus $l.cuadro
        wm geometry $d +$x1+$y1
        update
        grab -global $d
    }
    return
}

###############################################################################
# Selected
#    If you click on a list element, this procedures puts the selected
#    value in the entry widget, and closes the window list
###############################################################################
proc Selected { w y } {

    set d ${w}_desplegable
    set l $d.listado

    grab release $d

    $w.entry delete 0 end
    $w.entry insert 0 [$l.cuadro get [$l.cuadro nearest $y] ]

    focus $w.entry

    UnMapList $l

    return
}

###############################################################################
# KeySelected
#    Invoked if you use return to select an element, it writes it in the
#    entry widget and then closes the list window.
#
# Parameters
#    w: Path of the widget
#    which: '0' if the key pressed was 'Escape'
#           '1' if it was 'Return'
###############################################################################
proc KeySelected { w which } {

    set d ${w}_desplegable
    set l $d.listado

    if { $which==1 } {
        SetEntryText $l.cuadro $w.entry
    }

    focus $w.entry
    UnMapList $d

    return
}

###############################################################################
# DeleteItem
#    This procedure is invoked when the user presses the 'supr' button while
#    an element of the listbox is selected, as you have probably guessed, the
#    element gets deleted.
#
# Parameters:
#    w: path of the combobox
#    y: 'y' coordinate of the point in the listbox where the user clicked
###############################################################################
proc DeleteItem {w y} {
    variable cbArgs

    if {$cbArgs(-erasable)==0} {
        return
    }

    set d ${w}_desplegable
    set l $d.listado

    set index [$l.cuadro curselection]
    if {$index!=""} {
        $l.cuadro delete $index
        $l.cuadro configure -height [expr [$l.cuadro cget -height] - 1]
        if {$index<[expr [llength $cbArgs(items)]-1]} {
            $l.cuadro selection set $index
        } else {
            $l.cuadro selection set [expr $index -1]
        }
        set cbArgs(items) [lreplace $cbArgs(items) $index $index]
    }

    return
}

###############################################################################
# UnMapScrollBar
#    Erases the scroll bar in the window list in case it is not needed.
###############################################################################
proc UnMapScrollBar { listbox scrollbar } {

    set items [$listbox index end]
    set size  [$listbox cget -height]
    if {$items <= $size} {
        pack forget $scrollbar
        $listbox configure -height $items
    } else {
        pack $scrollbar -side right -fill y
    }

    return
}

###############################################################################
# ParseArguments
#    Gets the optional parameters passed to ComboBox into the
#    namespace variable 'cbArgs', the rest get the default values.
###############################################################################
proc ParseArguments {parameters} {
    variable cbArgs

    upvar $parameters args

    set cbArgs(-width)    50
    set cbArgs(-default)  ""
    set cbArgs(-erasable)  0

    array set cbArgs $args

    return
}

###############################################################################
# ComboBox
#    This procedure creates the combobox.
#
# Parameters:
#    w: full path to name the combobox.
#    items: list with the items to be displayed.
#    args: list with the optional parameters:
#           -default: the value that will appear preselected in the entry.
#           -width: width of the entry, defaults to 50
#           -erasable: '1' if the user can erase items from the listbox,
#            defaults to '0'.
###############################################################################
proc ComboBox { w items args} {
    variable cbArgs

    frame $w -relief sunken -bd 1
    array set cbArgs $args
    ParseArguments args

    set cbArgs(items) $items
    if {[info comm flecha] == {}} {
        set flecha {
            #define flecha_width 15
            #define flecha_height 15
                static char flecha_text_bits[] = {
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0xf8, 0x03, 0xf0, 0x01, 0xe0, 0x00, 0x40, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00
                }
        }
        image create bitmap flecha -data [set flecha]
        unset flecha
    }
# Entry box
    entry $w.entry -bd 1 -relief flat -width $cbArgs(-width) -bg white
    frame $w.fondo -bg gray
    $w.entry  insert 0 $cbArgs(-default)
    $w.entry  selection range 0 end	

# Button
    button $w.fondo.boton -image flecha -relief raised \
        -command [namespace code "MapList $w"] -width 8 -height 10
    pack $w.entry -side left -pady 1 -padx 1 -fill x
    pack $w.fondo -side right
    pack $w.fondo.boton -side bottom -pady 1

# Toplevel window with the list
    set d ${w}_desplegable
    toplevel $d -relief raised -bd 1 -bg White
    wm overrideredirect $d 1
    wm transient $d
    wm withdraw $d

# Frame for the listbox '$l'
    set l $d.listado
    frame $l
    pack $l -expand yes -fill y

# Scrollbar for the listbox '$l.barra'
    scrollbar $l.barra -bd 1 -command "$l.cuadro yview"

# Finally the listbox '$l.cuadro'
    listbox $l.cuadro -yscroll "$l.barra set" -setgrid 1 -bg White \
        -relief sunken -width $cbArgs(-width)
    pack $l.barra -side right -fill y
    pack $l.cuadro -side left -expand 1 -fill both

# 'Etiqueta' is used so all the elementos of the list can be binded
# in one go.
    set etiqueta [winfo name $w]_listado
    foreach q "$d $l $l.cuadro $l.barra" {
         bindtags $q [concat ${etiqueta} [bindtags $q]]
    }
    set numero [llength $cbArgs(items)]
    if {$cbArgs(items) != ""} {
        $l.cuadro delete 0 end
        foreach q $cbArgs(items) {
            $l.cuadro insert end $q
            if { $q==$cbArgs(-default) } {
                set i [lsearch $cbArgs(items) $q]
                $l.cuadro yview moveto [expr (double($i))/$numero]
                tkListboxMotion $l.cuadro \ $i
            }
        }
    }
    bind $l.cuadro <Motion> "
        tkListboxMotion %W \[%W index @%x,%y]
    "
    bind $l.cuadro <Enter> {
        tkCancelRepeat
    }
    bind $etiqueta <ButtonPress>  [namespace code {
        foreach q {rootx rooty width height} {
            set $q [winfo $q %W]
        }
        if {(%X < $rootx) || (%X > ($rootx+$width)) || \
                 (%Y < $rooty) || (($rooty+$height) < %Y)} {
            UnMapList %W
        }
    }]

    bind $etiqueta  <Return>   [namespace code "KeySelected [list $w] 1"]
    bind $etiqueta  <KP_Enter> [namespace code "KeySelected [list $w] 1"]
    bind $etiqueta  <Escape>   [namespace code "KeySelected [list $w] 0"]
    bind $l.cuadro  <1>        [namespace code "Selected    [list $w] %y"]
    bind $l.cuadro  <Delete>   [namespace code "DeleteItem  [list $w] %y"]
    bind ${w}.entry <Down>     [namespace code "MapList ${w}"]
    bind $l.cuadro <Configure> "
        [namespace code "UnMapScrollBar $l.cuadro $l.barra"]
    "

    return
}

}