# This file contains code for doing NNTP (network news) transactions.

namespace eval NNTP {
    # These are arrays for keeping track of information related
    # to NNTP connections. They are indexed by a "connection
    # handle", which is a unique string associated with each
    # NNTP connection. "nntpSocket" contains the name of the
    # socket used for communication with the server.
    # "nntpMsgHandler" contains the name of a procedure that
    # is passed the server's reply messages. "serverAllowsPosting"
    # contains a Boolean indicating whether the server allows
    # clisents to post messages to it.

    variable nntpSocket
    variable nntpMsgHandler
    variable serverAllowsPosting

    namespace export Open SetMsgHandler GetMsgHandler ResponseCode Close
    namespace export PostingAllowed GetGroupList GetNewGroups SetGroup
    namespace export GetHeader GetBody GetNewNews SetArticlePointer
    namespace export NextArticle PrevArticle Post ClientIsSlave IHave

    # "ResponseCode" extracts the numeric response code from
    # an NNTP reponse message.

    proc ResponseCode {msg} {
	set msg [string trimleft $msg]
	return [lindex [split $msg " "] 0]
    }

    # "Open" tries to open a news connection to host "host".
    # If it succeeds, it returns an "nntp connection handle",
    # which is a unique ID associated with the connection.
    # If it fails, it returns an error message, and returns an
    # error code of 1.
    #
    # Allowed options are:
    #
    #	-port <portNo>		TCP port to connect to; defaults to 119
    #
    #	-messagehandler <hnd>	Message handler routine; defaults to
    #				the empty string, which means do nothing
    #				with the NNTP reply messages.
    #
    # The message handler should be a proc that accepts one argument,
    # a response message that begins with a numeric response code.

    proc Open {host args} {
	variable nntpSocket
	variable nntpMsgHandler
	variable serverAllowsPosting

	# Set defaults for optional arguments, and parse the options.

	set opts(-port) 119
	set opts(-messagehandler) {}

	array set opts $args

	# Generate a unique handle name.

	set handleNum 0
	while {1} {
	    set nntpHandle "nntphandle$handleNum"
	    if {![info exists nntpSocket($nntpHandle)]} {
		break
	    }
	    incr handleNum
	}

	# Try to open the NNTP connection.

	set ecode [catch [list socket $host $opts(-port)] sock]

	if {$ecode != 0} {
	    return -code error "Couldn't make news connection: $sock"
	}

	# Got it! Save the socket, and set the message handling routine.

	set nntpSocket($nntpHandle) $sock
	set msgHandler $opts(-messagehandler)
	set nntpMsgHandler($nntpHandle) $msgHandler

	# Configure the socket so that it trims off the trailing CRLF.

	fconfigure $sock -translation crlf

	# Get the server's response string.

	set nChars [gets $sock response]

	if {$nChars < 0} {
	    catch [list close $sock]
	    unset nntpSocket($nntpHandle)
	    unset nntpMsgHandler($nntpHandle)
	    return -code error "End-of-file on socket"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Note whether the server allows posting.

	set resp [ResponseCode $response]
	set serverAllowsPosting($nntpHandle) [expr {$resp == 200}]

	# Return the connection handle.

	return $nntpHandle
    }

    # PostingAllowed returns a Boolean value indicating whether
    # the server connected to "nntpHandle" allows clients to
    # post messages to it.

    proc PostingAllowed {nntpHandle} {
	variable serverAllowsPosting
    }

    # "SetMsgHandler" sets the message handler procedure for
    # "nntpHandle" to "msgHandlerProc". "msgHandlerProc" should
    # accept exactly one argument (the server's response message).

    proc SetMsgHandler {nntpHandle msgHandlerProc} {
	variable nntpMsgHandler

	set nntpMsgHandler($nntpHandle) $msgHandlerProc
    }

    # "GetMsgHandler" retrieves the message
    # handler procedure for "nntpHandle".

    proc GetMsgHandler {nntpHandle} {
	variable nntpMsgHandler

	return $nntpMsgHandler($nntpHandle)
    }

    # "Close" closes the connection associated with "nntpHandle",
    # and does all necessary cleanup.

    proc Close {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler
	variable serverAllowsPosting

	# Try to send a QUIT command; then close the socket, and clean up.

	set sock $nntpSocket($nntpHandle)
	puts $sock "QUIT"
	flush $sock
	close $sock
	unset nntpSocket($nntpHandle)
	catch [list unset nntpMsgHandler($nntpHandle)]
	catch [list unset serverAllowsPosting($nntpHandle)]
    }

    # ReadGroupList reads a newsgroup list from the server. It assumes
    # that all the preliminary setup has been done, and that the
    # client only needs to read the newsgroup list and return it.

    proc ReadGroupList {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	set newsgroupList {}
	while {1} {
	    # Get a line of text; we're done if it consists of a single ".".

	    set nchars [gets $sock line]
	    if {$nchars < 0} {
		catch [list Close $nntpHandle]
		return -code error "Server disconnected prematurely"
	    }

	    if {[string compare $line "."] == 0} {
		return $newsgroupList
	    }

	    # If the first char is a ".", then trim it off, since
	    # an initial "." will be doubled.

	    set firstChar [string index $line 0]
	    if {[string compare $firstChar "."] == 0} {
		set line [string range $line 1 end]
	    }

	    # Extract the newsgroup name, first article number,
	    # last article number, and the post status.
	    # NOTE: we really shouldn't use "lindex" to do this,
	    # but this quick-and-dirty technique does seem to work...

	    set groupName [lindex $line 0]
	    scan [lindex $line 1] "%d" firstArticle
	    scan [lindex $line 2] "%d" lastArticle
	    set postStatus [lindex $line 3]
	    lappend newsgroupList $groupName \
			$firstArticle $lastArticle $postStatus
	}
    }

    # "GetGroupList" retrieves a list of newsgroup names from the
    # server associated with "nntpHandle".

    proc GetGroupList {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	puts $sock "LIST"
	flush $sock

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 215} {
	    return -code error "Newsgroups not retrieved"
	}

	return [ReadGroupList $nntpHandle]
    }

    # "GetNewGroups" retrieves a list of newsgroup names from the
    # server associated with "nntpHandle". It onlt retrieves those
    # groups that are new relative to "datime", where "datime"
    # is the date and time, specified as an integer in the format
    # returned by the "clock scan" command. The "-gmt" option is followed
    # by a Boolean indicating whether "datime" is expressed in Greenwich
    # Mean Time; it defaults to 0. The "-distlist" option is followed
    # by a Tcl list of "distributions", i.e. top-level newsgroup names
    # like "net", "alt", "comp", etc.

    proc GetNewGroups {nntpHandle datime args} {
	variable nntpSocket
	variable nntpMsgHandler

	set opts(-gmt) 0
	set opts(-distlist) {}

	array set opts $args

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Format the date/time properly, and put it
	# into the NEWGROUPS command line.

	set datimeFmt  [clock format $datime \
				-format "%y%m%d %H%M%S" -gmt $opts(-gmt)]
	set cmd "NEWGROUPS $datimeFmt"
	if {$opts(-gmt)} {
	    append cmd " GMT"
	}

	# Format the distribution list properly.

	if {[llength $opts(-distlist)] != 0} {
	    set fmtDistrib [join $opts(-distlist) ","]
	    append cmd " <$fmtDistrib>"
	}

	# Send the command to the server.

	puts $sock $cmd
	flush $sock

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 231} {
	    return -code error "Newsgroups not retrieved"
	}

	return [ReadGroupList $nntpHandle]
    }

    # "SetGroup" sets the newsgroup for the server associated with
    # "nntpHandle" to "groupName". If successful, it returns a
    # list containing three elements: the newsgroup name, the
    # number of the first available article in the newsgroup, and
    # the number of the last available article.

    proc SetGroup {nntpHandle groupName} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "GROUP" command, and get a response.

	puts $sock "GROUP $groupName"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 211} {
	    return -code error "Error setting newsgroup: $response"
	}

	# We're OK: return the info.

	set groupName [lindex $response 4]
	set firstArticleNo [lindex $response 2]
	set lastArticleNo [lindex $response 3]
	return [list $groupName $firstArticleNo $lastArticleNo]
    }

    # "ReadMessageText" reads a message from "nntpHandle".

    proc ReadMessageText {nntpHandle} {
	variable nntpSocket

	set sock $nntpSocket($nntpHandle)
	set text ""
	while {1} {
	    # Get a line of text; we're done if it consists of a single ".".

	    set nchars [gets $sock line]
	    if {$nchars < 0} {
		catch [list Close $nntpHandle]
		return -code error "Server disconnected prematurely"
	    }

	    if {[string compare $line "."] == 0} {
		return $text
	    }

	    # If the first char is a ".", then trim it off, since
	    # an initial "." will be doubled.

	    set firstChar [string index $line 0]
	    if {[string compare $firstChar "."] == 0} {
		set line [string range $line 1 end]
	    }

	    # Add the line...

	    append text $line "\n"
	}
    }

    # "GetHeader" reads the header for nntpHandle's currently-
    # selected article.

    proc GetHeader {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "HEAD" command, and get a response.

	puts $sock "HEAD"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 221} {
	    return -code error "Error reading article header: $response"
	}

	# Return the header text.

	return [ReadMessageText $nntpHandle]
    }

    # "GetBody" retrieves the body of nntpHandle's currently-
    # selected article.

    proc GetBody {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "BODY" command, and get a response.

	puts $sock "BODY"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 222} {
	    return -code error "Error reading article text: $response"
	}

	# Return the message body.

	return [ReadMessageText $nntpHandle]
    }

    # GetNewNews retrieves "new" news from the server represented
    # by "nntpHandle". An article is "new" if it was posted after
    # the date/time represented by "datime". "groupList" is a list
    # of newsgroups that we're interested in. The groups are specified
    # as described in RFC 977, i.e. they may contain "*" wildcards,
    # and may be preceded by "!" to indicate that groups matching
    # the given pattern are to be excluded. The "-gmt" option
    # must be followed by a Boolean value, indicating whether the
    # date/time is to be interpreted as Greenwich Mean Time. The
    # "-grouplist" argument must be followed by a Tcl list of
    # newsgroup specifiers; this is like the "-distlist" argument
    # of the GetNewGroups command.

    proc GetNewNews {nntpHandle groupList datime args} {
	variable nntpSocket
	variable nntpMsgHandler

	set opts(-gmt) 0
	set opts(-grouplist) {}

	array set opts $args

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Format the date/time properly, and put it
	# into the NEWNEWS command line.

	set datimeFmt  [clock format $datime \
				-format "%y%m%d %H%M%S" -gmt $opts(-gmt)]
	set cmd "NEWNEWS [join $groupList ","] $datimeFmt"
	if {$opts(-gmt)} {
	    append cmd " GMT"
	}

	# Format the group list properly.

	if {[llength $opts(-grouplist)] != 0} {
	    set fmtGroups [join $opts(-grouplist) ","]
	    append cmd " <$fmtGroups>"
	}

	# Send the command to the server.

	puts $sock $cmd
	flush $sock

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 230} {
	    return -code error "Newsgroups not retrieved"
	}

	return [split [ReadMessageText $nntpHandle] "\n"]
    }

    # "SetArticlePointer" sets the current article pointer for
    # "nntpHandle" to "articleNumber".

    proc SetArticlePointer {nntpHandle articleNumber} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "STAT" command, and get a response.

	puts $sock "STAT $articleNumber"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 223} {
	    return -code error "Error setting article number: $response"
	}

	return
    }

    # "NextArticle" advances nntpHandle's article pointer, i.e.
    # it moves the pointer to the next article. It returns the
    # number of the new current article.

    proc NextArticle {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "NEXT" command, and get a response.

	puts $sock "NEXT"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 223} {
	    return -code error "Error going to next article: $response"
	}

	# Return the article number.

	return [lindex $response 1]
    }

    # "PrevArticle" decrements nntpHandle's article pointer, i.e.
    # it moves the pointer to the previous article. It returns
    # the number of the new current article.

    proc PrevArticle {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "LAST" command, and get a response.

	puts $sock "LAST"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 223} {
	    return -code error "Error going to next article: $response"
	}

	# Return the article number.

	return [lindex $response 1]
    }

    # "Post" posts an article to group "groupName" via the server
    # represented by "nntpHandle". "header" is the article's header,
    # "body" is the body of the article.

    proc Post {nntpHandle groupName header body} {
	variable nntpSocket
	variable nntpMsgHandler
	variable serverAllowsPosting

	# Make sure we're allowed to post.

	if {!$serverAllowsPosting($nntpHandle)} {
	    return -code error "Server does not allow posting"
	}

	set sock $nntpSocket($nntpHandle)

	# Send the "POST" command.

	puts $sock "POST"
	flush $sock

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 340} {
	    return -code error "Could not post message"
	}

	# Write the message: header first, then body.

	set text "$header\n\n$body"
	foreach line [split $text "\n"] {
	    # If the line starts with a ".", send an extra "."

	    set firstChar [string index $line 0]
	    if {[string compare $firstChar "."] == 0} {
		set line ".$line"
	    }

	    puts $sock $line
	}

	# End with a single "." on a line.

	puts $sock "."
	flush $sock

	# Get the response.

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 240} {
	    return -code error "Posting failed"
	}
    }

    # ClientIsSlave indicates to the server that the client is a "slave",
    # i.e. it is a slave server requesting a newsfeed, rather than a
    # normal interactive user.

    proc ClientIsSlave {nntpHandle} {
	variable nntpSocket
	variable nntpMsgHandler

	set sock $nntpSocket($nntpHandle)
	set msgHandler $nntpMsgHandler($nntpHandle)

	# Send the "SLAVE" command, and get a response.

	puts $sock "SLAVE"
	flush $sock
	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	# Check for errors...

	set msgCode [ResponseCode $response]
	if {$msgCode != 202} {
	    return -code error "Error going to next article: $response"
	}

	return {}
    }

    # IHave indicates to the server connected to "nntpHandle" that the
    # client has message "messageID", whose contents are "Header"
    # and "body". If the server wants this message, then it sends it
    # to the server.

    proc IHave {nntpHandle messageID header body} {
	variable nntpSocket
	variable nntpMsgHandler
	variable serverAllowsPosting

	# Make sure we're allowed to post.

	if {!$serverAllowsPosting($nntpHandle)} {
	    return -code error "Server does not allow posting"
	}

	set sock $nntpSocket($nntpHandle)

	# Send the "IHAVE" command.

	puts $sock "IHAVE <$messageID>"
	flush $sock

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	set rcode [ResponseCode $response]
	if {$rcode == 435} {
	    return
	}

	if {$rcode != 335} {
	    return -code error "Could not transfer article"
	}

	# Write the message: header first, then body.

	set text "$header\n\n$body"
	foreach line [split $text "\n"] {
	    # If the line starts with a ".", send an extra "."

	    set firstChar [string index $line 0]
	    if {[string compare $firstChar "."] == 0} {
		set line ".$line"
	    }

	    puts $sock $line
	}

	# End with a single "." on a line.

	puts $sock "."
	flush $sock

	# Get the response.

	set nchars [gets $sock response]
	if {$nchars < 0} {
	    catch [list Close $nntpHandle]
	    return -code error "Server disconnected prematurely"
	}

	if {[string compare $msgHandler ""] != 0} {
	    $msgHandler $response
	}

	if {[ResponseCode $response] != 235} {
	    return -code error "Article transfer failed"
	}
    }
}

