static char udp_RCS[] = "$Id: udp_tcl.c,v 1.1.1.1 1997/05/15 21:02:59 millerms Exp $";
/*
 *----------------------------------------------------------------------
 *	AUTHOR:        	Michael Miller - msmiller@acm.org
 *----------------------------------------------------------------------
 */
#include "udp_tcl.h"

/*
 *	need to find a way to make this system dependent
 */
#define MAX_UDP_BUFFER 4096
char udp_error[256];

/*
 * channel callback declarations
 */
static Tcl_File UDP_get(ClientData instanceData, int direction);
static int UDP_close(ClientData instanceData, Tcl_Interp *interp);
static int UDP_ready(ClientData instanceData, int mask);
static void UDP_watch(ClientData instanceData, int mask);

static Tcl_ChannelType UDP_type = {
    "udp",         /* Type name. */
    NULL,          /* Set blocking/nonblocking mode.*/
    UDP_close,     /* Close proc. */
    NULL,          /* Input proc. */
    NULL,          /* Output proc. */
    NULL,          /* Seek proc. */
    NULL,          /* Set option proc. */
    NULL,          /* get option proc. */
    UDP_watch,     /* Initialize notifier. */
    UDP_ready,     /* Are there events? */
    UDP_get,       /* get Tcl_Files out of channel. */
};

/*
 * prototype declarations
 */
int UDP_open(ClientData , Tcl_Interp *, int , char *[]);
int UDP_conf(ClientData , Tcl_Interp *, int , char *[]);
int UDP_recv(ClientData , Tcl_Interp *, int , char *[]);
int UDP_send(ClientData , Tcl_Interp *, int , char *[]);
int UDP_Init(Tcl_Interp *);


/*
 *----------------------------------------------------------------------
 * udp_open	-	API for opening UDP socket
 *
 * needs	port
 * returns 	fd
 *			< 0		error
 *----------------------------------------------------------------------
 */
int udp_open(short localport) {
	int					fd;
	struct sockaddr_in	homeaddr;

	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		sprintf(udp_error,"%s","udp - socket");
		return -1;
	}
	memset(&homeaddr, 0, sizeof(homeaddr));
	homeaddr.sin_family = AF_INET;
	homeaddr.sin_addr.s_addr = 0;
	homeaddr.sin_port = htons(localport);
	if (bind(fd,(struct sockaddr *)&homeaddr, sizeof(homeaddr)) < 0) {
		sprintf(udp_error,"%s","udp - bind");
		return -2;
	}
	return fd;
}

/*
 *----------------------------------------------------------------------
 * udp_recv	-	API for reading a UDP socket
 *
 * needs	fd
 * returns 	message size		r_host r_port type message_string
 *			< 0					error
 *----------------------------------------------------------------------
 */
int udp_recv(int fd, char **r_host, short *r_port, void **message, int *size)
{
	int					actual_size, socksize, reply;
	int					buffer_size = MAX_UDP_BUFFER;
	short				port;
	char				number[32], *remotehost;
	struct hostent		*name;
	struct sockaddr_in	recvaddr;
	char 				*udp_mesg = NULL;

	socksize = sizeof(recvaddr);
	memset(number, 0, 32);
	memset(&recvaddr, 0, socksize);
	udp_mesg = (char *)calloc(1, MAX_UDP_BUFFER);
	if (udp_mesg == NULL) {
		sprintf(udp_error,"%s","udp - calloc");
		return -1;
	}

	actual_size = recvfrom(fd, (char *)udp_mesg, buffer_size, 0,
	(struct sockaddr *)&recvaddr, &socksize);
	if (actual_size < 0) {
		sprintf(udp_error,"%s","udp - recvfrom");
		return -1;
	}

	name = gethostbyaddr((const char *)&recvaddr.sin_addr, 
	sizeof(recvaddr.sin_addr), recvaddr.sin_family);
	if (name)
		remotehost = strdup(name->h_name);
	else
		remotehost = (char *)inet_ntoa(recvaddr.sin_addr);

	port = ntohs(recvaddr.sin_port);
	*r_host = strdup(remotehost);
	*r_port = port;
	*message = udp_mesg;
	return actual_size;
}

/*
 *----------------------------------------------------------------------
 * udp_send	-	API for writing to a UDP socket
 *
 * needs 	fd r_host r_port message_string
 * returns 	>= 0	no error
 *			< 0		error
 *----------------------------------------------------------------------
 */
int udp_send(int fd, char *r_host, short r_port, void *message, int buf_size)
{
	int					socksize, sendsize;
	struct hostent		*name;
	struct sockaddr_in	sendaddr;
	struct in_addr		inp;

	if (buf_size > MAX_UDP_BUFFER) {
		sprintf(udp_error,"%s %d","udp - MAX_UDP_BUFFER", MAX_UDP_BUFFER);
		return -1;
	}
	socksize = sizeof(sendaddr);
	memset(&sendaddr, 0, socksize);

	name = gethostbyname(r_host);
	if (name == NULL) {
		inp.s_addr = inet_addr(r_host);
		name = gethostbyaddr((char *)&inp, sizeof(inp), AF_INET);
		if (name == NULL) {
			sprintf(udp_error,"%s","udp - gethostbyname");
			return -2;
		}
	}
	memcpy(&sendaddr.sin_addr, name->h_addr, sizeof(sendaddr.sin_addr));
	sendaddr.sin_family = AF_INET;
	sendaddr.sin_port = htons(r_port);
	sendsize = sendto(fd, message, buf_size, 0, 
	(struct sockaddr *)&sendaddr, socksize);
	if (sendsize < 0) {
		sprintf(udp_error,"%s","udp - sendto");
		if (errno == EMSGSIZE)
			sprintf(udp_error,"%s","udp - sendto EMSGSIZE");
		return -3;
	}
	return sendsize;
}

/*
 *----------------------------------------------------------------------
 * UDP_close
 *----------------------------------------------------------------------
 */
static int UDP_close(ClientData instanceData, Tcl_Interp *interp)
{
	Tcl_File sockFile;
	int sock;
	int errorCode = 0;
	UDP_state *statePtr = (UDP_state *) instanceData;

	sockFile = statePtr->sock;
	sock = (int) Tcl_GetFileInfo(sockFile, NULL);
	Tcl_DeleteFileHandler(sockFile);
	ckfree((char *) statePtr);
	Tcl_FreeFile(sockFile);
	if (close(sock) < 0) {
		errorCode = errno;
	}
	return errorCode;
}

/*
 *----------------------------------------------------------------------
 * UDP_ready
 *----------------------------------------------------------------------
 */
static int UDP_ready(ClientData instanceData, int mask)
{
	UDP_state *statePtr = (UDP_state *) instanceData;
	return Tcl_FileReady(statePtr->sock, mask);
}

/*
 *----------------------------------------------------------------------
 * UDP_watch
 *----------------------------------------------------------------------
 */
static void UDP_watch(ClientData instanceData, int mask)
{
	UDP_state *statePtr = (UDP_state *) instanceData;
	Tcl_WatchFile (statePtr->sock, mask);
}

/*
 *----------------------------------------------------------------------
 * UDP_get
 *----------------------------------------------------------------------
 */
static Tcl_File UDP_get(ClientData instanceData, int direction)
{
	UDP_state *statePtr = (UDP_state *) instanceData;
	return statePtr->sock;
}


/*
 *----------------------------------------------------------------------
 * udp_configure	-	set remote host and port in channel
 *----------------------------------------------------------------------
 */
int UDP_conf(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	Tcl_Channel chan;
	UDP_state *statePtr;

	if (argc != 4) {
		interp->result = "udp_conf fileId remotehost remoteport";
		return TCL_ERROR;
	}
	chan = Tcl_GetChannel(interp, argv[1], NULL);
	if (chan == (Tcl_Channel) NULL) {
		return TCL_ERROR;
	}
	statePtr = (UDP_state *) Tcl_GetChannelInstanceData(chan);
	if (strlen(argv[2]) >= sizeof(statePtr->remotehost)) {
		interp->result = "hostname too long";
		return TCL_ERROR;
	}
	strcpy(statePtr->remotehost, argv[2]);
	statePtr->remoteport = atoi(argv[3]);
	return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 * udp_open	-	opens a UDP socket and addds the file descriptor to the
 *				tcl interpreter
 *
 * needs	port
 * returns 	fd
 *----------------------------------------------------------------------
 */
int UDP_open(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int		fd;
	Tcl_File sockFile;
	char channelName[20];
	Tcl_Channel chan;
	UDP_state *statePtr;

	if (argc != 2) {
		interp->result = "udp_open port";
		return TCL_ERROR;
	}
	memset(channelName, 0, sizeof(channelName));
	fd = udp_open(atoi(argv[1]));
	if (fd > 0) {
		sockFile = Tcl_GetFile((ClientData) fd, TCL_UNIX_FD);
		statePtr = (UDP_state *) ckalloc((unsigned) sizeof(UDP_state));
		statePtr->sock = sockFile;
		sprintf(channelName, "sock%d", 
				(int) Tcl_GetFileInfo(statePtr->sock, NULL));
		chan = Tcl_CreateChannel(&UDP_type, channelName,
			(ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
		Tcl_RegisterChannel(interp, chan);
		Tcl_AppendResult(interp, channelName, (char *)NULL);
		return TCL_OK;
	} else {
		Tcl_AppendResult(interp, udp_error, (char *)NULL);
		return TCL_ERROR;
	}
}

/*
 *----------------------------------------------------------------------
 * udp_recv	-	reads a UDP socket 
 * needs	fd
 * returns 	r_host r_port message_string
 *----------------------------------------------------------------------
 */
int UDP_recv(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int		reply;
	int 	bufsize;
	short 	remoteport;
	int mode=(TCL_READABLE | TCL_WRITABLE);
	Tcl_Channel chan;
	Tcl_File tf;
	int fd;
	char	*remotehost, *message, number[32];

	if (argc != 2) {
		interp->result = "udp_recv fileId";
		return TCL_ERROR;
	}
	chan = Tcl_GetChannel(interp, argv[1], &mode);
	if (chan == (Tcl_Channel) NULL) {
		return TCL_ERROR;
	}
	tf = Tcl_GetChannelFile(chan,(TCL_READABLE | TCL_WRITABLE));
	fd = (int) Tcl_GetFileInfo(tf, NULL);

	message = NULL;
	reply = udp_recv(fd, &remotehost, &remoteport, (void **)&message, &bufsize);
	if (reply > -1) {
		Tcl_AppendResult(interp, remotehost, (char *)NULL);
		memset(number, 0, 32);
		sprintf(number,"%d", remoteport);
		Tcl_AppendElement(interp, number);
		Tcl_AppendElement(interp, message);
		free(message);
		return TCL_OK;
	} else {
		Tcl_AppendResult(interp, udp_error, (char *)NULL);
		return TCL_ERROR;
	}
}

/*
 *----------------------------------------------------------------------
 * udp_send	-	sends a character string along a UDP socket
 *
 * needs 	fd r_host r_port message_string
 * returns 	error
 *----------------------------------------------------------------------
 */
int UDP_send(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	int		reply;
	Tcl_Channel chan;
	int chanMode=(TCL_READABLE | TCL_WRITABLE);
	Tcl_File tf;
	int fd;
	UDP_state *statePtr;

	if (argc != 3) {
		interp->result = "udp_send fileId string";
		return TCL_ERROR;
	}
	chan = Tcl_GetChannel(interp, argv[1], &chanMode);
	if (chan == (Tcl_Channel) NULL) {
		return TCL_ERROR;
	}
	statePtr = Tcl_GetChannelInstanceData(chan);
	tf = Tcl_GetChannelFile(chan,(TCL_READABLE | TCL_WRITABLE));
	fd = (int) Tcl_GetFileInfo(tf, NULL);
	reply = udp_send(fd, statePtr->remotehost, statePtr->remoteport,
		argv[2], strlen(argv[2]));
	if (reply > -1) {
		return TCL_OK;
	} else {
		Tcl_AppendResult(interp, udp_error, (char *)NULL);
		return TCL_ERROR;
	}
}

/*
 *----------------------------------------------------------------------
 * UDP_Init
 *----------------------------------------------------------------------
 */
int UDP_Init(Tcl_Interp *interp) 
{
	Tcl_CreateCommand(interp, "udp_open", UDP_open , 
		(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "udp_configure", UDP_conf , 
		(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "udp_send", UDP_send , 
		(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
	Tcl_CreateCommand(interp, "udp_recv", UDP_recv , 
		(ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
	return TCL_OK;
}
