
/*
 * TkTree.cc - member routines for class TkTree,
 *             implementation of the TCL tree command
 * 
 * -----------------------------------------------------------------------------
 * Copyright 1994 Allan Brighton.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies.  
 * Allan Brighton make no representations about the suitability of this 
 * software for any purpose. It is provided "as is" without express or 
 * implied warranty.
 * -----------------------------------------------------------------------------
 */

#include "TkTree.h"
#include "TkTreeNode.h"
#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <string.h>


/* 
 * declare a table of tree subcommands and the methods that handle them.
 * format: name, min_args, max_args, method
 */
static class TkTreeSubCmds {
public:
    char* name;      // method name
    int (TkTree::*fptr)(int argc, char* argv[]); 
    int min_args;    // minimum number of args
    int max_args;    // maximum number of args
    int mflag;       // set to 1 if the tree should be redrawn after calling
} subcmds_[] = { 
    {"addlink",       &TkTree::addlink,        3, 99,  1},
    {"ancestors",     &TkTree::ancestors,      1,  1,  0},
    {"getcanvas",     &TkTree::getcanvas,      0,  0,  0},
    {"canvas",        &TkTree::getcanvas,      0,  0,  0},
    {"child",         &TkTree::child,          1,  1,  0},
    {"isleaf",        &TkTree::isleaf,         1,  1,  0},
    {"isroot",        &TkTree::isroot,         1,  1,  0},
    {"movelink",      &TkTree::movelink,       2,  2,  1},
    {"nodeconfigure", &TkTree::nodeconfigure,  1, 99,  1},
    {"nodeconfig",    &TkTree::nodeconfigure,  1, 99,  1},
    {"prune",         &TkTree::prune,          1,  1,  1},
    {"parent",        &TkTree::parent,         1,  1,  0},
    {"root",          &TkTree::root,           1,  1,  1},
    {"rmlink",        &TkTree::rmlink,         1,  1,  1},
    {"sibling",       &TkTree::sibling,        1,  1,  0},
    {"subnodes",      &TkTree::subnodes,       1,  1,  0},
    {"draw",          &TkTree::draw,           0,  0,  0}
};


/* 
 * tree config options - used to process command line options and for the
 * widget "configure" subcommand.
 */
static Tk_ConfigSpec configSpecs_[] = {
    {TK_CONFIG_PIXELS, "-parentdistance", "parentDistance", "ParentDistance",
     "30", Tk_Offset(TkTreeOptions, parentDistance), 0},
    {TK_CONFIG_STRING, "-layout", "layout", "Layout",
     NULL, Tk_Offset(TkTreeOptions, layout), 0},
    {TK_CONFIG_SYNONYM, "-orient", "layout", NULL, NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
     "10", Tk_Offset(TkTreeOptions, borderWidth), 0},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};



/*
 * Call the given method in this class with the given arguments
 */
int TkTree::call(const char* name, int len, int argc, char* argv[])
{
    for(int i = 0; i < sizeof(subcmds_)/sizeof(*subcmds_); i++) {
	TkTreeSubCmds* t = &subcmds_[i];
	if (strncmp(t->name, name, len) == 0) {
	    if (check_args(name, argc, t->min_args, t->max_args) != TCL_OK)
		return TCL_ERROR;
	    if (t->mflag)
		eventually_redraw();
	    return (this->*t->fptr)(argc, argv);
	}
    }
    return TkWidget::call(name, len, argc, argv);
}


/*
 * A call to this function is made from the tkAppInit file at startup
 * to install the tree command
 */
extern "C"
int Tree_Init(Tcl_Interp* interp)  
{
    Tcl_CreateCommand(interp, "tree", TkTree::treeCmd, NULL, NULL);
    return TCL_OK;
}


/*
 * This static method is called by tcl to create a tree widget
 * Syntax: see tree man page
 */
int TkTree::treeCmd(ClientData, Tcl_Interp *interp, int argc, char* argv[])
{
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
	     argv[0], " path ?options?\"", NULL);
	return TCL_ERROR;
    }

    TkTree *t = new TkTree(interp, argc, argv);
    return t->status();
}


/* 
 * This event proc is used once to determine when the canvas window
 * has been mapped so we can get the size and center the tree
 * in it
 */
void TkTree::eventProc(ClientData clientData, XEvent *)
{
    TkTree* thisPtr = (TkTree*)clientData;
    Tk_DeleteEventHandler(thisPtr->canvasWin_, ExposureMask,
			  eventProc, (ClientData)thisPtr);
    thisPtr->center();
}


/*
 * Constructor: initialize a new tree with the command line args
 * 
 * This constructor is called for each tree declared in tk. The destructor
 * is called when the tree widget is destroyed.
 */
TkTree::TkTree(Tcl_Interp *interp, int argc, char* argv[])
: TkWidget(interp, "Canvas", configSpecs_, options_, argc, argv),
  canvas_(pname_),
  canvasWin_(Tk_NameToWindow(interp_, canvas_, Tk_MainWindow(interp))),
  rootNode_(new TkTreeNode(this, "", "")),
  options_(30, "horizontal", 10),
  center_flag_(1)
{
    Tk_CreateEventHandler(canvasWin_, ExposureMask, 
			  (Tk_EventProc*)eventProc, (ClientData)this);
    
    // do first time widget configuration
    initWidget(argc, argv);
}


/*
 * destructor - clean up tree nodes when widget cmd is destroyed
 */
TkTree::~TkTree()
{
    delete rootNode_;
    rootNode_ = NULL;
}


/*
 * This routine is called with the bounding box of the tree before drawing
 * We use this information to change the scrollregion of the tree's canvas
 */
void TkTree::setDrawArea(int x1, int y1, int x2, int y2) 
{
    int b = borderWidth();	   // border width around tree
    int p = parentDistance();	   // distance to parent nodes

    if (horizontal())  {
	x1 += p;
    }
    else {
	y1 += p;
    }

    // add the borderwidth as padding around the outside
    x1 -= b;    x2 += b;
    y1 -= b;    y2 += b;
    
    char buf[1024];
    sprintf(buf, "%s config -scrollregion {%d %d %d %d}", 
	    canvas_, x1, y1, x2, y2);
    Tcl_Eval(interp_, buf);
}


/*
 * find the named node and return a pointer to it or 0 if not found
 */
TkTreeNode* TkTree::findNode(const char* tag)
{
    TkTreeNode* node = (TkTreeNode*)rootNode_->find(tag);
    if (node == NULL) {
	error("can't find tree node: ", tag);
	return NULL;
    }
    return node;
}


/*
 * Set up the node's size, position and options
 */
int TkTree::setupNode(TkTreeNode* node, int argc, char* argv[])
{
    if (Tcl_VarEval(interp_, canvas_, " bbox {", node->tag(), "}", 0) != TCL_OK)
        return TCL_ERROR;
    
    // get the size of the node
    int x1, y1, x2, y2;
    sscanf(interp_->result, "%d %d %d %d", &x1, &y1, &x2, &y2);	
    node->box(x1, y1, x2, y2);
    
    // get the other options
    for (int i = 0; i < argc; i++) {
	if (strcmp(argv[i], "-border") == 0 && ++i < argc) {
	    int border = 0;
	    if (Tk_GetPixels(interp_, canvasWin_, argv[i], &border) != TCL_OK) {
		return TCL_ERROR;
	    } 
	    node->border(border);
	}
	else if (strcmp(argv[i], "-remove") == 0 && ++i < argc) {
	    node->removeCmd(argv[i]);
	}
	else {
	    return error("unknown tree node configuration option, should be -border or -remove");
	}
    }
    
    Tcl_ResetResult(interp_);
    return TCL_OK;
}


/*
 * add a new child node to the parent node in the tree
 * usage: addlink parent child line ?options?
 */
int TkTree::addlink(int argc, char* argv[])
{
    TkTreeNode* parent = findNode(argv[0]);
    if (parent == NULL) 
	return TCL_ERROR;
    
    TkTreeNode* child = new TkTreeNode(this, argv[1], argv[2], 0, 0, 0, 0, 0);
    if (setupNode(child, argc-3, argv+3) != TCL_OK)
        return TCL_ERROR;
        
    parent->addLink(child);
    return TCL_OK;
}


/*
 * unhook the child subtree and make it a subtree of the parent tree
 * usage: $tree movelink child parent
 */
int TkTree::movelink(int, char* argv[])
{
    TkTreeNode* parent = findNode(argv[1]);
    if (parent == NULL) 
	return TCL_ERROR;
	
    TkTreeNode* child = findNode(argv[0]);
    if (child == NULL) 
	return TCL_ERROR;

    /* don't allow the child to be itself, it's parent or
     * one of the parent's ancestors (no time travel allowed :-) 
     */ 
    if (parent == child || parent == child->parent()) {
	return error("illegal move");
    }
    for (TreeNode* p = parent; p && *p->tag(); p = p->parent()) {
	if (p == child) {
	    error("illegal move");
	    return TCL_ERROR;
	}
    }   
	
    child->unLink ();
    parent->addLink(child);
    
    return TCL_OK;
}
    
  

/*
 * remove the named node and its subnodes from the tree
 * usage: $tree rmlink tag
 */
int TkTree::rmlink(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    node->rmLink();
    return TCL_OK;
}
    
    
/*
 * Remove and delete the nodes children
 */
int TkTree::prune(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    node->prune();
    return TCL_OK;
}


/*
 * This procedure is redefined here from the parent class
 * to notice when the layout changes between horizontal and
 * vertical, so we can center the tree
 */
int TkTree::configureWidget(int argc, char* argv[], int flags)
{
    int h = horizontal();
    int ret = TkWidget::configureWidget(argc, argv, flags);
    center_flag_ += (h != horizontal());
    return ret;
}

    
/*
 * This command recalculates the node's size and also allows the same options
 * as the "addlink" sub-command (see the tree man page for details).
 */
int TkTree::nodeconfigure(int argc, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    if (setupNode((TkTreeNode*)node, --argc, ++argv) != TCL_OK)
        return TCL_ERROR;
        
    return TCL_OK;
}

    
/*
 * Make the named node the new root of the tree.
 */
int TkTree::root(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;
    
    if (node->parent()) {
        node->unLink ();
        rootNode_->prune();
        rootNode_->addLink(node);
    }
    
    return TCL_OK;
}
 
   
/*
 * return (in tcl) the name of the tree's canvas
 */
int TkTree::getcanvas(int, char**)
{
    // return the name of the canvas widget as a tcl result
    Tcl_SetResult(interp_, canvas_, TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns true if the node is a leaf node
 */
int TkTree::isleaf(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, (node->child() ? "0" : "1"), TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns true if the node is a root node
 * (root_ is invisible, so its children are the root nodes seen)
 */
int TkTree::isroot(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, ((node->parent() == rootNode_) ? "1" : "0"), TCL_STATIC);
    return TCL_OK;
}

    
/*
 * force a redraw of the tree in the canvas
 */
int TkTree::draw(int, char**)
{
    redraw();
    redraw_pending_ = 0;
    return TCL_OK;
}

     
/*
 * redraw the tree in the canvas
 */
void TkTree::redraw()
{
    if (center_flag_) {
	center();
	center_flag_ = 0;
	return;
    }
    rootNode_->drawTree();
}

     
/*
 * center the tree in the canvas
 */
void TkTree::center()
{
    int w = Tk_Width(canvasWin_);  
    int h = Tk_Height(canvasWin_); 

    // canvas may not be mapped yet, wait for event handler
    if (w <= 1 || h <= 1) {
	return;
    }

    // center the root node in the canvas
    if (horizontal())  {
        rootNode_->pos(TreePoint(0, h/2));
	rootNode_->drawTree();
	Tcl_VarEval(interp_, canvas_, " xview moveto 0", NULL);
    }
    else {
        rootNode_->pos(TreePoint(w/2, 0));
	rootNode_->drawTree();
	Tcl_VarEval(interp_, canvas_, " yview moveto 0", NULL);
    }
}


/*
 * this command returns the name of the node's child node
 * or the empty string if there is none
 */
int TkTree::child(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, (char*)(node->child() ? node->child()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this node's subnodes
 */
int TkTree::subnodes(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    for (TreeNode* p = node->child(); p; p = p->sibling()) 
        Tcl_AppendResult(interp_, "{", p->tag(), "} ", NULL);
    return TCL_OK;
}


/*
 * this command returns the name of the node's sibling node
 * or the empty string if there is none
 */
int TkTree::sibling(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, 
		     (char*)((node->sibling()) ? node->sibling()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * this command returns the name of the node's parent node
 * or the empty string if there is none
 */
int TkTree::parent(int, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    Tcl_SetResult(interp_, 
		     (char*)((node->parent()) ? node->parent()->tag() : ""), 
		     TCL_STATIC);
    return TCL_OK;
}


/*
 * This command returns a Tcl list of this nodes ancestors
 */
int TkTree::ancestors(int argc, char* argv[])
{
    TkTreeNode* node = findNode(argv[0]);
    if (node == NULL) 
	return TCL_ERROR;

    // count the ancestors
    int n = 0;
    const TreeNode* p;
    for (p = node->parent(); p && *p->tag(); p = p->parent()) {
	n++;
    }
    if (n == 0) 
	return TCL_OK;

    // allocate the array arg for the tcl list routines
    argc = n;
    argv = new char*[argc];
    for (p = node->parent(); p && *p->tag(); p = p->parent()) {
	argv[--n] = (char*)p->tag();
    }   

    // create the tcl list
    char* list = Tcl_Merge(argc, argv);
    if (list) 
	Tcl_AppendResult(interp_, list, NULL);
    else
	return TCL_ERROR;

    free(list);
    delete argv;
    return TCL_OK;
}

