# $Header: /usr/cvs/NEOSOFT/neowebscript-2.0/httpd/conf/class.tcl,v 1.2 1996/11/21 00:26:04 kunkee Exp $
#
# Copyright 1996 by NeoSoft, Inc. All rights reserved.
# This code was written by Randy Kunkee for NeoSoft.
# See LICENSE files in parent directories for information on use
# and redistribution.
#

Class member

member instproc init {args} {
	$self instvar parent
	set parent [rootname $self]
	eval $self next $args
	if [cequal $parent $self] {
		set parent {}
	} else {
		$parent append $self
	}
}

member instproc destroy {} {
	[$self set parent] delete $self
	$self next
}

member instproc text t { $self set text $t }

member instproc display {args} {
	$self set text
}

Class container 

container instproc init {args} {
	$self set members {}
	$self set itemsuffix {}
	eval $self next $args
}

container instproc itemsuffix s {$self set itemsuffix $s}

container instproc append m {
	$self instvar members
	lappend members $m
}

container instproc delete m {
	$self instvar members
	set i [lsearch $members $m]
	set members [lreplace $members $i $i]
}

container instproc destroy {} {
	foreach m [$self set members] {
		catch {$m destroy}
	}
	$self next
}


Class htattrs

htattrs instproc doattrs args {
	eval $self instvar $args
	set r {}
	foreach p $args {
		if [info exists $p] {append r " $p=\"[set $p]\""}
	}
	return $r
}

Class hypertext -superclass {container member}

hypertext instproc init {args} {
	$self instvar query text itemsuffix enabled
	set query {}
	set text {}
	set itemsuffix {}
	set enabled 1
	eval $self next $args
}

hypertext instproc display {{f 0} {t 0}} {
	$self instvar members query text itemsuffix enabled
	if !$enabled return
	set r $text
	foreach item $members {
		if $t {append r <td>}
		append r [$item display $f]
		if $t {append r </td>}
		append r $itemsuffix
	}
	if ![lempty $query] {
		set r [$query subst $r]
	}
	return $r
}

hypertext instproc query q { $self set query $q }
hypertext instproc text t { $self set text $t }
hypertext instproc itemsuffix t { $self set itemsuffix $t }
hypertext instproc enabled t { $self set enabled $t }


Class anchor -superclass hypertext

anchor instproc display {} {
	set r <A
	append r [$self doattrs href]
	append r >[$self next]</A>
}

Class table -superclass {hypertext htattrs}

table instproc init {args} {
	$self set info {}
	eval $self next $args
}

table instproc border p {$self set border $p}
table instproc cellspacing p {$self set cellspacing $p}
table instproc cellpadding p {$self set cellpadding $p}
table instproc width p {$self set width $p}
table instproc caption p {$self set caption $p}

# misc info (just gets stuck into the table)
table instproc info i {$self set info $i}

table instproc display {{f 0} {t 0}} {
	$self instvar members query info text border enabled caption
	if !$enabled return
	set r $text<table
	append r [$self doattrs cellspacing cellpadding width]
	if [info exists border] {
		append r " BORDER"
		if $border {append r =$border}
	}
	append r >
	if [info exists caption] {
		append r "<CAPTION>$caption</CAPTION>\n"
	}
	foreach item $members {
		append r <tr>
		append r "[$item display $f 1]"
		append r "</tr>\n"
	}
	append r "</table>"
	if ![lempty $query] {
		set r [$query subst $r]
	}
	return $r
}


Class querytable -superclass table

querytable instproc init {args} {
	eval $self next $args
	$self instvar query rowdata
	if ![info exists query] {error "$class $self: must set -query"}
	if ![info exists rowdata] {error "$class $self: must set -rowdata"}
}

querytable instproc rowdata rd {$self set rowdata $rd}

querytable instproc display {{f 0} {t 0}} {
	$self instvar members query info text border enabled rowdata
	if !$enabled return
	set r $text<table
	append r [$self doattrs cellspacing cellpadding width]
	if [info exists border] {
		append r " BORDER"
		if $border {append r =$border}
	}
	append r >
	$query foreach_tuple {
		append r <tr>
		foreach item [$rowdata set members] {
			append r [$query subst [$item display $f 1]]
		}
		append r "</tr>\n"
	}
	append r "</table>"
	return $r
}


# create class formitem, and implement -form, -text, and -display.


Class text -superclass member

text instproc display {{f 0} {t 0}} {
	if ![$self set enabled] return
	if $t { $self set text } else { return <td>[$self set text]</td> }
}

Class form_element -superclass {member htattrs}

# like to eventually handle isattr differently.  Just have a collection
# type of, eg. dbform, which implies -isattr 1.

form_element instproc init {args} {
	$self instvar name prefix isattr text enabled
	set name {}
	set prefix {}
	set isattr 1
	set enabled 1
	eval $self next $args
	if [lempty $name] { regexp {[^.]*$} $self name }
	if $isattr { $self set text "\[\$_var $name\]" }
}

form_element instproc name n { $self set name $n }
form_element instproc prefix p { $self set prefix $p }
form_element instproc isattr a { $self set isattr $a }
form_element instproc text t { $self set isattr 0; $self set text $t }
form_element instproc enabled t { $self set enabled $t }

Class output -superclass form_element

output instproc display {{f 0} {t 0}} {
	$self instvar size name type isattr prefix text enabled
	if !$enabled return
	#puts "$self $class $type $f $t"
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		append r $prefix
	}
	append r $text
	if $t {append r </td>}
	return $r
}


Class input -superclass form_element

# Here's an idea, have the form elements default to being database items
# unless -text is specified.

input instproc init {args} {
	$self set type text
	$self set name [translit . _ $self]
	eval $self next $args
}

# default enabled status to 0 for hidden types so they don't appear
# magically anywhere
input instproc type t {
	$self instvar type enabled
	$self set type $t
}
input instproc size s { $self set size $s }
input instproc checked s { $self set checked $s }
input instproc length s { $self set length $s }
input instproc maxlength s { $self set maxlength $s }
input instproc value s { $self set value $s }

input instproc display {{f 0} {t 0}} {
	$self instvar size name type isattr prefix text enabled
	if !$enabled return
	#debug "<br><code>$self $class $type $f $t<code>"
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		append r $prefix
	}
	if $f {
		append r "\n<INPUT TYPE=\"$type\" NAME=\"$name\" VALUE=\"$text\""
		append r [$self doattrs size value length maxlength]
		append r "></input>"
	} else {
		append r $text
	}
	if $t {append r </td>}

	return $r
}

Class checkbox -superclass input
checkbox instproc init args {
	eval $self next $args
	$self set type checkbox
	$self set value 1
}

checkbox instproc desc s {$self set desc $s}
checkbox instproc text t {error "-text is undefined for checkboxes"}

checkbox instproc display {{f 0} {t 0}} {
	$self instvar size name type isattr prefix desc enabled value
	if !$enabled return
	if ![info exists value] {set value $name}
	if ![info exists desc] {set desc $value}
	#debug "<br><code>$self $class $type $f $t<code>"
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		append r $prefix
	}
	if $f {
		append r "\n<INPUT TYPE=\"$type\" NAME=\"$name\""
		append r [$self doattrs value]
		append r "\[$self ischecked\]"
		append r "></input>$desc"
	} else {
		append r $desc
	}
	if $t {append r </td>}

	return $r
}

checkbox instproc ischecked {} {
	$self instvar isattr checked name value
	if $isattr {
		upvar $name dbvalue
		if [info exists dbvalue] {
			if [cequal $dbvalue $value] {
				return " CHECKED"
			}
			return
		}
	}
	if $checked {
		return " CHECKED"
	}
	return
}
# note: need some gifs for checkboxes and radio buttons to indicate they
# are checked when displayed outside of a form.


# Submit class
Class submit -superclass input

submit instproc init {args} {
	eval $self next $args
	$self instvar type name
	set type submit
	set name _$name
}

# Submit reset
Class reset -superclass input

reset instproc init {args} {
	eval $self next $args
	$self instvar type name
	set type reset
	set name _$name
}

Class edit_input -superclass input
edit_input instproc init args {
	$self set isattr 0
	$self set oidname oid
	eval $self next $args
}
edit_input instproc relname r {$self set relname $r}
edit_input instproc oidname o {$self set oidname $o}
edit_input instproc display {{f 0} {t 0}} {
	if !$f return
	$self next $f $t
}

Class delete_check -superclass edit_input
delete_check instproc init {args} {
	$self instvar relname oidname type isattr text
	set text Delete
	eval $self next $args
	set type checkbox
	$self set name "__DELETE/$relname/\$$oidname"
}

Class edit_button -superclass edit_input
edit_button instproc init args {
	$self instvar relname oidname type isattr
	$self text Edit
	eval $self next $args
	set type submit
	$self set name "__EDIT/$relname/\$$oidname"
}

Class update_button -superclass edit_input
update_button instproc init args {
	$self instvar relname oidname type text isattr
	$self text Update
	eval $self next $args
	set type submit
	$self set name "__UPDATE/$relname/\$$oidname"
}



# also need class to handle select and option.
Class select -superclass form_element

# select options:
#    -values list to set the list of values in the creations
#    -selected list of values that are selected
#	
# methods:
#   option value
#

select instproc init args {$self set selected {}; eval $self next $args}
select instproc values v {$self set values $v}
select instproc select v {$self set selected $v}
select instproc size v {$self set size $v}
select instproc multiple v {$self set multiple $v}
select instproc width v {$self set width $v}
select instproc height v {$self set height $v}

select instproc option {value {text {}}} {
	$self instvar values
	lappend values [list $value $text]
}

# would be nice if the selected could be delated until the final query
select instproc display {{f 0} {t 0}} {
	$self instvar name prefix isattr text values selected query enabled
	if !$enabled {
		if $t {return <td></td>}
		return
	}
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		append r $prefix
	}
	if $f {
		append r "\n<SELECT NAME=\"$name\""
		append r "[$self doattrs size multiple width height]>\n"
		foreach v $values {
			lassign $v vname vtext
			if [lempty $vtext] {set vtext $vname}
			append r "<OPTION \[$self selected [list $vname]\]"
			append r " VALUE=\"$vname\""
			append r ">$vtext</OPTION>\n"
		}
		append r "</SELECT>\n"
	} else {
		append r $text
	}
	if $t {append r </td>}
	return $r
}

select instproc selected {optionvalue} {
	$self instvar isattr selected name
	if $isattr {
		upvar $name dbvalue
		if [info exists dbvalue] {
			if [cequal $dbvalue $optionvalue] {
				return " SELECTED"
			}
			return
		}
	}
	if [cequal $selected $optionvalue] {
		return SELECTED
	}
	return
}

Class radio -superclass form_element

# would be nice if the selected could be delated until the final query
radio instproc display {{f 0} {t 0}} {
	$self instvar name prefix isattr text values selected query enabled
	if !$enabled return
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		set r $prefix
	}
	if $f {
		foreach v $values {
			lassign $v vname vtext
			if [lempty $vtext] {set vtext $vname}
			append r "<INPUT NAME=\"$name\" TYPE=radio VALUE=\"$vname\""
			append r "\[$self checked $vname\]"
			append r "><b>$vtext</b></INPUT>\n"
		}
	} else {
		append r $text
	}
	if $t {append r </td>}
	return $r
}

radio instproc checked {optionvalue} {
	$self instvar isattr selected name
	if $isattr {
		upvar $name dbvalue
		if [info exists dbvalue] {
			if [cequal $dbvalue $optionvalue] {
				return " CHECKED"
			}
			return
		}
	}
	if [cequal $selected $optionvalue] {
		return CHECKED
	}
	return
}

radio instproc init args {
	$self set selected {}
	eval $self next $args
}

radio instproc option {value {text {}}} {
	$self instvar values
	lappend values [list $value $text]
}



Class textarea -superclass form_element

textarea instproc init {args} {
	eval $self next $args
}

textarea instproc rows s { $self set rows $s }
textarea instproc cols s { $self set cols $s }
textarea instproc wrap s { $self set wrap $s }

textarea instproc display {{f 0} {t 0}} {
	$self instvar size name type prefix text enabled
	if !$enabled return
	if $t {
		set r <td>
		if ![lempty $prefix] { append r $prefix</td><td> }
	} else {
		set r $prefix
	}
	if $f {
		append r "\n<TEXTAREA NAME=\"$name\""
		$self instvar rows cols wrap
		foreach p {rows cols wrap} {
			if [info exists $p] {append r " $p=\"[set $p]\""}
		}
		append r >$text</textarea>
	} else {
		append r <BLOCKQUOTE>$text</BLOCKQUOTE>
	}
	if $t {append r "</td>\n"}
	return $r
}


# going to need a class for radio , checkbox, select, image hidden,
# submit


Class form -superclass {hypertext htattrs}

form instproc init {args} {
	eval $self next $args
}

form instproc method m {$self set method $m}
form instproc action a {$self set action $a}
form instproc table t {$self set table $t}

proc noquery attr {return}

form instproc display {{f 0} {t 0}} {
	$self instvar action query enabled table
	if !$enabled return
	if !$f {
		append r "\n<FORM[$self doattrs method action]>\n"
	}
	if [info exists table] {
		append r [$table display 1]
	}
	append r [$self next 1 $t]
	if !$f {append r "\n</FORM>\n"}
	if ![lempty $query] {
		set r [$query subst $r]
	} else {
		set _var noquery
		set r [subst $r]
	}
	return $r
}


Class sqlform -superclass form

sqlform instproc init {args} {
	$self instvar oidname
	set oidname oid
	eval $self next $args
	input $self.$oidname -type hidden -isattr 1
}

sqlform instproc oidname n { $self set oidname $n }

#
# subvar: a special variable to allow silently ignore undefined vars
proc subvar v {
	upvar $v w
	if [info exists w] {return $w}
	return
}


Class response

response instproc subst {_stuff} {
	$self instvar _attributes
	eval $self instvar $_attributes
	set _var subvar
	subst $_stuff
}

response instproc load_response {} {
	$self instvar _attributes _isresponse
	load_response _response
	set _attributes [array names _response]
	foreach _item $_attributes {
		$self set $_item $_response($_item)
	}
	set _isresponse [llength $_attributes]
	return $_isresponse
}

response instproc has pattern {
	$self instvar _attributes
	return [lmatch $_attributes $pattern]
}


Class query -superclass response

query instproc init {args} {
	$self instvar _parent _oidname _result _index _var _attributes
	set _var subvar
	set _result {}
	set _index -1
	set _parent [rootname $self]
	set _oidname oid
	set _attributes {}
	eval $self next $args
}

query instproc query q { $self set _query $q }
query instproc oidname n { $self set _oidname $n }
query instproc rowhtml n { $self set _rowhtml $n }
query instproc varsub n { $self set _var $n }
query instproc caption n { $self set _caption $n }
query instproc relname n { $self set _relname $n }
query instproc editattr n { $self set _editattr $n }
query instproc validproc n { $self set _validproc $n }
query instproc tableheading n { $self set _tableheading $n }

query instproc validate {{_attlist *}} {
	$self instvar _attributes _validproc
	if {$_attlist == "*"} {set _attlist $_attributes}
	eval $self instvar $_attlist
	set _errlist {}
	foreach _att $_attlist {
		set r [$_validproc $_att [set $_att]]
		if ![lempty $r] {
			lappend _errlist $_att
		}
	}
	return $_errlist
}

query instproc log {who what} {
    global webenv
    $self exec "insert into log values ([quote_sql $who], '$webenv(REMOTE_HOST)', 'now', [quote_sql $what]);"
}

query instproc index i {
	$self instvar _index _result
	if [cequal $i end] {
		set _index [expr [pg_result $_result -numTuples] - 1]
	} else {
		set _index $i
	}
}


# Note that _var indicates the name of a proc to fetch the procedure.
# for now this is just a proc which could be 'set' to get the values
# or do something more sophistocated.

query instproc exec {{_q {}}} {
	$self instvar _query _parent _attributes _result _numtup \
		_isresponse _index
	if [lempty $_q] {set _q $_query}
	#html "<p>exec query=$_q<p>"
	if ![lempty $_result] {
		pg_result $_result -clear
		set _result {}
	}
	$_parent open
	set _result [pg_exec [$_parent set dbconn] $_q]
	#html "_result=$_result, status = [pg_result $_result -status]<br>"
	set _result_status [pg_result $_result -status]
	if {$_result_status == "PGRES_TUPLES_OK"} {
		set _attributes [pg_result $_result -attributes]
		set _numtup [pg_result $_result -numTuples]
	} else {
		set _attributes {}
		set _numtup 0
	}
	set _index -1
	set _isresponse 0
	return [string match *_OK $_result_status]
}

query instproc dump_result {} {
	$self instvar _result
	if [lempty $_result] {return "It's not a result"}
	return [pg_result $_result -status]
}

# lookup will create a query demanding a match between the values of
# the attributes listed in _attlist to allow the user to quickly query
# the database after loading a response.

query instproc lookup {_attlist} {
	eval $self instvar $_attlist _relname _editattr _numtup
	foreach _attr $_attlist {
		lappend w $_attr=[quote_sql [set $_attr]]
	}
	if [info exists _editattr] {
		set _elist [join $_editattr ,]
	} else {
		set _elist *
	}
	$self exec "select $_elist from $_relname where [join $w AND];"
	if $_numtup {$self setvars 0}
	return $_numtup
}

query instproc destroy {} {
	if ![lempty [$self set _result]] {
		pg_result [$self set _result] -clear
	}
	$self next
}

query instproc setvars i {
	$self instvar _result _index _attributes _numtup
	#html "$self setvars $i" p
	set _index $i
	if $i==-1 {
		foreach a $_attributes {
			$self set $a $a
		}
		return
	}
	if $i<$_numtup {
		foreach a $_attributes v [pg_result $_result -getTuple $i] {
			$self set $a $v
		}
		return 1
	}
	foreach a $_attributes v {} {
		$self set $a $v
	}
	return 0
}

query instproc dump {} {
	foreach v [lsort [$self info vars]] {
		html "$v: [$self set $v]<br>"
	}
}

proc null args {return}

query instproc subst {_stuff} {
	$self instvar _attributes _index _var
	eval $self instvar $_attributes
	subst $_stuff
}

proc default v {
	upvar _attributes _a $v $v _defaults _d
	if [info exists _d($v)] {return $_d($v)}
	return
}

proc logged_on args {
	load_cookies
	if {[info exists cookies(neoworld)]} {
		if [dbfetch sessions $cookies(neoworld) userinfo] {
			if [lempty $args] {set args Login}
			uplevel set $args $userinfo(login)
			return 1
		}
	}
	return 0
}

query instproc attr {_att _val} {
	$self instvar _attributes $_att _isresponse
	if [lsearch $_attributes $_att]<0 {lappend _attributes $_att}
	set $_att $_val
	set _isresponse 1
}

query instproc sql_mdelete {} {
	$self instvar _attributes _isresponse _oidname _parent _result
	#html "_isresponse=$_isresponse<br>"
	if $_isresponse==0 {return 0}
	set q {}
	foreach del [lmatch $_attributes __DELETE/*] {
		lassign [split $del /] x relname oid
		append q "delete from $relname where oid=$oid;"
	}
	#html "q=$q<br>"
	if [lempty $q] {return 0}
	$self exec $q
}

query instproc sql_edit {} {
	$self instvar _attributes _isresponse _oidname _parent _result _numtup \
		_editattr
	if $_isresponse==0 {return 0}
	set q {}
	foreach edit [lmatch $_attributes __EDIT/*] {
		lassign [split $edit /] x relname oid
		append q "select [join $_editattr ,] from $relname where oid=$oid;"
	}
	#html "sql_edit q=$q<br>"
	if [lempty $q] {return 0}
	$self exec $q
}

#query instproc load_response {} {
#	$self instvar _attributes _isresponse
#	load_response _response
#	set _attributes [array names _response]
#	foreach _item $_attributes {
#		$self set $_item $_response($_item)
#	}
#	set _isresponse [llength $_attributes]
#	return $_isresponse
#}

# table value notes:
# 0 not in a table
# 1 called to generate a table row
#    rows escalate to generating table data
# 2 called to generate a table data
query instproc display {{f 0} {t 0} {_attlist {}}} {
	$self instvar _attributes _numtup _rowhtml _caption _tableheading
	#if ![info exists _rowhtml] {
	#	error "$display display called without -rowhtml initialized"
	#}
	if $_numtup==0 return
	eval $self instvar $_attributes
	if [lempty $_attlist] { set _attlist $_attributes }
	set r "\n<table border=1>\n"
	if [info exists _caption] {
		append r "<caption>$_caption</caption>\n"
	}
	if [info exists _rowhtml] {
		if [info exists _tableheading] {
			if ![lempty $_tableheading] {
				append r [$_tableheading display $f 1]
			}
		} else {
			$self setvars -1
			append r [translit "_" " " "<tr>[$_rowhtml display 0 1]</tr>\n"]
		}
		loop i 0 $_numtup {
			$self setvars $i
			append r "<tr>[$_rowhtml display $f 1]</tr>\n"
		}
	} else {
		if [info exists _tableheading] {
			if ![lempty $_tableheading] {
				$_tableheading display $f 1
			}
		} else {
			append r [translit "_" " " [make_tr $_attlist b]]
		}
		foreach col $_attlist {
			lappend _rowlist "\$$col"
		}
		loop i 0 $_numtup {
			$self setvars $i
			append r [subst [make_tr $_rowlist]]
		}
	}
	append r "</table>\n"
}

proc make_tr {columns {html {}}} {
	if [lempty $html] {
	    return <tr><td>[join $columns </td><td>]</$html></td></tr>
	} else {
	    return <tr><td><$html>[join $columns </$html></td><td><$html>]</$html></td></tr>
	}
}

# push the response into the database
query instproc sql_update {{attlist {}}} {
	$self instvar _attributes _isresponse _oidname _parent _result
	if $_isresponse==0 return
	if [lempty $attlist] { set attlist $_attributes }
	eval $self instvar $_attributes
	foreach update [lmatch $_attributes __UPDATE/*] {
		lassign [split $update /] _x _relname oid
	}
	if ![info exists _x] {return 0}
	set q {}
	foreach v $attlist {
		if [cequal $v oid]||[string match _* $v] continue
		if ![lempty $q] {append q ,}
		append q $v=[quote_sql [set $v]]
	}
	set q "update $_relname set $q where $_relname.$_oidname=[set $_oidname];"
	#html "sql_update q=$q\n"
	$self exec $q
}

# modify a record by replacing values, this one is more manual than sql_update
query instproc sql_modify {_keyname {_attlist {}}} {
	$self instvar _editattr _relname _result _parent _attributes
	if [lempty $_attlist] {set _attlist $_editattr}
	if {$_attlist == "*"} {set _attlist $_attributes}
	eval $self instvar $_attlist $_keyname
	foreach v $_attlist {
		if [cequal $v oid]||[string match _* $v]||[cequal $v $_keyname]  continue
		lappend q $v=[quote_sql [set $v]]
	}
	set q "update $_relname set [join $q ,] where $_relname.$_keyname=[quote_sql [set $_keyname]];"
	#html "sql_update q=$q\n"
	$self exec $q
}

# insert the response into the database
query instproc sql_insert {{attlist {}}} {
	$self instvar _attributes _isresponse _oidname _parent _result _relname
	if $_isresponse==0 {return 0}
	if [lempty $attlist] { set attlist $_attributes }
	eval $self instvar $_attributes
	set q {}
	set a {}
	foreach v $attlist {
		if [cequal $v oid]||[string match _* $v] continue
		lappend q [quote_sql [set $v]]
		lappend a $v
	}
	set q "insert into $_relname ([join $a ,]) values ([join $q ,]);"
	#html "sql_insert q=$q\n"
	#html "([id user]) sql_insert q=$q\n"
	$self exec $q
}

# insert the response into the database
query instproc sql_delete {} {
	$self instvar _attributes _isresponse _oidname _parent _result _relname
	#if $_isresponse==0 return
	set q "delete from $_relname where $_oidname='[$self set $_oidname]'::oid;"
	#html "sql_delete q=$q\n"
	$self exec $q
}

query instproc foreach_tuple body {
	$self instvar _isresponse _numtup
	if $_isresponse return
	loop i 0 $_numtup {
		$self setvars $i
		uplevel $body
	}
}

query instproc for_tuple {start end body} {
	$self instvar _isresponse _numtup
	if $_isresponse return
	switch $start start - begin {set i 0} current {set i $_index} default {
		error {syntax: for_tuple startindex endindex {body}}
	}
	switch $end end {set i $_numtup} default {
		error {syntax: for_tuple startindex endindex {body}}
	}
	loop i 0 $_numtup {
		$self setvars $i
		uplevel $body
	}
}

query instproc table_header {{attlist {}}} {
	$self instvar _attributes _oidname _numtup
	if [lempty $attlist] { set attlist $_attributes }
	set a {}
	foreach v $attlist {
		if [cequal $v $_oidname]||[string match _* $v] continue
		lappend a <b>$v</b>
	}
	set td {</td><td align=center>}
	return "<tr>$td[join $a $td]</td></tr>"
}


Class sqldb 

sqldb instproc init {n args} {
	$self instvar dbname dbconn host
	set host {}
	set dbname $n
	eval $self next $args
	set dbconn {}
}

sqldb instproc host h {$self set host "-host $h"}

sqldb instproc open {} {
	$self instvar dbname dbconn host
	if ![lempty $dbconn] return
	set dbconn [eval pg_connect $dbname $host]
}

sqldb instproc close {} {
	$self instvar dbname dbconn host
	if [lempty $dbconn] return
	pg_disconnect $dbconn
	set dbconn {}
}

# tear down query first, to clear all of the results.
sqldb instproc destroy {} {
	$self close
	$self next
}


Class defaults

defaults instproc default args { }

defaults instproc subst {_stuff} {
	set _var default
	subst $_stuff
}


defaults instproc attr {_att _val} {
	$self set _d($_att) $_val
}


proc default v {
	upvar _attributes _a $v $v _defaults _d
	if [info exists _d($v)] {return $_d($v)}
	return
}


proc quote_sql {string} {
    set string [join [split $string "\n"] "\\n"]
    regsub -all "(')" $string "\\'" string
    return '$string'
}

