#
# NeoWebScript - Server Side Programming based on Safe Tcl
#
# Copyright (C) NeoSoft, All Rights Reserved.  See NeoWebScript LICENSE
# files with this distribution for conditions on use and redistribution.
#
# These procs are loaded into safe interpreters
#
# $Id: client.tcl,v 1.1.1.1 1999/03/31 20:53:26 damon Exp $
#

#
# Crack the response (we're running as a pseudo-CGI in this case)
# into a global array of key-value pairs in the safe interpreter.
#
# Note info exists may return true when there is no data.
# We could check the action here and see if it's a POST
# and then we'd know which variable to look at.  The whole
# thing is somewhat kludgey.
#
# Now that we are supporting MIME types to handle file uploads, we
# need to find out if the response is mime, crack it out, and then
# handle it correctly.  Seek.  Locate.  Exterminate.
#
safe_proc load_response {{responseVar "response"} {multiple ""}} {
    global webenv
    upvar $responseVar response

    set responseString ""

    if {[info exists webenv(QUERY_STRING)]} {
	set responseString $webenv(QUERY_STRING)
    }

    if {[info exists webenv(NEO_POST_DATA)]} {
	set responseString $webenv(NEO_POST_DATA)
    }

    foreach key $multiple {
	set response(__$key) {}
    }

    foreach pair [split $responseString "&"] {
	set pair [split $pair "="]
	set key [unquote_string [lindex $pair 0]]
	set value [unquote_string [lindex $pair 1]]
	if [info exists response(__$key)] {
	    lappend response($key) $value
	} else {
	    if [info exists response($key)] {
		set response(__$key) {}
		set response($key) [list $response($key) $value]
	    } else {
		set response($key) $value
	    }
	}
    }

    if {[array exists response] && ([array size response] == 0)} {
        unset response
    }
}
# The above check is needed to fix a difference between Tcl 7.x and 8.x.
# 'array set arrayName list', where list is empty, will cause arrayName
# to exist in 8.x, but not in 7.x.  wierd.


#
# load Netscape's client-side cookie information into an array
#
safe_proc load_cookies {{cookieVar "cookies"}} {
    global webenv
    upvar $cookieVar cookies

    if ![info exists webenv(HTTP_COOKIE)] return

    foreach pair [split $webenv(HTTP_COOKIE) ";"] {
	set pair [split [string trim $pair] "="]
	set key [lindex $pair 0]
	set value [lindex $pair 1]
	set cookies($key) [list $value]
    }
}

safe_proc emit_standard_yahoo_link {} {
    html {<a href="http://www.yahoo.com"><img src="http://www.yahoo.com/images/recip.gif" alt="Yahoo!" border=0></a>}
}

safe_proc emit_standard_magellan_search_link {} {
    html {<form method=GET action="http://www.mckinley.com/extsearch.cgi">
<IMG SRC="http://images.mckinley.com/img/type.gif" ALIGN=MIDDLE>
<B>Search for:  </B> <input type=text name=query size=30 value="">
<INPUT type=submit value="Search">
</form>}
}

#
# convert an integer-seconds-since-1970 click value to
# RFC850 format, with the additional requirement that it be GMT only
# because that's the way netscape decided to do it.
#
safe_proc neo_clock_to_rfc850_gmt {seconds} {
    return [clock format $seconds -format "%a, %d-%b-%y %T GMT" -gmt 1]
}
#return [gm_timestr_822 $seconds]

#
# make_cookie_attributes encapsulates functions that were duplicated
# between the make_cookie and neo_make_cookie procs.
#
safe_proc make_cookie_attributes {paramsArray} {
    upvar $paramsArray params

    set cookieParams ""
    set expiresIn 0

    foreach {time num} [list days 86400 hours 3600 minutes 60] {
        if [info exists params($time)] {
	    incr expiresIn [expr $params($time) * $num]
	}
    }
    if {$expiresIn != 0} {
	append cookieParams "; expires=[neo_clock_to_rfc850_gmt [expr [clock seconds] + $expiresIn]]"
    }
    if [info exists params(path)] {
        append cookieParams "; path=$params(path)"
    }
    if [info exists params(domain)] {
        append cookieParams "; domain=$params(domain)"
    }
    if {[info exists params(secure)] && $params(secure) == 1} {
        append cookieParams "; secure"
    }

    return $cookieParams
}

#
# Create a cookie (send a browser a little message that you'll get
# back when they retrieve pages.)
#
# neo_make_cookie cookieName cookieValue [-days expireInDays]
#    [-hours expireInHours] [-minutes expireInMinutes]
#    [-path uriPathCookieAppliesTo]
#    [-secure 1|0]
#
safe_proc neo_make_cookie {name value args} {
    import_keyvalue_pairs params $args

    set badchars "\[ \t;\]"
    if {[regexp $badchars $name] || [regexp $badchars $value]} {
        return -code error "name/value pair \"$name=$value\" may not contain semicolons, spaces, or tabs"
    }
    set cookie "<meta http-equiv=\"Set-Cookie\" content=\"$name=$value"

    append cookie [make_cookie_attributes params]

    append cookie "\">"
    return $cookie
}

#
# Create a true cookie (send a browser a little message that you'll get
# back when they retrieve pages.)
# NB: This procedure uses the new true header method.  It must be
# called before any html has gone out.
#
# make_cookie cookieName cookieValue [-days expireInDays]
#    [-hours expireInHours] [-minutes expireInMinutes]
#    [-path uriPathCookieAppliesTo]
#    [-secure 1|0]
#
safe_proc make_cookie {name value args} {
    import_keyvalue_pairs params $args

    set badchars "\[ \t;\]"
    if {[regexp $badchars $name] || [regexp $badchars $value]} {
        return -code error "name/value pair \"$name=$value\" may not contain semicolons, spaces, or tabs"
    }
    set cookieKey "Set-cookie"
    set cookieValue "$name=$value"

    append cookieValue [make_cookie_attributes params]

    set_header $cookieKey $cookieValue
}

safe_and_trusted_proc parray {arrayName} {
    upvar 1 $arrayName array
    html "<pre><b>$arrayName</b>\n"
    foreach element [lsort [array names array]] {
	html "$element = $array($element)\n"
    }
    html "</pre>"
}

safe_proc dump_environment {} {
    global webenv
    parray webenv
}

safe_proc backlink {{linktext "Back"} {nolinktext ""}} {
    global webenv
    if ![info exists webenv(HTTP_REFERER)] {
	return $nolinktext
    }
    return "<a href=\"$webenv(HTTP_REFERER)\">$linktext</a>"
}

# nph_set_header and nph_send_headers are used in nph-*
# files to send out headers.  Usage is:
#
# nph_set_header key value
# ...
# nph_send_headers

safe_proc nph-set-header {headerKey headerValue} {
    global headers
    set headers($headerKey) $headerValue
}

safe_proc nph-send-headers {} {
    global headers webenv

    if ![info exists headers(Date)] {
	set headers(Date) [clock format [clock seconds]]
    }
    if ![info exists headers(Server)] {
	set headers(Server) $webenv(SERVER_SOFTWARE)
    }
    if ![info exists headers(Content-type)] {
	set headers(Content-type) "text/html"
    }
    html "HTTP/1.0 200 OK\n"
    html "Date: $headers(Date)\n"
    html "Server: $headers(Server)\n"
    html "Content-type: $headers(Content-type)\n"
    foreach key [array names headers] {
	if [lsearch {Date Server Content-type} $key]==-1 {
	    html "$key: $headers($key)\n"
	}
    }
    html "\n"
}

#
# Pick an element in a list at random and return it
# Now uses keyvalue pairs and will allow a text file as its list
#
safe_proc random_pick_html {args} {
    if {[cindex $args 0] == "-"} {
	import_keyvalue_pairs data $args
    }

    if [info exists data(file)] {
	set filename $data(file)
	if ![isfile_safe $filename] {
	    return -code error "$filename: Invalid pathname requested"
	}

        if [file exists $filename] {
            set fp [open $filename]
            set list ""
            while {[gets $fp line] != -1} {
                lappend list $line
            }
            close $fp
        } else {
            return -code error "$data(file): file does not exist"
        }
    } else {
	set list [lindex $args 0]
    }

    return [lindex $list [expr [clock seconds] % [llength $list]]]
}

#
# Increment a page counter based on the DOCUMENT_URI and return the result.
# Start it at zero if it doesn't exist (i.e. it returns 1 on the first
# try.)
#
safe_proc incr_page_counter {{start 0}} {
    global webenv
    if {![dbfetch pagecounters $webenv(DOCUMENT_URI) accessInfo] \
      || ![info exists accessInfo(counter)] \
      || $start > $accessInfo(counter)} {
	set accessInfo(counter) $start
    }
    set counter [incr accessInfo(counter)]
    dbstore pagecounters $webenv(DOCUMENT_URI) accessInfo
    return $counter
}
