# $Id: picproc.tcl,v 1.33 94/02/14 14:54:57 mangin Exp $

################################################################
####			Picasso                             ####
####			                                    ####
####	      Lucien Nocera & Frank Mangin                  ####
####			                                    ####
####	      Wed Dec  8 21:52:51 MET 1993                  ####
####			                                    ####
################################################################

set TmpNum 0
set ProcessId [id process]

################################################################
####		    Signal handling                         ####
################################################################

signal trap {SIGTERM SIGQUIT SIGINT} { cleanExit %S }

proc cleanExit {args} {
  global TmpDir

  if {$args != ""} {
    #  Signal generated exit  #
    puts stderr "caught signal $args - cleaning up"
    catch {exec rm [sglob "$TmpDir/picasso[id process]*"]}
    set dumpfile [mkTmpFile Dump]
    if {[catch {SaveDrawingInFile $dumpfile}]} {
      puts stderr "Couldn't dump drawing"
      catch {exec rm $dumpfile}
    } else {
      puts stderr "Drawing dumped to $dumpfile"
    }
    update
    destroy .
    exit 1
  } else {
    catch {exec rm [sglob "$TmpDir/picasso[id process]*"]}
    destroy .
  }
}

################################################################
####	       Useful general procedures                    ####
################################################################

########  Wait cursor  ########

set savedCursor {}

proc setCursor {name} {
  global savedCursor

  if {[lindex [. configure -cursor] 4] != "watch"} {
    set savedCursor [lindex [. configure -cursor] 4]
  }
  . configure -cursor "$name"
  update
}

proc waitCursor {} {
  global savedCursor

  if {[lindex [. configure -cursor] 4] != "watch"} {
    set savedCursor [lindex [. configure -cursor] 4]
  }
  . configure -cursor watch
  update
}

proc restoreCursor {} {
  global savedCursor
  . configure -cursor $savedCursor
  update
}
  
########  Temporary file names  ########
proc mkTmpFile {base} {
  global TmpNum TmpDir

  return "$TmpDir/picasso[id process]${base}[incr TmpNum]"
}
  
########  Safe glob  ########
proc sglob {path} {
  set new [glob -nocomplain -- $path]
  if {[lempty $new]} {
    set new [glob -nocomplain -- [file dirname $path]]
    if {[lempty $new]} {
      return $path
    } else {
      return "$new/[file tail $path]"
    }
  } else {
    return $new
  }
}

####  Utils  ####

proc cv {} {
  return .frame2.frame22.canvas
}

proc st {} {
  return .optionFr.styleFr.object
}

proc fontCv {} {
  return .optionFr.fontFr.showCv
}

proc al {} {
  return .optionFr.alignFr.f.frame36.canvas41
}

# Procedure Quit

proc Quit {} {
  global lastSavedState
  
  set cstate [getCurrentState]
  if {($lastSavedState != $cstate) &&
    ($cstate != {{} {} {} {}})} {
      ConfirmBox "Save current drawing before exiting?" \
	{save} { SaveDrawing } \
	{discard} { cleanExit} \
	{cancel}  { return }
    } else {
      cleanExit
    }
}

#  lvartail  #

proc lvartail {varname} {
  upvar 1 $varname var

  set l [expr "[llength $var] - 1"]
  set elt [lindex $var $l]
  incr l -1
  set var [lrange $var 0 $l]
  return $elt
}

#  llast  #

proc llast {l} {
  return [lindex $l [expr "[llength $l]-1"]]
}

################################################################
####		 Geometric calculations                     ####
################################################################

##  Pour remettre les coordonnees ds l'ordre  ##
proc dragMotion {tagOrId x y} {
  global x1fixed y1fixed

  set bb [[cv] coords $tagOrId]
  
  if {$bb != ""} {
    if {$x1fixed} {
      set x1 [lindex $bb 0]
      if {$x < $x1} {
	set newx1 $x
	set newx2 $x1
	set x1fixed 0
      } else {
	set newx1 $x1
	set newx2 $x
      }
    } else {
      set x2 [lindex $bb 2]
      if {$x < $x2} {
	set newx1 $x
	set newx2 $x2
      } else {
	set newx1 $x2
	set newx2 $x
	set x1fixed 1
      }
    }
    
    if {$y1fixed} {
      set y1 [lindex $bb 1]
      if {$y < $y1} {
	set newy1 $y
	set newy2 $y1
	set y1fixed 0
      } else {
	set newy1 $y1
	set newy2 $y
      }
    } else {
      set y2 [lindex $bb 3]
      if {$y < $y2} {
	set newy1 $y
	set newy2 $y2
      } else {
	set newy1 $y2
	set newy2 $y
	set y1fixed 1
      }
    }
    
    [cv] coords $tagOrId $newx1 $newy1 $newx2 $newy2
  }
}

##  Given a point and a bounding box, decide on which corner  ##
##  or side we are                                            ##

proc cornerOrSide {x y bb} {
  set x1 [lindex $bb 0]
  set y1 [lindex $bb 1]
  set x2 [lindex $bb 2]
  set y2 [lindex $bb 3]
  
  set dx1 [expr "($x-$x1)*($x-$x1)"]
  set dy1 [expr "($y-$y1)*($y-$y1)"]
  set dx2 [expr "($x-$x2)*($x-$x2)"]
  set dy2 [expr "($y-$y2)*($y-$y2)"]
  
  set d11 [expr "$dx1 + $dy1"]
  set d12 [expr "$dx1 + $dy2"]
  set d21 [expr "$dx2 + $dy1"]
  set d22 [expr "$dx2 + $dy2"]
  
  set dmin $d11
  set xid 1
  set yid 1
  
  if {$d12 < $dmin} {
    set dmin $d12
    set xid 1
    set yid 2
  }
  if {$d21 < $dmin} {
    set dmin $d21
    set xid 2
    set yid 1
  }
  if {$d22 < $dmin} {
    set dmin $d22
    set xid 2
    set yid 2
  }
  
  if { $xid == 1 } {
    set xside [expr "($x1 != $x2) && ($x - $x1+0.0)/($x2 - $x1) > 0.33"]
  } else {
    set xside [expr "($x1 != $x2) && ($x - $x2+0.0)/($x1 - $x2) > 0.33"]
  }
  
  if { $yid == 1 } {
    set yside [expr "($y1 != $y2) && ($y - $y1+0.0)/($y2 - $y1) > 0.33"]
  } else {
    set yside [expr "($y1 != $y2) && ($y - $y2+0.0)/($y1 - $y2) > 0.33"]
  }
  
  if { $yside && $xside } { return none }
  if { $xside } { return "sx$yid" }
  if { $yside } { return "sy$xid" }
  
  return "c$xid$yid"
}

################################################################
####		 Les modes operatoires                      ####
################################################################

proc globalBindings {} {
  global theBindings
  #  Help binding  #
  bind all <ButtonPress-3> { help %W }

  set bindList [split $theBindings "\n"]
  foreach spec $bindList {
    if {[catch {lindex $spec 0} event] ||
      [catch {lindex $spec 1} cmd]} { continue }
    if {![regexp {<[^>]+>} $event]} { continue }
    bind . $event "$cmd"
  }
}

################################################################
####		     Les mode hooks                         ####
################################################################

proc ModeEntryHook {mode} {
  global ModeFancyNames entryFillColor ModeState

  set info [keylget ModeFancyNames $mode]
  set ModeState 0
  
  switch $mode {
    ArcMode -
    PieMode {
      set info "$info : set center with mouse button 1"
    }
    CloseCurveMode -
    CloseLineMode {
      ##  set a fillColor if there is none since polygons  ##
      ##  have no outline option  ##
      global currentFillColor entryFillColor

      #  save it for exit hook  #
      set entryFillColor $currentFillColor
      if {$currentFillColor == {}} {
	deselect_all
	SetStyle fillColor black
      }
    }
    ReshapeMode {
      global Shaping ShapeId
      set Shaping 0
      set ShapeId {}
      deselect_all
    }
  }

  msg $info
}

proc ModeExitHook {mode} {
  global entryFillColor
  
  switch $mode {
    CloseCurveMode -
    CloseLineMode {
      ##  restore the fillColor  ##
      global entryFillColor
      deselect_all
      SetStyle fillColor $entryFillColor
    }
    PieMode -
    ArcMode {
      global ModeState
      set ModeState 0
      [cv] delete inCreation
      [cv] delete carcl1
      [cv] delete carcl2
    }
    ReshapeMode {
      global Shaping ShapeId
      set Shaping 0
      set ShapeId {}
      [cv] delete rsHandle
    }
    TextMode { ctextEnd }
    LatexMode { clatexEnd }
  }
}

################################################################
####		     Action Bindings                        ####
################################################################
proc Raise {} {
  ApplyOnSelection raise
}

proc Lower {} {
  ApplyOnSelection lower
}

proc Delete {} {
  ApplyOnSelection delete
}

proc ApplyOnSelection {action} {
  global CurrentMode
  
  [cv] addtag toApply withtag sOfr
  [cv] addtag toApply withtag inCreation
  
  if {[[cv] find withtag toApply] == {}} {
    warn "No item selected"
  } else {
    [cv] $action toApply
    [cv] dtag all toApply
  }

  [cv] raise ltxFr
  [cv] raise latex
}

################################################################
####		     Current Style                          ####
################################################################

##  There is one entry par picasso object type in currentStyle  ##
##  Each entry is the option line to apply to this type of object  ##

set currentStyle(rectangle)  {}
set currentStyle(frectangle) {} 
set currentStyle(orectangle) {} 
set currentStyle(drectangle) {}
set currentStyle(dline)      {}
set currentStyle(line)       {}
set currentStyle(curve)      {}
set currentStyle(oval)       {}
set currentStyle(doval)      {}
set currentStyle(arc)        {}
set currentStyle(pieslice)   {}
set currentStyle(polygon)    {}
set currentStyle(text)       {}
set currentStyle(image)      {}
set currentStyle(bitmap)     {}

##
##  Mapping between picasso styles
##  and canvas item config parameters
##

set Style2Config {
  {rectangle {
    {lineColor 		-outline	}
    {fillColor 		-fill		}
    {lineWidth		-width		}
    {stipple		-stipple	}}}
  {frectangle {
    {fillColor 		-fill		}
    {stipple		-stipple	}}}
  {orectangle {
    {lineColor 		-outline	}
    {lineWidth		-outlinewidth	}
    {dashes		-dashes		}}}
  {drectangle {
    {lineColor 		-outline	}
    {fillColor 		-fill		}
    {lineWidth		-width		}
    {stipple		-stipple	}
    {dashes		-dashes		}}}
  {dline {
    {lineColor 		-fill		}
    {lineWidth		-width		}
    {arrow		-arrow		}
    {dashes		-dashes		}}}
  {line {
    {lineColor 		-fill		}
    {lineWidth		-width		}
    {arrow		-arrow		}}}
  {curve {
    {lineColor 		-fill		}
    {lineWidth		-width		}
    {arrow		-arrow		}}}
  {oval {
    {lineColor 		-outline	}
    {fillColor 		-fill		}
    {lineWidth		-width		}
    {stipple		-stipple	}}}
  {doval {
    {lineColor 		-outline	}
    {fillColor 		-fill		}
    {lineWidth		-width		}
    {dashes		-dashes		}}}
  {arc {
    {lineColor 		-fill		}
    {lineWidth		-width		}}}
  {pieslice {
    {lineColor 		-outline	}
    {fillColor 		-fill		}
    {lineWidth		-width		}
    {stipple		-stipple	}}}
  {polygon {
    {fillColor 		-fill		}
    {stipple		-stipple	}}}
  {text {
    {lineColor 		-fill		}
    {font 		-font		}}}
  {bitmap {
    {lineColor 		-foreground	}
    {fillColor		-background	}}}
}

################

proc styleOverride {type opt value} {
  global currentStyle
  
  set i [lsearch -exact $currentStyle($type) $opt]
  if {$i < 0} {
    lappend currentStyle($type) $opt "$value"
  } else {
    incr i
    set currentStyle($type) [lreplace $currentStyle($type) $i $i "$value"]
  }
}

proc SetStyle {style value} {
  global currentStyle currentFillColor Style2Config
  ##  style is picasso defined; we translate it  ##
  ##  into appropriate canvas item options  ##

  if {$style == "fillColor"} {
    set currentFillColor "$value"
  }
  
  ##  Apply style on font displayer object  ##
  foreach item [[fontCv] find withtag all] {
    if {[keylget Style2Config \
	 [[fontCv] type $item].$style switch]} {
	   [fontCv] itemconfigure $item \
	     $switch $value
	   #  recenter object  #
	   [fontCv] coords $item \
	     [expr "[winfo width [fontCv]]/2"] \
	     [expr "[winfo height [fontCv]]/2"]
	   [fontCv] itemconfigure $item -anchor center
	 }
  }
  
  ##  Apply style on style viewer objects  ##
  foreach item [[st] find withtag show] {
    if {[keylget Style2Config \
	 [[st] type $item].$style switch]} {

	   [st] itemconfigure $item \
	     $switch $value
	 }
  }
  
  ##  Apply current style on selected objects  ##
  ##  and on objects being created  ##

  set items [concat [[cv] find withtag s] [[cv] find withtag inCreation]]

  foreach item $items {

    set type [[cv] type $item]

    if {($type == "arc") &&
      ([lindex [[cv] itemconfigure $item -style] 4] == "pieslice")} {
	set type pieslice
      }

    if {[keylget Style2Config $type.$style switch]} {
      [cv] itemconfigure $item $switch $value
    }
  }

  ##  Register change in currentStyle array  ##
  foreach type [keylget Style2Config] {
    if {[keylget Style2Config $type.$style switch]} {
      styleOverride $type $switch $value
    }
  }
  
  redrawFrames
}

################################################################
####		       OPERATIONS                           ####
################################################################

proc dragStartHook {} {
  grab [cv]
  focus [cv]
}

proc dragEndHook {} {
  grab release [cv]
  focus .
}

################################################################
####		       Selection                            ####
################################################################

##  TAGS : - s      => is selected and isn't frame  ##
##  - sOfr => is selected or is a select frame  ##
##  - sFr   => is a select frame  ##
##  - firstS => reference for align ##

proc notAFrame {item} {
  return [expr "[lsearch -exact [[cv] gettags $item] {sFr}] < 0"]
}

########  Select frames  ########

proc putFrame {gtag} {
  global sFrId

  lassign [[cv] bbox $gtag] x1 y1 x2 y2
  if {$x1 == ""} { return }
  incr x1 -1
  incr y1 -1
  incr x2
  incr y2
  return \
    [set sFrId($gtag) \
     [[cv] create rectangle \
      $x1 $y1 $x2 $y2 \
	-outline red -width 1 \
	-tags [list sOfr sFr x$gtag ctl]]]
}

##  Redraw frames around selected groups  ##

proc redrawFrames {} {
  
  [cv] delete sFr
  foreach gtag [getTopGroupTags s] {
    putFrame $gtag
  }
}

########  ########

set x1fixed 1
set y1fixed 1

proc deselect_all {} {
  [cv] delete   sFr
  [cv] dtag all s
  [cv] dtag all sOfr
  [cv] dtag all firstS
}

proc SelectAll {} {

  deselect_all

  ##  add selection tags and store existing groups  ##

  set gtags {}
  foreach item [[cv] find withtag all] {
    [cv] addtag s withtag $item
    [cv] addtag sOfr withtag $item
    lappend gtags [getTopGroupTags $item]
  }

  ## put a frame around each existing group ##
  foreach gtag [lrmdups $gtags] {
    putFrame $gtag
  }
}

proc iselect {x y} {
  global xdragAnchor ydragAnchor x1fixed y1fixed
  
  deselect_all

  set x [[cv] canvasx $x]
  set y [[cv] canvasy $y]
  
  if {([set item [[cv] find withtag current]] != "") &&
    [notAFrame current]} {
      ##  single item selection  ##
      set gtag [getTopGroupTags $item]
      [cv] addtag s withtag $gtag
      [cv] addtag sOfr withtag $gtag
      [cv] addtag firstS withtag $item
      putFrame $gtag
    } else {
      ##  no item -> start region selection  ##
      dragStartHook
      [cv] create rectangle $x $y $x $y \
	-outline red -tags {regionsFr}
      set x1fixed 1
      set y1fixed 1
    }
}

proc iselectRegion {x y} {
  dragMotion regionsFr [[cv] canvasx $x] [[cv] canvasy $y]
}

proc iselectRegionEnd {} {

  set bb [[cv] coords regionsFr]
  [cv] delete regionsFr
  
  if {$bb != ""} {
    #  find items enclosed in region  #
    set into [eval "[cv] find enclosed $bb"]
    #  find groups existing in these items  #
    set intogtags {}
    foreach item $into {
      lappend intogtags [getTopGroupTags $item]
    }
    #  only keep groups entirely included in region  #
    #  and corresponding items #
    set allintogtags {}
    set allinto {}
    foreach gtag [lrmdups $intogtags] {
      set gcontent [[cv] find withtag $gtag]
      if {[llength [intersect $gcontent $into]] == [llength $gcontent]} {
	lappend allintogtags $gtag
	lappend allinto $gcontent
      }
    }
    foreach gtag $allintogtags {
      [cv] addtag s withtag $gtag
      [cv] addtag sOfr withtag $gtag
      putFrame $gtag
    }
  }
  dragEndHook
}            

proc iselectAugment {} {
  global sFrId

  set item [[cv] find withtag current]
  if {($item != "") && [notAFrame $item]} {
    if {[lsearch [[cv] gettags $item] s] < 0} {
      #  select it  #
      set gtag [getTopGroupTags $item]
      [cv] addtag s withtag $gtag
      [cv] addtag sOfr withtag $gtag
      putFrame $gtag
    } else {
      #  deselect it  #
      set gtag [getTopGroupTags $item]
      [cv] delete withtag $sFrId($gtag)
      [cv] dtag $gtag s
      [cv] dtag $gtag sOfr
    }
  }
}

################################################################
####		    Drag management                         ####
################################################################

set Dragging 0

proc select_for_drag {} {
  global Dragging

  if {[set item [[cv] find withtag current]] != ""} {
    set tags [[cv] gettags $item]
    if {([lsearch -exact $tags "ctl"] < 0) &&
      [lsearch -exact $tags s] < 0} {
	set gtag [getTopGroupTags $item]
	[cv] addtag sForDrag withtag $gtag
	[cv] addtag s withtag $gtag
	[cv] addtag sOfr withtag $gtag
	[cv] addtag forDragFrame withtag [putFrame $gtag]
      }
  }

  set ids [[cv] find withtag s]
  if {[lempty $ids]} {
    set Dragging 0
    return ""
  } else {
    set Dragging 1
    return $ids
  }
}

proc deselect_for_drag {} {
  global Dragging
  
  if {$Dragging} {
    set Dragging 0
    [cv] delete forDragFrame
    [cv] dtag sForDrag s
    [cv] dtag sForDrag sOfr
    [cv] dtag all sForDrag
  }
}

