/**********************************************************************
  
  This file is part of the Bita-package, a Tcl/Tk-extension
  implementing binary typed array.

	 Copyright 1995,1996 Harald Kirsch (kir@iitb.fhg.de)

**********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <bitaP.h>
#include <util.h>

/**********************************************************************
  bitaCopyCmd -- copy (insert) values from one bita into another

  Called from bitaCmd according to argv[1]. The bita comes in
  ClientData. 

  SYNOPSIS:
  bita "copy" dstIndex srcBita ?from? ?to?

  RETURN(tcl):
  -- the name of the bita.
  
*****/
int
bitaCopyCmd(Bita b,
	    Tcl_Interp *ip,
	    int argc,
	    char **argv)
{
  unsigned elemSize = b->class->elemSize;
  Tcl_CmdInfo info;
  Bita srcB;

  unsigned dstIndex, start, end, count, newLen;
  char *tmp;
  char tmpAlloced;

  if( argc<4 || argc>6) {
    Tcl_AppendResult(ip, 
		     "wrong # args: should be `",
		     argv[0],
		     " copy dstIndex srcBita ?from? ?to?'", NULL);
    return TCL_ERROR;
  }

  /***** get the dstIndex */
  if( !checkedIndex(ip, argv[2], &dstIndex, b->length) )
    return TCL_ERROR;

  /***** get the srcBita (the deleteProc test might be a bit hacky) */
  if( !Tcl_GetCommandInfo(ip, argv[3], &info)
      || info.deleteProc!=(Tcl_CmdDeleteProc*)&bitaFree) {
    Tcl_AppendResult(ip, argv[3], " is no binary typed array", NULL);
    return TCL_ERROR;
  }

  srcB = (Bita)info.clientData;
  if( b->class != srcB->class ) {
    Tcl_AppendResult(ip, argv[3], " is not of the same type as ",
		     argv[0], NULL);
    return TCL_ERROR;
  }

  if( 0==srcB->length ) {
    Tcl_AppendResult(ip, argv[3], " is empty", NULL);
    return TCL_ERROR;
  }

  if( argc>4 ) {
    if( !checkedIndex(ip, argv[4], &start, srcB->length-1) ) 
      return TCL_ERROR;
  } else {
    start = 0;
  }

  if( argc>5 ) {
    if( !checkedIndex(ip, argv[5], &end, srcB->length-1) )
      return TCL_ERROR;
    end += 1;
  } else {
    end = srcB->length;
  }

  if( start>end ) {
    Tcl_SetResult(ip, 
		  "`from'-index may not be greater than `to'-index",
		  TCL_STATIC);
    return TCL_ERROR;
  }
  count = end - start;

  /***** 
    If both bitas are the same, we need to make a temporary copy of
    src. 
  *****/
  if( b==srcB ) {
    tmp = malloc(elemSize*count);
    if( !tmp ) {
      Tcl_SetResult(ip, "out of memory", TCL_STATIC);
      return TCL_ERROR;
    }
    memcpy(tmp, srcB->v+elemSize*start, elemSize*count);
    tmpAlloced = 1;
  } else {
    tmp = srcB->v+elemSize*start;
    tmpAlloced = 0;
  }

  /***** possibly realloc */
  newLen = b->length+count;
  if( b->size < newLen ) {
    char *t = realloc(b->v, newLen*elemSize);
    if( !t ) {
      Tcl_SetResult(ip, "out of memory", TCL_STATIC);
      if( tmpAlloced) free(tmp);
      return TCL_ERROR;
    }
    b->v = t;
    b->size = newLen;
  }

  /***** insert the values */
  memmove(b->v + elemSize*(dstIndex+count),
	  b->v + elemSize*dstIndex,
	  elemSize*(b->length-dstIndex));
  memcpy(b->v + elemSize*dstIndex, tmp, elemSize*count);
  b->length += count;

  if( tmpAlloced ) free(tmp);

  Tcl_SetResult(ip, argv[0], TCL_VOLATILE);
  return TCL_OK;
}
/**********************************************************************/
