/**********************************************************************
  
  This file is part of the Bita-package, a Tcl/Tk-extension
  implementing binary typed array.

	 Copyright 1995,1996 Harald Kirsch (kir@iitb.fhg.de)

**********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <bitaP.h>
#include <util.h>

/**********************************************************************
  bitaInsertCmd -- insert values given as tcl-list into bita.
  bitaSetCmd  -- overwrite bita wih values given as tcl-list

  Called from bitaCmd according to argv[1]. The bita comes in
  ClientData. 

  SYNOPSIS:
  bita "insert" dstIndex values
  bita "set" dstIndex values

  RETURN(tcl):
  -- the name of the bita.

  PRE:
  The calling function must make sure that argv[1] is exactly "insert"
  or "set".  
*****/
int
bitaInsertCmd(Bita b,
	      Tcl_Interp *ip,
	      int argc,
	      char **argv)
{
  unsigned elemSize = b->class->elemSize;

  unsigned dstIndex, newLen;
  int listc;
  char **listv;
  int i;
  char *tmp, *v;
  static char Tmp[32];	/* may save a call to malloc/free on most */
			/* cases */

  if( argc!=4 ) {
    Tcl_AppendResult(ip, 
		     "wrong # args: should be `",
		     argv[0], " ", argv[1],
		     " dstIndex values'", NULL);
    return TCL_ERROR;
  }

  /***** get the dstIndex */
  if( !checkedIndex(ip, argv[2], &dstIndex, b->length) )
    return TCL_ERROR;

  if( TCL_OK!=Tcl_SplitList(ip, argv[3], &listc, &listv) ) 
    return TCL_ERROR;

  /***** 
    allocate temporary, because we don't want to change anything
    in case of errors.
  *****/
  if( elemSize*listc <= 32 ) {
    tmp = Tmp;
  } else {
    tmp = malloc(elemSize*listc);
    if( !tmp ) {
      Tcl_SetResult(ip, "out of memory", TCL_STATIC);
      free((char*)listv);
      return TCL_ERROR;
    }
  }

  /***** convert all values */
  for(i=0, v=tmp; i<listc; i++, v+=elemSize) {
    if( TCL_OK!=b->class->stringToElem(listv[i], v, ip) ) {
      if( tmp!=Tmp ) free(tmp);
      free((char*)listv);
      return TCL_ERROR;
    }
  }
  free((char*)listv);

  /***** new length depending on "insert"/"set" command */
  if( argv[1][0]=='s' ) {
    newLen = dstIndex + listc;
    if( newLen < b->length ) newLen = b->length;
  } else {
    newLen = b->length+listc;
  }

  /***** possibly realloc */
  if( b->size < newLen ) {
    char *t = realloc(b->v, newLen*elemSize);
    if( !t ) {
      Tcl_SetResult(ip, "out of memory", TCL_STATIC);
      free(tmp);
      return TCL_ERROR;
    }
    b->v = t;
    b->size = newLen;
  }

  /***** open a gap, if "insert"-command */
  if( argv[1][0]=='i' ) {
    memmove(b->v + elemSize*(dstIndex+listc),
	    b->v + elemSize*dstIndex,
	    elemSize*(b->length-dstIndex));
  }
  b->length = newLen;

  /***** enter the new values */
  memcpy(b->v + elemSize*dstIndex, tmp, elemSize*listc);

  if( tmp!=Tmp ) free(tmp);

  Tcl_SetResult(ip, argv[0], TCL_VOLATILE);
  return TCL_OK;
}
/**********************************************************************/
