/**********************************************************************
  
  This file is part of the Bita-package, a Tcl/Tk-extension
  implementing binary typed array.

	 Copyright 1995,1996 Harald Kirsch (kir@iitb.fhg.de)

**********************************************************************/
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <sys/stat.h>
#include <unistd.h>
#include <limits.h>

#include <bitaP.h>
#include <util.h>

/**********************************************************************
  bitaReadCmd -- read values from binary file into bita

  Called from bitaCmd according to argv[1]. The bita comes in
  ClientData. 

  SYNOPSIS:
  bita "read" dstIndex file count

  Read the given number of values (not bytes) from the given file and
  insert them at dstIndex.

  RETURN(tcl):
  There is a problem with short reads (e.g. on non-blocking devices or
  on short files). Tcl_Read may e.g. return only 2 bytes although the
  bita is of type Int. In this case, the irregular bytes are returned
  along with the result as two integers between 0 and 255.
  Example: {3 60 91}
  For a bita of type Int this means that Tcl_Read returned 3 full
  values and got two additional bytes with the decimal values 60 and
  91. 

*****/
int
bitaReadCmd(Bita b,
	    Tcl_Interp *ip,
	    int argc,
	    char **argv)
{
  unsigned elemSize = b->class->elemSize;
  unsigned long l;

  unsigned dstIndex, count, newLen;
  int byteCount, valueCount, restCount, mode;
  Tcl_Channel file;
  static char buf[30];	/* enough for everthing %u might print */

  /***** BEGIN */
  if( argc!=5 ) {
    Tcl_AppendResult(ip, 
		     "wrong # args: should be `",
		     argv[0],
		     " read dstIndex file count'", NULL);
    return TCL_ERROR;
  }
  /***** get dstIndex */
  if( !checkedIndex(ip, argv[2], &dstIndex, b->length) )
    return TCL_ERROR;

  /***** get the file */
  file = Tcl_GetChannel(ip, argv[3], &mode);
  if( !file) {
    return TCL_ERROR;
  }
  if( !(mode&TCL_READABLE) ) {
    Tcl_AppendResult(ip, "channel `", argv[3], 
		     "' wasn't opened for reading", NULL);
    return TCL_ERROR;
  }

  /***** get the count */
  if( pickyUnsigned(argv[4], &l) || l>UINT_MAX) {
    Tcl_AppendResult(ip, "wrong count `", argv[4], "'", NULL);
    return TCL_ERROR;
  }
  count = (unsigned)l;

  /***** possibly realloc */
  newLen = b->length+count;
  if( b->size < newLen ) {
    char *tmp = realloc(b->v, newLen*elemSize);
    if( !tmp ) {
      Tcl_SetResult(ip, "out of memory", TCL_STATIC);
      return TCL_ERROR;
    }
    b->v = tmp;
    b->size = newLen;
  }

  /***** 
    Because of bloody MessyDos compatibility, Tcl_Read may be in
    translation mode (mood) meant to mess around with pairs of bytes
    which happen to have the values 0x0D and 0x0A. We now silently
    switch to binary mode and reset this mess afterwards.
  *****/
  if( TCL_OK!=bitaSetChannelMode(ip, file) ) return TCL_ERROR;

  /***** make room */
  memmove(b->v + elemSize*(dstIndex+count),
	  b->v + elemSize*dstIndex,
	  elemSize*(b->length-dstIndex));

  /***** READ */
  byteCount = Tcl_Read(file, b->v + elemSize*dstIndex,
		       elemSize*count);

  /***** before we check for a read error, we have to clean up the mess */
  if( byteCount<0 ) {
    valueCount = 0;
    restCount = 0;
  } else {
    valueCount = byteCount/elemSize;
    restCount = byteCount%elemSize;
  }
  sprintf(buf, "%d", valueCount);
  Tcl_AppendResult(ip, buf, NULL);

  if( restCount>0 ) {
    /***** rescue superfluous bytes */
    int i;
    unsigned char *start = b->v+elemSize*(dstIndex+valueCount);
    for(i=0; i<restCount; i++, start++) {
      sprintf(buf, " %d", (int)(*start));
      Tcl_AppendResult(ip, buf, NULL);
    }
  }

  if( valueCount<count ) {
    /***** close the gap */
    memmove(b->v + elemSize*(dstIndex+valueCount),
	    b->v + elemSize*(dstIndex+count),
	    elemSize*(b->length-dstIndex) );
  }
  b->length += valueCount;

  /***** now that the bita is stable, lets reset the channel mood */
  if( TCL_OK!=bitaResetChannelMode(ip, file) ) {
    if( byteCount>=0 )return TCL_ERROR;
  }

  /***** only now we can check for a read error */
  if( byteCount<0 ) {
    Tcl_FreeResult(ip);
    Tcl_AppendResult(ip, "error reading `",
		     argv[3], "': ",
		     Tcl_PosixError(ip), NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}

  
/**********************************************************************/
