/* 
 * SSLtcl-0.42.c
 *
 * This is SSLtcl - a loadable module for tcl that extends the sockate command
 * with SSL capabilitys. It works with tcl8.0 and SSleay-0.6.6 or SSLeay-0.8.1.
 * It is inspired by Bringgardners tclSSL, but with more functionality.
 *
 * Unix sockets that support SSL.  Much of this file is taken from 
 *      tclUnixChan.c.  My intent was to use as much of the core tcp
 *      code as posible.  The since many of the functions defined in
 *      tclUnixChan.c are static and I want to create a loadable
 *      module, I need to include them here but they
 *      are unchanged.
 *
 * Much of this file is taken from tclUnixChan.c and tclIOCmd.c in the same
 * way as tclSSL but with tcl8.0 as the base. 
 *
 * Copyright (c) 1997, 1998  Peter Antman
 * My part of the code in this file is licensed under
 * GNU/GPL
 *
 *This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file also contains code with following Copyrights:
 * Copyright (c) 1995-1997  Tony Bringardner
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 *
 * To do anyting usefull with it You have to link it against SSLeay, with
 * its special Copyright and Licencing.
 *
 * 
 *
 * load SSLtcl.so SSLtcl
 *
 * socket [-myaddr addr] [-myport myport] [-async] [-ssl [-verify NUM]
 * [-cert Certificate] [-key privateKey] [-CAfile CAfile] [-CAdir CAdir]
 * ] host port
 *
 * socket  -server command [-ssl -cert Certificate [-key privateKey]
 * [-verify NUM] [-Verify NUM] [-CAfile CAfile] [-CAdir CAdir] [-myaddr
 * addr] port
 *
 * DESCRIPTION
 *
 * SSLtcl creates a new tcl socket command with SSL capabilities. Se
 * socket(n) for a description of the tcl socket command. Used with -ssl
 * option it creates a SSL based connection. If -ssl is left out it
 * function as the normal socket command and only options described in
 * socket(n) is available. SSLtcls socket creates either a server SSL
 * socket or a client SSL socket, exactly as the normal socket command.
 *
 * CLIENT SOCKET
 *
 * If the -server option is not specified,  then  the  client side  of a
 * connection is opened and the command returns a channel identifier that
 * can be used for both  reading  and writing.   Port  and  host specify
 * a port to connect to;  there must be a server accepting connections on
 * this port. Port  is  an  integer  port  number  and  host is either a
 * domain-style name such as www.sunlabs.com or  a  numerical IP  address
 * such as 127.0.0.1.  Use localhost to refer to the host on which the
 * command is invoked. The following options may also be present before
 * host  to specify additional information about the connection:
 *
 *   -myaddr addr
 *       Addr  gives  the  domain-style name or numerical IP address of
 *       the client-side network interface to use for  the  connection.
 *       This option may be useful if the client machine has multiple
 *       network interfaces.  If  the  option  is  omitted  then  the
 *      client-side interface will be chosen by the system software.
 *
 *    -myport port
 *      Port specifies an integer port number  to  use  for the
 *      client's  side  of  the  connection.   If this option is
 *      omitted, the client's port number will be chosen at random by
 *      the system software.
 *
 *     -async
 *      The  -async  option will cause the client socket to be connected
 *      asynchronously. This  means  that  the socket  will be created
 *      immediately but may not yet be connected to the server, when the
 *      call to socket returns. When a gets or flush is done on the
 *      socket before the connection attempt succeeds or fails, if the
 *      socket is in blocking mode, the operation will wait until the
 *      connection is completed or fails. If the  socket  is  in
 *      nonblocking mode and a gets or flush is done on the socket
 *      before  the  connection attempt  succeeds  or  fails, the
 *      operation returns immediately and fblocked on the socket returns
 *      1.
 *
 *   -ssl
 *      The -ssl option turns SSL on an makes the following options
 *      available.
 *
 *   -verify num
 *      Turns verification of server on. num is an integer that
 *      specifies how deep in the certificate chain verification is
 *      required to succeed.
 *
 *   -key privateKey
 *      Specifies the private key that accompanies the own certificate.
 *      A certificate is only required for the client if it is going to
 *      be verified by the server. If left out will be set to the same
 *      value as given for -cert.
 *
 *   -cert Certificate
 *      The client certificate, pem encoded, that it will send to the
 *      server if required.
 *
 *    -CAfile CAfile
 *      A file containing certificates from Certificate Authorities that
 *      the client wants to trust. To verify a server a CAfile, or a
 *      CAdir, is required.
 *
 *   -CAdir CAdir
 *      A hashed directory that contains certificates from Certificate
 *       Authorities that the client wants to trust. To verify a server a
 *      CAdir, or a CAfile, is required.
 *    -ciphers cipherlist
 *       cipherlist is a colon separated list of ciphers that the clients
 *       wants to support. The first one will be tried first, and so on.
 *       See Supported ciphers below.
 *
 * SERVER SOCKET
 *
 * If the -server option is specified  then  the  new  socket will  be a
 * server  for the port given by port.  Tcl will automatically accept
 * connections to the given  port.   For each  connection Tcl will create
 *  a new channel that may be used to communicate with the  client.   Tcl
 *  then  invokes command  with  three additional arguments: the name of
 * the new channel, the address, in network address notation,  of  the
 * client's host, and the client's port number. The  following
 * additional  option  may  also be specified before host:
 *
 *   -myaddr addr
 *      Addr gives the domain-style name  or  numerical  IP address of
 *      the server-side network interface to use for the connection.
 *      This option may be  useful  if the server machine has multiple
 *      network interfaces.  If the option is omitted then the server
 *      socket  is bound  to the special address INADDR_ANY so that it
 *      can accept connections from any interface.
 *
 *   -ssl
 *      The -ssl option turns SSL on an makes the following options
 *      available.
 *
 *   -verify num
 *      Turns verification of client on. num is an integer that
 *      specifies how deep in the certificate chain verification is
 *      required to succeed. Will only report verification result, not
 *      stop the connection
 *
 *   -Verify num
 *      Same as -verify, but will stop the connection if verify fails.
 *
 *   -key privateKey
 *      Specifies the private key that accompanies the server
 *      certificate.  If left out will be set to the same value as given
 *      for -cert.
 *
 * 
 *   -cert Certificate
 *      The server certificate, pem encoded; it is required.
 *
 *   -CAfile CAfile
 *      A file contaning certificates from Certificate Authorities that
 *  
 *    the server wants to trust. To verify a client a CAfile, or a
 *      CAdir, is required.
 *
 *   -CAdir CAdir
 *      A hashed directory that contains certificates from Certificate
 *      Authorities that the server wants to trust. To verify a client a
 *      CAdir, or a CAfile, is required.
 *
 *     -ciphers cipherlist
 *       cipherlist is a colon separated list of ciphers that the server
 *       wants to support. The first one will be tried first, and so on.
 *       See Supported ciphers below.
 *
 *    -maxacloops num
 *       Specififies the numer of times the server should try to accept a
 *       connection. num is an integer. The defaultvalue is 100000. The
 *       option exist because of a problem with non SSL aware clients and
 *       server socket in non -blocking mode. Without a timeout the
 *       server will block for ever. I will try to find a better solution
 *       in the furure. If releveant connections keeps returning with
 *       bgerror try to increase this value.
 *
 * Server channels cannot be used for input or output;  their sole use is
 * to accept new client connections. The channels created for each
 * incoming client connection are opened for input  and  output.  Closing
 * the server channel shuts down  the server so that no new connections
 * will  be  accepted; however, existing connections will be unaffected.
 * Server  sockets  depend on the Tcl event mechanism to find out when
 * new connections are opened.  If  the application doesn't  enter the
 * event loop, for example by invoking the vwait command or calling the C
 * procedure  Tcl_DoOneEvent,  then no connections will be accepted.
 *
 *  SUPPORTED CIPHERS
 *
 * The cipeherslist is specified as a colon separated list of ciphers.
 * The following cipherabrewations, and  is supported:
 *
 *
 *      RC4-MD5
 *      EXP-RC4-MD5
 *      IDEA-CBC-MD5
 *      RC2-CBC-MD5
 *      EXP-RC2-CBC-MD5
 *      DES-CBC-MD5
 *      DES-CBC-SHA
 *      DES-CBC3-MD5
 *      DES-CBC3-SHA
 *      DES-CFB-M1
 *      NULL-MD5
 *
 * If You link SSLtcl against SSLeay-0.8.1 have a look in ssl.h for adi-
 * tional ciphers.
 */



#include        <tcl.h>
#include	"tclInt.h"	/* Internal definitions for Tcl. */
#include	"tclPort.h"	/* Portability features for Tcl. */
#include	<ssl.h>		/* SSL include  */
#include        <pem.h>         /* SSL include */
#include <x509.h>
#include <err.h>
#include <crypto.h>
/* #include <evp.h> */
#include <dh.h>
/* #include <bio.h> */
#ifdef  USE_FIONBIO


#       include <sys/ioctl.h>   /* For FIONBIO. */
#endif  /* USE_FIONBIO */

#if SSLEAY_VERSION_NUMBER < 0x600
#define ERR_print_errors_fp	ERR_print_errors
#endif

#if SSLEAY_VERSION_NUMBER < 0x0800
#define X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY \
		VERIFY_ERR_UNABLE_TO_GET_ISSUER
#define X509_V_OK VERIFY_OK
#define SSL_CTX_set_default_verify_paths(ctx) SSL_set_default_verify_paths(ctx)
#define SSL_CTX_load_verify_locations(ctx,cafile,capath) \
		 SSL_load_verify_locations(ctx,cafile,capath)
#define X509_verify_cert_error_string(error) X509_cert_verify_error_string(error)
#endif

#define TCLSSLCERT "tclSSL.pem"

#define SSL_CIPH "RC4-MD5:EXP-RC4-MD5:IDEA-CBC-MD5:RC2-CBC-MD5:EXP-RC2-CBC-MD5:DES-CBC-MD5:DES-CBC-SHA:DES-CBC3-MD5:DES-CBC3-SHA:DES-CFB-M1:NULL-MD5"


/*
 * sys/ioctl.h has already been included by tclPort.h.  Including termios.h
 * or termio.h causes a bunch of warning messages because some duplicate
 * (but not contradictory) #defines exist in termios.h and/or termio.h
 */
#undef NL0
#undef NL1
#undef CR0
#undef CR1
#undef CR2
#undef CR3
#undef TAB0
#undef TAB1
#undef TAB2
#undef XTABS
#undef BS0
#undef BS1
#undef FF0
#undef FF1
#undef ECHO
#undef NOFLSH
#undef TOSTOP
#undef FLUSHO
#undef PENDIN

#ifdef USE_TERMIOS
#   include <termios.h>
#else	/* !USE_TERMIOS */
#ifdef USE_TERMIO
#   include <termio.h>
#else	/* !USE_TERMIO */
#ifdef USE_SGTTY
#   include <sgtty.h>
#endif	/* USE_SGTTY */
#endif	/* !USE_TERMIO */
#endif	/* !USE_TERMIOS */

     /* FIXME
      * max_ac_lopp is used in WaitForConnect with SSL to have a timeout value
      * when the server socket is in non blocking mode and a none SSL aware
      * client tries to connect and sends someting. Othervise the loop
      * around SSL_accept will go on forever, locking the socket.
      * If I dont have a lopp around SSL_accept, a call from a none aware
      * client will lock up the server socket totaly untill the client
      * sends someting. That is, in non blocking mode.
      * This solution is not god. Om bad connections the time to finish a
      * relevant non blocking call might exeed the limit set here. The the
      * value of MAX_AC_LOOPS have to be higher.
      * This can also be done with the socket option -maxaclopps INT
      */

#define MAX_AC_LOOPS 100000

static int max_ac_loops = MAX_AC_LOOPS;

int verify_depth;

/*  The following are static functions in generic/tclIOCmd.c */
/* 
 * tclIOCmd.c --
 *
 *	Contains the definitions of most of the Tcl commands relating to IO.
 *
 * Copyright (c) 1995-1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tclIOCmd.c 1.119 97/07/25 20:49:23
 */


/*
 * Return at most this number of bytes in one call to Tcl_Read:
 */

#define	TCL_READ_CHUNK_SIZE	4096

/*
 * Callback structure for accept callback in a TCP server.
 */

typedef struct AcceptCallback {
    char *script;			/* Script to invoke. */
    Tcl_Interp *interp;			/* Interpreter in which to run it. */
} AcceptCallback;

/*
 * Static functions for this file:
 */

static void	AcceptCallbackProc _ANSI_ARGS_((ClientData callbackData,
	            Tcl_Channel chan, char *address, int port));
static void	RegisterTcpServerInterpCleanup _ANSI_ARGS_((Tcl_Interp *interp,
	            AcceptCallback *acceptCallbackPtr));
static void	TcpAcceptCallbacksDeleteProc _ANSI_ARGS_((
		    ClientData clientData, Tcl_Interp *interp));
static void	TcpServerCloseProc _ANSI_ARGS_((ClientData callbackData));
static void	UnregisterTcpServerInterpCleanupProc _ANSI_ARGS_((
		    Tcl_Interp *interp, AcceptCallback *acceptCallbackPtr));

/*
 *----------------------------------------------------------------------
 *
 * AcceptCallbackProc --
 *
 *	This callback is invoked by the TCP channel driver when it
 *	accepts a new connection from a client on a server socket.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Whatever the script does.
 *
 *----------------------------------------------------------------------
 */

static void
AcceptCallbackProc(callbackData, chan, address, port)
    ClientData callbackData;		/* The data stored when the callback
                                         * was created in the call to
                                         * Tcl_OpenTcpServer. */
    Tcl_Channel chan;			/* Channel for the newly accepted
                                         * connection. */
    char *address;			/* Address of client that was
                                         * accepted. */
    int port;				/* Port of client that was accepted. */
{
    AcceptCallback *acceptCallbackPtr;
    Tcl_Interp *interp;
    char *script;
    char portBuf[10];
    int result;

    acceptCallbackPtr = (AcceptCallback *) callbackData;

    /*
     * Check if the callback is still valid; the interpreter may have gone
     * away, this is signalled by setting the interp field of the callback
     * data to NULL.
     */
    
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {

        script = acceptCallbackPtr->script;
        interp = acceptCallbackPtr->interp;
        
        Tcl_Preserve((ClientData) script);
        Tcl_Preserve((ClientData) interp);

	TclFormatInt(portBuf, port);
        Tcl_RegisterChannel(interp, chan);
        result = Tcl_VarEval(interp, script, " ", Tcl_GetChannelName(chan),
                " ", address, " ", portBuf, (char *) NULL);
        if (result != TCL_OK) {
            Tcl_BackgroundError(interp);
	    Tcl_UnregisterChannel(interp, chan);
        }
        Tcl_Release((ClientData) interp);
        Tcl_Release((ClientData) script);
    } else {

        /*
         * The interpreter has been deleted, so there is no useful
         * way to utilize the client socket - just close it.
         */

        Tcl_Close((Tcl_Interp *) NULL, chan);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RegisterTcpServerInterpCleanup --
 *
 *	Registers an accept callback record to have its interp
 *	field set to NULL when the interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When, in the future, the interpreter is deleted, the interp
 *	field of the accept callback data structure will be set to
 *	NULL. This will prevent attempts to eval the accept script
 *	in a deleted interpreter.
 *
 *----------------------------------------------------------------------
 */

static void
RegisterTcpServerInterpCleanup(interp, acceptCallbackPtr)
    Tcl_Interp *interp;		/* Interpreter for which we want to be
                                 * informed of deletion. */
    AcceptCallback *acceptCallbackPtr;
    				/* The accept callback record whose
                                 * interp field we want set to NULL when
                                 * the interpreter is deleted. */
{
    Tcl_HashTable *hTblPtr;	/* Hash table for accept callback
                                 * records to smash when the interpreter
                                 * will be deleted. */
    Tcl_HashEntry *hPtr;	/* Entry for this record. */
    int new;			/* Is the entry new? */

    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
            "tclTCPAcceptCallbacks",
            NULL);
    if (hTblPtr == (Tcl_HashTable *) NULL) {
        hTblPtr = (Tcl_HashTable *) ckalloc((unsigned) sizeof(Tcl_HashTable));
        Tcl_InitHashTable(hTblPtr, TCL_ONE_WORD_KEYS);
        (void) Tcl_SetAssocData(interp, "tclTCPAcceptCallbacks",
                TcpAcceptCallbacksDeleteProc, (ClientData) hTblPtr);
    }
    hPtr = Tcl_CreateHashEntry(hTblPtr, (char *) acceptCallbackPtr, &new);
    if (!new) {
        panic("RegisterTcpServerCleanup: damaged accept record table");
    }
    Tcl_SetHashValue(hPtr, (ClientData) acceptCallbackPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * TcpAcceptCallbacksDeleteProc --
 *
 *	Assocdata cleanup routine called when an interpreter is being
 *	deleted to set the interp field of all the accept callback records
 *	registered with	the interpreter to NULL. This will prevent the
 *	interpreter from being used in the future to eval accept scripts.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Deallocates memory and sets the interp field of all the accept
 *	callback records to NULL to prevent this interpreter from being
 *	used subsequently to eval accept scripts.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static void
TcpAcceptCallbacksDeleteProc(clientData, interp)
    ClientData clientData;	/* Data which was passed when the assocdata
                                 * was registered. */
    Tcl_Interp *interp;		/* Interpreter being deleted - not used. */
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch hSearch;
    AcceptCallback *acceptCallbackPtr;

    hTblPtr = (Tcl_HashTable *) clientData;
    for (hPtr = Tcl_FirstHashEntry(hTblPtr, &hSearch);
             hPtr != (Tcl_HashEntry *) NULL;
             hPtr = Tcl_NextHashEntry(&hSearch)) {
        acceptCallbackPtr = (AcceptCallback *) Tcl_GetHashValue(hPtr);
        acceptCallbackPtr->interp = (Tcl_Interp *) NULL;
    }
    Tcl_DeleteHashTable(hTblPtr);
    ckfree((char *) hTblPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * TcpServerCloseProc --
 *
 *	This callback is called when the TCP server channel for which it
 *	was registered is being closed. It informs the interpreter in
 *	which the accept script is evaluated (if that interpreter still
 *	exists) that this channel no longer needs to be informed if the
 *	interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	In the future, if the interpreter is deleted this channel will
 *	no longer be informed.
 *
 *----------------------------------------------------------------------
 */

static void
TcpServerCloseProc(callbackData)
    ClientData callbackData;	/* The data passed in the call to
                                 * Tcl_CreateCloseHandler. */
{
    AcceptCallback *acceptCallbackPtr;
    				/* The actual data. */

    acceptCallbackPtr = (AcceptCallback *) callbackData;
    if (acceptCallbackPtr->interp != (Tcl_Interp *) NULL) {
        UnregisterTcpServerInterpCleanupProc(acceptCallbackPtr->interp,
                acceptCallbackPtr);
    }
    Tcl_EventuallyFree((ClientData) acceptCallbackPtr->script, TCL_DYNAMIC);
    ckfree((char *) acceptCallbackPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * UnregisterTcpServerInterpCleanupProc --
 *
 *	Unregister a previously registered accept callback record. The
 *	interp field of this record will no longer be set to NULL in
 *	the future when the interpreter is deleted.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Prevents the interp field of the accept callback record from
 *	being set to NULL in the future when the interpreter is deleted.
 *
 *----------------------------------------------------------------------
 */

static void
UnregisterTcpServerInterpCleanupProc(interp, acceptCallbackPtr)
    Tcl_Interp *interp;		/* Interpreter in which the accept callback
                                 * record was registered. */
    AcceptCallback *acceptCallbackPtr;
    				/* The record for which to delete the
                                 * registration. */
{
    Tcl_HashTable *hTblPtr;
    Tcl_HashEntry *hPtr;

    hTblPtr = (Tcl_HashTable *) Tcl_GetAssocData(interp,
            "tclTCPAcceptCallbacks", NULL);
    if (hTblPtr == (Tcl_HashTable *) NULL) {
        return;
    }
    hPtr = Tcl_FindHashEntry(hTblPtr, (char *) acceptCallbackPtr);
    if (hPtr == (Tcl_HashEntry *) NULL) {
        return;
    }
    Tcl_DeleteHashEntry(hPtr);
}


/* End of tclIOCmd.c */

/*  tclUnixChan.c start  */
/* 
 * tclUnixChan.c
 *
 *	Common channel driver for Unix channels based on files, command
 *	pipes and TCP sockets.
 *
 * Copyright (c) 1995-1997 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tclUnixChan.c 1.203 97/06/20 13:03:18
 */

/*
 * The following structure is used to set or get the serial port
 * attributes in a platform-independant manner.
 */
 
typedef struct TtyAttrs {
    int baud;
    int parity;
    int data;
    int stop;
} TtyAttrs;

/*
 * This structure describes per-instance state of a file based channel.
 */

typedef struct FileState {
    Tcl_Channel channel;	/* Channel associated with this file. */
    int fd;			/* File handle. */
    int validMask;		/* OR'ed combination of TCL_READABLE,
				 * TCL_WRITABLE, or TCL_EXCEPTION: indicates
				 * which operations are valid on the file. */
    struct FileState *nextPtr;	/* Pointer to next file in list of all
				 * file channels. */
} FileState;

/*
 * List of all file channels currently open.
 */

static FileState *firstFilePtr = NULL;

/*
 * This structure describes per-instance state of a tcp based channel.
 */

typedef struct TcpState {
    Tcl_Channel channel;	/* Channel associated with this file. */
    int fd;			/* The socket itself. */
    int flags;			/* ORed combination of the bitfields
				 * defined below. */
    Tcl_TcpAcceptProc *acceptProc;
				/* Proc to call on accept. */
    ClientData acceptProcData;	/* The data for the accept proc. */
    SSL *ssl;				/* Struct for SSL processing */
    SSL_CTX  *ctx;                      /* Contect for SSL    */
    int sslerr;				/* I'm sure there another way to do this, but... */
  int server;                   /* its a server socket */   
  int needssl;                  /* we are using ssl */
  int orig_fd;                  /* to know when to free ctx */
} TcpState;

/* FIXME, verify_error have to be global */
int verify_error = X509_V_OK;

/*
 * These bits may be ORed together into the "flags" field of a TcpState
 * structure.
 */

#define TCP_ASYNC_SOCKET	(1<<0)	/* Asynchronous socket. */
#define TCP_ASYNC_CONNECT	(1<<1)	/* Async connect in progress. */

/*
 * The following defines the maximum length of the listen queue. This is
 * the number of outstanding yet-to-be-serviced requests for a connection
 * on a server socket, more than this number of outstanding requests and
 * the connection request will fail.
 */

#ifndef	SOMAXCONN
#define SOMAXCONN	100
#endif

#if	(SOMAXCONN < 100)
#undef	SOMAXCONN
#define	SOMAXCONN	100
#endif

/*
 * The following defines how much buffer space the kernel should maintain
 * for a socket.
 */

#define SOCKET_BUFSIZE	4096

/*
 * Static routines for this file:
 */

static TcpState *	CreateSocket _ANSI_ARGS_((Tcl_Interp *interp,
			    int port, char *host, int server,
			    char *myaddr, int myport, int async, int needssl,char *key,char *cert,char *CAfile,char *CAdir,char *ciphers,int verify));
static int		CreateSocketAddress _ANSI_ARGS_(
			    (struct sockaddr_in *sockaddrPtr,
			    char *host, int port));
static int		FileBlockModeProc _ANSI_ARGS_((
    			    ClientData instanceData, int mode));
static int		FileCloseProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp));
static int		FileGetHandleProc _ANSI_ARGS_((ClientData instanceData,
		            int direction, ClientData *handlePtr));
static int		FileInputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toRead, int *errorCode));
static int		FileOutputProc _ANSI_ARGS_((
			    ClientData instanceData, char *buf, int toWrite,
                            int *errorCode));
static int		FileSeekProc _ANSI_ARGS_((ClientData instanceData,
			    long offset, int mode, int *errorCode));
static void		FileWatchProc _ANSI_ARGS_((ClientData instanceData,
		            int mask));
static void		TcpAccept _ANSI_ARGS_((ClientData data, int mask));
static int		TcpBlockModeProc _ANSI_ARGS_((ClientData data,
        		    int mode));
static int		TcpCloseProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp));
static int		TcpGetHandleProc _ANSI_ARGS_((ClientData instanceData,
		            int direction, ClientData *handlePtr));
static int		TcpGetOptionProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp, char *optionName,
			    Tcl_DString *dsPtr));
static int		TcpInputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toRead,  int *errorCode));
static int		TcpOutputProc _ANSI_ARGS_((ClientData instanceData,
		            char *buf, int toWrite, int *errorCode));
static void		TcpWatchProc _ANSI_ARGS_((ClientData instanceData,
		            int mask));
static int		TtyParseMode _ANSI_ARGS_((Tcl_Interp *interp,
			    CONST char *mode, int *speedPtr, int *parityPtr,
			    int *dataPtr, int *stopPtr));
static void		TtyGetAttributes _ANSI_ARGS_((int fd,
			    TtyAttrs *ttyPtr));
static int		TtyGetOptionProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp, char *optionName,
			    Tcl_DString *dsPtr));
static void		TtyInit _ANSI_ARGS_((int fd));
static void		TtySetAttributes _ANSI_ARGS_((int fd,
			    TtyAttrs *ttyPtr));
static int		TtySetOptionProc _ANSI_ARGS_((ClientData instanceData,
			    Tcl_Interp *interp, char *optionName, 
			    char *value));
static int		WaitForConnect _ANSI_ARGS_((TcpState *statePtr,
		            int *errorCodePtr));

/*
 * This structure describes the channel type structure for file based IO:
 */

static Tcl_ChannelType fileChannelType = {
    "file",				/* Type name. */
    FileBlockModeProc,			/* Set blocking/nonblocking mode.*/
    FileCloseProc,			/* Close proc. */
    FileInputProc,			/* Input proc. */
    FileOutputProc,			/* Output proc. */
    FileSeekProc,			/* Seek proc. */
    NULL,				/* Set option proc. */
    NULL,				/* Get option proc. */
    FileWatchProc,			/* Initialize notifier. */
    FileGetHandleProc,			/* Get OS handles out of channel. */
};

/*
 * This structure describes the channel type structure for serial IO.
 * Note that this type is a subclass of the "file" type.
 */

static Tcl_ChannelType ttyChannelType = {
    "tty",				/* Type name. */
    FileBlockModeProc,			/* Set blocking/nonblocking mode.*/
    FileCloseProc,			/* Close proc. */
    FileInputProc,			/* Input proc. */
    FileOutputProc,			/* Output proc. */
    NULL,				/* Seek proc. */
    TtySetOptionProc,			/* Set option proc. */
    TtyGetOptionProc,			/* Get option proc. */
    FileWatchProc,			/* Initialize notifier. */
    FileGetHandleProc,			/* Get OS handles out of channel. */
};

/*
 * This structure describes the channel type structure for TCP socket
 * based IO:
 */

static Tcl_ChannelType tcpChannelType = {
    "tcp",				/* Type name. */
    TcpBlockModeProc,			/* Set blocking/nonblocking mode.*/
    TcpCloseProc,			/* Close proc. */
    TcpInputProc,			/* Input proc. */
    TcpOutputProc,			/* Output proc. */
    NULL,				/* Seek proc. */
    NULL,				/* Set option proc. */
    TcpGetOptionProc,			/* Get option proc. */
    TcpWatchProc,			/* Initialize notifier. */
    TcpGetHandleProc,			/* Get OS handles out of channel. */
};


/*
 *----------------------------------------------------------------------
 *
 * FileBlockModeProc --
 *
 *	Helper procedure to set blocking and nonblocking modes on a
 *	file based channel. Invoked by generic IO level code.
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or non-blocking mode.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
FileBlockModeProc(instanceData, mode)
    ClientData instanceData;		/* File state. */
    int mode;				/* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    FileState *fsPtr = (FileState *) instanceData;
    int curStatus;

#ifndef USE_FIONBIO
    curStatus = fcntl(fsPtr->fd, F_GETFL);
    if (mode == TCL_MODE_BLOCKING) {
	curStatus &= (~(O_NONBLOCK));
    } else {
	curStatus |= O_NONBLOCK;
    }
    if (fcntl(fsPtr->fd, F_SETFL, curStatus) < 0) {
	return errno;
    }
    curStatus = fcntl(fsPtr->fd, F_GETFL);
#else
    if (mode == TCL_MODE_BLOCKING) {
	curStatus = 0;
    } else {
	curStatus = 1;
    }
    if (ioctl(fsPtr->fd, (int) FIONBIO, &curStatus) < 0) {
	return errno;
    }
#endif
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * FileInputProc --
 *
 *	This procedure is invoked from the generic IO level to read
 *	input from a file based channel.
 *
 * Results:
 *	The number of bytes read is returned or -1 on error. An output
 *	argument contains a POSIX error code if an error occurs, or zero.
 *
 * Side effects:
 *	Reads input from the input device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileInputProc(instanceData, buf, toRead, errorCodePtr)
    ClientData instanceData;		/* File state. */
    char *buf;				/* Where to store data read. */
    int toRead;				/* How much space is available
                                         * in the buffer? */
    int *errorCodePtr;			/* Where to store error code. */
{
    FileState *fsPtr = (FileState *) instanceData;
    int bytesRead;			/* How many bytes were actually
                                         * read from the input device? */

    *errorCodePtr = 0;
    
    /*
     * Assume there is always enough input available. This will block
     * appropriately, and read will unblock as soon as a short read is
     * possible, if the channel is in blocking mode. If the channel is
     * nonblocking, the read will never block.
     */

    bytesRead = read(fsPtr->fd, buf, (size_t) toRead);
    if (bytesRead > -1) {
        return bytesRead;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * FileOutputProc--
 *
 *	This procedure is invoked from the generic IO level to write
 *	output to a file channel.
 *
 * Results:
 *	The number of bytes written is returned or -1 on error. An
 *	output argument	contains a POSIX error code if an error occurred,
 *	or zero.
 *
 * Side effects:
 *	Writes output on the output device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileOutputProc(instanceData, buf, toWrite, errorCodePtr)
    ClientData instanceData;		/* File state. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCodePtr;			/* Where to store error code. */
{
    FileState *fsPtr = (FileState *) instanceData;
    int written;

    *errorCodePtr = 0;
    written = write(fsPtr->fd, buf, (size_t) toWrite);
    if (written > -1) {
        return written;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * FileCloseProc --
 *
 *	This procedure is called from the generic IO level to perform
 *	channel-type-specific cleanup when a file based channel is closed.
 *
 * Results:
 *	0 if successful, errno if failed.
 *
 * Side effects:
 *	Closes the device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
FileCloseProc(instanceData, interp)
    ClientData instanceData;	/* File state. */
    Tcl_Interp *interp;		/* For error reporting - unused. */
{
    FileState *fsPtr = (FileState *) instanceData;
    FileState **nextPtrPtr;
    int errorCode = 0;

    Tcl_DeleteFileHandler(fsPtr->fd);
    if (!TclInExit()
	    || ((fsPtr->fd != 0) && (fsPtr->fd != 1) && (fsPtr->fd != 2))) {
	if (close(fsPtr->fd) < 0) {
	    errorCode = errno;
	}
    }
    for (nextPtrPtr = &firstFilePtr; (*nextPtrPtr) != NULL;
	 nextPtrPtr = &((*nextPtrPtr)->nextPtr)) {
	if ((*nextPtrPtr) == fsPtr) {
	    (*nextPtrPtr) = fsPtr->nextPtr;
	    break;
	}
    }
    ckfree((char *) fsPtr);
    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * FileSeekProc --
 *
 *	This procedure is called by the generic IO level to move the
 *	access point in a file based channel.
 *
 * Results:
 *	-1 if failed, the new position if successful. An output
 *	argument contains the POSIX error code if an error occurred,
 *	or zero.
 *
 * Side effects:
 *	Moves the location at which the channel will be accessed in
 *	future operations.
 *
 *----------------------------------------------------------------------
 */

static int
FileSeekProc(instanceData, offset, mode, errorCodePtr)
    ClientData instanceData;			/* File state. */
    long offset;				/* Offset to seek to. */
    int mode;					/* Relative to where
                                                 * should we seek? Can be
                                                 * one of SEEK_START,
                                                 * SEEK_SET or SEEK_END. */
    int *errorCodePtr;				/* To store error code. */
{
    FileState *fsPtr = (FileState *) instanceData;
    int newLoc;

    newLoc = lseek(fsPtr->fd, offset, mode);

    *errorCodePtr = (newLoc == -1) ? errno : 0;
    return newLoc;
}

/*
 *----------------------------------------------------------------------
 *
 * FileWatchProc --
 *
 *	Initialize the notifier to watch the fd from this channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets up the notifier so that a future event on the channel will
 *	be seen by Tcl.
 *
 *----------------------------------------------------------------------
 */

static void
FileWatchProc(instanceData, mask)
    ClientData instanceData;		/* The file state. */
    int mask;				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    FileState *fsPtr = (FileState *) instanceData;

    /*
     * Make sure we only register for events that are valid on this file.
     * Note that we are passing Tcl_NotifyChannel directly to
     * Tcl_CreateFileHandler with the channel pointer as the client data.
     */

    mask &= fsPtr->validMask;
    if (mask) {
	Tcl_CreateFileHandler(fsPtr->fd, mask,
		(Tcl_FileProc *) Tcl_NotifyChannel,
		(ClientData) fsPtr->channel);
    } else {
	Tcl_DeleteFileHandler(fsPtr->fd);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * FileGetHandleProc --
 *
 *	Called from Tcl_GetChannelFile to retrieve OS handles from
 *	a file based channel.
 *
 * Results:
 *	Returns TCL_OK with the fd in handlePtr, or TCL_ERROR if
 *	there is no handle for the specified direction. 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
FileGetHandleProc(instanceData, direction, handlePtr)
    ClientData instanceData;	/* The file state. */
    int direction;		/* TCL_READABLE or TCL_WRITABLE */
    ClientData *handlePtr;	/* Where to store the handle.  */
{
    FileState *fsPtr = (FileState *) instanceData;

    if (direction & fsPtr->validMask) {
	*handlePtr = (ClientData) fsPtr->fd;
	return TCL_OK;
    } else {
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TtySetOptionProc --
 *
 *	Sets an option on a channel.
 *
 * Results:
 *	A standard Tcl result. Also sets interp->result on error if
 *	interp is not NULL.
 *
 * Side effects:
 *	May modify an option on a device.
 *      Sets Error message if needed (by calling Tcl_BadChannelOption).
 *
 *----------------------------------------------------------------------
 */

static int		
TtySetOptionProc(instanceData, interp, optionName, value)
    ClientData instanceData;	/* File state. */
    Tcl_Interp *interp;		/* For error reporting - can be NULL. */
    char *optionName;		/* Which option to set? */
    char *value;		/* New value for option. */
{
    FileState *fsPtr = (FileState *) instanceData;
    unsigned int len;
    TtyAttrs tty;

    len = strlen(optionName);
    if ((len > 1) && (strncmp(optionName, "-mode", len) == 0)) {
	if (TtyParseMode(interp, value, &tty.baud, &tty.parity, &tty.data,
		&tty.stop) != TCL_OK) {
	    return TCL_ERROR;
	}
	/*
	 * system calls results should be checked there. -- dl
	 */

	TtySetAttributes(fsPtr->fd, &tty);
	return TCL_OK;
    } else {
	return Tcl_BadChannelOption(interp, optionName, "mode");
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TtyGetOptionProc --
 *
 *	Gets a mode associated with an IO channel. If the optionName arg
 *	is non NULL, retrieves the value of that option. If the optionName
 *	arg is NULL, retrieves a list of alternating option names and
 *	values for the given channel.
 *
 * Results:
 *	A standard Tcl result. Also sets the supplied DString to the
 *	string value of the option(s) returned.
 *
 * Side effects:
 *	The string returned by this function is in static storage and
 *	may be reused at any time subsequent to the call.
 *      Sets Error message if needed (by calling Tcl_BadChannelOption).
 *
 *----------------------------------------------------------------------
 */

static int		
TtyGetOptionProc(instanceData, interp, optionName, dsPtr)
    ClientData instanceData;	/* File state. */
    Tcl_Interp *interp;		/* For error reporting - can be NULL. */
    char *optionName;		/* Option to get. */
    Tcl_DString *dsPtr;		/* Where to store value(s). */
{
    FileState *fsPtr = (FileState *) instanceData;
    unsigned int len;
    char buf[32];
    TtyAttrs tty;

    if (optionName == NULL) {
	Tcl_DStringAppendElement(dsPtr, "-mode");
	len = 0;
    } else {
	len = strlen(optionName);
    }
    if ((len == 0) || 
	    ((len > 1) && (strncmp(optionName, "-mode", len) == 0))) {
	TtyGetAttributes(fsPtr->fd, &tty);
	sprintf(buf, "%d,%c,%d,%d", tty.baud, tty.parity, tty.data, tty.stop);
	Tcl_DStringAppendElement(dsPtr, buf);
	return TCL_OK;
    } else {
	return Tcl_BadChannelOption(interp, optionName, "mode");
    }
}

#undef DIRECT_BAUD
#ifdef B4800
#   if (B4800 == 4800)
#	define DIRECT_BAUD
#   endif
#endif

#ifdef DIRECT_BAUD
#   define TtyGetSpeed(baud)   ((unsigned) (baud))
#   define TtyGetBaud(speed)   ((int) (speed))
#else

static struct {int baud; unsigned long speed;} speeds[] = {
#ifdef B0
    {0, B0},
#endif
#ifdef B50
    {50, B50},
#endif
#ifdef B75
    {75, B75},
#endif
#ifdef B110
    {110, B110},
#endif
#ifdef B134
    {134, B134},
#endif
#ifdef B150
    {150, B150},
#endif
#ifdef B200
    {200, B200},
#endif
#ifdef B300
    {300, B300},
#endif
#ifdef B600
    {600, B600},
#endif
#ifdef B1200
    {1200, B1200},
#endif
#ifdef B1800
    {1800, B1800},
#endif
#ifdef B2400
    {2400, B2400},
#endif
#ifdef B4800
    {4800, B4800},
#endif
#ifdef B9600
    {9600, B9600},
#endif
#ifdef B14400
    {14400, B14400},
#endif
#ifdef B19200
    {19200, B19200},
#endif
#ifdef EXTA
    {19200, EXTA},
#endif
#ifdef B28800
    {28800, B28800},
#endif
#ifdef B38400
    {38400, B38400},
#endif
#ifdef EXTB
    {38400, EXTB},
#endif
#ifdef B57600
    {57600, B57600},
#endif
#ifdef _B57600
    {57600, _B57600},
#endif
#ifdef B76800
    {76800, B76800},
#endif
#ifdef B115200
    {115200, B115200},
#endif
#ifdef _B115200
    {115200, _B115200},
#endif
#ifdef B153600
    {153600, B153600},
#endif
#ifdef B230400
    {230400, B230400},
#endif
#ifdef B307200
    {307200, B307200},
#endif
#ifdef B460800
    {460800, B460800},
#endif
    {-1, 0}
};

/*
 *---------------------------------------------------------------------------
 *
 * TtyGetSpeed --
 *
 *	Given a baud rate, get the mask value that should be stored in
 *	the termios, termio, or sgttyb structure in order to select that
 *	baud rate.
 *
 * Results:
 *	As above.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static unsigned long
TtyGetSpeed(baud)
    int baud;			/* The baud rate to look up. */
{
    int bestIdx, bestDiff, i, diff;
    
    bestIdx = 0;
    bestDiff = 1000000;
    
    /*
     * If the baud rate does not correspond to one of the known mask values,
     * choose the mask value whose baud rate is closest to the specified
     * baud rate.
     */

    for (i = 0; speeds[i].baud >= 0; i++) {
	diff = speeds[i].baud - baud;
	if (diff < 0) {
	    diff = -diff;
	}
	if (diff < bestDiff) {
	    bestIdx = i;
	    bestDiff = diff;
	}
    }
    return speeds[bestIdx].speed;
}

/*
 *---------------------------------------------------------------------------
 *
 * TtyGetBaud --
 *
 *	Given a speed mask value from a termios, termio, or sgttyb
 *	structure, get the baus rate that corresponds to that mask value.
 *
 * Results:
 *	As above.  If the mask value was not recognized, 0 is returned.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
TtyGetBaud(speed)
    unsigned long speed;	/* Speed mask value to look up. */
{
    int i;
    
    for (i = 0; speeds[i].baud >= 0; i++) {
	if (speeds[i].speed == speed) {
	    return speeds[i].baud;
	}
    }
    return 0;
}

#endif	/* !DIRECT_BAUD */


/*
 *---------------------------------------------------------------------------
 *
 * TtyInit --
 *
 *	Given file descriptor that refers to a serial port, 
 *	initialize the serial port to a set of sane values so that
 *	Tcl can talk to a device located on the serial port.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Serial device initialized.
 *
 *---------------------------------------------------------------------------
 */

static void
TtyInit(fd)
    int fd;			/* Open file descriptor for serial port to
				 * be initialized. */
{
#ifdef USE_TERMIOS
    struct termios termios;

    tcgetattr(fd, &termios);
    termios.c_iflag = IGNBRK;
    termios.c_oflag = 0;
    termios.c_lflag = 0;
    termios.c_cflag |= CREAD;
    termios.c_cc[VMIN] = 60;
    termios.c_cc[VTIME] = 2;
    tcsetattr(fd, TCSANOW, &termios);
#else	/* !USE_TERMIOS */
#ifdef USE_TERMIO
    struct termio termio;

    ioctl(fd, TCGETA, &termio);
    termio.c_iflag = IGNBRK;
    termio.c_oflag = 0;
    termio.c_lflag = 0;
    termio.c_cflag |= CREAD;
    termio.c_cc[VMIN] = 60;
    termio.c_cc[VTIME] = 2;
    ioctl(fd, TCSETAW, &termio);
#else	/* !USE_TERMIO */
#ifdef USE_SGTTY
    struct sgttyb sgttyb;

    ioctl(fd, TIOCGETP, &sgttyb);
    sgttyb.sg_flags &= (EVENP | ODDP);
    sgttyb.sg_flags |= RAW;
    ioctl(fd, TIOCSETP, &sgttyb);
#endif	/* USE_SGTTY */
#endif	/* !USE_TERMIO */
#endif	/* !USE_TERMIOS */
}

/*
 *---------------------------------------------------------------------------
 *
 * TtyGetAttributes --
 *
 *	Get the current attributes of the specified serial device.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
 
static void
TtyGetAttributes(fd, ttyPtr)
    int fd;			/* Open file descriptor for serial port to
				 * be queried. */
    TtyAttrs *ttyPtr;		/* Buffer filled with serial port
				 * attributes. */
{
#ifdef USE_TERMIOS
    int parity, data;
    struct termios termios;

    tcgetattr(fd, &termios);
    ttyPtr->baud = TtyGetBaud(cfgetospeed(&termios));
    
    parity = 'n';
#ifdef PAREXT
    switch ((int) (termios.c_cflag & (PARENB | PARODD | PAREXT))) {
	case PARENB		      : parity = 'e'; break;
	case PARENB | PARODD	      :	parity = 'o'; break;
	case PARENB |	       PAREXT : parity = 's'; break;
	case PARENB | PARODD | PAREXT :	parity = 'm'; break;
    }
#else	/* !PAREXT */
    switch ((int) (termios.c_cflag & (PARENB | PARODD))) {
	case PARENB		      : parity = 'e'; break;
	case PARENB | PARODD	      :	parity = 'o'; break;
    }
#endif	/* !PAREXT */
    ttyPtr->parity = parity;

    data = termios.c_cflag & CSIZE;
    ttyPtr->data = (data == CS5) ? 5 : (data == CS6) ? 6 :
	    (data == CS7) ? 7 : 8;

    ttyPtr->stop = (termios.c_cflag & CSTOPB) ? 2 : 1;
#else	/* !USE_TERMIOS */
#ifdef USE_TERMIO
    int parity, data;
    struct termio termio;

    ioctl(fd, TCGETA, &termio);
    ttyPtr->baud = TtyGetBaud(termio.c_cflag & CBAUD);
    parity = 'n';
    switch (termio.c_cflag & (PARENB | PARODD | PAREXT)) {
	case PARENB		      : parity = 'e'; break;
	case PARENB | PARODD	      :	parity = 'o'; break;
	case PARENB |	       PAREXT : parity = 's'; break;
	case PARENB | PARODD | PAREXT :	parity = 'm'; break;
    }
    ttyPtr->parity = parity;

    data = termio.c_cflag & CSIZE;
    ttyPtr->data = (data == CS5) ? 5 : (data == CS6) ? 6 :
	    (data == CS7) ? 7 : 8;

    ttyPtr->stop = (termio.c_cflag & CSTOPB) ? 2 : 1;
#else	/* !USE_TERMIO */
#ifdef USE_SGTTY
    int parity;
    struct sgttyb sgttyb;

    ioctl(fd, TIOCGETP, &sgttyb);
    ttyPtr->baud = TtyGetBaud(sgttyb.sg_ospeed);
    parity = 'n';
    if (sgttyb.sg_flags & EVENP) {
	parity = 'e';
    } else if (sgttyb.sg_flags & ODDP) {
	parity = 'o';
    }
    ttyPtr->parity = parity;
    ttyPtr->data = (sgttyb.sg_flags & (EVENP | ODDP)) ? 7 : 8;
    ttyPtr->stop = 1;
#else	/* !USE_SGTTY */
    ttyPtr->baud = 0;
    ttyPtr->parity = 'n';
    ttyPtr->data = 0;
    ttyPtr->stop = 0;
#endif	/* !USE_SGTTY */
#endif	/* !USE_TERMIO */
#endif	/* !USE_TERMIOS */
}

/*
 *---------------------------------------------------------------------------
 *
 * TtySetAttributes --
 *
 *	Set the current attributes of the specified serial device. 
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
 
static void
TtySetAttributes(fd, ttyPtr)
    int fd;			/* Open file descriptor for serial port to
				 * be modified. */
    TtyAttrs *ttyPtr;		/* Buffer containing new attributes for
				 * serial port. */
{
#ifdef USE_TERMIOS
    int parity, data, flag;
    struct termios termios;

    tcgetattr(fd, &termios);
    cfsetospeed(&termios, TtyGetSpeed(ttyPtr->baud));
    cfsetispeed(&termios, TtyGetSpeed(ttyPtr->baud));

    flag = 0;
    parity = ttyPtr->parity;
    if (parity != 'n') {
	flag |= PARENB;
#ifdef PAREXT
	termios.c_cflag &= ~PAREXT;
	if ((parity == 'm') || (parity == 's')) {
	    flag |= PAREXT;
	}
#endif
	if ((parity == 'm') || (parity == 'o')) {
	    flag |= PARODD;
	}
    }
    data = ttyPtr->data;
    flag |= (data == 5) ? CS5 : (data == 6) ? CS6 : (data == 7) ? CS7 : CS8;
    if (ttyPtr->stop == 2) {
	flag |= CSTOPB;
    }

    termios.c_cflag &= ~(PARENB | PARODD | CSIZE | CSTOPB);
    termios.c_cflag |= flag;
    tcsetattr(fd, TCSANOW, &termios);

#else	/* !USE_TERMIOS */
#ifdef USE_TERMIO
    int parity, data, flag;
    struct termio termio;

    ioctl(fd, TCGETA, &termio);
    termio.c_cflag &= ~CBAUD;
    termio.c_cflag |= TtyGetSpeed(ttyPtr->baud);

    flag = 0;
    parity = ttyPtr->parity;
    if (parity != 'n') {
	flag |= PARENB;
	if ((parity == 'm') || (parity == 's')) {
	    flag |= PAREXT;
	}
	if ((parity == 'm') || (parity == 'o')) {
	    flag |= PARODD;
	}
    }
    data = ttyPtr->data;
    flag |= (data == 5) ? CS5 : (data == 6) ? CS6 : (data == 7) ? CS7 : CS8;
    if (ttyPtr->stop == 2) {
	flag |= CSTOPB;
    }

    termio.c_cflag &= ~(PARENB | PARODD | PAREXT | CSIZE | CSTOPB);
    termio.c_cflag |= flag;
    ioctl(fd, TCSETAW, &termio);

#else	/* !USE_TERMIO */
#ifdef USE_SGTTY
    int parity;
    struct sgttyb sgttyb;

    ioctl(fd, TIOCGETP, &sgttyb);
    sgttyb.sg_ospeed = TtyGetSpeed(ttyPtr->baud);
    sgttyb.sg_ispeed = TtyGetSpeed(ttyPtr->baud);

    parity = ttyPtr->parity;
    if (parity == 'e') {
	sgttyb.sg_flags &= ~ODDP;
	sgttyb.sg_flags |= EVENP;
    } else if (parity == 'o') {
	sgttyb.sg_flags &= ~EVENP;
	sgttyb.sg_flags |= ODDP;
    }
    ioctl(fd, TIOCSETP, &sgttyb);
#endif	/* USE_SGTTY */
#endif	/* !USE_TERMIO */
#endif	/* !USE_TERMIOS */
}

/*
 *---------------------------------------------------------------------------
 *
 * TtyParseMode --
 *
 *	Parse the "-mode" argument to the fconfigure command.  The argument
 *	is of the form baud,parity,data,stop.
 *
 * Results:
 *	The return value is TCL_OK if the argument was successfully
 *	parsed, TCL_ERROR otherwise.  If TCL_ERROR is returned, an
 *	error message is left in interp->result (if interp is non-NULL).
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
 
static int
TtyParseMode(interp, mode, speedPtr, parityPtr, dataPtr, stopPtr)
    Tcl_Interp *interp;		/* If non-NULL, interp for error return. */
    CONST char *mode;		/* Mode string to be parsed. */
    int *speedPtr;		/* Filled with baud rate from mode string. */
    int *parityPtr;		/* Filled with parity from mode string. */
    int *dataPtr;		/* Filled with data bits from mode string. */
    int *stopPtr;		/* Filled with stop bits from mode string. */
{
    int i, end;
    char parity;
    static char *bad = "bad value for -mode";

    i = sscanf(mode, "%d,%c,%d,%d%n", speedPtr, &parity, dataPtr,
	    stopPtr, &end);
    if ((i != 4) || (mode[end] != '\0')) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, bad, ": should be baud,parity,data,stop",
		    NULL);
	}
	return TCL_ERROR;
    }
    if (strchr("noems", parity) == NULL) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, bad,
		    " parity: should be n, o, e, m, or s", NULL);
	}
	return TCL_ERROR;
    }
    *parityPtr = parity;
    if ((*dataPtr < 5) || (*dataPtr > 8)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, bad, " data: should be 5, 6, 7, or 8",
		    NULL);
	}
	return TCL_ERROR;
    }
    if ((*stopPtr < 0) || (*stopPtr > 2)) {
	if (interp != NULL) {
	    Tcl_AppendResult(interp, bad, " stop: should be 1 or 2", NULL);
	}
	return TCL_ERROR;
    }
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * TcpBlockModeProc --
 *
 *	This procedure is invoked by the generic IO level to set blocking
 *	and nonblocking mode on a TCP socket based channel.
 *
 * Results:
 *	0 if successful, errno when failed.
 *
 * Side effects:
 *	Sets the device into blocking or nonblocking mode.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpBlockModeProc(instanceData, mode)
    ClientData instanceData;		/* Socket state. */
    int mode;				/* The mode to set. Can be one of
                                         * TCL_MODE_BLOCKING or
                                         * TCL_MODE_NONBLOCKING. */
{
    TcpState *statePtr = (TcpState *) instanceData;
    int setting;
    
#ifndef	USE_FIONBIO
    setting = fcntl(statePtr->fd, F_GETFL);
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting &= (~(O_NONBLOCK));
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting |= O_NONBLOCK;
    }
    if (fcntl(statePtr->fd, F_SETFL, setting) < 0) {
        return errno;
    }
#endif

#ifdef	USE_FIONBIO
    if (mode == TCL_MODE_BLOCKING) {
        statePtr->flags &= (~(TCP_ASYNC_SOCKET));
        setting = 0;
        if (ioctl(statePtr->fd, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    } else {
        statePtr->flags |= TCP_ASYNC_SOCKET;
        setting = 1;
        if (ioctl(statePtr->fd, (int) FIONBIO, &setting) == -1) {
            return errno;
        }
    }
#endif

    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * WaitForConnect --
 *
 *	Waits for a connection on an asynchronously opened socket to
 *	be completed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The socket is connected after this function returns.
 *
 *----------------------------------------------------------------------
 */

static int
WaitForConnect(statePtr, errorCodePtr)
    TcpState *statePtr;		/* State of the socket. */
    int *errorCodePtr;		/* Where to store errors? */
{
    int timeOut;		/* How long to wait. */
    int state;			/* Of calling TclWaitForFile. */
    int flags;			/* fcntl flags for the socket. */
    int timer = 0;              /* to stop SSL_accept in non
				   blocking mode if errors */

    /*
     * If an asynchronous connect is in progress, attempt to wait for it
     * to complete before reading.
     */

	/* DEBUG
	*/
#ifdef PDEBUG
	/* printf("Innan allt i waitforconnect %d\n", state); */
#endif
    
    if (statePtr->flags & TCP_ASYNC_CONNECT) {
        if (statePtr->flags & TCP_ASYNC_SOCKET) {
            timeOut = 0;
        } else {
            timeOut = -1;
        }
        errno = 0;
        state = TclUnixWaitForFile(statePtr->fd,
		TCL_WRITABLE | TCL_EXCEPTION, timeOut);
        if (!(statePtr->flags & TCP_ASYNC_SOCKET)) {
#ifndef	USE_FIONBIO
            flags = fcntl(statePtr->fd, F_GETFL);
            flags &= (~(O_NONBLOCK));
            (void) fcntl(statePtr->fd, F_SETFL, flags);
#endif

#ifdef	USE_FIONBIO
            flags = 0;
            (void) ioctl(statePtr->fd, FIONBIO, &flags);
#endif
        }
        if (state & TCL_EXCEPTION) {
            return -1;
        }
        if (state & TCL_WRITABLE) {
            statePtr->flags &= (~(TCP_ASYNC_CONNECT));
        } else if (timeOut == 0) {
            *errorCodePtr = errno = EWOULDBLOCK;
            return -1;
        }
    }
    if ( statePtr->needssl ) {

  if ( statePtr->server == 2 ) {

#ifdef PDEBUG
  puts("innan SSL_accept");
#endif
  
/* To handle no-blocking IO we use this while loop, Peter Antman */
while(!SSL_is_init_finished(statePtr->ssl)) 
  {
  
  if((statePtr->sslerr = SSL_accept(statePtr->ssl)) <= 0 ) {
    
		  verify_error=SSL_get_verify_result(statePtr->ssl);
		  #ifdef PDEBUG
		  fprintf(stderr,"verify_serror %d", verify_error);
		  #endif
		  if (verify_error != X509_V_OK) {
			fprintf(stderr,"verify error:%s\n",
				X509_verify_cert_error_string(verify_error));
			statePtr->server = 3;
			timer = 0;
			statePtr->sslerr = -1;
			*errorCodePtr = TCL_EXCEPTION;
			return -1;
		   } else {
		     timer++;
		     /* fprintf(stderr, "timer: %d",timer) */
		     /* FIXME, this is a temporary solution to the problem
		      * with server in nonblocking mode. If more than 2000
		      * rounds with error, return i TCL_EXCEPTION.
		      * This is in particualr a problem when non SSL
		      * client tries to connect to the socket. 
		      * More: how a bout a timeout if the client
		      * does not send anyting? */

		     if (timer > max_ac_loops) {
		       
		       /*  fprintf(stderr, "timer: %d",timer); */
		          statePtr->server = 3;
			  timer = 0;
			  statePtr->sslerr = -1;
			  *errorCodePtr = TCL_EXCEPTION;
 			  return -1;
		     }
		     

		     #ifdef PDEBUG
		     puts("Error i accept");
		     
		     switch (SSL_get_error(statePtr->ssl,verify_error))
		       {
			case SSL_ERROR_NONE:
			  puts("ssl error none");
			  break;
                        case SSL_ERROR_WANT_WRITE:
			  puts("ssl error want write");
			  break;
                        case SSL_ERROR_WANT_READ:
			  puts("ssl error want read");
			  break;
                        case SSL_ERROR_WANT_X509_LOOKUP:
			  puts("ssl X509");
			  break;
                        case SSL_ERROR_SYSCALL:
			  puts("syscall");
                        case SSL_ERROR_SSL:
			  puts("error ssl");
			/*   statePtr->server = 3; */
/* 			  return(statePtr->sslerr = -1); */
/* 			  break; */
                        case SSL_ERROR_ZERO_RETURN:
			  puts("zeror");
			  break;
                               
                               
                             
		       }
		     #endif
			     
			       
			     
		     /*  printf(stderr, "SSL AcceptError (%d) i (%s)\n", */
			     /* 				statePtr->sslerr, ERR_error_string(ERR_get_error())); */
		    #ifdef PDEBUG
		     ERR_print_errors_fp(stderr); 
		     #endif
			statePtr->sslerr = -1;
        		
		   }
  } else {
    timer = 0;
			statePtr->sslerr = 0;
			PEM_write_SSL_SESSION(stderr,SSL_get_session(statePtr->ssl));
	  }
		statePtr->server = 3;

  }
  }
  return statePtr->sslerr;
    }
return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpInputProc --
 *
 *	This procedure is invoked by the generic IO level to read input
 *	from a TCP socket based channel.
 *
 *	NOTE: We cannot share code with FilePipeInputProc because here
 *	we must use recv to obtain the input from the channel, not read.
 *
 * Results:
 *	The number of bytes read is returned or -1 on error. An output
 *	argument contains the POSIX error code on error, or zero if no
 *	error occurred.
 *
 * Side effects:
 *	Reads input from the input device of the channel.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpInputProc(instanceData, buf, bufSize, errorCodePtr)
    ClientData instanceData;		/* Socket state. */
    char *buf;				/* Where to store data read. */
    int bufSize;			/* How much space is available
                                         * in the buffer? */
    int *errorCodePtr;			/* Where to store error code. */
{
    TcpState *statePtr = (TcpState *) instanceData;
    int bytesRead, state;

    *errorCodePtr = 0;
    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
        return -1;
    }
	if ( statePtr->needssl ) {

    		bytesRead = SSL_read(statePtr->ssl, buf, bufSize);

	} else {
    		bytesRead = recv(statePtr->fd, buf, bufSize,0);
	}
    if (bytesRead > -1) {
        return bytesRead;
    }
    if (errno == ECONNRESET) {

        /*
         * Turn ECONNRESET into a soft EOF condition.
         */
        
        return 0;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpOutputProc --
 *
 *	This procedure is invoked by the generic IO level to write output
 *	to a TCP socket based channel.
 *
 *	NOTE: We cannot share code with FilePipeOutputProc because here
 *	we must use send, not write, to get reliable error reporting.
 *
 * Results:
 *	The number of bytes written is returned. An output argument is
 *	set to a POSIX error code if an error occurred, or zero.
 *
 * Side effects:
 *	Writes output on the output device of the channel.
 *
 *----------------------------------------------------------------------
 */

static int
TcpOutputProc(instanceData, buf, toWrite, errorCodePtr)
    ClientData instanceData;		/* Socket state. */
    char *buf;				/* The data buffer. */
    int toWrite;			/* How many bytes to write? */
    int *errorCodePtr;			/* Where to store error code. */
{
    TcpState *statePtr = (TcpState *) instanceData;
    int written;
    int state;				/* Of waiting for connection. */

    *errorCodePtr = 0;
    state = WaitForConnect(statePtr, errorCodePtr);
    if (state != 0) {
        return -1;
    }

	if ( statePtr->needssl ) {
    		written = SSL_write(statePtr->ssl, buf, toWrite);
	} else {
    		written = send(statePtr->fd, buf, toWrite,0);
	}

    if (written > -1) {
        return written;
    }
    *errorCodePtr = errno;
    return -1;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpCloseProc --
 *
 *	This procedure is invoked by the generic IO level to perform
 *	channel-type-specific cleanup when a TCP socket based channel
 *	is closed.
 *
 * Results:
 *	0 if successful, the value of errno if failed.
 *
 * Side effects:
 *	Closes the socket of the channel.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpCloseProc(instanceData, interp)
    ClientData instanceData;	/* The socket to close. */
    Tcl_Interp *interp;		/* For error reporting - unused. */
{
    TcpState *statePtr = (TcpState *) instanceData;
    int errorCode = 0;

    /*
     * Delete a file handler that may be active for this socket if this
     * is a server socket - the file handler was created automatically
     * by Tcl as part of the mechanism to accept new client connections.
     * Channel handlers are already deleted in the generic IO channel
     * closing code that called this function, so we do not have to
     * delete them here.
     */
#ifdef PDEBUG
    puts("Stnger proc");
#endif

#if SSLEAY_VERSION_NUMBER < 0x0800

#else
    if ( statePtr->needssl ) {SSL_shutdown(statePtr->ssl); }
#endif
     	if ( statePtr->needssl ) { SSL_free(statePtr->ssl); }

	if (statePtr->needssl) {
	  if (statePtr->fd == statePtr->orig_fd) {
#ifdef PDEBUG
	    puts("Freeing ctx");
#endif
	    SSL_CTX_free(statePtr->ctx);
	  }
	  
	}

    Tcl_DeleteFileHandler(statePtr->fd);

    if (close(statePtr->fd) < 0) {
	errorCode = errno;
    }
    ckfree((char *) statePtr);

    return errorCode;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetOptionProc --
 *
 *	Computes an option value for a TCP socket based channel, or a
 *	list of all options and their values.
 *
 *	Note: This code is based on code contributed by John Haxby.
 *
 * Results:
 *	A standard Tcl result. The value of the specified option or a
 *	list of all options and	their values is returned in the
 *	supplied DString. Sets Error message if needed.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
TcpGetOptionProc(instanceData, interp, optionName, dsPtr)
    ClientData instanceData;     /* Socket state. */
    Tcl_Interp *interp;          /* For error reporting - can be NULL. */
    char *optionName;	         /* Name of the option to
				  * retrieve the value for, or
				  * NULL to get all options and
				  * their values. */
    Tcl_DString *dsPtr;	         /* Where to store the computed
				  * value; initialized by caller. */
{
    TcpState *statePtr = (TcpState *) instanceData;
    struct sockaddr_in sockname;
    struct sockaddr_in peername;
    struct hostent *hostEntPtr;
    int size = sizeof(struct sockaddr_in);
    size_t len = 0;
    char buf[128];

    if (optionName != (char *) NULL) {
        len = strlen(optionName);
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 'p') &&
                    (strncmp(optionName, "-peername", len) == 0))) {
        if (getpeername(statePtr->fd, (struct sockaddr *) &peername, &size)
		>= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-peername");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(peername.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(peername.sin_addr));
            }
            sprintf(buf, "%d", ntohs(peername.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        } else {
            /*
             * getpeername failed - but if we were asked for all the options
             * (len==0), don't flag an error at that point because it could
             * be an fconfigure request on a server socket. (which have
             * no peer). same must be done on win&mac.
             */

            if (len) {
                if (interp) {
                    Tcl_AppendResult(interp, "can't get peername: ",
                                     Tcl_PosixError(interp),
                                     (char *) NULL);
                }
                return TCL_ERROR;
            }
        }
    }

    if ((len == 0) ||
            ((len > 1) && (optionName[1] == 's') &&
                    (strncmp(optionName, "-sockname", len) == 0))) {
        if (getsockname(statePtr->fd, (struct sockaddr *) &sockname, &size)
		>= 0) {
            if (len == 0) {
                Tcl_DStringAppendElement(dsPtr, "-sockname");
                Tcl_DStringStartSublist(dsPtr);
            }
            Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            hostEntPtr = gethostbyaddr((char *) &(sockname.sin_addr),
                    sizeof(peername.sin_addr), AF_INET);
            if (hostEntPtr != (struct hostent *) NULL) {
                Tcl_DStringAppendElement(dsPtr, hostEntPtr->h_name);
            } else {
                Tcl_DStringAppendElement(dsPtr, inet_ntoa(sockname.sin_addr));
            }
            sprintf(buf, "%d", ntohs(sockname.sin_port));
            Tcl_DStringAppendElement(dsPtr, buf);
            if (len == 0) {
                Tcl_DStringEndSublist(dsPtr);
            } else {
                return TCL_OK;
            }
        } else {
	    if (interp) {
		Tcl_AppendResult(interp, "can't get sockname: ",
				 Tcl_PosixError(interp),
				 (char *) NULL);
	    }
	    return TCL_ERROR;
	}
    }

    if (len > 0) {
        return Tcl_BadChannelOption(interp, optionName, "peername sockname");
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TcpWatchProc --
 *
 *	Initialize the notifier to watch the fd from this channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Sets up the notifier so that a future event on the channel will
 *	be seen by Tcl.
 *
 *----------------------------------------------------------------------
 */

static void
TcpWatchProc(instanceData, mask)
    ClientData instanceData;		/* The socket state. */
    int mask;				/* Events of interest; an OR-ed
                                         * combination of TCL_READABLE,
                                         * TCL_WRITABLE and TCL_EXCEPTION. */
{
    TcpState *statePtr = (TcpState *) instanceData;

    if (mask) {
	Tcl_CreateFileHandler(statePtr->fd, mask,
		(Tcl_FileProc *) Tcl_NotifyChannel,
		(ClientData) statePtr->channel);
    } else {
	Tcl_DeleteFileHandler(statePtr->fd);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TcpGetHandleProc --
 *
 *	Called from Tcl_GetChannelFile to retrieve OS handles from inside
 *	a TCP socket based channel.
 *
 * Results:
 *	Returns TCL_OK with the fd in handlePtr, or TCL_ERROR if
 *	there is no handle for the specified direction. 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static int
TcpGetHandleProc(instanceData, direction, handlePtr)
    ClientData instanceData;	/* The socket state. */
    int direction;		/* Not used. */
    ClientData *handlePtr;	/* Where to store the handle.  */
{
    TcpState *statePtr = (TcpState *) instanceData;

    *handlePtr = (ClientData)statePtr->fd;
    return TCL_OK;
}


/*
 *---------------------------------------------------------------------
 *
 * VerifyCallbackSSL --
 *
 *       Callback from SSLeay to perform customized handling
 *       of verify errors
 *
 * Result:
 *       Set error (I think)
 *
 * Side effects:
 *       Its purpose
 *
 * By Peter Antman
 * inspired by s_cb.c
 *
 *-----------------------------------------------------------------------
 */
#if SSLEAY_VERSION_NUMBER >= 0x0800
int
VerifyCallbackSSL(int ok,X509_STORE_CTX *ctx)
    {
    X509 *xs=X509_STORE_CTX_get_current_cert(ctx);
    int depth=X509_STORE_CTX_get_error_depth(ctx);
    int error=X509_STORE_CTX_get_error(ctx);
#else
int
VerifyCallbackSSL(ok, xs, xi, depth, error, arg)
		  int ok;
		  X509 *xs;
		  X509 *xi;
		  int depth;
		  int error;
		  char *arg;
{
#endif
	  /* FIXME, verify_error is global */
	char *s;
	/* Staticly set by Peter Antman, FIXME */
	/* int verify_depth = 1; */

			  #ifdef PDEBUG
	fprintf(stderr,"depth r %d i callback\n", verify_depth); 
	fprintf(stderr,"verify error:num=%d:%s\n",error,
			X509_verify_cert_error_string(error));
#endif

	/* is this realy ok, what are we getting */
#if SSLEAY_VERSION_NUMBER < 0x0800
    s=(char *)X509_NAME_oneline(X509_get_subject_name(xi));
#else
	s=(char *)X509_NAME_oneline(X509_get_subject_name(xs),NULL,0);
#endif
	fprintf(stderr,"Cafile: %s",s);
	free(s);


#if SSLEAY_VERSION_NUMBER < 0x0800
	s=(char *)X509_NAME_oneline(X509_get_subject_name(xs));
#else
	s=(char *)X509_NAME_oneline(X509_get_subject_name(xs),NULL,0);
#endif
	if (s == NULL)
		{
		ERR_print_errors_fp(stderr);
		return(0);
		}
	fprintf(stderr,"depth=%d %s\n",depth,s);
	free(s);
	if (error == X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT_LOCALLY)
		{
#if SSLEAY_VERSION_NUMBER < 0x0800
	s=(char *)X509_NAME_oneline(X509_get_issuer_name(xs));
#else
		s=(char *)X509_NAME_oneline(X509_get_issuer_name(xs),NULL,0);
#endif
		if (s == NULL)
			{
			fprintf(stderr,"verify error\n");
			ERR_print_errors_fp(stderr);
			return(0);
			}
		fprintf(stderr,"issuer= %s\n",s);
		free(s);
		}
	if (!ok)
		{
		fprintf(stderr,"verify error:num=%d:%s\n",error,
			X509_verify_cert_error_string(error));
		if (verify_depth <= depth)
			{
			ok=1;
			verify_error=X509_V_OK;
			}
		else
			{
			ok=0;
			verify_error=error;
			}
		}
	fprintf(stderr,"verify return:%d\n",ok);

#ifdef PDEBUG
	puts("innan return callback");
#endif
	return(ok);
}

/*
 *---------------------------------------------------------------------
 * SetCertStuffSSL --
 *
 *     Set certification stuff
 *
 * Result:
 *     Sets verification stuff up in the ctx-structure for SSLeay
 *
 * Sideeffects
 *     No need to do it in other places
 *
 * By Peter Antman,
 * inspired from s_cb.c
 *
 *----------------------------------------------------------------------
 */
int
SetCertStuffSSL(ctx, cert_file, key_file)
 SSL_CTX *ctx;
 char *cert_file;
 char *key_file;
{
#if SSLEAY_VERSION_NUMBER < 0x0800

#else
  SSL *ssl;
  X509 *x509;
#endif
  #ifdef PDEBUG
	puts("Stter cert stuff");
#endif
	if (cert_file != NULL)
		{
		if (SSL_CTX_use_certificate_file(ctx,cert_file,
			SSL_FILETYPE_PEM) <= 0)
			{
			fprintf(stderr,"unable to set certificate file\n");
			ERR_print_errors_fp(stderr);
			return(0);
			}
		if (key_file == NULL) key_file=cert_file;
		if (SSL_CTX_use_PrivateKey_file(ctx,key_file,
			SSL_FILETYPE_PEM) <= 0)
			{
			fprintf(stderr,"unable to set public key file\n");
			ERR_print_errors_fp(stderr);
			return(0);
			}
#if SSLEAY_VERSION_NUMBER < 0x0800

#else
		/* Experiment, from s_client.c */
		ssl = SSL_new(ctx);
		x509 = SSL_get_certificate(ssl);

		if (x509 != NULL) {
		  EVP_PKEY_copy_parameters(X509_get_pubkey(x509),SSL_get_privatekey(ssl));
		}
		SSL_free(ssl);
		/* End of experiment */
#endif		
		/* Now we know that a key and cert have been set against
		 * the SSL context */
		if (!SSL_CTX_check_private_key(ctx))
			{
			fprintf(stderr,"Private key does not match the certificate public key\n");
			return(0);
			}
		}
	return(1);
}


/*
 *----------------------------------------------------------------------
 *
 * CreateSocket --
 *
 *	This function opens a new socket in client or server mode
 *	and initializes the TcpState structure.
 *
 * Results:
 *	Returns a new TcpState, or NULL with an error in interp->result,
 *	if interp is not NULL.
 *
 * Side effects:
 *	Opens a socket.
 *
 *----------------------------------------------------------------------
 */

static TcpState *
CreateSocket(interp, port, host, server, myaddr, myport, async,needssl, key, cert,CAfile,CAdir,ciphers,verify)
    Tcl_Interp *interp;		/* For error reporting; can be NULL. */
    int port;			/* Port number to open. */
    char *host;			/* Name of host on which to open port.
				 * NULL implies INADDR_ANY */
    int server;			/* 1 if socket should be a server socket,
				 * else 0 for a client socket. */
    char *myaddr;		/* Optional client-side address */
    int myport;			/* Optional client-side port */
    int async;			/* If nonzero and creating a client socket,
                                 * attempt to do an async connect. Otherwise
                                 * do a synchronous connect or bind. */
    int  needssl;		/* configure for a secure socket  */
    char *key;                  /* Private key for SSL server     */
    char *cert;                 /* Certificate for SSL server */
    char *CAfile;
    char *CAdir;
    char *ciphers;
    int verify;
{
    int status, sock, asyncConnect, curState, origState;
    struct sockaddr_in sockaddr;	/* socket address */
    struct sockaddr_in mysockaddr;	/* Socket address for client */
    TcpState *statePtr;
    char *chiph;
    char *env_cipher;                   /* default cipher list */
    char *cafile;

    /* SSL *ssl = (SSL *)0; */
/*     SSL_CTX *ctx = (SSL_CTX *)0; */
    SSL *ssl = NULL;
    SSL_CTX *ctx = NULL;
    sock = -1;
    origState = 0;

    chiph = NULL;
    env_cipher = SSL_CIPH;
    cafile = "/usr/local/src/SSLeay-0.6.6/apps/demoCA/MinCA-cert.pem";

    if (! CreateSocketAddress(&sockaddr, host, port)) {
	goto addressError;
    }
    if ((myaddr != NULL || myport != 0) &&
	    ! CreateSocketAddress(&mysockaddr, myaddr, myport)) {
	goto addressError;
    }

    sock = socket(AF_INET, SOCK_STREAM, 0);
    if (sock < 0) {
	goto addressError;
    }

    if ( needssl ) {
      	SSL_load_error_strings();
	ERR_load_crypto_strings();

#if SSLEAY_VERSION_NUMBER >= 0x0800
	SSLeay_add_ssl_algorithms();
#endif
#if SSLEAY_VERSION_NUMBER < 0x0800
	ctx = SSL_CTX_new();
#else
	 #ifdef PDEBUG
	  puts("v 8.0");
	  #endif
	if (server) {
		ctx = SSL_CTX_new(SSLv23_server_method());
	} else {
		ctx = SSL_CTX_new(SSLv23_client_method());
	}
#endif



	
	/*
	 * Staticly set verify_depth, FIXME
	 *
	 * Following code by Peter Antman, up to SSL_new
	 *
	 * int verify_depth = 1;
	 */

	if ((!SSL_CTX_load_verify_locations(ctx,CAfile,CAdir)) ||
		(!SSL_CTX_set_default_verify_paths(ctx)))
		{
		fprintf(stderr,"error seting default verify locations\n");
		ERR_print_errors_fp(stderr);
		goto end;
		}
	if (!SetCertStuffSSL(ctx,cert,key))
		goto end;

	if (ciphers == NULL) {
	  #ifdef PDEBUG
	  puts("Stter defaultciphers");
	  #endif
		SSL_CTX_set_cipher_list(ctx,env_cipher); 
	} else {
	  #ifdef PDEBUG
	printf("Stter cipher %s\n", ciphers);
#endif

		SSL_CTX_set_cipher_list(ctx,ciphers);
	}

	  #ifdef PDEBUG
	printf("CAfile i socketcmd %s\n", CAfile);
#endif

	
	


	
	SSL_CTX_set_verify(ctx,verify,VerifyCallbackSSL);
	ssl = SSL_new(ctx);
	SSL_set_fd(ssl,sock);

	
	#ifdef PDEBUG
	printf("Efter SSL_set_fd %d\n", sock);
#endif
    }

    /*
     * Set the close-on-exec flag so that the socket will not get
     * inherited by child processes.
     */

    fcntl(sock, F_SETFD, FD_CLOEXEC);
    
    /*
     * Set kernel space buffering
     */

    TclSockMinimumBuffers(sock, SOCKET_BUFSIZE);

    asyncConnect = 0;
    status = 0;

/* new place for statePtr aloc - not good, what if needssl 0 */
    statePtr = (TcpState *) ckalloc((unsigned) sizeof(TcpState));
    statePtr->flags = 0;
    if (asyncConnect) {
        statePtr->flags = TCP_ASYNC_CONNECT;
	/* maybe FIXME fd instead of sock, as in orig */
    }
    statePtr->fd = sock;
    statePtr->orig_fd = sock;
    
    /* FIXME, ckeck to get real sockets to work */
    if (needssl) {
      statePtr->needssl = needssl;
      statePtr->ssl = ssl;
    } else {
      statePtr->needssl = 0;
    }

    statePtr->ctx = ctx;
    statePtr->sslerr = 0;
    statePtr->server = 0;

/* end here */

    if (server) {

	/*
	 * Set up to reuse server addresses automatically and bind to the
	 * specified port.
	 */
    	/*debug
	*/
#ifdef PDEBUG
	printf("Innan bind socket %d\n", sock);
#endif
	status = 1;
	(void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char *) &status,
		sizeof(status));
	status = bind(sock, (struct sockaddr *) &sockaddr,
                sizeof(struct sockaddr));
	if (status != -1) {
	    status = listen(sock, SOMAXCONN);
	} 
    } else {
	if (myaddr != NULL || myport != 0) { 
	    curState = 1;
	    (void) setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,
                    (char *) &curState, sizeof(curState));
	    status = bind(sock, (struct sockaddr *) &mysockaddr,
		    sizeof(struct sockaddr));
	    if (status < 0) {
		goto bindError;
	    }
	}

	/*
	 * Attempt to connect. The connect may fail at present with an
	 * EINPROGRESS but at a later time it will complete. The caller
	 * will set up a file handler on the socket if she is interested in
	 * being informed when the connect completes.
	 */

        if (async) {
#ifndef	USE_FIONBIO
            origState = fcntl(sock, F_GETFL);
            curState = origState | O_NONBLOCK;
            status = fcntl(sock, F_SETFL, curState);
#endif

#ifdef	USE_FIONBIO
            curState = 1;
            status = ioctl(sock, FIONBIO, &curState);
#endif            
        } else {
            status = 0;
        }
	
	if (status > -1) {
            status = connect(sock, (struct sockaddr *) &sockaddr,
                    sizeof(sockaddr));
            if (status < 0) {
                if (errno == EINPROGRESS) {
                    asyncConnect = 1;
                    status = 0;
                }
            }
        }
	if (needssl && (status > -1)) {
#ifdef PDEBUG
	printf("Innan SSL_connect i socketCmd %d\n", sock);
#endif
	SSL_set_connect_state(ssl); 
	/* while(!SSL_is_init_finished(ssl))  */
    		if ( (status = SSL_connect(ssl)) != 1) {
 #ifdef PDEBUG
	printf("Status frn connect %d\n", status);
#endif
			status = -1;
        		if (interp != NULL) {
			  /* Changed by Peter Antman */
			  if (verify_error != X509_V_OK )
				{
				  Tcl_AppendResult(interp, "SSL connect error ", "verify error: ", X509_verify_cert_error_string(verify_error), NULL);
			       
				}
			  else
				{
				  Tcl_AppendResult(interp, "SSL connect error ", NULL);
				
				}
        			
				
			}
    		}
/*	 */
		printf("Cipher is %s:", SSL_get_cipher(ssl));
		PEM_write_SSL_SESSION(stderr,SSL_get_session(ssl));
	} 



    }

bindError:
    if (status < 0) {
	if ( ssl ) { 
		SSL_free(ssl); 
	}
        if (interp != NULL) {
            Tcl_AppendResult(interp, "couldn't open socket: ",
                    Tcl_PosixError(interp), (char *) NULL);
        }
        if (sock != -1) {
            close(sock);
        }
        return NULL;
    }

    /*
     * Allocate a new TcpState for this socket.
     */
#ifdef PDEBUG
	printf("Innan allock av statePtr i socketcmd %d\n", sock);
#endif
	/* old place of statePtr aloc */

    
/* end here */
    return statePtr;

addressError:
    if (sock != -1) {
        close(sock);
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "couldn't open socket: ",
		Tcl_PosixError(interp), (char *) NULL);
    }
	if ( ssl ) { 
		SSL_free(ssl); 
	}
    return NULL;

    /* By Peter Antman */
end:
    if (ctx != NULL)
      SSL_CTX_free(ctx);
    Tcl_AppendResult(interp, "couldnt set certstuff", NULL);
    return NULL;
    
}

/*
 *----------------------------------------------------------------------
 *
 * CreateSocketAddress --
 *
 *	This function initializes a sockaddr structure for a host and port.
 *
 * Results:
 *	1 if the host was valid, 0 if the host could not be converted to
 *	an IP address.
 *
 * Side effects:
 *	Fills in the *sockaddrPtr structure.
 *
 *----------------------------------------------------------------------
 */

static int
CreateSocketAddress(sockaddrPtr, host, port)
    struct sockaddr_in *sockaddrPtr;	/* Socket address */
    char *host;				/* Host.  NULL implies INADDR_ANY */
    int port;				/* Port number */
{
    struct hostent *hostent;		/* Host database entry */
    struct in_addr addr;		/* For 64/32 bit madness */

    (void) memset((VOID *) sockaddrPtr, '\0', sizeof(struct sockaddr_in));
    sockaddrPtr->sin_family = AF_INET;
    sockaddrPtr->sin_port = htons((unsigned short) (port & 0xFFFF));
    if (host == NULL) {
	addr.s_addr = INADDR_ANY;
    } else {
        addr.s_addr = inet_addr(host);
        if (addr.s_addr == -1) {
            hostent = gethostbyname(host);
            if (hostent != NULL) {
                memcpy((VOID *) &addr,
                        (VOID *) hostent->h_addr_list[0],
                        (size_t) hostent->h_length);
            } else {
#ifdef	EHOSTUNREACH
                errno = EHOSTUNREACH;
#else
#ifdef ENXIO
                errno = ENXIO;
#endif
#endif
                return 0;	/* error */
            }
        }
    }
        
    /*
     * NOTE: On 64 bit machines the assignment below is rumored to not
     * do the right thing. Please report errors related to this if you
     * observe incorrect behavior on 64 bit machines such as DEC Alphas.
     * Should we modify this code to do an explicit memcpy?
     */

    sockaddrPtr->sin_addr.s_addr = addr.s_addr;
    return 1;	/* Success. */
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenSSLClient --
 *
 *	Opens a TCP client socket and creates a channel around it.
 *
 * Results:
 *	The channel or NULL if failed.  An error message is returned
 *	in the interpreter on failure.
 *
 * Side effects:
 *	Opens a client socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenSSLClient(interp, port, host, myaddr, myport, async,needssl,key,cert,CAfile,CAdir,ciphers,verify)
    Tcl_Interp *interp;			/* For error reporting; can be NULL. */
    int port;				/* Port number to open. */
    char *host;				/* Host on which to open port. */
    char *myaddr;			/* Client-side address */
    int myport;				/* Client-side port */
    int async;				/* If nonzero, attempt to do an
                                         * asynchronous connect. Otherwise
                                         * we do a blocking connect. */
    int needssl;                        /* Do a secure client */
	char *key;
	char *cert;
	char *CAfile;
	char *CAdir;
	char *ciphers;
	int verify;
{
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */
#ifdef PDEBUG
	printf("Fre socket i opensslclient %d\n", port);
	
#endif
    statePtr = CreateSocket(interp, port, host, 0, myaddr, myport, async,needssl,key,cert,CAfile,CAdir,ciphers,verify);
    if (statePtr == NULL) {
	return NULL;
    }
#ifdef PDEBUG
	printf("Efter socket i opensslclient %d\n", statePtr->fd);
#endif

    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;

    sprintf(channelName, "sock%d", statePtr->fd);

    statePtr->channel = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
    if (Tcl_SetChannelOption(interp, statePtr->channel, "-translation",
	    "auto crlf") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, statePtr->channel);
        return NULL;
    }
    return statePtr->channel;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_MakeTcpClientChannel --
 *
 *	Creates a Tcl_Channel from an existing client TCP socket.
 *
 * Results:
 *	The Tcl_Channel wrapped around the preexisting TCP socket.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_MakeTcpClientChannel(sock)
    ClientData sock;		/* The socket to wrap up into a channel. */
{
    TcpState *statePtr;
    char channelName[20];

    statePtr = (TcpState *) ckalloc((unsigned) sizeof(TcpState));
    statePtr->fd = (int) sock;
    statePtr->orig_fd = (int) sock;
    statePtr->acceptProc = NULL;
    statePtr->acceptProcData = (ClientData) NULL;

    sprintf(channelName, "sock%d", statePtr->fd);
    
    statePtr->channel = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, (TCL_READABLE | TCL_WRITABLE));
    if (Tcl_SetChannelOption((Tcl_Interp *) NULL, statePtr->channel,
	    "-translation", "auto crlf") == TCL_ERROR) {
        Tcl_Close((Tcl_Interp *) NULL, statePtr->channel);
        return NULL;
    }
    return statePtr->channel;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_OpenSSLServer --
 *
 *	Opens a TCP server socket and creates a channel around it.
 *
 * Results:
 *	The channel or NULL if failed. If an error occurred, an
 *	error message is left in interp->result if interp is
 *	not NULL.
 *
 * Side effects:
 *	Opens a server socket and creates a new channel.
 *
 *----------------------------------------------------------------------
 */

Tcl_Channel
Tcl_OpenSSLServer(interp, port, myHost, acceptProc, acceptProcData, needssl,key,cert,CAfile,CAdir,ciphers,verify)
    Tcl_Interp *interp;			/* For error reporting - may be
                                         * NULL. */
    int port;				/* Port number to open. */
    char *myHost;			/* Name of local host. */
    Tcl_TcpAcceptProc *acceptProc;	/* Callback for accepting connections
                                         * from new clients. */
    ClientData acceptProcData;		/* Data for the callback. */
    int needssl;                        /* Create a secure server  */
    char *key;
    char *cert;
	char *CAfile;
	char *CAdir;
	char *ciphers;
	int verify;
{
    TcpState *statePtr;
    char channelName[20];

    /*
     * Create a new client socket and wrap it in a channel.
     */
		/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan CreateSocket i opensslserver %d\n", port);
#endif
    statePtr = CreateSocket(interp, port, myHost, 1, NULL, 0, 0,needssl,key,cert,CAfile,CAdir,ciphers,verify);
    if (statePtr == NULL) {
	return NULL;
    }

    statePtr->acceptProc = acceptProc;
    statePtr->acceptProcData = acceptProcData;

    /*
     * Set up the callback mechanism for accepting connections
     * from new clients.
     */
	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan createfilehandler i openserver %d\n", statePtr->fd);
#endif
    Tcl_CreateFileHandler(statePtr->fd, TCL_READABLE, TcpAccept,
            (ClientData) statePtr);
	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Efter createfilehandler i openserver %d\n", statePtr->fd);
#endif
    sprintf(channelName, "sock%d", statePtr->fd);
    statePtr->channel = Tcl_CreateChannel(&tcpChannelType, channelName,
            (ClientData) statePtr, 0);
    return statePtr->channel;
}


static void doSSLAccept (sockState,newSockState,newsock)
    TcpState *sockState;		/* Client data of server socket. */
    TcpState *newSockState;		/* Client data of server socket. */
	int newsock ;
{
	       
		
		
		newSockState->sslerr = 0;
	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan key i doSSLAccept %d\n", newsock);
#endif
		


		newSockState->ctx = sockState->ctx;

		newSockState->ssl = SSL_new(newSockState->ctx);

    		SSL_set_fd(newSockState->ssl, newsock);



}

/*
 *----------------------------------------------------------------------
 *
 * TcpAccept --
 *	Accept a TCP socket connection.  This is called by the event loop.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Creates a new connection socket. Calls the registered callback
 *	for the connection acceptance mechanism.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
static void
TcpAccept(data, mask)
    ClientData data;			/* Callback token. */
    int mask;				/* Not used. */
{
    TcpState *sockState;		/* Client data of server socket. */
    int newsock;			/* The new client socket */
    TcpState *newSockState;		/* State for new socket. */
    struct sockaddr_in addr;		/* The remote address */
    int len;				/* For accept interface */
    char channelName[20];

    sockState = (TcpState *) data;

    len = sizeof(struct sockaddr_in);
	#ifdef PDEBUG
	puts("Innan newsocket");
#endif

    newsock = accept(sockState->fd, (struct sockaddr *)&addr, &len);
    if (newsock < 0) {
        return;
    }
	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Efter accept i TclAccept %d\n", newsock);
#endif
    /*
     * Set close-on-exec flag to prevent the newly accepted socket from
     * being inherited by child processes.
     */

    (void) fcntl(newsock, F_SETFD, FD_CLOEXEC);
    
    newSockState = (TcpState *) ckalloc((unsigned) sizeof(TcpState));

    newSockState->flags = 0;
    newSockState->fd = newsock;
    newSockState->orig_fd = sockState->fd;
    newSockState->acceptProc = (Tcl_TcpAcceptProc *) NULL;
    newSockState->acceptProcData = (ClientData) NULL;
    newSockState->server = 2;

    if (sockState->needssl) {
      newSockState->needssl = 1;
    } else {
      newSockState->needssl = 0;
    }
    
    /* FIXME, test to get real sockets working */
    /* newSockState->ssl = NULL; */ 
        
    sprintf(channelName, "sock%d", newsock);
    newSockState->channel = Tcl_CreateChannel(&tcpChannelType, channelName,
	    (ClientData) newSockState, (TCL_READABLE | TCL_WRITABLE));

    Tcl_SetChannelOption((Tcl_Interp *) NULL, newSockState->channel,
	    "-translation", "auto crlf");
	
	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan doSSLAcept i TclAccept %d\n", newsock);
#endif
    if ( sockState->needssl ) 
	doSSLAccept(sockState,newSockState,newsock);

    if (sockState->acceptProc != (Tcl_TcpAcceptProc *) NULL) {
	(sockState->acceptProc) (sockState->acceptProcData,
		newSockState->channel, inet_ntoa(addr.sin_addr),
		ntohs(addr.sin_port));
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Tcl_SSLSocketCmd --
 * replacement for Tcl_SocketCmd
 *
 *	This procedure is invoked to process the "socket" Tcl command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Creates a socket based channel.
 *
 *----------------------------------------------------------------------
 */

int
Tcl_SSLSocketCmd(notUsed, interp, argc, argv)
    ClientData notUsed;			/* Not used. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    int a, server, port;
    char *arg, *copyScript, *host, *script;
    char *myaddr = NULL;
    int myport = 0;
    int async = 0;
    int ssl = 0;
    char *key = NULL;
    char *cert = NULL;
    char *CAfile = NULL;
    char *CAdir = NULL;
    char *ciphers = NULL;
    int verify = SSL_VERIFY_NONE;
    int ver = 0;
    int Ver = 0;
    Tcl_Channel chan;
    AcceptCallback *acceptCallbackPtr;
    
    server = 0;
    script = NULL;
    verify_depth = 0;


    if (TclHasSockets(interp) != TCL_OK) {
	return TCL_ERROR;
    }

    for (a = 1; a < argc; a++) {
        arg = argv[a];
	if (arg[0] == '-') {
	    if (strcmp(arg, "-server") == 0) {
                if (async == 1) {
                    Tcl_AppendResult(interp,
                            "cannot set -async option for server sockets",
                            (char *) NULL);
                    return TCL_ERROR;
                }
		server = 1;
		a++;
		if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -server option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                script = argv[a];
            } else if (strcmp(arg, "-ssl") == 0) {
		ssl = 1;
            } else if (strcmp(arg, "-cert") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -cert option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                cert = argv[a];
            } else if (strcmp(arg, "-key") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -key option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                key = argv[a];
	    } else if (strcmp(arg, "-CAfile") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -CAfile option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                CAfile = argv[a];
	    } else if (strcmp(arg, "-CAdir") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -CAdir option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                CAdir = argv[a];
	    } else if (strcmp(arg, "-ciphers") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -ciphers option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		
                ciphers = argv[a];

	    } else if (strcmp(arg, "-verify") == 0) {
	      a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -verify option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		verify_depth = atoi(argv[a]);
                 ver=1;
	    } else if (strcmp(arg, "-Verify") == 0) {
	      a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -Verify option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		verify_depth = atoi(argv[a]);
                  Ver=1;
	    } else if (strcmp(arg, "-maxacloops") == 0) {
	      a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -maxacloops option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		max_ac_loops = atoi(argv[a]);
            } else if (strcmp(arg, "-myaddr") == 0) {
	      a++;
              if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myaddr option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
                myaddr = argv[a];
            } else if (strcmp(arg, "-myport") == 0) {
		a++;
                if (a >= argc) {
		    Tcl_AppendResult(interp,
			    "no argument given for -myport option",
                            (char *) NULL);
		    return TCL_ERROR;
		}
		if (TclSockGetPort(interp, argv[a], "tcp", &myport)
                    != TCL_OK) {
		    return TCL_ERROR;
		}
            } else if (strcmp(arg, "-async") == 0) {
                if (server == 1) {
                    Tcl_AppendResult(interp,
                            "cannot set -async option for server sockets",
                            (char *) NULL);
                    return TCL_ERROR;
                }
                async = 1;
	    } else {
		Tcl_AppendResult(interp, "bad option \"", arg,
                        "\", must be -async, -myaddr, -myport, or -server",
                        (char *) NULL);
		return TCL_ERROR;
	    }
	} else {
	    break;
	}
    }
    if (server) {
        host = myaddr;		/* NULL implies INADDR_ANY */
	if (myport != 0) {
	    Tcl_AppendResult(interp, "Option -myport is not valid for servers",
		    NULL);
	    return TCL_ERROR;
	}
    } else if (a < argc) {
	host = argv[a];
	a++;
    } else {
wrongNumArgs:
	Tcl_AppendResult(interp, "wrong # args: should be either:\n",
		argv[0],
                " ?-ssl? ?-myaddr addr? ?-myport myport? ?-async? ?-verify NUM? ?-cert Certificate? ?-key privateKey? ?-CAfile CAfile? ?-CAdir CAdir? ?-ciphers cipherlist? host port\n",
		argv[0],
                " -server command ?-ssl? ?-verify NUM? ?-Verify NUM? ?-key privateKey? ?-cert Certificate? ?-CAfile CAfile? ?-CAdir CAdir? ?-ciphers cipherlist? ?maxacloops NUM? ?-myaddr addr? port\n",
                (char *) NULL);
        return TCL_ERROR;
    }

    if (a == argc-1) {
	if (TclSockGetPort(interp, argv[a], "tcp", &port) != TCL_OK) {
	    return TCL_ERROR;
	}
    } else {
	goto wrongNumArgs;
    }
    
    /* verify_depth = 1; */

    if (server) {
	if ( ssl && !cert ) {
		Tcl_AppendResult(interp, "Certificate is required for an SSL server\n",
			"Use (-cert filename) to set!",NULL);
		return TCL_ERROR;
	}
	
	if ( ssl && ver )
	   verify=SSL_VERIFY_PEER|SSL_VERIFY_CLIENT_ONCE;
	if ( ssl && Ver )
	  verify=SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT|SSL_VERIFY_CLIENT_ONCE;
		/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan accept Callback i socketCmd %d\n", server);
#endif
        acceptCallbackPtr = (AcceptCallback *) ckalloc((unsigned)
                sizeof(AcceptCallback));
        copyScript = ckalloc((unsigned) strlen(script) + 1);
        strcpy(copyScript, script);
        acceptCallbackPtr->script = copyScript;
        acceptCallbackPtr->interp = interp;

	/* DEBUG
	*/
#ifdef PDEBUG
	printf("Innan openserver i socketCmd %d\n", server);
#endif
        chan = Tcl_OpenSSLServer(interp, port, host, AcceptCallbackProc,
                (ClientData) acceptCallbackPtr,ssl,key,cert,CAfile,CAdir,ciphers,verify);
        if (chan == (Tcl_Channel) NULL) {
            ckfree(copyScript);
            ckfree((char *) acceptCallbackPtr);
            return TCL_ERROR;
        }

        /*
         * Register with the interpreter to let us know when the
         * interpreter is deleted (by having the callback set the
         * acceptCallbackPtr->interp field to NULL). This is to
         * avoid trying to eval the script in a deleted interpreter.
         */

        RegisterTcpServerInterpCleanup(interp, acceptCallbackPtr);
        
        /*
         * Register a close callback. This callback will inform the
         * interpreter (if it still exists) that this channel does not
         * need to be informed when the interpreter is deleted.
         */
        
        Tcl_CreateCloseHandler(chan, TcpServerCloseProc,
                (ClientData) acceptCallbackPtr);
    } else {
	#ifdef PDEBUG
	printf("Innan opensslclient in socketcmd %d\n", port);
#endif
	if ( ssl && ver ) {
	  #ifdef PDEBUG
	printf("Stter verify fr Client %d\n", verify);
#endif
	  verify = SSL_VERIFY_PEER;
	}
        chan = Tcl_OpenSSLClient(interp, port, host, myaddr, myport, async,ssl,key,cert,CAfile,CAdir,ciphers,verify);
        if (chan == (Tcl_Channel) NULL) {
            return TCL_ERROR;
        }
    }
    Tcl_RegisterChannel(interp, chan);            
    Tcl_AppendResult(interp, Tcl_GetChannelName(chan), (char *) NULL);
    
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * TclSSL_Init --
 *
 *	This is a package initialization procedure, which is called
 *	by Tcl when this package is to be added to an interpreter.
 *
 * Results:  TclSSL configured and loaded
 *
 * Side effects:
 *	 rename some of the standard tcl functions dealing with sockets.  This
 *       will let existing code run correctly with ssl.
 *
 *----------------------------------------------------------------------
 */

int
Ssltcl_Init(interp)
    Tcl_Interp *interp;		/* Interpreter in which the package is
				 * to be made available. */
{
    int code;

	ERR_load_SSL_strings();
    code = Tcl_PkgProvide(interp, "SSLtcl", "1.0");
    if (code != TCL_OK) {
	return code;
    }
    Tcl_CreateCommand(interp, "socket", Tcl_SSLSocketCmd , (ClientData) 0,
	    (Tcl_CmdDeleteProc *) NULL);
    return TCL_OK;
}















