/*
 * $Id: auth_context.c,v 1.2 1997/01/07 18:04:52 kenh Exp $
 *
 * auth_context - Glue for manipulationg authorization contexts for
 *		  Tcl-Kerberos 5
 */

#ifndef LINT
static char rcsid[]=
	"$Id: auth_context.c,v 1.2 1997/01/07 18:04:52 kenh Exp $";
#endif

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <krb5.h>
#include <com_err.h>
#include <tcl.h>

#include "tcl-krb5.h"

Tcl_HashTable AuthCon_Table;
int AuthCon_Counter = 0;

/*
 * Functions for manipulating auth_contexts
 *
 * The situation here is much like it is with credential caches; we
 * have a hash table that keeps the string to auth_context mapping
 * for us.  We're going to use "authcon<num>" for the string handle.
 */

/* Glue to krb5_auth_con_init */

int
Krb5AuthConInitCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		   char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_auth_context auth_context;
	char auth_name[256];

	if (argc != 1) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (CreateAuthContext(interp, context, &auth_context, auth_name) !=
	    TCL_OK)
		return TCL_ERROR;
	
	Tcl_SetResult(interp, auth_name, TCL_VOLATILE);

	return TCL_OK;
}

/* Glue to krb5_auth_con_free */

int
Krb5AuthConFreeCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		   char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_auth_context auth_context;
	krb5_error_code code;
	Tcl_HashEntry *entry;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, &entry)
	    != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_free(context, auth_context))) {
		Tcl_AppendResult(interp, "krb5_auth_con_free failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DeleteHashEntry(entry);

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_getkey
 */

int
Krb5AuthConGetkeyCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_keyblock *keyblock;
	Tcl_DString dstr;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_getkey(context, auth_context, &keyblock))) {
		Tcl_AppendResult(interp, "krb5_auth_con_getkey failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	if (keyblock == NULL) {
		Tcl_AppendResult(interp, "no key in that auth_context",
				 (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringInit(&dstr);

	if (KeyblockToDString(interp, keyblock, &dstr) != TCL_OK) {
		Tcl_DStringFree(&dstr);
		return TCL_ERROR;
	}

	Tcl_DStringResult(interp, &dstr);

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_getlocalsubkey
 */

int
Krb5AuthConGetlocalsubkeyCmd(ClientData clientData, Tcl_Interp *interp,
			     int argc, char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_keyblock *keyblock;
	Tcl_DString dstr;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_getlocalsubkey(context, auth_context,
						 &keyblock))) {
		Tcl_AppendResult(interp, "krb5_auth_con_getlocalsubkey "
				 "failed: ", error_message(code),
				 (char *) NULL);
		return TCL_ERROR;
	}

	if (keyblock == NULL) {
		Tcl_AppendResult(interp, "no local subkey in that auth_context",
				 (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringInit(&dstr);

	if (KeyblockToDString(interp, keyblock, &dstr) != TCL_OK) {
		Tcl_DStringFree(&dstr);
		return TCL_ERROR;
	}

	Tcl_DStringResult(interp, &dstr);

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_getremotesubkey
 */

int
Krb5AuthConGetremotesubkeyCmd(ClientData clientData, Tcl_Interp *interp,
			      int argc, char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_keyblock *keyblock;
	Tcl_DString dstr;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_getremotesubkey(context, auth_context,
						  &keyblock))) {
		Tcl_AppendResult(interp, "krb5_auth_con_getremotesubkey "
				 "failed: ", error_message(code),
				 (char *) NULL);
		return TCL_ERROR;
	}

	if (keyblock == NULL) {
		Tcl_AppendResult(interp, "no remote subkey in that "
				 "auth_context", (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringInit(&dstr);

	if (KeyblockToDString(interp, keyblock, &dstr) != TCL_OK) {
		Tcl_DStringFree(&dstr);
		return TCL_ERROR;
	}

	Tcl_DStringResult(interp, &dstr);

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_getflags
 */

int
Krb5AuthConGetflagsCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_int32 flags;
	Tcl_DString dstr;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_getflags(context, auth_context, &flags))) {
		Tcl_AppendResult(interp, "krb5_auth_con_getflags failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringInit(&dstr);

	FlagToDString(AuthConFlagTable, flags, &dstr);

	Tcl_DStringResult(interp, &dstr);

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_setflags
 */

int
Krb5AuthConSetflagsCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_int32 flags = 0;
	int i, tempflag;

	if (argc < 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context ?flag? ?flag ...?\"",
				 (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	for (i = 2; i < argc; i++) {
		if (StringToNum(AuthConFlagTable, argv[i], &tempflag) !=
		    TCL_OK) {
			Tcl_AppendResult(interp, "Invalid flag: \"", argv[i],
					 "\"", (char *) NULL);
			return TCL_ERROR;
		}
		flags |= tempflag;
	}
		
	if ((code = krb5_auth_con_setflags(context, auth_context, flags))) {
		Tcl_AppendResult(interp, "krb5_auth_con_setflags failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_genaddrs
 */

int
Krb5AuthConGenaddrsCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		     char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	int flags = 0;
	int i, tempflag, fd;
	Tcl_Channel channel;
	Tcl_File file;

	if (argc < 3) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " auth_context channelId ?flag? ?flag ...?\"",
				 (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;

	/*
	 * Get a file descriptor from the channel ID
	 */

	channel = Tcl_GetChannel(interp, argv[2], NULL);

	file = Tcl_GetChannelFile(channel, TCL_WRITABLE);

	if (file == NULL) {
		Tcl_AppendResult(interp, "cannot get Unix file descriptor "
				 "for channel \"", argv[2], "\"",
				 (char *) NULL);
		return TCL_ERROR;
	}

	fd = (int) Tcl_GetFileInfo(file, NULL);

	for (i = 3; i < argc; i++) {
		if (StringToNum(GenaddrsFlagTable, argv[i], &tempflag) !=
		    TCL_OK) {
			Tcl_AppendResult(interp, "Invalid flag: \"", argv[i],
					 "\"", (char *) NULL);
			return TCL_ERROR;
		}
		flags |= tempflag;
	}
		
	if ((code = krb5_auth_con_genaddrs(context, auth_context, fd, flags))) {
		Tcl_AppendResult(interp, "krb5_auth_con_genaddrs failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_setrcache
 */

int
Krb5AuthConSetrcacheCmd(ClientData clientData, Tcl_Interp *interp, int argc,
			char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_rcache rcache;

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context rcache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK)
		return TCL_ERROR;
	
	if (GetReplayCache(interp, argv[2], &rcache, NULL) != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_auth_con_setrcache(context, auth_context, rcache))) {
		Tcl_AppendResult(interp, "krb5_auth_con_setrcache failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	return TCL_OK;
}

/*
 * Glue to krb5_auth_con_setuseruserkey
 */

int
Krb5AuthConSetuseruserkeyCmd(ClientData clientData, Tcl_Interp *interp,
			     int argc, char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_auth_context auth_context;
	krb5_keyblock keyblock;

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "auth_context rcache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetAuthContext(interp, context, argv[1], &auth_context, NULL)
	    != TCL_OK) {
		return TCL_ERROR;
	}
	
	if (StringToKeyblock(interp, argv[2], &keyblock) != TCL_OK) {
		return TCL_ERROR;
	}
	
	if ((code = krb5_auth_con_setuseruserkey(context, auth_context,
						 &keyblock))) {
		Tcl_AppendResult(interp, "krb5_auth_con_setuseruserkey "
				 "failed: ", error_message(code));
		krb5_free_keyblock_contents(context, &keyblock);
		return TCL_ERROR;
	}

	krb5_free_keyblock_contents(context, &keyblock);

	return TCL_OK;
}

/*
 * Internal function to create authorization contexts
 */

int
CreateAuthContext(Tcl_Interp *interp, krb5_context context,
		  krb5_auth_context *auth_context, char *auth_name)
{
	krb5_error_code code;
	Tcl_HashEntry *entry;
	int new;

	if ((code = krb5_auth_con_init(context, auth_context))) {
		Tcl_AppendResult(interp, "krb5_auth_con_init failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	sprintf(auth_name, "authcon%d", AuthCon_Counter++);

	entry = Tcl_CreateHashEntry(&AuthCon_Table, auth_name, &new);

	Tcl_SetHashValue(entry, (ClientData) *auth_context);

	return TCL_OK;
}

/*
 * Internal function to find authorization contexts
 */

int
GetAuthContext(Tcl_Interp *interp, krb5_context context, char *auth_name,
	      krb5_auth_context *auth_context, Tcl_HashEntry **ret_entry)
{
	Tcl_HashEntry *entry;

	if ((entry = Tcl_FindHashEntry(&AuthCon_Table, auth_name)) == NULL) {
		Tcl_AppendResult(interp, "can not find auth_context named \"",
				 auth_name, "\"", (char *) NULL);
		return TCL_ERROR;
	}

	*auth_context = (krb5_auth_context) Tcl_GetHashValue(entry);

	if (ret_entry != NULL)
		*ret_entry = entry;
	
	return TCL_OK;
}
