/*
 * $Id: ccache.c,v 1.1 1997/01/02 23:01:22 kenh Exp $
 *
 * ccache - Glue routines for credential cache access functions for
 *	    Tcl-Kerberos 5
 *
 */

#ifndef LINT
static char rcsid[]=
	"$Id: ccache.c,v 1.1 1997/01/02 23:01:22 kenh Exp $";
#endif

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <krb5.h>
#include <com_err.h>
#include <tcl.h>

#include "tcl-krb5.h"

Tcl_HashTable Ccache_Table;
int Ccache_Counter = 0;

/*
 * Glue to the krb5_cc_* functions
 *
 * Since we have to maintain a ccache pointer, we're going to create a
 * "ccache" handle.  This is just an ascii string which is stored in a
 * hash table, which points to the _real_ ccache info.
 */

/* Glue to krb5_cc_default() */

int
Krb5CcDefaultCmd(ClientData clientData, Tcl_Interp *interp, int argc,
	      char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_ccache ccache;
	krb5_error_code code;
	char ccache_name[256];
	Tcl_HashEntry *entry;
	int new;

	if (argc != 1) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 "\"", (char *) NULL);
		return TCL_ERROR;
	}

	if ((code = krb5_cc_default(context, &ccache))) {
		Tcl_AppendResult(interp, "krb5_cc_default failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	sprintf(ccache_name, "ccache%d", Ccache_Counter++);

	entry = Tcl_CreateHashEntry(&Ccache_Table, ccache_name, &new);

	Tcl_SetHashValue(entry, (ClientData) ccache);

	Tcl_SetResult(interp, ccache_name, TCL_VOLATILE);

	return TCL_OK;
}

/* Glue to krb5_cc_get_principal */

int
Krb5CcGetPrincipalCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		   char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_ccache ccache;
	krb5_error_code code;
	krb5_principal princ;
	char *princ_string;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " ccache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetKrb5Ccache(interp, context, argv[1], &ccache, NULL) != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_cc_get_principal(context, ccache, &princ))) {
		Tcl_AppendResult(interp, "krb5_cc_get_principal failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	if ((code = krb5_unparse_name(context, princ, &princ_string))) {
		Tcl_AppendResult(interp, "krb5_unparse_name failed: ",
				 error_message(code), (char *) NULL);
		krb5_free_principal(context, princ);
		return TCL_ERROR;
	}

	Tcl_SetResult(interp, princ_string, TCL_VOLATILE);

	free(princ_string);
	krb5_free_principal(context, princ);

	return TCL_OK;
}

/* Glue to krb5_cc_close */

int
Krb5CcCloseCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		   char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_ccache ccache;
	Tcl_HashEntry *entry;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " ccache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetKrb5Ccache(interp, context, argv[1], &ccache, &entry) != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_cc_close(context, ccache))) {
		Tcl_AppendResult(interp, "krb5_cc_close failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DeleteHashEntry(entry);

	return TCL_OK;
}

/* Glue to krb5_cc_destroy */

int
Krb5CcDestroyCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		   char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_ccache ccache;
	Tcl_HashEntry *entry;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " ccache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetKrb5Ccache(interp, context, argv[1], &ccache, &entry) != TCL_OK)
		return TCL_ERROR;

	if ((code = krb5_cc_destroy(context, ccache))) {
		Tcl_AppendResult(interp, "krb5_cc_close failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DeleteHashEntry(entry);

	return TCL_OK;
}

/*
 * Internal function to find credential caches
 */

int
GetKrb5Ccache(Tcl_Interp *interp, krb5_context context, char *ccache_name,
	      krb5_ccache *ccache, Tcl_HashEntry **ret_entry)
{
	Tcl_HashEntry *entry;

	if ((entry = Tcl_FindHashEntry(&Ccache_Table, ccache_name)) == NULL) {
		Tcl_AppendResult(interp, "can not find ccache named \"",
				 ccache_name, "\"", (char *) NULL);
		return TCL_ERROR;
	}

	*ccache = (krb5_ccache) Tcl_GetHashValue(entry);

	if (ret_entry != NULL)
		*ret_entry = entry;
	
	return TCL_OK;
}
