/*
 * $Id: rcache.c,v 1.2 1997/01/07 18:04:52 kenh Exp $
 *
 * rcache - Routines to manipulate replay caches for Tcl-Kerberos 5
 */

#ifndef LINT
static char rcsid[]=
	"$Id: rcache.c,v 1.2 1997/01/07 18:04:52 kenh Exp $";
#endif

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <krb5.h>
#include <com_err.h>
#include <tcl.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "tcl-krb5.h"

Tcl_HashTable Rcache_Table;
int Rcache_Counter = 0;

/*
 * Interface to some of the replay cache functions.  We use a string
 * key to a hash table to represent replay cache information.
 */

/*
 * Glue to krb5_get_server_rcache
 */

int
Krb5GetServerRcacheCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		       char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_data princ_data;
	krb5_rcache rcache;
	char rcache_name[256];

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " princ_data\"", (char *) NULL);
		return TCL_ERROR;
	}

	princ_data.data = argv[1];
	princ_data.length = strlen(argv[1]);

	if ((code = krb5_get_server_rcache(context,
					   (const krb5_data *) &princ_data,
					   &rcache))) {
		Tcl_AppendResult(interp, "krb5_get_server_rcache failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	if (CreateReplayCache(rcache, rcache_name) != TCL_OK) {
		Tcl_AppendResult(interp, "Could not create replay cache",
				 (char *) NULL);
		krb5_rc_close(context, rcache);
		return TCL_ERROR;
	}

	Tcl_SetResult(interp, rcache_name, TCL_VOLATILE);

	return TCL_OK;
}

/*
 * Glue to krb5_rc_close
 */

int
Krb5RcCloseCmd(ClientData clientData, Tcl_Interp *interp, int argc,
	       char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_rcache rcache;
	Tcl_HashEntry *entry;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " replay_cache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetReplayCache(interp, argv[1], &rcache, &entry) != TCL_OK) {
		return TCL_ERROR;
	}

	if ((code = krb5_rc_close(context, rcache))) {
		Tcl_AppendResult(interp, "krb5_rc_close failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DeleteHashEntry(entry);

	return TCL_OK;
}

/*
 * Glue to krb5_rc_destroy
 */

int
Krb5RcDestroyCmd(ClientData clientData, Tcl_Interp *interp, int argc,
		 char *argv[])
{
	krb5_context context = (krb5_context) clientData;
	krb5_error_code code;
	krb5_rcache rcache;
	Tcl_HashEntry *entry;

	if (argc != 2) {
		Tcl_AppendResult(interp, "wrong # args, should be \"", argv[0],
				 " replay_cache\"", (char *) NULL);
		return TCL_ERROR;
	}

	if (GetReplayCache(interp, argv[1], &rcache, &entry) != TCL_OK) {
		return TCL_ERROR;
	}

	if ((code = krb5_rc_destroy(context, rcache))) {
		Tcl_AppendResult(interp, "krb5_rc_destroy failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DeleteHashEntry(entry);

	return TCL_OK;
}

/*
 * Internal function to create replay caches.  Note that this function
 * doesn't actually _create_ replay caches; it just stores them into the
 * hash table.
 */

int
CreateReplayCache(krb5_rcache rcache, char *rcache_name)
{
	Tcl_HashEntry *entry;
	int new;

	sprintf(rcache_name, "rcache%d", Rcache_Counter++);

	entry = Tcl_CreateHashEntry(&Rcache_Table, rcache_name, &new);

	Tcl_SetHashValue(entry, (ClientData) rcache);

	return TCL_OK;
}

/*
 * Internal function to find replay caches
 */

int
GetReplayCache(Tcl_Interp *interp, char *rcache_name, krb5_rcache *rcache,
	       Tcl_HashEntry **ret_entry)
{
	Tcl_HashEntry *entry;

	if ((entry = Tcl_FindHashEntry(&Rcache_Table, rcache_name)) == NULL) {
		Tcl_AppendResult(interp, "can not find replay cache named \"",
				 rcache_name, "\"", (char *) NULL);
		return TCL_ERROR;
	}

	*rcache = (krb5_rcache) Tcl_GetHashValue(entry);

	if (ret_entry != NULL)
		*ret_entry = entry;
	
	return TCL_OK;
}
