/*
 * $Id: conversion.c,v 1.3 1997/01/07 18:04:52 kenh Exp $
 *
 * conversion - Various functions used for converting data to and from Tcl
 *		for Tcl-Kerberos 5
 */

#ifndef LINT
static char rcsid[]=
	"$Id: conversion.c,v 1.3 1997/01/07 18:04:52 kenh Exp $";
#endif

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <krb5.h>
#include <com_err.h>
#include <tcl.h>

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "tcl-krb5.h"

/*
 * These are tables that hold string constants that get converted to
 * flag values.
 */

StringToNumTable SnameTypeTable[] = {
	{ "KRB5_NT_SRV_HST", KRB5_NT_SRV_HST },
	{ "KRB5_NT_UNKNOWN", KRB5_NT_UNKNOWN },
	{ "KRB5_NT_SRV_INST", KRB5_NT_SRV_INST },
	{ "KRB5_NT_SRV_XHST", KRB5_NT_SRV_XHST },
	{ "KRB5_NT_UID", KRB5_NT_UID },
	{ NULL, 0 },
};

StringToNumTable ApOptsTable[] = {
	{ "AP_OPTS_MUTUAL_REQUIRED", AP_OPTS_MUTUAL_REQUIRED },
	{ "AP_OPTS_USE_SESSION_KEY", AP_OPTS_USE_SESSION_KEY },
	{ "AP_OPTS_USE_SUBKEY", AP_OPTS_USE_SUBKEY },
	{ NULL, 0 },
};

StringToNumTable AuthConFlagTable[] = {
	{ "KRB5_AUTH_CONTEXT_DO_TIME", KRB5_AUTH_CONTEXT_DO_TIME },
	{ "KRB5_AUTH_CONTEXT_RET_TIME", KRB5_AUTH_CONTEXT_RET_TIME },
	{ "KRB5_AUTH_CONTEXT_DO_SEQUENCE", KRB5_AUTH_CONTEXT_DO_SEQUENCE },
	{ "KRB5_AUTH_CONTEXT_RET_SEQUENCE", KRB5_AUTH_CONTEXT_RET_SEQUENCE },
	{ NULL, 0 },
};

StringToNumTable GenaddrsFlagTable[] = {
	{ "KRB5_AUTH_CONTEXT_GENERATE_LOCAL_ADDR", KRB5_AUTH_CONTEXT_GENERATE_LOCAL_ADDR },
	{ "KRB5_AUTH_CONTEXT_GENERATE_REMOTE_ADDR", KRB5_AUTH_CONTEXT_GENERATE_REMOTE_ADDR },
	{ "KRB5_AUTH_CONTEXT_GENERATE_LOCAL_FULL_ADDR", KRB5_AUTH_CONTEXT_GENERATE_LOCAL_FULL_ADDR },
	{ "KRB5_AUTH_CONTEXT_GENERATE_REMOTE_FULL_ADDR", KRB5_AUTH_CONTEXT_GENERATE_REMOTE_FULL_ADDR },
	{ NULL, 0 },
};

StringToNumTable TicketFlagTable[] = {
	{ "TKT_FLG_FORWARDABLE", TKT_FLG_FORWARDABLE },
	{ "TKT_FLG_FORWARDED", TKT_FLG_FORWARDED },
	{ "TKT_FLG_PROXIABLE", TKT_FLG_PROXIABLE },
	{ "TKT_FLG_PROXY", TKT_FLG_PROXY },
	{ "TKT_FLG_MAY_POSTDATE", TKT_FLG_MAY_POSTDATE },
	{ "TKT_FLG_POSTDATED", TKT_FLG_POSTDATED },
	{ "TKT_FLG_INVALID", TKT_FLG_INVALID },
	{ "TKT_FLG_RENEWABLE", TKT_FLG_RENEWABLE },
	{ "TKT_FLG_INITIAL", TKT_FLG_INITIAL },
	{ "TKT_FLG_PRE_AUTH", TKT_FLG_PRE_AUTH },
	{ "TKT_FLG_HW_AUTH", TKT_FLG_HW_AUTH },
	{ NULL, 0 },
};

StringToNumTable AddrtypeTable[] = {
	{ "ADDRTYPE_INET", ADDRTYPE_INET },
	{ "ADDRTYPE_CHAOS", ADDRTYPE_CHAOS },
	{ "ADDRTYPE_XNS", ADDRTYPE_XNS },
	{ "ADDRTYPE_ISO", ADDRTYPE_ISO },
	{ "ADDRTYPE_DDP", ADDRTYPE_DDP },
	{ "ADDRTYPE_ADDRPORT", ADDRTYPE_ADDRPORT },
	{ "ADDRTYPE_IPPORT", ADDRTYPE_IPPORT },
	{ NULL, 0 },
};

StringToNumTable EnctypeTable[] = {
	{ "ENCTYPE_NULL", ENCTYPE_NULL },
	{ "ENCTYPE_DES_CBC_CRC", ENCTYPE_DES_CBC_CRC },
	{ "ENCTYPE_DES_CBC_MD4", ENCTYPE_DES_CBC_MD4 },
	{ "ENCTYPE_DES_CBC_MD5", ENCTYPE_DES_CBC_MD5 },
	{ "ENCTYPE_DES_CBC_RAW", ENCTYPE_DES_CBC_RAW },
	{ "ENCTYPE_DES3_CBC_SHA", ENCTYPE_DES3_CBC_SHA },
	{ "ENCTYPE_DES3_CBC_RAW", ENCTYPE_DES3_CBC_RAW },
	{ "ENCTYPE_UNKNOWN", ENCTYPE_UNKNOWN },
	{ NULL, 0 },
};

StringToNumTable GetCredsTable[] = {
	{ "KRB5_GC_USER_USER", KRB5_GC_USER_USER },
	{ "KRB5_GC_CACHED", KRB5_GC_CACHED },
	{ NULL, 0 },
};

/*
 * Convert a string constant into a corresponding numeric value
 */

int
StringToNum(StringToNumTable *table, char *string, int *num)
{
	int i;

	for (i = 0; table[i].name != NULL; i++)
		if (strcmp(string, table[i].name) == 0) {
			*num = table[i].value;
			return TCL_OK;
		}
	
	return TCL_ERROR;
}

/*
 * Convert a numeric value into it's corresponding string constant
 */

int
NumToString(StringToNumTable *table, int num, char **string)
{
	int i;

	for (i = 0; table[i].name != NULL; i++)
		if (table[i].value == num) {
			*string = table[i].name;
			return TCL_OK;
		}
	
	return TCL_ERROR;
}

/*
 * Convert one or more set flags into a Tcl_DString list of flag names
 */

void
FlagToDString(StringToNumTable *table, int flags, Tcl_DString *dstr)
{
	int i;

	for (i = 0; table[i].name != NULL; i++) {
		if (flags & table[i].value) {
			Tcl_DStringAppendElement(dstr, table[i].name);
		}
	}
}

/*
 * Convert a krb5_keyblock to a Tcl_DString
 */

int
KeyblockToDString(Tcl_Interp *interp, krb5_keyblock *keyblock,
		  Tcl_DString *dstr)
{
	char temp[1024], *type;

	/*
	 * First, include the magic number
	 */

	sprintf(temp, "%d", keyblock->magic);

	Tcl_DStringAppendElement(dstr, temp);

	/*
	 * Next, include the encryption type.  Print the string name
	 * of the encryption type if possible
	 */

	if (NumToString(EnctypeTable, keyblock->enctype, &type) == TCL_OK) {
		Tcl_DStringAppendElement(dstr, type);
	} else {
		sprintf(temp, "%d", keyblock->enctype);
		Tcl_DStringAppendElement(dstr, temp);
	}

	/*
	 * Now append the actual key itself
	 */

	if (DataToDString(interp, keyblock->contents, keyblock->length,
			  dstr) != TCL_OK) {
		return TCL_ERROR;
	}

	return TCL_OK;
}

/*
 * Convert a krb5_enc_tkt_part to a Tcl_DString
 *
 * Sigh.  This is nasty.
 */

int
EncTktPartToDString(Tcl_Interp *interp, krb5_context context,
		    krb5_enc_tkt_part *enc_part, Tcl_DString *dstr)
{
	krb5_error_code code;
	char temp[1024], *name;
	int i;

	/*
	 * First, append the magic number
	 */

	sprintf(temp, "%d", enc_part->magic);

	Tcl_DStringAppendElement(dstr, temp);

	/*
	 * Next, a list of flags
	 */

	Tcl_DStringStartSublist(dstr);

	FlagToDString(TicketFlagTable, enc_part->flags, dstr);

	Tcl_DStringEndSublist(dstr);

	Tcl_DStringStartSublist(dstr);

	if (KeyblockToDString(interp, enc_part->session, dstr) != TCL_OK)
		return TCL_ERROR;
	
	Tcl_DStringEndSublist(dstr);

	if ((code = krb5_unparse_name(context, enc_part->client, &name))) {
		Tcl_AppendResult(interp, "krb5_unparse_name failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringAppendElement(dstr, name);

	krb5_xfree(name);

	Tcl_DStringStartSublist(dstr);

	sprintf(temp, "%d", enc_part->transited.magic);

	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", enc_part->transited.tr_type);

	Tcl_DStringAppendElement(dstr, temp);

	if (DataToDString(interp, enc_part->transited.tr_contents.data,
			  enc_part->transited.tr_contents.length, dstr)
	    != TCL_OK) {
		return TCL_ERROR;
	}

	Tcl_DStringEndSublist(dstr);

	Tcl_DStringStartSublist(dstr);

	sprintf(temp, "%d", enc_part->times.authtime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", enc_part->times.starttime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", enc_part->times.endtime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", enc_part->times.renew_till);
	Tcl_DStringAppendElement(dstr, temp);

	Tcl_DStringEndSublist(dstr);

	Tcl_DStringStartSublist(dstr);

	for (i = 0; enc_part->caddrs[i] != NULL; i++) {
		Tcl_DStringStartSublist(dstr);
		if (AddressToDString(interp, enc_part->caddrs[i], dstr) !=
		    TCL_OK)
			return TCL_ERROR;
		Tcl_DStringEndSublist(dstr);
	}

	Tcl_DStringEndSublist(dstr);

	return TCL_OK;
}

/*
 * Convert a krb5_address to a Tcl DString
 */

int
AddressToDString(Tcl_Interp *interp, krb5_address *address, Tcl_DString *dstr)
{
	char temp[1024], *type;

	sprintf(temp, "%d", address->magic);

	Tcl_DStringAppendElement(dstr, temp);

	if (NumToString(AddrtypeTable, address->addrtype, &type) == TCL_OK) {
		Tcl_DStringAppendElement(dstr, type);
	} else {
		sprintf(temp, "%d", address->addrtype);
		Tcl_DStringAppendElement(dstr, temp);
	}

	if (DataToDString(interp, address->contents, address->length,
			  dstr) != TCL_OK) {
		return TCL_ERROR;
	}
	
	return TCL_OK;
}

/*
 * Convert credentials to a DString
 */

int
CredsToDString(Tcl_Interp *interp, krb5_context context, krb5_creds *creds,
	       Tcl_DString *dstr)
{
	krb5_error_code code;
	char temp[1024], *name;
	int i;

	sprintf(temp, "%d", creds->magic);
	Tcl_DStringAppendElement(dstr, temp);

	if ((code = krb5_unparse_name(context, creds->client, &name))) {
		Tcl_AppendResult(interp, "krb5_unparse_name failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringAppendElement(dstr, name);

	krb5_xfree(name);

	if ((code = krb5_unparse_name(context, creds->server, &name))) {
		Tcl_AppendResult(interp, "krb5_unparse_name failed: ",
				 error_message(code), (char *) NULL);
		return TCL_ERROR;
	}

	Tcl_DStringAppendElement(dstr, name);

	krb5_xfree(name);

	Tcl_DStringStartSublist(dstr);

	if (KeyblockToDString(interp, &creds->keyblock, dstr) != TCL_OK)
		return TCL_ERROR;
	
	Tcl_DStringEndSublist(dstr);

	Tcl_DStringStartSublist(dstr);

	sprintf(temp, "%d", creds->times.authtime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", creds->times.starttime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", creds->times.endtime);
	Tcl_DStringAppendElement(dstr, temp);

	sprintf(temp, "%d", creds->times.renew_till);
	Tcl_DStringAppendElement(dstr, temp);

	Tcl_DStringEndSublist(dstr);

	sprintf(temp, "%d", creds->is_skey);
	Tcl_DStringAppendElement(dstr, temp);

	Tcl_DStringStartSublist(dstr);

	FlagToDString(TicketFlagTable, creds->ticket_flags, dstr);

	Tcl_DStringEndSublist(dstr);

	Tcl_DStringStartSublist(dstr);

	for (i = 0; creds->addresses[i] != NULL; i++) {
		Tcl_DStringStartSublist(dstr);
		if (AddressToDString(interp, creds->addresses[i], dstr) !=
		    TCL_OK)
			return TCL_ERROR;
		Tcl_DStringEndSublist(dstr);
	}

	Tcl_DStringEndSublist(dstr);

	if (DataToDString(interp, creds->ticket.data, creds->ticket.length,
			  dstr) != TCL_OK) {
		return TCL_ERROR;
	}

	if (DataToDString(interp, creds->second_ticket.data,
			  creds->second_ticket.length, dstr) != TCL_OK) {
		return TCL_ERROR;
	}

	return TCL_OK;
}

/*
 * Convert random hex data into a DString
 */

int
DataToDString(Tcl_Interp *interp, krb5_octet *data, int size, Tcl_DString *dstr)
{
	char temp[1024], *c;
	int i;

	if (size > 512) {
		Tcl_AppendResult(interp, "Data to be converted is too large",
				 (char *) NULL);
		return TCL_ERROR;
	}

	temp[0] = '\0';

	for (i = 0, c = temp; i < size; i++) {
		sprintf(c, "%02x", data[i]);
		c += 2;
	}

	Tcl_DStringAppendElement(dstr, temp);

	return TCL_OK;
}

/*
 * Convert a hex string back to binary data
 */

int
StringToData(Tcl_Interp *interp, char *string, krb5_octet **ret, int *size)
{
	krb5_octet *data;
	char *s = string;
	int strsize = strlen(string);
	int tmp, i;

	if (strsize == 0) {
		*ret = NULL;
		*size = 0;
		return TCL_OK;
	}

	strsize /= 2;

	if ((data = (krb5_octet *) malloc(strsize)) == NULL) {
		Tcl_SetErrno(ENOMEM);
		return TCL_ERROR;
	}

	for (i = 0; i < strsize; i++) {
		if (sscanf(s, "%2x", &tmp) != 1) {
			Tcl_AppendResult(interp, "Invalid data in hex string",
					 (char *) NULL);
			krb5_xfree(data);
			return TCL_ERROR;
		}
		data[i] = (krb5_octet) tmp;
		s += 2;
	}

	*ret = data;
	*size = strsize;

	return TCL_OK;
}

/*
 * Convert list-formatted keyblock data back into a krb5_keyblock structure
 */

int
StringToKeyblock(Tcl_Interp *interp, char *string, krb5_keyblock *keyblock)
{
	int argc;
	char **argv;

	if (Tcl_SplitList(interp, string, &argc, &argv) != TCL_OK) {
		return TCL_ERROR;
	}

	if (argc != 3) {
		Tcl_AppendResult(interp, "Wrong # of elements in keyblock "
				 "list, should have been 3", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (strlen(argv[0]) > 0) {
		if (Tcl_GetInt(interp, argv[0], &keyblock->magic) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			Tcl_Free((char *) argv);
			return TCL_ERROR;
		}
	}

	if (strlen(argv[1]) == 0) {
		Tcl_AppendResult(interp, " encryption type not specified",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (StringToNum(EnctypeTable, argv[1], (int *) &keyblock->enctype)
	    != TCL_OK) {
		Tcl_AppendResult(interp, "invalid encryption type \"",
				 argv[1], "\"", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (strlen(argv[2]) > 0) {
		if (StringToData(interp, argv[2], &keyblock->contents,
				 &keyblock->length) != TCL_OK) {
			Tcl_AppendResult(interp, " while converting keyblock "
					 "list", (char *) NULL);
			Tcl_Free((char *) argv);
			return TCL_ERROR;
		}
	} else {
		keyblock->length = 0;
		keyblock->contents = NULL;
	}

	return TCL_OK;
}

/*
 * Convert list-formatted credential data back into a krb5_creds structure
 */

int
StringToCreds(Tcl_Interp *interp, krb5_context context, char *string,
	      krb5_creds *creds)
{
	krb5_error_code code;
	int retval = TCL_ERROR;
	int argc;
	char **argv = NULL;

	memset((char *) creds, 0, sizeof(krb5_creds));

	if (Tcl_SplitList(interp, string, &argc, &argv) != TCL_OK) {
		goto exit;
	}

	if (argc != 10) {
		Tcl_AppendResult(interp, "Wrong # of elements in credential "
				 "list, should have been 10", (char *) NULL);
		goto exit;
	}

	if (strlen(argv[0]) > 0) {
		if (Tcl_GetInt(interp, argv[0], &creds->magic) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[1]) > 0) {
		if ((code = krb5_parse_name(context, argv[1],
					    &creds->client))) {
			Tcl_AppendResult(interp, "krb5_parse_name failed: ",
					 error_message(code), " during "
					 "credential list conversion",
					 (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[2]) > 0) {
		if ((code = krb5_parse_name(context, argv[2],
					    &creds->server))) {
			Tcl_AppendResult(interp, "krb5_parse_name failed: ",
					 error_message(code), " during "
					 "credential list conversion",
					 (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[3]) > 0) {
		if (StringToKeyblock(interp, argv[3], &creds->keyblock) !=
		    TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[4]) > 0) {
		if (StringToTimes(interp, argv[4], &creds->times) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[5]) > 0) {
		if (Tcl_GetBoolean(interp, argv[5], (int *) &creds->is_skey)
		    != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[6]) > 0) {
		if (StringToFlags(interp, argv[6], TicketFlagTable, 
				  &creds->ticket_flags) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[7]) > 0) {
		int largc, i;
		char **largv;

		if (Tcl_SplitList(interp, argv[7], &largc, &largv) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}

		creds->addresses = (krb5_address **)
				malloc((largc + 1) * sizeof(krb5_address *));
		
		if (creds->addresses == NULL) {
			Tcl_SetErrno(ENOMEM);
			Tcl_AppendResult(interp, "no memory during credential "
					"list conversion", (char *) NULL);
			goto exit;
		}

		for (i = 0; i <= largc; i++) {
			creds->addresses[i] = NULL;
		}

		for (i = 0; i < largc; i++) {
			creds->addresses[i] = (krb5_address *)
						malloc(sizeof(krb5_address));
			if (StringToAddress(interp, largv[i],
					    creds->addresses[i]) != TCL_OK) {
				Tcl_AppendResult(interp, " during credential "
						 "list conversion",
						 (char *) NULL);
				goto exit;
			}
		}

		Tcl_Free((char *) largv);
	}

	if (strlen(argv[8]) > 0) {
		if (StringToData(interp, argv[8],
				(krb5_octet **) &creds->ticket.data,
				 &creds->ticket.length) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	if (strlen(argv[9]) > 0) {
		if (StringToData(interp, argv[9],
				 (krb5_octet **) &creds->second_ticket.data,
				 &creds->second_ticket.length) != TCL_OK) {
			Tcl_AppendResult(interp, " during credential list "
					 "conversion", (char *) NULL);
			goto exit;
		}
	}

	retval = TCL_OK;

exit:
	if (argv)
		Tcl_Free((char *) argv);
	if (creds->client && retval != TCL_OK)
		krb5_free_principal(context, creds->client);
	if (creds->server && retval != TCL_OK)
		krb5_free_principal(context, creds->server);
	if (creds->keyblock.contents && retval != TCL_OK)
		krb5_xfree(creds->keyblock.contents);
	if (creds->addresses && retval != TCL_OK)
		krb5_free_addresses(context, creds->addresses);
	if (creds->ticket.data && retval != TCL_OK)
		krb5_xfree(creds->ticket.data);
	if (creds->second_ticket.data && retval != TCL_OK)
		krb5_xfree(creds->second_ticket.data);

	return retval;
}

/*
 * Convert a list of flags into a bitmapped krb5_flags value
 */
int
StringToFlags(Tcl_Interp *interp, char *string, StringToNumTable *table,
	      krb5_flags *flags)
{
	int argc, i;
	char **argv;
	int temp;

	if (Tcl_SplitList(interp, string, &argc, &argv) != TCL_OK) {
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	for (i = 0; i < argc; i++) {
		if (StringToNum(table, argv[i], &temp) != TCL_OK) {
			Tcl_AppendResult(interp, " invalid flag \"",
					 argv[i], "\"", (char *) NULL);
			Tcl_Free((char *) argv);
			return TCL_ERROR;
		}
		*flags |= temp;
	}

	return TCL_OK;
}

/*
 * Convert a string into a krb5_ticket_times structure
 */

int
StringToTimes(Tcl_Interp *interp, char *string, krb5_ticket_times *times)
{
	int argc;
	char **argv;

	if (Tcl_SplitList(interp, string, &argc, &argv) != TCL_OK) {
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (argc != 4) {
		Tcl_AppendResult(interp, "wrong # elements in times "
				 "list, should be 4", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, argv[0], &times->authtime) != TCL_OK) {
		Tcl_AppendResult(interp, " while converting authtime",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, argv[1], &times->starttime) != TCL_OK) {
		Tcl_AppendResult(interp, " while converting starttime",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, argv[2], &times->endtime) != TCL_OK) {
		Tcl_AppendResult(interp, " while converting endtime",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (Tcl_GetInt(interp, argv[3], &times->renew_till) != TCL_OK) {
		Tcl_AppendResult(interp, " while converting renew_till time",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	Tcl_Free((char *) argv);

	return TCL_OK;
}

/*
 * Convert a string into a krb5_address
 */

int
StringToAddress(Tcl_Interp *interp, char *string, krb5_address *address)
{
	int argc;
	char **argv;

	if (Tcl_SplitList(interp, string, &argc, &argv) != TCL_OK) {
		return TCL_ERROR;
	}

	if (argc != 3) {
		Tcl_AppendResult(interp, "wrong # elements in address list, "
				 "must be 3", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	memset((char *) address, 0, sizeof(krb5_address));

	if (strlen(argv[0]) > 0) {
		if (Tcl_GetInt(interp, argv[0], &address->magic) != TCL_OK) {
			Tcl_AppendResult(interp, "while converting address "
					 "list", (char *) NULL);
			Tcl_Free((char *) argv);
			return TCL_ERROR;
		}
	}

	if (strlen(argv[1]) == 0) {
		Tcl_AppendResult(interp, "address type required",
				 (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (StringToNum(AddrtypeTable, argv[1], (int *) &address->addrtype)
	    != TCL_OK) {
		Tcl_AppendResult(interp, " invalid address type \"",
				 argv[1], "\"", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	if (StringToData(interp, argv[2], &address->contents,
			 &address->length) != TCL_OK) {
		Tcl_AppendResult(interp, " in address list", (char *) NULL);
		Tcl_Free((char *) argv);
		return TCL_ERROR;
	}

	return TCL_OK;
}
