/*
 * $Id: init.c,v 1.9 1997/01/06 22:50:11 kenh Exp $
 *
 * kerberos5 - a Tcl extension for Kerberos 5
 *
 * This module provides a set of routines that lets a Tcl programmer access
 * the Kerberos 5 API.  Using these functions, you can authenticate with
 * remote principals (both as a client _and_ server) and pass integrity-proof
 * (KRB_SAFE) or encrypted (KRB_PRIV) messages between a client and server.
 *
 * Providing a mapping between Tcl and the structures used by Kerberos 5
 * was quite a challenge.  Three basic approaches were taken:
 *
 * 1) For things that had a canonical string representation (such as
 *    principals or flags), the string representation was always used
 *    in Tcl.  The glue routines convert to and from Kerberos internal forms
 *    as necessary.  For flags an internal table is kept to facilitate this
 *    mapping.  A list of OR-ed flags are represented as a Tcl list.
 * 
 * 2) Binary structures that contained user-digestable data (such as a
 *    krb5_keyblock).  All of the components of the structure are converted
 *    to a Tcl list, with each member of the structure being an element in
 *    the list.  Typically integers are treated as integers, and krb5_data
 *    types are arrays of hex numbers which correspond to the data stored
 *    in the krb5_data type.
 *
 * 3) Structures that are opaque or contain data that has no user
 *    representation (e.g. function pointers).  An example of a structure
 *    such as this would be an krb5_auth_context, or a krb5_ccache.  These
 *    structures are stored in a hash table using a unique string as the
 *    key into this table (e.g. - authcon5).  The string is used as the
 *    structure reference in Tcl, much in the same way as channel handles
 *    are used.  The glue routines look up the entries in a hash table
 *    and retrieve a pointer to the structure at the appropriate time.
 */

#ifndef LINT
static char rcsid[]=
	"$Id: init.c,v 1.9 1997/01/06 22:50:11 kenh Exp $";
#endif

#include <sys/types.h>
#include <unistd.h>
#include <tcl.h>
#include <krb5.h>

#include "tcl-krb5.h"

/*
 * This array holds all the commands we are going to create, and their
 * corresponding functions
 */

static struct _CmdArray {
	char *name;
	Tcl_CmdProc *func;
} CmdArray[] = {
	{ "krb5_get_default_realm", Krb5GetDefaultRealmCmd },
	{ "krb5_set_default_realm", Krb5SetDefaultRealmCmd },
	{ "krb5_sname_to_principal", Krb5SnameToPrincipalCmd },
	{ "krb5_cc_default", Krb5CcDefaultCmd },
	{ "krb5_cc_get_principal", Krb5CcGetPrincipalCmd },
	{ "krb5_cc_close", Krb5CcCloseCmd },
	{ "krb5_cc_destroy", Krb5CcDestroyCmd },
	{ "krb5_auth_con_init", Krb5AuthConInitCmd },
	{ "krb5_auth_con_free", Krb5AuthConFreeCmd },
	{ "krb5_auth_con_getkey", Krb5AuthConGetkeyCmd },
	{ "krb5_auth_con_getlocalsubkey", Krb5AuthConGetlocalsubkeyCmd },
	{ "krb5_auth_con_getremotesubkey", Krb5AuthConGetremotesubkeyCmd },
	{ "krb5_auth_con_getflags", Krb5AuthConGetflagsCmd },
	{ "krb5_auth_con_setflags", Krb5AuthConSetflagsCmd },
	{ "krb5_auth_con_genaddrs", Krb5AuthConGenaddrsCmd },
	{ "krb5_auth_con_setrcache", Krb5AuthConSetrcacheCmd },
	{ "krb5_auth_con_setuseruserkey", Krb5AuthConSetuseruserkeyCmd },
	{ "krb5_sendauth", Krb5SendauthCmd },
	{ "krb5_write_mk_safe", Krb5WriteMkSafeCmd },
	{ "krb5_read_rd_safe", Krb5ReadRdSafeCmd },
	{ "krb5_get_server_rcache", Krb5GetServerRcacheCmd },
	{ "krb5_rc_close", Krb5RcCloseCmd },
	{ "krb5_rc_destroy", Krb5RcDestroyCmd },
	{ "krb5_recvauth", Krb5RecvauthCmd },
	{ "krb5_write_mk_priv", Krb5WriteMkPrivCmd },
	{ "krb5_read_rd_priv", Krb5ReadRdPrivCmd },
	{ "krb5_get_credentials", Krb5GetCredentialsCmd },
	{ "krb5_read_message", Krb5ReadMessageCmd },
	{ "krb5_write_message", Krb5WriteMessageCmd },
	{ NULL, NULL },
};

/*
 * Our package initializer routine.  Create our new interpreter commands
 */

int
Kerberos_Init(Tcl_Interp *interp)
{
	krb5_context context;
	int i;

	/*
	 * First, create a krb5_context and initialize the error table
	 */
	
	krb5_init_context(&context);
	krb5_init_ets(context);

	/*
	 * Create our commands, with the ClientData being our krb5_context
	 */

	for (i = 0; CmdArray[i].name != NULL; i++)
		Tcl_CreateCommand(interp, CmdArray[i].name, CmdArray[i].func,
				  (ClientData) context, (void (*)()) NULL);

	/*
	 * Initialze our hash tables
	 */

	Tcl_InitHashTable(&Ccache_Table, TCL_STRING_KEYS);
	Tcl_InitHashTable(&AuthCon_Table, TCL_STRING_KEYS);
	Tcl_InitHashTable(&Rcache_Table, TCL_STRING_KEYS);

	/*
	 * Tell the Tcl package interface that we exist
	 */

	if (Tcl_PkgProvide(interp, "Kerberos", "1.0") != TCL_OK)
		return TCL_ERROR;
	
	return TCL_OK;
}
