/*
 * ratFolder.c --
 *
 *      This file contains basic support code for the folder commands. Each
 *      folder type is created using an unique command. This command returns
 *      a folder handler, which when invoked calls the RatFolderCmd()
 *      procedure with a pointer to a RatFolderInfo structure as clientData.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notices is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"

/*
 * The number of folders opened. This is used when making
 * the folder entities.
 */
static int numFolders = 0;

/*
 * Global variables used to controll the RatFolderSortCompare function
 */
static char **rat_SortList;
static int rat_SortOrder;

static int RatFolderOpen _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]));
static int RatFolderCmd _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]));
static void RatFolderSort _ANSI_ARGS_((Tcl_Interp *interp,
	RatFolderInfo *infoPtr));
static int RatFolderSortCompare _ANSI_ARGS_((const void *arg1,
	const void *arg2));
static int RatFolderSubjSame _ANSI_ARGS_((char *subj1, char *subj2));


/*
 *----------------------------------------------------------------------
 *
 * RatFolderInit --
 *
 *      Initializes the folder commands.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	The folder creation commands are created in interp. And the
 *	Std folder is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderInit(Tcl_Interp *interp)
{
    RatInitMessages();
    if (TCL_OK != RatStdFolderInit(interp)) {
	return TCL_ERROR;
    }
    if (TCL_OK != RatDbFolderInit(interp)) {
	return TCL_ERROR;
    }
    Tcl_CreateCommand(interp, "RatOpenFolder", RatFolderOpen,
	    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RatParseExp", RatParseExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatGetExp", RatGetExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatFreeExp", RatFreeExpCmd, NULL, NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderOpen --
 *
 *      See the INTERFACE specification
 *
 * Results:
 *      The return value is normally TCL_OK and a foilder handle is left
 *	in interp->result; if something goes wrong TCL_ERROR is returned
 *	and an error message will be left in interp->result.
 *
 * Side effects:
 *	The folder creation commands are created in interp.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderOpen(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr;
    char buf[256];
    int i;

    if (!currentHost) {
	RatInitCurrent(interp);
    }
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" prot [args]\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(argv[1], "std")) {
	infoPtr = RatStdFolderCreate(interp, argc, argv);
    } else if (!strcmp(argv[1], "dbase")) {
	infoPtr = RatDbFolderCreate(interp, argc, argv);
    } else {
	Tcl_SetResult(interp,
		"Wrong protocol. Only 'std' and 'dbase' are supported",
		TCL_STATIC);
	return TCL_ERROR;
    }
    if (NULL == infoPtr) {
	return TCL_ERROR;
    }
    infoPtr->allocated = infoPtr->number;
    infoPtr->msgCmdPtr = (char **) ckalloc(infoPtr->allocated*sizeof(char*));
    for (i=0; i<infoPtr->allocated; i++) {
	infoPtr->msgCmdPtr[i] = (char *) NULL;
    }
    infoPtr->presentationOrder = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    infoPtr->hidden = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    RatFolderSort(interp, infoPtr);
    sprintf(buf, "RatFolder%d", numFolders++);
    Tcl_CreateCommand(interp, buf, RatFolderCmd,
    	    (ClientData) infoPtr, (Tcl_CmdDeleteProc *) NULL);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderCmd --
 *
 *      Main folder entity procedure. This procedure implements the
 *	folder commands mentioned in ../doc/interface. In order to make
 *	this a tad easier it uses the procedures defined in the
 *	RatFolderInfo structure :-)
 *
 * Results:
 *      Depends on the input :-)
 *
 * Side effects:
 *	The specified folder may be modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderCmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr = (RatFolderInfo*) clientData;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'u') && (strncmp(argv[1], "update", length) == 0)
	    && (length > 1)) {
	int i, number, oldNumber;

	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " update\"", (char *) NULL);
	    return TCL_ERROR;
	}
	oldNumber = infoPtr->visible;
	number = (*infoPtr->updateProc)(infoPtr, interp);
	if (number < 0) {
	    RatLog(interp, RAT_WARN, interp->result, 0);
	    number = 0;
	} else {
	    infoPtr->number += number;
	    if (infoPtr->number > infoPtr->allocated) {
		infoPtr->allocated = infoPtr->number;
		infoPtr->msgCmdPtr = (char **) REALLOC(infoPtr->msgCmdPtr,
			infoPtr->allocated*sizeof(char*));
		infoPtr->presentationOrder = (int *) REALLOC(
			infoPtr->presentationOrder,
			infoPtr->allocated*sizeof(int));
		infoPtr->hidden = (int *) REALLOC(infoPtr->hidden,
			infoPtr->allocated*sizeof(int));
	    }
	    for (i=infoPtr->number-number; i<infoPtr->number; i++) {
		infoPtr->msgCmdPtr[i] = (char *) NULL;
	    }
	    RatFolderSort(interp, infoPtr);
	    number = infoPtr->visible - oldNumber;
	    if (number < 0) {
		number = 0;
	    }
	}
	Tcl_ResetResult(interp);
	sprintf(interp->result, "%d", number);
	return TCL_OK;

    } if ((c == 'c') && (strncmp(argv[1], "close", length) == 0)
	    && (length > 2)) {
	int i, ret;
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " close\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(infoPtr->name);
	for(i=0; i < infoPtr->number; i++) {
	    if (NULL != infoPtr->msgCmdPtr[i]) {
		(void)Tcl_DeleteCommand(interp, infoPtr->msgCmdPtr[i]);
	    }
	}
	ret = (*infoPtr->closeProc)(infoPtr, interp);
	ckfree(infoPtr->msgCmdPtr);
	ckfree(infoPtr->presentationOrder);
	ckfree(infoPtr->hidden);
	ckfree(infoPtr);
	(void)Tcl_DeleteCommand(interp, argv[0]);
	return ret;

    } if ((c == 's') && (strncmp(argv[1], "setName", length) == 0)
	    && (length > 3)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setName name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(infoPtr->name);
	infoPtr->name = (char *) ckalloc(strlen(argv[2])+1);
	strcpy(infoPtr->name, argv[2]);
	return TCL_OK;

    } if ((c == 'i') && (strncmp(argv[1], "info", length) == 0)
	    && (length > 2)) {
	int infoArgc;
	char *infoArgv[3];
	char numberBuf[16], sizeBuf[16];
	char *list;

	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " info\"", (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(numberBuf, "%d", infoPtr->visible);
	sprintf(sizeBuf, "%d", infoPtr->size);
	infoArgv[0] = infoPtr->name;
	infoArgv[1] = numberBuf;
	infoArgv[2] = sizeBuf;
	infoArgc = 3;
	list = Tcl_Merge(infoArgc, infoArgv);
	Tcl_SetResult(interp, list, TCL_DYNAMIC);
	return TCL_OK;

    } if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)
	    && (length > 1)) {
	ListExpression *exprPtr;
	Tcl_DString result;
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " list format\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (exprPtr = RatParseList(argv[2]))) {
	    Tcl_SetResult(interp, "Illegal list format", TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&result);
	for (i=0; i < infoPtr->visible; i++) {
	    Tcl_DStringAppendElement(&result, RatDoList(interp, exprPtr,
		    infoPtr->infoProc, (ClientData)infoPtr, i));
	}
	RatFreeListExpression(exprPtr);
	Tcl_DStringResult(interp, &result);
	return TCL_OK;

    } if ((c == 'g') && (strcmp(argv[1], "get") == 0)) {
	int index;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get index\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (NULL == infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]]) {
	    infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]] =
		    (*infoPtr->createProc)(infoPtr, interp,
		    infoPtr->presentationOrder[index]);
	}
	Tcl_SetResult(interp,
		infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]],
		TCL_VOLATILE);
	return TCL_OK;

    } if ((c == 's') && (strncmp(argv[1], "setFlag", length) == 0)
	    && (length > 3)) {
	int index, value;
	RatFlag flag;

	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setFlag index flag value\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[4], &value)) {
	    return TCL_ERROR;
	}

	(*infoPtr->setFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag, value);
	return TCL_OK;

    } if ((c == 'g') && (strncmp(argv[1], "getFlag", length) == 0)
	    && (length > 3)) {
	int index;
	RatFlag flag;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " getFlag index flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	sprintf(interp->result, "%d",
		(*infoPtr->getFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag));
	return TCL_OK;

    } if ((c == 'f') && (strncmp(argv[1], "flagged", length) == 0)
	    && (length > 1)) {
	RatFlag flag;
	char buf[64];
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " flagged flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[2], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[2], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[2], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[2], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	for (i=0; i<infoPtr->number; i++) {
	    if ((*infoPtr->getFlagProc)(infoPtr, interp,
		    infoPtr->presentationOrder[i], flag)) {
		sprintf(buf, "%d", i);
		Tcl_AppendElement(interp, buf);
	    }
	}
	return TCL_OK;

    } if ((c == 'e') && (strncmp(argv[1], "expunge", length) == 0)
	    && (length > 1)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " expunge\"", (char *) NULL);
	    return TCL_ERROR;
	}
	(*infoPtr->expungeProc)(infoPtr, interp);
	RatFolderSort(interp, infoPtr);
	return TCL_OK;

    } if ((c == 'i') && (strncmp(argv[1], "insert", length) == 0)
	    && (length > 2)) {
	Tcl_CmdInfo cmdInfo;
	int i, result;

	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " insert message ?message ...?\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (argc-2 > infoPtr->allocated-infoPtr->number) {
	    infoPtr->allocated += argc;
	    infoPtr->msgCmdPtr = (char**)REALLOC(infoPtr->msgCmdPtr,
	    	    infoPtr->allocated*sizeof(char*));
	    infoPtr->presentationOrder = (int*)REALLOC(
		    infoPtr->presentationOrder, infoPtr->allocated*sizeof(int));
	}
	for(i=2; i<argc; i++) {
	    if (0 == Tcl_GetCommandInfo(interp, argv[i], &cmdInfo)
		    || NULL == cmdInfo.clientData) {
		Tcl_AppendResult(interp, "error \"", argv[i],
			"\" is not a valid message command", (char *) NULL);
		return TCL_ERROR;
	    }
	}
	result = (*infoPtr->insertProc)(infoPtr, interp, argc-2, &argv[2]);
	RatFolderSort(interp, infoPtr);
	return result;

    } if ((c == 't') && (strncmp(argv[1], "type", length) == 0)
	    && (length > 2)) {
	Tcl_SetResult(interp, infoPtr->type, TCL_STATIC);
	return TCL_OK;

    } if ((c == 'f') && (strncmp(argv[1], "find", length) == 0)
	    && (length > 1)) {
	int msgNo, i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " find message\"", (char *) NULL);
	    return TCL_ERROR;
	}

	strcpy(interp->result, "-1");
	for (msgNo=0; msgNo < infoPtr->number; msgNo++) {
	    if (infoPtr->msgCmdPtr[msgNo]
	    	    && !strcmp(infoPtr->msgCmdPtr[msgNo], argv[2])) {
	        for (i=0; msgNo != infoPtr->presentationOrder[i]; i++);
	        sprintf(interp->result, "%d", i);
	    }
	}
	return TCL_OK;

    } if ((c == 'm') && (strncmp(argv[1], "match", length) == 0)
	    && (length > 1)) {
	Tcl_DString ds;
	int i, expId;
	char buf[8];

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " match expId\"", (char *) NULL);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&ds);
	expId = atoi(argv[2]);

	for (i=0; i<infoPtr->number; i++) {
	    if (RatExpMatch(interp, expId, infoPtr->infoProc,
		    (ClientData)infoPtr, i)) {
		sprintf(buf, "%d", i);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}
	Tcl_DStringResult(interp, &ds);
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be update, close, setName, info, list, ",
		"get, setFlag, getFlag, flagged, expunge or insert",
		(char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSort --
 *
 *      Sorts the folder according to the users wishes. The user may
 *	communicates their will via the folder_sort variable. Currently
 *	The following methods are implemented:
 *	  folder		- Sorts in native folder order
 *	  reverseFolder		- The reverse of the above
 *	  date			- By date sent
 *	  reverseDate		- By reverse date sent
 *	  subjectsByDate	- Group messages with the same subject
 *				  and sort the groups by the earliest date
 *				  in each group. NOT IMPLEMENTED YET
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The presentation order member of the RatFolderInfo structure
 *	is initialized. The size of the folder is updated.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatFolderSort(Tcl_Interp *interp, RatFolderInfo *infoPtr)
{
    char *sortMethod=Tcl_GetVar2(interp,"option","folder_sort",TCL_GLOBAL_ONLY);
    int i, j, snarf, numParm, notused, listArgc, *p=infoPtr->presentationOrder,
	seen;
    char *s, **pArgv, **listArgv, *argv[4];

    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "dsn_snarf_reports",
	    TCL_GLOBAL_ONLY), &snarf);

    infoPtr->size = 0;
    for (i=0; i<infoPtr->number; i++) {
	infoPtr->hidden[i] = 0;
	infoPtr->presentationOrder[i] = i;
	s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,RAT_FOLDER_SIZE,i);
	if (s) {
	    infoPtr->size += atoi(s);
	}
	s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,RAT_FOLDER_TYPE,i);
	if (s && !strcasecmp(s, "multipart/report")) {
	    s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		RAT_FOLDER_PARAMETERS, i);
	    if (!s) {
		continue;
	    }
	    Tcl_SplitList(interp, s, &numParm, &pArgv);
	    for (j=0; j<numParm; j++) {
		if (!strcasecmp(pArgv[j], "report-type delivery-status")) {
		    /*
		     * This is a DSN; call the proper authorities
		     */
		    seen=(*infoPtr->getFlagProc)(infoPtr, interp, i, RAT_SEEN);
		    if (!infoPtr->msgCmdPtr[i]) {
			infoPtr->msgCmdPtr[i] =
				(*infoPtr->createProc)(infoPtr, interp, i);
		    }
		    if (RatDSNHandle(interp, infoPtr->msgCmdPtr[i]) && snarf) {
			infoPtr->hidden[i] = 1;
			(*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_DELETED,1);
		    }
		    (*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_SEEN,seen);
		}
	    }
	    ckfree(pArgv);
	}
    }

    if (!strcmp(sortMethod, "subject")) {
	int j, start, end, burrow, *datePtr, *snugglePtr;
	char **subjArgv, *s;

	subjArgv = (char**)malloc(infoPtr->number*sizeof(char*));
	datePtr = (int*)malloc(infoPtr->number*sizeof(int));
	snugglePtr = (int*)malloc(infoPtr->number*sizeof(int));
	for (i=0; i<infoPtr->number; i++) {
	    s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		    RAT_FOLDER_SUBJECT, i);
	    if (s && *s) {
		subjArgv[i] =  strdup(s);
	    } else {
		subjArgv[i] =  "";
	    }
	    s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		    RAT_FOLDER_DATE_N, i);
	    if (s) {
		datePtr[i] = atoi(s);
	    } else {
		datePtr[i] = 0;
	    }
	}
	for (i=0; i<infoPtr->number; i++) {
	    snugglePtr[i] = 0;
	    burrow = 0;
	    start = 0;
	    end = i+1;
	    for (j=0; j<i; j++) {
		if (RatFolderSubjSame(subjArgv[i], subjArgv[p[j]])) {
		    snugglePtr[i] = 1;
		    snugglePtr[p[j]] = 1;
		    burrow = 1;
		    start = j;
		    for (j++; j < i &&
			 RatFolderSubjSame(subjArgv[i],subjArgv[p[j]]); j++);
		    end = j;
		    break;
		}
	    } 
	    for (j=start; j<end && j<i; j++) {
		if (datePtr[i]<datePtr[p[j]] && (burrow || !snugglePtr[p[j]])) {
		    break;
		}
	    }
	    if (j < i) {
		memmove(&p[j+1], &p[j], (i-j)*sizeof(int));
	    }
	    p[j] = i;
	}
	for (i=0; i<infoPtr->number; i++) {
	    if (*subjArgv[i]) {
		ckfree(subjArgv[i]);
	    }
	}
	ckfree(subjArgv);
	ckfree(datePtr);

    } else if (!strcmp(sortMethod, "reverseFolder")) {
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = infoPtr->number-i-1;
	}

    } else if (!strcmp(sortMethod, "date")) {
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	infoPtr->visible = infoPtr->number;
	argv[0] = "foo";
	argv[1] = "list";
	argv[2] = "%D";
	RatFolderCmd((ClientData)infoPtr, interp, 3, argv);
	Tcl_SplitList(interp, interp->result, &listArgc, &listArgv);
	rat_SortList = listArgv;
	rat_SortOrder = 0;
	qsort((void*)p, infoPtr->number,
		sizeof(int), RatFolderSortCompare);
	ckfree(listArgv);
	Tcl_ResetResult(interp);

    } else if (!strcmp(sortMethod, "reverseDate")) {
	for (i=0; i<infoPtr->number; i++) {
	    infoPtr->presentationOrder[i] = i;
	}
	infoPtr->visible = infoPtr->number;
	argv[0] = "foo";
	argv[1] = "list";
	argv[2] = "%D";
	RatFolderCmd((ClientData)infoPtr, interp, 3, argv);
	Tcl_SplitList(interp, interp->result, &listArgc, &listArgv);
	rat_SortList = listArgv;
	rat_SortOrder = 1;
	qsort((void*)p, infoPtr->number, sizeof(int), RatFolderSortCompare);
	ckfree(listArgv);
	Tcl_ResetResult(interp);

    } else {	/* default to folder native order */
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
    }
    for (i=j=0; i < infoPtr->number; i++) {
	if (!infoPtr->hidden[p[i]]) {
	    p[j++] = p[i];
	}
    }
    infoPtr->visible = j;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSortCompare --
 *
 * 	This is a helper function to RatFolderSort, it is used as
 *	comparision function for qsort. The behaviour is controlled
 *	via two global variables:
 *	rat_SortList -	Points to the list which contains the values to
 *			be sorted.
 *	rat_SortOrder -	Indicates the order: 0 = increasing, 1 = decreasing
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The presentation order member of the RatFolderInfo structure
 *	is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderSortCompare(const void *arg1, const void *arg2)
{
    long val1, val2;
    int result;

    val1 = atoi(rat_SortList[*((int*)arg1)]);
    val2 = atoi(rat_SortList[*((int*)arg2)]);
    if (val1 > val2) {
	result = 1;
    } else if (val2 > val1) {
	result = -1;
    } else {
	result = 0;
    }
    if (rat_SortOrder) {
	result *= -1;
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSubjSame --
 *
 * Checks if the two subjects belongs to the same thread.
 *
 * Results:
 *	True if they do.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderSubjSame(char *subj1, char *subj2)
{
    char *s1 = subj1, *s2 = subj2;
    int l1, l2;

    /*
     * For each subject we first try to find the start of the actual text
     * (i.e. without any leading Re:'s and whitespaces. Then we find how
     * long the text is (ignore trailing whitespaces).
     */
    while (*s1) {
	while (*s1 && isspace(*s1)) s1++;
	if (!strncasecmp(s1, "re", 2) && (':' == s1[2] || isspace(s1[2]))) {
	    s1 += 2;
	    if (*s1 == ':') {
		s1++;
	    }
	} else {
	    break;
	}
    }
    l1 = strlen(s1);
    while (l1 && isspace(s1[l1-1])) l1--;
    while (*s2) {
	while (*s2 && isspace(*s2)) s2++;
	if (!strncasecmp(s2, "re", 2) && (':' == s2[2] || isspace(s2[2]))) {
	    s2 += 2;
	    if (*s2 == ':') {
		s2++;
	    }
	} else {
	    break;
	}
    }
    l2 = strlen(s2);
    while (l2 && isspace(s2[l2-1])) l2--;

    /*
     * Now we are ready to compare them. If any of the subject are less
     * than 10 characters we say that they must match exactly otherwise it
     * is enough that they match as long as they are.
     */
    if (l1 < 10 || l2 < 10) {
	return (l1 == l2 && !strcmp(s1, s2));
    }
    return !strncmp(s1, s2, (l1>l2 ? l2 : l1));
}

/*
 *----------------------------------------------------------------------
 *
 * RatGetMsgInfo --
 *
 * Gets info from message structure and formats it somewhat. None of the
 * informations items needs both the messagePtr and the eltPtr. The following
 * table describes which needs which:
 *
 *  Info			envPtr	bodyPtr	eltPtr	size
 *  RAT_FOLDER_SUBJECT		needed	-	-	-
 *  RAT_FOLDER_NAME		needed	-	-	-
 *  RAT_FOLDER_MAIL		needed	-	-	-
 *  RAT_FOLDER_NAME_RECIPIENT	needed	-	-	-
 *  RAT_FOLDER_MAIL_RECIPIENT	needed	-	-	-
 *  RAT_FOLDER_SIZE		-	-	-	needed
 *  RAT_FOLDER_SIZE_F		-	-	-	needed
 *  RAT_FOLDER_DATE_F		needed	-	needed	-
 *  RAT_FOLDER_DATE_N		needed	-	needed	-
 *  RAT_FOLDER_STATUS		    [not supported ]
 *  RAT_FOLDER_TYPE		-	needed	-	-
 *  RAT_FOLDER_PARAMETERS	-	needed	-	-
 *  RAT_FOLDER_INDEX		    [not supported ]
 *
 * Results:
 *	A pointer to a string which is valid at least until the next call
 *	to this structure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char *
RatGetMsgInfo(Tcl_Interp *interp, RatFolderInfoType type, ENVELOPE *envPtr,
	BODY *bodyPtr, MESSAGECACHE *eltPtr, int size)
{
    static Tcl_DString ds;
    static int initialized = 0;
    time_t time, zonediff;
    MESSAGECACHE dateElt, *dateEltPtr;
    PARAMETER *parmPtr;
    ADDRESS *adrPtr;
    struct tm tm;

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    } else {
	Tcl_DStringSetLength(&ds, 0);
    }

    switch (type) {
	case RAT_FOLDER_SUBJECT:
	    return envPtr->subject;
	case RAT_FOLDER_NAME:
	    if (!envPtr->from) {
		return "";
	    }
	    if (RatAddressIsMe(interp, envPtr->from, 1)) {
		if (envPtr->to && envPtr->to->personal) {
		    Tcl_DStringSetLength(&ds, 0);
		    Tcl_DStringAppend(&ds,
			    Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY),-1);
		    Tcl_DStringAppend(&ds, ": ", 2);
		    Tcl_DStringAppend(&ds, envPtr->to->personal, -1);
		    return Tcl_DStringValue(&ds);
		}
	    } else if (envPtr->from->personal) {
		return envPtr->from->personal;
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL:
	    Tcl_DStringSetLength(&ds, 0);
	    if (RatAddressIsMe(interp, envPtr->from, 1)) {
		adrPtr = envPtr->to;
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY),-1);
		Tcl_DStringAppend(&ds, ": ", 2);
	    } else {
		adrPtr = envPtr->from;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		return "";
	    }
	    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_NAME_RECIPIENT:
	    if (!envPtr->to) {
		return "";
	    }
	    if (RatAddressIsMe(interp, envPtr->to, 1)) {
		if (envPtr->from->personal) {
		    Tcl_DStringSetLength(&ds, 0);
		    Tcl_DStringAppend(&ds,
			    Tcl_GetVar2(interp, "t","from",TCL_GLOBAL_ONLY),-1);
		    Tcl_DStringAppend(&ds, ": ", 2);
		    Tcl_DStringAppend(&ds, envPtr->from->personal, -1);
		    return Tcl_DStringValue(&ds);
		}
	    } else if (envPtr->to->personal) {
		return envPtr->to->personal;
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:
	    Tcl_DStringSetLength(&ds, 0);
	    if (RatAddressIsMe(interp, envPtr->to, 1)) {
		adrPtr = envPtr->from;
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "from", TCL_GLOBAL_ONLY),-1);
		Tcl_DStringAppend(&ds, ": ", 2);
	    } else {
		adrPtr = envPtr->to;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		return "";
	    }
	    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SIZE:
	    Tcl_DStringSetLength(&ds, 256);
	    sprintf(Tcl_DStringValue(&ds), "%d", size);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SIZE_F:
	    Tcl_DStringSetLength(&ds, 256);
	    sprintf(Tcl_DStringValue(&ds), "RatMangleNumber %d", size);
	    Tcl_GlobalEval(interp, Tcl_DStringValue(&ds));
	    Tcl_DStringGetResult(interp, &ds);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_DATE_F:
	    if (envPtr->date) {
		mail_parse_date(&dateElt, envPtr->date);
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    Tcl_DStringSetLength(&ds, 256);
	    sprintf(Tcl_DStringValue(&ds), "RatFormatDate %d %d %d %d %d %d",
		    dateEltPtr->year, dateEltPtr->month, dateEltPtr->day,
		    dateEltPtr->hours, dateEltPtr->minutes,dateEltPtr->seconds);
	    Tcl_GlobalEval(interp, Tcl_DStringValue(&ds));
	    Tcl_DStringGetResult(interp, &ds);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_DATE_N:
	    if (envPtr->date) {
		mail_parse_date(&dateElt, envPtr->date);
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    Tcl_DStringSetLength(&ds, 256);
	    tm.tm_sec = dateEltPtr->seconds;
	    tm.tm_min = dateEltPtr->minutes;
	    tm.tm_hour = dateEltPtr->hours;
	    tm.tm_mday = dateEltPtr->day;
	    tm.tm_mon = dateEltPtr->month - 1;
	    tm.tm_year = dateEltPtr->year+70;
	    tm.tm_wday = 0;
	    tm.tm_yday = 0;
	    tm.tm_isdst = -1;
	    time = mktime(&tm);
	    zonediff = (dateEltPtr->zhours*60+dateEltPtr->zminutes)*60;
	    if (!dateEltPtr->zoccident) {
		zonediff *= -1;
	    }
	    time += zonediff;
	    sprintf(Tcl_DStringValue(&ds), "%ld", time);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_TYPE:
	    Tcl_DStringSetLength(&ds, strlen(bodyPtr->subtype)+2+
		    strlen(body_types[bodyPtr->type]));
	    sprintf(Tcl_DStringValue(&ds), "%s/%s",
		    body_types[bodyPtr->type],
		    bodyPtr->subtype);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_PARAMETERS:
	    for (parmPtr = bodyPtr->parameter; parmPtr;
		    parmPtr = parmPtr->next) {
		Tcl_DStringStartSublist(&ds);
		Tcl_DStringAppendElement(&ds, parmPtr->attribute);
		Tcl_DStringAppendElement(&ds, parmPtr->value);
		Tcl_DStringEndSublist(&ds);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->to);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_FROM:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->from);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SENDER:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->sender);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_CC:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->cc);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_REPLY_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->reply_to);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_STATUS:	/*fallthrough */
	case RAT_FOLDER_INDEX:	/*fallthrough */
	case RAT_FOLDER_END:
	    return NULL;
    }
    return NULL;
}
