/*
 * ratFrMessage.c --
 *
 *	This file contains code which implements free messages.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"

/*
 * The ClientData for each message entity
 */
typedef struct FrMessageInfo {
    MESSAGE *messagePtr;
    char *headers;
} FrMessageInfo;

/*
 * The ClientData for each bodypart entity
 */
typedef struct FrBodyInfo {
    char *mark;
    char *text;
} FrBodyInfo;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numFrMessages = 0;

/*
 * The number of bodypart entities created. This is used to create new
 * unique command names.
 */
static int numFrBodies = 0;

static RatGetHeadersProc Fr_GetHeadersProc;
static RatGetEnvelopeProc Fr_GetEnvelopeProc;
static RatCreateBodyProc Fr_CreateBodyProc;
static RatFetchTextProc Fr_FetchTextProc;
static RatEnvelopeProc Fr_EnvelopeProc;
static RatFetchFirstTextProc Fr_FetchFirstTextProc;
static RatSetIndexProc Fr_SetIndexProc;
static RatMsgDeleteProc Fr_MsgDeleteProc;
static RatMakeChildrenProc Fr_MakeChildrenProc;
static RatFetchBodyProc Fr_FetchBodyProc;
static RatBodyDeleteProc Fr_BodyDeleteProc;
static RatInfoProc Fr_GetInfoProc;


/*
 *----------------------------------------------------------------------
 *
 * RatFrMessagesInit --
 *
 *      Initializes the given MessageProcInfo entry for a free message
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given MessageProcInfo is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatFrMessagesInit(MessageProcInfo *messageProcInfoPtr)
{
    messageProcInfoPtr->getHeadersProc = Fr_GetHeadersProc;
    messageProcInfoPtr->getEnvelopeProc = Fr_GetEnvelopeProc;
    messageProcInfoPtr->getInfoProc = Fr_GetInfoProc;
    messageProcInfoPtr->createBodyProc = Fr_CreateBodyProc;
    messageProcInfoPtr->fetchTextProc = Fr_FetchTextProc;
    messageProcInfoPtr->envelopeProc = Fr_EnvelopeProc;
    messageProcInfoPtr->fetchFirstTextProc = Fr_FetchFirstTextProc;
    messageProcInfoPtr->setIndexProc = Fr_SetIndexProc;
    messageProcInfoPtr->msgDeleteProc = Fr_MsgDeleteProc;
    messageProcInfoPtr->makeChildrenProc = Fr_MakeChildrenProc;
    messageProcInfoPtr->fetchBodyProc = Fr_FetchBodyProc;
    messageProcInfoPtr->bodyDeleteProc = Fr_BodyDeleteProc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFrMessageCreate --
 *
 *      Creates a free message entity
 *
 * Results:
 *	The name of the new message entity.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatFrMessageCreate(Tcl_Interp *interp, char *data, int length)
{
    FrMessageInfo *msgFrPtr=(FrMessageInfo*)ckalloc(sizeof(FrMessageInfo));
    MessageInfo *msgPtr=(MessageInfo*)ckalloc(sizeof(MessageInfo));
    char *name = (char*)ckalloc(16);
    char *bodyData;
    int headerLength;

    for (headerLength = 0; data[headerLength]; headerLength++) {
	if (data[headerLength] == '\n' && data[headerLength+1] == '\n') {
	    headerLength++;
	    break;
	}
	if (data[headerLength]=='\r' && data[headerLength+1]=='\n'
		&& data[headerLength+2]=='\r' && data[headerLength+3]=='\n') {
	    headerLength += 2;
	    break;
	}
    }

    bodyData = (char*)malloc(length+1);
    memcpy(bodyData, data, length);
    bodyData[length] = '\0';

    msgPtr->interp = interp;
    msgPtr->folderInfoPtr = NULL;
    msgPtr->name = name;
    msgPtr->type = RAT_FREE_MESSAGE;
    msgPtr->bodyInfoPtr = NULL;
    msgPtr->msgNo = 0;
    msgPtr->clientData = (ClientData)msgFrPtr;
    msgFrPtr->messagePtr = RatParseMsg(interp, bodyData);
    msgFrPtr->headers = (char*)ckalloc(headerLength+1);
    strncpy(msgFrPtr->headers, data, headerLength);
    msgFrPtr->headers[headerLength] = '\0';

    sprintf(name, "RatFrMsg%d", numFrMessages++);
    Tcl_CreateCommand(interp, name, RatMessageCmd, (ClientData) msgPtr,
	    (Tcl_CmdDeleteProc *)RatMessageDelete);
    return name;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetHeadersProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetHeadersProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    return frMsgPtr->headers;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetEnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetEnvelopeProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    static char buf[1024];
    MESSAGECACHE elt;
    ADDRESS *adrPtr;
    time_t date;
    struct tm tm, *tmPtr;

    if (frMsgPtr->messagePtr->env->return_path) {
	adrPtr = frMsgPtr->messagePtr->env->sender;
    } else if (frMsgPtr->messagePtr->env->sender) {
	adrPtr = frMsgPtr->messagePtr->env->sender;
    } else {
	adrPtr = frMsgPtr->messagePtr->env->from;
    }
    if (!strcmp(currentHost, adrPtr->host)) {
	sprintf(buf, "From %s", adrPtr->mailbox);
    } else {
	sprintf(buf, "From ");
	rfc822_address(buf+5, adrPtr);
    }
    mail_parse_date(&elt, frMsgPtr->messagePtr->env->date);
    tm.tm_sec = elt.seconds;
    tm.tm_min = elt.minutes;
    tm.tm_hour = elt.hours;
    tm.tm_mday = elt.day;
    tm.tm_mon = elt.month - 1;
    tm.tm_year = elt.year+69;
    tm.tm_wday = 0;
    tm.tm_yday = 0;
    tm.tm_isdst = -1;
    date = (int)mktime(&tm);
    tmPtr = gmtime(&date);
    sprintf(buf + strlen(buf), " %s %s %2d %02d:%02d GMT 19%02d\n",
	    dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
	    tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min, tmPtr->tm_year);
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_CreateBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static BodyInfo*
Fr_CreateBodyProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)ckalloc(sizeof(FrBodyInfo));
    msgPtr->bodyInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);

    msgPtr->bodyInfoPtr->cmdName = (char*)msgPtr->bodyInfoPtr
	    + sizeof(BodyInfo);
    sprintf(msgPtr->bodyInfoPtr->cmdName, "RatFrBody%d", numFrBodies++);
    msgPtr->bodyInfoPtr->msgPtr = msgPtr;
    msgPtr->bodyInfoPtr->bodyPtr = frMsgPtr->messagePtr->body;
    msgPtr->bodyInfoPtr->firstbornPtr = NULL;
    msgPtr->bodyInfoPtr->nextPtr = NULL;
    msgPtr->bodyInfoPtr->containedEntity = NULL;
    msgPtr->bodyInfoPtr->clientData = (ClientData)frBodyInfoPtr;
    frBodyInfoPtr->text = frMsgPtr->messagePtr->text.text.data +
			  frMsgPtr->messagePtr->text.offset;
    frBodyInfoPtr->mark = frBodyInfoPtr->text;
    Tcl_CreateCommand(interp, msgPtr->bodyInfoPtr->cmdName,
	    RatBodyCmd, (ClientData) msgPtr->bodyInfoPtr, NULL);
    return msgPtr->bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_FetchTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_FetchTextProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    return frMsgPtr->messagePtr->text.text.data +
	   frMsgPtr->messagePtr->text.offset;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_EnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static ENVELOPE*
Fr_EnvelopeProc(MessageInfo *msgPtr)
{
    return ((FrMessageInfo*)msgPtr->clientData)->messagePtr->env;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_FetchFirstTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static BODY*
Fr_FetchFirstTextProc(MessageInfo *msgPtr, char **dataPtrPtr, int *lengthPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)msgPtr->bodyInfoPtr->clientData;
    BODY *bodyPtr = frMsgPtr->messagePtr->body;
    int offset = 0;

    while (TYPETEXT != bodyPtr->type) {
	if (TYPEMULTIPART != bodyPtr->type) {
	    bodyPtr = NULL;
	} else {
	    bodyPtr = &bodyPtr->nested.part->body;
	    offset = bodyPtr->contents.offset;
	}
    }
    if (bodyPtr) {
	*dataPtrPtr = frBodyInfoPtr->text + offset;
	*lengthPtr = bodyPtr->size.bytes;
    }
    return bodyPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_SetIndexProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_SetIndexProc(MessageInfo *msgPtr, int index)
{
     return;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_MsgDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_MsgDeleteProc(MessageInfo *msgPtr)
{
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    ckfree(frMsgPtr->messagePtr->text.text.data);
    mail_free_envelope(&frMsgPtr->messagePtr->env);
    mail_free_body(&frMsgPtr->messagePtr->body);
    ckfree(frMsgPtr->messagePtr);
    ckfree(frMsgPtr->headers);
    ckfree(frMsgPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_MakeChildrenProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_MakeChildrenProc(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    BodyInfo *partInfoPtr, **partInfoPtrPtr;
    FrBodyInfo *frPartInfoPtr;
    PART *partPtr;

    if (!bodyInfoPtr->firstbornPtr) {
	partInfoPtrPtr = &bodyInfoPtr->firstbornPtr;
	for (partPtr = bodyPtr->nested.part; partPtr;
		partPtr = partPtr->next) {
	    frPartInfoPtr = (FrBodyInfo*)ckalloc(sizeof(FrBodyInfo));
	    partInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);
	    *partInfoPtrPtr = partInfoPtr;
	    partInfoPtr->cmdName = (char*)partInfoPtr+sizeof(BodyInfo);
	    sprintf(partInfoPtr->cmdName, "RatFrBody%d", numFrBodies++);
	    partInfoPtr->bodyPtr = &partPtr->body;
	    partInfoPtr->firstbornPtr = NULL;
	    partInfoPtr->nextPtr = NULL;
	    partInfoPtr->containedEntity = NULL;
	    partInfoPtrPtr = &partInfoPtr->nextPtr;
	    partInfoPtr->msgPtr = bodyInfoPtr->msgPtr;
	    partInfoPtr->clientData = (ClientData)frPartInfoPtr;
	    frPartInfoPtr->text = frBodyInfoPtr->mark +
				  partPtr->body.contents.offset;
	    if (TYPEMULTIPART == partPtr->body.type) {
		frPartInfoPtr->mark = frBodyInfoPtr->text;
	    } else {
		frPartInfoPtr->mark = frPartInfoPtr->text;
	    }
	    Tcl_CreateCommand(interp, partInfoPtr->cmdName, RatBodyCmd,
		    (ClientData) partInfoPtr, NULL);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_FetchBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_FetchBodyProc(BodyInfo *bodyInfoPtr, int *lengthPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;

    *lengthPtr = bodyInfoPtr->bodyPtr->contents.text.size;
    return frBodyInfoPtr->text;
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_BodyDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Fr_BodyDeleteProc(BodyInfo *bodyInfoPtr)
{
    FrBodyInfo *frBodyInfoPtr = (FrBodyInfo*)bodyInfoPtr->clientData;
    ckfree(frBodyInfoPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Fr_GetInfoProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Fr_GetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int index)
{
    static char buf[32];
    MessageInfo *msgPtr = (MessageInfo*)clientData;
    FrMessageInfo *frMsgPtr = (FrMessageInfo*)msgPtr->clientData;
    MESSAGECACHE elt;
    char *cPtr;
    int i;

    switch (type) {
	case RAT_FOLDER_SUBJECT:	/* fallthrough */
	case RAT_FOLDER_NAME:		/* fallthrough */
	case RAT_FOLDER_MAIL:		/* fallthrough */
	case RAT_FOLDER_NAME_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_TYPE:		/* fallthrough */
	case RAT_FOLDER_TO:		/* fallthrough */
	case RAT_FOLDER_FROM:		/* fallthrough */
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:	/* fallthrough */
	case RAT_FOLDER_PARAMETERS:
	    return RatGetMsgInfo(interp, type, frMsgPtr->messagePtr->env,
		    frMsgPtr->messagePtr->body, NULL, 0);
	case RAT_FOLDER_SIZE:		/* fallthrough */
	case RAT_FOLDER_SIZE_F:
	    return RatGetMsgInfo(interp, type, NULL, NULL, NULL,
		    frMsgPtr->messagePtr->header.text.size +
		    frMsgPtr->messagePtr->text.text.size);
	case RAT_FOLDER_DATE_F:	/* fallthrough */
	case RAT_FOLDER_DATE_N:
	    mail_parse_date(&elt, frMsgPtr->messagePtr->env->date);
	    return RatGetMsgInfo(interp, type, frMsgPtr->messagePtr->env,
		    NULL, &elt, 0);
	case RAT_FOLDER_STATUS:
	    cPtr = frMsgPtr->headers;
	    do {
		if (strncasecmp(cPtr, "status:", 7)) {
		    int seen, deleted, marked, answered, me;
		    ADDRESS *addressPtr;

		    seen = deleted = marked = answered = me = 0;
		    for (i=7; cPtr[i]; i++) {
			switch (cPtr[i]) {
			case 'R': seen = 1;	break;
			case 'D': deleted = 1;	break;
			case 'F': marked = 1;	break;
			case 'A': answered = 1;	break;
			}
		    }
		    for (addressPtr = frMsgPtr->messagePtr->env->to;
			    !me && addressPtr;
			    addressPtr = addressPtr->next) {
			if (RatAddressIsMe(interp, addressPtr, 1)) {
			    me = 1;
			}
		    }
		    i = 0;
		    if (!seen) {
			buf[i++] = 'N';
		    }
		    if (deleted) {
			buf[i++] = 'D';
		    }
		    if (marked) {
			buf[i++] = 'F';
		    }
		    if (answered) {
			buf[i++] = 'A';
		    }
		    if (me) {
			buf[i++] = '+';
		    } else {
			buf[i++] = ' ';
		    }
		    buf[i] = '\0';
		    return buf;
		}
	    } while ((cPtr = strchr(cPtr, '\n')) && cPtr[1]);
	    return "";
	case RAT_FOLDER_INDEX:
	    return "1";
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}
