/*
 * ratSMTP.c --
 *
 *	This file contains basic support for sending messages via SMTP.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "rat.h"

/*
 * Timeout values:
 *    INITIAL_TIMEOUT	How long to wait for the initial greeting.
 *    COMMAND_TIMEOUT	How long for wait for command completion
 */
#define INITIAL_TIMEOUT 60
#define COMMAND_TIMEOUT 120

/*
 * Each channel has one of these structures. The channel handler is actually
 * just a pointer to this structure.
 */
typedef struct SMTPChannelPriv {
    Tcl_Channel channel;
    unsigned int mime : 1;	/* True if the peer supports 8 bit mime */
    unsigned int dsn : 1;	/* True if the peer supports DSN */
} SMTPChannelPriv;

/*
 * Local functions
 */
static char *RatTimedGets(Tcl_Interp *interp, Tcl_Channel channel, int timeout);
static int RatSendCommand(Tcl_Interp *interp, Tcl_Channel channel, char *cmd);
static int RatSendRcpt(Tcl_Interp *interp, Tcl_Channel channel,
	ADDRESS *adrPtr, DSNhandle handle, int *size, int verbose);


/*
 *----------------------------------------------------------------------
 *
 * RatTimedGets --
 *
 *      A gets with timeout. the channel must be in nonblocking mode.
 *
 * Results:
 *	Returns a pointer to a static area containing the read string.
 *
 * Side effects:
 *	The previous result is overwritten
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatTimedGets(Tcl_Interp *interp, Tcl_Channel channel, int timeout)
{
    static Tcl_DString ds;
    static int dsInit = 0;

    if (!dsInit) {
	dsInit = 1;
	Tcl_DStringInit(&ds);
    } else {
	Tcl_DStringSetLength(&ds, 0);
    }

    Tcl_SetChannelOption(interp, channel, "-blocking", "0");
    while (-1 == Tcl_Gets(channel, &ds)) {
	if (Tcl_InputBlocked(channel) && timeout) {
	    sleep(1),
	    timeout--;
	} else {
	    Tcl_SetChannelOption(interp, channel, "-blocking", "1");
	    return NULL;
	}
    }
    Tcl_SetChannelOption(interp, channel, "-blocking", "1");
    return Tcl_DStringValue(&ds);
}

/*
 *----------------------------------------------------------------------
 *
 * RatSendCommand --
 *
 *      Send a command to the SMTP peer, wait for and parse the result.
 *
 * Results:
 *	A standard Tcl result and an eventual error messages in 
 *	interp->result.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatSendCommand(Tcl_Interp *interp, Tcl_Channel channel, char *cmd)
{
    char *reply;
    int result;

    Tcl_Write(channel, cmd, -1);
    if ('\n' != cmd[strlen(cmd)-1]) {
	Tcl_Write(channel, "\n", -1);
    }
    do {
	reply = RatTimedGets(interp, channel, COMMAND_TIMEOUT);
	if (reply) {
	    if ('2' == reply[0] || '3' == reply[0]) {
		result = TCL_OK;
	    } else {
		Tcl_SetResult(interp, reply, TCL_VOLATILE);
		result = TCL_ERROR;
	    }
	} else {
	    Tcl_SetResult(interp, "Timeout from SMTP server", TCL_STATIC);
	    result = TCL_ERROR;
	}
    } while (TCL_OK == result && '-' == reply[4]);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSendRcpt --
 *
 *      Send the RCPT TO statements to the SMTP peer.
 *
 * Results:
 *	Returns the number of failed addresses. The result string of interp
 *	will contain more information in case of failure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatSendRcpt(Tcl_Interp *interp, Tcl_Channel channel, ADDRESS *adrPtr,
	DSNhandle handle, int *size, int verbose)
{
    char buf[1024], adr[1024], msg[1024];
    unsigned char *cPtr;
    int failures = 0, i;

    for (; adrPtr; adrPtr = adrPtr->next) {
	adr[0] = '\0';
	rfc822_address(adr, adrPtr);
	*size += strlen(adr)+8;
	sprintf(buf, "RCPT TO:<%s>", adr);
	if (handle) {
	    RatDSNAddRecipient(interp, handle,  adr);
	    sprintf(buf+strlen(buf), " NOTIFY=SUCCESS,FAILURE,DELAY");
	    sprintf(buf+strlen(buf), " ORCPT=rfc822;");
	    for (i = strlen(buf),cPtr = adr; *cPtr; cPtr++) {
		if (*cPtr < 33 || *cPtr > 126 || *cPtr == '+' || *cPtr == '=') {
		    buf[i++] = '+';
		    buf[i++] = alphabetHEX[*cPtr>>4];
		    buf[i++] = alphabetHEX[*cPtr&0xf];
		} else {
		    buf[i++] = *cPtr;
		}
	    }
	    buf[i] = '\0';
	}
	if (3 == verbose) {
	    sprintf(msg, "%s %s...", Tcl_GetVar2(interp, "t", "send_rcpt",
		    TCL_GLOBAL_ONLY), adr);
	    RatLog(interp, RAT_PARSE, msg, 1);
	    
	}
	if (TCL_OK != RatSendCommand(interp, channel, buf)) {
	    failures++;
	}
    }
    return failures;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSMTPOpen --
 *
 *      Open an SMTP channel.
 *
 * Results:
 *	Returns a channel handler.
 *
 * Side effects:
 *	A new channel is created.
 *
 *
 *----------------------------------------------------------------------
 */

SMTPChannel
RatSMTPOpen (Tcl_Interp *interp, char *host, int verbose)
{
    SMTPChannelPriv *chPtr;
    char *reply, buf[1024], *msg;
    int port;

    if (2 != sscanf(host, "%s[^:]:%d", buf, &port)) {
	strcpy(buf, host);
	port = 25;	/* The default SMTP port */
    }

    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "opening_connection", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    chPtr = (SMTPChannelPriv*)ckalloc(sizeof(SMTPChannelPriv));
    chPtr->mime = chPtr->dsn = 0;
    /*
     * Your compiler may complain that there are too many arguments to
     * Tcl_OpenTcpClient() below. This is a symtom of you having a prerelease
     * of tcl7.5 installed. In this case you must upgrade to the real
     * releases of tcl7.5/tk4.1 and reconfigure tkrat (this MUST be done).
     * After rerunning configure you may build tkrat.
     */
    if (NULL == (chPtr->channel = Tcl_OpenTcpClient(interp,port,buf,NULL,0,0))){
	ckfree(chPtr);
	return NULL;
    }
    Tcl_SetChannelOption(interp, chPtr->channel, "-buffering", "line");
    Tcl_SetChannelOption(interp, chPtr->channel, "-translation", "crlf");

    /*
     * Get initial greeting
     */
    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "wait_greeting", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    do {
	reply = RatTimedGets(interp, chPtr->channel, INITIAL_TIMEOUT);
	if (!reply || '2' != reply[0]) {
	    Tcl_Close(interp, chPtr->channel);
	    ckfree(chPtr);
	    return NULL;
	}
    } while (strncmp("220 ", reply, 4));

    /*
     * Send EHLO (HELO) and get capabilities
     */
    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "get_capabilities", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    sprintf(buf, "EHLO %s\n", currentHost);
    Tcl_Write(chPtr->channel, buf, -1);
    reply = RatTimedGets(interp, chPtr->channel, COMMAND_TIMEOUT);
    if (!reply || '2' != reply[0]) {
	sprintf(buf, "HELO %s.%s\n", currentHost, currentHost);
	Tcl_Write(chPtr->channel, buf, -1);
	reply = RatTimedGets(interp, chPtr->channel, COMMAND_TIMEOUT);
    }
    while (reply && strncmp("250 ", reply, 4)) {
	if (!reply) {
	    Tcl_Close(interp, chPtr->channel);
	    ckfree(chPtr);
	    return NULL;
	}
	if (!strncmp("8BITMIME", &reply[4], 8)) {
	    chPtr->mime = 1;
	} else if (!strncmp("DSN", &reply[4], 3)) {
	    chPtr->dsn = 1;
	}
	reply = RatTimedGets(interp, chPtr->channel, COMMAND_TIMEOUT);
    }

    if (verbose > 1) {
	RatLog(interp, RAT_PARSE, "", 1);
    }
    return (SMTPChannel)chPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSMTPClose --
 *
 *      Close an SMTP channel.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	THe chanel is closed.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatSMTPClose (Tcl_Interp *interp, SMTPChannel channel, int verbose)
{
    SMTPChannelPriv *chPtr = (SMTPChannelPriv*)channel;
    char *msg;

    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "closing", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    Tcl_Write(chPtr->channel, "QUIT\n", -1);
    Tcl_Close(interp, chPtr->channel);
    ckfree(chPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatSMTPSend --
 *
 *      Send a message with SMTP over the specified channel.
 *
 * Results:
 *	A standard Tcl result and an eventual error messages in 
 *	interp->result.
 *
 * Side effects:
 *	The DSN structures may be updated.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatSMTPSend (Tcl_Interp *interp, SMTPChannel channel, ENVELOPE *envPtr,
	BODY *bodyPtr, int doDSN, int verbose)
{
    SMTPChannelPriv *chPtr = (SMTPChannelPriv*)channel;
    char buf[1024], *header, *msg;
    int failures = 0, size = 4096;
    DSNhandle handle = NULL;

    /*
     * Check input and reset stream to known status.
     */
    if (!(envPtr->to || envPtr->cc || envPtr->bcc)) {
	Tcl_SetResult(interp, "No recipients specified", TCL_STATIC);
	return TCL_ERROR;
    }
    if (TCL_OK != RatSendCommand(interp, chPtr->channel, "RSET")) {
	return TCL_ERROR;
    }

    /*
     * Check if we should request DSN
     */
    if (doDSN && !chPtr->dsn) {
	RatLog(interp, RAT_WARN, "DSN not avaliable", 0);
	doDSN = 0;
    }

    /*
     * Send envelope information.
     */
    if (verbose > 1) {
	if (verbose == 2) {
	    msg = Tcl_GetVar2(interp, "t", "send_envelope", TCL_GLOBAL_ONLY);
	} else {
	    msg = Tcl_GetVar2(interp, "t", "send_from", TCL_GLOBAL_ONLY);
	}
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    sprintf(buf, "MAIL FROM:<");
    rfc822_address(buf, envPtr->return_path);
    strcat(buf, ">");
    if (chPtr->mime) {
	strcat(buf, " BODY=8BITMIME");
    }
    size += strlen(buf);
    size += (envPtr->remail?strlen(envPtr->remail):0);
    size += (envPtr->date?strlen(envPtr->date):0);
    size += (envPtr->subject?strlen(envPtr->subject):0);
    size += (envPtr->in_reply_to?strlen(envPtr->in_reply_to):0);
    size += (envPtr->message_id?strlen(envPtr->message_id):0);
    size += (envPtr->newsgroups?strlen(envPtr->newsgroups):0);
    size += (envPtr->followup_to?strlen(envPtr->followup_to):0);
    size += (envPtr->references?strlen(envPtr->references):0);
    if (doDSN) {
	RatGenId(NULL, interp, 0, NULL);
	handle = RatDSNStartMessage(interp, interp->result, envPtr->subject);
	strcat(buf, " ENVID=");
	strcat(buf, interp->result);
    }
    if (TCL_OK != RatSendCommand(interp, chPtr->channel, buf)) {
	return TCL_ERROR;
    }
    failures += RatSendRcpt(interp, chPtr->channel, envPtr->to, handle, &size,
	    verbose);
    failures += RatSendRcpt(interp, chPtr->channel, envPtr->cc, handle, &size,
	    verbose);
    failures += RatSendRcpt(interp, chPtr->channel, envPtr->bcc, handle, &size,
	    verbose);
    if (failures) {
	goto error;
    }

    /*
     * Send message data
     */
    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "send_data", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    if (TCL_OK != RatSendCommand(interp, chPtr->channel, "DATA")) {
	goto error;
    }
    header = (char*)malloc(size);
    rfc822_output(header, envPtr, bodyPtr, RatTclPutsSMTP, chPtr->channel,
	    chPtr->mime);
    free(header);
    if (verbose > 1) {
	msg = Tcl_GetVar2(interp, "t", "wait_ack", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, msg, 1);
    }
    if (TCL_OK != RatSendCommand(interp, chPtr->channel, ".")) {
	goto error;
    }
    if (handle) {
	RatDSNFinish(interp, handle);
    }

    return TCL_OK;

error:
    RatDSNAbort(interp, handle);
    RatSendCommand(interp, chPtr->channel, "RSET");
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSMTPSupportDSN --
 *
 *      Check if a host supports DSN.
 *
 * Results:
 *	TCL_OK and interp->result will contain "1" if the host supports
 *	DSN and "0" otherwise.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatSMTPSupportDSN(ClientData dummy, Tcl_Interp *interp, int argc, char **argv)
{
    SMTPChannelPriv *chPrivPtr;
    SMTPChannel channel;
    int verbose, result;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" hostname\"", (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "smtp_verbose",
	    TCL_GLOBAL_ONLY), &verbose);
    channel = RatSMTPOpen(interp, argv[1], verbose);
    if (channel) {
	chPrivPtr = (SMTPChannelPriv*)channel;
	result = chPrivPtr->dsn;
	RatSMTPClose(interp, channel, verbose);
    } else {
	result = 0;
    }
    if (verbose) {
	RatLog(interp, RAT_PARSE, "", 1);
    }
    Tcl_ResetResult(interp);
    sprintf(interp->result, "%d", result);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatTclPutsSMTP --
 *
 *	Like RatTclPuts but also escapes sigle dots with double dots.
 *
 * Results:
 *      Always returns 1L.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

long
RatTclPutsSMTP(void *stream_x, char *string)
{
    Tcl_Channel channel = (Tcl_Channel)stream_x;
    char *cPtr;

    if ('.' == string[0]) {
	Tcl_Write(channel, ".", -1);
    }

    while ((cPtr = strstr(string, "\015\012."))) {
	if (-1 == Tcl_Write(channel, string, cPtr-string+3)
		|| -1 == Tcl_Write(channel, ".", -1)) {
	    return 0;
	}
	string = cPtr+3;
    }
    if (-1 == Tcl_Write(channel, string, -1)) {
	return 0;
    } else {
	return 1;
    }
}
