/* 
 * ratSender.c --
 *
 *	This is the subprocess which handles the actual sending of messages.
 *
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include <tk.h>
#include "rat.h"

/*
 * Create bodypart s procedure
 */
static BODY *RatCreateBody(Tcl_Interp *interp, char *handler, int *errorFlag,
	char *charset, Tcl_DString *files);
static int RatSenderSend(Tcl_Interp *interp, char *prefix,
	Tcl_DString *usedVariables, Tcl_DString *files);


/*
 *----------------------------------------------------------------------
 *
 * RatSender --
 *
 *	This routine runs the sending process.
 *
 * Results:
 *	None, this routine never returns
 *
 * Side effects:
 *      Messages may be sent
 *
 *
 *----------------------------------------------------------------------
 */

void
RatSender(Tcl_Interp *interp)
{
    Tcl_DString usedVariables, files, result;
    char buf[1024], **argv;
    int argc, i;

    Tcl_DStringInit(&usedVariables);
    Tcl_DStringInit(&files);
    Tcl_DStringInit(&result);

    while ((fgets(buf, sizeof(buf), stdin), !feof(stdin))
	    && !strncmp(buf, "SEND", 4)) {
	(void)Tcl_SplitList(interp, buf, &argc, &argv);
	if (TCL_OK == RatSenderSend(interp, argv[2], &usedVariables, &files)) {
	    Tcl_DStringAppendElement(&result, "SENT");
	    Tcl_DStringAppendElement(&result, argv[1]);
	    Tcl_DStringAppendElement(&result, argv[3]);
	    free(argv);
	    (void)Tcl_SplitList(interp, Tcl_DStringValue(&files), &argc, &argv);
	    for (i=0; i<argc; i++) {
		(void)unlink(argv[i]);
	    }
	} else {
	    Tcl_DStringAppendElement(&result, "FAILED");
	    Tcl_DStringAppendElement(&result, argv[1]);
	    Tcl_DStringAppendElement(&result, argv[2]);
	    Tcl_DStringAppendElement(&result, argv[3]);
	    Tcl_DStringAppendElement(&result, interp->result);
	}
	free(argv);
	(void)Tcl_SplitList(interp, Tcl_DStringValue(&usedVariables), &argc,
		&argv);
	for (i=0; i<argc; i++) {
	    (void)Tcl_UnsetVar(interp, argv[i], TCL_GLOBAL_ONLY);
	}
	free(argv);
	fprintf(stdout, "%s", Tcl_DStringValue(&result));
	fputc('\0', stdout);
	fflush(stdout);
	Tcl_DStringSetLength(&result, 0);
	Tcl_DStringSetLength(&usedVariables, 0);
	Tcl_DStringSetLength(&files, 0);
    }
    exit(0);
}


/*
 *----------------------------------------------------------------------
 *
 * RatSenderSend --
 *
 *	Send a specified message
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *      A message is sent.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatSenderSend(Tcl_Interp *interp, char *prefix, Tcl_DString *usedVariablesPtr,
	Tcl_DString *filesPtr)
{
    char *charset = Tcl_GetVar2(interp, "option", "charset", TCL_GLOBAL_ONLY);
    int listArgc, i, errorFlag = 0, requestDSN, useFrom, verbose;
    char *tmp, buf[10000], **listArgv, *handler, *saveTo;
    SMTPChannel smtpChannel;
    ENVELOPE *msg;
    BODY *body;

    /*
     * Extract the message
     */
    if (TCL_OK != RatHoldExtract(interp, prefix, usedVariablesPtr, filesPtr)) {
	return TCL_ERROR;
    }
    handler = strdup(interp->result);

    /*
     * Construct the headers
     */
    if (!(tmp = Tcl_GetVar2(interp, handler, "request_dsn", TCL_GLOBAL_ONLY))
	    || TCL_OK != Tcl_GetBoolean(interp, tmp, &requestDSN)) {
	requestDSN = 0;
    }
    tmp = Tcl_GetVar2(interp, handler, "to", TCL_GLOBAL_ONLY);
    msg = mail_newenvelope ();
    rfc822_parse_adrlist(&msg->to, tmp, currentHost);
    msg->remail = cpystr(Tcl_GetVar2(interp,handler,"remail",TCL_GLOBAL_ONLY));
    if ((tmp = Tcl_GetVar2(interp, handler, "return_path", TCL_GLOBAL_ONLY)) 
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&msg->return_path, tmp, currentHost);
    } else {
	msg->return_path = mail_newaddr();
	msg->return_path->personal =
		currentPersonalName ? cpystr(currentPersonalName) : NULL;
	msg->return_path->mailbox = cpystr(currentMailboxName);
	msg->return_path->host = cpystr(currentHost);
    }
    if (NULL==(msg->date=Tcl_GetVar2(interp,handler,"date",TCL_GLOBAL_ONLY))) {
	rfc822_date(buf);
	msg->date = buf;
    }
    msg->date = cpystr(msg->date);
    if (TCL_OK != Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option",
	    "use_from", TCL_GLOBAL_ONLY), &useFrom)) {
	useFrom = 0;
    }
    if (useFrom && (tmp = Tcl_GetVar2(interp, handler, "from", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&msg->from, tmp, currentHost);
    }
    if (msg->from) {
	ADDRESS *adrPtr;

	for (adrPtr = msg->from; adrPtr; adrPtr = adrPtr->next) {
	    if (RatAddressIsMe(interp, adrPtr, 0)) {
		break;
	    }
	}
	if (!adrPtr) {
	    msg->sender = mail_newaddr();
	    msg->sender->personal =
		    currentPersonalName ? cpystr(currentPersonalName) : NULL;
	    msg->sender->mailbox = cpystr(currentMailboxName);
	    msg->sender->host = cpystr(currentHost);
	} else {
	    msg->sender = NULL;
	}
    } else {
	msg->from = mail_newaddr();
	msg->from->personal =
		currentPersonalName ? cpystr(currentPersonalName) : NULL;
	msg->from->mailbox = cpystr(currentMailboxName);
	msg->from->host = cpystr(currentHost);
    }
    if ((tmp = Tcl_GetVar2(interp, handler, "reply_to", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&msg->reply_to, tmp, currentHost);
    }
    msg->subject = (char*)RatEncodeHeaderLine((unsigned char*)
	    Tcl_GetVar2(interp, handler, "subject", TCL_GLOBAL_ONLY),
	    charset, 9, 1);
    if ((tmp = Tcl_GetVar2(interp, handler, "cc", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&msg->cc, tmp, currentHost);
    }
    if ((tmp = Tcl_GetVar2(interp, handler, "bcc", TCL_GLOBAL_ONLY))
	    && !RatIsEmpty(tmp)) {
	rfc822_parse_adrlist(&msg->bcc, tmp, currentHost);
    }
    msg->in_reply_to = cpystr(Tcl_GetVar2(interp, handler, "in_reply_to",
	    TCL_GLOBAL_ONLY));
    msg->message_id = cpystr(Tcl_GetVar2(interp, handler, "message_id",
	    TCL_GLOBAL_ONLY));
    msg->newsgroups = NULL;

    /*
     * Construct the body
     */
    if ((tmp = Tcl_GetVar2(interp, handler, "body", TCL_GLOBAL_ONLY))) {
	body = RatCreateBody(interp, tmp, &errorFlag, charset, filesPtr);
    } else {
	body = mail_newbody ();
    }
    if (errorFlag) {
	goto error;
    }

    /*
     * Send the message
     */
    Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "smtp_verbose",
	    TCL_GLOBAL_ONLY), &verbose);
    if (1 == verbose) {
	tmp = Tcl_GetVar2(interp, "t", "sending_message", TCL_GLOBAL_ONLY);
	RatLog(interp, RAT_PARSE, tmp, 1);
    }
    tmp = Tcl_GetVar2(interp, "option", "sendprot", TCL_GLOBAL_ONLY);
    if (tmp && !strcmp(tmp, "smtp")) {
	listArgc = 0;
	if ((tmp = Tcl_GetVar2(interp,"option","smtp_hosts",TCL_GLOBAL_ONLY))) {
	    Tcl_SplitList(interp, tmp, &listArgc, &listArgv);
	}
	if (0 == listArgc) {
	    Tcl_SetResult(interp, 
			  "Configuration error; no valid SMTP hosts given",
		          TCL_STATIC);
	    goto error;
	}
	for (i=0, smtpChannel=NULL; !smtpChannel && i<listArgc; i++) {
	    smtpChannel = RatSMTPOpen(interp, listArgv[i], verbose);
	}
	ckfree(listArgv);
	if (!smtpChannel) {
	    Tcl_SetResult(interp, "No valid SMTP hosts", TCL_STATIC);
	    goto error;
	}
	if (TCL_OK != RatSMTPSend(interp, smtpChannel, msg, body, requestDSN,
		verbose)) {
	    Tcl_DString ds;
	    Tcl_DStringInit(&ds);
	    Tcl_DStringGetResult(interp, &ds);
	    RatSMTPClose(interp, smtpChannel, verbose);
	    Tcl_DStringResult(interp, &ds);
	    goto error;
	}
	RatSMTPClose(interp, smtpChannel, verbose);

    } else if (tmp && !strcmp(tmp, "prog")) {
	Tcl_Channel channel;
	Tcl_DString ds;
	ADDRESS *adrPtr;

	Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "sendprog_8bit",
		TCL_GLOBAL_ONLY), &i);
	if (NULL == (tmp = Tcl_GetVar2(interp,"option", "sendprog",
		     TCL_GLOBAL_ONLY))) {
	    Tcl_SetResult(interp, "Invalid send program", TCL_STATIC);
	    goto error;
	}
	Tcl_DStringInit(&ds);
	Tcl_DStringAppend(&ds, tmp, -1);
	for (adrPtr = msg->to; adrPtr; adrPtr = adrPtr->next) {
	    buf[0] = '\0';
	    rfc822_address(buf, adrPtr);
	    Tcl_DStringAppendElement(&ds, buf);
	}
	for (adrPtr = msg->cc; adrPtr; adrPtr = adrPtr->next) {
	    buf[0] = '\0';
	    rfc822_address(buf, adrPtr);
	    Tcl_DStringAppendElement(&ds, buf);
	}
	for (adrPtr = msg->bcc; adrPtr; adrPtr = adrPtr->next) {
	    buf[0] = '\0';
	    rfc822_address(buf, adrPtr);
	    Tcl_DStringAppendElement(&ds, buf);
	}

	if (TCL_OK != Tcl_SplitList(interp, Tcl_DStringValue(&ds),
		    &listArgc, &listArgv)
		|| (NULL == (channel = Tcl_OpenCommandChannel(interp, listArgc,
		    listArgv, TCL_STDIN|TCL_STDOUT|TCL_STDERR)))) {
	    Tcl_DStringFree(&ds);
	    Tcl_SetResult(interp, "Invalid send program", TCL_STATIC);
	    goto error;
	} else {
	    Tcl_DStringFree(&ds);
	    ckfree(listArgv);
	    rfc822_output(buf, msg, body, RatTclPuts, channel, i);
	    Tcl_Close(interp, channel);
	}
    } else {
	sprintf(buf, "Invalid send protocol '%s'", tmp ? tmp : "<NULL>");
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	goto error;
    }
    if (verbose) {
	RatLog(interp, RAT_PARSE, "", 1);
    }

    saveTo = Tcl_GetVar2(interp, handler, "save_to", TCL_GLOBAL_ONLY);
    if (saveTo && *saveTo) {
	Tcl_DString saveArg;
	Tcl_Channel channel;
	char saveFile[1024];
	struct tm *tmPtr;
	int perm;
	time_t now;

	Tcl_DStringInit(&saveArg);
	tmp = Tcl_GetVar2(interp, "option", "tmp", TCL_GLOBAL_ONLY);
	RatGenId(NULL, interp, 0, NULL);
	sprintf(saveFile, "%s/rat.%s", tmp, interp->result);

	now = time(NULL);
	tmPtr = gmtime(&now);
	sprintf(buf, "From %s@%s %s %s %2d %02d:%02d GMT 19%02d\n",
		currentMailboxName, currentHost,
		dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
		tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min,
		tmPtr->tm_year);
	strcat(buf, "Status: RO\n");

	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
		TCL_GLOBAL_ONLY), &perm);
	if (NULL == (channel=Tcl_OpenFileChannel(interp,saveFile,"a",perm))){
	    Tcl_AppendResult(interp, "Failed to save copy of message: ",
		    Tcl_PosixError(interp), (char*) NULL);
	    goto error;
	}
	Tcl_Write(channel, buf, strlen(buf));
	rfc822_output(buf, msg, body, RatTclPuts, channel, 1);
	Tcl_Close(interp, channel);

	Tcl_DStringAppendElement(&saveArg, saveFile);
	Tcl_DStringAppendElement(&saveArg, saveTo);
	buf[0] = '\0';
	rfc822_write_address(buf, msg->to);
	Tcl_DStringAppendElement(&saveArg, buf);
	if (msg->from) {
	    buf[0] = '\0';
	    rfc822_write_address(buf, msg->from);
	} else {
	    sprintf(buf, "%s@%s", currentMailboxName, currentHost);
	}
	Tcl_DStringAppendElement(&saveArg, buf);
	buf[0] = '\0';
	rfc822_write_address(buf, msg->cc);
	Tcl_DStringAppendElement(&saveArg, buf);
	Tcl_DStringAppendElement(&saveArg, msg->subject);
	fprintf(stdout, "SAVE %s", Tcl_DStringValue(&saveArg));
	fputc('\0', stdout);
	fflush(stdout);
	Tcl_DStringFree(&saveArg);
    }

    mail_free_envelope(&msg);
    mail_free_body(&body);
    return TCL_OK;

error:
    if (verbose) {
	RatLog(interp, RAT_PARSE, "", 1);
    }
    mail_free_envelope(&msg);
    mail_free_body(&body);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RatCreateBody --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in interp->result. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */
static BODY*
RatCreateBody(Tcl_Interp *interp, char *handler, int *errorFlag, char *charset,
	Tcl_DString *filesPtr)
{
    BODY *body = mail_newbody();
    char *type;
    char *encoding;
    char *parameter;
    char buf[1024];
    char *filename = NULL;

    if (NULL == (type = Tcl_GetVar2(interp, handler, "type",
	    TCL_GLOBAL_ONLY))) {
	RatLog(interp, RAT_ERROR, "Internal error, no bhandler(type)", 0);
	(*errorFlag)++;
	return body;
    }
    if (!strcasecmp(type, "text"))		body->type = TYPETEXT;
    else if (!strcasecmp(type, "multipart"))	body->type = TYPEMULTIPART;
    else if (!strcasecmp(type, "message"))	body->type = TYPEMESSAGE;
    else if (!strcasecmp(type, "application"))	body->type = TYPEAPPLICATION;
    else if (!strcasecmp(type, "audio"))	body->type = TYPEAUDIO;
    else if (!strcasecmp(type, "image"))	body->type = TYPEIMAGE;
    else if (!strcasecmp(type, "video"))	body->type = TYPEVIDEO;
    else					body->type = TYPEOTHER;
    if (NULL == (encoding = Tcl_GetVar2(interp, handler, "encoding",
	    TCL_GLOBAL_ONLY))) {
	body->encoding = ENC7BIT;
    } else {
	if (!strcasecmp(encoding, "7bit"))	    body->encoding = ENC7BIT;
	else if (!strcasecmp(encoding, "8bit"))	    body->encoding = ENC8BIT;
	else if (!strcasecmp(encoding, "binary"))   body->encoding = ENCBINARY;
	else if (!strcasecmp(encoding, "base64"))   body->encoding = ENCBASE64;
	else if (!strcasecmp(encoding, "quoted-printable"))
					body->encoding = ENCQUOTEDPRINTABLE;
	else {
	    sprintf(buf, "Unkown encoding %s\n", encoding);
	    (*errorFlag)++;
	    RatLog(interp, RAT_ERROR, buf, 0);
	}
    }
    if (NULL == (body->subtype = Tcl_GetVar2(interp, handler, "subtype",
	    TCL_GLOBAL_ONLY))) {
	RatLog(interp, RAT_ERROR, "Internal error, no bhandler(subtype)", 0);
	(*errorFlag)++;
	return body;
    }
    body->subtype = cpystr(body->subtype);
    if ((parameter = Tcl_GetVar2(interp,handler,"parameter",TCL_GLOBAL_ONLY))) {
	PARAMETER **parmPtrPtr = &body->parameter;
	int parmListArgc, parmArgc, i;
	char **parmListArgv, **parmArgv;
	Tcl_SplitList(interp, parameter, &parmListArgc, &parmListArgv);
	for (i=0; i<parmListArgc; i++) {
	    if (TCL_ERROR == Tcl_SplitList(interp, parmListArgv[i], &parmArgc,
		    &parmArgv) || 2 != parmArgc) {
		sprintf(buf, "Illegal parameter: %s", parmListArgv[i]);
		RatLog(interp, RAT_ERROR, buf, 0);
		(*errorFlag)++;
	    } else {
		*parmPtrPtr = mail_newbody_parameter();
		(*parmPtrPtr)->attribute = cpystr(parmArgv[0]);
		(*parmPtrPtr)->value = cpystr(parmArgv[1]);
		parmPtrPtr = &(*parmPtrPtr)->next;
	    }
	    ckfree(parmArgv);
	}
	ckfree(parmListArgv);
    }
    body->id = cpystr(Tcl_GetVar2(interp, handler, "id", TCL_GLOBAL_ONLY));
    if (body->id && !strlen(body->id)) {
	body->id = NULL;
    }
    body->description = (char*)RatEncodeHeaderLine((unsigned char*)
	    Tcl_GetVar2(interp, handler, "description", TCL_GLOBAL_ONLY),
	    charset, 13, 1);
    if (body->description && !strlen(body->description)) {
	body->description = NULL;
    }

    if (TYPEMULTIPART == body->type) {
	PART **partPtrPtr = &body->nested.part;
	int childrenArgc, i;
	char **childrenArgv;
	char *children = Tcl_GetVar2(interp, handler, "children",
		TCL_GLOBAL_ONLY);
	Tcl_SplitList(interp, children, &childrenArgc, &childrenArgv);
	for (i=0; i<childrenArgc; i++) {
	    *partPtrPtr = mail_newbody_part();
	    (*partPtrPtr)->body = *RatCreateBody(interp, childrenArgv[i],
		    errorFlag, charset, filesPtr);
	    partPtrPtr = &(*partPtrPtr)->next;
	}
    } else if (TYPEMESSAGE == body->type) {
	struct stat sbuf;
	char *message;
	int fd;

	if (NULL == (filename = Tcl_GetVar2(interp, handler,"filename",
		TCL_GLOBAL_ONLY))) {
	    RatLog(interp, RAT_ERROR,"Internal error, no bhandler(filename)",0);
	    (*errorFlag)++;
	    return body;
	}
	if (-1 == (fd = open(filename, O_RDONLY))) {
	    sprintf(buf, "Failed to open file \"%s\": %s", filename,
		    Tcl_PosixError(interp));
	    RatLog(interp, RAT_ERROR, buf, 0);
	    (*errorFlag)++;
	    return body;
	}
	fstat(fd, &sbuf);
	message = (char*)ckalloc(sbuf.st_size+1);
	read(fd, message, sbuf.st_size);
	message[sbuf.st_size] = '\0';
	(void)close(fd);

	body->nested.msg = RatParseMsg(interp, message);
	body->contents.text.data = message;
	body->contents.text.size = sbuf.st_size;

    } else {
	FILE *fp;
	struct stat statbuf;
	int len = 0, allocated, ci;
	char *data;

	if (NULL == (filename = Tcl_GetVar2(interp, handler,"filename",
		TCL_GLOBAL_ONLY))) {
	    RatLog(interp, RAT_ERROR,"Internal error, no bhandler(filename)",0);
	    (*errorFlag)++;
	    return body;
	}
	if (NULL == (fp = fopen(filename, "r"))) {
	    sprintf(buf, "Failed to open file \"%s\": %s", filename,
		    Tcl_PosixError(interp));
	    RatLog(interp, RAT_ERROR, buf, 0);
	    (*errorFlag)++;
	    return body;
	}
	fstat(fileno(fp), &statbuf);
	allocated = (statbuf.st_size*105)/100;
	data = (char*)ckalloc(allocated);
	if (TYPETEXT == body->type) {
	    while (EOF != (ci = getc(fp))) {
		if (len >= allocated-1) {
		    allocated += 1024;
		    data = (char*)REALLOC(data, allocated);
		}
		if (ci == '\n') {
		    data[len++] = '\r';
		}
		data[len++] = ci;
	    }
	} else {
	    fread(data, statbuf.st_size, 1, fp);
	    len = statbuf.st_size;
	}
	data[len] = '\0';
	fclose(fp);
	body->contents.text.data = (unsigned char*)data;
	body->size.bytes = body->contents.text.size = len;
    }
    if (filename) {
	int removeFile = 0;
	char *tmp;

	if ((tmp = Tcl_GetVar2(interp, handler,"removeFile", TCL_GLOBAL_ONLY))){
	    Tcl_GetBoolean(interp, tmp, &removeFile);
	}
	if (removeFile && filesPtr) {
	    Tcl_DStringAppendElement(filesPtr, filename);
	}
    }

    return body;
}
