/*
 * ratStdMessage.c --
 *
 *	This file contains code which implements standard c-client messages.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratStdFolder.h"

/*
 * The ClientData for each message entity
 */
typedef struct StdMessageInfo {
    MAILSTREAM *stream;
    MESSAGECACHE *eltPtr;
    ENVELOPE *envPtr;
    BODY *bodyPtr;
    RatStdFolderType type;
    char *host;
    char *user;
} StdMessageInfo;

/*
 * The ClientData for each bodypart entity
 */
typedef struct StdBodyInfo {
    char *section;
} StdBodyInfo;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numStdMessages = 0;

/*
 * The number of bodypart entities created. This is used to create new
 * unique command names.
 */
static int numStdBodies = 0;


/*
 *----------------------------------------------------------------------
 *
 * RatStdMessagesInit --
 *
 *      Initializes the given MessageProcInfo entry for a c-client message
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given MessageProcInfo is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatStdMessagesInit(MessageProcInfo *messageProcInfoPtr)
{
    messageProcInfoPtr->getHeadersProc = Std_GetHeadersProc;
    messageProcInfoPtr->getEnvelopeProc = Std_GetEnvelopeProc;
    messageProcInfoPtr->getInfoProc = Std_GetInfoProc;
    messageProcInfoPtr->createBodyProc = Std_CreateBodyProc;
    messageProcInfoPtr->fetchTextProc = Std_FetchTextProc;
    messageProcInfoPtr->envelopeProc = Std_EnvelopeProc;
    messageProcInfoPtr->fetchFirstTextProc = Std_FetchFirstTextProc;
    messageProcInfoPtr->setIndexProc = Std_SetIndexProc;
    messageProcInfoPtr->msgDeleteProc = Std_MsgDeleteProc;
    messageProcInfoPtr->makeChildrenProc = Std_MakeChildrenProc;
    messageProcInfoPtr->fetchBodyProc = Std_FetchBodyProc;
    messageProcInfoPtr->bodyDeleteProc = Std_BodyDeleteProc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatStdMessageCreate --
 *
 *      Creates a std message entity
 *
 * Results:
 *	The name of the new message entity.
 *
 * Side effects:
 *	The message's long cache entry is locked until the message is
 *	deleted.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatStdMessageCreate(Tcl_Interp *interp, RatFolderInfoPtr folderInfoPtr,
	MAILSTREAM *stream, int msgNo, RatStdFolderType type,
	char *host, char *user)
{
    StdMessageInfo *msgStdPtr=(StdMessageInfo*)ckalloc(sizeof(StdMessageInfo));
    MessageInfo *msgPtr=(MessageInfo*)ckalloc(sizeof(MessageInfo));
    char *name = (char*)ckalloc(16);

    msgPtr->interp = interp;
    msgPtr->folderInfoPtr = folderInfoPtr;
    msgPtr->name = name;
    msgPtr->type = RAT_CCLIENT_MESSAGE;
    msgPtr->bodyInfoPtr = NULL;
    msgPtr->msgNo = msgNo;
    msgPtr->clientData = (ClientData)msgStdPtr;
    msgStdPtr->stream = stream;
    msgStdPtr->envPtr =
	    mail_fetch_structure(stream, msgNo, &msgStdPtr->bodyPtr, NIL);
    if (NULL == msgStdPtr->envPtr) {
	RatLog(interp, RAT_FATAL, "Failed to fetch message structure", 0);
	exit(1);
    }
    msgStdPtr->eltPtr = mail_elt(stream, msgNo);
    msgStdPtr->eltPtr->lockcount++;
    msgStdPtr->type = type;
    msgStdPtr->host = host;
    msgStdPtr->user = user;
    sprintf(name, "RatStdMsg%d", numStdMessages++);
    Tcl_CreateCommand(interp, name, RatMessageCmd, 
	    (ClientData) msgPtr, (Tcl_CmdDeleteProc *)RatMessageDelete);
    return name;
}


/*
 *----------------------------------------------------------------------
 *
 * RatStdEasyCopyingOK --
 *
 *      Check if we can lets c-client handle the copying of this message
 *
 * Results:
 *	A boolean which says if it is OK or not.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatStdEasyCopyingOK(MessageInfo *msgPtr, char *prot, char *destination,
	char *user)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    int nameLen;

    switch (stdMsgPtr->type) {
	case RAT_BERKELY:
	    return !strcasecmp(prot, "file");
	case RAT_MH:
	    return !strcasecmp(prot, "mh");
	case RAT_POP:
	    return 0;
	case RAT_IMAP:
	    if(strcasecmp(prot,"imap")) {
		return 0;
	    }
	    for (nameLen=0; '}' != destination[nameLen+1]
		    && '/' != destination[nameLen+1]; nameLen++);
	    return (!strncasecmp(stdMsgPtr->host, &destination[1], nameLen) &&
		    !strcasecmp(stdMsgPtr->user, user));
    }
    return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * RatStdMessageCopy --
 *
 *      Copy a message to aother c-client folder.
 *
 * Results:
 *	A boolean which says if it went OK or not.
 *
 * Side effects:
 *	The destination folder is modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatStdMessageCopy (Tcl_Interp *interp, MessageInfo *msgPtr, char *destination)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    char *cPtr, seq[16];

    sprintf(seq, "%d", msgPtr->msgNo);
    switch (stdMsgPtr->type) {
	case RAT_BERKELY:	/* fallthrough */
	case RAT_MH:	/* fallthrough */
	case RAT_POP:	/* fallthrough */
	    if (T == mail_copy_full(stdMsgPtr->stream, seq, destination, 0)) {
		return TCL_OK;
	    }
	    return TCL_ERROR;
	case RAT_IMAP:
	    cPtr = strchr(destination, '}');
	    if (cPtr && mail_copy_full(stdMsgPtr->stream, seq, &cPtr[1], 0)) {
		return TCL_OK;
	    }
	    return TCL_ERROR;
    }
    return TCL_ERROR;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_GetHeadersProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Std_GetHeadersProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    static char *header = NULL;
    static int headerSize = 0;
    char *fetchedHeader = mail_fetchheader(stdMsgPtr->stream, msgPtr->msgNo);
    int length = strlen(fetchedHeader);

    if (length > 2 && fetchedHeader[length-3] == '\n') {
	length -= 2;
    }

    if (length+32 > headerSize) {
	headerSize = length+32;
	header = (char*)REALLOC(header, headerSize);
    }
    memmove(header, fetchedHeader, length);
    if (stdMsgPtr->eltPtr->seen) {
	strcpy(&header[length], "Status: RO\r\n");
	length += strlen(&header[length]);
    }
    if (stdMsgPtr->eltPtr->answered) {
	strcpy(&header[length], "X-Status: A\r\n");
	length += strlen(&header[length]);
    }
    strcpy(&header[length], "\r\n");

    return header;
}

/*
 *----------------------------------------------------------------------
 *
 * Std_GetEnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Std_GetEnvelopeProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    static char buf[1024];
    ADDRESS *adrPtr;
    time_t date;
    struct tm tm, *tmPtr;

    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;

    if (stdMsgPtr->envPtr->return_path) {
	adrPtr = stdMsgPtr->envPtr->sender;
    } else if (stdMsgPtr->envPtr->sender) {
	adrPtr = stdMsgPtr->envPtr->sender;
    } else {
	adrPtr = stdMsgPtr->envPtr->from;
    }
    if (!adrPtr) {
	sprintf(buf, "From unkown");
    } else if (!strcmp(currentHost, adrPtr->host)) {
	sprintf(buf, "From %s", adrPtr->mailbox);
    } else {
	sprintf(buf, "From ");
	rfc822_address(buf+5, adrPtr);
    }
    tm.tm_sec = stdMsgPtr->eltPtr->seconds;
    tm.tm_min = stdMsgPtr->eltPtr->minutes;
    tm.tm_hour = stdMsgPtr->eltPtr->hours;
    tm.tm_mday = stdMsgPtr->eltPtr->day;
    tm.tm_mon = stdMsgPtr->eltPtr->month - 1;
    tm.tm_year = stdMsgPtr->eltPtr->year+69;
    tm.tm_wday = 0;
    tm.tm_yday = 0;
    tm.tm_isdst = -1;
    date = (int)mktime(&tm);
    tmPtr = gmtime(&date);
    sprintf(buf + strlen(buf), " %s %s %2d %02d:%02d GMT 19%02d\n",
	    dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
	    tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min, tmPtr->tm_year);
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_CreateBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

BodyInfo*
Std_CreateBodyProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    StdBodyInfo *stdBodyInfoPtr = (StdBodyInfo*)ckalloc(sizeof(StdBodyInfo));
    msgPtr->bodyInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);

    msgPtr->bodyInfoPtr->cmdName = (char*)msgPtr->bodyInfoPtr
	    + sizeof(BodyInfo);
    sprintf(msgPtr->bodyInfoPtr->cmdName, "RatStdBody%d", numStdBodies++);
    mail_fetchstructure(stdMsgPtr->stream, msgPtr->msgNo,
	    &msgPtr->bodyInfoPtr->bodyPtr);
    msgPtr->bodyInfoPtr->firstbornPtr = NULL;
    msgPtr->bodyInfoPtr->nextPtr = NULL;
    msgPtr->bodyInfoPtr->msgPtr = msgPtr;
    msgPtr->bodyInfoPtr->containedEntity = NULL;
    msgPtr->bodyInfoPtr->clientData = (ClientData)stdBodyInfoPtr;
    if (TYPEMULTIPART == msgPtr->bodyInfoPtr->bodyPtr->type) {
        stdBodyInfoPtr->section = NULL;
    } else {
        stdBodyInfoPtr->section = strdup("1");
    }
    Tcl_CreateCommand(interp, msgPtr->bodyInfoPtr->cmdName,
	    RatBodyCmd, (ClientData) msgPtr->bodyInfoPtr, NULL);
    return msgPtr->bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_FetchTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Std_FetchTextProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    return mail_fetchtext(stdMsgPtr->stream, msgPtr->msgNo);
}


/*
 *----------------------------------------------------------------------
 *
 * Std_EnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

ENVELOPE*
Std_EnvelopeProc(MessageInfo *msgPtr)
{
    return ((StdMessageInfo*)msgPtr->clientData)->envPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_FetchFirstTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

BODY*
Std_FetchFirstTextProc(MessageInfo *msgPtr, char **dataPtrPtr, int *lengthPtr)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    BODY *bodyPtr = stdMsgPtr->bodyPtr;

    while (TYPETEXT != bodyPtr->type) {
	if (TYPEMULTIPART != bodyPtr->type) {
	    bodyPtr = NULL;
	    break;
	}
	bodyPtr = &bodyPtr->nested.part->body;
    }
    if (bodyPtr) {
	*dataPtrPtr = mail_fetchbody(stdMsgPtr->stream, msgPtr->msgNo, "1",
		(unsigned long*)lengthPtr);
    }
    return bodyPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_SetIndexProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

void
Std_SetIndexProc(MessageInfo *msgPtr, int index)
{
    msgPtr->msgNo = index;
}


/*
 *----------------------------------------------------------------------
 *
 * Std_MsgDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

void
Std_MsgDeleteProc(MessageInfo *msgPtr)
{
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    stdMsgPtr->eltPtr->lockcount--;
    ckfree(stdMsgPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Std_MakeChildrenProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

void
Std_MakeChildrenProc(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    StdBodyInfo *stdBodyInfoPtr = (StdBodyInfo*)bodyInfoPtr->clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    BodyInfo *partInfoPtr, **partInfoPtrPtr;
    StdBodyInfo *partStdInfoPtr;
    int index = 1;
    PART *partPtr;

    if (!bodyInfoPtr->firstbornPtr) {
	partInfoPtrPtr = &bodyInfoPtr->firstbornPtr;
	for (partPtr = bodyPtr->nested.part; partPtr;
		partPtr = partPtr->next) {
	    partInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);
	    partStdInfoPtr = (StdBodyInfo*)ckalloc(sizeof(StdBodyInfo));
	    *partInfoPtrPtr = partInfoPtr;
	    partInfoPtr->cmdName = (char*)partInfoPtr+sizeof(BodyInfo);
	    sprintf(partInfoPtr->cmdName, "RatStdBody%d", numStdBodies++);
	    partInfoPtr->bodyPtr = &partPtr->body;
	    partInfoPtr->firstbornPtr = NULL;
	    partInfoPtr->nextPtr = NULL;
	    partInfoPtr->containedEntity = NULL;
	    partInfoPtrPtr = &partInfoPtr->nextPtr;
	    partInfoPtr->msgPtr = bodyInfoPtr->msgPtr;
	    partInfoPtr->clientData = (ClientData)partStdInfoPtr;
	    if (stdBodyInfoPtr->section) {
		partStdInfoPtr->section = (char*)ckalloc(
			strlen(stdBodyInfoPtr->section)+8);
		sprintf(partStdInfoPtr->section, "%s.%d",
			stdBodyInfoPtr->section, index++);
	    } else {
		partStdInfoPtr->section = (char*)ckalloc(8);
		sprintf(partStdInfoPtr->section, "%d", index++);
	    }
	    Tcl_CreateCommand(interp, partInfoPtr->cmdName, RatBodyCmd,
		    (ClientData) partInfoPtr, NULL);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Std_FetchBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Std_FetchBodyProc(BodyInfo *bodyInfoPtr, int *lengthPtr)
{
    StdMessageInfo *stdMsgPtr =(StdMessageInfo*)bodyInfoPtr->msgPtr->clientData;
    return mail_fetchbody(stdMsgPtr->stream, bodyInfoPtr->msgPtr->msgNo,
	    ((StdBodyInfo*)(bodyInfoPtr->clientData))->section,
	    (unsigned long*)lengthPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Std_BodyDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

void
Std_BodyDeleteProc(BodyInfo *bodyInfoPtr)
{
    StdBodyInfo *partStdInfoPtr = (StdBodyInfo*)bodyInfoPtr->clientData;
    if (partStdInfoPtr->section) {
	ckfree(partStdInfoPtr->section);
    }
    ckfree(bodyInfoPtr->clientData);
}

/*
 *----------------------------------------------------------------------
 *
 * Std_GetInfoProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Std_GetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int notused)
{
    static char buf[32];
    MessageInfo *msgPtr = (MessageInfo*)clientData;
    StdMessageInfo *stdMsgPtr = (StdMessageInfo*)msgPtr->clientData;
    ADDRESS *addressPtr;
    int i, me, presIndex;

    switch (type) {
	case RAT_FOLDER_SUBJECT:	/* fallthrough */
	case RAT_FOLDER_NAME:		/* fallthrough */
	case RAT_FOLDER_MAIL:		/* fallthrough */
	case RAT_FOLDER_NAME_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_SIZE:		/* fallthrough */
	case RAT_FOLDER_SIZE_F:		/* fallthrough */
	case RAT_FOLDER_TYPE:		/* fallthrough */
	case RAT_FOLDER_PARAMETERS:	/* fallthrough */
	case RAT_FOLDER_DATE_F:		/* fallthrough */
	case RAT_FOLDER_DATE_N:		/* fallthrough */
	case RAT_FOLDER_TO:		/* fallthrough */
	case RAT_FOLDER_FROM:		/* fallthrough */
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:
	    return RatGetMsgInfo(interp, type, stdMsgPtr->envPtr,
		    stdMsgPtr->bodyPtr, stdMsgPtr->eltPtr,
		    stdMsgPtr->eltPtr->rfc822_size);
	case RAT_FOLDER_STATUS:
	    me = 0;
	    for (addressPtr = stdMsgPtr->envPtr->to; !me && addressPtr;
		    addressPtr = addressPtr->next) {
		if (RatAddressIsMe(interp, addressPtr, 1)) {
		    me = 1;
		}
	    }
	    i = 0;
	    if (!stdMsgPtr->eltPtr->seen) {
		buf[i++] = 'N';
	    }
	    if (stdMsgPtr->eltPtr->deleted) {
		buf[i++] = 'D';
	    }
	    if (stdMsgPtr->eltPtr->flagged) {
		buf[i++] = 'F';
	    }
	    if (stdMsgPtr->eltPtr->answered) {
		buf[i++] = 'A';
	    }
	    if (me) {
		buf[i++] = '+';
	    } else {
		buf[i++] = ' ';
	    }
	    buf[i] = '\0';
	    return buf;
	case RAT_FOLDER_INDEX:
	    if (msgPtr->folderInfoPtr) {
		for (i=0; i< msgPtr->folderInfoPtr->number; i++) {
		    presIndex = msgPtr->folderInfoPtr->presentationOrder[i];
		    if (msgPtr->folderInfoPtr->msgCmdPtr[presIndex] ==
			    msgPtr->name) {
			sprintf(buf, "%d", i+1);
			return buf;
		    }
		}
	    }
	    return "1";
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}
