# alias.tcl --
#
# Code which handles aliases.
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.

# List of alias windows
set aliasWindows {}

# True if the aliases have been modified
set aliasMod 0

# Aliases --
#
# Display the aliases window
#
# Arguments:

proc Aliases {} {
    global idCnt t option defaultFont aliasWindows

    # Create identifier
    set id al[incr idCnt]
    set w .$id
    upvar #0 $id hd

    set hd(old,name) ""
    set hd(old,fname) ""
    set hd(old,content) ""

    # Create toplevel
    toplevel $w
    wm title $w $t(aliases)
    set hd(w) $w

    # Create list
    frame $w.l
    scrollbar $w.l.scroll \
	    -relief raised \
	    -bd 1 \
	    -highlightthickness 0 \
	    -command "$w.l.list yview"
    listbox $w.l.list \
	    -yscroll "$w.l.scroll set" \
	    -relief raised \
	    -bd 1 \
	    -exportselection false \
	    -highlightthickness 0 \
	    -font $defaultFont \
	    -selectmode single \
	    -width 30 \
	    -height 15
    set hd(listbox) $w.l.list
    pack $w.l.scroll -side right -fill y
    pack $w.l.list -expand 1 -fill both
    bind $hd(listbox) <ButtonRelease-1> "AliasSelect $id"

    # Create detail
    frame $w.d
    label $w.d.nlab -text $t(alias):
    entry $w.d.nent -textvariable ${id}(name)
    label $w.d.flab -text $t(fullname):
    entry $w.d.fent -textvariable ${id}(fname)
    label $w.d.clab -text $t(alias_content):
    text $w.d.ctext -relief sunken -wrap word -height 5 -setgrid true
    bindtags $w.d.ctext [list $w.d.ctext Text . all]
    bind $w.d.ctext <Return> {break}
    set hd(text) $w.d.ctext
    Size $w.d.ctext aliasCont
    grid $w.d.nlab -sticky e
    grid $w.d.nent -row 0 -column 1 -sticky ew
    grid $w.d.flab -sticky e
    grid $w.d.fent -row 1 -column 1 -sticky ew
    grid $w.d.clab 
    grid $w.d.ctext - -sticky nswe
    grid rowconfigure $w.d 3 -weight 1
    grid columnconfigure $w.d 1 -weight 1

    # Create buttons
    frame $w.b
    button $w.b.apply -text $t(apply) -width 11 -command "AliasApply $id" \
	    -state disabled
    button $w.b.clear -text $t(new) -width 11 -command "AliasClear $id"
    button $w.b.delete -text $t(delete) -width 11 -command "AliasDelete $id" \
	    -state disabled
    button $w.b.dismiss -text $t(dismiss) -width 11 -command "AliasDismiss $id"
    grid $w.b.apply $w.b.clear -padx 5 -pady 5
    grid $w.b.delete $w.b.dismiss  -padx 5 -pady 5
    set hd(apply) $w.b.apply
    set hd(delete) $w.b.delete

    # Pack the frames
    pack $w.l -side left -fill y
    pack $w.d -side top -fill both -padx 5 -pady 5 -expand 1
    pack $w.b

    # Populate list
    AliasesPopulate $id

    # Place window
    Place $w aliases
    lappend aliasWindows $id

    # Start traces
    trace variable hd(name) w "AliasTrace $id"
    trace variable hd(fname) w "AliasTrace $id"
    bind $hd(text) <KeyRelease> "AliasTrace $id hd content w"
    bind $w <Destroy> AliasSave
}

# AliasesPopulate --
#
# Populate the aliases window
#
# Arguments:
# handler - The handler of the alias window

proc AliasesPopulate {handler} {
    upvar #0 $handler hd

    set sel [$hd(listbox) curselection]
    if [string length $sel] {
	set old [$hd(listbox) get $sel]
    }
    RatAlias list alias
    set hd(aliasIds) [lsort [array names alias]]
    set top [lindex [$hd(listbox) yview] 0]
    $hd(listbox) delete 0 end
    foreach a $hd(aliasIds) {
	$hd(listbox) insert end [format "%-8s %-20s" $a [lindex $alias($a) 0]]
    }
    $hd(listbox) yview moveto $top
    if [string length $sel] {
	set i [lsearch -exact [$hd(listbox) get 0 end] $old]
	if { -1 != $i } {
	    $hd(listbox) selection set $i
	    $hd(listbox) see $i
	}
    }
    AliasSelect $handler
}

# AliasSelect --
#
# Select an alias and edit it
#
# Arguments:
# handler - The handler of the alias window

proc AliasSelect {handler} {
    upvar #0 $handler hd

    $hd(text) delete 1.0 end
    if ![llength [$hd(listbox) curselection]] {
	set hd(old,name) ""
	set hd(old,fname) ""
	set hd(old,content) ""
	set hd(name) ""
	set hd(fname) ""
	$hd(delete) configure -state disabled
	return
    }
    RatAlias list alias
    set a [lindex $hd(aliasIds) [$hd(listbox) curselection]]
    set hd(old,name) $a
    set hd(old,fname) [lindex $alias($a) 0]
    set hd(old,content) [lindex $alias($a) 1]
    $hd(text) insert end $hd(old,content)
    set hd(name) $hd(old,name)
    set hd(fname) $hd(old,fname)
    $hd(delete) configure -state normal
}

# AliasClear --
#
# Clear the alias entry field
#
# Arguments:
# handler - The handler of the alias window

proc AliasClear {handler} {
    upvar #0 $handler hd

    set hd(old,name) ""
    set hd(old,fname) ""
    set hd(name) ""
    set hd(fname) ""
    $hd(text) delete 1.0 end
    $hd(listbox) selection clear 0 end
    $hd(delete) configure -state disabled
}

# AliasTrace --
#
# Trace hanges in the input fields
#
# Arguments:
# handler - The handler of the alias window
# name1	  - The array the variable belongs to
# name2   - The element in that array
# op	  - The operation performed on the variable

proc AliasTrace {handler name1 name2 op} {
    upvar #0 $handler hd

    if { ([string compare $hd(old,name) $hd(name)]
	    || [string compare $hd(old,fname) $hd(fname)]
	    || [string compare [string trim $hd(old,content)] \
			       [string trim [$hd(text) get 1.0 end]]])
	    && [string length $hd(name)]} {
	$hd(apply) configure -state normal
    } else {
	$hd(apply) configure -state disabled
    }
}

# AliasApply --
#
# Apply the changes in the alias definition
#
# Arguments:
# handler - The handler of the alias window

proc AliasApply {handler} {
    global aliasMod
    upvar #0 $handler hd

    set i [$hd(listbox) curselection]
    if [llength $i] {
	RatAlias delete [lindex $hd(aliasIds) $i]
	set hd(aliasIds) [lreplace $hd(aliasIds) $i $i]
    }
    RatAlias add $hd(name) $hd(fname) [string trim [$hd(text) get 1.0 end]]
    set ni [lsearch -exact [lsort [concat $hd(aliasIds) $hd(name)]] \
	    $hd(name)]
    AliasUpdate
    $hd(listbox) selection clear 0 end
    $hd(listbox) selection set $ni
    AliasSelect $handler
    set aliasMod 1
}

# AliasDelete --
#
# Delete the current alias
#
# Arguments:
# handler - The handler of the alias window

proc AliasDelete {handler} {
    global aliasWindows aliasMod
    upvar #0 $handler hd

    RatAlias delete [lindex $hd(aliasIds) [$hd(listbox) curselection]]
    foreach aw $aliasWindows {
	AliasesPopulate $aw
    }
    set aliasMod 1
}

# AliasDismiss --
#
# Dismiss the aliases window.
#
# Arguments:
# handler - The handler of the alias window

proc AliasDismiss {handler} {
    global aliasWindows aliasMod
    upvar #0 $handler hd

    set i [lsearch -exact $aliasWindows $handler]
    set aliasWindows [lreplace $aliasWindows $i $i]
    RecordSize $hd(text) aliasCont
    RecordPos $hd(w) aliases
    destroy $hd(w)
}

# AliasUpdate --
#
# Updates all the alias windows
#
# Arguments:

proc AliasUpdate {} {
    global aliasWindows

    foreach aw $aliasWindows {
	AliasesPopulate $aw
    }
}

# AliasSave --
#
# Save the aliases (if needed)
#
# Arguments:

proc AliasSave {} {
    global option aliasMod

    if $aliasMod {
	RatAlias save $option(aliases_file)
	set aliasMod 0
    }
}

# AliasExtract --
#
# Extracts aliases from the current message
#
# Arguments:
# handler - The handler of the folder window

proc AliasExtract {handler} {
    global idCnt t
    upvar #0 $handler fh

    # Extract the addresses
    set adrlist {}
    foreach a [$fh(current) get from return_path reply_to sender cc bcc to] {
	if [$a isMe] {
	    continue
	}
	set good 1
	foreach a2 $adrlist {
	    if ![$a compare $a2] {
		set good 0
		break
	    }
	}
	if $good {
	    lappend adrlist $a
	}
    }

    # Check that we found something
    if ![llength $adrlist] {
	Popup $t(could_not_find_adr)
	return
    }

    RatAlias list alias

    # Create identifier
    set id al[incr idCnt]
    set w .$id
    upvar #0 $id hd

    # Create toplevel
    toplevel $w
    wm title $w $t(extract_adr)
    set hd(w) $w

    # Create frame with aliases to add
    frame $w.f
    label $w.f.use -text $t(use)
    label $w.f.name -text $t(alias)
    label $w.f.fname -text $t(fullname)
    label $w.f.content -text $t(alias_content)
    grid $w.f.use $w.f.name $w.f.fname $w.f.content -sticky w
    foreach a $adrlist {
	incr idCnt
	set hd($idCnt,use) 1
	set name [string tolower [lindex [$a get name] 0]]
	if {![string length $name] || [info exists alias($name)]} {
	    set name2 [string tolower [lindex [split [$a get mail] @.] 0]]
	    if ![string length $name] {
		set name $name2
	    }
	    if [info exist alias($name2)] {
		for {set i 2} {[info exists alias($name2)]} {incr i} {
		    set name2 $name$i
		}
	    }
	    set name $name2
	}
	set alias($name) ""
	set hd($idCnt,name) $name
	set hd($idCnt,fname) [$a get name]
	set hd($idCnt,content) [$a get mail]
	checkbutton $w.f.c$idCnt -variable ${id}($idCnt,use)
	entry $w.f.en$idCnt -textvariable ${id}($idCnt,name) -width 8
	entry $w.f.ef$idCnt -textvariable ${id}($idCnt,fname) -width 20
	entry $w.f.ec$idCnt -textvariable ${id}($idCnt,content) -width 30
	grid $w.f.c$idCnt $w.f.en$idCnt $w.f.ef$idCnt $w.f.ec$idCnt -sticky we
    }
    grid columnconfigure $w.f 1 -weight 1
    grid columnconfigure $w.f 2 -weight 1
    grid columnconfigure $w.f 3 -weight 1
    pack $w.f -side top -fill both

    # Create buttons
    OkButtons $w $t(add_aliases) $t(cancel) "AliasExtractDone $id"
    pack $w.buttons -side bottom -pady 5 -fill x

    Place $w extractAlias
}

# AliasExtractDone --
#
# The alias extract window is now done.
#
# Arguments:
# handler - The handler of the extract window
# action  - Which action we should take

proc AliasExtractDone {handler action} {
    upvar #0 $handler hd
    global aliasMod t

    # Find which entries we should use
    set ids {}
    foreach i [array names hd *,use] {
	if $hd($i) {
	    lappend ids [lindex [split $i ,] 0]
	}
    }

    if { 1 == $action} {
	# Add the aliases
	foreach id $ids {
	    if ![string length $hd($id,name)] {
		Popup $t(missing_alias_name)
		continue
	    }
	    RatAlias add $hd($id,name) $hd($id,fname) $hd($id,content)
	    set aliasMod 1
	}
	if { 1 == $aliasMod } {
	    AliasUpdate
	    AliasSave
	}
    }

    RecordPos $hd(w) extractAlias
    destroy $hd(w)
    unset hd
}

# AliasChooser --
#
# Pops up a window where the user may select an alias. This alias is
# then returned.
#
# Arguments:
# master    - The text widget that is to be master for this window

proc AliasChooser {master} {
    global idCnt t defaultFont

    # Create identifier
    set id al[incr idCnt]
    set w .$id
    upvar #0 $id hd

    # Create toplevel
    toplevel $w
    wm transient $w [winfo toplevel $master]
    wm title $w $t(alias_chooser)
    set hd(w) $w
    set hd(search) ""

    # Find coordinates for window
    set bbox [$master bbox insert]
    set x [expr [winfo rootx $master]+[lindex $bbox 0]+5]
    set y [expr [winfo rooty $master]+[lindex $bbox 1]-20]
    wm geom $w +$x+$y

    # Build the list
    scrollbar $w.scroll \
	    -relief raised \
	    -bd 1 \
	    -highlightthickness 0 \
	    -command "$w.list yview"
    listbox $w.list \
	    -yscroll "$w.scroll set" \
	    -relief raised \
	    -bd 1 \
	    -exportselection false \
	    -highlightthickness 0 \
	    -font $defaultFont \
	    -selectmode single
    Size $w.list aliasChooser
    pack $w.scroll -side right -fill y
    pack $w.list -expand 1 -fill both
    set hd(list) $w.list

    # Bind keys
    bind $w <Control-c> "set ${id}(done) 0"
    bind $w <Key-Escape> "set ${id}(done) 0"
    bind $w <Key-Return> "set ${id}(done) 1"
    bind $w <Key-Tab> "set ${id}(done) 1"
    bind $w.list <ButtonRelease-1> "set ${id}(done) 1"
    bind $w <Key-Up> "AliasChooserMoveSel $id up"
    bind $w <Key-Down> "AliasChooserMoveSel $id down"
    bind $w <Key> "AliasChooserSearch $id %A"

    # Populate list
    RatAlias list alias
    set hd(aliasIds) [lsort [array names alias]]
    foreach a $hd(aliasIds) {
	$hd(list) insert end [format "%-8s %-20s" $a [lindex $alias($a) 0]]
    }
    $hd(list) selection set 0

    # Set focus and grab
    set hd(oldFocus) [focus]
    grab $w
    focus $w

    # Wait for action
    tkwait variable ${id}(done)
    RecordSize $w.list aliasChooser
    if {1 == $hd(done)} {
	set ret [lindex $hd(aliasIds) [$hd(list) curselection]]
    } else {
	set ret ""
    }
    destroy $w
    focus $hd(oldFocus)
    unset hd
    return $ret
}

# AliasChooserMoveSel --
#
# Move the selection in the chooser.
#
# Arguments:
# handler   - The handler which defines this selection window
# direction - Which direction we should move the selection.

proc AliasChooserMoveSel {handler direction} {
    upvar #0 $handler hd

    set cur [$hd(list) curselection]
    $hd(list) selection clear $cur

    if [string compare up $direction] {
	if {[incr cur] > [$hd(list) size]} {
	    incr cur -1
	}
	if {[expr $cur/[$hd(list) size].0] >= [lindex [$hd(list) yview] 1]} {
	    $hd(list) yview $cur
	}
    } else {
	if {$cur > 0} {
	    incr cur -1
	    if {[expr $cur/[$hd(list) size].0] < [lindex [$hd(list) yview] 0]} {
		$hd(list) yview scroll -1 pages
	    }
	}
    }
    $hd(list) selection set $cur
    set hd(search) ""
}

# AliasChooserSearch --
#
# Searches the chooser list.
#
# Arguments:
# handler   - The handler which defines this selection window
# key	    - The pressed key

proc AliasChooserSearch {handler key} {
    upvar #0 $handler hd

    if {1 != [scan $key "%s" key2]} {
	return
    }
    set hd(search) "$hd(search)$key2"
    set i [lsearch -glob $hd(aliasIds) "$hd(search)*"]
    if {-1 == $i} {
	bell
	set hd(search) ""
    } else {
	$hd(list) selection clear [$hd(list) curselection]
	$hd(list) selection set $i
	$hd(list) see $i
    }
}
