# preferences.tcl --
#
# This file controls the preferences window. There is just one preferences
# window, if the user requests a new preferences window when the window
# already exists, it is just unmapped and raised (if possible).
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.

# The last wisited window
set lastPref PrefGeneral

# Preferences --
#
# Make sure the user sees an preferences window.
#
# Arguments:

proc Preferences {} {
    if ![winfo exists .pref] {
	BuildPreferences
    } else {
	wm deiconify .pref
    }
}


# BuildPreferences --
#
# Builds the preferences window.
#
# Arguments:

proc BuildPreferences {} {
    global t pref lastPref

    # Initialize defaults
    set pref(options) {}

    # Create toplevel
    set w .pref
    toplevel $w
    wm title $w $t(preferences)

    # The page selector
    menubutton $w.page -textvariable pref(active) -indicatoron 1 -width 20 \
		  -relief raised -menu $w.page.m
    menu $w.page.m
    $w.page.m add command -label $t(general) -command {PrefCheck; PrefGeneral}
    $w.page.m add command -label $t(folderwindow) \
	    -command {PrefCheck; PrefFolder}
    $w.page.m add command -label $t(dbase) -command {PrefCheck; PrefDbase}
    $w.page.m add command -label $t(watcher) -command {PrefCheck; PrefWatcher}
    $w.page.m add command -label $t(compose_name) \
	    -command {PrefCheck; PrefCompose}
    $w.page.m add command -label $t(sending) -command {PrefCheck; PrefSending}
    $w.page.m add command -label $t(notification) \
	    -command {PrefCheck; PrefNotification}

    # The canvas
    frame $w.f -relief sunken -bd 1
    scrollbar $w.f.scroll \
	    -relief sunken \
	    -bd 1 \
	    -command "$w.f.canvas yview" \
	    -highlightthickness 0
    canvas $w.f.canvas -yscrollcommand "$w.f.scroll set" -highlightthickness 0
    Size $w.f.canvas prefCanvas
    pack $w.f.scroll -side right -fill y
    pack $w.f.canvas -side left -expand 1 -fill both
    set pref(frame) [frame $w.f.canvas.f]
    set pref(cid) [$w.f.canvas create window 0 0 \
	    -anchor nw \
	    -window $w.f.canvas.f]
    set pref(canvas) $w.f.canvas

    # The buttons
    frame $w.buttons
    frame $w.buttons.def -relief sunken -bd 1
    button $w.buttons.def.ok -text $t(apply) -command PrefApply
    pack $w.buttons.def.ok -padx 4 -pady 4
    button $w.buttons.cancel -text $t(reset) \
	    -command {
			global pref
			set pref(options) {}
			eval $pref(cmd)
		     }
    button $w.buttons.dismiss -text $t(dismiss) \
	    -command "PrefCheck; \
		      RecordSize $w.f.canvas prefCanvas;
		      RecordPos $w preferences;
		      destroy $w"
    pack $w.buttons.def \
	 $w.buttons.cancel \
         $w.buttons.dismiss -side left -expand 1
    bind $w <Return> PrefApply

    pack $w.page -side top -pady 5
    pack $w.buttons -side bottom -fill x -pady 5
    pack $w.f -side top -pady 10 -expand 1 -fill both

    Place $w preferences

    $lastPref
}


# PrefApply --
#
# Applies any changes to the preferences made in the current window.
#
# Arguments:

proc PrefApply {} {
    global option pref t folderWindowList

    switch $pref(cmd) {
    Prefgeneral {
	    if [string compare $option(mailcap_path) $pref(opt,mailcap_path)] {
		RatMailcapReload
	    }
	}
    PrefWatcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    PrefNotification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    PrefCompose {
	    if {[string compare $option(signature) $pref(opt,signature)]
		    && 1 == [llength [info commands RatUP_Signature]]} {
		Popup $t(sig_cmd_takes_precedence)
	    }
	}
    PrefSending {
	    if {[string compare $option(sendprog) $pref(opt,sendprog)]
		    && ![file executable [lindex $pref(opt,sendprog) 0]]} {
		Popup $t(warning_sendprog)
	    }
	}
    PrefFolder {
	    if [string compare $option(icon) $pref(opt,icon)] {
		SetIcon . $pref(opt,icon)
	    }
	    if [string compare $option(message_llength) \
			       $pref(opt,message_llength)] {
		foreach fw $folderWindowList {
		    upvar #0 $fw fh
		    $fh(message_list) configure \
			    -height $pref(opt,message_llength)
		}
	    }
	}
    }
    set hasChanged 0
    set needRestart 0
    foreach opt $pref(options) {
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set option($opt) $pref(opt,$opt)
	    set hasChanged 1
	    if { -1 != [lsearch -exact {language charset fontsize \
		    main_window_name icon_name default_folder \
		    watcher_name masquerade_as} $opt]} {
		set needRestart 1
	    }
	}
    }

    if $hasChanged {
	switch $pref(cmd) {
	PrefFolder {
		foreach f $folderWindowList {
		    Sync $f update
		}
	    }
	}

	SaveOptions
	if ![string compare PrefSending $pref(cmd)] {
	    RatSend kill
	}
    }
    if $needRestart {
	Popup $t(need_restart)
    }
}


# PrefCheck --
#
# Checks if there are any unapplied changes and if there is the user is
# queried if he wants to apply them.
#
# Arguments:

proc PrefCheck {} {
    global option pref t

    switch $pref(cmd) {
    PrefWatcher {
	    set pref(opt,watcher_time) [list [list std $pref(opt,finterval)] \
					     [list dbase $pref(opt,dinterval)]]
	}
    PrefNotification {
	    set pref(opt,dsn_verbose) [list [list failed $pref(opt,failed)] \
					[list delayed $pref(opt,delayed)] \
					[list delivered $pref(opt,delivered)] \
					[list relayed $pref(opt,relayed)] \
					[list expanded $pref(opt,expanded)]]
	}
    }
    foreach opt $pref(options) {
	if [string compare $option($opt) $pref(opt,$opt)] {
	    set value [RatDialog $t(unapplied_changes_title) \
		    $t(unapplied_changes) {} 0 $t(apply) $t(discard)]
	    if { 0 == $value } {
		PrefApply
	    }
	    return
	}
    }
}


# OptionMenu --
#
# Generates an option menu. The generated menu will have window name "w"
# and will set the "varName" variable. The different options are
# controlled by the value arguments. Each value argument is a list of
# two elemnts. The first is the value to set "varName" to and the second
# is the text to show. The menubutton will use "textVar" as the textvariable.
#
# Arguments:
# w	  -	Name of menubutton to create
# varName -	Variable to set to value
# textVar -	Variable to use for the text we show
# values  -	A list of lists which describes the values of this button
# locked  -	A boolean value which controls if the button is enabled

proc OptionMenu {w varName textVar values locked} {
    upvar #0 $varName var
    upvar #0 $textVar text

    set width 20
    menubutton $w -textvariable $textVar -indicatoron 1 \
		  -relief raised -menu $w.m
    menu $w.m -tearoff 0
    foreach elem $values {
	if ![string compare [lindex $elem 0] $var] {
	    set text [lindex $elem 1]
	}
	$w.m add command -label [lindex $elem 1] \
		-command "set $varName [list [lindex $elem 0]]; \
			  set $textVar [list [lindex $elem 1]]"
	if { $width < [string length [lindex $elem 1]]} {
	    set width [string length [lindex $elem 1]]
	}
    }
    if $locked {
	$w configure -state disabled
    }
    $w configure -width $width
}


# PrefGeneral --
#
# Shows the folder preferences
#
# Arguments:

proc PrefGeneral {} {
    global t option pref lastPref

    set lastPref PrefGeneral
    set pref(active) $t(general)
    set pref(cmd) PrefGeneral
    set pref(options) {language info_changes aliases_file dbase_dir hold_dir
		       userproc permissions print_command tmp fontsize
		       mail_steal keep_pos color_set iconic mailcap_path
		       terminal print_header}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }
    
    frame $f.lang
    label $f.lang.label -text $t(language): -width 20 -anchor e
    set lang {}
    foreach l [GetLanguages] {
	if ![string length [GetFont [lindex $l 2] 0 {}]] {
	    continue
	}
	lappend lang [lrange $l 0 1]
    }
    OptionMenu $f.lang.mbut pref(opt,language) pref(text,language) \
	    $lang [RatIsLocked option(language)]
    pack $f.lang.label \
	 $f.lang.mbut -side left -padx 5
    
    frame $f.color
    label $f.color.label -text $t(color_scheme): -width 20 -anchor e
    set width 20
    menubutton $f.color.mbut -textvariable pref(text,color_set) \
	    -indicatoron 1 -relief raised -menu $f.color.mbut.m
    menu $f.color.mbut.m -tearoff 0
    foreach c { {gray85 black} {PeachPuff2 black} {bisque black}
		{SlateBlue1 black} {SteelBlue4 white} {SkyBlue1 black}
		{aquamarine2 black} {SpringGreen4 black}} {
	set name $t([lindex $c 0])
	if ![string compare $c $option(color_set)] {
	    set pref(text,color_set) $name
	}
	$f.color.mbut.m add command -label $name \
		-command "set pref(opt,color_set) [list $c]; \
			  set pref(text,color_set) [list $name]; \
			  SetColor $c" \
		-background [lindex $c 0] -foreground [lindex $c 1]
	if { $width < [string length $name]} {
	    set width [string length $name]
	}
    }
    $f.color.mbut configure -width $width
    if [RatIsLocked option(color_set)] {
	$f.color.mbut configure -state disabled
    }
    pack $f.color.label \
	 $f.color.mbut -side left -padx 5

    frame $f.iconic
    label $f.iconic.label -text $t(startup_mode): -width 20 -anchor e
    OptionMenu $f.iconic.mbut pref(opt,iconic) pref(text,iconic) \
	    [list [list 1 $t(iconic)] [list 0 $t(normal)]] \
	    [RatIsLocked option(iconic)]
    pack $f.iconic.label \
	 $f.iconic.mbut -side left -padx 5
    
    frame $f.infoch
    label $f.infoch.label -text $t(show_changes): -width 20 -anchor e
    OptionMenu $f.infoch.mbut pref(opt,info_changes) pref(text,info_changes) \
	    [list [list 1 $t(show)] [list 0 $t(dont_show)]] \
	    [RatIsLocked option(info_changes)]
    pack $f.infoch.label \
	 $f.infoch.mbut -side left -padx 5
    
    frame $f.smail
    label $f.smail.label -text $t(check_stolen_mail): -width 20 -anchor e
    OptionMenu $f.smail.mbut pref(opt,mail_steal) pref(text,mail_steal) \
	    [list [list 1 $t(check)] [list 0 $t(dont_check)]] \
	    [RatIsLocked option(mail_steal)]
    pack $f.smail.label \
	 $f.smail.mbut -side left -padx 5
    
    frame $f.alias
    label $f.alias.label -text $t(aliases_file): -width 20 -anchor e
    entry $f.alias.entry -textvariable pref(opt,aliases_file) -relief sunken \
	    -width 60
    if [RatIsLocked option(aliases_file)] {
	$f.alias.entry configure -state disabled -relief flat
    }
    pack $f.alias.label \
	 $f.alias.entry -side left -padx 5
    
    frame $f.print
    label $f.print.label -text $t(print_command): -width 20 -anchor e
    entry $f.print.entry -textvariable pref(opt,print_command) -relief sunken \
	    -width 60
    if [RatIsLocked option(print_command)] {
	$f.print.entry configure -state disabled -relief flat
    }
    pack $f.print.label \
	 $f.print.entry -side left -padx 5
    
    frame $f.prhd
    label $f.prhd.label -text $t(print_headers): -width 20 -anchor e
    OptionMenu $f.prhd.mbut pref(opt,print_header) pref(text,print_header) \
	    [list [list all $t(all)] \
		  [list selected $t(selected)] \
		  [list none $t(none)]] \
	    [RatIsLocked option(print_header)]
    pack $f.prhd.label \
	 $f.prhd.mbut -side left -padx 5

    frame $f.tmp
    label $f.tmp.label -text $t(tmp_dir): -width 20 -anchor e
    entry $f.tmp.entry -textvariable pref(opt,tmp) -relief sunken \
	    -width 60
    if [RatIsLocked option(tmp)] {
	$f.tmp.entry configure -state disabled -relief flat
    }
    pack $f.tmp.label \
	 $f.tmp.entry -side left -padx 5
    
    frame $f.perm
    label $f.perm.label -text $t(file_permissions): -width 20 -anchor e
    entry $f.perm.entry -textvariable pref(opt,permissions) -relief sunken \
	    -width 60
    if [RatIsLocked option(permissions)] {
	$f.perm.entry configure -state disabled -relief flat
    }
    pack $f.perm.label \
	 $f.perm.entry -side left -padx 5
    
    frame $f.user
    label $f.user.label -text $t(userproc_file): -width 20 -anchor e
    entry $f.user.entry -textvariable pref(opt,userproc) -relief sunken \
	    -width 60
    if [RatIsLocked option(userproc)] {
	$f.user.entry configure -state disabled -relief flat
    }
    pack $f.user.label \
	 $f.user.entry -side left -padx 5
    
    frame $f.hold
    label $f.hold.label -text $t(hold_dir): -width 20 -anchor e
    entry $f.hold.entry -textvariable pref(opt,hold_dir) -relief sunken \
	    -width 60
    if [RatIsLocked option(hold_dir)] {
	$f.hold.entry configure -state disabled -relief flat
    }
    pack $f.hold.label \
	 $f.hold.entry -side left -padx 5
    
    frame $f.fsize
    label $f.fsize.label -text $t(character_size): -width 20 -anchor e
    OptionMenu $f.fsize.mbut pref(opt,fontsize) pref(text,fontsize) \
	    [list [list 8 "8"] \
		  [list 10 "10"] \
		  [list 12 "12"] \
		  [list 14 "14"] \
		  [list 18 "18"] \
		  [list 24 "24"]] \
	    [RatIsLocked option(fontsize)]
    pack $f.fsize.label \
	 $f.fsize.mbut -side left -padx 5
    
    frame $f.rpos
    label $f.rpos.label -text $t(remember_pos): -width 20 -anchor e
    OptionMenu $f.rpos.mbut pref(opt,keep_pos) pref(text,keep_pos) \
	    [list [list 1 $t(do_remember)] [list 0 $t(dont_remember)]] \
	    [RatIsLocked option(keep_pos)]
    pack $f.rpos.label \
	 $f.rpos.mbut -side left -padx 5
    
    frame $f.mpath
    label $f.mpath.label -text $t(mailcap_path): -width 20 -anchor e
    entry $f.mpath.entry -textvariable pref(opt,mailcap_path) -relief sunken \
	    -width 60
    if [RatIsLocked option(mailcap_path)] {
	$f.mpath.entry configure -state disabled -relief flat
    }
    pack $f.mpath.label \
	 $f.mpath.entry -side left -padx 5
    
    frame $f.term
    label $f.term.label -text $t(terminal_command): -width 20 -anchor e
    entry $f.term.entry -textvariable pref(opt,terminal) -relief sunken \
	    -width 60
    if [RatIsLocked option(terminal)] {
	$f.term.entry configure -state disabled -relief flat
    }
    pack $f.term.label \
	 $f.term.entry -side left -padx 5
    
    pack $f.lang \
	 $f.iconic \
	 $f.color \
	 $f.infoch \
	 $f.rpos \
	 $f.smail \
	 $f.alias \
	 $f.print \
	 $f.prhd \
	 $f.tmp \
	 $f.perm \
	 $f.user \
	 $f.hold \
	 $f.fsize \
	 $f.mpath \
	 $f.term -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefFolder --
#
# Shows the folder preferences
#
# Arguments:

proc PrefFolder {} {
    global t option pref lastPref

    set lastPref PrefFolder
    set pref(active) $t(folderwindow)
    set pref(cmd) PrefFolder
    set pref(options) {main_window_name icon_name list_format show_header
		       show_header_selection folder_sort start_selection
		       log_timeout keep_conn cache_passwd icon message_llength}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }
    
    frame $f.mname
    label $f.mname.label -text $t(window_name): -width 20 -anchor e
    entry $f.mname.entry -textvariable pref(opt,main_window_name) \
	    -relief sunken -width 60
    if [RatIsLocked option(main_window_name)] {
	$f.mname.entry configure -state disabled -relief flat
    }
    pack $f.mname.label \
	 $f.mname.entry -side left -padx 5
    
    frame $f.iname
    label $f.iname.label -text $t(icon_name): -width 20 -anchor e
    entry $f.iname.entry -textvariable pref(opt,icon_name) -relief sunken \
	    -width 60
    if [RatIsLocked option(icon_name)] {
	$f.iname.entry configure -state disabled -relief flat
    }
    pack $f.iname.label \
	 $f.iname.entry -side left -padx 5
 
    frame $f.icon
    label $f.icon.label -text $t(icon_bitmap): -width 20 -anchor e
    OptionMenu $f.icon.mbut pref(opt,icon) pref(text,icon) \
	    [list [list normal $t(normal_bitmap)] \
		  [list small $t(small_bitmap)] \
		  [list none $t(none)]] \
	    [RatIsLocked option(icon)]
    pack $f.icon.label \
	 $f.icon.mbut -side left -padx 5

    frame $f.llen
    label $f.llen.label -text $t(messages_shown): -width 20 -anchor e
    entry $f.llen.entry -textvariable pref(opt,message_llength) -relief sunken \
	    -width 60
    if [RatIsLocked option(message_llength)] {
	$f.llen.entry configure -state disabled -relief flat
    }
    pack $f.llen.label \
	 $f.llen.entry -side left -padx 5
    
    frame $f.list
    label $f.list.label -text $t(list_format): -width 20 -anchor e
    entry $f.list.entry -textvariable pref(opt,list_format) -relief sunken \
	    -width 60
    if [RatIsLocked option(list_format)] {
	$f.list.entry configure -state disabled -relief flat
    }
    pack $f.list.label \
	 $f.list.entry -side left -padx 5
    
    frame $f.show
    label $f.show.label -text $t(show_headers): -width 20 -anchor e
    OptionMenu $f.show.mbut pref(opt,show_header) pref(text,show_header) \
	    [list [list all $t(show_all_headers)] \
		  [list selected $t(show_selected_headers)] \
		  [list none $t(show_no_headers)]] \
	    [RatIsLocked option(show_header)]
    pack $f.show.label \
	 $f.show.mbut -side left -padx 5

    frame $f.ssel
    label $f.ssel.label -text $t(show_header_selection): -width 20 -anchor e
    entry $f.ssel.entry -textvariable pref(opt,show_header_selection) \
	    -relief sunken -width 60
    if [RatIsLocked option(show_header_selection)] {
	$f.ssel.entry configure -state disabled -relief flat
    }
    pack $f.ssel.label \
	 $f.ssel.entry -side left -padx 5
    
    frame $f.fsort
    label $f.fsort.label -text $t(sort_order): -width 20 -anchor e
    OptionMenu $f.fsort.mbut pref(opt,folder_sort) pref(text,folder_sort) \
	    [list [list subject $t(subject_by_date)] \
		  [list folder $t(natural_order)] \
		  [list reverseFolder $t(reverse_natural_order)] \
		  [list date $t(by_date)] \
		  [list reverseDate $t(reverse_date)]] \
	    [RatIsLocked option(folder_sort)]
    pack $f.fsort.label \
	 $f.fsort.mbut -side left -padx 5

    frame $f.startsel
    label $f.startsel.label -text $t(start_selection): -width 20 -anchor e
    OptionMenu $f.startsel.mbut pref(opt,start_selection) \
	    pref(text,start_selection) \
	    [list [list first $t(first_message)] \
		  [list last $t(last_message)] \
		  [list first_new $t(first_new_message)] \
		  [list before_new $t(before_first_new_message)]] \
	    [RatIsLocked option(start_selection)]
    pack $f.startsel.label \
	 $f.startsel.mbut -side left -padx 5

    frame $f.logt
    label $f.logt.label -text $t(log_timeout): -width 20 -anchor e
    entry $f.logt.entry -textvariable pref(opt,log_timeout) \
	    -relief sunken -width 60
    if [RatIsLocked option(log_timeout)] {
	$f.logt.entry configure -state disabled -relief flat
    }
    pack $f.logt.label \
	 $f.logt.entry -side left -padx 5
    
    frame $f.keep
    label $f.keep.label -text $t(keep_conn): -width 20 -anchor e
    entry $f.keep.entry -textvariable pref(opt,keep_conn) \
	    -relief sunken -width 60
    if [RatIsLocked option(keep_conn)] {
	$f.keep.entry configure -state disabled -relief flat
    }
    pack $f.keep.label \
	 $f.keep.entry -side left -padx 5
    
    frame $f.passwd
    label $f.passwd.label -text $t(cache_passwd): -width 20 -anchor e
    OptionMenu $f.passwd.mbut pref(opt,cache_passwd) pref(text,cache_passwd) \
	    [list [list 1 $t(do_cache)] \
		  [list 0 $t(do_not_cache)]] \
	    [RatIsLocked option(cache_passwd)]
    pack $f.passwd.label \
	 $f.passwd.mbut -side left -padx 5

    pack $f.mname \
	 $f.iname \
	 $f.icon \
	 $f.llen \
	 $f.list \
	 $f.show \
	 $f.ssel \
	 $f.fsort \
	 $f.startsel \
	 $f.logt \
	 $f.keep \
	 $f.passwd -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefDbase --
#
# Shows the dbase preferences
#
# Arguments:

proc PrefDbase {} {
    global t option pref lastPref

    set lastPref PrefDbase
    set pref(active) $t(dbase)
    set pref(cmd) PrefDbase
    set pref(options) {dbase_dir def_extype def_exdate dbase_backup \
		       expire_interval chunksize}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }
    
    frame $f.dbase
    label $f.dbase.label -text $t(dbase_dir): -width 20 -anchor e
    entry $f.dbase.entry -textvariable pref(opt,dbase_dir) -relief sunken \
	    -width 60
    if [RatIsLocked option(dbase_dir)] {
	$f.dbase.entry configure -state disabled -relief flat
    }
    pack $f.dbase.label \
	 $f.dbase.entry -side left -padx 5
    
    frame $f.extype
    label $f.extype.label -text $t(extype): -width 20 -anchor e
    OptionMenu $f.extype.mbut pref(opt,def_extype) pref(text,def_extype) \
	    [list [list none $t(none)] \
		  [list remove $t(remove)] \
		  [list incoming $t(add_incoming)] \
		  [list backup $t(backup)]] \
	    [RatIsLocked option(def_extype)]
    pack $f.extype.label \
	 $f.extype.mbut -side left -padx 5

    frame $f.exdate
    label $f.exdate.label -text $t(exdate): -width 20 -anchor e
    entry $f.exdate.entry -textvariable pref(opt,def_exdate) -relief sunken \
	    -width 60
    if [RatIsLocked option(def_exdate)] {
	$f.exdate.entry configure -state disabled -relief flat
    }
    pack $f.exdate.label \
	 $f.exdate.entry -side left -padx 5

    frame $f.backup
    label $f.backup.label -text $t(dbase_backup): -width 20 -anchor e
    entry $f.backup.entry -textvariable pref(opt,dbase_backup) -relief sunken \
	    -width 60
    if [RatIsLocked option(dbase_backup)] {
	$f.backup.entry configure -state disabled -relief flat
    }
    pack $f.backup.label \
	 $f.backup.entry -side left -padx 5

    frame $f.chunksize
    label $f.chunksize.label -text $t(chunksize): -width 20 -anchor e
    entry $f.chunksize.entry -textvariable pref(opt,chunksize) \
	    -relief sunken -width 60
    if [RatIsLocked option(chunksize)] {
	$f.chunksize.entry configure -state disabled -relief flat
    }
    pack $f.chunksize.label \
	 $f.chunksize.entry -side left -padx 5
    
    frame $f.interval
    label $f.interval.label -text $t(expire_interval): -width 20 -anchor e
    entry $f.interval.entry -textvariable pref(opt,expire_interval) \
	    -relief sunken -width 60
    if [RatIsLocked option(expire_interval)] {
	$f.interval.entry configure -state disabled -relief flat
    }
    pack $f.interval.label \
	 $f.interval.entry -side left -padx 5
    
    pack $f.dbase \
	 $f.extype \
	 $f.exdate \
	 $f.backup \
	 $f.chunksize \
	 $f.interval -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefWatcher --
#
# Shows the watcher preferences
#
# Arguments:

proc PrefWatcher {} {
    global t option pref lastPref

    set lastPref PrefWatcher
    set pref(active) $t(watcher)
    set pref(cmd) PrefWatcher
    set pref(options) {watcher_enable watcher_time watcher_name watcher_font
		       watcher_show watcher_bell watcher_format
		       watcher_max_height}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }

    frame $f.enable
    label $f.enable.label -text $t(watcher_enable): -width 20 -anchor e
    OptionMenu $f.enable.mbut pref(opt,watcher_enable) \
	    pref(text,watcher_enable) \
	    [list [list 1 $t(enabled)] \
		  [list 0 $t(disabled)]] \
	    [RatIsLocked option(watcher_enable)]
    pack $f.enable.label \
	 $f.enable.mbut -side left -padx 5

    frame $f.time
    label $f.time.label -text $t(intervals): -width 20 -anchor e
    frame $f.time.f
    label $f.time.f.label -text $t(stdfolders): -width 17 -anchor e
    entry $f.time.f.entry -textvariable pref(opt,finterval) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_time)] {
	$f.time.f.entry configure -state disabled -relief flat
    }
    pack $f.time.f.label \
	 $f.time.f.entry -side left -padx 5
    frame $f.time.d
    label $f.time.d.label -text $t(dbase_folders): -width 17 -anchor e
    entry $f.time.d.entry -textvariable pref(opt,dinterval) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_time)] {
	$f.time.d.entry configure -state disabled -relief flat
    }
    pack $f.time.d.label \
	 $f.time.d.entry -side left -padx 5
    pack $f.time.label -side left -anchor n
    pack $f.time.f \
	 $f.time.d -side top
    foreach elem $option(watcher_time) {
	if ![string compare [lindex $elem 0] std] {
	    set pref(opt,finterval) [lindex $elem 1]
	}
	if ![string compare [lindex $elem 0] dbase] {
	    set pref(opt,dinterval) [lindex $elem 1]
	}
    }
    
    frame $f.name
    label $f.name.label -text $t(window_name): -width 20 -anchor e
    entry $f.name.entry -textvariable pref(opt,watcher_name) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_name)] {
	$f.name.entry configure -state disabled -relief flat
    }
    pack $f.name.label \
	 $f.name.entry -side left -padx 5
    
    frame $f.max
    label $f.max.label -text $t(max_height): -width 20 -anchor e
    entry $f.max.entry -textvariable pref(opt,watcher_max_height) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_max_height)] {
	$f.max.entry configure -state disabled -relief flat
    }
    pack $f.max.label \
	 $f.max.entry -side left -padx 5
    
    frame $f.font
    label $f.font.label -text $t(font): -width 20 -anchor e
    entry $f.font.entry -textvariable pref(opt,watcher_font) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_font)] {
	$f.font.entry configure -state disabled -relief flat
    }
    pack $f.font.label \
	 $f.font.entry -side left -padx 5
    
    frame $f.show
    label $f.show.label -text $t(show_messages): -width 20 -anchor e
    OptionMenu $f.show.mbut pref(opt,watcher_show) pref(text,watcher_show) \
	    [list [list new $t(new)] \
		  [list all $t(all)]] \
	    [RatIsLocked option(watcher_show)]
    pack $f.show.label \
	 $f.show.mbut -side left -padx 5

    frame $f.format
    label $f.format.label -text $t(list_format): -width 20 -anchor e
    entry $f.format.entry -textvariable pref(opt,watcher_format) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_format)] {
	$f.format.entry configure -state disabled -relief flat
    }
    pack $f.format.label \
	 $f.format.entry -side left -padx 5
    
    frame $f.bell
    label $f.bell.label -text $t(bell_ringings): -width 20 -anchor e
    entry $f.bell.entry -textvariable pref(opt,watcher_bell) \
	    -relief sunken -width 60
    if [RatIsLocked option(watcher_bell)] {
	$f.bell.entry configure -state disabled -relief flat
    }
    pack $f.bell.label \
	 $f.bell.entry -side left -padx 5
    
    pack $f.enable \
	 $f.time \
	 $f.name \
	 $f.font \
	 $f.max \
	 $f.show \
	 $f.format \
	 $f.bell -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefCompose --
#
# Shows the compose preferences
#
# Arguments:

proc PrefCompose {} {
    global t option pref lastPref

    set lastPref PrefCompose
    set pref(active) $t(compose_name)
    set pref(cmd) PrefCompose
    set pref(options) {compose_headers editor reply_lead no_subject
		       signature attribution forwarded_message
		       lookup_name reply_to}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }

    frame $f.rto
    label $f.rto.label -text $t(default_reply_to): -width 20 -anchor e
    entry $f.rto.entry -textvariable pref(opt,reply_to) \
	    -relief sunken -width 60
    if [RatIsLocked option(reply_to)] {
	$f.rto.entry configure -state disabled -relief flat
    }
    pack $f.rto.label \
	 $f.rto.entry -side left -padx 5
    
    frame $f.head
    label $f.head.label -text $t(headers): -width 20 -anchor e
    entry $f.head.entry -textvariable pref(opt,compose_headers) -relief sunken \
	    -width 60
    if [RatIsLocked option(compose_headers)] {
	$f.head.entry configure -state disabled -relief flat
    }
    pack $f.head.label \
	 $f.head.entry -side left -padx 5
    
    frame $f.edit
    label $f.edit.label -text $t(external_editor): -width 20 -anchor e
    entry $f.edit.entry -textvariable pref(opt,editor) -relief sunken \
	    -width 60
    if [RatIsLocked option(editor)] {
	$f.edit.entry configure -state disabled -relief flat
    }
    pack $f.edit.label \
	 $f.edit.entry -side left -padx 5
    
    frame $f.attr
    label $f.attr.label -text $t(attribution): -width 20 -anchor e
    entry $f.attr.entry -textvariable pref(opt,attribution) -relief sunken \
	    -width 60
    if [RatIsLocked option(attribution)] {
	$f.attr.entry configure -state disabled -relief flat
    }
    pack $f.attr.label \
	 $f.attr.entry -side left -padx 5
    
    frame $f.forw
    label $f.forw.label -text $t(forwarded_label): -width 20 -anchor e
    entry $f.forw.entry -textvariable pref(opt,forwarded_message) \
	    -relief sunken -width 60
    if [RatIsLocked option(forwarded_message)] {
	$f.forw.entry configure -state disabled -relief flat
    }
    pack $f.forw.label \
	 $f.forw.entry -side left -padx 5
    
    frame $f.reply
    label $f.reply.label -text $t(reply_lead): -width 20 -anchor e
    entry $f.reply.entry -textvariable pref(opt,reply_lead) -relief sunken \
	    -width 60
    if [RatIsLocked option(reply_lead)] {
	$f.reply.entry configure -state disabled -relief flat
    }
    pack $f.reply.label \
	 $f.reply.entry -side left -padx 5
    
    frame $f.nosub
    label $f.nosub.label -text $t(no_subject): -width 20 -anchor e
    entry $f.nosub.entry -textvariable pref(opt,no_subject) -relief sunken \
	    -width 60
    if [RatIsLocked option(no_subject)] {
	$f.nosub.entry configure -state disabled -relief flat
    }
    pack $f.nosub.label \
	 $f.nosub.entry -side left -padx 5
    
    frame $f.sig
    label $f.sig.label -text $t(signature_file): -width 20 -anchor e
    entry $f.sig.entry -textvariable pref(opt,signature) -relief sunken \
	    -width 60
    if [RatIsLocked option(signature)] {
	$f.sig.entry configure -state disabled -relief flat
    }
    pack $f.sig.label \
	 $f.sig.entry -side left -padx 5

    frame $f.lookup
    label $f.lookup.label -text $t(lookup_name): -width 20 -anchor e
    OptionMenu $f.lookup.mbut pref(opt,lookup_name) pref(text,lookup_name) \
	    [list [list true $t(do_lookup)] \
		  [list false $t(dont_lookup)]] \
	    [RatIsLocked option(lookup_name)]
    pack $f.lookup.label \
	 $f.lookup.mbut -side left -padx 5

    pack $f.rto \
	 $f.head \
	 $f.edit \
	 $f.attr \
	 $f.forw \
	 $f.reply \
	 $f.nosub \
	 $f.sig \
	 $f.lookup -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefSending --
#
# Shows the sending preferences
#
# Arguments:

proc PrefSending {} {
    global t option pref lastPref

    set lastPref PrefSending
    set pref(active) $t(sending)
    set pref(cmd) PrefSending
    set pref(options) {masquerade_as domain sendprot smtp_hosts sendprog \
		       sendprog_8bit charset from smtp_verbose delivery_mode}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }

    frame $f.from
    label $f.from.label -text $t(use_from_address): -width 20 -anchor e
    entry $f.from.entry -textvariable pref(opt,from) -relief sunken \
	    -width 60
    if {$option(use_from)} {
	set frLock 0
    } else {
	set frLock 1
    }
    if [RatIsLocked option(from)] {
	set frLock 1
    }
    if $frLock {
	$f.from.entry configure -state disabled -relief flat
    }
    pack $f.from.label \
	 $f.from.entry -side left -padx 5

    frame $f.masq
    label $f.masq.label -text $t(masquerade_host_as): -width 20 -anchor e
    entry $f.masq.entry -textvariable pref(opt,masquerade_as) -relief sunken \
	    -width 60
    if [RatIsLocked option(masquerade_as)] {
	$f.masq.entry configure -state disabled -relief flat
    }
    pack $f.masq.label \
	 $f.masq.entry -side left -padx 5

    frame $f.domain
    label $f.domain.label -text $t(domain): -width 20 -anchor e
    entry $f.domain.entry -textvariable pref(opt,domain) -relief sunken \
	    -width 60
    if [RatIsLocked option(domain)] {
	$f.domain.entry configure -state disabled -relief flat
    }
    pack $f.domain.label \
	 $f.domain.entry -side left -padx 5

    frame $f.mode
    label $f.mode.label -text $t(delivery_mode): -width 20 -anchor e
    OptionMenu $f.mode.mbut pref(opt,delivery_mode) pref(text,delivery_mode) \
	    [list [list direct $t(direct)] [list deferred $t(deferred)]] \
	    [RatIsLocked option(delivery_mode)]
    pack $f.mode.label \
	 $f.mode.mbut -side left -padx 5

    frame $f.sendprot
    label $f.sendprot.label -text $t(sendprot): -width 20 -anchor e
    OptionMenu $f.sendprot.mbut pref(opt,sendprot) pref(text,sendprot) \
	    [list [list smtp $t(smtp)] [list prog $t(user_program)]] \
	    [RatIsLocked option(sendprot)]
    pack $f.sendprot.label \
	 $f.sendprot.mbut -side left -padx 5

    frame $f.smtp
    label $f.smtp.label -text $t(smtp_hosts): -width 20 -anchor e
    entry $f.smtp.entry -textvariable pref(opt,smtp_hosts) -relief sunken \
	    -width 60
    if [RatIsLocked option(smtp_hosts)] {
	$f.smtp.entry configure -state disabled -relief flat
    }
    pack $f.smtp.label \
	 $f.smtp.entry -side left -padx 5

    frame $f.smtpv
    label $f.smtpv.label -text $t(smtpv): -width 20 -anchor e
    OptionMenu $f.smtpv.mbut pref(opt,smtp_verbose) pref(text,smtp_verbose) \
	    [list [list 0 $t(none)] \
	          [list 1 $t(terse)] \
	          [list 2 $t(normal)] \
	          [list 3 $t(verbose)]] \
	    [RatIsLocked option(smtp_verbose)]
    pack $f.smtpv.label \
	 $f.smtpv.mbut -side left -padx 5

    frame $f.sendprog
    label $f.sendprog.label -text $t(sendprog): -width 20 -anchor e
    entry $f.sendprog.entry -textvariable pref(opt,sendprog) -relief sunken \
	    -width 60
    if [RatIsLocked option(sendprog)] {
	$f.sendprog.entry configure -state disabled -relief flat
    }
    pack $f.sendprog.label \
	 $f.sendprog.entry -side left -padx 5

    frame $f.sendprei
    label $f.sendprei.label -text $t(progin): -width 20 -anchor e
    OptionMenu $f.sendprei.mbut pref(opt,sendprog_8bit) \
	    pref(text,sendprog_8bit) \
	    [list [list true $t(eightbit)] \
	          [list false $t(sevenbit)]] \
	    [RatIsLocked option(sendprog_8bit)]
    pack $f.sendprei.label \
	 $f.sendprei.mbut -side left -padx 5

    frame $f.char
    label $f.char.label -text $t(charset): -width 20 -anchor e
    entry $f.char.entry -textvariable pref(opt,charset) -relief sunken \
	    -width 60
    if [RatIsLocked option(charset)] {
	$f.char.entry configure -state disabled -relief flat
    }
    pack $f.char.label \
	 $f.char.entry -side left -padx 5

    pack $f.from \
	 $f.masq \
	 $f.domain \
	 $f.mode \
	 $f.sendprot \
	 $f.smtp \
	 $f.smtpv \
	 $f.sendprog \
	 $f.sendprei \
	 $f.char -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}


# PrefNotification --
#
# Shows the notification preferences
#
# Arguments:

proc PrefNotification {} {
    global t option pref lastPref

    set lastPref PrefNotification
    set pref(active) $t(notification)
    set pref(cmd) PrefNotification
    set pref(options) {dsn_directory dsn_request dsn_snarf_reports \
		       dsn_expiration dsn_verbose}
    foreach opt $pref(options) {
	set pref(opt,$opt) $option($opt)
    }

    # Prepare frame for our windows
    set f $pref(frame)
    if [llength [pack slaves $f]] {
	eval "destroy [pack slaves $f]"
    }
    
    frame $f.dir
    label $f.dir.label -text $t(dsn_directory): -width 20 -anchor e
    entry $f.dir.entry -textvariable pref(opt,dsn_directory) -relief sunken \
	    -width 60
    if [RatIsLocked option(dsn_directory)] {
	$f.dir.entry configure -state disabled -relief flat
    }
    pack $f.dir.label \
	 $f.dir.entry -side left -padx 5
    
    frame $f.default
    label $f.default.label -text $t(default_action): -width 20 -anchor e
    OptionMenu $f.default.mbut pref(opt,dsn_request) pref(text,dsn_request) \
	    [list [list 0 $t(not_request_dsn)] \
		  [list 1 $t(request_dsn)]] \
	    [RatIsLocked option(dsn_request)]
    button $f.default.button -text $t(probe) -command PrefProbeDSN
    pack $f.default.label \
	 $f.default.mbut -side left -padx 5
    pack $f.default.button

    frame $f.snarf
    label $f.snarf.label -text $t(folderwindow): -width 20 -anchor e
    OptionMenu $f.snarf.mbut pref(opt,dsn_snarf_reports) \
	    pref(text,dsn_snarf_reports) \
	    [list [list 0 $t(not_snarf_dsn)] \
		  [list 1 $t(snarf_dsn)]] \
	    [RatIsLocked option(dsn_snarf_reports)]
    pack $f.snarf.label \
	 $f.snarf.mbut -side left -padx 5

    frame $f.expire
    label $f.expire.label -text $t(dsn_expiration): -width 20 -anchor e
    entry $f.expire.entry -textvariable pref(opt,dsn_expiration) \
	    -relief sunken -width 60
    pack $f.expire.label \
	 $f.expire.entry -side left -padx 5
    if [RatIsLocked option(dsn_expiration)] {
	$f.expire.entry configure -state disabled -relief flat
    }

    foreach elem $option(dsn_verbose) {
	set pref(opt,[lindex $elem 0]) [lindex $elem 1]
    }
    frame $f.verbose
    label $f.verbose.label -text $t(report_level): -width 20 -anchor e
    pack $f.verbose.label -side left -anchor n
    foreach cat {failed delayed delivered relayed expanded} {
	set sf $f.verbose.$cat
	frame $sf
	label $sf.l -text $t($cat): -width 12 -anchor e
	OptionMenu $sf.mbut pref(opt,$cat) pref(text,$cat) \
		[list [list none $t(rl_none)] \
		      [list status $t(rl_status)] \
		      [list notify $t(rl_notify)]] \
		[RatIsLocked option(dsn_verbose)]
	pack $sf.l $sf.mbut -side left -padx 5
	pack $sf -side top -anchor w
    }
    
    pack $f.dir \
	 $f.default \
	 $f.snarf \
	 $f.expire \
	 $f.verbose -side top -fill x -pady 5
    update idletasks
    set bbox [$pref(canvas) bbox $pref(cid)]
    eval {$pref(canvas) configure -scrollregion $bbox}
}

# PrefProbeDSN --
#
# Probe the current SMTP servers for DSN support.
#
# Arguments:

proc PrefProbeDSN {} {
    global idCnt option t defaultFont

    # Create identifier
    set id probeWin[incr idCnt]
    upvar #0 $id hd
    set w .$id
    set hd(w) $w

    # Create toplevel
    toplevel $w
    wm title $w $t(probe)

    if [string compare $option(sendprot) smtp] {
	message $w.message -aspect 600 -text $t(dsn_need_smtp)
	button $w.button -text $t(dismiss) \
		-command "RecordPos $w prefProbeDSN; destroy $w; unset $id"
	pack $w.message \
	     $w.button -side top -padx 5 -pady 5
	return
    }

    set row 0
    foreach h $option(smtp_hosts) {
	label $w.l$row -text $h -width 32 -anchor e
	label $w.a$row -textvariable ${id}($h) -font $defaultFont -width 32 \
		-anchor w
	grid $w.l$row -row $row -column 0 -sticky e
	grid $w.a$row -row $row -column 1 -sticky w
	incr row
    }
    button $w.button -text $t(dismiss) -command "destroy $w; unset $id" \
	    -state disabled
    grid $w.button -row $row -column 0 -columnspan 2
    Place $w prefProbeDSN

    foreach h $option(smtp_hosts) {
        set hd($h) $t(probing)...
	update idletasks
	if [RatSMTPSupportDSN $h] {
	    set hd($h) $t(supports_dsn)
	} else {
	    set hd($h) $t(no_dsn_support)
	}
    }
    $w.button configure -state normal
}
