# show.tcl --
#
# This file contains code which handles the actual displaying of a message
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# A list of types we can show
set GoodTypes {text/plain message/rfc822 image/gif}

# A list of parameters to show for different types
set showParams(text/plain) {charset}
set showParams(message/external-body) {access-type site server name directory}

# ShowNothing --
#
# Clear the show window
#
# Arguments:
# handler -	The handler which identifies the show text widget

proc ShowNothing handler {
    $handler configure -state normal
    $handler delete 0.0 end
    $handler configure -state disabled
}

# Show --
#
# Shows the given message in the show portion of a folder window
#
# Arguments:
# handler -	The handler which identifies the show text widget
# msg     -	Message to show

proc Show {handler msg} {
    upvar #0 hd$handler fh
    global option langCharset

    set fh(id) 0
    set fh(current) $msg

    # Enable updates
    $handler configure -state normal

    # Reset the tags to a known status
    $handler tag delete [$handler tag names]
    $handler tag configure HeaderName -font [GetFont $langCharset 0 bold]
    $handler tag configure Center -justify center -spacing1 5m -spacing3 5m

    # Delete old subwindows
    foreach slave [winfo children $handler] {
	destroy $slave
    }
    # Do other misc cleanup
    if [info exists fh(show_cleanup_cmds)] {
	foreach clear $fh(show_cleanup_cmds) {
	    eval $clear
	}
	set fh(show_cleanup_cmds) {}
    }
    foreach n [array names fh charset_*] {
	unset fh($n)
    }
    set fh(lastType) ""

    # Clear the display
    $handler delete 0.0 end

    # Show the message
    ShowMessage 1 $handler $msg
    BuildStructMenu $handler $msg

    # Don't allow the user to change this
    $handler configure -state disabled
}

# InsertHeader --
#
# Inserts a header row into the text widget.
#
# Arguments:
# gtag	 - Tag to add to text elements
# w      - Text widget to insert into
# header - List of header rows

proc InsertHeader {gtag w header {width 1}} {
    global t
    upvar #0 hd$w fh

    set hn [lindex $header 0]
    regsub -all -- - [string tolower $hn] _ hni
    if [info exists t($hni)] {
	set hn $t($hni)
    }
    $w insert insert [format "%${width}s: " $hn] "HeaderName $gtag"

    foreach h [lindex $header 1] {
	set charset [lindex $h 0]
	if [info exists fh(charset_$charset)] {
	    set tag $fh(charset_$charset)
	} else {
	    set font [GetFont $charset 0 {}]
	    if [string length $font] {
		set tag t[incr fh(id)]
		set fh(charset_$charset) $tag
		$w tag configure $tag -font $font
	    }
	}
	if [info exists tag] {
	    $w insert insert [lindex $h 1] "$tag $gtag"
	    unset tag
	} else {
	    $w insert insert " ??? "
	}
    }
    $w insert insert "\n" $gtag
}

# ShowMessage --
#
# Inserts a message entity at the end. This is done by first inserting the
# selected headers and then the body via ShowBody.
#
# Arguments:
# first	  -	Indicates if this is the top message or an embedded message
# handler -	The handler which identifies the show text widget
# msg     -	Message to show

proc ShowMessage {first handler msg} {
    global option t
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    set tag t[incr fh(id)]
    if ![info exists msgInfo(show,$msg)] {
        set msgInfo(show,$msg) 1
    }
    if {0 == $first} {
	$handler tag bind $tag <3> "tk_popup $fh(struct_menu) %X %Y \
			        \[lsearch \[set hd${handler}(struct_list)\] \
			        $msg\]"
    }

    if ![string length $msg] {
	$handler insert insert "\[$t(empty_message)\]" $tag
	return
    }

    # Add a newline if this isn't the first message
    if {0 == $first} {
	$handler insert insert "\n" $tag
    }

    switch $fh(show_header) {
    all	{
	    foreach h [$msg headers split] {
		InsertHeader $tag $handler $h
	    }
	}
    selected {
	    foreach h [$msg headers split] {
		set header([string tolower [lindex $h 0]]) [lindex $h 1]
	    }
	    set length 5
	    foreach f $option(show_header_selection) {
		if { $length < [string length $f]} {
		    set length [string length $f]
		}
	    }
	    foreach f $option(show_header_selection) {
		set n [string tolower $f]
		if [info exists header($n)] {
		    InsertHeader $tag $handler [list $f $header($n)] $length
		}
	    }
	}
    default	{ }
    }

    # Insert the body
    set body [$msg body]
    if ![info exists msgInfo(show,$body)] {
	set msgInfo(show,$body) 1
    }
    if $msgInfo(show,$msg) {
	$handler insert insert "\n"
	if ![info exists msgInfo(show,$body)] {
	    set msgInfo(show,$body) 1
	}
    } else {
        set msgInfo(show,$body) 0
    }
    ShowBody $handler $body $msg
}

# ShowBody --
#
# Inserts a bodypart entity at the end, this is really just a switch which
# calls upon the commands which does the actual work.
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowBody {handler body msg} {
    global option
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    if !$msgInfo(show,$body) {
        return
    }

    # We will need this later on.
    set type [string tolower [$body type]]

    # Draw separator between items only if the previous entity was of
    # the same type and not multipart.
    set mtype [lindex $type 0]
    if [string compare $mtype multipart] {
        if ![string compare $mtype $fh(lastType)] {
	    frame $handler.f[incr fh(id)] -width 12c -height 2 \
		    -relief sunken -bd 2
	    $handler window create insert -window $handler.f$fh(id) -pady 10
	    $handler insert insert "\n"
        }
        set fh(lastType) $mtype
    }

    # Switch for subroutines which does the actual drawing
    switch -glob [lindex $type 0]/[lindex $type 1] {
    text/plain			{ ShowTextPlain $handler $body $msg}
    text/*			{ ShowTextOther $handler $body $msg}
    multipart/alternative	{ ShowMultiAlt $handler $body $msg }
    multipart/*			{ ShowMultiMixed $handler $body $msg }
    message/rfc822		{ ShowMessage 0 $handler [$body message] }
    message/delivery-status	{ ShowDSN $handler $body }
    image/gif           	{ ShowImage $handler $body gif }
    default			{ ShowDefault $handler $body }
    }
}

# ShowTextPlain --
#
# Show text/plain entities, should handle different fonts...
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowTextPlain {handler body msg} {
    global option defaultFont
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    set tag t[incr fh(id)]
    $handler tag bind $tag <3> "tk_popup $fh(struct_menu) %X %Y \
			        \[lsearch \[set hd${handler}(struct_list)\] \
			        $body\]"
    set action [$body getShowCharset]
    if ![string compare good [lindex $action 0]] {
	$handler tag configure $tag -font [GetFont [lindex $action 1] 0 {}]
	$handler insert insert [$body data false [lindex $action 1]] $tag
    } else {
	global t
	set w $handler.w[incr fh(id)]
	frame $w -relief raised -bd 2 -cursor top_left_arrow
	if [string length [$body description]] {
	    label $w.desc -text "$t(description): [$body description]" \
		    -font $defaultFont
	    pack $w.desc -side top -anchor nw
	}
	set charset [$body parameter charset]
	label $w.type \
	    	-text "$t(here_is_text) '$charset' $t(which_cant_be_shown)"
	label $w.modlab -text $t(mode):
	menubutton $w.mode -menu $w.mode.m -indicatoron 1 -relief raised \
	    	-textvariable hd${handler}(mode,$body)
	button $w.save -text $t(save_to_file) -command "SaveBody $body"
	pack $w.type -side top -anchor w
	pack $w.modlab $w.mode -side left
	pack $w.save -side left -padx 20
	set l 0
	menu $w.mode.m -tearoff 0
	foreach s {dont_show show_sevenbit show_all} {
	    $w.mode.m add command -label $t($s) \
		    -command "ShowTextCharset $body $msg hd$handler $handler \
					      $tag $s"
	    if {[string length $t($s)] > $l} {
		set l [string length $t($s)]
		$w.mode configure -width $l
	    }
	}
	if ![info exists msgInfo(show,$body,how)] {
	    set msgInfo(show,$body,how) show_sevenbit
	}
	$handler window create insert -window $w -pady 5
	foreach win [concat $w [pack slaves $w]] {
	    bind $win <3> "tk_popup $fh(struct_menu) %X %Y \
			   \[lsearch \[set hd${handler}(struct_list)\] \
			   $body\]"
	}
	$handler insert insert "\n" $tag
	$handler mark set ${body}_s insert
	$handler mark set ${body}_e insert
	$handler mark gravity ${body}_s left
	$handler mark gravity ${body}_e right
	ShowTextCharset $body $msg hd$handler $handler $tag \
		$msgInfo(show,$body,how)
    }
}

# ShowTextCharset --
#
# Shows text in a nonstandard character set
#
# Arguments:
# body    -	The bodypart to show
# msg     -	The message name
# hd	  -	The array to use for global variables
# w 	  -	The handler which identifies the show text widget
# tag	  -	The tag this entity should have
# mode	  -	What to show

proc ShowTextCharset {body msg hd w tag mode} {
    upvar #0 $hd fh \
    	     msgInfo_$msg msgInfo
    global t

    $w configure -state normal
    $w delete ${body}_s ${body}_e
    $w mark set insert ${body}_s

    set fh(mode,$body) $t($mode)
    set msgInfo(show,$body,how) $mode

    switch $mode {
    show_sevenbit {
	    $w insert insert [$body data false 7bit] $tag
	}
    show_all {
	    $w insert insert [$body data false [$body parameter charset]] $tag
	}
    }
    $w configure -state disabled
}

# ShowTextOther --
#
# Show text parts other that text/plain.
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowTextOther {handler body msg} {
    global t defaultFont langCharset
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    set type [$body type]
    set typename [string tolower [lindex $type 0]/[lindex $type 1]]
    set mailcap [$body findShowCommand]
    set width 0

    set w $handler.w[incr fh(id)]
    frame $w -relief raised -bd 2 -cursor top_left_arrow
    if [string length [lindex $mailcap 4]] {
	if ![catch {image create bitmap $body.icon -file [lindex $mailcap 4]}] {
	    label $w.icon -image $body.icon
	    pack $w.icon -side left
	}
    }
    text $w.text -relief flat -cursor top_left_arrow
    $w.text tag configure Bold -font [GetFont $langCharset 0 bold]
    if [string length [$body description]] {
	set width [string length "$t(description): [$body description]"]
	$w.text insert insert "$t(description): " Bold [$body description]\n
    }
    if [string length [lindex $mailcap 3]] {
	set l [string length "$t(type_description): [lindex $mailcap 3]"]
	if { $l > $width } {
	    set width $l
	}
	$w.text insert insert "$t(type_description): " Bold \
		[lindex $mailcap 3]\n
    }
    set typestring [string tolower [lindex $type 0]/[lindex $type 1]]
    set size [RatMangleNumber [$body size]]
    set l [string length "$t(here_is): $typestring ($size bytes)"]
    if { $l > $width } {
	set width $l
    }
    $w.text insert insert "$t(here_is): " Bold "$typestring  ($size bytes)" {}
    $w.text configure \
	    -height [lindex [split [$w.text index end-1c] .] 0] \
	    -width $width \
	    -state disabled
    pack $w.text -side top -anchor w

    if [string length [lindex $mailcap 0]] {
	button $w.view_ext -text $t(view) -command "RunMailcap $body {$mailcap}"
	pack $w.view_ext -side left -padx 10
    }
    if ![info exists msgInfo(show,$body,tp)] {
	if [string length [lindex $mailcap 0]] {
	    set msgInfo(show,$body,tp) 0
	} else {
	    set msgInfo(show,$body,tp) 1
	}
    }
    checkbutton $w.tp -text $t(view_as_text) \
	    -relief raised \
	    -variable msgInfo_${msg}(show,$body,tp) \
	    -command "ShowTextOtherDo $handler $body $msg" \
	    -padx 4 -pady 4
    button $w.save -text $t(save_to_file) -command "SaveBody $body"
    button $w.view_int -text $t(view_source) -command "ShowSource $body"
    pack $w.tp $w.save $w.view_int -side left -padx 5

    $handler window create insert -window $w -padx 5 -pady 5
    foreach win [concat $w [pack slaves $w]] {
	bind $win <3> "tk_popup $fh(struct_menu) %X %Y \
		       \[lsearch \[set hd${handler}(struct_list)\] \
		       $body\]"
    }
    $handler insert insert "\n"

    $handler mark set ${body}_s insert
    $handler mark set ${body}_e insert
    $handler mark gravity ${body}_s left
    $handler mark gravity ${body}_e right
    ShowTextOtherDo $handler $body $msg
}

# ShowTextOtherDo --
#
# Subfunction of ShowTextOther shows not/shows the text as text/plain
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowTextOtherDo {handler body msg} {
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo
    
    $handler configure -state normal
    $handler delete ${body}_s ${body}_e
    $handler mark set insert ${body}_s
    if $msgInfo(show,$body,tp) {
	ShowTextPlain $handler $body $msg
    }
    $handler configure -state disabled
}

# ShowMultiAlt --
#
# Show a multipart/alternative object
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowMultiAlt {handler body msg} {
    global GoodTypes
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo
    set found 0

    if ![info exists msgInfo(alternated,$body)] {
        set msgInfo(alternated,$body) 1
        set children [$body children]
        foreach c $children {
            set msgInfo(show,$c) 0
	}
        for {set i [expr [llength $children]-1]} {$i >= 0} {incr i -1} {
            set child [lindex $children $i]
	    set tc $child
            set typelist [$child type]
	    while {1 == [regexp -nocase multipart [lindex $typelist 0]]} {
		set tc [lindex [$tc children] 0]
		set typelist [$tc type]
	    }
            set type [string tolower [lindex $typelist 0]/[lindex $typelist 1]]
            if { -1 != [lsearch $GoodTypes $type] } {
		set msgInfo(show,$child) 1
		set found 1
                break
            }
        }
    }

    foreach child [$body children] {
	if !$found {
            set msgInfo(show,$child) 1
	}
	ShowBody $handler $child $msg
    }
}

# ShowMultiMixed --
#
# Show a multipart/mixed object
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# msg     -	The message name

proc ShowMultiMixed {handler body msg} {
    global option t
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    foreach child [$body children] {
	if ![info exists msgInfo(show,$child)] {
	    set msgInfo(show,$child) $msgInfo(show,$msg)
	}
	ShowBody $handler $child $msg
    }
}

# ShowImageGif --
#
# Show image/gif entities
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show
# subtype -	The type of image

proc ShowImage {handler body subtype} {
    global option
    upvar #0 hd$handler fh

    set tag t[incr fh(id)]
    set filename $option(tmp)/rat.[RatGenId]
    set fid [open $filename w 0600]
    $body saveData $fid 0 0
    close $fid
    switch $subtype {
    gif    { set img [image create photo -format gif -file $filename] }
    }
    exec rm -f $filename
    set c [canvas $handler.c[incr fh(id)] -width [image width $img] \
	    -height [image height $img]]
    $c create image 0 0 -image $img -anchor nw
    $handler insert insert " " "Center $tag"
    $handler window create insert -window $c
    $handler insert insert "\n" $tag
    lappend fh(show_cleanup_cmds) "image delete $img"
    $handler tag bind $tag <3> "tk_popup $fh(struct_menu) %X %Y \
				 \[lsearch \[set hd${handler}(struct_list)\] \
				 $body\]"
    bind $c <3> "tk_popup $fh(struct_menu) %X %Y \
			  \[lsearch \[set hd${handler}(struct_list)\] \
			  $body\]"
}


# ShowDSN --
#
# Show message/delivery-status entities
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The body which contains this DSN

proc ShowDSN {handler body} {
    global t
    upvar #0 hd$handler fh

    set tag t[incr fh(id)]
    $handler tag bind $tag <3> "tk_popup $fh(struct_menu) %X %Y \
				 \[lsearch \[set hd${handler}(struct_list)\] \
				 $body\]"
    $handler insert insert \n $tag
    set dsn [$body dsn]
    foreach h [lindex $dsn 0] {
	set hn [lindex $h 0]
	if [info exists t([string tolower $hn])] {
	    set hn $t([string tolower $hn])
	}
	$handler insert insert [format "%21s: " $hn] "$tag HeaderName"
	$handler insert insert [lindex $h 1]\n $tag
    }
    foreach r [lindex $dsn 1] {
	$handler insert insert \n $tag
	foreach h $r {
	    set hn [lindex $h 0]
	    if [info exists t([string tolower $hn])] {
		set hn $t([string tolower $hn])
	    }
	    $handler insert insert [format "%21s: " $hn] "$tag HeaderName"
	    $handler insert insert [lindex $h 1]\n $tag
	}
    }
    $handler insert insert \n $tag
}

# ShowDefault --
#
# The default type shower, just iserts a marker that here is an object
# of this type.
#
# Arguments:
# handler -	The handler which identifies the show text widget
# body    -	The bodypart to show

proc ShowDefault {handler body} {
    global t defaultFont langCharset
    upvar #0 hd$handler fh

    set type [$body type]
    set typename [string tolower [lindex $type 0]/[lindex $type 1]]
    set mailcap [$body findShowCommand]
    set width 0

    set w $handler.w[incr fh(id)]
    frame $w -relief raised -bd 2 -cursor top_left_arrow
    if [string length [lindex $mailcap 4]] {
	if ![catch {image create bitmap $body.icon -file [lindex $mailcap 4]}] {
	    label $w.icon -image $body.icon
	    pack $w.icon -side left
	}
    }
    text $w.text -relief flat -cursor top_left_arrow
    $w.text tag configure Bold -font [GetFont $langCharset 0 bold]
    if [string length [$body description]] {
	set width [string length "$t(description): [$body description]"]
	$w.text insert insert "$t(description): " Bold [$body description]\n
    }
    if [string length [lindex $mailcap 3]] {
	set l [string length "$t(type_description): [lindex $mailcap 3]"]
	if { $l > $width } {
	    set width $l
	}
	$w.text insert insert "$t(type_description): " Bold [lindex $mailcap 3]\n
    }
    set l [string length "$t(here_is): [string tolower [lindex $type 0]/[lindex $type 1]] ([RatMangleNumber [$body size]] bytes)"]
    if { $l > $width } {
	set width $l
    }
    $w.text insert insert "$t(here_is): " Bold \
	    "[string tolower [lindex $type 0]/[lindex $type 1]]" {} \
	    " ([RatMangleNumber [$body size]] bytes)"
    $w.text configure \
	    -height [lindex [split [$w.text index end-1c] .] 0] \
	    -width $width \
	    -state disabled
    pack $w.text -side top -anchor w

    if [string length [lindex $mailcap 0]] {
	button $w.view_ext -text $t(view) -command "RunMailcap $body {$mailcap}"
	pack $w.view_ext -side left -padx 10
    }
    button $w.save -text $t(save_to_file) -command "SaveBody $body"
    button $w.view_int -text $t(view_source) -command "ShowSource $body"
    pack $w.save $w.view_int -side left -padx 5
    $handler window create insert -window $w -padx 5 -pady 5
    foreach win [concat $w [pack slaves $w]] {
	bind $win <3> "tk_popup $fh(struct_menu) %X %Y \
		       \[lsearch \[set hd${handler}(struct_list)\] \
		       $body\]"
    }
    $handler insert insert "\n"
}

# ShowHome --
#
# Scroll to the top of the document
#
# Arguments:
# handler -	The handler which identifies the show text widget

proc ShowHome {handler} {
    $handler yview moveto 0
}

# ShowPageDown --
#
# Move the show one page down
#
# Arguments:
# handler -	The handler which identifies the show text widget

proc ShowPageDown {handler} {
    $handler yview scroll 1 pages
}

# ShowPageUp --
#
# Move the show one page up
#
# Arguments:
# handler -	The handler which identifies the show text widget

proc ShowPageUp {handler} {
    $handler yview scroll -1 pages
}

# SaveBody --
#
# Save a bodypart to file.
#
# Arguments:
# body -	The bodypart to save

proc SaveBody {body} {
    global idCnt t option

    # Create identifier
    set id showSave[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    if [regexp -nocase text [lindex [$body type] 0]] {
	set hd(convertNL) 1
    } else {
	set hd(convertNL) 0
    }

    # Create toplevel
    toplevel $w
    wm title $w $t(save_to_file)

    # Filechooser
    set hd(name) [$body filename]
    frame $w.chooser
    set fh [FileSelectorCreate $w.chooser ${id}(name) "set ${id}(done) 1" 0 0]

    # Newline conversion
    checkbutton $w.localnl -text $t(convert_to_local_nl) \
	    -variable ${id}(convertNL)

    # OK and cancel buttons
    frame $w.buttons
    button $w.buttons.ok -text $t(ok) -command "FileSelectorUpdate $fh 1"
    CreateModeMenu $w.buttons.mode $id $option(permissions)
    button $w.buttons.cancel -text $t(cancel) -command "set ${id}(done) 0"
    pack $w.buttons.ok \
	 $w.buttons.mode \
	 $w.buttons.cancel -side left -expand 1

    # Pack it
    pack $w.buttons -side bottom -fill x
    pack $w.localnl -side bottom -anchor w -pady 5 -padx 5
    pack $w.chooser -fill both -expand 1

    Place $w saveBody
    grab $w
    tkwait variable ${id}(done)

    # Do save
    if { 1 == $hd(done) } {
	FileSelectorDone $fh
	if [file exists $hd(name)] {
	    set action [RatDialog $t(file_exists) \
				  "$hd(name) $t(already_exists)." \
				  {} 1 $t(overwrite) $t(cancel)]
	} else {
	    set action 0
	}
	if {0 == $action} {
	    set mode [GetMode $id]
	    if { 0 == [catch "open $hd(name) w $mode" fh]} {
		$body saveData $fh false $hd(convertNL)
		close $fh
		# Defeat the umask
		catch {exec chmod [format %#o $perm] $hd(name)}
	    } else {
		RatLog 4 "$t(save_failed): $fh"
	    }
	}
    }

    RecordPos $w saveBody
    destroy $w
    unset hd
    focus $oldFocus
}

# BuildStructMenu
#
# Builds the structure menu
#
# Arguments:
# handler -	The handler which identifies the show text widget
# m       -	The menu to build
# msg     -	The message we should display the structure of

proc BuildStructMenu {handler msg} {
    upvar #0 hd$handler fh

    # Clear the old menu
    $fh(struct_menu) delete 0 end
    foreach slave [winfo children $fh(struct_menu)] {
	destroy $slave
    }
    set fh(struct_list) {}

    # Check if we got anything
    if ![string length $msg] { return }

    BuildStructEntry $handler $fh(struct_menu) $msg [$msg body] ""
}

# BuildStructEntry
#
# Builds an entry in the structure menu
#
# Arguments:
# handler -	The handler which identifies the show text widget
# m       -	The menu to build
# msg     -	The message we should display the structure of
# body    -	The bodypart to describe
# preamble-	The preamble to add before this entry

proc BuildStructEntry {handler m msg body preamble} {
    global t showParams
    upvar #0 hd$handler fh \
    	     msgInfo_$msg msgInfo

    if ![string length $body] { return }

    lappend fh(struct_list) $body
    set sm $m.m[incr fh(id)]
    set typepair [$body type]
    set type [string tolower [lindex $typepair 0]/[lindex $typepair 1]]
    $m add cascade -label $preamble$type -menu $sm
    menu $sm -tearoff 0 \
    	     -disabledforeground [lindex [$m configure -activeforeground] 3]

    if [string length [$body description]] {
        $sm add command -label [$body description] -state disabled
    }
    # Do parameters
    if [info exists showParams($type)] {
        set sp $showParams($type)
        foreach param $sp {
            set value [$body parameter $param]
            if [string length $param] {
		$sm add command -label "$t($param): $value" \
				-state disabled
            }
        }
    } else {
        set sp {{}}
    }
    $sm add command -label "$t(size): [RatMangleNumber [$body size]]" \
    		    -state disabled
    $sm add separator
    if ![info exists msgInfo(show,$body)] {
	set msgInfo(show,$body) 1
    }
    $sm add checkbutton -label $t(show) -onvalue 1 -offvalue 0 \
    	    -variable msgInfo_${msg}(show,$body) \
    	    -command "Show $handler $fh(current)"
    $sm add command -label $t(save_to_file)... -command "SaveBody $body"
    $sm add command -label $t(view_source)... -command "ShowSource $body"

    # See if we have children to show
    switch -glob $type {
    message/rfc822
        {
	    set message [$body message]
	    set fh(struct_list) [lreplace $fh(struct_list) end end $message]
	    if [string length $message] {
		BuildStructEntry $handler $m $message [$message body] \
			"$preamble  "
	    }
        }
    multipart/*
        {
            foreach c [$body children] {
                BuildStructEntry $handler $m $msg $c "$preamble  "
            }
        }
    }
}

# RunMailcap --
#
# Runs the mailcap entry for a body
#
# Arguments:
# body    -	The handler which identifies the entity to show
# mailcap -	The mailcap entry to use

proc RunMailcap {body mailcap} {
    global option t idCnt defaultFont

    set id id[incr idCnt]
    upvar #0 $id rm
    set cmd [lindex $mailcap 0]

    # Fix files
    set rm(fileName) $option(tmp)/rat.[RatGenId]
    if [regexp %s $cmd] {
	regsub -all %s $cmd $rm(fileName) cmd
    } else {
	if [lindex $mailcap 1] {
	    Popup "$t(cant_pipe): \"$cmd\""
	    unset rm
	    return
	}
	set cmd "cat $rm(fileName) | $cmd"
    }
    set f [open $rm(fileName) w]
    $body saveData $f 0 0
    close $f
    if [lindex $mailcap 1] {
	# This command needs a terminal
	RatBgExec ${id}(existStatus) "$option(terminal) $cmd"
    } elseif [lindex $mailcap 2] {
	# This command produces lots of output
	set w .$id
	toplevel $w
	wm title $w $t(external_viewer)
	text $w.text \
		-yscroll "$w.scroll set" \
		-relief sunken -bd 1 \
		-setgrid 1 \
		-font $defaultFont \
		-highlightthickness 0
	Size $w.text extView
	scrollbar $w.scroll \
		-relief sunken \
		-bd 1 \
		-command "$w.text yview" \
		-highlightthickness 0
	button $w.button -text $t(dismiss) -command "RecordPos $w extView; \
		RecordSize $w.text extView; destroy $w"
	pack $w.button -side bottom -pady 5
	pack $w.scroll -side right -fill y
	pack $w.text -expand 1 -fill both
	Place $w extView
	$w.text insert insert [eval "exec $cmd"]
	$w.text configure -state disabled
	exec rm -f $rm(fileName) &
	unset rm
	return
    } else {
	# This command manages its own output
	RatBgExec ${id}(existStatus) $cmd
    }
    trace variable rm(existStatus) w RunMailcapDone
}

# RunMailcapDone --
#
# This gets called when the show command has run and should clean
# things up.
#
# Arguments:
# name1, name2 -        Variable specifiers
# op           -        Operation

proc RunMailcapDone {name1 name2 op} {
    upvar #0 $name1 rm

    exec rm -f $rm(fileName) &
    unset rm
}

