# vfolderdef.tcl -
#
# Create and edit vfolders
#
# Each virtual folder is described by one entry in the vFolderDef array.
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# VFolderDef --
#
# Create the vfolderdef window
#
# Arguments:

proc VFolderDef {} {
    global t vFolderRowHeight vFolderFrameHeight vFolderChanged

    set vFolderChanged 0

    # Create identifier
    set id vfolderdef
    set w .$id
    if [winfo exists $w] {
	wm deiconify $w
	raise $w
	return
    }
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]


    # Create toplevel
    toplevel $w
    wm title $w $t(vfolderdef)

    # Populate window
    frame $w.mbar -relief raised -bd 1
    menubutton $w.mbar.w -menu $w.mbar.w.m -text $t(window)
    menu $w.mbar.w.m
    $w.mbar.w.m add command -label $t(close) \
	    -command "RecordPos $w vFolderDef; \
	    RecordSize $w.canvas vFolderDef; destroy $w; unset $id"
    menubutton $w.mbar.folder -menu $w.mbar.folder.m -text $t(folder)
    menu $w.mbar.folder.m
    $w.mbar.folder.m add command -label $t(create_submenu)... \
	    -command {set vFolderCurrent {0 end}; VFolderCreateMenu}
    $w.mbar.folder.m add command -label $t(new_file)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew file}
    $w.mbar.folder.m add command -label $t(new_mh)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew mh}
    $w.mbar.folder.m add command -label $t(new_dbase)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew dbase}
    $w.mbar.folder.m add command -label $t(new_imap)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew imap}
    $w.mbar.folder.m add command -label $t(new_pop3)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew pop3}
    $w.mbar.folder.m add command -label $t(new_dynamic)... \
	    -command {set vFolderCurrent {0 end}; VFolderNew dynamic}
    menubutton $w.mbar.import -menu $w.mbar.import.m -text $t(import)
    menu $w.mbar.import.m
    $w.mbar.import.m add command -label $t(directory)... \
	    -command VFolderImportDirectory
    $w.mbar.import.m add command -label $t(import_IMAP)... \
	    -command VFolderImportIMAP

    pack $w.mbar.w \
	 $w.mbar.folder \
	 $w.mbar.import -side left -padx 5
    scrollbar $w.scrolly -relief raised -command "$w.canvas yview" \
	    -highlightthickness 0 -bd 1
    scrollbar $w.scrollx -relief raised -command "$w.canvas xview" \
	    -highlightthickness 0 -orient horizontal -bd 1
    canvas $w.canvas -yscrollcommand "$w.scrolly set" \
	    -xscrollcommand "$w.scrollx set" -relief raised -bd 1 \
	    -highlightthickness 0 -selectborderwidth 0
    Size $w.canvas vFolderDef

    grid $w.mbar -sticky ew -columnspan 2
    grid $w.canvas -sticky nsew
    grid $w.scrolly -column 1 -row 1 -sticky ns
    grid $w.scrollx -sticky ew
    grid columnconfigure $w 0 -weight 1
    grid rowconfigure $w 1 -weight 1

    menu $w.m -tearoff 0
    $w.m add command -label $t(edit)... -command EditVFolderEntry
    $w.m add command -label $t(delete)... -command VFolderDelete
    $w.m add command -label $t(set_to_inbox) \
	    -command {VFolderSetToSpecial Inbox}
    $w.m add command -label $t(set_to_save_outgoing) \
	    -command {VFolderSetToSpecial Save}
    $w.m add separator
    $w.m add command -label $t(create_submenu)... -command VFolderCreateMenu
    $w.m add command -label $t(new_file)... -command {VFolderNew file}
    $w.m add command -label $t(new_mh)... -command {VFolderNew mh}
    $w.m add command -label $t(new_dbase)... -command {VFolderNew dbase}
    $w.m add command -label $t(new_imap)... -command {VFolderNew imap}
    $w.m add command -label $t(new_pop3)... -command {VFolderNew pop3}
    $w.m add command -label $t(new_dynamic)... -command {VFolderNew dynamic}

    # Find out how big the listboxes are
    listbox $w.canvas.l -height 1 -width 1 -highlightthickness 0
    set id [$w.canvas create window 0 0 -window $w.canvas.l \
	    -anchor nw]
    set y1 [lindex [$w.canvas bbox $id] 3]
    $w.canvas.l configure -height 2
    set y2 [lindex [$w.canvas bbox $id] 3]
    set vFolderRowHeight [expr $y2-$y1]
    set vFolderFrameHeight [expr ($y1-($y2-$y1))/2]

    bind $w.mbar <Destroy> VFolderWindowDestroy

    RedrawVFolderList
    Place $w vFolderDef
}

# RedrawVFolderList --
#
# Redraws the list of vfolders. This is done by first clearing the canvas
# and then call DrawVFolderStruct which does the actual drawing.
#
# Arguments:

proc RedrawVFolderList {} {
    global vFolderX vFolderY vFolderCI maxX maxY

    set c .vfolderdef.canvas
    foreach w [winfo children $c] {
	destroy $w
    }
    $c delete all
    set vFolderX 10
    set vFolderY 10
    set vFolderCI 0
    set maxX 0
    set maxY 0

    DrawVFolderStruct 0 0

    $c configure -scrollregion [list 0 0 [expr $maxX+10] [expr $maxY+10]]
}

# DrawVFolderStruct --
#
# Draws a given struct, and its children by calling itself recursively.
# It returns the Y-coordinate of its upper edge.
#
# Arguments:
# sIdent -	The ident of the struct to be drawn
# column -	The column the struct is to be drawn in

proc DrawVFolderStruct {sIdent column} {
    global vFolderX vFolderY option vFolderCI vFolderStruct vFolderSave \
	   vFolderRowHeight vFolderFrameHeight maxX maxY vFolderInbox 

    set c .vfolderdef.canvas
    set l $c.l$vFolderCI
    incr vFolderCI
    listbox $l -width $option(vfoldername_width) -relief groove \
	    -height [llength $vFolderStruct($sIdent)] -highlightthickness 0 \
	    -selectmode single
    bind $l <1> "VFolderPostMenu %X %Y %W $sIdent \[%W index @%x,%y\]"
    set x [lindex $vFolderX $column]
    set y [lindex $vFolderY $column]
    set id [$c create window $x $y -anchor nw -window $l -tags $sIdent]
    set bbox [$c bbox $id]
    if {[llength $vFolderX] == [expr $column+1]} {
	lappend vFolderX [expr [lindex $bbox 2]+40]
	lappend vFolderY [lindex $bbox 1]
    }
    set c1 [expr $column+1]
    if {[lindex $vFolderY $c1] < [lindex $bbox 1]} {
	set vFolderY [lreplace $vFolderY $c1 $c1 [lindex $bbox 1]]
    }
    set isFirst 1
    set x1 [lindex $bbox 2]
    set y1 [expr [lindex $bbox 1]+$vFolderFrameHeight+$vFolderRowHeight/2]
    set x2 [lindex $vFolderX $c1]
    foreach e $vFolderStruct($sIdent) {
	set prefix " "
	if ![string compare vfolder [lindex $e 0]] {
	    if ![string compare $vFolderInbox [lindex $e 1]] {
		set prefix "*"
	    }
	    if ![string compare $vFolderSave [lindex $e 1]] {
		set prefix ">"
	    }
	}
	$l insert end ${prefix}[lindex $e 2]
	if ![string compare [lindex $e 0] struct] {
	    set newy [DrawVFolderStruct [lindex $e 1] $c1]
	    if $isFirst {
		set isFirst 0
		if {$newy > [lindex $bbox 1]} {
		    incr y1 [expr $newy-[lindex $bbox 1]]
		    $c move $id 0 [expr $newy-[lindex $bbox 1]]
		}
	    }
	    set y2 [expr $newy+$vFolderFrameHeight+$vFolderRowHeight/2]
	    $c create line $x1 $y1 $x2 $y2
	}
	incr y1 $vFolderRowHeight
    }
    set bbox [$c bbox $id]
    if {[lindex $bbox 2] > $maxX} {
	set maxX [lindex $bbox 2]
    }
    if {[lindex $bbox 3] > $maxY} {
	set maxY [lindex $bbox 3]
    }
    bind $l <3> "VFolderStartDrag $c %x %y $id $sIdent \[%W index @%x,%y\]"
    bind $l <B3-Motion> "VFolderDrag $c %x %y"
    bind $l <B3-ButtonRelease> "VFolderStopDrag $c"
    set vFolderY [lreplace $vFolderY $column $column [expr [lindex $bbox 3]+10]]
    return [lindex $bbox 1]
}

# VFolderPostMenu:
# Post a the menu for a given entry. The arguments are: the root window
# coordinates, the listbox, the ident of the vFolderStruct and the index
# of the active member.
# VFolderPostMenu --
#
# Post the menu for the given entry.
#
# Arguments:
# x, y   -	Position of pointer, this is where we will popup the menu
# listW  -	Name of the list widget the entry belongs to
# sIdent -	Identifier of the vFolderStruct the entry belongs to
# index  -	Index in the above mentioned vFolderStruct

proc VFolderPostMenu {x y listW sIdent index} {
    global vFolderCurrent vFolderStruct vFolderInbox vFolderSave t

    set vFolderCurrent [list $sIdent $index]
    set elem [lindex $vFolderStruct($sIdent) $index]
    if ![llength $elem] {
	.vfolderdef.m entryconfigure 0 -state disabled
	.vfolderdef.m entryconfigure 1 -state disabled
	.vfolderdef.m entryconfigure 2 -state disabled
    } else {
	.vfolderdef.m entryconfigure 0 -state normal
	if [string compare $vFolderInbox [lindex $elem 1]] {
	    .vfolderdef.m entryconfigure 1 -state normal
	    .vfolderdef.m entryconfigure 2 -state normal
	} else {
	    .vfolderdef.m entryconfigure 1 -state disabled
	    .vfolderdef.m entryconfigure 2 -state disabled
	}
	if ![string compare vfolder [lindex $elem 0]] {
	    if [string compare $vFolderSave [lindex $elem 1]] {
		.vfolderdef.m entryconfigure 3 \
			-label $t(set_to_save_outgoing) -state normal
	    } else {
		.vfolderdef.m entryconfigure 3 \
			-label $t(unset_to_save_outgoing) -state normal
	    }
	} else {
	    .vfolderdef.m entryconfigure 3 \
		    -label $t(set_to_save_outgoing) -state disabled
	}
    }
    bind .vfolderdef.m <Unmap> "$listW selection clear $index"
    tk_popup .vfolderdef.m $x $y
}

# VFolderStartDrag --
#
# Invoked when the users starts to drag an item (or at least wants to).
#
# Arguments:
# c      -	Widget command name
# x, y   -	Place from which the user wants to pickup an object,
#		relative to the listbox nw corner.
# id	 -	The itemId of the listbox in the canvas
# sIdent -	Ident of the vfolderstruct which contains the object
# index  -	Index of object in struct

proc VFolderStartDrag {c x y id sIdent index} {
    global vFLastX vFLastY vFOffsetX vFOffsetY vFolderCurrent vFolderStruct
    set vFolderCurrent [list $sIdent $index]
    set vFLastX $x
    set vFLastY $y
    set name [lindex [lindex $vFolderStruct($sIdent) $index] 2]
    set bbox [$c bbox $id]
    set vFOffsetX [lindex $bbox 0]
    set vFOffsetY [lindex $bbox 1]
    label $c.l -text $name -relief ridge
    $c create window [expr $vFLastX+$vFOffsetX] [expr $vFLastY+$vFOffsetY] \
	    -window $c.l -tags drag -anchor se
}

# VFolderDrag --
#
# Does the acutal dragging.
#
# Arguments:
# c    -	Canvas widget command name
# x, y -	Place to drag the item to

proc VFolderDrag {c x y} {
    global vFLastX vFLastY
    $c move drag [expr $x-$vFLastX] [expr $y-$vFLastY]
    set vFLastX $x
    set vFLastY $y
}

# VFolderStopDrag --
#
# The user is tired of dragging this heavy object around and wants to drop
# it.
#
# Arguments:
# c    -	Canvas widget command name

proc VFolderStopDrag {c} {
    global vFLastX vFLastY vFolderRowHeight vFolderFrameHeight \
	   vFOffsetX vFOffsetY vFolderStruct vFolderCurrent vFolderChanged

    set x [expr $vFLastX+$vFOffsetX]
    set y [expr $vFLastY+$vFOffsetY]
    destroy $c.l
    $c delete drag
    set id [$c find overlapping $x $y $x $y]

    # See if we have found something
    if {1 != [llength $id]} {
	return
    }
    if {"window" != [$c type $id]} {
	return
    }
    set w [$c itemcget $id -window]
    set iy [expr $y-[lindex [$c bbox $id] 1]]
    set indexN [expr int(($iy-$vFolderFrameHeight/2) /$vFolderRowHeight)]
    set sIdentN [$c itemcget $id -tags]
    if { 1 < [llength $sIdentN] } {
	set sIdentN [lreplace $sIdentN \
		[lsearch $sIdentN current] [lsearch $sIdentN current]]
    }
    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set item [lindex $vFolderStruct($sIdent) $index]
    set vFolderStruct($sIdent) [lreplace $vFolderStruct($sIdent) $index $index]
    set vFolderStruct($sIdentN) [linsert $vFolderStruct($sIdentN) $indexN $item]
    incr vFolderChanged
    RedrawVFolderList
}

# VFolderNew --
# This procedure is invoked when the user wants to create a new vfolder.
# The struct it should be inserted into is specified via the vFolderCurrent
# global variable.
#
# Arguments:
# type -	The type of the new vfolder

proc VFolderNew {type} {
    global vFolderCurrent vFolderDefIdent vFolderStruct vFolderChanged \
	   vFolderDef option env

    switch $type {
    file      {set vFolderDef($vFolderDefIdent) {{} file {}}}
    mh        {set vFolderDef($vFolderDefIdent) {{} mh {}}}
    dbase     {set vFolderDef($vFolderDefIdent) \
	        [list {} dbase {} $option(def_extype) $option(def_exdate)]}
    imap      {set vFolderDef($vFolderDefIdent) \
	        "{} imap \\\\{:$option(imap_port)/imap\\\\} $env(USER)"}
    pop3      {set vFolderDef($vFolderDefIdent) {{} pop3 \\{/pop3\\} {}}}
    dynamic   {set vFolderDef($vFolderDefIdent) {{} dynamic {} sender}}
    }
    set vFolderName [EditVFolder $vFolderDefIdent]
    if ![string length $vFolderName] {
	unset vFolderDef($vFolderDefIdent)
	return
    }
    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index \
	    [list vfolder $vFolderDefIdent $vFolderName]]
    incr vFolderDefIdent
    RedrawVFolderList
}

# VFolderCreateMenu --
# Asks the user for a name and then creates a new empty submenu with
# that name. The new submenu is inserted into the structure at the place
# indicated by the vFolderCurrent global variable.
#
# Arguments:

proc VFolderCreateMenu {} {
    global vFolderCurrent t idCnt

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm title $w $t(create_submenu)

    # Folder name
    frame $w.name
    label $w.name.label -text $t(name):
    entry $w.name.entry -textvariable ${id}(name) -width 20
    pack $w.name.entry \
	 $w.name.label -side right

    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.name \
	 $w.buttons -side top -fill both -padx 5 -pady 5

    Place $w vFolderCreateMenu
    focus $w.name.entry
    grab $w
    tkwait variable ${id}(done)
    RecordPos $w vFolderCreateMenu
    destroy $w
    focus $oldFocus

    if {1 == $hd(done)} {
	global vFolderStruct vFolderStructIdent vFolderChanged
	incr vFolderChanged
	incr vFolderStructIdent
	set sIdent [lindex $vFolderCurrent 0]
	set index [lindex $vFolderCurrent 1]
	set vFolderStruct($sIdent) [linsert $vFolderStruct($sIdent) $index \
		[list struct $vFolderStructIdent $hd(name)]]
	set vFolderStruct($vFolderStructIdent) {}
	RedrawVFolderList
    }
    unset hd
}

# EditVFolderEntry --
#
# Edit an entry in the vfolder menu
#
# Arguments:

proc EditVFolderEntry {} {
    global t idCnt vFolderCurrent vFolderStruct vFolderChanged vFolderDef

    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set elem [lindex $vFolderStruct($sIdent) $index]
    switch [lindex $elem 0] {
    struct  {
	    set ident [lindex $elem 1]
	    set id cf[incr idCnt]
	    set w .$id
	    upvar #0 $id hd
	    set hd(done) 0
	    set oldFocus [focus]

	    # Create toplevel
	    toplevel $w
	    wm title $w $t(name)

	    # Folder name
	    set hd(name) [lindex $elem 2]
	    frame $w.name
	    label $w.name.label -text $t(name): -width 15 -anchor e
	    entry $w.name.entry -textvariable ${id}(name) -width 20
	    pack $w.name.label $w.name.entry -side left
	    pack $w.name -side top -expand 1 -padx 5 -pady 5
	    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
	    pack $w.buttons -fill both -padx 5 -pady 5

	    Place $w editVfolderEntry
	    focus $w.name.entry
	    grab $w
	    while 1 {
		tkwait variable ${id}(done)
		if { 1 == $hd(done) } {
		    if ![string length $hd(name)] {
			Popup $t(need_name)
			continue
		    }
		    incr vFolderChanged
		    set vFolderStruct($sIdent) \
			    [lreplace $vFolderStruct($sIdent) $index $index \
			     [list struct $ident $hd(name)]]
		}
		break
	    }
	    RecordPos $w editVfolderEntry
	    destroy $w
	    focus $oldFocus
	    unset hd
	}
    vfolder {
	    EditVFolder [lindex $elem 1]
	    set vFolderStruct($sIdent) \
		    [lreplace $vFolderStruct($sIdent) $index $index \
		      	  [list vfolder \
				[lindex $elem 1] \
				[lindex $vFolderDef([lindex $elem 1]) 0]]]
	}
    }
    RedrawVFolderList
}

# VFolderDelete --
#
# This procedure asks the user if it is really sure it wants to delete
# the specified item. The item is specified via the vFolderCurrent
# global variable.
#
# Arguments:

proc VFolderDelete {} {
    global t vFolderCurrent vFolderDef vFolderStruct vFolderChanged \
	   vFolderInbox vFolderSave

    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]

    set elem [lindex $vFolderStruct($sIdent) $index]
    if ![string compare [lindex $elem 0] struct] {
	if [llength $vFolderStruct([lindex $elem 1])] {
	    if [RatDialog ! $t(submenu_not_empty) {} 0 $t(delete) \
		    $t(cancel)] {
		return
	    }
	    VFolderDeleteStruct -1 [lindex $elem 1]
	}
	unset vFolderStruct([lindex $elem 1])
    } else {
	set vd $vFolderDef([lindex $elem 1])
	if ![string compare imap [lindex $vd 1]] {
	    set ret [RatDialog ! $t(delete_imap) {} 0 $t(delete) \
		    $t(dont_delete) $t(cancel)]
	    if {2 == $ret} {
		return
	    } 
	    if {0 == $ret} {
		catch {RatImapDeleteFolder [lindex [lindex $vd 2] 0] \
			[lindex $vd 3]}
	    }
	}
	unset vFolderDef([lindex $elem 1])
    }
    set vFolderStruct($sIdent) [lreplace $vFolderStruct($sIdent) $index $index]
    if ![info exists vFolderDef($vFolderInbox)] {
	set vFolderInbox [lindex [lsort -integer [array names vFolderDef]] 0]
    }
    if ![info exists vFolderDef($vFolderInbox)] {
	set vFolderSave {}
    }
    incr vFolderChanged
    RedrawVFolderList
}

# VFolderSetToSpecial --
#
# This procedure marks the selected folder as a special mailbox.
#
# Arguments:
# what - What kind of of special feature this mailbox have

proc VFolderSetToSpecial {what} {
    global t vFolderCurrent vFolderStruct vFolderChanged
    upvar #0 vFolder$what special

    set sIdent [lindex $vFolderCurrent 0]
    set index [lindex $vFolderCurrent 1]
    set elem [lindex $vFolderStruct($sIdent) $index]
    if {![string compare Save $what] &&
	    ![string compare $special [lindex $elem 1]]} {
	set special {}
    } else {
	set special [lindex $elem 1]
    }
    incr vFolderChanged
    RedrawVFolderList
}

# VFolderDeleteStruct --
#
# Remove a struct and all contained folders (as well ass all children
#
# Arguments:
# op	 - Operation to perform on IMAP-folders (-1=dont_know 0=delete 1=keep)
# sIdent - id of struct

proc VFolderDeleteStruct {op sIdent} {
    global vFolderStruct vFolderDef t

    foreach elem $vFolderStruct($sIdent) {
	if ![string compare [lindex $elem 0] struct] {
	    VFolderDeleteStruct $op [lindex $elem 1]
	    unset vFolderStruct([lindex $elem 1])
	} else {
	    set vd $vFolderDef([lindex $elem 1])
	    if {![string compare imap [lindex $vd 1]] && 1 != $op} {
		if { -1 == $op} {
		    set op [RatDialog ! $t(delete_imap) {} 0 \
			    $t(delete) \
		    	    $t(dont_delete)]
		}
		if {0 == $op} {
		    catch {RatImapDeleteFolder [lindex [lindex $vd 2] 0] \
			    [lindex $vd 3]}
		}
	    }
	    unset vFolderDef([lindex $elem 1])
	}
    }
}

# EditVFolder --
#
# Lets the user edit a vfolder.
#
# Arguments:
# ident -	The identity in the vFolderDef array of the vfolder

proc EditVFolder {ident} {
    global t idCnt vFolderChanged vFolderDef
    upvar #0 vFolderDef($ident) def

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set oldFocus [focus]

    # Create toplevel
    toplevel $w
    wm title $w "$t(vfolderedit)  ([lindex $def 1])"

    # Folder name
    set hd(name) [lindex $def 0]
    frame $w.name
    label $w.name.label -text $t(folder_name): -width 15 -anchor e
    entry $w.name.entry -textvariable ${id}(name) -width 20
    pack $w.name.label $w.name.entry -side left
    pack $w.name -side top -expand 1 -padx 5 -pady 5
    bind $w <Return> {}

    # Insert type specific widgets
    switch [lindex $def 1] {
    file {
	    set hd(fname) [lindex $def 2]
	    frame $w.chooser -relief sunken -bd 1
	    set fh [FileSelectorCreate $w.chooser ${id}(fname) \
		    "set ${id}(done) 1" 0 0]
	    pack $w.chooser -side top -fill both -expand 1 -padx 5 -pady 5
	}
    mh {
	    set hd(mh_name) [string range [lindex $def 2] 4 end]
	    frame $w.mhn
	    label $w.mhn.label -text $t(mh_name):
	    entry $w.mhn.entry -textvariable ${id}(mh_name) -width 20
	    frame $w.chooser -relief sunken -bd 1
	    pack $w.mhn.label \
		 $w.mhn.entry -side left -pady 5
	    pack $w.mhn -side top
	}
    dbase {
	    set hd(keywords) [lindex $def 2]
	    set hd(extype) [lindex $def 3]
	    set hd(exdate) [lindex $def 4]
	    frame $w.keywords
	    label $w.keywords.label -text $t(keywords):
	    entry $w.keywords.entry -textvariable ${id}(keywords) -width 20
	    pack $w.keywords.entry \
		 $w.keywords.label -side right
	    frame $w.extype
	    label $w.extype.label -text $t(extype):
	    frame $w.extype.b
	    radiobutton $w.extype.b.none -text $t(none) \
		    -variable ${id}(extype) -value none
	    radiobutton $w.extype.b.remove -text $t(remove) \
		    -variable ${id}(extype) -value remove
	    radiobutton $w.extype.b.incoming -text $t(add_incoming) \
		    -variable ${id}(extype) -value incoming
	    radiobutton $w.extype.b.backup -text $t(backup) \
		    -variable ${id}(extype) -value backup
	    radiobutton $w.extype.b.custom -variable ${id}(extype) \
		    -value custom -state disabled
	    pack $w.extype.b.none \
		 $w.extype.b.remove \
		 $w.extype.b.incoming \
		 $w.extype.b.backup \
		 $w.extype.b.custom -side top -anchor w
	    pack $w.extype.b \
		 $w.extype.label -side right -anchor nw
	    frame $w.exdate
	    label $w.exdate.label -text $t(exdate):
	    entry $w.exdate.entry -textvariable ${id}(exdate) -width 20
	    pack $w.exdate.entry \
		 $w.exdate.label -side right
	    pack $w.keywords \
		 $w.extype \
		 $w.exdate -side top -fill both
	}
    imap {
	    set hd(user) [lindex $def 3]
	    set h [lindex [lindex $def 2] 0]
	    regsub {(/|\}).+} [string trim $h \{] {} host
	    set hlist [split $host :]
	    set hd(host) [lindex $hlist 0]
	    if {2 == [llength $hlist]} {
		set hd(port) [lindex $hlist 1]
	    }
	    regsub {.+\}} $h {} hd(mbox)
	    set oldHost $hd(host)
	    set oldMbox $hd(mbox)
	    frame $w.host
	    label $w.host.label -text $t(host): -width 17 -anchor e
	    entry $w.host.entry -textvariable ${id}(host) -width 30
	    pack $w.host.label $w.host.entry -side left
	    frame $w.user
	    label $w.user.label -text $t(user): -width 17 -anchor e
	    entry $w.user.entry -textvariable ${id}(user) -width 30
	    pack $w.user.label $w.user.entry -side left
	    frame $w.mbox
	    label $w.mbox.label -text $t(mbox): -width 17 -anchor e
	    entry $w.mbox.entry -textvariable ${id}(mbox) -width 30
	    pack $w.mbox.label $w.mbox.entry -side left
	    frame $w.port
	    label $w.port.label -text $t(port): -width 17 -anchor e
	    entry $w.port.entry -textvariable ${id}(port) -width 30
	    pack $w.port.label $w.port.entry -side left
	    pack $w.host \
		 $w.user \
		 $w.mbox \
		 $w.port -side top -fill both
	}
    pop3 {
	    set hd(user) [lindex $def 3]
	    set h [lindex [lindex $def 2] 0]
	    regsub {(/|\}).+} [string trim $h \{] {} host
	    set hlist [split $host :]
	    set hd(host) [lindex $hlist 0]
	    if {2 == [llength $hlist]} {
		set hd(port) [lindex $hlist 1]
	    }
	    frame $w.host
	    label $w.host.label -text $t(host): -width 17 -anchor e
	    entry $w.host.entry -textvariable ${id}(host) -width 30
	    pack $w.host.label $w.host.entry -side left
	    frame $w.user
	    label $w.user.label -text $t(user): -width 17 -anchor e
	    entry $w.user.entry -textvariable ${id}(user) -width 30
	    pack $w.user.label $w.user.entry -side left
	    frame $w.port
	    label $w.port.label -text $t(port): -width 17 -anchor e
	    entry $w.port.entry -textvariable ${id}(port) -width 30
	    pack $w.port.label $w.port.entry -side left
	    pack $w.host \
		 $w.user \
		 $w.port -side top -fill both
	}
    dynamic {
	    set hd(fname) [lindex $def 2]
	    frame $w.chooser -relief sunken -bd 1
	    set fh [FileSelectorCreate $w.chooser ${id}(fname) \
		    "set ${id}(done) 1" 1 1]
	    pack $w.chooser -side top -fill both -expand 1 -padx 5 -pady 5
	}
    }
    
    # OK and cancel buttons
    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.buttons -side bottom -fill both -padx 5 -pady 5

    Place $w editVfolder
    focus $w.name.entry
    grab $w
    while 1 {
	tkwait variable ${id}(done)

	if { 1 == $hd(done) } {
	    if ![string length $hd(name)] {
		Popup $t(need_name)
		continue
	    }
	    switch [lindex $def 1] {
	    file {
		    FileSelectorCleanup $fh
		    if {![file isfile $hd(fname)] && ([file exists $hd(fname)]
			    || ![file isdirectory [file dirname $hd(fname)]])} {
			Popup "$t(illegal_file_spec): $hd(fname)"
			continue
		    }
		    FileSelectorDone $fh
		    set vFolderDef($ident) [list $hd(name) file $hd(fname)]
		}
	    mh {
		    if {![string length $hd(mh_name)]} {
		        Popup $t(need_mh_name)
			continue
		    }
		    set vFolderDef($ident) [list $hd(name) mh \
			    "#MH/$hd(mh_name)"]
		}
	    dbase {
		    if ![string length $hd(keywords)] {
		        Popup $t(need_keyword)
			continue
		    }
		    set vFolderDef($ident) [list $hd(name) dbase \
			    $hd(keywords) $hd(extype) $hd(exdate)]
		}
	    imap {
		    if {![string length $hd(host)]
			    || ![string length $hd(user)]} {
		        Popup $t(need_host_and_user)
			continue
		    }
		    if [string length $hd(port)] {
			set host $hd(host):$hd(port)
		    } else {
			set host $hd(host)
		    }
		    set vFolderDef($ident) [list $hd(name) imap \
			    \{{$host}$hd(mbox)\} $hd(user)]
		    if {[string compare $oldHost $hd(host)]
			    || [string compare $oldMbox $hd(mbox)]} {
			catch {RatImapCreateFolder "{$host}$hd(mbox)" $hd(user)}
		    }
		}
	    pop3 {
		    if {![string length $hd(host)]
			    || ![string length $hd(user)]} {
		        Popup $t(need_host_and_user)
			continue
		    }
		    if [string length $hd(port)] {
			set host $hd(host):$hd(port)
		    } else {
			set host $hd(host)
		    }
		    set vFolderDef($ident) [list $hd(name) pop3 \
			    \{{$host/pop3}\} $hd(user)]
		}
	    dynamic {
		    FileSelectorCleanup $fh
		    if {![file isdirectory $hd(fname)]} {
			Popup "$t(illegal_file_spec): $hd(fname)"
			continue
		    }
		    FileSelectorDone $fh
		    set vFolderDef($ident) \
			    [list $hd(name) dynamic $hd(fname) sender]
		}
	    }
	    incr vFolderChanged
	    RecordPos $w editVfolder
	    destroy $w
	    focus $oldFocus
	    set name $hd(name)
	    unset hd
	    return $name
	} else {
	    RecordPos $w editVfolder
	    destroy $w
	    focus $oldFocus
	    unset hd
	    return ""
	}
    }
}

# VFolderWrite --
#
# Writes the list of vfolders to disk
#
# Arguments:

proc VFolderWrite {} {
    global option vFolderDef vFolderStruct vFolderDefIdent vFolderStructIdent \
	   vFolderVersion vFolderInbox vFolderSave

    set fh [open $option(ratatosk_dir)/vfolderlist w]
    puts $fh "set vFolderVersion $vFolderVersion"
    puts $fh "set vFolderStructIdent $vFolderStructIdent"
    puts $fh "set vFolderDefIdent $vFolderDefIdent"
    puts $fh "set vFolderInbox $vFolderInbox"
    puts $fh "set vFolderSave [list $vFolderSave]"
    foreach str [array names vFolderStruct] {
	set vs {}
	foreach e $vFolderStruct($str) {
	    switch [lindex $e 0] {
	    struct  {
		    set keep [info exists vFolderStruct([lindex $e 1])]
		}
	    vfolder {
		    set keep [info exists vFolderDef([lindex $e 1])]
		}
	    }
	    if $keep {
		lappend vs $e
	    } else {
		puts "Warning, removing {$e}"
	    }
	}
	puts $fh "set vFolderStruct($str) [list $vs]"
    }
    foreach elem [array names vFolderDef] {
	puts $fh "set vFolderDef($elem) [list $vFolderDef($elem)]"
    }
    close $fh
}

# VFolderWindowDestroy --
#
# This procedure is called when the vfolder window is destroyed. It
# writes any changes to the vfolder structre to disk, if needed.
#
# Arguments:

proc VFolderWindowDestroy {} {
    global vFolderChanged
    if $vFolderChanged {
	VFolderWrite
    }
}

# VFolderImportDirectory --
#
# Import a folder structure from disk (i.e. the folders are stored in
# directories on the disk.
#
# Arguments:

proc VFolderImportDirectory {} {
    global vFolderDefIdent vFolderStruct vFolderChanged vFolderStructIdent\
	   vFolderDef t idCnt

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0
    set hd(file_pattern) {*}
    set hd(dir_pattern) {*}

    # Create toplevel
    toplevel $w
    wm title $w $t(import_directory)

    # Create fileselector
    set hd(fname) {}
    frame $w.chooser -relief sunken -bd 1
    set fh [FileSelectorCreate $w.chooser ${id}(fname) "set ${id}(done) 1" 1 1]
    pack $w.chooser -side top -fill both -expand 1 -padx 5 -pady 5

    # The patterns
    frame $w.pat
    label $w.pat.lab -text $t(patterns_to_use)
    label $w.pat.labf -text $t(files):
    entry $w.pat.entf -textvariable ${id}(file_pattern)
    label $w.pat.labd -text $t(directories):
    entry $w.pat.entd -textvariable ${id}(dir_pattern)
    grid $w.pat.lab -columnspan 2
    grid $w.pat.labf -row 1 -column 0 -sticky e
    grid $w.pat.entf -row 1 -column 1 -sticky ew
    grid $w.pat.labd -row 2 -column 0 -sticky e
    grid $w.pat.entd -row 2 -column 1 -sticky ew
    pack $w.pat

    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    pack $w.buttons \
	 $w.chooser -side bottom -fill both -padx 5 -pady 5

    Place $w vFolderImportDirectory
    grab $w
    tkwait variable ${id}(done)
    RecordPos $w vFolderImportDirectory
    destroy $w

    if {1 == $hd(done)} {
	FileSelectorDone $fh
	incr vFolderChanged
	set sid [incr vFolderStructIdent]

	set name [file tail [string trimright $hd(fname) /]]
	if [file isdirectory $hd(fname)] {
	    lappend vFolderStruct(0) [list struct $sid $name]
	    VFolderScanDir $sid $hd(fname) $hd(file_pattern) $hd(dir_pattern)
	} else {
	    lappend vFolderStruct(0) [list vfolder $vFolderDefIdent $name]
	    set vFolderDef($vFolderDefIdent) [list $name file $hd(fname)]
	    incr vFolderDefIdent
	}
	RedrawVFolderList
    }
    unset hd
}

# VFolderScanDir --
#
# Scan a directory and build a structure from it.
#
# Arguments:
# sid	       - Structure id for our struct entry
# dirName      - Name of the directory
# file_pattern - pattern to match files against
# dir_pattern  - pattern to match directories against

proc VFolderScanDir {sid dirName file_pattern dir_pattern} {
    global vFolderDefIdent vFolderStruct vFolderDef vFolderStructIdent t 

    set vFolderStruct($sid) {}
    foreach f [lsort [glob -nocomplain $dirName/*]] {
	set name [file tail $f]
	if ![file readable $f] {
	    Popup "$t(cant_read): $f"
	    continue
	}
	if [file isdirectory $f] {
	    if ![string match $dir_pattern $name] {
		continue
	    }
	    lappend vFolderStruct($sid) \
		    [list struct [incr vFolderStructIdent] $name]
	    VFolderScanDir $vFolderStructIdent $f $file_pattern $dir_pattern
	} elseif [file isfile $f] {
	    if ![string match $file_pattern $name] {
		continue
	    }
	    lappend vFolderStruct($sid) [list vfolder $vFolderDefIdent $name]
	    set vFolderDef($vFolderDefIdent) [list $name file $f]
	    incr vFolderDefIdent
	} else {
	    Popup "$f $t(ignored_not_file_not_dir)"
	}
    }
}

# VFolderImportIMAP --
#
# Import a folder structure from an IMAP server.
#
# Arguments:

proc VFolderImportIMAP {} {
    global t env idCnt vFolderChanged option \
	   vFolderStruct vFolderStructIdent

    # Create identifier
    set id cf[incr idCnt]
    set w .$id
    upvar #0 $id hd
    set hd(done) 0

    # Create toplevel
    toplevel $w
    wm title $w $t(import_IMAP)

    # Create the entries
    set hd(host) {}
    set hd(port) $option(imap_port)
    set hd(user) $env(USER)
    set hd(pattern) {*}
    set row 0
    label $w.host_label -text $t(host):
    entry $w.host_entry -textvariable ${id}(host) -width 32
    grid $w.host_label -row $row -sticky e -padx 5 -pady 5
    grid $w.host_entry -row $row -column 1 -sticky ew -padx 5
    incr row
    label $w.port_label -text $t(port):
    entry $w.port_entry -textvariable ${id}(port) -width 32
    grid $w.port_label -row $row -sticky e -padx 5 -pady 5
    grid $w.port_entry -row $row -column 1 -sticky ew -padx 5
    incr row
    label $w.user_label -text $t(user):
    entry $w.user_entry -textvariable ${id}(user) -width 32
    grid $w.user_label -row $row -sticky e -padx 5 -pady 5
    grid $w.user_entry -row $row -column 1 -sticky ew -padx 5
    incr row
    label $w.pattern_label -text $t(pattern):
    entry $w.pattern_entry -textvariable ${id}(pattern) -width 32
    grid $w.pattern_label -row $row -sticky e -padx 5 -pady 5
    grid $w.pattern_entry -row $row -column 1 -sticky ew -padx 5
    incr row

    OkButtons $w $t(ok) $t(cancel) "set ${id}(done)"
    grid $w.buttons -row $row -columnspan 2 -pady 10 -sticky ew

    Place $w vFolderImportIMAP
    focus $w.host_entry
    grab $w
    while 1 {
	tkwait variable ${id}(done)
	if {(1 == $hd(done)) && (0 == [string length $hd(host)])} {
	    Popup $t(host_required)
	} else {
	    break
	}
    }
    RecordPos $w vFolderImportIMAP
    destroy $w

    if {1 == $hd(done)} {
	if [string length $hd(port)] {
	    set cmd {RatListIMAP $hd(host):$hd(port) $hd(user) $hd(pattern)}
	} else {
	    set cmd {RatListIMAP $hd(host) $hd(user) $hd(pattern)}
	}
	if [catch $cmd folders] {
	    Popup "$t(import_failed): $folders"
	} else {
	    incr vFolderChanged
	    incr vFolderStructIdent
	    lappend vFolderStruct(0) [list struct $vFolderStructIdent \
		    $hd(user)@$hd(host)]
	    VFolderBuildIMAPStruct $folders $vFolderStructIdent $hd(user)
	    RedrawVFolderList
	}
    }
    unset hd
}

# VFolderBuildIMAPStruct --
#
# Recursively build the folder structre from the data returned by RatListIMAP.
#
# Arguments:
# flist  - List of folders on this level
# sIdent - The ident of the struct to build
# user   - The username

proc VFolderBuildIMAPStruct {flist sIdent user} {
    global vFolderDefIdent vFolderStruct vFolderStructIdent vFolderDef

    set vFolderStruct($sIdent) {}
    foreach f $flist {
	if [lindex $f 1] {
	    lappend vFolderStruct($sIdent) [list vfolder $vFolderDefIdent \
		    [lindex $f 0]]
	    set vFolderDef($vFolderDefIdent) [list [lindex $f 0] imap \
		    [lindex $f 2] $user]
	    incr vFolderDefIdent
	} else {
	    incr vFolderStructIdent
	    lappend vFolderStruct($sIdent) [list struct $vFolderStructIdent \
		    [lindex $f 0]]
	    VFolderBuildIMAPStruct [lindex $f 2] $vFolderStructIdent $user
	}
    }
}
