/*
 * ratAddress.c --
 *
 *	This file contains basic support for handling addresses.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "rat.h"

/*
 * This struct defines an alias.
 */
typedef struct {
    char *fullname;	/* Long name of alias (phrase part) */
    char *content;	/* Content that alias expands to */
} AliasInfo;

/*
 * This table contains all the aliases.
 */
Tcl_HashTable aliasTable;

/*
 * This table contains all the users
 */
Tcl_HashTable userTable;

/*
 * The number of address entities created. This is used to create new
 * unique command names.
 */
static int numAddresses = 0;


/*
 *----------------------------------------------------------------------
 *
 * RatInitAddresses --
 *
 *      This routine takes an address list as argument and constructs a
 *	list of address entities of it.
 *
 * Results:
 *	A list of address entity names is appended to interp->result.
 *
 * Side effects:
 *	New address entities are created,
 *
 *
 *----------------------------------------------------------------------
 */

void
RatInitAddresses(Tcl_Interp *interp, ADDRESS *addressPtr)
{
    ADDRESS *adrPtr, *newPtr;
    Tcl_DString result;
    char name[32], *s;

    Tcl_DStringInit(&result);
    Tcl_DStringGetResult(interp, &result);
    for (adrPtr = addressPtr; adrPtr; adrPtr = adrPtr->next) {
	newPtr = mail_newaddr();
	if (adrPtr->personal)	{
	    s = RatDecodeHeader(interp, adrPtr->personal);
	    newPtr->personal = cpystr(RatDecodeHeaderFull(interp, s));
	}
	if (adrPtr->adl)	newPtr->adl = cpystr(adrPtr->adl);
	if (adrPtr->mailbox)	newPtr->mailbox = cpystr(adrPtr->mailbox);
	if (adrPtr->host)	newPtr->host = cpystr(adrPtr->host);
	if (adrPtr->error)	newPtr->error = cpystr(adrPtr->error);
	sprintf(name, "RatAddress%d", numAddresses++);
	Tcl_CreateCommand(interp, name, RatAddress, (ClientData) newPtr,
		RatDeleteAddress);
	Tcl_DStringAppendElement(&result, name);
    }
    Tcl_DStringResult(interp, &result);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddress --
 *
 *      This routine handles the address entity commands. See ../doc/interface
 *	for a documentation of them.
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *	May be some
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddress(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    ADDRESS *adrPtr = (ADDRESS*)clientData;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(argv[1], "isMe")) {
	if (RatAddressIsMe(interp, adrPtr, 1)) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	}
	return TCL_OK;
	
    } else if (!strcmp(argv[1], "compare")) {
	Tcl_CmdInfo info;
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " compare address\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (0 == Tcl_GetCommandInfo(interp, argv[2], &info)) {
	    Tcl_AppendResult(interp, "there is no address entity \"",
		    argv[2], "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (RatAddressCompare(adrPtr, (ADDRESS*)info.clientData)) {
	    Tcl_SetResult(interp, "1", TCL_STATIC);
	} else {
	    Tcl_SetResult(interp, "0", TCL_STATIC);
	}
	return TCL_OK;

    } else if (!strcmp(argv[1], "set")) {
	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " set personal name host\"", (char *) NULL);
	    return TCL_ERROR;
	}
	free(adrPtr->mailbox);
	if (adrPtr->personal) {
	    free(adrPtr->personal);
	}
	if (adrPtr->host) {
	    free(adrPtr->host);
	}
	adrPtr->personal = cpystr(argv[2]);
	adrPtr->mailbox = cpystr(argv[3]);
	adrPtr->host = cpystr(argv[4]);
	return TCL_OK;

    } else if (!strcmp(argv[1], "get")) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get form\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (!strcasecmp(argv[2], "rfc822")) {
	    if (adrPtr->personal) {
		char *charset = Tcl_GetVar2(interp, "option", "charset",
			TCL_GLOBAL_ONLY);
		char *personal = (char*)RatEncodeHeaderLine(
	    		(unsigned char*)adrPtr->personal, charset, 0, 1);
		Tcl_ResetResult(interp);
		sprintf(interp->result, "%s <%s>", personal,
			RatAddressMail(adrPtr));
		ckfree(personal);
	    } else {
		Tcl_SetResult(interp, RatAddressMail(adrPtr), TCL_VOLATILE);
	    }
	    return TCL_OK;

	} else if (!strcasecmp(argv[2], "mail")) {
	    Tcl_SetResult(interp, RatAddressMail(adrPtr), TCL_VOLATILE);
	    return TCL_OK;

	} else if (!strcasecmp(argv[2], "name")) {
	    if (adrPtr->personal) {
		Tcl_SetResult(interp, adrPtr->personal, TCL_VOLATILE);
	    }
	    return TCL_OK;

	} else {
	    Tcl_AppendResult(interp, "bad form \"", argv[2],
		    "\": must be one of rfc822, mail or name", (char *) NULL);
	    return TCL_ERROR;
	}
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of isMe, compare, set or get", (char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatDeleteAddress --
 *
 *      Frees the client data of an address entity.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDeleteAddress(ClientData clientData)
{
    ADDRESS *adrPtr = (ADDRESS*)clientData;
    if (adrPtr->personal)	free(adrPtr->personal);
    if (adrPtr->adl)		free(adrPtr->adl);
    if (adrPtr->mailbox)	free(adrPtr->mailbox);
    if (adrPtr->host)		free(adrPtr->host);
    if (adrPtr->error)		free(adrPtr->error);
    ckfree(adrPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressIsMe --
 *
 *      Checks if the address points to me.
 *
 * Results:
 *	If it is then non zero is returned otherwise zero.
 *
 * Side effects:
 *	May initialize the current* variables
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddressIsMe(Tcl_Interp *interp, ADDRESS *adrPtr, int useUP)
{
    if (adrPtr->mailbox && !strcasecmp(adrPtr->mailbox, currentMailboxName)
	    && (adrPtr->host && !strcasecmp(adrPtr->host, currentHost))) {
	return 1;
    } else if (useUP) {
	Tcl_CmdInfo cmdInfo;
	if (Tcl_GetCommandInfo(interp, "RatUP_IsMe", &cmdInfo)) {
	    Tcl_DString cmd;
	    int isMe;
	    Tcl_DStringInit(&cmd);
	    Tcl_DStringAppendElement(&cmd, "RatUP_IsMe");
	    Tcl_DStringAppendElement(&cmd,adrPtr->mailbox?adrPtr->mailbox:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->host?adrPtr->host:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->personal?adrPtr->personal:"");
	    Tcl_DStringAppendElement(&cmd,adrPtr->adl?adrPtr->adl:"");
	    Tcl_Eval(interp, Tcl_DStringValue(&cmd));
	    Tcl_GetBoolean(interp, interp->result, &isMe);
	    Tcl_DStringFree(&cmd);
	    return isMe;
	}
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressCompare --
 *
 *      Check if two addresses are equal.
 *
 * Results:
 *	If they are then zero is returned otherwise non zero.
 *
 * Side effects:
 *	May initialize the current* variables
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAddressCompare(ADDRESS *adr1Ptr, ADDRESS *adr2Ptr)
{
    if (((adr1Ptr->mailbox && adr2Ptr->mailbox
		&& !strcasecmp(adr1Ptr->mailbox, adr2Ptr->mailbox))
  	      || adr1Ptr->mailbox == adr2Ptr->mailbox)
   	    && ((adr1Ptr->host && adr2Ptr->host
		&& !strcasecmp(adr1Ptr->host, adr2Ptr->host))
	      || adr1Ptr->host == adr2Ptr->host)) {
	return 0;
    } else {
	return 1;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressTranslate --
 *
 *      Let the user do their translation of this address.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The address may be affected.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatAddressTranslate(Tcl_Interp *interp, ADDRESS *adrPtr)
{
    Tcl_CmdInfo cmdInfo;
    Tcl_DString cmd;
    char **argv;
    int argc;

    if (!Tcl_GetCommandInfo(interp, "RatUP_Translate", &cmdInfo)) {
	return;
    }
    Tcl_DStringInit(&cmd);
    Tcl_DStringAppendElement(&cmd, "RatUP_Translate");
    Tcl_DStringAppendElement(&cmd,adrPtr->mailbox?adrPtr->mailbox:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->host?adrPtr->host:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->personal?adrPtr->personal:"");
    Tcl_DStringAppendElement(&cmd,adrPtr->adl?adrPtr->adl:"");
    Tcl_Eval(interp, Tcl_DStringValue(&cmd));
    if (   TCL_OK != Tcl_SplitList(interp, interp->result, &argc, &argv)
	|| 4 != argc) {
	char buf[2048];
	sprintf(buf, "%s: %s", Tcl_GetVar2(interp, "t", "translate_error",
		TCL_GLOBAL_ONLY), Tcl_DStringValue(&cmd));
	RatLog(interp, RAT_ERROR, buf, 0);
    } else {
	if (   (*argv[0]&&(!adrPtr->mailbox||strcmp(argv[0],adrPtr->mailbox)))
	    || (!*argv[0] && adrPtr->mailbox)) {
	    if (adrPtr->mailbox) {
		free(adrPtr->mailbox);
	    }
	    if (*argv[0]) {
		adrPtr->mailbox = cpystr(argv[0]);
	    } else {
		adrPtr->mailbox = NULL;
	    }
	}
	if (   (*argv[1]&&(!adrPtr->host||strcmp(argv[1],adrPtr->host)))
	    || (!*argv[1] && adrPtr->host)) {
	    if (adrPtr->host) {
		free(adrPtr->host);
	    }
	    if (*argv[1]) {
		adrPtr->host = cpystr(argv[1]);
	    } else {
		adrPtr->host = cpystr(currentHost);
	    }
	}
	if (   (*argv[2]&&(!adrPtr->personal||strcmp(argv[2],adrPtr->personal)))
	    || (!*argv[2] && adrPtr->personal)) {
	    if (adrPtr->personal) {
		free(adrPtr->personal);
	    }
	    if (*argv[2]) {
		adrPtr->personal = cpystr(argv[2]);
	    } else {
		adrPtr->personal = NULL;
	    }
	}
	if (   (*argv[3]&&(!adrPtr->adl||strcmp(argv[3],adrPtr->adl)))
	    || (!*argv[3] && adrPtr->adl)) {
	    if (adrPtr->adl) {
		free(adrPtr->adl);
	    }
	    if (*argv[3]) {
		adrPtr->adl = cpystr(argv[3]);
	    } else {
		adrPtr->adl = NULL;
	    }
	}
    }
    Tcl_DStringFree(&cmd);
}

/*
 *----------------------------------------------------------------------
 *
 * RatAlias --
 *
 *      Implements the RatAlias command as per ../doc/interface
 *
 * Results:
 *	Probably.
 *
 * Side effects:
 *	Probably.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatAlias(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(argv[1], "add")) {
	AliasInfo *aliasPtr;
	Tcl_HashEntry *entryPtr;
	char *key;
	int new;

	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " add name fullname content\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (!strlen(argv[2])) {
	    Tcl_SetResult(interp, "The name can not be an empty string",
		    TCL_STATIC);
	    return TCL_OK;
	}
	key = cpystr(argv[2]);
	lcase(key);
	aliasPtr = (AliasInfo*)ckalloc(sizeof(AliasInfo)+2+strlen(argv[3])+
		strlen(argv[4]));
	aliasPtr->fullname = (char*)aliasPtr+sizeof(AliasInfo);
	aliasPtr->content = aliasPtr->fullname + strlen(argv[3]) + 1;
	strcpy(aliasPtr->fullname, argv[3]);
	strcpy(aliasPtr->content, argv[4]);
	entryPtr = Tcl_CreateHashEntry(&aliasTable, key, &new);
	Tcl_SetHashValue(entryPtr, (ClientData)aliasPtr);
	return TCL_OK;
	
    } else if (!strcmp(argv[1], "delete")) {
	Tcl_HashEntry *entryPtr;
	char key[1024];
	int i;

	for (i=2; i<argc; i++) {
	    strncpy(key, argv[i], sizeof(key));
	    lcase(key);
	    if ((entryPtr = Tcl_FindHashEntry(&aliasTable, key))) {
		ckfree((AliasInfo*)Tcl_GetHashValue(entryPtr));
		Tcl_DeleteHashEntry(entryPtr);
	    }
	}
	return TCL_OK;

    } else if (!strcmp(argv[1], "list")) {
	Tcl_HashEntry *entryPtr;
	Tcl_HashSearch search;
	AliasInfo *aliasPtr;
	Tcl_DString list;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " list var\"", (char *) NULL);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&list);
	for (entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
		entryPtr; entryPtr = Tcl_NextHashEntry(&search)) {
	    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
	    Tcl_DStringAppendElement(&list, aliasPtr->fullname);
	    Tcl_DStringAppendElement(&list, aliasPtr->content);
	    Tcl_SetVar2(interp, argv[2], Tcl_GetHashKey(&aliasTable, entryPtr),
		    Tcl_DStringValue(&list), 0);
	    Tcl_DStringFree(&list);
	}
	return TCL_OK;
	
    } else if (!strcmp(argv[1], "read")) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " read filename\"", (char *) NULL);
	    return TCL_ERROR;
	}
	return Tcl_EvalFile(interp, argv[2]);
	
    } else if (!strcmp(argv[1], "save")) {
	Tcl_HashEntry *entryPtr;
	Tcl_HashSearch search;
	AliasInfo *aliasPtr;
	Tcl_DString list, buffer;
	char buf[1024];
	FILE *fp;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " save filename\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (NULL == (fp = fopen(Tcl_TildeSubst(interp,argv[2],&buffer), "w"))) {
	    Tcl_DStringFree(&buffer);
	    sprintf(buf, "Failed to open file \"%s\": %s", argv[2],
		    Tcl_PosixError(interp));
	    RatLog(interp, RAT_ERROR, buf, 0);
	    return TCL_ERROR;
	}
	Tcl_DStringFree(&buffer);

	Tcl_DStringInit(&list);
	for (entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
		entryPtr; entryPtr = Tcl_NextHashEntry(&search)) {
	    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
	    Tcl_DStringAppendElement(&list,
		    Tcl_GetHashKey(&aliasTable, entryPtr));
	    Tcl_DStringAppendElement(&list, aliasPtr->fullname);
	    Tcl_DStringAppendElement(&list, aliasPtr->content);
	    fputs("RatAlias add ", fp);
	    fputs(Tcl_DStringValue(&list), fp);
	    fputs("\n", fp);
	    Tcl_DStringSetLength(&list, 0);
	}
	Tcl_DStringFree(&list);

	if ( 0 != fclose(fp)) {
	    sprintf(buf, "Failed to close file \"%s\": %s", argv[2],
		    Tcl_PosixError(interp));
	    RatLog(interp, RAT_ERROR, buf, 0);
	    return TCL_ERROR;
	}
	return TCL_OK;
	
    } else if (!strcmp(argv[1], "expand1") || !strcmp(argv[1], "expand2")) {
	ADDRESS *adrPtr, *baseAdrPtr = NULL;
	struct passwd *pwPtr;
	Tcl_HashEntry *entryPtr, *userPtr = NULL;
	Tcl_DString newList;
	AliasInfo *aliasPtr;
	char buf[1024], *sPtr, *cPtr;
	int level, iterations, rescan, useTable, new;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " expand[12] adrlist\"", (char *) NULL);
	    return TCL_ERROR;
	}

	/*
	 * Ignore empty addresses
	 */
	for (sPtr = argv[2]; *sPtr && isspace(*sPtr); sPtr++);
	if (!*sPtr) {
	    return TCL_OK;
	}

	Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "lookup_name",
			               TCL_GLOBAL_ONLY), &useTable);
	Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "alias_iterations",
			               TCL_GLOBAL_ONLY), &iterations);

	if (!strcmp(argv[1], "expand1")) {
	    level = 0;
	} else {
	    level = 1;
	}
	Tcl_DStringInit(&newList);
	Tcl_DStringAppend(&newList, sPtr, -1);
	for (cPtr = Tcl_DStringValue(&newList); *cPtr; cPtr++) {
	    if (isspace(*cPtr)) {
		*cPtr = ' ';
	    }
	}
	do {
	    rescan = 0;
	    rfc822_parse_adrlist(&baseAdrPtr, Tcl_DStringValue(&newList), "");
	    Tcl_DStringSetLength(&newList, 0);
	    for (adrPtr = baseAdrPtr; adrPtr; adrPtr = adrPtr->next) {
		/*
		 * Check if this matches an alias. If the host is an empty
		 * string we just try the mailbox name as key, otherwise we
		 * try mailbox@host.
		 */
		if (adrPtr->error || (adrPtr->host && adrPtr->host[0] == '.')){
		    if (adrPtr->error) {
			Tcl_SetResult(interp, adrPtr->error, TCL_VOLATILE);
		    } else {
			Tcl_SetResult(interp, adrPtr->host, TCL_VOLATILE);
		    }
		    Tcl_DStringFree(&newList);
		    mail_free_address(&baseAdrPtr);
		    return TCL_ERROR;
		}
		entryPtr = NULL;
		if (adrPtr->host && *adrPtr->host) {
		    sprintf(buf, "%s@%s", adrPtr->mailbox, adrPtr->host);
		} else {
		    strcpy(buf, adrPtr->mailbox);
		}
		lcase(buf);
		if (useTable && 
			!(entryPtr = Tcl_FindHashEntry(&aliasTable, buf))) {
		    if ((pwPtr = getpwnam(buf))) {
			userPtr = Tcl_CreateHashEntry(&userTable,
				pwPtr->pw_name, &new);
			strcpy(buf, pwPtr->pw_gecos);
			if ((cPtr = strchr(buf, ','))) {
			    *cPtr = '\0';
			}
			Tcl_SetHashValue(userPtr, (ClientData)strdup(buf));
		    }
		}
		if (entryPtr) {
		    aliasPtr = (AliasInfo*) Tcl_GetHashValue(entryPtr);
		    if (level) {
			rescan = 1;
			Tcl_DStringAppend(&newList, aliasPtr->content, -1);
		    } else {
			sPtr = RatAddressMail(adrPtr);
			Tcl_DStringAppend(&newList, sPtr, -1);
			if (aliasPtr->fullname) {
			    sprintf(buf, " (%s)", aliasPtr->fullname);
			    Tcl_DStringAppend(&newList, buf, strlen(buf));
			}
		    }
		} else if ((!adrPtr->host || !*adrPtr->host) &&
			   !adrPtr->personal && !level && userPtr) {
		    sPtr = RatAddressMail(adrPtr);
		    Tcl_DStringAppend(&newList, sPtr, -1);
		    sprintf(buf, " (%s)", (char*)Tcl_GetHashValue(userPtr));
		    Tcl_DStringAppend(&newList, buf, -1);

		} else {
		    sPtr = RatAddressMail(adrPtr);
		    Tcl_DStringAppend(&newList, sPtr, -1);
		    if (adrPtr->personal && strlen(adrPtr->personal)&& !level) {
			sprintf(buf, " (%s)", adrPtr->personal);
			Tcl_DStringAppend(&newList, buf, -1);
		    }
		}
		if (adrPtr->next) {
		    Tcl_DStringAppend(&newList, ", ", 2);
		}
	    }
	    mail_free_address(&baseAdrPtr);
	    if (0 == --iterations && rescan) {
		RatLog(interp, RAT_WARN, 
			"Too many alias iterations (alias loop?)", 0);
		rescan = 0;
	    }
	} while(rescan);
	Tcl_DStringResult(interp, &newList);
	return TCL_OK;
	
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be one of add, list, read, save, expand1 or expand2",
		(char *) NULL);
	return TCL_ERROR;
    }

}

/*
 *----------------------------------------------------------------------
 *
 * RatAddressMail --
 *
 *      Prints the mail address in rfc822 format of an ADDRESS entry.
 *	Only one address is printed and there is NO fullname.
 *
 * Results:
 *	Pointer to a static storage area where the string is stored.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatAddressMail(ADDRESS *adrPtr)
{
    static char *store = NULL;
    static int length = 0;
    int quote = 0, grow, l, i;
    char *cPtr;

    /*
     * We start by checking if the mailbox part contains any special
     * chracters (actually specials, SPACE and CTLS). While we do that we
     * also calculate how much it will grow when converted to qtext.
     * If we must quote it then we do so.
     * After that we append the host part if any.
     */
    for (cPtr = adrPtr->mailbox, grow=0; *cPtr; cPtr++) {
	if (strchr("\"\\\n", *cPtr)) {
	    quote = 1;
	    grow++;
	}
	if (*cPtr <= 32 || strchr("()<>@,;:[]", *cPtr)) {
	    quote = 1;
	}
    }
    if (quote) {
	l = 2+strlen(adrPtr->mailbox)+grow
		+(adrPtr->host?strlen(adrPtr->host)+1:0)+1;
	if (l > length) {
	    length = l;
	    store = REALLOC(store, length);
	}
	i = 0;
	store[i++] = '"';
	for (cPtr = adrPtr->mailbox, grow=0; *cPtr; cPtr++) {
	    if (strchr("\"\\\n", *cPtr)) {
		store[i++] = '\\';
	    }
	    store[i++] = *cPtr;
	}
	store[i++] = '"';
    } else {
	l = strlen(adrPtr->mailbox)+(adrPtr->host?strlen(adrPtr->host)+1:0)+1;
	if (l > length) {
	    length = l;
	    store = REALLOC(store, length);
	}
	strcpy(store, adrPtr->mailbox);
	i = strlen(store);
    }
    if (adrPtr->host && *adrPtr->host) {
	store[i++] = '@';
	strcpy(&store[i], adrPtr->host);
    } else {
	store[i++] = '\0';
    }
    return store;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSplitAddresses --
 *
 *	This routine takes an address list as argument and splits it.
 *
 * Results:
 *	A list of addresses contained in the argument
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */
 
int
RatSplitAddresses(ClientData clientData, Tcl_Interp *interp, int argc,
	char *argv[])
{
    char *srcPtr, *dstPtr, *cPtr, *adr;
    Tcl_DString result;
    int inq = 0, inc = 0;
 
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" addresslist\"", (char *) NULL);
        return TCL_ERROR;
    }
 
    adr = (char*)malloc(strlen(argv[1])+1);
    Tcl_DStringInit(&result);
 
    for (srcPtr = argv[1], dstPtr = adr; *srcPtr; srcPtr++) {
	switch (*srcPtr) {
	case '\\':
	    if (srcPtr[1]) {
		*dstPtr++ = *srcPtr++;
	    }
	    *dstPtr++ = *srcPtr;
	    break;
	case '"': 
	    if (inq) {
		inq = 0;
	    } else {
		inq = 1;
	    }
	    *dstPtr++ = *srcPtr;
	    break;
	case '(':
	    inc = 1;
	    *dstPtr++ = *srcPtr;
	    break;
	case ')':
	    inc = 0;
	    *dstPtr++ = *srcPtr;
	    break;
	case ',':
	    if (!inq && !inc) {
		for (dstPtr--; dstPtr >= adr && isspace(*dstPtr); dstPtr--);
		if (++dstPtr != adr) {
		    *dstPtr = '\0';
		    for (cPtr = adr; *cPtr && isspace(*cPtr); cPtr++);
		    if (cPtr) {
			Tcl_DStringAppendElement(&result, cPtr);
		    }
		    dstPtr = adr;
		}
		break;
	    }
	    /* fallthrough */
	default:
	    *dstPtr++ = *srcPtr;
	    break;
	}
    }
    if (dstPtr != adr) {
	*dstPtr = '\0';
	Tcl_DStringAppendElement(&result, adr);
    }
 
    Tcl_DStringResult(interp, &result);
    free(adr);
    return TCL_OK;
}

