/*
 * ratDbMessage.c --
 *
 *	This file contains code which implements dbase messages.
 *
 * TkRat software and its included text is Copyright 1996 by Martin Forssen.
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"

/*
 * The ClientData for each message entity
 */
typedef struct DbMessageInfo {
    MESSAGE *messagePtr;
} DbMessageInfo;

/*
 * The ClientData for each bodypart entity
 */
typedef struct DbBodyInfo {
    char *text;
} DbBodyInfo;

/*
 * The number of message entities created. This is used to create new
 * unique command names.
 */
static int numDbMessages = 0;

/*
 * The number of bodypart entities created. This is used to create new
 * unique command names.
 */
static int numDbBodies = 0;

static RatGetHeadersProc Db_GetHeadersProc;
static RatGetEnvelopeProc Db_GetEnvelopeProc;
static RatInfoProc Db_GetInfoProc;
static RatCreateBodyProc Db_CreateBodyProc;
static RatFetchTextProc Db_FetchTextProc;
static RatEnvelopeProc Db_EnvelopeProc;
static RatFetchFirstTextProc Db_FetchFirstTextProc;
static RatSetIndexProc Db_SetIndexProc;
static RatMsgDeleteProc Db_MsgDeleteProc;
static RatMakeChildrenProc Db_MakeChildrenProc;
static RatFetchBodyProc Db_FetchBodyProc;
static RatBodyDeleteProc Db_BodyDeleteProc;


/*
 *----------------------------------------------------------------------
 *
 * RatDbMessagesInit --
 *
 *      Initializes the given MessageProcInfo entry for a dbase message
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The given MessageProcInfo is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatDbMessagesInit(MessageProcInfo *messageProcInfoPtr)
{
    messageProcInfoPtr->getHeadersProc = Db_GetHeadersProc;
    messageProcInfoPtr->getEnvelopeProc = Db_GetEnvelopeProc;
    messageProcInfoPtr->getInfoProc = Db_GetInfoProc;
    messageProcInfoPtr->createBodyProc = Db_CreateBodyProc;
    messageProcInfoPtr->fetchTextProc = Db_FetchTextProc;
    messageProcInfoPtr->envelopeProc = Db_EnvelopeProc;
    messageProcInfoPtr->fetchFirstTextProc = Db_FetchFirstTextProc;
    messageProcInfoPtr->setIndexProc = Db_SetIndexProc;
    messageProcInfoPtr->msgDeleteProc = Db_MsgDeleteProc;
    messageProcInfoPtr->makeChildrenProc = Db_MakeChildrenProc;
    messageProcInfoPtr->fetchBodyProc = Db_FetchBodyProc;
    messageProcInfoPtr->bodyDeleteProc = Db_BodyDeleteProc;
}


/*
 *----------------------------------------------------------------------
 *
 * RatDbMessageCreate --
 *
 *      Creates a dbase message entity
 *
 * Results:
 *	The name of the new message entity.
 *
 * Side effects:
 *	None.
 *	deleted.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatDbMessageCreate(Tcl_Interp *interp, RatFolderInfoPtr infoPtr, int index)
{
    DbMessageInfo *msgDbPtr=(DbMessageInfo*)ckalloc(sizeof(DbMessageInfo));
    MessageInfo *msgPtr=(MessageInfo*)ckalloc(sizeof(MessageInfo));
    char *name = (char*)ckalloc(16);

    msgPtr->interp = interp;
    msgPtr->folderInfoPtr = infoPtr;
    msgPtr->name = name;
    msgPtr->type = RAT_DBASE_MESSAGE;
    msgPtr->bodyInfoPtr = NULL;
    msgPtr->msgNo = index;
    msgPtr->clientData = (ClientData)msgDbPtr;
    msgDbPtr->messagePtr = RatDbGetMessage(interp, index);
    sprintf(name, "RatDbMsg%d", numDbMessages++);
    Tcl_CreateCommand(interp, name, RatMessageCmd,
    	    (ClientData) msgPtr, (Tcl_CmdDeleteProc *) RatMessageDelete);
    return name;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_GetHeadersProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_GetHeadersProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    return RatDbGetHeaders(interp, msgPtr->msgNo);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_GetEnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_GetEnvelopeProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    static char buf[1024];
    RatDbEntry *entryPtr = RatDbGetEntry(msgPtr->msgNo);
    struct tm *tmPtr;
    time_t date;

    date = atoi(entryPtr->content[DATE]);
    tmPtr = gmtime(&date);
    sprintf(buf, "From %s %s %s %2d %02d:%02d GMT 19%02d\n",
	    entryPtr->content[FROM], dayName[tmPtr->tm_wday],
	    monthName[tmPtr->tm_mon], tmPtr->tm_mday, tmPtr->tm_hour,
	    tmPtr->tm_min, tmPtr->tm_year);
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_CreateBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static BodyInfo*
Db_CreateBodyProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    DbMessageInfo *dbMsgPtr = (DbMessageInfo*)msgPtr->clientData;
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)ckalloc(sizeof(DbBodyInfo));
    msgPtr->bodyInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);

    msgPtr->bodyInfoPtr->cmdName = (char*)msgPtr->bodyInfoPtr
	    + sizeof(BodyInfo);
    sprintf(msgPtr->bodyInfoPtr->cmdName, "RatDbBody%d", numDbBodies++);
    msgPtr->bodyInfoPtr->msgPtr = msgPtr;
    msgPtr->bodyInfoPtr->bodyPtr = dbMsgPtr->messagePtr->body;
    msgPtr->bodyInfoPtr->firstbornPtr = NULL;
    msgPtr->bodyInfoPtr->nextPtr = NULL;
    msgPtr->bodyInfoPtr->containedEntity = NULL;
    msgPtr->bodyInfoPtr->clientData = (ClientData)dbBodyInfoPtr;
    dbBodyInfoPtr->text =
	dbMsgPtr->messagePtr->text.text.data+dbMsgPtr->messagePtr->text.offset;
    Tcl_CreateCommand(interp, msgPtr->bodyInfoPtr->cmdName,
	    RatBodyCmd, (ClientData) msgPtr->bodyInfoPtr, NULL);
    return msgPtr->bodyInfoPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_FetchTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_FetchTextProc(Tcl_Interp *interp, MessageInfo *msgPtr)
{
    return RatDbGetText(interp, msgPtr->msgNo);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_EnvelopeProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static ENVELOPE*
Db_EnvelopeProc(MessageInfo *msgPtr)
{
    return ((DbMessageInfo*)msgPtr->clientData)->messagePtr->env;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_FetchFirstTextProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static BODY*
Db_FetchFirstTextProc(MessageInfo *msgPtr, char **dataPtrPtr, int *lengthPtr)
{
    DbMessageInfo *dbMsgPtr = (DbMessageInfo*)msgPtr->clientData;
    BODY *bodyPtr = dbMsgPtr->messagePtr->body;
    long offset = dbMsgPtr->messagePtr->text.offset;

    while (TYPETEXT != bodyPtr->type) {
	if (TYPEMULTIPART != bodyPtr->type) {
	    bodyPtr = NULL;
	    break;
	} else {
	    bodyPtr = &bodyPtr->nested.part->body;
	    offset = bodyPtr->contents.offset;
	}
    }
    if (bodyPtr) {
	*dataPtrPtr = dbMsgPtr->messagePtr->text.text.data + offset;
	*lengthPtr = bodyPtr->size.bytes;
    }
    return bodyPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_SetIndexProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_SetIndexProc(MessageInfo *msgPtr, int index)
{
     return;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_MsgDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_MsgDeleteProc(MessageInfo *msgPtr)
{
    DbMessageInfo *dbMsgPtr = (DbMessageInfo*)msgPtr->clientData;
    ckfree(dbMsgPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Db_MakeChildrenProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_MakeChildrenProc(Tcl_Interp *interp, BodyInfo *bodyInfoPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;
    BODY *bodyPtr = bodyInfoPtr->bodyPtr;
    BodyInfo *partInfoPtr, **partInfoPtrPtr;
    DbBodyInfo *dbPartInfoPtr;
    PART *partPtr;

    if (!bodyInfoPtr->firstbornPtr) {
	partInfoPtrPtr = &bodyInfoPtr->firstbornPtr;
	for (partPtr = bodyPtr->nested.part; partPtr;
		partPtr = partPtr->next) {
	    dbPartInfoPtr = (DbBodyInfo*)ckalloc(sizeof(DbBodyInfo));
	    partInfoPtr = (BodyInfo*)ckalloc(sizeof(BodyInfo)+16);
	    *partInfoPtrPtr = partInfoPtr;
	    partInfoPtr->cmdName = (char*)partInfoPtr+sizeof(BodyInfo);
	    sprintf(partInfoPtr->cmdName, "RatDbBody%d", numDbBodies++);
	    partInfoPtr->bodyPtr = &partPtr->body;
	    partInfoPtr->firstbornPtr = NULL;
	    partInfoPtr->nextPtr = NULL;
	    partInfoPtr->containedEntity = NULL;
	    partInfoPtrPtr = &partInfoPtr->nextPtr;
	    partInfoPtr->msgPtr = bodyInfoPtr->msgPtr;
	    partInfoPtr->clientData = (ClientData)dbPartInfoPtr;
	    dbPartInfoPtr->text = dbBodyInfoPtr->text+
				  partPtr->body.contents.offset;
	    Tcl_CreateCommand(interp, partInfoPtr->cmdName, RatBodyCmd,
		    (ClientData) partInfoPtr, NULL);
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * Db_FetchBodyProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static char*
Db_FetchBodyProc(BodyInfo *bodyInfoPtr, int *lengthPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;

    *lengthPtr = bodyInfoPtr->bodyPtr->contents.text.size;
    return dbBodyInfoPtr->text;
}


/*
 *----------------------------------------------------------------------
 *
 * Db_BodyDeleteProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

static void
Db_BodyDeleteProc(BodyInfo *bodyInfoPtr)
{
    DbBodyInfo *dbBodyInfoPtr = (DbBodyInfo*)bodyInfoPtr->clientData;
    ckfree(dbBodyInfoPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * Db_GetInfoProc --
 *
 *      See ratFolder.h
 *
 *----------------------------------------------------------------------
 */

char*
Db_GetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int notused)
{
    MessageInfo *msgPtr = (MessageInfo*)clientData;
    MESSAGE *messagePtr = ((DbMessageInfo*)msgPtr->clientData)->messagePtr;
    static Tcl_DString ds;
    static int initialized = 0;

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    }

    switch (type) {
	case RAT_FOLDER_SUBJECT:	/* fallthrough */
	case RAT_FOLDER_NAME:		/* fallthrough */
	case RAT_FOLDER_MAIL:		/* fallthrough */
	case RAT_FOLDER_NAME_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:	/* fallthrough */
	case RAT_FOLDER_SIZE:		/* fallthrough */
	case RAT_FOLDER_SIZE_F:		/* fallthrough */
	case RAT_FOLDER_TYPE:		/* fallthrough */
	case RAT_FOLDER_PARAMETERS:	/* fallthrough */
	case RAT_FOLDER_DATE_F:		/* fallthrough */
	case RAT_FOLDER_DATE_N:		/* fallthrough */
	case RAT_FOLDER_TO:		/* fallthrough */
	case RAT_FOLDER_FROM:		/* fallthrough */
	case RAT_FOLDER_STATUS:		/* fallthrough */
	case RAT_FOLDER_INDEX:
	    return Db_InfoProc(interp, (ClientData)msgPtr->folderInfoPtr, type,
		    -msgPtr->msgNo);
	case RAT_FOLDER_SENDER:		/* fallthrough */
	case RAT_FOLDER_CC:		/* fallthrough */
	case RAT_FOLDER_REPLY_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    if (type == RAT_FOLDER_SENDER) {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->sender);
	    } else if (type == RAT_FOLDER_CC) {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->cc);
	    } else {
		rfc822_write_address(Tcl_DStringValue(&ds),
			messagePtr->env->reply_to);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_END:
	    break;
    }
    return NULL;
}
