# folder.tcl --
#
# This file contains code which handles a folder window.
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# The list of folder windows
set folderWindowList {}

# FolderWindowInit --
#
# Initializes a folder window, that is it populates it with widgets and
# installs all callbacks. It reaturns the handler that should be used for
# this folder window.
#
# Arguments:
# w -		Window which the folder window should be packed into

proc FolderWindowInit w {
    global t idCnt statusText option defaultFont folderWindowList \
	   valueFont ratDeferred

    # Create the handler
    set handler f[incr idCnt]
    lappend folderWindowList $handler
    upvar #0 $handler fh

    # Initialize variables
    set fh(toplevel) [winfo toplevel $w]
    set fh(w) $w
    set fh(folder_name) {}
    set fh(folder_size) {}
    set fh(folder_messages) {}
    set fh(groupMessageLists) {}
    set fh(message_scroll) $w.messlist.scroll
    set fh(message_list) $w.messlist.list
    set fh(after_id) {}
    set fh(group) {}
    set fh(text) $w.text.text
    upvar #0 hd$fh(text) texth
    set texth(show_header) $option(show_header)
    set texth(struct_menu) $w.mbar.message.m.structmenu

    # The menu and information line
    frame $w.mbar -relief raised -bd 1

    # Tkrat menu
    menubutton $w.mbar.tkrat -menu $w.mbar.tkrat.m -text TkRat
    menu $w.mbar.tkrat.m
    $w.mbar.tkrat.m add command -label $t(send_deferred) \
	    -command SendDeferred -state disabled
    trace variable ratDeferred w \
	    "TraceDeferred $w.mbar.tkrat.m [$w.mbar.tkrat.m index end]"
    $w.mbar.tkrat.m add command -label $t(notifications)... -command ShowDSNList
    $w.mbar.tkrat.m add checkbutton -label $t(watcher) \
	    -variable option(watcher_enable) -onvalue 1 -offvalue 0
    $w.mbar.tkrat.m add separator
    $w.mbar.tkrat.m add command -label $t(reread_aliases) -command AliasRead
    $w.mbar.tkrat.m add command -label $t(reread_userproc) -command ReadUserproc
    $w.mbar.tkrat.m add command -label $t(import_aliases) -command ScanAliases
    $w.mbar.tkrat.m add command -label $t(reread_mailcap) \
	    -command RatMailcapReload
    $w.mbar.tkrat.m add command -label $t(take_mail)... \
	    -command "MailSteal $handler 0"
    $w.mbar.tkrat.m add separator
    $w.mbar.tkrat.m add command -label $t(check_dbase)... \
	    -command "DbaseCheck 0"
    $w.mbar.tkrat.m add command -label $t(check_fix_dbase)... \
	    -command "DbaseCheck 1"
    $w.mbar.tkrat.m add separator
    $w.mbar.tkrat.m add command -label $t(see_log)... -command SeeLog
    $w.mbar.tkrat.m add separator
    $w.mbar.tkrat.m add command -label $t(quit) -command "Quit $handler"
    set fh(quit_menu) $w.mbar.tkrat.m
    set fh(quit_index) [$w.mbar.tkrat.m index end]
    
    # Folder menu
    menubutton $w.mbar.folder -menu $w.mbar.folder.m -text $t(folders)
    menu $w.mbar.folder.m -postcommand "PostFolder $handler $w.mbar.folder.m"

    # Message menu
    menubutton $w.mbar.message -menu $w.mbar.message.m -text $t(message)
    menu $w.mbar.message.m
    set fh(message_menu_nokeep) {}
    $w.mbar.message.m add command -label $t(compose)... -command Compose
    set fh(compose_menu) $w.mbar.message.m
    set fh(compose_index) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(reply_sender)... \
	    -command "FolderReply $handler sender"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    set fh(replys_menu) $w.mbar.message.m
    set fh(replys_index) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(reply_all)... \
	    -command "FolderReply $handler all"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    set fh(replya_menu) $w.mbar.message.m
    set fh(replya_index) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(forward_inline)... \
	    -command "FolderSomeCompse $handler ComposeForwardInline"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    set fh(forward_i_menu) $w.mbar.message.m
    set fh(forward_i_index) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(forward_as_attachment)... \
	    -command "FolderSomeCompse $handler ComposeForwardAttachment"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    set fh(forward_a_menu) $w.mbar.message.m
    set fh(forward_a_index) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(getheld)... -command ComposeHeld
    $w.mbar.message.m add command -label $t(extract_adr)... \
	    -command "AliasExtract $handler"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    $w.mbar.message.m add separator
    $w.mbar.message.m add cascade -label $t(move) -menu $w.mbar.message.m.move
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    menu $w.mbar.message.m.move \
	     -postcommand "PostMove $handler current $w.mbar.message.m.move"
    $w.mbar.message.m add command -label $t(delete) \
	    -command "SetFlag $handler deleted 1; FolderNext $handler"
    set fh(delete_menu) $w.mbar.message.m
    set fh(delete_index) [$w.mbar.message.m index end]
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(undelete) \
	    -command "SetFlag $handler deleted 0; FolderNext $handler"
    set fh(undelete_menu) $w.mbar.message.m
    set fh(undelete_index) [$w.mbar.message.m index end]
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    $w.mbar.message.m add command -label $t(print) \
	    -command "Print $handler current"
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    $w.mbar.message.m add cascade -label $t(structure) \
    	    -menu $w.mbar.message.m.structmenu
    lappend fh(message_menu_nokeep) [$w.mbar.message.m index end]
    $w.mbar.message.m add separator
    $w.mbar.message.m add radiobutton -label $t(show_all_headers) \
	    -variable hd$fh(text)(show_header) -value all \
	    -command "FolderSelect $fh(message_list) $handler; SaveOptions"
    $w.mbar.message.m add radiobutton -label $t(show_selected_headers) \
	    -variable hd$fh(text)(show_header) -value selected \
	    -command "FolderSelect $fh(message_list) $handler; SaveOptions"
    $w.mbar.message.m add radiobutton -label $t(show_no_headers) \
	    -variable hd$fh(text)(show_header) -value no \
	    -command "FolderSelect $fh(message_list) $handler; SaveOptions"

    # Group menu
    menubutton $w.mbar.group -menu $w.mbar.group.m -text $t(group)
    menu $w.mbar.group.m -postcommand "SetupGroupMenu $w.mbar.group.m $handler"
    $w.mbar.group.m add command -label $t(create_in_win)... \
	    -command "GroupMessageList $handler"
    $w.mbar.group.m add command -label $t(create_by_expr)... \
	    -command "ExpCreate $handler"
    $w.mbar.group.m add cascade -label $t(use_saved_expr) \
	    -menu $w.mbar.group.m.saved
    menu $w.mbar.group.m.saved \
	    -postcommand "ExpBuildMenu $w.mbar.group.m.saved $handler"
    $w.mbar.group.m add command -label $t(clear_group) \
	    -command "GroupClear $handler"
    $w.mbar.group.m add separator
    $w.mbar.group.m add command -label $t(delete) -command "SetFlag \
	    $handler deleted 1 \[\$${handler}(folder_handler) flagged flagged\]"
    $w.mbar.group.m add command -label $t(undelete) -command "SetFlag \
	    $handler deleted 0 \[\$${handler}(folder_handler) flagged flagged\]"
    $w.mbar.group.m add command -label $t(print) \
	    -command "Print $handler group"
    $w.mbar.group.m add cascade -label $t(move) -menu $w.mbar.group.m.move
    menu $w.mbar.group.m.move \
	     -postcommand "PostMove $handler group $w.mbar.group.m.move"

    # Admin menu
    menubutton $w.mbar.admin -menu $w.mbar.admin.m -text $t(admin)
    menu $w.mbar.admin.m
    $w.mbar.admin.m add command -label $t(update_folder) \
	    -command "Sync $handler update"
    set fh(update_menu) $w.mbar.admin.m
    set fh(update_index) [$w.mbar.admin.m index end]
    $w.mbar.admin.m add command -label $t(sync_folder) \
	    -command "Sync $handler expunge"
    set fh(sync_menu) $w.mbar.admin.m
    set fh(sync_index) [$w.mbar.admin.m index end]
    $w.mbar.admin.m add separator
    $w.mbar.admin.m add command -label $t(newedit_folder)... \
	    -command VFolderDef
    $w.mbar.admin.m add command -label $t(aliases)... -command Aliases
    $w.mbar.admin.m add command -label $t(preferences)... -command Preferences
    $w.mbar.admin.m add command -label $t(define_keys)... \
	    -command "KeyDef folder"
    $w.mbar.admin.m add command -label $t(saved_expr)... -command ExpHandleSaved
    $w.mbar.admin.m add separator
    $w.mbar.admin.m add radiobutton -label $t(subject_by_date) \
	    -variable option(folder_sort) -value subject \
	    -command {foreach f $folderWindowList {Sync $f update}; SaveOptions}
    $w.mbar.admin.m add radiobutton -label $t(natural_order) \
	    -variable option(folder_sort) -value folder \
	    -command {foreach f $folderWindowList {Sync $f update}; SaveOptions}
    $w.mbar.admin.m add radiobutton -label $t(reverse_natural_order) \
	    -variable option(folder_sort) -value reverseFolder \
	    -command {foreach f $folderWindowList {Sync $f update}; SaveOptions}
    $w.mbar.admin.m add radiobutton -label $t(by_date) \
	    -variable option(folder_sort) -value date \
	    -command {foreach f $folderWindowList {Sync $f update}; SaveOptions}
    $w.mbar.admin.m add radiobutton -label $t(reverse_date) \
	    -variable option(folder_sort) -value reverseDate \
	    -command {foreach f $folderWindowList {Sync $f update}; SaveOptions}

    # Help menu
    menubutton $w.mbar.help -menu $w.mbar.help.m -text $t(help)
    menu $w.mbar.help.m
    $w.mbar.help.m add command -label $t(version)... -command Version
    $w.mbar.help.m add command -label Ratatosk... -command Ratatosk
    $w.mbar.help.m add command -label $t(help_window)... -command Help

    # The structure menu (constructed by the Show routine)
    menu $texth(struct_menu) -tearoff 0

    # Pack the menus into the menu bar
    pack $w.mbar.tkrat \
	 $w.mbar.folder \
	 $w.mbar.message \
	 $w.mbar.group \
	 $w.mbar.admin -side left -padx 5
    pack $w.mbar.help -side right -padx 5

    # The information part
    frame $w.info -relief raised -bd 1
    label $w.info.flabel -text $t(name):
    label $w.info.fname -textvariable ${handler}(folder_name) \
        -font $valueFont -anchor w
    label $w.info.slabel -text $t(size):
    label $w.info.size -textvariable ${handler}(folder_size) -width 5 \
	-anchor w -font $valueFont
    label $w.info.mlabel -text $t(messages):
    label $w.info.messages -textvariable ${handler}(folder_messages) -width 4 \
        -anchor w -font $valueFont
    pack $w.info.messages \
         $w.info.mlabel \
         $w.info.size \
         $w.info.slabel -side right
    pack $w.info.flabel -side left
    pack $w.info.fname -side left -fill x -expand 1

    # The message list
    frame $w.messlist
    scrollbar $fh(message_scroll) \
        -relief sunken \
        -command "$w.messlist.list yview" \
	-highlightthickness 0
    listbox $fh(message_list) \
        -yscroll "$fh(message_scroll) set" \
        -bd 0 \
        -font $defaultFont \
        -exportselection false \
	-highlightthickness 0 \
	-selectmode single \
	-height $option(message_llength)
    pack $fh(message_scroll) -side right -fill y
    pack $fh(message_list) -side left -expand 1 -fill both

    # The status line
    label $w.statustext -textvariable statusText -relief raised \
	    -bd 1 -font $valueFont -width 80
    
    # The command buttons
    frame $w.buttons
    menubutton $w.buttons.move -text $t(move) -bd 1 -relief raised \
	    -menu $w.buttons.move.m -indicatoron 1
    menu $w.buttons.move.m \
	    -postcommand "PostMove $handler current $w.buttons.move.m"
    button $w.buttons.delete -text $t(delete) -bd 1 -highlightthickness 0 \
	    -command "SetFlag $handler deleted 1; FolderNext $handler"
    button $w.buttons.compose -text $t(compose)... -bd 1 -highlightthickness 0 \
	    -command Compose
    button $w.buttons.reply_sender -text $t(reply_sender)... -bd 1 \
	    -highlightthickness 0 -command "FolderReply $handler sender"
    button $w.buttons.reply_all -text $t(reply_all)... -bd 1 \
	    -highlightthickness 0 -command "FolderReply $handler all"
    pack $w.buttons.move \
	 $w.buttons.delete \
	 $w.buttons.compose \
	 $w.buttons.reply_sender \
	 $w.buttons.reply_all -side left -expand 1 -fill x

    # The actual text
    frame $w.text -relief raised -bd 1
    scrollbar $w.text.scroll \
        -relief sunken \
        -command "$w.text.text yview" \
	-highlightthickness 0
    text $fh(text) \
        -yscroll "$w.text.scroll set" \
        -setgrid true \
        -relief raised \
        -wrap char \
        -font $defaultFont \
	-bd 0 \
	-highlightthickness 0
    Size $w.text.text folderM
    pack $w.text.scroll -side right -fill y
    pack $w.text.text -side left -expand yes -fill both

    # Pack all the parts into the window
    pack $w.mbar -side top -fill x
    pack $w.info -side top -fill x
    pack $w.messlist -side top -fill both
    pack $w.statustext -side top -fill x -anchor center
    pack $w.buttons -side top -fill x
    pack $w.text -side top -expand yes -fill both

    # Do bindings
    bind $w <Enter>		"focus $w"
    bind $w.messlist.list <ButtonRelease-1>\
				"FolderSelect $w.messlist.list $handler"
    bind $w.messlist.list <ButtonPress-3> \
			"SetFlag $handler flagged toggle \[%W index @%x,%y\]"
    bind $fh(toplevel) <Map>	"WatcherSleep $handler"
    FolderBind $handler
    wm protocol $fh(toplevel) WM_DELETE_WINDOW "Quit $handler"

    return $handler
}


# FolderBind --
#
# Bind the key definitions for a folder window
#
# Arguments:
# handler - The handler which identifies the fodler window

proc FolderBind {handler} {
    upvar #0 $handler fh

    RatBind $fh(w) folder_key_compose   "Compose" \
	$fh(compose_menu) $fh(compose_index)
    RatBind $fh(w) folder_key_quit	"Quit $handler" \
	$fh(quit_menu) $fh(quit_index)
    RatBind $fh(w) folder_key_nextu	"FolderSelectUnread $handler; break"
    RatBind $fh(w) folder_key_sync	"Sync $handler expunge" \
	$fh(sync_menu) $fh(sync_index)
    RatBind $fh(w) folder_key_update    "Sync $handler update" \
	$fh(update_menu) $fh(update_index)
    RatBind $fh(w) folder_key_delete    "SetFlag $handler deleted 1; \
	FolderNext $handler" $fh(delete_menu) $fh(delete_index)
    RatBind $fh(w) folder_key_undelete  "SetFlag $handler deleted 0; \
	FolderNext $handler" $fh(undelete_menu) $fh(undelete_index)
    RatBind $fh(w) folder_key_flag	"SetFlag $handler flagged toggle;
	FolderNext $handler"
    RatBind $fh(w) folder_key_next	"FolderNext $handler"
    RatBind $fh(w) folder_key_prev	"FolderPrev $handler"
    RatBind $fh(w) folder_key_home	"ShowHome $fh(text)"
    RatBind $fh(w) folder_key_pagedown  "ShowPageDown $fh(text)"
    RatBind $fh(w) folder_key_pageup    "ShowPageUp $fh(text)"
    RatBind $fh(w) folder_key_replya    "FolderReply $handler all" \
	$fh(replya_menu) $fh(replya_index)
    RatBind $fh(w) folder_key_replys    "FolderReply $handler sender" \
	$fh(replys_menu) $fh(replys_index)
    RatBind $fh(w) folder_key_forward_i \
	"FolderSomeCompse $handler ComposeForwardInline" \
	$fh(forward_i_menu) $fh(forward_i_index)
    RatBind $fh(w) folder_key_forward_a \
	"FolderSomeCompse $handler ComposeForwardAttachment"\
	$fh(forward_a_menu) $fh(forward_a_index)
    RatBind $fh(w) folder_key_cycle_header "CycleShowHeader $handler"
}


# FolderRead --
#
# Reads and the given folder and calls draw_folder_list to show the content.
# If there already is an active folder it is closed. As arguments it expects
# a folderwindow handler and a command which when run opens the folder.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# foldercmd -	Command which creates the new folder

proc FolderRead {handler foldercmd} {
    global option inbox t
    upvar #0 $handler fh

    # First we expunge the old folder
    if [info exists fh(folder_handler)] {
	$fh(folder_handler) close
	if ![string compare $inbox $fh(folder_handler)] {
	    set inbox ""
	}
	unset fh(folder_handler)
    }

    # Open the new folder
    set id [RatLog 2 $t(opening_folder)... explicit]
    if [catch $foldercmd folder_handler] {
	set fh(folder_name) ""
	set fh(folder_messages) ""
	set fh(folder_size) ""
	ShowNothing $fh(text)
	catch {unset fh(current); unset fh(folder_handler)}
	$fh(message_list) delete 0 end
	RatClearLog $id
	FolderButtons $handler 0
	return
    }
    RatClearLog $id

    # Update our information
    set fh(folder_handler) $folder_handler
    set i [$fh(folder_handler) info]
    set fh(folder_name) [lindex $i 0]
    set fh(folder_messages) [RatMangleNumber [lindex $i 1]]
    FolderDrawList $handler
    set i [$fh(folder_handler) info]
    if { -1 == [lindex $i 2]} {
	set fh(folder_size) ???
    } else {
	set fh(folder_size) [RatMangleNumber [lindex $i 2]]
    }
    switch $option(folder_sort) {
    reverseFolder {
	    set dir -1
	    set start 0
	}
    reverseDate {
	    set dir -1
	    set start 0
	}
    default {
	    set dir 1
	    set start [expr [$fh(message_list) size]-1]
	}
    }
    if { 0 != [$fh(message_list) size]} {
	switch $option(start_selection) {
	last {
		set index [expr [$fh(message_list) size]-1]
	    }
	first_new {
		set index [FolderGetNextUnread $handler $start $dir]
	    }
	before_new {
		set index [FolderGetNextUnread $handler $start $dir]
		if { 0 == [$fh(folder_handler) getFlag $index seen]} {
		    incr index -$dir
		    if {$index < 0 || $index >= [$fh(message_list) size]} {
			incr index $dir
		    }
		}
	    }
	default {	# And first
		set index 0
	    }
	}
	$fh(message_list) selection set $index
	$fh(message_list) see $index
    }
    FolderSelect $fh(message_list) $handler

    # Initialize watcher
    if [string length $fh(after_id)] {
	after cancel $fh(after_id)
	set fh(after_id) {}
    }
    set type [$folder_handler type]
    foreach timePair $option(watcher_time) {
	if ![string compare [lindex $timePair 0] $type] {
	    set time [expr 1000*[lindex $timePair 1]]
	    if $time {
		set fh(after_id) [after $time WatcherDoRound $handler $type]
	    }
	}
    }
    return $folder_handler
}

# FolderDrawList --
#
# Constructs the list of messages in the folder and shows this.
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderDrawList {handler} {
    upvar #0 $handler fh
    global option

    $fh(message_list) delete 0 end
    eval "$fh(message_list) insert 0 \
	    [$fh(folder_handler) list $option(list_format)]"
    foreach w $fh(groupMessageLists) {
	GroupMessageListUpdate $w $handler
    }
}

# FolderSelect --
#
# Handle the selection of a message
#
# Arguments:
# l       -	The listbox the message was selected from
# handler -	The handler which identifies the folder window

proc FolderSelect {l handler} {
    upvar #0 $handler fh
    global option

    set index [$l curselection]
    if {0 == [string length $index]} {
	catch {unset fh(current)}
	FolderButtons $handler 0
	ShowNothing $fh(text)
	return
    }
    if ![info exists fh(current)] {
	FolderButtons $handler 1
    }
    set fh(current) [$fh(folder_handler) get $index]
    set seen [$fh(folder_handler) getFlag $index seen]
    $fh(folder_handler) setFlag $index seen 1
    Show $fh(text) $fh(current)
    if { 0 == $seen } {
	set top [lindex [$fh(message_list) yview] 0]
	$fh(message_list) delete $index
	$fh(message_list) insert $index [lindex [[$fh(folder_handler) get \
		$index] list $option(list_format)] 0]
	$fh(message_list) selection set $index
	$fh(message_list) yview moveto $top
    }
}

# FolderNext --
#
# Advance to the next message in the folder
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderNext {handler} {
    upvar #0 $handler fh
    set ml $fh(message_list)

    if ![string length [$ml curselection]] { return }

    set index [expr 1+[$ml curselection]]
    if { $index >= [$ml size] } {
	return
    }

    $ml selection clear [$fh(message_list) curselection]
    $ml selection set $index
    if {[expr $index/[$ml size].0] >= [lindex [$ml yview] 1]} {
	$ml yview $index
    }
    FolderSelect $ml $handler
}

# FolderPrev --
#
# Retreat to the previous message in the folder
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderPrev {handler} {
    upvar #0 $handler fh
    set ml $fh(message_list)

    if ![string length [$ml curselection]] { return }

    set index [expr [$ml curselection]-1]
    if {$index < 0} {
	return
    }

    $ml selection clear [$fh(message_list) curselection]
    $ml selection set $index
    if {[expr $index/[$ml size].0] < [lindex [$ml yview] 0]} {
	$ml yview scroll -1 pages
    }
    FolderSelect $ml $handler
}

# SetFlag --
#
# Set a flag to a specified value for the current message or a set of
# messages given as argument.
#
# Arguments:
# handler  -	The handler which identifies the folder window
# flag	   -	The flag to set
# value    -	The new value of the deletion flag ('1' or '0')
# messages -	Ano optional list of messages to do this to

proc SetFlag {handler flag value {messages {}}} {
    upvar #0 $handler fh
    global option

    if ![string length [$fh(message_list) curselection]] { return }

    set sel [$fh(message_list) curselection]
    if ![string length $messages] {
	set messages $sel
    }
    set top [lindex [$fh(message_list) yview] 0]
    foreach i $messages {
	if [string compare toggle $value] {
	    $fh(folder_handler) setFlag $i $flag $value
	} else {
	    if [$fh(folder_handler) getFlag $i $flag] {
		set v 0 
	    } else {
		set v 1
	    }
	    $fh(folder_handler) setFlag $i $flag $v
	}
	$fh(message_list) delete $i
	$fh(message_list) insert $i [lindex [[$fh(folder_handler) get $i] \
		list $option(list_format)] 0]
    }
    $fh(message_list) selection set $sel
    $fh(message_list) yview moveto $top
}

# Quit --
#
# Closes the given folder window
#
# Arguments:
# handler -	The handler which identifies the folder window

proc Quit {handler} {
    global expAfter logAfter ratDeferred ratSenderSending t composeWindowList
    upvar #0 $handler fh

    if [info exists composeWindowList] {
	if {0 < [llength $composeWindowList]} {
	    if {0 == [RatDialog $t(really_quit) $t(compose_sessions) \
		    {} 0 $t(dont_quit) $t(do_quit)]} {
		return
	    }
	}
    }
    if {0 < $ratDeferred} {
	if {1 < $ratDeferred} {
	    set text "$t(you_have) $ratDeferred $t(deferred_messages)"
	} else {
	    set text "$t(you_have) 1 $t(deferred_message)"
	}
	if {0 == [RatDialog $t(send_deferred) $text {} 0 \
		$t(send_now) $t(no)]} {
	    tkwait window [SendDeferred]
	}
    }
    if $ratSenderSending {
	RatLog 2 $t(waiting_on_sender) explicit
	while {1 == $ratSenderSending} {
	    tkwait variable ratSenderSending
	}
	RatLog 2 "" explicit
    }
    if [string length $fh(after_id)] {
	after cancel $fh(after_id)
    }
    if [string length $expAfter] {
	after cancel $expAfter
    }
    if [string length $logAfter] {
	after cancel $logAfter
    }
    if [info exists fh(folder_handler)] {
	$fh(folder_handler) expunge
	$fh(folder_handler) close
    }
    RecordPos [winfo toplevel $fh(w)] folder
    RecordSize $fh(text) folderM
    if [info exists fh(watcher_w)] {
	if [info exists fh(watcher_geom)] {
	    wm geom $fh(watcher_w) $fh(watcher_geom)
	    RecordPos $fh(watcher_w) watcher
	}
	RecordSize $fh(watcher_list) watcher
    }
    SavePos
    unset fh
    destroy .
}

# Sync --
#
# Does an expunge or update on the current folder.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# operation -	Which operation we should do 'expunge' or 'update'

proc Sync {handler operation} {
    upvar #0 $handler fh

    if ![info exists fh(folder_handler)] { return }

    set indexBefore [$fh(message_list) curselection]
    set listBefore [$fh(message_list) get 0 end]
    if [string length $indexBefore] {
	set subject [lindex $listBefore $indexBefore]
	set topBefore [expr round([lindex [$fh(message_list) yview] 0]*\
			        [$fh(message_list) size])]
	set msg $fh(current)
    }
    set topBefore [lindex [$fh(message_list) yview] 0]

    $fh(folder_handler) $operation
    $fh(message_list) selection clear 0 end
    FolderDrawList $handler

    # Update information
    set i [$fh(folder_handler) info]
    set fh(folder_messages) [RatMangleNumber [lindex $i 1]]
    if { -1 == [lindex $i 2]} {
	set fh(folder_size) ???
    } else {
	set fh(folder_size) [RatMangleNumber [lindex $i 2]]
    }
    if { 0 == [lindex $i 1] } {
	FolderSelect $fh(message_list) $handler
	return
    }
    set listAfter [$fh(message_list) get 0 end]
    set index 0

    # Check if our element is still in there
    if [string length $indexBefore] {
	set index [$fh(folder_handler) find $msg]
	if { -1 != $index } {
	    $fh(message_list) selection set $index
	    $fh(message_list) yview moveto $topBefore
	} else {
	    if ![string length [$fh(message_list) curselection]] {
		# Search for adjacent elements
		set list [lrange $listBefore [expr $indexBefore+1] end]
		for {set i $indexBefore} {$i >= 0} {incr i -1} {
		    lappend list [lindex $listBefore $i]
		}
		foreach element $list {
		    if {-1!=[set index [lsearch -exact $listAfter $element]]} {
			$fh(message_list) selection set $index
			FolderSelect $fh(message_list) $handler
			break
		    }
		}
	    }
	}
    }
    if ![string length [$fh(message_list) curselection]] {
	$fh(message_list) selection set 0
	FolderSelect $fh(message_list) $handler
    }
    $fh(message_list) see [$fh(message_list) curselection]
}

# FolderReply --
#
# Construct a reply to a message and update the messages status if the
# reply was sent.
#
# Arguments:
# handler   -	The handler which identifies the folder window
# recipient -	Who the reply should be sent to 'sender' or 'all'

proc FolderReply {handler recipient} {
    upvar #0 $handler fh

    if ![string length [$fh(message_list) curselection]] { return }

    set current $fh(current)
    set hd [ComposeReply $current $recipient]
    upvar #0 $hd sendHandler
    trace variable sendHandler w "FolderReplySent $hd $handler $current"
}
proc FolderReplySent {hd handler current name1 name2 op} {
    upvar #0 $handler fh
    upvar #0 $hd sendHandler
    global option

    trace vdelete $name1 $op "FolderReplySent $hd $handler $current"

    if [info exists sendHandler(do)] {
	if ![string compare send $sendHandler(do)] {
	    set index [$fh(folder_handler) find $current]
	    if { -1 != $index } {
		if ![$fh(folder_handler) getFlag $index answered] {
		    SetFlag $handler answered 1 $index
		}
	    }
	}
    }
}

# FolderSomeCompse --
#
# Run a compose function on a message and update the message status if the
# message actually was sent.
#
# Arguments:
# handler   -	The handler which identifies the folder window

proc FolderSomeCompse {handler composeFunc} {
    upvar #0 $handler fh
    global option

    if ![string length [$fh(message_list) curselection]] { return }

    set current $fh(current)
    set hd [$composeFunc $current]
    upvar #0 $hd sendHandler
    trace variable sendHandler w "FolderReplySent $hd $handler $current"
}

# PostFolder --
#
# Populate the folder menu
#
# Arguments:
# handler -	The handler which identifies the folder window
# m       -	The menu which we should populate

proc PostFolder {handler m} {
    global t option

    $m delete 1 end
    VFolderBuildMenu $m 0 "VFolderOpen $handler" 0
    $m add separator
    $m add command -label $t(open_file)... -command "SelectFileFolder $handler"
    $m add command -label $t(open_dbase)... \
	    -command "SelectDbasefolder $handler"
    FixMenu $m
}

# PostMove --
#
# Populate the move menu
#
# Arguments:
# handler -	The handler which identifies the folder window
# which	  -	Which set of messages we should move (current or group)
# m       -	The menu which we should populate

proc PostMove {handler which m} {
    upvar #0 $handler fh
    global t

    if [string compare "group" $which] {
	set a 1
    } else {
	set a 0
    }
    $m delete 1 end
    VFolderBuildMenu $m 0 \
	    "VFolderInsert $handler $a \[GetMsgSet $handler $which\]" 1
    $m add separator
    $m add command -label $t(to_file)... \
	    -command "VFolderInsert $handler $a \[GetMsgSet $handler $which\] \
		      \[InsertIntoFile\]"
    $m add command -label $t(to_dbase)... \
	    -command "VFolderInsert $handler $a \[GetMsgSet $handler $which\] \
		      \[InsertIntoDBase\]"
    FixMenu $m
}

# GetMsgSet --
#
# Get the messages that the current operation should be performed on
#
# Arguments:
# handler -	The handler which identifies the folder window
# which	  -	Which set of messages we should move (current or group)

proc GetMsgSet {handler which} {
    upvar #0 $handler fh

    if [string compare group $which] {
	return $fh(current)
    } else {
	set msgs {}
	foreach i [$fh(folder_handler) flagged flagged] {
	    lappend msgs [$fh(folder_handler) get $i]
	}
	return $msgs
    }
}

# FolderButtons --
#
# Enable or disable the buttons in the folder window which depends on
# an active message.
#
# Arguments:
# handler -	The handler which identifies the folder window
# onoff   -	The new state of the buttons

proc FolderButtons {handler onoff} {
    upvar #0 $handler fh
    set w $fh(w)

    if $onoff {
	set state normal
    } else {
	set state disabled
    }
    foreach b [list $w.buttons.move \
		    $w.buttons.delete \
		    $w.buttons.reply_sender \
		    $w.buttons.reply_all] {
	$b configure -state $state
    }
    foreach i $fh(message_menu_nokeep) {
	$w.mbar.message.m entryconfigure $i -state $state
    }
}

# FolderGetNextUnread --
#
# Return the index of the next unread message in folder after index,
# or index if none is found.
#
# Arguments:
# handler -	The handler which identifies the folder window
# index   -	Where in the list to start looking

proc FolderGetNextUnread {handler index dir} {
    upvar #0 $handler fh

    set size [$fh(message_list) size]
    if {0 == $size} {
	return 0
    }
    for {set i [expr $index+$dir]} {$i != $index} {incr i $dir} {
	if {$i >= $size} {
	    set i 0
	} elseif {$i < 0} {
	    set i [expr $size-1]
	}
	if {$i == $index} {
	    return 0
	}
	if { 0 == [$fh(folder_handler) getFlag $i seen]} {
	    return $i
	}
    }
    return $index
}


# FolderSelectUnread --
#
# Selects the next unread message in the folder.
#
# Arguments:
# handler -	The handler which identifies the folder window

proc FolderSelectUnread {handler} {
    upvar #0 $handler fh

    set index [$fh(message_list) curselection]
    if {0 == [llength $index]} {
	return
    }
    set i [FolderGetNextUnread $handler $index 1]
    $fh(message_list) selection clear $index
    $fh(message_list) selection set $i
    $fh(message_list) see $i
    FolderSelect $fh(message_list) $handler
}

# TraceDeferred --
#
# Trace the ratDeferred variable
#
# Arguments:
# m	- The menu to update
# index - The menu entry to update
# name1	- notused
# name2	- notused
# op	- notused

proc TraceDeferred {m index name1 name2 op} {
     global ratDeferred t

    if $ratDeferred {
	set state normal
    } else {
	set state disabled
    }
    $m entryconfigure $index -label "$t(send_deferred)    \[$ratDeferred\]" \
	    -state $state
}

# GroupMessageList --
#
# Pops a message list that lets the user select messages for a group
#
# Arguments:
# handler -	The handler which identifies the folder window

proc GroupMessageList {handler} {
    global idCnt defaultFont t option
    upvar #0 $handler fh

    # Create identifier
    set id f[incr idCnt]
    set w .$id

    # Create toplevel
    toplevel $w
    wm title $w $t(edit_group)
    frame $w.f
    scrollbar $w.f.scroll \
        -relief sunken \
        -command "$w.f.list yview" \
	-highlightthickness 0
    listbox $w.f.list \
        -yscroll "$w.f.scroll set" \
        -font $defaultFont \
        -exportselection false \
	-highlightthickness 0 \
	-selectmode multiple \
	-setgrid true
    Size $w.f.list gFolderL
    pack $w.f.scroll -side right -fill y
    pack $w.f.list -side left -expand 1 -fill both
    frame $w.buttons
    button $w.buttons.ok -text $t(ok) \
	    -command "GroupMessageListDone $w $handler 1"
    button $w.buttons.sel -text $t(select_all) \
	    -command "$w.f.list selection set 0 end"
    button $w.buttons.unsel -text $t(deselect_all) \
	    -command "$w.f.list selection clear 0 end"
    button $w.buttons.cancel -text $t(cancel) \
	    -command "GroupMessageListDone $w $handler 0"
    pack $w.buttons.ok \
	 $w.buttons.sel \
	 $w.buttons.unsel \
	 $w.buttons.cancel -side left -expand 1
    pack $w.buttons -side bottom -fill x -pady 5
    pack $w.f -expand 1 -fill both

    eval "$w.f.list insert 0 [$fh(folder_handler) list $option(list_format)]"
    foreach i [$fh(folder_handler) flagged flagged] {
	$w.f.list selection set $i
    }
    lappend fh(groupMessageLists) $w

    Place $w groupMessageList
}

# GroupMessageListUpdate --
#
# Update the message list since the underlying folder was updated
#
# Arguments:
# w	  -	The group selection window
# handler -	The handler which identifies the folder window

proc GroupMessageListUpdate {w handler} {
    upvar #0 $handler fh
    global option

    $w.f.list delete 0 end
    eval "$w.f.list insert 0 [$fh(folder_handler) list $option(list_format)]"
    foreach i [$fh(folder_handler) flagged flagged] {
	$w.f.list selection set $i
    }
}

# GroupMessageListDone --
#
# Calls when the grouping is done
#
# Arguments:
# w	  -	The group selection window
# handler -	The handler which identifies the folder window
# done    -	The users selection (1=ok, 0=cancel)

proc GroupMessageListDone {w handler done} {
    upvar #0 $handler fh
    global option

    if $done {
	set top [lindex [$fh(message_list) yview] 0]
	set sel [$fh(message_list) curselection]
	set toset [$w.f.list curselection]
	set isset [$fh(folder_handler) flagged flagged]
	for {set i 0} {$i < [$w.f.list size]} {incr i} {
	    set s [expr -1 != [lsearch $toset $i]]
	    if {$s != [expr -1 != [lsearch $isset $i]]} {
		$fh(folder_handler) setFlag $i flagged $s
		$fh(message_list) delete $i
		$fh(message_list) insert $i [lindex \
			[[$fh(folder_handler) get $i] \
			 list $option(list_format)] 0]
	    }
	}
	foreach i [$w.f.list curselection] {
	    if ![$fh(folder_handler) getFlag $i flagged] {
	    }
	}
	$fh(message_list) yview moveto $top
	$fh(message_list) selection set $sel
    }
    RecordPos $w groupMessageList
    RecordSize $w.f.list gFolderL
    set index [lsearch $w $fh(groupMessageLists)]
    set fh(groupMessageLists) [lreplace $fh(groupMessageLists) $index $index]
    destroy $w
}

# GroupClear --
#
# Removes the flag from every message
#
# Arguments:
# handler -	The handler which identifies the folder window

proc GroupClear {handler} {
    upvar #0 $handler fh
    global option

    set top [lindex [$fh(message_list) yview] 0]
    set sel [$fh(message_list) curselection]
    foreach i [$fh(folder_handler) flagged flagged] {
	$fh(folder_handler) setFlag $i flagged 0
	$fh(message_list) delete $i
	$fh(message_list) insert $i [lindex [[$fh(folder_handler) get $i] \
		list $option(list_format)] 0]
    }
    $fh(message_list) yview moveto $top
    $fh(message_list) selection set $sel
}

# SetupGroupMenu --
#
# Setup the entries in the group menu
#
# Arguments:
# m	  -	The menu command name
# handler -	The handler which identifies the folder window

proc SetupGroupMenu {m handler} {
    upvar #0 $handler fh

    if ![info exists fh(folder_handler)] {
	set s disabled
    } elseif {[llength [$fh(folder_handler) flagged flagged]]} {
	set s normal
    } else {
	set s disabled
    }
    $m entryconfigure 4 -state $s
    $m entryconfigure 6 -state $s
    $m entryconfigure 7 -state $s
    $m entryconfigure 8 -state $s
    $m entryconfigure 9 -state $s
}

# CycleShowHeader --
#
# Cycle through the values of the show_header option
#
# Arguments:
# handler -	The handler which identifies the folder window

proc CycleShowHeader {handler} {
    global option
    upvar #0 $handler fh
    upvar #0 hd$fh(text) texth

    switch $texth(show_header) {
    all		{ set texth(show_header) selected }
    selected	{ set texth(show_header) no }
    no		{ set texth(show_header) all }
    }
    FolderSelect $fh(message_list) $handler
    SaveOptions
}
