# font.tcl --
#
# Handles cataloguing and selection of fonts
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# This will be the list of fonts
# Each font is represented by one list of four entries:
#	encoding size attributes name
set font_List {}

# AddFont --
#
# Add a font to the list of known fonts.
#
# Arguments:
# encoding   - The encoding of this font
# size       - The size difference from "normal"
# attributes - The attributes
# name	     - The name of the font

proc AddFont {encoding size attributes name} {
    global font_List fontEncoding
    lappend font_List [list $encoding $size $attributes $name]

    # Add encoding
    set encoding [string tolower $encoding]
    if {![info exists fontEncoding($encoding)]} {
	set fontEncoding($encoding) 1
    }

    # Add US-ASCII if compatible
    if {[RatEncodingCompat us-ascii $encoding]
	    && ![info exists fontEncoding(us-ascii)]} {
       set fontEncoding(us-ascii) 1
    }
}


# RemoveFonts --
#
# Remove slected fonts from the list of known fonts.
#
# Arguments:
# name - Name of fonts to remove (may be regexp)

proc RemoveFonts {name} {
    global font_List

    set fl {}
    foreach f $font_List {
	set n [lindex $f 3]
	if ![regexp -- $name $n] {
	    lappend fl $f
	}
    }
    set font_List $fl
}


# GetFont --
#
# Get the font that best matches the requirements
#
# Arguments:
# encoding   - The encoding of this font
# size       - The size difference from "normal"
# attributes - The attributes

proc GetFont {encoding size attributes} {
    global font_List option

    # The ultimate result
    set name {}
    set equivalence 100

    incr size $option(fontsize)

    foreach font $font_List {
	# Check encoding
	if ![RatEncodingCompat $encoding [lindex $font 0]] {
	    continue
	}
	# calculate equivalence rating
	set e [expr 4*abs([lindex $font 1]-$size)]
	set attr [lindex $font 2]
	foreach a $attributes {
	    set i [lsearch -exact $attr $a]
	    if { -1 == $i} {
		incr e 2
	    } else {
		set attr [lreplace $attr $i $i]
	    }
	}
	incr e [llength $attr]
	# Check if good
	if {$e < $equivalence} {
	    set equivalence $e
	    set name [lindex $font 3]
	    if {0 == $e} {
		return $name
	    }
	}
    }
    return $name
}


# RatEncodingCompat --
#
# Checks if an encoding is compatible with another
#
# Arguments:
# wanted - The encoding we want to use
# avail  - The available encoding

proc RatEncodingCompat {wanted avail} {
    set wanted [string tolower $wanted]
    set avail [string tolower $avail]

    # Check if equal
    if ![string compare $wanted $avail] {
	return 1
    }

    # US-ASCII is compatible with iso-8859-*
    if {![string compare us-ascii $wanted] && [regexp iso-8859- $avail]} {
	return 1
    }
    return 0
}


# RatSelectEncoding --
#
# Select which encoding to express some characters in
#
# Arguments:
# encoding - The current encoding of the characters

proc RatSelectEncoding {encoding} {
    global fontEncoding option

    # First we try to find it
    if [info exists fontEncoding($encoding)] {
	return $encoding
    }

    # Then we try to find a compatible one
    foreach e [array names fontEncoding] {
	if [RatEncodingCompat $encoding $e] {
	    return $e
	}
    }

    # Finally we fall back to our default encoding
    return $option(charset)
}
