# help.tcl --
#
# This file contains code which handles help windows
#
#
#  TkRat software and its included text is Copyright 1996 by Martin Forssen.
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.

# Help --
#
# Creates a new help window and shows the requested help-entry (or an
# introduction if none is specified).
#
# Arguments:
# section  - The section to show (may be empty)

proc Help {{subject tkrat}} {
    global idCnt t help option defaultFont

    # Initialize help texts (if needed)
    if ![info exists help(general)] {
	InitMessages $option(language) help
    }

    # Create identifier
    set id helpWin[incr idCnt]
    upvar #0 $id hd
    set w .$id

    # Create toplevel
    toplevel $w
    wm title $w $t(help_window)

    # Subtopics frame
    frame $w.subtopic
    label $w.subtopic.label -text $t(subtopic)
    scrollbar $w.subtopic.scroll \
	-relief sunken \
	-command "$w.subtopic.list yview" \
	-highlightthickness 0
    listbox $w.subtopic.list \
	-yscroll "$w.subtopic.scroll set" \
	-width 20 \
	-relief sunken \
	-bd 1 \
	-exportselection false \
	-highlightthickness 0
    pack $w.subtopic.label -side top
    pack $w.subtopic.scroll -side right -fill y
    pack $w.subtopic.list -expand 1 -fill both
    set hd(subtopics) $w.subtopic.list
    set hd(subtopic_nodes) {}

    # Path frame
    frame $w.path
    label $w.path.label -text $t(parent_topics)
    scrollbar $w.path.scroll \
	-relief sunken \
	-command "$w.path.list yview" \
	-highlightthickness 0
    listbox $w.path.list \
	-yscroll "$w.path.scroll set" \
	-height 6 \
	-width 20 \
	-relief sunken \
	-bd 1 \
	-exportselection false \
	-highlightthickness 0
    pack $w.path.label -side top
    pack $w.path.scroll -side right -fill y
    pack $w.path.list -expand 1 -fill both
    set hd(path) $w.path.list
    set hd(path_nodes) {}

    # Text frame
    frame $w.t
    scrollbar $w.t.scroll \
	-relief sunken \
	-command "$w.t.text yview" \
	-highlightthickness 0
    text $w.t.text \
	-yscroll "$w.t.scroll set" \
	-width 80 \
	-height 50 \
	-setgrid true \
	-wrap word \
	-relief sunken \
	-bd 1 \
	-highlightthickness 0 \
	-font $defaultFont 
    pack $w.t.scroll -side right -fill y
    pack $w.t.text -expand 1 -fill both
    set hd(text) $w.t.text

    button $w.dismiss -text $t(dismiss) -command "RecordPos $w help; destroy $w"

    # Bindings
    bind $hd(path) <ButtonRelease-1> \
	"if {1 == \[llength \[%W curselection\]\]} { \
		ShowHelp $id \[lindex \$${id}(path_nodes) \[%W curselection\]\]\
	 }"
    bind $hd(subtopics) <ButtonRelease-1> \
	"if {1 == \[llength \[%W curselection\]\]} { \
		ShowHelp $id \[lindex \$${id}(subtopics_nodes) \
			\[%W curselection\]\]\
	 }"

    # Pack window
    pack $w.t -side left -expand 1 -fill both -padx 5 -pady 5
    pack $w.dismiss -side bottom -padx 10 -pady 10
    pack $w.path -side top -padx 5 -pady 5
    pack $w.subtopic -expand 1 -fill both -padx 5 -pady 5

    Place $w help

    ShowHelp $id $subject
}

# ShowHelp --
#
# Populates the help window
#
# Arguments:
# id   - The help-window identifier
# node - The node to show

proc ShowHelp {id node} {
    global help
    upvar #0 $id hd

    # The text window
    $hd(text) configure -state normal
    $hd(text) delete 0.0 end
    $hd(text) insert end $help($node)
    $hd(text) configure -state disabled

    # The subtopics list
    $hd(subtopics) delete 0 end
    if [info exists help(struct,$node)] {
	foreach s $help(struct,$node) {
	    $hd(subtopics) insert end $help(title,$s)
	}
	set hd(subtopics_nodes) $help(struct,$node)
    }

    # The parent topics list
    set i [lsearch -exact $hd(path_nodes) $node]
    if { -1 == $i } {
	lappend hd(path_nodes) $node
	$hd(path) insert end $help(title,$node)
    } else {
	incr i
	if { $i < [llength $hd(path_nodes)]} {
	    set hd(path_nodes) [lreplace $hd(path_nodes) $i end]
	    $hd(path) delete $i end
	}
    }
    $hd(path) selection clear 0 end
}
