## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "enhancedTextwidget.tcl"
 #                                    created: 04/12/98 {22:45:38 PM} 
 #                                
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html
 #  
 # Copyright (c) 1998-2001  Vince Darley
 # 
 # See the file "license.terms" for information on use and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

# Namespace used to over-ride the text widget, and to store
# a variety of enhanced-text-widget-specific information.
namespace eval tw {    
    variable lmatch [list "(" "\{" "\[" "\""]
    variable rmatch [list ")" "\}" "\]" "\""]
    
    variable split
    variable split_focus
    variable splitter
    variable hooks
}

#
# tw::Hook --
# 
# Can be used like 'set' to attach global information to all
# enhanced text windows.  It is used for a variety of predefined
# hooks.  These are:
# 
# activate w -- this is now the foremost editing window
# 
# deactivate w -- this window was the foremost and is about to lose
# the focus
# 
# lock w l -- the user has clicked in the 'lock' region of the status
# control and the window has changed to status $l (0 or 1 as to whether
# unlocked or locked)
# 
# save w -- the user has asked that the window be saved.
# 
# dirty w d -- the window has become dirty ($d == 1) or clean ($d == 0).
# 
# position {row col} -- the current cursor position is given by the
# text string {row col}.
# 
# hyper cmd -- the user has clicked on a hypertext link in the window
# and the $cmd should be evaluated in Tcl.
# 
# The goal of all these hooks, of course, is to abstract away the
# 'enhanced text widget' from the rest of Alphatk/AlphaTcl.
# 
proc tw::Hook {name args} {
    variable hooks
    eval [list ::set hooks($name)] $args
}

proc tw::CallHook {name args} {
    variable hooks
    if {[info exists hooks($name)]} {
	eval $hooks($name) $args
    }
}

tw::Hook position    [list .status.w.position configure -text]
tw::Hook activate    [list ::alpha::divertTkCallToWindow ::activateHook]
tw::Hook deactivate  [list ::alpha::divertTkCallToWindow ::deactivateHook]
tw::Hook lock        [list ::alpha::divertTkCallToWindow ::alphatk_lockClick]
tw::Hook save        [list ::alpha::divertTkCallToWindow ::save]
tw::Hook dirty       [list ::alpha::divertTkCallToWindow ::dirtyHook]
tw::Hook hyper       [list ::alphatk_hyper]

# For each text widget subcommand, create a dummy procedure 'tw::subcmd'
# for our purposes of wrapping the text widget.  Most of these subcommands
# are actually over-ridden below, but we do this exhaustively, so that
# even if the text widget gains new subcommands with new Tk releases
# (e.g. Tk 8.4 adds 'edit'), we make sure we cover them all.
if {[info commands ::tw::configure] == ""} {
    text .alphatktextdummy
    catch {.alphatktextdummy alphatk} msg
    destroy .alphatktextdummy

    regsub ".*must be " $msg "" msg
    regsub ", or " $msg ", " msg
    regsub -all ", " $msg " " msg
    foreach subcmd [split $msg ] {
	proc tw::$subcmd {w args} \
	  "uplevel 1 ::tw::\[getSplitFocus \$w\] $subcmd \$args"
    }
    unset msg
    unset subcmd
}

# These two procedures are the only ones which
# know that text widgets are called '$W.text$num'
proc tw::_uniqueTextWidget {W} {
    # Find an unused path name of a specific form inside the window (if
    # the frame is split, we'll use more and more of these).
    set i 1
    while {[winfo exists $W.text$i]} {
	incr i
    }
    return $W.text$i
}

# Given the widget path of a text window, return
# the widget path of the corresponding '$to' widget
# (a scrollbar, margin, toolbar, etc).
proc tw::_textTo {w to} {
    regsub "\\.text" $w ".$to" wres
    set wres
}

proc tw::MakeWindow {W {text ""} args} {
    set w [_uniqueTextWidget $W]
    # The vertical scrollbar
    set ws [_textTo $w scroll]
    
    variable $w
    
    set minimal 0
    
    foreach {arg val} $args {
	switch -- $arg {
	    "-encoding" {
		if {$val != ""} {
		    set ${w}(encoding) $val
		}
	    }
	    "-font" {
		set font $val
	    }
	    "-tabsize" {
		set ${w}(tabsize) $val
	    }
	    "-minimal" {
		set minimal $val
	    }
	    default {
		error "Bad argument $arg to MakeWindow"
	    }
	}
    }
    set ${w}(linenumbers) 0
    
    if {!$minimal} {
	foreach {p v} [list encoding "" dirty 0 read-only 0 tabsize 8] {
	    if {![info exists ${w}($p)]} {
		set ${w}($p) $v
	    }
	}
    }
    
    text $w -relief flat -bd 2 -yscrollcommand "$ws set" \
      -bg white -height 0 -wrap none
    if {[info exists font]} {
	$w configure -font $font
    }

    global tcl_platform
    if {$tcl_platform(platform) == "macintosh"} {
	$w configure -highlightthickness 0
    }
    scrollbar $ws -command [list ::tw::yview $w]
    
    # Find next free row
    set row [lindex [grid size $W] 1]
    
    # Both styles use exactly three columns.
    if {$minimal} {
	# A minimal window takes up just one row, and cannot
	# have a horizontal scrollbar.  It also doesn't have
	# the toolbar at the top right
	grid $w -sticky news -column 1 -row $row
	grid $ws -sticky nse -column 2 -row $row
	grid rowconfigure $W $row -weight 1
    } else {
	# A more complex window takes up four rows, and the
	# vertical scrollbar is complicated by the small
	# toolbar in the top right.  Anyone can add things
	# to that toolbar with 'tw::Toolbar'.
	frame $W.rt -highlightcolor red -height 48
	frame $W.splitter -height 8 -relief raised -borderwidth 3 \
	  -highlightthickness 1 -width [winfo reqwidth $ws]
	bindtags $W.splitter Splitter
	grid $w -sticky news -column 1 -row $row -rowspan 3
	grid $W.rt -sticky nsew -column 2 -row $row
	incr row
	grid $W.splitter -sticky we -column 2 -row $row
	incr row
	grid $ws -sticky nse -column 2 -row $row
	global horScrollBar
	tw::horizScrollbar $w $horScrollBar
    
	label $W.rt.dirty -image clean
	bindtags $W.rt.dirty [concat "Lock" [bindtags $W.rt.dirty]]

	grid $W.rt.dirty -row 0 -sticky ew
	grid columnconfigure $W.rt 0 -weight 1

	grid rowconfigure $W $row -weight 1
    }

    update idletasks
    # These three need only be set once per window, since they
    # apply to all columns (i.e. all split windows) which are
    # added.  However, it doesn't harm to redo them each time.
    grid columnconfigure $W 2 -weight 0 -minsize [winfo reqwidth $ws]
    grid columnconfigure $W 1 -weight 1
    grid columnconfigure $W 0 -weight 0

    Textwidget $w
    $w insert 1.0 $text
    $w mark set insert 1.0
    
    _updateMargin $w
    
    rename $w ::tw::$w
    proc ::$w {cmd args} \
      "uplevel 1 \[list namespace eval ::tw \"\$cmd $w \$args\"\]"

    return $w
}

proc tw::Textwidget {w args} {
    global blockCursor tabSize
    bindtags $w [concat Alpha AlphaStyle [bindtags $w]]
    eval [list $w configure -wrap none] $args
    tw::setTabSize $w $tabSize
    if {$blockCursor} {
	# Not in any version of Tk yet.
	#catch {$w configure -blockcursor 1 -insertbackground grey25}
    }
    $w tag configure blink -background black -foreground white
    $w tag configure backsel -background darkgray

    global colorInds
    $w tag configure color1 -foreground [rgbToColor $colorInds(blue)]
    $w tag configure color2 -foreground [rgbToColor $colorInds(cyan)]
    $w tag configure color3 -foreground [rgbToColor $colorInds(green)]
    $w tag configure color4 -foreground [rgbToColor $colorInds(magenta)]
    $w tag configure color5 -foreground [rgbToColor $colorInds(red)]
    $w tag configure color6 -foreground [rgbToColor $colorInds(white)]
    $w tag configure color7 -foreground [rgbToColor $colorInds(yellow)]
    if {[info exists colorInds(color_8)]} {
	$w tag configure color8 -foreground [rgbToColor $colorInds(color_8)]
    } else {
	$w tag configure color8 -foreground blue
    }
    if {[info exists colorInds(color_9)]} {
	$w tag configure color9 -foreground [rgbToColor $colorInds(color_9)]
    } else {
	$w tag configure color9 -foreground blue
    }
    if {[info exists colorInds(color_10)]} {
	$w tag configure color10 -foreground [rgbToColor $colorInds(color_10)]
    } else {
	$w tag configure color10 -foreground blue
    }
    if {[info exists colorInds(color_11)]} {
	$w tag configure color11 -foreground [rgbToColor $colorInds(color_11)]
    } else {
	$w tag configure color11 -foreground blue
    }
    if {[info exists colorInds(color_12)]} {
	$w tag configure color12 -foreground [rgbToColor $colorInds(color_12)]
    } else {
	$w tag configure color12 -foreground blue
    }
    if {[info exists colorInds(color_13)]} {
	$w tag configure color13 -foreground [rgbToColor $colorInds(color_13)]\
	  -underline 1
    } else {
	$w tag configure color13 -underline 1
    }
    if {[info exists colorInds(color_14)]} {
	$w tag configure color14 -foreground [rgbToColor $colorInds(color_14)]\
	  -underline 1
    } else {
	$w tag configure color14 -underline 1
    }
    if {[info exists colorInds(color_15)]} {
	$w tag configure color15 -foreground [rgbToColor $colorInds(color_15)]\
	  -underline 1
    } else {
	$w tag configure color15 -underline 1
    }

    $w tag bind color15 <ButtonPress> "text_cmd hyper activate %x %y ; break"
    $w tag bind color15 <Enter> "text_cmd hyper enter %x %y ; break"
    $w tag bind color15 <Motion> "text_cmd hyper motion %x %y ; break"
    $w tag bind color15 <Leave> "text_cmd hyper leave %x %y ; break"

    if {[info tclversion] >= 8.3} {
	$w tag configure elidden -elide 1
    }
}

proc tw::Toolbar {w cmd path} {
    switch -- $cmd {
	"path" {
	    return $w.rt.$path
	}
	"add" {
	    set num [llength [grid slaves $w.rt]]
	    grid $w.rt.$path -row $num -sticky ew
	}
    }
}

proc tw::bbox {w args} { uplevel 1 ::tw::[getSplitFocus $w] bbox $args } 
proc tw::cget {w args} { uplevel 1 ::tw::[getSplitFocus $w] cget $args } 
proc tw::compare {w args} { uplevel 1 ::tw::[getSplitFocus $w] compare $args } 
proc tw::configure {w args} { 
    uplevel 1 ::tw::[getSplitFocus $w] configure $args 
} 
proc tw::debug {w args} { uplevel 1 ::tw::[getSplitFocus $w] debug $args } 
proc tw::dump {w args} { uplevel 1 ::tw::[getSplitFocus $w] dump $args } 
proc tw::dlineinfo {w args} { 
    uplevel 1 ::tw::[getSplitFocus $w] dlineinfo $args 
} 
proc tw::get {w args} { uplevel 1 ::tw::[getSplitFocus $w] get $args } 
proc tw::image {w args} { uplevel 1 ::tw::[getSplitFocus $w] image $args } 
proc tw::index {w args} { uplevel 1 ::tw::[getSplitFocus $w] index $args } 
proc tw::mark {w args} {
    set res [uplevel 1 ::tw::[getSplitFocus $w] mark $args]
    if {([lindex $args 0] == "set") && ([lindex $args 1] == "insert")} {
	CallHook position [::split [index $w insert] .]
    }
    return $res
} 
proc tw::scan {w args} { uplevel 1 ::tw::[getSplitFocus $w] scan $args } 
proc tw::tag {w args} { uplevel 1 ::tw::[getSplitFocus $w] tag $args } 
proc tw::window {w args} { uplevel 1 ::tw::[getSplitFocus $w] window $args } 
proc tw::xview {w args} { uplevel 1 ::tw::[getSplitFocus $w] xview $args } 

# This is uplevel 2 because of the use of 'namespace eval' in
# the renamed window procedure.
proc tw::search {w args} {
    #puts stderr "$w $args"
    global useMultilineSearchRoutines
    if {$useMultilineSearchRoutines} {
	uplevel 2 tw::multisearch $w $args
    } else {
	uplevel 2 tw::[getSplitFocus $w] search $args
    }
}

# This can be used internally where we know we only want to
# look at a single line.  For example the forward/backward
# word functionality (also used by syntax colouring) uses
# this.  It's obviously significantly faster, and is therefore
# useful when we need to call a proc hundreds or thousands of
# times very quickly (as with syntax colouring).
proc tw::singlesearch {w args} {
    uplevel 1 tw::[getSplitFocus $w] search $args
}

proc tw::getSplitFocus {w} {
    variable split
    if {[info exists split($w)]} {
	variable split_focus
	return $split_focus($split($w))
    } else {
	return $w
    }
}

proc tw::base_window {w} {
    variable split
    return [expr {[info exists split($w)] ? $split($w) : $w}]
}

# return list of first and last viewable positions
proc tw::_viewable {w} {
    variable $w
    set first [expr {int([$w index @0,0])}]
    set last [expr {int([$w index "@[winfo width $w],[winfo height $w]"])}]
    set ${w}(viewable) [list $first $last]
}

proc tw::viewable {w args} {
    if {[llength $args]} {
	set first [$w index @0,0]
	set last [$w index "@[winfo width $w],[winfo height $w]"]
	foreach pos $args {
	    if {[$w compare $pos < $first]} {
		lappend res -1
	    } elseif {[$w compare $pos > $last]} {
		lappend res 1
	    } else {
		lappend res 0
	    }
	}
	return $res
    } else {
	_viewable $w
    }
}

proc tw::silent_insert {w where args} { 
    variable split
    set where [$w index $where]
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    set old [_viewable $ww]
	    uplevel 1 [list ::tw::$ww insert $where] $args
	    eval [list arrangeToColourNewlyRevealed $ww] $old [list insert] $where
	    _updateMargin $ww
	}
    } else {
	set old [_viewable $w]
	uplevel 1 [list ::tw::$w insert $where] $args 
	eval [list arrangeToColourNewlyRevealed $w] $old [list insert] $where
	_updateMargin $w
    }
}

proc tw::silent_delete {w args} { 
    variable split
    # We must have normalized index positions
    foreach p $args {
	lappend idx [index $w $p]
    }
    if {[info exists split($w)]} {
	variable splitter
	foreach ww $splitter([set split($w)]) {
	    set old [_viewable $ww]
	    uplevel 1 [list ::tw::$ww delete] $idx
	    eval [list arrangeToColourNewlyRevealed $ww] $old [list delete] $idx
	    _updateMargin $ww
	}
    } else {
	set old [_viewable $w]
	uplevel 1 [list ::tw::$w delete] $idx
	eval [list arrangeToColourNewlyRevealed $w] $old [list delete] $idx
	_updateMargin $w
    }
}

proc tw::windowCleanup {w} {
    variable split
    if {[info exists split($w)]} {
	variable splitter
	variable split_focus
	set original $split($w)
	foreach ww $splitter($original) {
	    unset split($ww)
	    if {$ww != $original} {
		rename ::$ww {}
	    }
	}
	unset splitter($original)
	unset split_focus($original)
    }
    variable colouring
    if {[info exists colouring($w)]} {
	unset colouring($w)
    }
}

proc tw::split {w} {
    variable split
    if {[info exists split($w)]} {
	return
    }
    variable split_focus
    set split_focus($w) $w

    set W [winfo parent $w]
    
    set sash [frame $W.split -height 6 -bd 3 -relief raised]
    grid $W.split -sticky ew -column 0 -columnspan 3 -row 4

    set ww [MakeWindow $W [$w get 1.0 "end -1c"] \
      -minimal 1 -font [$w cget -font]]

    bind $sash <Button-1> "tw::_startGrip $sash %y $w $ww"
    bind $sash <B1-Motion> "tw::_handleGrip $sash %Y $w $ww"
    bind $sash <B1-ButtonRelease-1> "tw::_endGrip $sash %y $w $ww"

    variable splitter
    if {[info exists split($w)]} {
	set original $split($w)
	lappend splitter($original) $ww
    } else {
	set original $w
	set split($w) $w
	lappend splitter($original) $w $ww
    }
    set split($ww) $original
    # Set up correct bindings
    bindtags $ww [bindtags $w]
    after idle [list ::tw::arrangeToColour $ww {} {}]

}

proc tw::otherPane {w} {
    variable split
    if {![info exists split($w)]} {
	return -code error "No other pane"
    }
    variable splitter
    variable split_focus
    set original $split($w)
    foreach ww $splitter($original) {
	if {$ww != $split_focus($w)} {
	    focus $ww
	    return
	}
    }
}

proc tw::_startGrip {sash y w ww} {
    $sash configure -relief sunken
    grab $sash
}

# not an ideal solution, but kind of works.
proc tw::_handleGrip {sash y w ww} {
    set tophalf [expr {[winfo height $w] + ($y-[winfo rooty $sash])}]
    set height [expr {[winfo height $w] + [winfo height $ww]}]
    if {[set ww [expr {100*$tophalf/$height}]] < 0} {
	set ww 0
    }
    grid rowconfigure [winfo toplevel $w] 2 -weight $ww
    if {[set ww [expr {100-100*$tophalf/$height}]] < 0} {
	set ww 0
    }
    grid rowconfigure [winfo toplevel $w] 5 -weight $ww
    update
}

proc tw::_endGrip {sash y w ww} {
    $sash configure -relief raised
    grab release $sash
}


proc tw::toggleSplit {w} {
    variable split
    if {[info exists split($w)]} {
	unsplit $w
    } else {
	split $w
    }
}

proc tw::_destroyTextWidget {ww} {
    # Remove the procedure which is overriding
    # the tk widget name.
    rename ::$ww {}
    # Remove any possible bindings which may trigger
    # side-effects (esp. for destroy)
    bindtags $ww $ww
    # Destroy the text widget
    destroy $ww
    # Cleanup all state information stored in this
    # variable
    variable $ww
    catch {unset $ww}
    # Destroy the associated vertical scrollbar
    destroy [_textTo $ww scroll]
}

proc tw::unsplit {w} {
    variable split
    variable splitter
    variable split_focus
    set original $split($w)
    set W [winfo parent $original]
    foreach ww $splitter($original) {
	unset split($ww)
	if {$ww != $original} {
	    # Destroy the splitting widget
	    destroy [winfo parent $ww].split
	    # Destroy the text widget
	    _destroyTextWidget $ww
	}
    }
    grid rowconfigure $W 5 -weight 0
    grid rowconfigure $W 4 -weight 0
    grid rowconfigure $W 2 -weight 1
    grid rowconfigure $W 1 -weight 0
    unset splitter($original)
    unset split_focus($original)
}


proc splitter {w} {
    $w configure -relief sunken
    text_cmd toggleSplit
}

proc tw::_updateMargin {w} {
    variable $w
    
    set wm [_textTo $w margin]
    
    if {[set ${w}(linenumbers)]} {
	if {![winfo exists $wm]} {
	    # Implementation of a thin margin on the left for
	    # various control options.
	    text $wm -width 1 -borderwidth 2 -bg [color background] \
	      -relief flat
	    $wm configure -font [$w cget -font]
	    $wm tag configure right -justify right
	    
	    array set info [grid info $w]
	    grid $wm -sticky wns -row $info(-row) \
	      -column 0 -rowspan $info(-rowspan)
	    
	    set repeat -1
	    if {[info tclversion] >= 8.3} {
		$wm tag configure elidden -elide 1
	    }
	} else {
	    set repeat [expr {int([$w index end]) -1}]
	}

	if {![info exists ${w}(numlines)] || ([set ${w}(numlines)] != $repeat)} {
	    set repeat [expr {int([$w index end]) -1}]
	    if {[winfo exists $wm]} {
		set str ""
		for {set i 1} {$i <= $repeat} {incr i} {
		    if {$i == 1} {
			set str "$i"
		    } else {
			append str "\n$i"
		    }
		}
		$wm delete 0.0 end
		$wm insert end $str right
		$wm configure -width [string length $repeat]
		$wm yview moveto [lindex [$w yview] 0]
	    }
	    set ${w}(numlines) $repeat
	}
    } else {
	destroy $wm
    }
}

proc tw::toggleLineNumbers {w {create 1}} {
    variable $w

    if {[info exists ${w}(linenumbers)]} {
	set ${w}(linenumbers) [expr {1- [set ${w}(linenumbers)]}]
    } else {
	set ${w}(linenumbers) $create
    }

    _updateMargin $w
}

# Central location for creating and destroying the horizontal scrollbar
proc tw::horizScrollbar {w {create 1}} {
    variable $w
    
    if {[info exists ${w}(horizScrollbar)]} {
	set ${w}(horizScrollbar) [expr {1- [set ${w}(horizScrollbar)]}]
    } else {
	set ${w}(horizScrollbar) $create
    }
    set wh [tw::_textTo $w hscroll]
    if {[set ${w}(horizScrollbar)]} {
	scrollbar $wh -command "$w xview" -orient horizontal
	grid $wh -sticky sew -column 0 -row 3 -columnspan 2
	$w configure -xscrollcommand "$wh set"
    } else {
	if {[winfo exists $wh]} {
	    destroy $wh
	    $w configure -xscrollcommand ""
	}
    }
}

proc tw::addHyper {w from to hyper} {
    variable $w
    set ${w}(hyper:$from:$to) $hyper
}

proc tw::hyper {w what x y} {
    switch -- $what {
	"leave" {
	    message ""
	    $w config -cursor xterm
	    return
	}
    }
    if {[info tclversion] < 8.4} {
	set pos [tkTextClosestGap $w $x $y]
    } else {
	set pos [tk::TextClosestGap $w $x $y]
    }
    set range [$w tag prevrange color15 $pos]
    if {[lindex $range 1] < $pos} {
	#echo "Weird, I thought I had a hyper"
	return
    }
    variable $w
    set from [lindex $range 0]
    set to [lindex $range 1]
    if {![info exists ${w}(hyper:$from:$to)]} {
	# probably just underlined text
	return
    }
    set cmd [set ${w}(hyper:$from:$to)]
    
    switch -- $what {
	"activate" {
	    eval ::select $range
	    update
	    $w tag delete sel
	    update
	    CallHook hyper $cmd
	}
	"enter" - 
	"motion" {
	    message "---> '$cmd'"
	    $w config -cursor arrow
	}
    }
}

proc tw::balance {w} {
    set pair [eval findBalancedPair $w [selectLimits]]
    eval [list select $w] $pair
    message "Matching braces selected, from [join $pair { to }]."
}

proc tw::findBalancedPair {w {before insert} {after insert}} {
    variable lmatch
    variable rmatch
    while {1} {
	while {1} {
	    set f [singlesearch $w -forwards -regexp \
	      -- "\[\]\[\{\(\)\}\]" $after end]
	    if {$f == "" || ![text::isEscaped $f]} {
		break
	    }
	    set after "$f +1c"
	}
	while {1} {
	    set b [singlesearch $w -backwards -regexp \
	      -- "\[\]\[\{\(\)\}\]" $before 1.0]
	    if {$b == "" || ![text::isEscaped $b]} {
		break
	    }
	    set before "$b -1c"
	}
	
	# echo "$b $before $after $f"
	if {([string length $f] == 0) || ([string length $b] == 0)} {
	    error "No matching delimiters"
	}
	if {[set i [lsearch -exact $rmatch [$w get $f]]] != -1} {
	    # we found a backwards looking element while looking forwards
	    # We need to find its partner
	    return [list [match $w [get $w $f] "$f -1c"] [index $w "$f +1c"]]
	} elseif {[set i [lsearch -exact $lmatch [get $w $b]]] != -1} {
	    # we found the opposite, which is also ok
	    return [list $b [index $w "[match $w [get $w $b] "$b +1c"] +1c"]]
	} else {
	    # hmm, this is harder
	    set before [match $w [get $w $b] "$b -1c"]
	    set after "[match $w [get $w $f] "$f+1c"] +1c"
	}
    }
}

proc tw::matchError {w dir look pos} {
    if {$pos != ""} {
	set start [lindex [::split [set pos [index $w $pos]] .] 0]
	error "No match $dir for '$look' from $pos; possible unmatched\
	  delimiter at: [string trim [get $w ${start}.0 [expr {$start+1}].0]]"
    } else {
	error "No further braces found"
    }
}

# Return position before matching quotation mark
proc tw::matchQuoteBack {w pos limit} {
    set pos "$pos +1c"
    while {1} {
	#update ; if {$::alpha::abort} {set ::alpha::abort 0 ; error "Action aborted by user."}
	set pos1 [singlesearch $w -backwards -- \" $pos $limit]
	if {$pos1 == ""} { matchError $w back \" $pos }
	if {![text::isEscaped $pos1]} {
	    break
	} else {
	    set pos "$pos1 -1c"
	}
    }
    return $pos1
}

# Return position after matching quotation mark
proc tw::matchQuoteForward {w pos limit} {
    while {1} {
	#update ; if {$::alpha::abort} {set ::alpha::abort 0 ; error "Action aborted by user."}
	set pos1 [singlesearch $w -forwards -- \" $pos $limit]
	if {$pos1 == ""} { matchError $w forward \" $pos }
	if {![text::isEscaped $pos1]} {
	    break
	} else {
	    set pos "$pos1 +1c"
	}
    }
    return "$pos1 +1c"
}

proc tw::matchForward {w pos looking limit} {
    variable rmatch
    variable lmatch
    while 1 {
	while {1} {
	    #update ; if {$::alpha::abort} {set ::alpha::abort 0 ; error "Action aborted by user."}
	    set pos1 [singlesearch $w -forwards -regexp \
	      -- $::tw::matchlook $pos $limit]
	    if {$pos1 == ""} { matchError $w forward [lindex ${looking} end] $pos }
	    set pos $pos1
	    if {![text::isEscaped $pos]} {
		break
	    } else {
		set pos "$pos +1c"
	    }
	}
	set char [$w get $pos]
	if {$char == "\""} {
	    set pos [matchQuoteForward $w "$pos +1c" $limit]
	    continue
	}
	if {[lindex $looking end] == $char} {
	    set looking [lreplace $looking end end]
	    if {$looking == ""} {
		return $pos
	    }
	} else {
	    if {[lsearch -exact $rmatch $char] == -1} {
		lappend looking [lindex $rmatch [lsearch -exact $lmatch $char]]
	    } else {
		matchError $w forward [lindex ${looking} end] $pos
	    }
	}
	set pos "$pos +1c"
    }
}

set tw::matchlook "\[\]\"\)\}\[\(\{\]"

proc tw::matchBack {w pos looking limit} {
    variable rmatch
    variable lmatch
    set pos "$pos +1c"
    while 1 {
	while {1} {
	    #update ; if {$::alpha::abort} {set ::alpha::abort 0 ; error "Action aborted by user."}
	    set pos1 [singlesearch $w -backwards -regexp \
	      -- $::tw::matchlook $pos $limit]
	    if {$pos1 == ""} { matchError $w back [lindex ${looking} end] $pos }
	    set pos $pos1
	    if {![text::isEscaped $pos]} {
		break
	    }
	}
	set char [$w get $pos]
	if {$char == "\""} {
	    set pos [matchQuoteBack $w "$pos -1c" $limit]
	    continue
	}
	if {[lindex $looking end] == $char} {
	    set looking [lreplace $looking end end]
	    if {$looking == ""} {
		return "$pos"
	    }
	} else {
	    if {[lsearch -exact $lmatch $char] == -1} {
		lappend looking [lindex $lmatch [lsearch -exact $rmatch $char]]
	    } else {
		matchError $w back [lindex ${looking} end] "$pos -1c"
	    }
	}
    }
}

proc tw::match {w char pos {limit ""}} {
    # echo "tw::match $w $char $pos $limit"
    variable lmatch
    variable rmatch
    if {[set i [lsearch -exact $lmatch $char]] != -1} {
	lappend looking [lindex $rmatch $i]
	if {$limit == ""} { 
	    set limit end 
	} else {
	    set limit "$pos + ${limit}c"
	}
	return [matchForward $w $pos $looking $limit]
    } elseif {[set i [lsearch -exact $rmatch $char]] != -1} {
	lappend looking [lindex $lmatch $i]
	if {$limit == "0" || $limit == ""} { 
	    set limit 1.0 
	} else {
	    set limit "$pos - ${limit}c"
	}
	return [matchBack $w "$pos" $looking $limit]
    } else {
	error "Char '$char' unrecognised by match"
    }
}

proc tw::select {w from to} {
    $w tag add sel $from $to
}

proc tw::undo {w {allcontiguous 1}} {
    variable $w
    if {![info exists ${w}(undo)]} {
	return
    }
    set first 1
    while 1 {
	set action [lindex [set ${w}(undo)] end]
	if {$action == ""} {break}
	switch -- [lindex $action 0] {
	    "insert" {
		set len [string length [join [lrange $action 2 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != $new_pos]} {
			break
		    }	    
		}
	    }
	    "delete" {
		set len [string length [join [lrange $action 3 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != "$new_pos + ${len}c"]} {
			break
		    }	    
		}
	    }
	}
	# perform action
	# adjust the undo/redo lists
	lappend ${w}(redo) $action
	set ${w}(undo) [lrange [set ${w}(undo)] 0 [expr {[llength [set ${w}(undo)]] -2}]]
	
	switch -- [lindex $action 0] {
	    "insert" {
		if {$len > 1} {
		    uplevel 1 [list ::tw::silent_delete $w $where [list $where +${len}c]]
		} else {
		    uplevel 1 ::tw::silent_delete $w $where
		}
		set new_pos [index $w "$where - 1c"]
	    }
	    "delete" {
		uplevel 1 [list ::tw::silent_insert $w $where [lindex $action 3]]
		set new_pos $where
	    }
	}
	if {!$allcontiguous} {
	    break
	}
	set first 0
    }
    
    # undirty if necessary
    if {[llength [set ${w}(undo)]] == 0} { 
	unset ${w}(undo)
	dirty $w 0
    }
    if {[info exists new_pos]} {
	# Move insertion (and view) to the correct position.
	goto $w "$new_pos +1c"
    }
}

proc tw::redo {w {allcontiguous 1}} {
    variable $w
    if {![info exists ${w}(redo)]} {
	return
    }
    if {![info exists ${w}(undo)]} {
	dirty $w 1
    }
    set first 1
    while 1 {
	set action [lindex [set ${w}(redo)] end]
	if {$action == ""} {break}
	switch -- [lindex $action 0] {
	    "insert" {
		set len [string length [join [lrange $action 2 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w $where != $new_pos]} {
			break
		    }	    
		}
	    }
	    "delete" {
		set len [string length [join [lrange $action 3 end] ""]]
		set where [lindex $action 1]
		if {!$first} {
		    if {[compare $w "$where +${len}c" != $new_pos]} {
			break
		    }	    
		}
	    }
	}
	# perform action
	# adjust the undo/redo lists
	lappend ${w}(undo) $action
	set ${w}(redo) [lrange [set ${w}(redo)] 0 [expr {[llength [set ${w}(redo)]] -2}]]
	
	switch -- [lindex $action 0] {
	    "delete" {
		if {$len > 1} {
		    uplevel 1 [list ::tw::silent_delete $w $where [list $where +${len}c]]
		} else {
		    uplevel 1 ::tw::silent_delete $w $where
		}
		set new_pos $where
	    }
	    "insert" {
		uplevel 1 [list ::tw::silent_insert $w $where [lindex $action 2]]
		set new_pos [index $w "$where + ${len}c"]
	    }
	}
	if {!$allcontiguous} {
	    break
	}
	set first 0
    }
    # undirty if necessary
    if {[llength [set ${w}(redo)]] == 0} { 
	unset ${w}(redo)
    }
    if {[info exists new_pos]} {
	goto $w $new_pos
    }
}

proc tw::statusConfigure {w what args} {
    variable $w
    switch -- $what {
	"image" {
	    return [eval [list [winfo parent $w].rt.dirty configure -image] $args]
	}
	"background" {
	    return [eval [list [winfo parent $w].rt.dirty configure -background] $args]
	}
	default {
	    return -code error "bad option $what"
	}
    }
}

proc tw::lockClick {w {ctrl 0}} {
    variable $w
    if {[info exists ${w}(shell)]} {
	message "Clicking doesn't affect shell windows."
	return
    }
    switch -- [[winfo parent $w].rt.dirty cget -image] {
	"lock" {
	    if {!$ctrl} {return}
	    if {[set ${w}(dirty)]} {
		message "Clicking only affects locked or clean windows."
		return
	    }
	    read_only $w 0
	    CallHook lock $w 0
	}
	"" - 
	"clean" {
	    if {!$ctrl} {return}
	    if {[set ${w}(dirty)]} {
		message "Clicking only affects locked or clean windows."
		return
	    }
	    read_only $w 1
	    CallHook lock $w 1
	}
	"dirty" {
	    CallHook save $w
	}
	default {
	    alertnote "What!"
	}
    }
    
}

proc tw::readvar {w var} {
    variable $w
    set ${w}($var)
}

proc tw::read_only {w {d 1}} {
    variable $w
    # If there's no change
    if {[set ${w}(read-only)] == $d} {return}

    if {$d} {
	tw::dirty $w 0
	$w configure -state disabled
	[winfo parent $w].rt.dirty configure -image lock
    } else {
	$w configure -state normal
	[winfo parent $w].rt.dirty configure -image clean
    }
    set ${w}(read-only) $d
}

proc tw::dirty {w {d 1}} {
    set w [base_window $w]
    variable $w
    if {[info exists ${w}(shell)]} {
	return
    }
    if {[set ${w}(dirty)] != $d} {
	set ${w}(dirty) $d
	CallHook dirty $w $d
    }
    if {$d == 0} {
	if {[info exists ${w}(undo)]} {
	    unset ${w}(undo)
	}
	# This prevents a redo after an 'undo makes window unmodified'
	# so we remove it for the moment
	#if {[info exists ${w}(redo)]} {unset ${w}(redo)}
    }
    
    if {[$w cget -state] == "disabled"} {
	# it's a read-only window
	if {![dialog::yesno -y "Continue" -n "Throw error" \
	  "Modified a read-only window!"]} {
	    return -code error "Modified a read-only window!"
	}
    }
    
    [winfo parent $w].rt.dirty configure -image [expr {$d ? "dirty" : "clean"}]
}

proc tw::save {w} {
    variable undo
    variable redo
    variable $w
    if {[set ${w}(dirty)] != 0} {
	set ${w}(dirty) 0
	CallHook dirty $w 0
    }
    [winfo parent $w].rt.dirty configure -image clean \
      -background [default::color background]
    if {[info exists ${w}(undo)]} {
	unset ${w}(undo)
    }
    if {[info exists ${w}(redo)]} {
	unset ${w}(redo)
    }
}

proc tw::goto {w where} {
    set range [$w tag ranges backsel]
    if {$range != ""} {
	eval $w tag remove backsel $range
    }
    set range [$w tag ranges sel]
    if {$range != ""} {
	eval $w tag remove sel $range
    }
    mark $w set insert $where
    see $w insert
}

proc tw::insert {w where args} {
    if {[$w cget -state] == "disabled"} {return}
    variable split
    global wordWrap fillColumn
    # make sure we use the 'where' which corresponds to the correct pane if a
    # window has been split into pieces.  To do this we must turn it into a 
    # canonical line.col form
    set char [join $args ""]
    if {[string length $char] == 0} {
	# Insert empty string is a no-op
	return
    }
    set where [index $w $where]
    if {[info exists split($w)]} {
	set w $split($w)
    }
    variable $w
    # Basic wrap technique is as follows:
    # (i) Only bother with any wrapping if it's a single character being
    # inserted.
    # (ii) Given it's a single char, any whitespace just moves to the next
    # line
    # (iii) If not whitespace, check if we're in the middle of a word or
    # starting a new word.  The former shifts the whole word to the next
    # line, the latter just moves the word we're typing to the next line.
    if {[info exists wordWrap] && $wordWrap} {
	# only wrap if it's a single character
	if {[string length $char] == 1} {
	    set where [index $w $where]
	    if {[lindex [::split $where .] 1] >= $fillColumn} {
		switch -- $char {
		    "\r" - "\n" {}
		    " " {
			set args [list "\n"]
			set addReturn 1
		    }
		    "\t" {
			set args [concat [list "\n" ""] $args]
			set addReturn 1
		    }
		    default {
			if {[regexp "\[ \t\r\n\]" [get $w "$where -1c"]]} {
			    set args [concat [list "\n" ""] $args]
			    set addReturn 1
			} else {
			    # we've got a word going back
			    set p [search $w -backwards -regexp -- "\[ \t\r\n\]" $where]
			    mark $w set tmp-wordwrap $where
			    if {![compare $w $p <= "$where linestart"]} {
				delete $w $p
				insert $w $p "\n"
			    }
			    set where [index $w tmp-wordwrap]
			    mark $w unset tmp-wordwrap
			    set addReturn 1
			}
		    }
		}
	    }
	}
    }
    
    if {![info exists ${w}(shell)]} {
	if {![info exists ${w}(undo)]} {
	    dirty $w
	}
	set where [index $w $where]
	lappend ${w}(undo) "insert $where $args"
	if {[info exists ${w}(redo)]} {
	    unset ${w}(redo)
	}
	global autoSave changesLim
	if {$autoSave && [string length $changesLim]} {
	    if {[llength [set ${w}(undo)]] >= $changesLim} {
		set doAnAutoSave 1
	    }
	}
    }
    eval [list silent_insert $w $where] $args

    if {[info exists addReturn]} {
	global indentOnReturn
	if {$indentOnReturn} {
	    mark $w set tmp-wordwrap insert
	    bind::IndentLine
	    mark $w set insert tmp-wordwrap
	    mark $w unset tmp-wordwrap
	}
    }
    CallHook position [::split [index $w insert] .]

    if {[info exists doAnAutoSave]} {
	CallHook save $w
    }
}

proc tw::delete {w where1 {where2 ""}} {
    if {[$w cget -state] == "disabled"} {return}
    # make sure we use the 'where' which corresponds to the correct pane if a
    # window has been split into pieces.  To do this we must turn it into a 
    # canonical line.col form
    set where1 [index $w $where1]
    if {$where2 != ""} {
	set where2 [index $w $where2]
	if {[compare $w $where1 == $where2]} {
	    return
	}
    }
    variable split
    if {[info exists split($w)]} {
	set w $split($w)
    }
    variable $w
    if {![info exists ${w}(shell)]} {
	if {![info exists ${w}(undo)]} {
	    dirty $w
	}
	if {$where2 != ""} { 
	    lappend ${w}(undo) [list delete $where1 $where2 [::tw::$w get $where1 $where2]]
	} else {
	    lappend ${w}(undo) [list delete $where1 $where2 [::tw::$w get $where1]]
	}
	if {[info exists ${w}(redo)]} {
	    unset ${w}(redo)
	}
    }
    if {$where2 != ""} {
	silent_delete $w $where1 $where2
    } else {
	silent_delete $w $where1
    }
}

proc tw::backward_word {w pos} {
    global wordBreak wordBreakPreface
    set reg "(^|\r|\n|$wordBreakPreface)$wordBreak"

    set to [singlesearch $w -backwards -regexp -- $reg "$pos -1c" 1.0]
    if {$to == ""} {
	return 1.0
    } else {
	set reg "($wordBreakPreface|\r|\n|^)"
	if {[regexp -- $reg [get $w $to] match]} {
	    if {[string length $match]} {
		return "$to +1c"
	    } else {
		return $to
	    }
	} else {
	    return "$to +1c"
	}
    }
}

proc tw::forward_word {w pos} {
    global wordBreak
    set to [singlesearch $w -forwards -regexp -count c -- "$wordBreak" $pos]
    if {$to == ""} {
	return end
    } else {
	return "$to +${c}c"
    }
}

proc tw::double_click {w x y} {
    if {[info tclversion] < 8.4} {
	set cur [tkTextClosestGap $w $x $y]
    } else {
	set cur [tk::TextClosestGap $w $x $y]
    }
    select $w [backward_word $w "$cur +1c"] [forward_word $w "$cur -1c"]
}

proc tw::binding_capture {w bt} {
    variable $w
    if {[info exists ${w}(bindtags)]} {
	error "Already got binding capture!"
    } else {
	set ${w}(bindtags) [bindtags $w]
	bindtags $w "BindReset $bt BindNoMatch"
	bind BindReset <Key> [list tw::binding_reset $w]
    }
}

bind BindNoMatch <Key> [list message "No matching prefixed binding."]

proc tw::binding_reset {w} {
    variable $w
    message ""
    if {[info exists ${w}(bindtags)]} {
	bindtags $w [set ${w}(bindtags)]
	unset ${w}(bindtags)
    } else {
	error "No previous binding capture!"
    }
}

proc tw::setTabSize {w {v ""}} {
    variable $w
    if {$v == ""} {
	set v [set ${w}(tabsize)]
    } else {
	set ${w}(tabsize) $v
    }
    # If we have a patched text widget which can tab properly
    if {![catch {$w cget -fixedtabs}]} {
	$w configure -tabs ""
	$w configure -fixedtabs $v
	return
    }
    if {$v == 8} {
	$w configure -tabs ""
    } else {
	set charWidth [font measure [$w cget -font] " "]
	$w configure -tabs [expr {$v * $charWidth}]
    }
}

proc tw::encoding {w args} {
    variable $w
    switch -- [llength $args] {
	0 {
	    set ${w}(encoding)
	}
	1 {
	    set val [lindex $args 0]
	    set ${w}(encoding) $val
	}
	default {
	    error "Wrong number of args"
	}
    }
}

proc tw::setFontsTabs {w} {
    variable $w
    set fnt [$w cget -font]
    regexp {([^0-9]+)([0-9]+)} $fnt "" fnt size
    foreach {f s t} [chooseFontTab [string trim $fnt] $size [set ${w}(tabsize)]] {}
    $w configure -font "$f $s"
    setTabSize $w $t
}

proc tw::see {w args} {
    #puts "see $w $args"
    set old [_viewable $w]
    set ret [uplevel 1 ::tw::[getSplitFocus $w] see $args]
    set new [$w yview]
    set wm [_textTo $w margin]
    if {[winfo exists $wm]} {
	eval [list ::$wm yview] moveto [lindex $new 0]
    }
    eval [list arrangeToColourNewlyRevealed $w] $old
    return $ret
}

proc tw::yview {w args} {
    #puts "yview $w $args"
    if {[llength $args]} {
	set old [_viewable $w]
	set ret [uplevel 1 [list $w yview] $args]
	set new [$w yview]
	set wm [_textTo $w margin]
	if {[winfo exists $wm]} {
	    eval [list ::$wm yview] moveto [lindex $new 0]
	}
	# now recolour as appropriate
	eval [list arrangeToColourNewlyRevealed $w] $old
	return $ret
    } else {
	# '$w' is a command in the 'tw' namespace so this
	# is not an infinite loop!
	$w yview
    }
}

proc tw::windowConfigure {w width height} {
    variable $w
    if {![info exists ${w}(viewable)]} {
	_viewable $w
	return
    }
    set old [set ${w}(viewable)]
    # now recolour as appropriate
    eval [list arrangeToColourNewlyRevealed $w] $old
}

proc tw::collapse {w pos} {
    if {[info tclversion] < 8.3} { echo "Collapsing requires Tcl 8.3" ; return }
    set tags [$w tag names $pos]
    if {[lsearch -exact $tags elidden] != -1} {
	set nr [$w tag prevrange elidden $pos]
    } else {
	set nr [$w tag nextrange elidden "$pos linestart" "$pos lineend +1c"]
    }
    if {[llength $nr]} {
	eval [list $w tag remove elidden] $nr
	set wm [_textTo $w margin]
	if {[winfo exists $wm]} {
	    eval [list ::$wm tag remove elidden] $nr
	}
    } else {
	foreach {s a1 a2 bs be} [procs::findEnclosing $pos proc 1] {}
	$w tag add elidden "$bs +1c" "$be lineend +1c"
	#puts "$bs $be"
	set wm [_textTo $w margin]
	if {[winfo exists $wm]} {
	    ::$wm tag add elidden "$bs lineend" "$be lineend +1c"
	}
    }
}

proc tw::activateHook {n} {
    set range [$n tag ranges backsel]
    if {$range != ""} {
	eval $n tag remove backsel $range
	eval $n tag add sel $range
    }
    variable split
    if {[info exists split($n)]} {
	variable split_focus
	set original $split($n)
	set split_focus($original) $n
    }
    CallHook activate [base_window $n]
}

proc tw::deactivateHook {n} {
    variable $n
    if {[info exists ${n}(bindtags)]} {
	tw::binding_reset $n
    }
    set range [$n tag ranges sel]
    if {$range != ""} {
	eval $n tag remove sel $range
	eval $n tag add backsel $range
    }
    CallHook deactivate [base_window $n]
}

bind AlphaStyle <Configure> {tw::windowConfigure "%W" %w %h}
bind AlphaStyle <FocusIn> {::tw::activateHook %W}
bind AlphaStyle <FocusOut> {::tw::deactivateHook %W}
bind AlphaStyle <Key-bracketright> [list flash "\]"]
bind AlphaStyle <Key-parenright> [list flash "\)"]
bind Text <Double-Button-1> "::tw::double_click %W %x %y"
bind Lock <[lindex $alpha::modifier_keys 0]-Button-1> "text_cmd lockClick 1"
bind Lock <Button-1> "text_cmd lockClick 0"

if {[info tclversion] < 8.4} {
# Default behaviour in Alpha is to create a selection with
# a shift-click, even if no selection currently exists.
# So, we modify this core text widget procedure.  This
# is fixed in Tk 8.4
proc tkTextResetAnchor {w index} {
    if {![string compare [$w tag ranges sel] ""]} {
	#$w mark set anchor $index
	return
    }

    set a [$w index $index]
    set b [$w index sel.first]
    set c [$w index sel.last]
    if {[$w compare $a < $b]} {
	$w mark set anchor sel.last
	return
    }
    if {[$w compare $a > $c]} {
	$w mark set anchor sel.first
	return
    }
    scan $a "%d.%d" lineA chA
    scan $b "%d.%d" lineB chB
    scan $c "%d.%d" lineC chC
    if {$lineB < $lineC+2} {
	set total [string length [$w get $b $c]]
	if {$total <= 2} {
	    return
	}
	if {[string length [$w get $b $a]] < ($total/2)} {
	    $w mark set anchor sel.last
	} else {
	    $w mark set anchor sel.first
	}
	return
    }
    if {($lineA-$lineB) < ($lineC-$lineA)} {
	$w mark set anchor sel.last
    } else {
	$w mark set anchor sel.first
    }
}
}

# Override with new version which doesn't use 'catch' to avoid
# generating a long stack trace every time the window doesn't
# have a current selection (because of the way we wrap up the text
# widget, this stack trace is longer than it ought to be...)

if {[info tclversion] < 8.4} {
    # Old versions of Tk.
    proc tkTextInsert {w s} {
	if {[string equal $s ""] || [string equal [$w cget -state] "disabled"]} {
	    return
	}
	if {[llength [$w tag ranges sel]]} {
	    if {[$w compare sel.first <= insert] \
		    && [$w compare sel.last >= insert]} {
		$w delete sel.first sel.last
	    }
	}
	$w insert insert $s
	$w see insert
    }
}
