## -*-Tcl-*-
 # ###################################################################
 #  Alphatk - the ultimate editor
 # 
 #  FILE: "syntaxColouring.tcl"
 #                                    created: 04/12/98 {22:45:38 PM} 
 #                                last update: 12/14/2001 {13:04:40 PM}}} 
 #  Author: Vince Darley
 #  E-mail: vince.darley@kagi.com
 #    mail: Flat 10, 98 Gloucester Terrace, London W2 6HP
 #     www: http://www.santafe.edu/~vince/Alphatk.html/Alphatk.html
 #  
 # Copyright (c) 1998-2002  Vince Darley
 # 
 # See the file "license.terms" for information on use and redistribution
 # of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 # 
 # In particular, while this is 'open source', it is NOT free, and
 # cannot be copied in full or in part except according to the terms
 # of the license agreement.
 # 
 # ###################################################################
 ##

namespace eval tw {}

set tw::colorList [list blue cyan green magenta red white yellow]

# Don't remove marks or selection when recolouring.
set tw::nonColorMarkRegexp {^(mark:|(back)?sel$)}

## 
 # -------------------------------------------------------------------------
 # 
 # "tw::arrangeToColourNewlyRevealed" --
 # 
 # This procedure is called whenever the window is scrolled (either
 # with the scrollbar or the '$w see' command), or if text is deleted
 # from or inserted into the current window (since this may result in
 # new lines becoming visible at either or both top/bottom of the
 # window).
 # 
 # Here 'oldtop', 'oldbottom' are the previous line numbers of
 # the first and last lines before the recent changes.  The procedure
 # itself will query '_viewable' to determine the current top and 
 # bottom.  Finally, some text may have been deleted between the
 # indices [lindex $args 0] and [lindex $args 1], if given, and
 # in that case we have to make sure to adjust the relative line
 # numbers in the calculations of what lines are newly visible.
 #  
 # -------------------------------------------------------------------------
 ##
proc tw::arrangeToColourNewlyRevealed {w oldtop oldbottom args} {
    #puts "$w $oldtop $oldbottom $args"
    foreach {top bottom} [_viewable $w] {}
    if {[llength $args]} {
	set from [expr {int([lindex $args 1])}]
	set to [lindex $args 2]
	if {$to == ""} {
	    set to [expr {int([$w index insert])}]
	} else {
	    set to [expr {int([$w index $to])}]
	}
	if {[lindex $args 0] == "delete"} {
	    # diff should be negative.
	    set diff [expr {$from - $to}]
	    lappend range $from $from
	} else {
	    # Text added
	    set diff [expr {$to - $from}]
	    if {$to > $top && $from < $bottom} {
		# Colour entire range, provided some of it is onscreen
		lappend range $from $to
	    }
	}
	if {$diff == 0} {
	    #lappend range $from $from
	} else {
	    if {$to < $oldtop} {
		incr oldtop $diff 
		incr oldbottom $diff
	    } elseif {$from > $oldbottom} {
		# do nothing
	    } elseif {$from >= $oldtop && $to <= $oldbottom} {
		incr oldbottom $diff
	    } elseif {$from < $oldtop} {
		# deleted range overlaps top of window
		incr oldbottom $diff
		set oldtop $from
	    } else {
		# deleted range overlaps bottom of window
		set oldbottom $from
	    }
	}
    }
    
    #puts [list $oldtop $oldbottom $top $bottom]
    if {$top < $oldtop} {
	# Some new text has appeared at the top
	if {$bottom < $oldtop} {
	    lappend range $top $bottom
	} else {
	    lappend range $top $oldtop
	}
    }
    if {$bottom > $oldbottom} {
	# Some new text has appeared at the bottom
	if {$top > $oldbottom} {
	    lappend range $top $bottom
	} else {
	    lappend range $oldbottom $bottom
	}
    }
    if {[info exists range]} {
	#puts "color $range"
	eval [list arrangeToColour $w] $range
    }
}

proc tw::arrangeToColorIfVisible {w from to} {
    foreach {top bottom} [_viewable $w] {}
    set from [expr {int([$w index $from])}]
    set to [expr {int([$w index $to])}]
    if {$from < $top} { set from $top }
    if {$to > $bottom} { set to $bottom }
    if {$from < $to} {
	arrangeToColour $w $from $to
    }
}

proc tw::arrangeToColour {w args} {
    #puts stderr "Perhaps colour $w $args"
    global coloring
    if {!$coloring} {return}
    variable $w
    if {![info exists ${w}(colouring)]} {
	set ${w}(colouring) $args
	after idle [list tw::colourSoon $w]
    } else {
	eval lappend ${w}(colouring) $args
    }
}

proc tw::colourSoon {w} {
    #puts stderr "Go $w"
    if {![winfo viewable $w]} {
	tkwait visibility $w 
    }
    after idle [list tw::colourWinFraction $w]
}

proc tw::colourWinFraction {w} {
    variable ${w}
    variable split
    if {[info exists ${w}(colouring)]} {
	set args [set ${w}(colouring)]
	unset ${w}(colouring)
    } else {
	# Window probably destroyed.
	return
    }
    #puts stderr "Colourise $w $args"
    if {[info exists split($w)]} {
	set m $::win::Modes($::win::tktitle($split($w)))
    } else {
	set m $::win::Modes($::win::tktitle($w))
    }
    #puts "colourWinFraction $w $m $args"
    foreach {st end} $args {
	#puts stderr "Colouring $w in range $st $end"
	if {![string length $st] && ![string length $end]} {
	    set el [lindex [::split [$w index end] .] 0]
	    set yv [$w yview]
	    set st [expr {int([lindex $yv 0] * $el)}]
	    # Need extra 1 to colour lines which are half off the bottom
	    # (actually rounding up any non-zero fraction would be ideal)
	    set end [expr {int(1.0 + [lindex $yv 1] * $el)}]
	}
	if {$st == $end} {
	    colourLine $w $m "${st}.0"
	} else {
	    if {($end - $st) > 10} {
		# We need to colour a lot, therefore it's worth
		# checking whether some is off screen
		set el [lindex [::split [$w index end] .] 0]
		set yv [$w yview]
		set w_st [expr {int([lindex $yv 0] * $el)}]
		# Need extra 1 to colour lines which are half off the bottom
		# (actually rounding up any non-zero fraction would be ideal)
		set w_end [expr {int(1.0 + [lindex $yv 1] * $el)}]
		if {$w_st > $st} {set st $w_st}
		if {$w_end < $end} {set end $w_end}
	    }
	    set pos_end [tw::index $w ${end}.0]
	    set pos_start ${st}.0
	    while {[$w compare $pos_start <= $pos_end]} {
		set pos_start [colourLine $w $m $pos_start]
		if {[$w compare $pos_start == end]} {break}
	    }
	}
    }
}

proc tw::colourIndexRange {w m from to} {
    while {[$w compare $from < $to]} {
	# Word 1 - after the current point
	set ws1 [$w index [backward_word $w $from]]
	set we1 [$w index [forward_word $w $from]]
	
	colourWord $w $m $ws1 $we1
	set from $we1
    }
}

proc tw::colourLine {w m {li ""}} {
    #puts "colourLine $w $m $li"
    if {![info exists ::${m}::lineRegexp]} {return "$li lineend +1c"}
    if {$li == ""} { set li [$w index "insert linestart"] }
    set txt [$w get $li [set lend [$w index "${li} lineend"]]]
    while {[eval [list regexp [set ::${m}::lineRegexp] $txt ""] \
      [set ::${m}::lineVars]]} {
	#echo "-$space-$comment-$quote-$txt-"
	set li [$w index "$li + [string length $space]c"]
	if {[info exists comment] && ($comment != "")} {
	    $w tag add [set ::${m}::specialColors(comment)] $li "$li lineend"
	    break
	}
	if {[info exists multicomment] && ($multicomment != "")} {
	    set mc [lindex [set ::${m}::multiComment] 1]
	    set endc [$w search $mc $li end]
	    #puts stderr "got: $li, $mc, $endc"
	    if {$endc != ""} {
		# If the insertion point is in the middle of this section,
		# don't colour, otherwise we tend to just colour vast pieces
		# of code when the user types the opening comment
		if {1 || [$w compare insert <= $li] \
		  || [$w compare insert > $endc]} {
		    set end [$w index "$endc + [string length $mc]c"]
		    $w tag add [set ::${m}::specialColors(comment)] $li $end
		    set range [$w tag prevrange [set ::${m}::specialColors(comment)] "$li +1c"]
		    if {[llength $range]} {
			set index1 [lindex $range 1]
			if {[$w compare $index1 > $end]} {
			    $w tag remove [set ::${m}::specialColors(comment)] "$end" $index1
			    #puts stderr "Remove $end, $index1"
			    if {[$w compare $index1 > "$end lineend"]} {
				# We have just removed tags from a large
				# block of text; arrange to recolour it
				# soon.
				arrangeToColorIfVisible $w \
				  "$endc lineend +1c" $index1
			    }
			}
		    }
		    return "$endc lineend +1c"
		} else {
		    $w tag add [set ::${m}::specialColors(comment)] $li insert
		}
	    }
	}
	if {[info exists quote] && ($quote != "")} {
	    set le [$w index "$li +[string length $quote]c"]
	    if {[$w get "$le -1c"] != "\""} {
		# We presumably reached the end of the line without
		# finding a match
		set endc [$w search -regexp "\[^\\\\\]\"" $le end]
		if {$endc != ""} {
		    # If the insertion point is in the middle of this
		    # section, don't colour, otherwise we tend to just
		    # colour vast pieces of code when the user types the
		    # opening quote!
		    if {[$w compare insert < $li] \
		      || [$w compare insert > $endc]} {
			$w tag add [set ::${m}::specialColors(quote)] $li "$endc + 2c"
			return "$endc lineend +1c"
		    }
		}
	    }
	    $w tag add [set ::${m}::specialColors(quote)] $li $le
	    set li $le
	    set quote ""
	} else {
	    if {$txt == ""} {break}
	    set firstChar [string index $txt 0]
	    if {[info exists ::${m}::specialChars($firstChar)]} {
		set color [set ::${m}::specialChars($firstChar)]
		$w tag add $color $li "$li +1c"
		set li [$w index "$li +1c"]
		set txt [string range $txt 1 end]
	    } else {
		if {[regexp -- $::wordBreakPreface $firstChar]} {
		    set li [$w index "$li +1c"]
		    set txt [string range $txt 1 end]
		} else {
		    set we1 [$w index [forward_word $w ${li}]]
		    if {[$w compare $we1 > $lend]} {
			set offplus [colourWord $w $m $li $lend]
		    } else {
			# Could be a bug in multisearch
			if {[$w compare $we1 <= $li]} {break}
			set offplus [colourWord $w $m $li $we1]
		    }
		    set li $we1
		    set txt [string range $txt $offplus end]
		}
	    }
	}
    }
    return "$lend +1c"
}

# Use full paths of variables to avoid 'global', which will speed
# up this heavily-used procedure.
proc tw::colourWord {w m i1 i2} {
    set len [string length [set word [$w get $i1 $i2]]]
    if {!$len} {return 0}
    if {[info exists ::${m}::specialColors(quote)]} {
	$w tag remove [set ::${m}::specialColors(quote)] $i1 $i2
    }
    if {[info exists ::${m}::keywords($word)]} {
	$w tag add [set ::${m}::keywords($word)] $i1 $i2
    } elseif {[info exists ::${m}::magicPrefix]} {
	if {[string index $word 0] == [set ::${m}::magicPrefix]} {
	    if {$len == 1 || [regexp -- $::wordBreak [string range $word 0 1]]} {
		$w tag add [set ::${m}::magicColor] $i1 $i2
	    } else {
		$w tag add [set ::${m}::magicColor] $i1 "$i1 + 1c"
	    }
	}
    }
    set len
}


